/*******************************************************************************
 Copyright 2023 NXP 

 NXP Confidential and Proprietary. This software is owned or controlled by NXP
 and may only be used strictly in accordance with the applicable license terms.
 By expressly accepting such terms or by downloading, installing, activating
 and/or otherwise using the software, you are agreeing that you have read, and
 that you agree to comply with and are bound by, such license terms. If you do
 not agree to be bound by the applicable license terms, then you may not retain,
 install, activate or otherwise use the software.
 *******************************************************************************/

package com.freescale.system.browser.epl.help;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.viewers.IFilter;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import com.freescale.system.browser.epl.ISectionDescriptorEpl;
import com.freescale.system.browser.epl.ISectionEpl;
import com.freescale.system.browser.epl.TabDescriptorEpl;
import com.freescale.system.browser.epl.TabEpl;

public class CustomTabDescriptorEpl extends TabDescriptorEpl {

	/**
	 * The tab's ID as specified in the extension (XML)
	 */
	private String fId;

	/**
	 * The ID attribute used in the systemTabs extension point.
	 */
	private static final String ATT_ID = "id"; //$NON-NLS-1$
	
	/**
	 * List of section descriptor this tab owns.
	 */
	protected List<ICustomSectionDescriptorEpl> fSectionDescriptors = new ArrayList<>(5);

	/**
	 * Constructor for CustomTabDescriptor.
	 * 
	 * @param configurationElement
	 *            the extension point configuration element for the tab descriptor.
	 */
	public CustomTabDescriptorEpl(IConfigurationElement configurationElement) {
		super(configurationElement);
		
		fId = configurationElement.getAttribute(ATT_ID);
		fTitle = configurationElement.getAttribute(ATT_TITLE);

		String iconstr = configurationElement.getAttribute(ATT_IMAGE);
		if (iconstr != null) {
			fIcon = AbstractUIPlugin.imageDescriptorFromPlugin(
					configurationElement.getDeclaringExtension().getNamespaceIdentifier(), iconstr);
		}
		
		if (fId == null || fTitle == null) {
			// the tab id, label and category are mandatory - log error
			handleTabError(configurationElement, null);
		}
	}
	
	/**
	 * Append the description of a section that a tab of our type will contain
	 * 
	 * @param target
	 *            the section descriptor to append.
	 */
	protected void addSectionDescriptor(ISectionDescriptorEpl sectionDescriptor) {
		if (sectionDescriptor instanceof ICustomSectionDescriptorEpl) {
			ICustomSectionDescriptorEpl target = (ICustomSectionDescriptorEpl)sectionDescriptor;
		  if (!insertSectionDescriptor(target)) {
			fSectionDescriptors.add(target);
		  }
		}
	}

	/**
	 * Insert the section descriptor into the section descriptor list.
	 * 
	 * @param target
	 *            the section descriptor to insert.
	 * @return <code>true</code> if the target descriptor was added to the
	 *         descriptors list.
	 */
	private boolean insertSectionDescriptor(final ICustomSectionDescriptorEpl target) {
		if (target.getAfterSection().equals(TOP)) {
			fSectionDescriptors.add(0, target);
			return true;
		}
		int i = 0;
		for (ICustomSectionDescriptorEpl sectionDecriptor : fSectionDescriptors) {
			if (target.getAfterSection().equals(sectionDecriptor.getId())) {
				fSectionDescriptors.add(i + 1, target);
				return true;
			} else {
				if (sectionDecriptor.getAfterSection().equals(target.getId())) {
					fSectionDescriptors.add(i, target);
					return true;
				}
			}
			i++;
		}
		return false;
	}
	
	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.TabDescriptor#createTab()
	 */
	@Override
	protected TabEpl createTabEx() {
		List <ISectionEpl> sections = new ArrayList<>(fSectionDescriptors.size());
		
		for (final ICustomSectionDescriptorEpl descriptor : fSectionDescriptors) {
			sections.add(descriptor.getSection());
		}
		
		TabEpl tab = new TabEpl(this, sections);		
		return tab;
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabItem#getId()
	 */
	@Override
	public String getId() {
		return fId;
	}
	
	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.TabDescriptor#showForInput(java.lang.Object)
	 */
	public boolean showForInput(Object input) {
		// If at least one section is applicable, we show the tab
		for (ICustomSectionDescriptorEpl descriptor : fSectionDescriptors) {
			IFilter filter = descriptor.getFilter();
			if (filter != null) {
				if (filter.select(input)) {
					return true;
				}
			} else {
				return true;
			}
		}
		return false;
	}
}
