/**
* Copyright 2019, 2023 NXP
*
* SPDX-License-Identifier: EPL-1.0

*/
 package com.nxp.s32ds.ext.rcp.utils.epl.viewers;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Supplier;

import org.eclipse.jface.viewers.ITreeContentProvider;

/**
 * Class is used in {@link ContainerFilteredCheckboxTreeViewer}. Intended to
 * store and manage a check-box styled tree elements selection. A
 * selection/deselection of element propagates to children with the same state.
 * It also change state of parent - add parent in selected set if all children
 * are selected, and remove parent from selected set if some child is not
 * selected.
 */

public class TreeContentCheckStateHolder {

	private final Set<Object> selectedElements = new HashSet<>();

	private final Supplier<ITreeContentProvider> contentProviderSupplier;

	public TreeContentCheckStateHolder(Supplier<ITreeContentProvider> contentProviderSupplier) {
		this.contentProviderSupplier = contentProviderSupplier;
	}

	public void clear() {
		selectedElements.clear();
	}

	public Object[] getCheckedElements() {
		return selectedElements.toArray();
	}

	public boolean isChecked(Object element) {
		return selectedElements.contains(element);
	}

	public boolean isChildChecked(Object element) {
		Object[] children = contentProviderSupplier.get().getChildren(element);
		if (children == null) {
			return false;
		}
		for (Object chid : children) {
			if (isChecked(chid) || isChildChecked(chid)) {
				return true;
			}
		}
		return false;
	}

	public void setChecked(Object element, boolean checked) {
		applyState(element, checked);
		updateChildrenElements(element, checked);
		updateParentElement(element, checked);
	}

	protected void applyState(Object element, boolean checked) {
		if (checked) {
			selectedElements.add(element);
		} else {
			selectedElements.remove(element);
		}
	}

	protected void updateChildrenElements(Object element, boolean checked) {
		Object[] children = contentProviderSupplier.get().getChildren(element);
		if (children == null) {
			return;
		}
		for (Object child : children) {
			applyState(child, checked);
			updateChildrenElements(child, checked);
		}
	}

	protected void updateParentElement(Object element, boolean checked) {
		ITreeContentProvider contentProvider = contentProviderSupplier.get();
		Object parent = contentProvider.getParent(element);
		if (parent == null) {
			return;
		}

		if (checked) {
			// getChildren should not return null - otherwise error in provider
			for (Object child : contentProvider.getChildren(parent)) {
				if (!isChecked(child)) {
					checked = false;
					break;
				}
			}
		}
		applyState(parent, checked);
		updateParentElement(parent, checked);
	}

}