/*
 * Copyright 2014-2017 Freescale
 * Copyright 2018,2021,2023 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

package com.nxp.sa.ui.common.epl;

import java.io.File;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.sourcelookup.ISourceLookupDirector;
import org.eclipse.debug.ui.sourcelookup.SourceLookupDialog;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.freescale.sa.Messages;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.LaunchConfigUtils;
import com.freescale.sa.util.Utils;
import com.ibm.icu.text.MessageFormat;

/**
 * Dialog that lets you select a replacement for the missing source file and
 * modifies the source locator accordingly.
 * 
 * Functionality is similar to
 * org.eclipse.cdt.debug.internal.ui.sourcelookup.CSourceNotFoundEditor
 */
public class SASourceLookupDialog extends Dialog {

	private static final String UID_KEY = ".uid"; //$NON-NLS-1$
	private static final String UID_CLASS_NAME = SASourceLookupDialog.class
			.getName();

	/* uids for the buttons */
	public static final String UID_LOCATE_FILE_BUTTON = UID_CLASS_NAME
			+ "locateFileButton"; //$NON-NLS-1$
	public static final String UID_EDIT_LOOKUP_BUTTON = UID_CLASS_NAME
			+ "editLookupButton"; //$NON-NLS-1$

	private Button locateFileButton;
	private Button editLookupButton;
	private Button doNotPromptAgainButton;
	private Label errorMessage;
	private Color redColor;
	private String missingFile = CommonConstants.EMPTY_STRING;
	private boolean doNotPromptAgain = false;
	
	private IPath newPath;

	/*
	 * launch configuration object which will be updated after locating the file
	 * or the source lookup paths
	 */
	private ILaunchConfiguration config = null;

	/*
	 * resultsConfig file name which will be updated after locating the file or
	 * the source lookup paths
	 */
	private String resultsConfig = null;

	/**
	 * Constructor. It requires the parent shell, the missing file (path) and an
	 * ILaunchConfiguration object to be used for lookup and updated
	 * 
	 * @param parent
	 *            the parent shell
	 * @param file
	 *            path of the missing file
	 * @param launchConfig
	 *            the launch configuration object
	 */
	public SASourceLookupDialog(Shell parent, String file,
			ILaunchConfiguration launchConfig) {
		super(parent);
		super.setShellStyle(SWT.OK | SWT.RESIZE);
		config = launchConfig; 
		if (file != null) {
			missingFile = file;
		}
	}

	/**
	 * Constructor. It requires the parent shell, the missing file (path) and an
	 * ILaunchConfiguration object to be used for lookup and updated
	 * 
	 * @param parent
	 *            the parent shell
	 * @param file
	 *            path of the missing file
	 * @param resultsConfigFileName
	 *            the resultsConfig file
	 */
	public SASourceLookupDialog(Shell parent, String file,
			String resultsConfigFileName) {
		super(parent);
		super.setShellStyle(SWT.OK | SWT.RESIZE);
		
		if ((resultsConfigFileName != null) 
				&& (new File(resultsConfigFileName).exists())) {
			resultsConfig = resultsConfigFileName;
		}
		if (file != null) {
			missingFile = file;
		}
	}

	@Override
	protected Control createDialogArea(Composite _parent) {

		getShell().setText(Messages.SourceNotFound_2);

		Composite outerComposite = new Composite(_parent, SWT.NONE);

		outerComposite.setLayout(new GridLayout());
		outerComposite.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Label text = new Label(outerComposite, SWT.WRAP);

		text.setLayoutData(new GridData(GridData.FILL_BOTH));
		text.setBackground(outerComposite.getBackground());
		if (missingFile.length() > 0) {
			String message = MessageFormat.format(Messages.SourceNotFound_00,
					new Object[] { missingFile });

			text.setText(message);
		} else {
			text.setText(Messages.SourceNotFound_0);
		}
		text.pack();
		
		final Label tip = new Label(outerComposite, SWT.WRAP);

		tip.setLayoutData(new GridData(GridData.FILL_BOTH));
		tip.setBackground(outerComposite.getBackground());
		tip.setText(Messages.SourceNotFound_7 + "\n\n"); //$NON-NLS-1$
		errorMessage = new Label(outerComposite, SWT.WRAP);
		errorMessage.setLayoutData(new GridData(GridData.FILL_BOTH));
		redColor = new Color(Display.getCurrent().getActiveShell().getDisplay(), 255, 0 ,50);
		errorMessage.setForeground(redColor);
		errorMessage.addDisposeListener(new DisposeListener() {
			
			@Override
			public void widgetDisposed(DisposeEvent arg0) {
				redColor.dispose();
				
			}
		});
		createButtons(outerComposite);

		return _parent;
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		// create OK button only, with another ID. Otherwise, the dialog will
		// return immediately.
		createButton(parent, IDialogConstants.FINISH_ID,
				IDialogConstants.OK_LABEL, true);
	}

	@Override
	protected void buttonPressed(int buttonId) {
		if (IDialogConstants.FINISH_ID == buttonId) {
			okPressed();
		}
	}

	protected void createButtons(Composite parent) {
		{
			GridData data;

			locateFileButton = new Button(parent, SWT.PUSH);
			data = new GridData();
			data.grabExcessHorizontalSpace = false;
			data.grabExcessVerticalSpace = false;
			locateFileButton.setLayoutData(data);
			locateFileButton.setText(Messages.SourceNotFound_1);
			locateFileButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent evt) {
					locateFile();
				}
			});
			locateFileButton.setData(UID_KEY, UID_LOCATE_FILE_BUTTON);
		}
		{
			GridData data;

			editLookupButton = new Button(parent, SWT.PUSH);
			data = new GridData();
			data.grabExcessHorizontalSpace = false;
			data.grabExcessVerticalSpace = false;
			editLookupButton.setLayoutData(data);
			editLookupButton.setText(Messages.SourceNotFound_5);
			if (config != null) { // enable button for launch path mapping
									// solution
				editLookupButton.setEnabled(true);
				editLookupButton.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent evt) {
						try {
							editSourceLookupPath();
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				});
				editLookupButton.setData(UID_KEY, UID_EDIT_LOOKUP_BUTTON);
			} else if (resultsConfig != null) { // disable button for result
												// config path mapping solution
				editLookupButton.setEnabled(false);
			}
		}
		{
			GridData data;

			doNotPromptAgainButton = new Button(parent, SWT.CHECK);
			data = new GridData();
			data.grabExcessHorizontalSpace = false;
			data.grabExcessVerticalSpace = false;
			doNotPromptAgainButton.setLayoutData(data);
			doNotPromptAgainButton
					.setText(Messages.SourceNotFound_do_not_locate_file_again);
			doNotPromptAgainButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent evt) {
					doNotPromptAgain = doNotPromptAgainButton.getSelection();
				}
			});
		}

	}

	protected void locateFile() {
		FileDialog dialog = new FileDialog(getShell(), SWT.NONE);
		Path missingPath = new Path(missingFile);

		dialog.setText(Messages.SourceNotFound_2);
		dialog.setFilterNames(new String[] { "*." + missingPath.getFileExtension() }); //$NON-NLS-1$
		dialog.setFileName(Utils.simplifiedFileName(missingFile, true));

		dialog.setFilterExtensions(new String[] { "*." + missingPath.getFileExtension() }); //$NON-NLS-1$
		String res = dialog.open();

		if (res != null) {
			if ((new File(res)).exists()) {
				newPath = new Path(res);
				if (newPath.lastSegment().equalsIgnoreCase(
						missingPath.lastSegment())) {
	
					if (missingPath.segmentCount() > 1) {
						int missingPathSegCount = missingPath.segmentCount() - 2;
						int newPathSegCount = newPath.segmentCount() - 2;
	
						while (missingPathSegCount >= 0 && newPathSegCount >= 0) {
							if (!newPath.segment(newPathSegCount).equalsIgnoreCase(
									missingPath.segment(missingPathSegCount))) {
								break;
							}
							newPathSegCount--;
							missingPathSegCount--;
						}
						IPath compPath = missingPath.removeLastSegments(missingPath
								.segmentCount() - missingPathSegCount - 1);
						IPath newSourcePath = newPath.removeLastSegments(newPath
								.segmentCount() - newPathSegCount - 1);
	
						try {
							if (config != null) {
								SASourceLookupUtils.addSourceMappingToLaunch(
										config, compPath, newSourcePath);
							} else if (resultsConfig != null) {
								SASourceLookupUtils
										.addSourceMappingToResultsConfig(
												resultsConfig, missingPath, newPath);
							}
							errorMessage.setText(CommonConstants.EMPTY_STRING);
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				}
			} else {
				errorMessage.setText(Messages.SourceNotFound_3);
			}
		}
	}

	/**
	 * Edits the source lookup path associated with the active launch config
	 * copy. After the path is edited, source lookup is performed again and this
	 * editor is closed.
	 */
	protected void editSourceLookupPath() throws CoreException {
		if (config != null) {
			ISourceLookupDirector director = SASourceLookupUtils
					.getSourceLookupDirector(config);

			final SourceLookupDialog dialog = new SourceLookupDialog(
					getShell(), director);

			if (dialog.open() == Window.OK) {
				LaunchConfigUtils.updateLaunchConfigurationFromDirector(config,
						director);
			}
		} 
	}

	/**
	 * @return true if the "Do Not Locate..." checkbox has ben checked
	 */
	public boolean getDoNotPromptAgain() {
		return doNotPromptAgain;
	}
	
	public IPath getNewPath() {
		return newPath;
	}
}
