/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * Copyright 2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *******************************************************************************/

package com.freescale.system.browser.epl;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.freescale.system.browser.epl.TabbedWidgetFactoryEpl;


/**
 * Composite responsible for drawing the tabbed sheet page. 
 * Uses a <code>TabbedWidgetFactory</code> to create the widgets. 
 */
public class TabbedCompositeEpl extends Composite {

	private TabbedWidgetFactoryEpl fFactory;

	private Composite fMainComposite;

	private Composite fLeftComposite;

	private ScrolledComposite fScrolledComposite;

	private Composite fTabComposite;

	private TabbedTitleEpl fTitle;

	private TabbedListEpl fListComposite;

	private boolean fGainedFocus;

	private boolean fDisplayTitle;

	private Label fOsBannerTextLabel;

	private Label fOsBannerImageLabel;

	/**
	 * Constructor for a TabbedComposite
	 * 
	 * @param parent
	 *            the parent widget.
	 * @param fFactory
	 *            the widget factory.
	 * @param displayTitle
	 *            if <code>true</code>, then the title bar will be displayed.
	 */
	public TabbedCompositeEpl(Composite parent,
			TabbedWidgetFactoryEpl factory, boolean displayTitle) {
		super(parent, SWT.NO_FOCUS);
		fFactory = factory;
		fDisplayTitle = displayTitle;

		createMainComposite();
	}

	/**
	 * Get the tabbed list, which is the list of tabs on the left hand
	 * side of this composite.
	 * 
	 * @return the tabbed property list.
	 */
	public TabbedListEpl getList() {
		return fListComposite;
	}

	/**
	 * Get the tabbed property title bar.
	 * 
	 * @return the tabbed property title bar or <code>null</code> if not used.
	 */
	public TabbedTitleEpl getTitle() {
		return fTitle;
	}

	/**
	 * Get the tab composite where sections display their property contents.
	 * 
	 * @return the tab composite.
	 */
	public Composite getTabComposite() {
		return fTabComposite;
	}

	/**
	 * Get the scrolled composite which surrounds the title bar and tab
	 * composite.
	 * 
	 * @return the scrolled composite.
	 */
	public ScrolledComposite getScrolledComposite() {
		return fScrolledComposite;
	}

	/**
	 * Clean up th widgets containe by this composite.
	 * 
	 * @see org.eclipse.swt.widgets.Widget#dispose()
	 */
	public void dispose() {
		fListComposite.dispose();
		if (fTitle != null) {
			fTitle.dispose();
		}
		super.dispose();
	}
	
	/**
	 * Create the main composite.
	 */
	protected void createMainComposite() {
		fMainComposite = fFactory.createComposite(this, SWT.NO_FOCUS);
		fMainComposite.setLayout(new FormLayout());
		FormData formData = new FormData();

		formData.left = new FormAttachment(0, 0);
		formData.right = new FormAttachment(100, 0);
		formData.top = new FormAttachment(0, 0);
		formData.bottom = new FormAttachment(100, 0);
		fMainComposite.setLayoutData(formData);

		createMainContents();

		fMainComposite.addFocusListener(new FocusListener() {

			public void focusGained(FocusEvent e) {
				fGainedFocus = true;
				fMainComposite.redraw();
			}

			public void focusLost(FocusEvent e) {
				fGainedFocus = false;
				fMainComposite.redraw();
			}
		});
		fMainComposite.addPaintListener(new PaintListener() {

			public void paintControl(PaintEvent e) {
				if (fGainedFocus) {
					Rectangle r = getClientArea();

					e.gc.drawFocus(0, 0, r.width, r.height);
				}
			}
		});
	}

	/**
	 * Create the contents in the main composite.
	 */
	protected void createMainContents() {
		// On the left side, we create a top area (the "banner") that contains 
		// the OS's name and icon, and beneath it we put the tab list.
		fLeftComposite = fFactory.createComposite(fMainComposite, SWT.NO_FOCUS);
		fLeftComposite.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true));
		GridLayout gridLayout = new GridLayout(); 
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		gridLayout.horizontalSpacing = 0;
		gridLayout.verticalSpacing = 0;
		fLeftComposite.setLayout(gridLayout);

		fScrolledComposite = fFactory.createScrolledComposite(fMainComposite, SWT.H_SCROLL | SWT.V_SCROLL | SWT.NO_FOCUS);
		fScrolledComposite.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true));
		fScrolledComposite.setLayout(new FormLayout());

		FormData formData = new FormData();

		formData.left = new FormAttachment(fLeftComposite, 0);
		formData.right = new FormAttachment(100, 0);
		formData.top = new FormAttachment(0, 0);
		formData.bottom = new FormAttachment(100, 0);
		fScrolledComposite.setLayoutData(formData);

		formData = new FormData();
		formData.left = new FormAttachment(0, 0);
		formData.right = new FormAttachment(fScrolledComposite, 0);
		formData.top = new FormAttachment(0, 0);
		formData.bottom = new FormAttachment(100, 0);
		fLeftComposite.setLayoutData(formData);

		// On the right is where we show the content of the active tab (with a 
		// scroll bar). At the top of it, we create a title area (TitleTab 
		// instance) which will show the label for the active debug context
		// as optionally provided to us by the tabbed page contributor.
		final Composite rightComposite = fFactory.createComposite(fScrolledComposite, SWT.NO_FOCUS);
		rightComposite.setLayoutData(new GridData());
		FormLayout formLayout = new FormLayout();
		formLayout.marginHeight = 0;
		formLayout.marginWidth = 0;
		rightComposite.setLayout(formLayout);

		fScrolledComposite.setContent(rightComposite);
		fScrolledComposite.setAlwaysShowScrollBars(false);
		fScrolledComposite.setExpandVertical(true);
		fScrolledComposite.setExpandHorizontal(true);

		Composite osBannerComposite = new Composite(fLeftComposite, SWT.NONE);
		osBannerComposite.setLayoutData(new GridData(GridData.FILL, GridData.BEGINNING, true, false));
		osBannerComposite.setLayout(new GridLayout(2, false));
		fOsBannerImageLabel = new Label(osBannerComposite, SWT.NONE);
		fOsBannerImageLabel.setLayoutData(new GridData(GridData.BEGINNING, GridData.BEGINNING, false, false));
		fOsBannerTextLabel = new Label(osBannerComposite, SWT.NONE);
		fOsBannerTextLabel.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true));
		fOsBannerTextLabel.setBackground(osBannerComposite.getBackground());
		
		fListComposite = new TabbedListEpl(fLeftComposite, fFactory);
		fListComposite.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true));

		if (fDisplayTitle) {
			fTitle = new TabbedTitleEpl(rightComposite, fFactory);
		}

		fTabComposite = fFactory.createComposite(rightComposite, SWT.NO_FOCUS);
		fTabComposite.setLayout(new FormLayout());
		fTabComposite.setBackground(osBannerComposite.getBackground());

		if (fDisplayTitle) {
			FormData data = new FormData();

			data.left = new FormAttachment(0, 0);
			data.right = new FormAttachment(100, 0);
			data.top = new FormAttachment(0, 0);
			fTitle.setLayoutData(data);
		}

		FormData data = new FormData();

		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		if (fDisplayTitle) {
			data.top = new FormAttachment(fTitle, 0);
		} else {
			data.top = new FormAttachment(0, 0);
		}
		data.bottom = new FormAttachment(100, 0);
		fTabComposite.setLayoutData(data);

		fListComposite.addPaintListener(new PaintListener() {

			public void paintControl(PaintEvent e) {
				Point leftSize = fLeftComposite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				FormData formData2 = new FormData();

				formData2.left = new FormAttachment(0, leftSize.x);
				formData2.right = new FormAttachment(100, 0);
				formData2.top = new FormAttachment(0, 0);
				formData2.bottom = new FormAttachment(100, 0);
				fScrolledComposite.setLayoutData(formData2);
				fMainComposite.layout(true);
			}
		});
	}

	/**
	 * Get the widget m_factory.
	 * 
	 * @return the widget factory.
	 */
	protected TabbedWidgetFactoryEpl getFactory() {
		return fFactory;
	}
	
	public void setOsNameAndImage(String name, Image image) {
		fOsBannerTextLabel.setText(name);
		fOsBannerImageLabel.setImage(image);
		fOsBannerTextLabel.getParent().layout(true, true);
	}
}
