/*******************************************************************************
 * Copyright (c) 2012 Andrew Gvozdev and others.
 * Copyright 2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 *
 * Contributors:
 *     Andrew Gvozdev - initial API and implementation
 *     IBM Corporation
 *******************************************************************************/
package com.freescale.s32ds.cross.cdt.utils.epl;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.ICommandLauncher;
import org.eclipse.cdt.core.resources.IConsole;
import org.eclipse.cdt.core.resources.RefreshScopeManager;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;

public class ExternalToolRunnerHelperEPL implements Closeable {

	private IProject project;
	private IConsole console;
	private OutputStream stdout = null;
	private OutputStream stderr = null;

	private boolean isStreamsOpen = false;
	private boolean isCancelled = false;

	private OutputStream consoleInfo = null;

	private long startTime = 0;
	private long endTime = 0;

	private ICommandLauncher launcher;
	private IPath buildCommand;
	private String[] args;
	private IPath pathToWorkingDirectory;
	private String[] environment;

	/**
	 * Constructor.
	 */
	public ExternalToolRunnerHelperEPL(IProject project) {
		this.project = project;
		this.console = CCorePlugin.getDefault().getConsole(); // Get a build
																// console for
																// the project
		console.start(project);
	}

	/**
	 * Set parameters for the launch.
	 * 
	 * @param environment - String[] array of environment variables in format
	 *                    "var=value" suitable for using as "envp" with
	 *                    Runtime.exec(String[] cmdarray, String[] envp, File dir)
	 */
	public void setLaunchParameters(ICommandLauncher launcher, IPath buildCommand, String[] args,
			IPath pathToWorkingDirectory, String[] environment) {
		this.launcher = launcher;
		launcher.setProject(project);
		// Print the command for visual interaction.
		launcher.showCommand(true);

		this.buildCommand = buildCommand;
		this.args = args;
		this.pathToWorkingDirectory = pathToWorkingDirectory;
		this.environment = environment;
	}

	/**
	 * Open and set up streams for use by {@link ExternalToolRunnerHelperEPL}. This
	 * must be followed by {@link #close()} to close the streams. Use try...finally
	 * for that.
	 *
	 * @param outputStream - output stream
	 * @throws CoreException
	 */
	public void prepareStreams(OutputStream outputStream) throws CoreException {
		this.isStreamsOpen = true;
		this.stdout = outputStream;
		this.stderr = this.console.getErrorStream();
	}

	/**
	 * Launch external tool and process console output.
	 *
	 * @param monitor - progress monitor in the initial state where
	 *                {@link IProgressMonitor#beginTask(String, int)} has not been
	 *                called yet.
	 * @throws CoreException
	 * @throws IOException
	 */
	public int execute(IProgressMonitor monitor) throws CoreException, IOException {
		int status = ICommandLauncher.ILLEGAL_COMMAND;

		String errMsg = null;
		Process process = launcher.execute(buildCommand, args, environment, pathToWorkingDirectory, monitor);
		if (process != null) {
			try {
				// Close the input of the Process explicitly.
				// We will never write to it.
				process.getOutputStream().close();
			} catch (IOException e) {
			}
		} else {
			errMsg = launcher.getErrorMessage();
		}

		status = launcher.waitAndRead(stdout, console.getOutputStream(), monitor);
		if (status != ICommandLauncher.OK) {
			errMsg = launcher.getErrorMessage();
		}

		if (errMsg != null && !errMsg.isEmpty()) {
			stderr.write(errMsg.getBytes(StandardCharsets.UTF_8.name()));
		}

		return status;
	}

	/**
	 * Print the specified greeting to the console. Note that start time of the
	 * build is recorded by this method.
	 *
	 * This method may open an Info stream which must be closed by call to
	 * {@link #goodbye()} after all informational messages are printed.
	 */
	public void greeting(String msg) {
		startTime = System.currentTimeMillis();
		if (consoleInfo == null) {
			try {
				consoleInfo = console.getInfoStream();
			} catch (CoreException e) {
				CCorePlugin.log(e);
			}
		}
		if (consoleInfo != null) {
			toConsole(ExternalToolRunnerHelperEPL.timeStamp(startTime) + "**** " + msg + " ****"); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	/**
	 * Close all streams except console Info stream which is handled by
	 * {@link #greeting(String)}/{@link #goodbye()}.
	 */
	@Override
	public void close() throws IOException {
		if (!isStreamsOpen)
			return;

		try {
			if (stdout != null)
				stdout.close();
		} catch (Exception e) {
			CCorePlugin.log(e);
		} finally {
			stdout = null;
			try {
				if (stderr != null)
					stderr.close();
			} catch (Exception e) {
				CCorePlugin.log(e);
			} finally {
				stderr = null;
			}
		}
		isStreamsOpen = false;
	}

	/**
	 * Print a standard footer to the console and close Info stream (must be open
	 * with one of {@link #greeting(String)} calls). That prints duration of the
	 * build determined by start time recorded in {@link #greeting(String)}.
	 *
	 * <br>
	 * <strong>Important: {@link #close()} the streams BEFORE calling this method to
	 * properly flush all outputs</strong>
	 */
	public void goodbye(String msg) {
		Assert.isTrue(startTime != 0, "Start time must be set before calling this method."); //$NON-NLS-1$
		Assert.isTrue(consoleInfo != null,
				"consoleInfo must be open with greetings(...) call before using this method."); //$NON-NLS-1$

		endTime = System.currentTimeMillis();
		String duration = durationToString(endTime - startTime);
//		String msg = isCancelled ? CCorePlugin.getFormattedString("PreprocessRunnerHelper.preprocessCancelled", duration) //$NON-NLS-1$
//		: CCorePlugin.getFormattedString("PreprocessRunnerHelper.preprocessFinished", duration); //$NON-NLS-1$
		String newMsg = isCancelled ? new String(msg + " Cancelled (took " + duration.toString() + ")")
				: new String(msg + " Finished (took " + duration.toString() + ")"); //$NON-NLS-1$
		String goodbye = '\n' + timeStamp(endTime) + newMsg + '\n';

		try {
			toConsole(goodbye);
		} finally {
			try {
				consoleInfo.close();
			} catch (Exception e) {
				CCorePlugin.log(e);
			} finally {
				consoleInfo = null;
			}
		}
	}

	/**
	 * Print a message to the console info output. Note that this message is colored
	 * with the color assigned to "Info" stream.
	 * 
	 * @param msg - message to print.
	 */
	private void toConsole(String msg) {
		Assert.isNotNull(console, "Streams must be created and connected before calling this method"); //$NON-NLS-1$
		try {
			consoleInfo.write((msg + "\n").getBytes(StandardCharsets.UTF_8.name())); //$NON-NLS-1$
		} catch (Exception e) {
			CCorePlugin.log(e);
		}
	}

	/**
	 * Supply timestamp to prefix informational messages.
	 */
	@SuppressWarnings("nls")
	private static String timeStamp(long time) {
		return new SimpleDateFormat("HH:mm:ss").format(new Date(time)) + " ";
	}

	/**
	 * Convert duration to human friendly format.
	 */
	@SuppressWarnings("nls")
	private static String durationToString(long duration) {
		String result = "";
		long days = TimeUnit.MILLISECONDS.toDays(duration);
		if (days > 0) {
			result += days + "d,";
		}
		long hours = TimeUnit.MILLISECONDS.toHours(duration) % 24;
		if (hours > 0) {
			result += hours + "h:";
		}
		long minutes = TimeUnit.MILLISECONDS.toMinutes(duration) % 60;
		if (minutes > 0) {
			result += minutes + "m:";
		}
		long seconds = TimeUnit.MILLISECONDS.toSeconds(duration) % 60;
		if (seconds > 0) {
			result += seconds + "s.";
		}
		long milliseconds = TimeUnit.MILLISECONDS.toMillis(duration) % 1000;
		result += milliseconds + "ms";

		return result;
	}

	/**
	 * Refresh project in the workspace.
	 *
	 * @param configName - the configuration to refresh
	 */
	public void refreshProject(String configName) {
		try {
			// Do not allow the cancel of the refresh, since the builder is external
			// to Eclipse, files may have been created/modified and we will be out-of-sync.
			// The caveat is for huge projects, it may take sometimes at every build.
			// Use the refresh scope manager to refresh
			RefreshScopeManager refreshManager = RefreshScopeManager.getInstance();
			IWorkspaceRunnable runnable = refreshManager.getRefreshRunnable(project, configName);
			ResourcesPlugin.getWorkspace().run(runnable, null, IWorkspace.AVOID_UPDATE, null);
		} catch (CoreException e) {
		}
	}
}
