/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * Copyright 2017-2019 NXP
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Libor Ukropec (Freescale) - code customized for Processor Expert
 *     Lukas Tyc (NXP) - Add feature(method createAwtColorFromSwtRGB)
 *                     - In method createTextImage add parameter to set foreground text color
 *******************************************************************************/
package com.nxp.swtools.derivative.swt;

/*
 * Convert between SWT Image and AWT BufferedImage
 *
 * For a list of all SWT example snippets see
 * http://www.eclipse.org/swt/snippets/
 */
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DirectColorModel;
import java.awt.image.FilteredImageSource;
import java.awt.image.ImageFilter;
import java.awt.image.ImageProducer;
import java.awt.image.IndexColorModel;
import java.awt.image.RGBImageFilter;
import java.awt.image.WritableRaster;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;

/**
 * Used for converting images between AWT and SWT.
 */
public class AWTConverter {

	/**
	 * Converts SWT image in a form of ImageData into AWT in a form of BufferedImage.
	 * @param data ImageData from SWT image (NonNull)
	 * @return AWT BufferedImage (NonNull)
	 */
	/*@NonNull*/
	public static BufferedImage convertToAWT(/*@NonNull*/ ImageData data) {
		assert (data != null);
		ColorModel colorModel = null;
		PaletteData palette = data.palette;
		if (palette.isDirect) {
			colorModel = new DirectColorModel(data.depth, palette.redMask, palette.greenMask, palette.blueMask);
			BufferedImage bufferedImage = new BufferedImage(colorModel, colorModel.createCompatibleWritableRaster(data.width, data.height), false, null);
			for (int y = 0; y < data.height; y++) {
				for (int x = 0; x < data.width; x++) {
					int pixel = data.getPixel(x, y);
					RGB rgb = palette.getRGB(pixel);
					bufferedImage.setRGB(x, y,  (rgb.red << 16) | (rgb.green << 8) | rgb.blue);
				}
			}
			assert (bufferedImage != null);
			return bufferedImage;
		} else {
			RGB[] rgbs = palette.getRGBs();
			byte[] red = new byte[rgbs.length];
			byte[] green = new byte[rgbs.length];
			byte[] blue = new byte[rgbs.length];
			for (int i = 0; i < rgbs.length; i++) {
				RGB rgb = rgbs[i];
				red[i] = (byte)rgb.red;
				green[i] = (byte)rgb.green;
				blue[i] = (byte)rgb.blue;
			}
			if (data.transparentPixel != -1) {
				colorModel = new IndexColorModel(data.depth, rgbs.length, red, green, blue, data.transparentPixel);
			} else {
				colorModel = new IndexColorModel(data.depth, rgbs.length, red, green, blue);
			}
			BufferedImage bufferedImage = new BufferedImage(colorModel, colorModel.createCompatibleWritableRaster(data.width, data.height), false, null);
			WritableRaster raster = bufferedImage.getRaster();
			int[] pixelArray = new int[1];
			for (int y = 0; y < data.height; y++) {
				for (int x = 0; x < data.width; x++) {
					int pixel = data.getPixel(x, y);
					pixelArray[0] = pixel;
					raster.setPixel(x, y, pixelArray);
				}
			}
			assert (data != null);
			return bufferedImage;
		}
	}

	/**
	 * Converts AWT in a form of BufferedImage into SWT image in a form of ImageData.
	 * @param bufferedImage from AWT (NonNull)
	 * @return SWT ImageData
	 */
	/*@Nullable*/
	public static ImageData convertToSWT(/*@NonNull*/ BufferedImage bufferedImage) {
		assert (bufferedImage != null);
		if (bufferedImage.getColorModel() instanceof DirectColorModel) {
			DirectColorModel colorModel = (DirectColorModel)bufferedImage.getColorModel();
			PaletteData palette = new PaletteData(colorModel.getRedMask(), colorModel.getGreenMask(), colorModel.getBlueMask());
			ImageData data = new ImageData(bufferedImage.getWidth(), bufferedImage.getHeight(), colorModel.getPixelSize(), palette);
			for (int y = 0; y < data.height; y++) {
				for (int x = 0; x < data.width; x++) {
					int rgb = bufferedImage.getRGB(x, y);
					int pixel = palette.getPixel(new RGB((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF));
					data.setPixel(x, y, pixel);
					if (colorModel.hasAlpha()) {
						data.setAlpha(x, y, (rgb >> 24) & 0xFF);
					}
				}
			}
			return data;
		} else if (bufferedImage.getColorModel() instanceof IndexColorModel) {
			IndexColorModel colorModel = (IndexColorModel)bufferedImage.getColorModel();
			int size = colorModel.getMapSize();
			byte[] reds = new byte[size];
			byte[] greens = new byte[size];
			byte[] blues = new byte[size];
			colorModel.getReds(reds);
			colorModel.getGreens(greens);
			colorModel.getBlues(blues);
			RGB[] rgbs = new RGB[size];
			for (int i = 0; i < rgbs.length; i++) {
				rgbs[i] = new RGB(reds[i] & 0xFF, greens[i] & 0xFF, blues[i] & 0xFF);
			}
			PaletteData palette = new PaletteData(rgbs);
			ImageData data = new ImageData(bufferedImage.getWidth(), bufferedImage.getHeight(), colorModel.getPixelSize(), palette);
			data.transparentPixel = colorModel.getTransparentPixel();
			WritableRaster raster = bufferedImage.getRaster();
			int[] pixelArray = new int[1];
			for (int y = 0; y < data.height; y++) {
				for (int x = 0; x < data.width; x++) {
					raster.getPixel(x, y, pixelArray);
					data.setPixel(x, y, pixelArray[0]);
				}
			}
			return data;
		}
		return null;
	}

	/**
	 * Creates a text image
	 * @param text to be paint
	 * @param width image width
	 * @param height image height
	 * @param backgroundColor background color (NonNull)
	 * @param foregroundColor foreground color (NonNull)
	 * @return BufferedImage
	 */
	/*@NonNull*/
	public static BufferedImage createTextImage(/*@Nullable*/ String text, int width, int height,/*@NonNull*/ Color backgroundColor,/*@NonNull*/ Color foregroundColor) {
		assert (backgroundColor != null);
		assert (foregroundColor != null);
		return createTextImage(text, width, height, backgroundColor, foregroundColor, new Font("SansSerif", SWT.None, 10)); //$NON-NLS-1$
	}

	/**
	 * Creates a text image
	 * @param text to be paint
	 * @param width image width
	 * @param height image height
	 * @param backgroundColor background color (NonNull)
	 * @param foregroundColor foreground color (NonNull)
	 * @param font to create text image with (NonNull)
	 * @return BufferedImage
	 */
	/*@NonNull */
	public static BufferedImage createTextImage(/*@Nullable*/ String text, int width, int height,/*@NonNull*/ Color backgroundColor,/*@NonNull*/ Color foregroundColor, /*@NonNull*/ Font font) {
		assert (backgroundColor != null);
		assert (foregroundColor != null);
		assert (font != null);
		BufferedImage bufferedImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		Graphics graphics = bufferedImage.getGraphics();
		graphics.setColor(backgroundColor);
		graphics.fillRect(0, 0, width, height);
		graphics.setColor(foregroundColor);
		graphics.setFont(font);
		if (graphics instanceof Graphics2D) {
			Graphics2D g2d = (Graphics2D) graphics;
			g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
		}
		graphics.drawString(text, 0, font.getSize());
		assert (bufferedImage != null);
		return bufferedImage;
	}

	/**
	 * Convert input image to BufferedImage
	 * @param image to convert it to BufferedImage (NonNull)
	 * @return BufferedImage instance
	 */
	private static /*@NonNull*/ BufferedImage imageToBufferedImage(/*@NonNull*/ Image image) {
		assert (image != null);
		BufferedImage bufferedImage = new BufferedImage(image.getWidth(null), image.getHeight(null), BufferedImage.TYPE_INT_ARGB);
		Graphics2D g2 = bufferedImage.createGraphics();
		g2.drawImage(image, 0, 0, null);
		g2.dispose();
		return bufferedImage;
	}

	/**
	 * Creates image which is the same as given image except that pixels of the input color are set as transparent
	 * @param image (NonNull)
	 * @param color which will be set as transparent (NonNull)
	 * @return image which is the same as given image except that pixels of the input color are set as transparent
	 */
	private static /*@Nullable*/ Image makeColorTransparent(/*@NonNull*/ BufferedImage image, final /*@NonNull*/ Color color) {
		assert (image != null);
		assert (color != null);
		ImageFilter filter = new RGBImageFilter() {
			// the color we are looking for, Alpha bits are set to opaque
			public int markerRGB = color.getRGB() | 0xFF000000;

			/* (non-Javadoc)
			 * @see java.awt.image.RGBImageFilter#filterRGB(int, int, int)
			 */
			@Override
			public final int filterRGB(int x, int y, int rgb) {
				if ((rgb | 0xFF000000) == markerRGB) {
					// Mark the alpha bits as zero - transparent
					return 0x00FFFFFF & rgb;
				} else {
					return rgb;
				}
			}
		};

		ImageProducer ip = new FilteredImageSource(image.getSource(), filter);
		return Toolkit.getDefaultToolkit().createImage(ip);
	}

	/**
	 * Creates new instance of BufferedImage which is the same as given image except that pixels of input color are set as transparent
	 * @param image to set pixels of input color as transparent (NonNull)
	 * @param color which will be set as transparent (NonNull)
	 * @return new instance of BufferedImage where pixels of input color are transparent
	 */
	public static /*@NonNull*/ BufferedImage createImageWithTransparentColor(/*@NonNull*/ BufferedImage image, /*@NonNull*/ Color color) {
		assert (image != null);
		assert (color != null);
		Image img = makeColorTransparent(image, color);
		if (img == null) {
			return image;
		}
		return imageToBufferedImage(img);
	}

	/**
	 * @param colorRGB color RGB model
	 * @return AWT color representing given RGB model
	 */
	/*@NonNull*/
	public static Color createAwtColorFromSwtRGB(/*@NonNull*/ RGB colorRGB) {
		return new Color(colorRGB.red, colorRGB.green, colorRGB.blue);
	}

}
