"use strict";

/* Copyright 2016: Freescale Semiconductor, Inc. All Rights Reserved. */
/* Copyright 2017-2018 NXP */
/* Script to export registers data used for Pins tool v3 */

/** Example script for export registers content - user modifications possible. */

// User defined name for exported file name used in CLI processing
var exportedRegFileName = "exported_registers.h"                 // generated exported registers content file name

/**
 * Gets reset value of register from database
 * dbReg - reference to database register
 * returns reset value
 */
function getResetValue(dbReg) {
  var value = java.math.BigInteger.valueOf(0);
  var dbBitFields = dbReg.getBitFields();
  for (var fi = 0; fi < dbBitFields.length; fi++) {
    var dbBitField = dbBitFields[fi];
    value = getBitwiseOr(value, getBitwiseShiftLeft(dbBitField.getResetValue().doubleValue(), dbBitField.getOffset()));
  }
  return (value);
}

/**
 * Gets bitwise and. Native bitwise operation in JavaScript works only with 31-bit numbers because of implicit conversion to signed 32-bit integer.
 * a - value 1
 * b - value 2
 * returns result of bitwise AND
 */
function getBitwiseAnd(a,b) {
  if (a < 0 || b < 0) {
    PExOut.log('export_registers.js: One of operands (' + a + ', ' + b + ') is negative number.');
  }
  var result = 0;
  var mask = 0xFFFF;
  var divisor = mask + 1;
  var i = 0;
  while ((a != 0) && (b != 0)) {
    var r = ((a & mask) & (b & mask));
    for (var j = 0; j < i; j++) {
      r *= divisor;
    }
    result += r;
    a = Math.floor(a / divisor);
    b = Math.floor(b / divisor);
    i++;
  }
  return result;
}

/**
 * Gets bitwise or. Native bitwise operation in JavaScript works only with 31-bit numbers because of implicit conversion to signed 32-bit integer.
 * a - value 1
 * b - value 2
 * returns result of bitwise OR
 */
function getBitwiseOr(a,b) {
  if (a < 0 || b < 0) {
    PExOut.log('export_registers.js: One of operands (' + a + ', ' + b + ') is negative number.');
  }
  var result = 0;
  var mask = 0xFFFF;
  var divisor = mask + 1;
  var i = 0;
  while ((a != 0) || (b != 0)) {
    var r = ((a & mask) | (b & mask));
    for (var j = 0; j < i; j++) {
      r *= divisor;
    }
    result += r;
    a = Math.floor(a / divisor);
    b = Math.floor(b / divisor);
    i++;
  }
  return result;
}

/**
 * Gets bitwise xor. Native bitwise operation in JavaScript works only with 31-bit numbers because of implicit conversion to signed 32-bit integer.
 * a - value 1
 * b - value 2
 * returns result of bitwise XOR
 */
function getBitwiseXor(a,b) {
  if (a < 0 || b < 0) {
    PExOut.log('export_registers.js: One of operands (' + a + ', ' + b + ') is negative number.');
  }
  var result = 0;
  var mask = 0xFFFF;
  var divisor = mask + 1;
  var i = 0;
  while ((a != 0) || (b != 0)) {
    var r = ((a & mask) ^ (b & mask));
    for (var j = 0; j < i; j++) {
      r *= divisor;
    }
    result += r;
    a = Math.floor(a / divisor);
    b = Math.floor(b / divisor);
    i++;
  }
  return result;
}

/**
 * Gets value shifted left by a number. Native bitwise operation in JavaScript works only with 31-bit numbers because of implicit conversion to signed 32-bit integer.
 * a - value which is shifted left by n
 * n - number
 * returns shifted value
 */
function getBitwiseShiftLeft(a, n) {
  if (a < 0 || n < 0) {
    PExOut.log('export_registers.js: One of operands (' + a + ', ' + n + ') is negative number.');
  }
 return (a * Math.pow(2, n));
}

/**
 * Gets value converted to hex format
 * bitFieldValueX - number for conversion
 * returns converted value (e.g. 0x0Fu)
 */
function getNumberConvertedToHex(bitFieldValueX) {
  var hexNumberPart = bitFieldValueX.toString(16).toUpperCase();
  while(hexNumberPart.length < 8) {
    hexNumberPart = "0" + hexNumberPart;
  }
  return ("0x" + hexNumberPart);
}

/**
 *  Gets string aligned to given column at the line using added spaces to the end
 *  str - string to be aligned
 *  col - number of column to be aligned for
 */
function getStringAlignedToColumn(str, col) {
  while (str.length < col) {
    str += ' '; // align with space
  }
  return str;
}

/**
 * Prints string array
 * array - string array (one item on one line)
 * stringAtBeginningOfEachLine - string which is placed at the beginning of each line
 * stringAtBeginningOfFirstLine - string which is placed at the beginning of the first line
 * return value - no data; just printing into output
 */
function printStringArray(array,stringAtBeginningOfEachLine,stringAtBeginningOfFirstLine) {
  if (array.length > 0) {
    for (var lineIndex = 0; lineIndex < array.length; lineIndex++) {
      var line = array[lineIndex];
      if (lineIndex == 0) {
        line = stringAtBeginningOfFirstLine + line;
      }
      line = stringAtBeginningOfEachLine + line;
      PExOut.gen(line);
    }
  }
}

// Register database Object
var registerDatabaseObject = PExProcessor.getRegistersDB();
var allComponents = PExProject.getAllComponents();               // Each component represents one user function
//
var peripherals = null;
if (registerDatabaseObject != null) {
    peripherals = registerDatabaseObject.getPeripherals();
}

//
var componentCoreIds = new Array();                              // Core ID's array. It says core ID for each UI user-defined function
var configurationStrategies = new Array();                       // Configuration strategies for getting right register configuration for given user function
var identifierPrefixes = new Array();                            // Array of identifier prefixes of each UI user-defined function
var textOptions =  null;
//
for (var pc = 0; pc < allComponents.length; pc++) {
    var funcNameItem = allComponents[pc].findItemBySymbol("FunctionName");
    if (funcNameItem != null) {
        configurationStrategies[pc] = funcNameItem.getText();
        //PExOut.log('export_registers.js: funcNameItem ' + funcNameItem.getText());
    }
    textOptions = allComponents[pc].getComponentOptions();
    if (textOptions != null) {
        var options = JSON.parse(textOptions);
        componentCoreIds.push(options["coreID"]);
        //PExOut.log('export_registers.js: Component options: ' + JSON.stringify(options));
        var identifierPrefix = options["prefix"];
        if (identifierPrefix == null) {
            identifierPrefixes.push((configurationStrategies[pc] + '_').toUpperCase());
        }
        else {
            identifierPrefixes.push(identifierPrefix);
        }
    }
}
var configurationStrategy = null;                                //

// Get multi-core information
var coreIds = null;
var coreList = PExProcessor.getCoresList();
if (coreList != null) {
    coreIds = Object.keys(JSON.parse(coreList))
}

// Process all cores with non-empty configurations - filtering for cores with an assigned function(s)
var notEmptyCoreIds = new Array();
for (var coreIndex in coreIds) {
    var coreId = coreIds[coreIndex];
    if (componentCoreIds.indexOf(coreId) > -1) {
        notEmptyCoreIds.push(coreId);
    }
}

var registerList = null;                                         // List of registers for pin muxing and pin functional properties; register is defined by name, clear mask and set mask
var clockGateRegisterList = null;                                // List of registers for clock gate enable; register is defined by name, clear mask and set mask

///////////////////////////////////////////////////////////////
// Functions definition ...

/**
 * Analyzes register settings for pin muxing and functional pin properties (electrical features) from the tool.
 * It iterates through all configuration steps and registers and search all register modifications for export.
 * pc - index of the configuration/function (internally component)
 * configurationSteps - array of configuration register steps
 */
function analyzeRegisterConfigurationSequence(pc,configurationSteps) {
    // Analyzing register configuration data
    registerList[pc] = new Object();
    var configurationRegisterList = registerList[pc];  // list of all registers modfied in current sequence
    clockGateRegisterList[pc] = new Object();
    var configurationClockGateRegisterList = clockGateRegisterList[pc];
    for (var si = 0; si < configurationSteps.length; si++) {  // go through all configuration steps
        var configurationRegisters = configurationSteps[si].getRegistersConfigurations(); // get register configurations
        for (var ri = 0; ri < configurationRegisters.length; ri++) {  // analyze all registers
            var registerName = configurationRegisters[ri].getRegisterName();
            var registerClrMask = configurationRegisters[ri].getClrRegMask();
            var registerSetMask = configurationRegisters[ri].getSetRegMask();
            //
            var dbPeripheral = registerDatabaseObject.getPeripheralByFullName(registerName);
            var dbRegister = registerDatabaseObject.getRegisterByFullName(registerName);
            //
            if ((dbPeripheral != null) && (dbRegister != null)) {
                /* ***** Preparation of pin muxing, functional property register configuration lists ***** */
                configurationRegisterList[registerName] = new Object();
                configurationRegisterList[registerName].regDb = dbRegister;
                configurationRegisterList[registerName].registerName = configurationRegisters[ri].getRegisterName();
                configurationRegisterList[registerName].registerOffset = dbRegister.getOffset();
                configurationRegisterList[registerName].registerClrMask = registerClrMask;
                configurationRegisterList[registerName].registerSetMask = registerSetMask;
                //PExOut.log('export_registers.js: ' + registerName + ' @offset: ' + dbRegister.getOffset() + ', clrMask=: ' + registerClrMask + '; setMask=: ' + registerSetMask);

                /* ***** Preparation of clock gate configuration lists ***** */
                var clockGate = registerDatabaseObject.getClockGate(dbPeripheral, dbRegister);
                if (clockGate != null) {
                    var clockGateRegisterFullName = clockGate.getControlPeripheralName() + '_' + clockGate.getControlRegisterName();
                    var clockGateField = clockGate.getControlBitField();
                    var clockGateStates = clockGate.getStates();
                    var clockGateDbRegister = registerDatabaseObject.getRegisterByFullName(clockGateRegisterFullName);
                    if ((clockGateField != null) && (clockGateDbRegister != null)) {
                        for (var gindex = 0; gindex < clockGateStates.length; gindex++) {
                            var aGState = clockGateStates[gindex];
                            var clockGateShiftedValue = getBitwiseShiftLeft(aGState.getValue(), clockGateField.getOffset());
                            if (aGState.getName() == "enabled") {
                                var reg = configurationRegisterList[clockGateRegisterFullName];
                                if (reg == null) {
                                    //PExOut.log('export_registers.js: clockGateRegisterFullName ' + clockGateRegisterFullName + ' is not defined yet - create new Object() first.');
                                    configurationRegisterList[clockGateRegisterFullName] = new Object(); // create new clockGateRegisterFullName register configuration
                                    configurationRegisterList[clockGateRegisterFullName].regDb = clockGateDbRegister;
                                    configurationRegisterList[clockGateRegisterFullName].registerName = clockGateRegisterFullName;
                                    configurationRegisterList[clockGateRegisterFullName].registerOffset = clockGateDbRegister.getOffset();
                                    configurationRegisterList[clockGateRegisterFullName].registerClrMask = getBitwiseXor(clockGateShiftedValue, clockGateField.getRegisterMask().doubleValue());
                                    configurationRegisterList[clockGateRegisterFullName].registerSetMask = clockGateShiftedValue;
                                    //PExOut.log('export_registers.js: clockGateRegisterFullName ' + clockGateRegisterFullName + ' @offset: ' + clockGateDbRegister.getOffset() + ', clrMask=: ' + getBitwiseXor(clockGateShiftedValue, clockGateField.getRegisterMask().doubleValue()) + '; setMask=: ' + clockGateShiftedValue);
                                }
                                else {
                                    //PExOut.log('export_registers.js: clockGateRegisterFullName ' + clockGateRegisterFullName + ' is already existing, update the init values.');
                                    var clrMask = getBitwiseOr(reg.registerClrMask, getBitwiseXor(clockGateShiftedValue, clockGateField.getRegisterMask().doubleValue()))
                                    var setMask = getBitwiseOr(reg.registerSetMask, clockGateShiftedValue);
                                    configurationRegisterList[clockGateRegisterFullName].registerClrMask = clrMask;
                                    configurationRegisterList[clockGateRegisterFullName].registerSetMask = setMask;
                                    //PExOut.log('export_registers.js: clockGateRegisterFullName ' + reg.registerName + ' @offset: ' + reg.registerOffset + ', =clrMask=: ' + getBitwiseOr(reg.registerClrMask, getBitwiseXor(clockGateShiftedValue, clockGateField.getRegisterMask().doubleValue())) + '; =setMask=: ' + getBitwiseOr(reg.registerSetMask, clockGateShiftedValue));
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

/**
 * Processes register configuration sequence for all pin function configurations.
 * coreId - processor core Id
 */
function processRegisterConfigurationSequence(coreId) {
    for (var pc = 0; pc < allComponents.length; pc++) {  // go through all components - UI user-defined functions
        if (coreId == componentCoreIds[pc]) {  // process the function if assigned for current coreId
            var configurationSteps = PExProcessor.getRegisterConfigurationSequence(true, configurationStrategies[pc], null);
            analyzeRegisterConfigurationSequence(pc,configurationSteps);
        }
    }
}

/**
 * Enrolls one register configuration set of register modifications given by the pre-analyzed configuration sequence.
 * configuration - given configuration from where the register modifications are need to be enrolled for export
 */
function enrolOneRegisterConfiguration(configuration) {
    var configurationRegisterList = registerList[configuration];
    for (var ri in configurationRegisterList) {
        var reg = configurationRegisterList[ri];
        if (reg != null) {
            var regMask = 0xFFFFFFFF;
            // enrol one register config data for exporting register content
            //PExOut.log('export_registers.js: ' + reg.registerName + ' @offset: ' + reg.registerOffset + ', clrMask=: ' + reg.registerClrMask + '; setMask=: ' + reg.registerSetMask);
            var regNameDef = identifierPrefixes[configuration] + reg.registerName + '_VALUE'; // FIXME: optionally prefixed according to script bool setting
            //var regNameVal = getBitwiseAnd(getResetValue(reg.regDb), getBitwiseOr(getBitwiseXor(regMask,reg.registerClrMask), reg.registerSetMask));
            var regNameVal = (getResetValue(reg.regDb) & (regMask ^ reg.registerClrMask)) | reg.registerSetMask;
            var regNameTxt = '   /*!< Register name: ' + reg.registerName + ' */'; // FIXME: optionally commented according to script bool setting
            PExOut.gen(getStringAlignedToColumn('#define ' + regNameDef, 80) + getNumberConvertedToHex(regNameVal) + regNameTxt);
        }
    }
}

/**
 * Exports all register modifications
 */
function exportRegisters() {
    for (var coreIndex in notEmptyCoreIds) {
        registerList = new Object(); // will store all register objects for further export
        clockGateRegisterList = new Object(); // will store clock gate register objects for further export

        // get current core id to perform the export registers for
        var coreId = notEmptyCoreIds[coreIndex];

        /** pre-process register configuration sequence content(s) */
        processRegisterConfigurationSequence(coreId);

        /** set 'temporary' output file required for script processing */
        //PExOut.setOutputFile(exportedRegFileName); // Note: the tool will then save the output file into file specified in the UI

        /** collect register modifications through all configurations defined in the tool UI */
        for (var pc = 0; pc < allComponents.length; pc++) {
            if (coreId == componentCoreIds[pc]) {
                PExOut.gen('');
                configurationStrategy = configurationStrategies[pc];
                PExOut.gen('/*FUNCTION**********************************************************************');
                PExOut.gen(' *');
                PExOut.gen(' * Function Name : ' + configurationStrategy);
                printStringArray(allComponents[pc].getDescription(), ' * ', 'Description   : ');
                PExOut.gen(' *');
                PExOut.gen(' *END**************************************************************************/');
                PExOut.gen('');
                //
                enrolOneRegisterConfiguration(pc); // enrol one register configuration as per all user-defined functions
            }
        }
    }
}


///////////////////////////////////////////////////////////////
// Export registers data ...
//

/** set 'temporary' output file required for script processing */
// Note: the UI tool will override the output file name into name specified by user in the UI dialog
PExOut.setOutputFile(exportedRegFileName); // change the 'exportedRegFileName' value in the beginning of the script if different output name for CLI is required

/** User-defined Pins Tool functions & register data export first ... */
PExOut.gen('/*******************************************************************************');
PExOut.gen(' * Export Registers - Modified Values ');
PExOut.gen(' ******************************************************************************/');
PExOut.gen('');

PExOut.gen('/*');
PExOut.gen(PExProject.getYamlState());                           // Print project status including current processor selection
PExOut.gen(' */');
//
exportRegisters();                                               // Export User-defined functions register modifications
PExOut.gen('');
PExOut.gen('');

/** And then (optionally?) export register after reset values ... */
// Note: For LPC/Kinetis do not include after reset register values within exported registers content file.

PExOut.gen('');
PExOut.gen('/*******************************************************************************');
PExOut.gen(' * EOF');
PExOut.gen(' ******************************************************************************/');


//eof.