/*******************************************************************************
 * Copyright (c) 2006-2011 Freescale Semiconductor.
 * Copyright 2022 NXP
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Freescale Semiconductor - initial API and implementation
 *     NXP - Dsf session mechanics fix
 *******************************************************************************/
package com.freescale.system.browser.ui;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.eclipse.cdt.dsf.datamodel.DMContexts;
import org.eclipse.cdt.dsf.datamodel.IDMContext;

import org.eclipse.cdt.dsf.debug.service.IRunControl.IExecutionDMContext;
import org.eclipse.cdt.dsf.gdb.launching.GdbLaunch;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.contexts.DebugContextEvent;
import org.eclipse.debug.ui.contexts.IDebugContextListener;
import org.eclipse.debug.ui.contexts.IDebugContextManager;
import org.eclipse.debug.ui.contexts.IDebugContextService;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.SubActionBars;
import org.eclipse.ui.part.IPageBookViewPage;
import org.eclipse.ui.part.IPageSite;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.part.PageSite;
import org.eclipse.ui.part.ViewPart;

import com.freescale.s32ds.debug.ka.KAPluginActivator;
import com.freescale.system.browser.internal.SystemBrowserActivator;
import com.freescale.system.browser.internal.ui.SystemContributor;
import com.freescale.system.browser.internal.ui.tabbed.ITabbedPage;
import com.freescale.system.browser.internal.ui.tabbed.ITabbedPageContributor;
import com.freescale.system.browser.internal.ui.tabbed.TabbedDefaultPage;
import com.freescale.system.browser.internal.ui.tabbed.TabbedSheetPage;
import com.freescale.system.browser.model.OSAwarenessData;

/**
 * The System Browser view. It is a multipage view using a
 * <code>org.eclipse.ui.part.PageBook</code> control to house the pages. We
 * maintain a collection of page records to keep track of all pages in the view
 * (the single visisble one and the rest--the nonvisible ones).
 * 
 * This is a singleton. Has related constants, move carefully.
 */
public class TabbedSystemView extends ViewPart {

	/**
	 * The page book. Every system contributor has a page (and only one) in the
	 * view. Each page takes up the entire view and so only one is shown at any
	 * point in time. We automatically activate the appropriate page given the
	 * active debug context (selection in the Debug view).
	 */
	private PageBook fBook = null;

	/**
	 * Our singleton instance.
	 */
	private static volatile TabbedSystemView sInstance;

	/**
	 * The page record for the default page (the "no info available" one). It
	 * shows when no system contributor is subject to being active.
	 */
	private PageRec fDefaultPageRec = null;

	/**
	 * Lock to create pages
	 */
	private Object createPageLock = new Object();

	/**
	 * Collection of pages in the view.
	 */
	private List<PageRec> fPageRecs = new ArrayList<>();

	/**
	 * Registered listeners that will react to key changes in the view.
	 * Currently, the only such notification is a "page was added" one.
	 */
	private ArrayList<IPropertyChangeListener> fPropertyListeners = new ArrayList<>();

	/**
	 * Map page contributor to page record.
	 */
	private Map<ITabbedPageContributor, PageRec> fMapContributorToRec = new HashMap<>();

	/**
	 * The page rec associated with the active page or null.
	 */
	private PageRec fActiveRec = null;

	private IDebugContextListener fDebugContextListener;
	private Object m_session;

	/**
	 * A data structure used to store the information about a single page within
	 * this multipage view.
	 */
	protected static class PageRec {
		public ITabbedPageContributor part;
		public IPageBookViewPage page;
		public SubActionBars subActionBars;

		public PageRec(ITabbedPageContributor part, IPageBookViewPage page) {
			this.part = part;
			this.page = page;
		}

		public void dispose() {
			part = null;
			page = null;
		}
	}

	/**
	 * Constructor. This is a singleton.
	 */
	public TabbedSystemView() {
		super();
	}

	/**
	 * Gets the singleton instance of this view
	 * 
	 * @return the instance
	 */
	public static TabbedSystemView getInstance() {
		if (null == sInstance) {
			sInstance = new TabbedSystemView();
		}
		return sInstance;
	}

	/**
	 * The <code>TabbedSystemView</code> implementation of this
	 * <code>IWorkbenchPart</code> method creates a <code>PageBook</code>
	 * control with its default page showing.
	 */
	@Override
	public void createPartControl(Composite parent) {
		// Create the page book.
		fBook = new PageBook(parent, SWT.NONE);
		// Create the default page (the "no system info available" one)
		TabbedDefaultPage defaultPage = new TabbedDefaultPage();
		providePageSite(defaultPage);
		defaultPage.createControl(fBook);
		fDefaultPageRec = new PageRec(null, defaultPage);
		preparePage(fDefaultPageRec);
		showPageRec(fDefaultPageRec);

		// listen to active debug context changes
		setupContextListener();
	}

	/**
	 * Get the CDT process associated with the active debug context and activate
	 * the page that supports it
	 */
	private void changeContext(final IExecutionDMContext newContext) {
		if (fActiveRec != null && fActiveRec.page instanceof ITabbedPage) {
			if (newContext != null) {
				OSAwarenessData newSessionData = new OSAwarenessData(newContext);
				((ITabbedPage) fActiveRec.page).setPageInput(newSessionData);
			} else {
				((ITabbedPage) fActiveRec.page).setPageInput(OSAwarenessData.getDummyData());
			}
		}
	}

	/**
	 * Gets the PageRec for the given tabbed page or creates one if it doesn't
	 * exist yet.
	 */
	public PageRec getOrCreatePageRec(final ITabbedPage page) {
		// special case the default "no system available" page
		if (page.equals(fDefaultPageRec.page)) {
			return fDefaultPageRec;
		}

		synchronized (createPageLock) {
			PageRec rec = getPageRec(page);
			ITabbedPageContributor contr = page.getContributor();
			if (rec == null && contr != null) {
				rec = createPage(contr);
			}
			return rec;
		}
	}

	/**
	 * Add a listener for key view events. Calling this with an
	 * already-registered listener simply no-ops
	 * 
	 * @param listener
	 *            the interested object
	 */
	public void addPropertyChangeListener(IPropertyChangeListener listener) {
		if (!fPropertyListeners.contains(listener)) {
			fPropertyListeners.add(listener);
		}
	}

	/**
	 * Remove a listener for key view events. Calling this with a listener that
	 * isn't registered is harmless.
	 * 
	 * @param listener
	 *            the no-longer interested object
	 */
	public void removePropertyChangeListener(IPropertyChangeListener listener) {
		fPropertyListeners.remove(listener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		// First set focus on the page book, in case the page doesn't properly
		// handle setFocus
		if (fBook != null && !fBook.isDisposed()) {
			fBook.setFocus();
		}
		// Then set focus on the actice page, if any
		if (fActiveRec != null) {
			fActiveRec.page.setFocus();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		// stop listening to debug context changes
		IDebugContextManager contextManager = DebugUITools.getDebugContextManager();
		IDebugContextService contextService = contextManager.getContextService(getSite().getWorkbenchWindow());
		contextService.removeDebugContextListener(fDebugContextListener);

		changeContext(null);

		// Deref all of the pages.
		fActiveRec = null;
		if (fDefaultPageRec != null) {
			// check for null since the default page may not have
			// been created (ex. perspective never visible)
			fDefaultPageRec.page.dispose();
			fDefaultPageRec.page = null;
			fDefaultPageRec = null;
		}

		// remove all pages
		if (fPageRecs != null) {
			removeSessionPagesData();
			fPageRecs = null;
		}

		fPropertyListeners = null;

		super.dispose();
	}

	/**
	 * Initializes the given page with a page site.
	 * 
	 * This needs to happen after the page is created but before creating its
	 * controls.
	 * 
	 * @param page
	 *            The page to initialize
	 */
	protected void providePageSite(IPageBookViewPage page) {
		try {
			page.init(new PageSite(getViewSite()));
		} catch (PartInitException exc) {
			SystemBrowserActivator.log(exc.getStatus());
		}
	}

	/**
	 * Make the specified page the active one in the view.
	 * 
	 * @param pageRec
	 *            the page record containing the page to make active
	 */
	protected void showPageRec(PageRec pageRec) {
		// If already showing do nothing
		if (fActiveRec == pageRec) {
			return;
		}

		// Deactivate action bars of old page
		if (fActiveRec != null) {
			fActiveRec.subActionBars.deactivate();
		}

		// Show new page.
		fActiveRec = pageRec;
		Control pageControl = fActiveRec.page.getControl();
		if (pageControl != null && !pageControl.isDisposed()) {
			// Verify that the page control is not disposed
			// If we are closing, it may have already been disposed
			fBook.showPage(pageControl);
			fActiveRec.subActionBars.activate();
			refreshGlobalActionHandlers();
			getViewSite().getActionBars().updateActionBars();
		}
	}

	/**
	 * Destroys the page and its record
	 */
	protected void doDestroyPage(ITabbedPageContributor part, PageRec rec) {
		((ITabbedPage) rec.page).dispose();
		rec.page = null;
		rec.dispose();
	}

	/**
	 * Accessor method. See fBook.
	 */
	protected PageBook getPageBook() {
		return fBook;
	}

	/**
	 * Returns the page record for the given contributor.
	 */
	protected PageRec getPageRec(ITabbedPageContributor contributor) {
		return fMapContributorToRec.get(contributor);
	}

	/**
	 * Returns the page record for the given page of this view.
	 * 
	 * @param page
	 *            the page
	 * @return the corresponding page record, or <code>null</code> if not found
	 */
	protected PageRec getPageRec(IPageBookViewPage page) {
		for (PageRec rec : fPageRecs) {
			if (rec.page.equals(page)) {
				return rec;
			}
		}
		return null;
	}

	/**
	 * Creates a page for a given contributor. Adds it to the pagebook but does
	 * not show it yet.
	 * 
	 * @param part
	 *            The contributor we are making a page for.
	 * @return the newly created page record
	 */
	private PageRec createPage(ITabbedPageContributor part) {
		PageRec rec = null;

		final ITabbedPage page = part.getTabbedSheetPage();
		if (page != null) {
			providePageSite(page);
			page.createControl(getPageBook());
			rec = new PageRec(part, page);

			preparePage(rec);
			fPageRecs.add(rec);
			fMapContributorToRec.put(part, rec);

			// Inform the listeners that a new page has just been created
			// Send them the entire collection so they know what we have
			List<?> clone = (List<?>) ((ArrayList<PageRec>) fPageRecs).clone();
			informListeners(fPropertyListeners, clone);
		} else {
			SystemBrowserActivator.log(IStatus.ERROR, "ITabbedPageContributor didn't provide a TabbedPage."); //$NON-NLS-1$
		}

		return rec;
	}

	/**
	 * Prepares the page in the given page record for use in this view.
	 * 
	 * @param rec
	 *            the Page Record that holds the page
	 */
	private void preparePage(PageRec rec) {
		IPageSite site = null;

		if (getPageRec(rec.page) == null) {
			site = rec.page.getSite();
			if (site == null) {
				site = new PageSite(getViewSite());
			}

			rec.subActionBars = (SubActionBars) site.getActionBars();
			rec.subActionBars.addPropertyChangeListener(actionBarPropListener);
			rec.page.setActionBars(rec.subActionBars);
		} else {
			site = rec.page.getSite();
			rec.subActionBars = (SubActionBars) site.getActionBars();
		}
	}

	/**
	 * Removes a page record. If it is the last reference to the page dispose of
	 * it - otherwise just decrement the reference count.
	 * 
	 * @param rec
	 *            the page record
	 */
	private void disposePage(PageRec rec) {
		fMapContributorToRec.remove(rec.part);

		if (rec.subActionBars != null) {
			rec.subActionBars.removePropertyChangeListener(actionBarPropListener);
			rec.subActionBars.dispose();
			rec.subActionBars = null;
		}

		// Be proactive and dispose the page's control. The page's control is
		// a child of this view's control so if this view is closed, the
		// page's control will already be disposed.
		Control control = rec.page.getControl();
		if (control != null && !control.isDisposed()) {
			control.dispose();
		}

		((PageSite) rec.page.getSite()).deactivate();
		doDestroyPage(rec.part, rec);
	}

	/**
	 * Inform the listeners that the property has changed by generating a
	 * <code>PropertyChangeEvent event</code>.
	 * 
	 * @param listeners
	 *            the lsit of listeners
	 * @param obj
	 *            the new changed property object
	 */
	private void informListeners(List<IPropertyChangeListener> listeners, Object obj) {
		for (IPropertyChangeListener listener : listeners) {
			listener.propertyChange(new PropertyChangeEvent(this, "NewPageInstantiated", null, obj)); //$NON-NLS-1$
		}
	}

	/**
	 * Refreshes the global actions for the active page.
	 */
	private void refreshGlobalActionHandlers() {
		// Clear old actions.
		IActionBars bars = getViewSite().getActionBars();

		bars.clearGlobalActionHandlers();

		// Set new actions.
		Map<?, ?> newActionHandlers = fActiveRec.subActionBars.getGlobalActionHandlers();
		if (newActionHandlers != null) {
			Set<?> keys = newActionHandlers.entrySet();
			Iterator<?> iter = keys.iterator();
			while (iter.hasNext()) {
				Map.Entry<?, ?> entry = (Map.Entry<?, ?>) iter.next();
				bars.setGlobalActionHandler((String) entry.getKey(), (IAction) entry.getValue());
			}
		}
	}

	/**
	 * The action bar property listener.
	 */
	private IPropertyChangeListener actionBarPropListener = new IPropertyChangeListener() {
		@Override
		public void propertyChange(PropertyChangeEvent event) {
			if (SubActionBars.P_ACTION_HANDLERS.equals(event.getProperty()) && fActiveRec != null
					&& event.getSource() == fActiveRec.subActionBars) {
				refreshGlobalActionHandlers();
			}
		}
	};

	/**
	 * Based on the active debug context (if there is any) assign the
	 * appropriate page as the current one in the page book. If there's no
	 * active debug context, or there is but it doesn't meet our criteria, then
	 * we show the default page (says "no info available"). We may have to
	 * create the page to display if it hasn't been created yet; they are lazily
	 * created. SP: Changed as DSF comes.
	 */
	private void chooseCurrentPage(final ILaunch launch) {
		PageRec pagerec = null;
		if (launch != null) {
			SystemBrowserActivator.log(1, "chooseCurrentPage =>" + launch.toString()); //$NON-NLS-1$
			ILaunchConfiguration launchConfig = launch.getLaunchConfiguration();
			try {
				SystemContributor systemContributor = null;

				// Is this debugger using OS Awareness? If yes, then the OS
				// Contributor dictates which system contributor to use. If no,
				// then the session may still involve a system browser but there
				// we need to see which SystemBrowser claims to support the
				// debugger
				final String osContributorId = launchConfig.getAttribute(KAPluginActivator.OSPLUGIN_LAUNCH_ATTR,
						OSAwarenessData.DEFAULT_OS_CONTRIBUTOR_ID);
				systemContributor = SystemContributor.getSystemContributor(osContributorId);

				if (systemContributor != null) {
					Set<Entry<ITabbedPageContributor, PageRec>> set = fMapContributorToRec.entrySet();
					Iterator<Entry<ITabbedPageContributor, PageRec>> iterator = set.iterator();
					while (iterator.hasNext()) {
						Entry<ITabbedPageContributor, PageRec> entry = iterator.next();
						if (entry.getKey().getContributorId().equals(systemContributor.getId())) {
							pagerec = entry.getValue();
						}
					}
					if (pagerec == null) {
						TabbedSheetPage newPage = new TabbedSheetPage(systemContributor);
						pagerec = getOrCreatePageRec(newPage);
					}
				}
			} catch (CoreException e) {
				SystemBrowserActivator.log(e.getStatus());
			}
		}

		if (pagerec == null) {
			pagerec = fDefaultPageRec;
		}
		showPageRec(pagerec);
	}

	private void setupContextListener() {
		final IDebugContextManager contextManager = DebugUITools.getDebugContextManager();
		IDebugContextService contextService = contextManager.getContextService(getSite().getWorkbenchWindow());

		fDebugContextListener = new IDebugContextListener() {
			@Override
			public void debugContextChanged(DebugContextEvent event) {
				if ((event.getFlags() & DebugContextEvent.ACTIVATED) != 0) {
					ISelection s = event.getContext();
					setDebugContext(s);
				}
			}
		};
		contextService.addDebugContextListener(fDebugContextListener);
		setDebugContext(contextService.getActiveContext());
	}

	/**
	 * Called on new selection in Debug View.
	 */
	private void setDebugContext(final ISelection selection) {
		IExecutionDMContext context = null;
		if (selection instanceof IStructuredSelection && !selection.isEmpty()) {
			// take only first
			final Object element = ((IStructuredSelection) selection).getFirstElement();
			context = getExecutionDMContext(element);

			if (context != null) {
				ILaunch launch = context.getAdapter(ILaunch.class);
				if (!contextActive(launch, context)) {
					context = null;
				}
			} else {
				removeSessionPagesData();
			}
		}
		changeContext(context);
	}

	private boolean contextActive(Object element, IExecutionDMContext context) {
		if (element instanceof GdbLaunch) {
			GdbLaunch launch = (GdbLaunch) element;
			DsfSession session = launch.getSession();
			if (session == null || !session.isActive()) {
				removeSessionPagesData();
			} else {
				String sessionId = session.getId();
				if (m_session == null || !m_session.equals(sessionId)) {
					m_session = sessionId;
					chooseCurrentPage(launch);
				}
				return true;
			}
		}
		return false;
	}

	private IExecutionDMContext getExecutionDMContext(Object obj) {
		if (obj instanceof IAdaptable) {
			IAdaptable context = (IAdaptable) obj;
			IDMContext idmContext = context.getAdapter(IDMContext.class);
			if (idmContext != null) {
				return DMContexts.getAncestorOfType(idmContext, IExecutionDMContext.class);
			}
		}
		return null;
	}

	private void removeSessionPagesData() {
		m_session = null;
		showPageRec(fDefaultPageRec);
		fPageRecs.forEach(pageRec -> disposePage(pageRec));
		fPageRecs.clear();
	}
}