/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * Copyright (c) 2010 Freescale Semiconductor Inc.  All rights reserved.
 * Copyright 2023 NXP
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *******************************************************************************/
package com.freescale.system.browser.ui;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.part.Page;

import com.freescale.system.browser.model.OSAwarenessData;


/**
 * Represents a section.
 * <p>
 * The lifecycle of an ISection is as follows:
 * <ul>
 * <li><code>ISection.createControls()</code></li>
 * <li><code>ISection.setSectionInput()</code></li>
 * <li><code>ISection.aboutToBeShown()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * <li><code>ISection.aboutToBeHidden()</code></li>
 * <li><code>ISection.dispose()</code></li>
 * </ul>
 * </p>
 * <p>
 * Implementors of this class should be aware that a section instance might be
 * reused for different input objects (as long as they are valid section
 * inputs). It means that <code>ISection.setSectionInput</code> can be called at any
 * time between <code>ISection.createControls</code> and
 * <code>ISection.dispose</code>.
 * </p>
 * <p>
 * When an input change event occurs, such as a tab selection or a workbench
 * selection change, an ISection is sent:
 * <ul>
 * <li><code>ISection.setSectionInput()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * </ul>
 * </p>
 * <p>
 * When an activation event occurs, such as the contributor activation
 * event, an ISection is sent:
 * <ul>
 * <li><code>ISection.setSectionInput()</code></li>
 * <li><code>ISection.aboutToBeShown()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * <li><code>ISection.setInput()</code></li>
 * <li><code>ISection.refresh()</code></li>
 * </ul>
 * This is because both a tab selection event and an input selection event have
 * occurred.
 * </p>
 * <p>
 * This interface should not be extended or implemented. New section instances
 * should be created using <code>AbstractSection</code>.
 * </p>
 * @see com.freescale.system.browser.internal.ui.tabbed.TabbedSheetPage
 * 
 * @author Anthony Hunter
 * @version 1.0
 */
public interface ISection {

	/**
	 * Creates the controls for the section.
	 * <p>
	 * Clients should take advantage of the widget factory provided by the
	 * framework to achieve a common look between sections.
	 * </p>
	 * @see org.eclipse.ui.views.properties.tabbed.TabbedSheetPage#getWidgetFactory()
	 * 
	 * @param parent
	 *            the parent composite for the section.
	 * @param tabbedPage
	 *            the tabbed page.
	 */
	public abstract void createControls(Composite parent,
			Page tabbedPage);

	/**
	 * Notifies the section that the active debug context has changed.
	 * The section should update its UI to reflect the new context
	 * @param input the active debug context 
	 */
	public abstract void setSectionInput(OSAwarenessData input);

	/**
	 * Returns the value passed into the most recent setInput() call.
	 */
	public abstract Object getSectionInput();
	
	/**
	 * Notifies the section that its controls are about to be shown. It is
	 * expected that sections enable domain related functions in this method,
	 * most commonly add listeners.
	 * <p>
	 * Since the controls are not visible, the section should wait for the
	 * refresh() before updating the section controls.
	 * </p>
	 */
	public abstract void aboutToBeShown();

	/**
	 * Notifies the section that its controls are about to be hidden. It is
	 * expected that sections disable domain related functions in this method,
	 * most commonly remove listeners.
	 */
	public abstract void aboutToBeHidden();

	/**
	 * Dispose this section.
	 */
	public abstract void dispose();

	/**
	 * Returns the minimum height needed by this section. A return value of
	 * <code>SWT.DEFAULT</code> indicates that no minimum height is defined.
	 * 
	 * @return the minimum height needed by this section.
	 */
	public abstract int getMinimumHeight();

	/**
	 * Determine whether this section would like extra height space in case
	 * there is some left. Normally this is true when the section is the last to
	 * be displayed on a tab or is the only section on a tab.
	 * @return <code>true</code> if this section would like extra height space.
	 */
	public abstract boolean shouldUseExtraSpace();

	/**
	 * Refresh the contents of the controls displayed in this section.
	 * The typical implementation calls Viewer.refresh() on all jFace 
	 * viewers in its section.
	 */
	public abstract void refresh();
	
	/**
	 * Invalidate the data exposed by this section. This does not directly 
	 * cause the section to gather new context, less display it. However, it 
	 * should do so next time it is asked to display the section content.
	 * To get the whole package (invalidate and force new content to be 
	 * gathered and shown), call refresh() instead.
	 * 
	 */
	public abstract void invalidate();
	
	/**
	 * Allows a section to customize the action bars that will be available when
	 * the section is shown. This hook is called every time the section's
	 * containing tab is about to show.
	 * 
	 * The toolbar is wiped clean prior to this hook being called; the System
	 * Browser will add any actions it has after those added by this method.
	 * 
	 * @param actionBars
	 *            the action bars for the page
	 */
	public void setActionBarsForTab(IActionBars actionBars);
}
