/*******************************************************************************
 * Copyright (c) 2001, 2006, 2019, 2022 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *     NXP - modified by and for NXP
 *******************************************************************************/
package com.freescale.system.browser.internal.ui.tabbed;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.cdt.dsf.debug.service.IRunControl.IExecutionDMContext;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.Page;

import com.freescale.system.browser.internal.SystemBrowserActivator;
import com.freescale.system.browser.internal.ui.SystemContributor;
import com.freescale.system.browser.internal.ui.Tab;
import com.freescale.system.browser.internal.ui.TabDescriptor;
import com.freescale.system.browser.internal.ui.TabListContentProvider;
import com.freescale.system.browser.internal.ui.TabbedComposite;
import com.freescale.system.browser.internal.ui.TabbedSystemViewer;
import com.freescale.system.browser.internal.ui.TabbedTitle;
import com.freescale.system.browser.internal.ui.builtintab.tasks.TasksTabDescriptor;
import com.freescale.system.browser.model.ISimpleTableDataProvider;
import com.freescale.system.browser.model.OSAwarenessData;
import com.freescale.system.browser.ui.IActionProvider;
import com.freescale.system.browser.ui.ISection;
import com.freescale.system.browser.ui.TabbedWidgetFactory;


/**
 * A page that provides a tabbed UI. 
 */
public class TabbedSheetPage extends Page implements ITabbedPage,
		ILabelProviderListener {
	/**
	 * A tabbed composite.
	 */
	private TabbedComposite fTabbedComposite;

	/**
	 * The widget factory.
	 */
	private TabbedWidgetFactory fWidgetFactory;

	/**
	 * The page contributor.
	 */
	private ITabbedPageContributor fPageContributor;

	/**
	 * The tabbed registry.
	 */
	private SystemContributor fSystemContributor;
	
	/**
	 * A content provider for the tab list.
	 */
	protected IStructuredContentProvider fTabListContentProvider;

	/**
	 * A tabbed system viewer.
	 */
	private TabbedSystemViewer fTabbedSystemViewer;

	/**
	 * The currently selected tab.
	 */
	private Tab fCurrentTab;

	/**
	 * A mapping between tab descriptors and tabs.
	 */
	private Map<ITabDescriptor, Tab> fDescriptorToTab;

	/**
	 * A mapping between tabs and composites. Contains only tabs shown
	 * to the user.
	 */
	private Map<Tab, Composite> fTabToComposite;

	/**
	 * A list representing a selection queue.
	 */
	private List<String> fSelectionQueue;

	/**
	 * Locking state of the selection queue. 
	 */
	private boolean fSelectionQueueLocked;

	/**
	 * List containing tab selection listeners.
	 */
	private List<ITabSelectionListener> fTabSelectionListeners;

	/**
	 * We store the action bar object so that we can call the provider when
	 * there's a change in the tab selection. He may want to add/remove actions
	 * based on the active tab.
	 */
	private IActionBars fActionBars;

	/**
	 * The active debug context, as set in us via setInput()
	 */
	private OSAwarenessData fContext;

	/**
	 * Label provider for the ListViewer.
	 */
	static class TabbedPageLabelProvider extends LabelProvider {
		@Override
		public String getText(Object element) {
			if (element instanceof TabDescriptor) {
				return ((TabDescriptor)element).getLabel();
			}
			return null;
		}
	}


	/**
	 * SelectionChangedListener for the ListViewer.
	 */
	class SelectionChangedListener implements ISelectionChangedListener {

		/**
		 * Shows the tab associated with the selection.
		 */
		@Override
		public void selectionChanged(SelectionChangedEvent event) {
			IStructuredSelection selection = (IStructuredSelection)event.getSelection();
			Tab tab = null;
			TabDescriptor descriptor = (TabDescriptor)selection.getFirstElement();

			if (descriptor == null) {
				// pretend the tab is empty.
				hideTab(fCurrentTab);
			} else {
				// create tab if necessary
				// can not cache based on the id - tabs may have the same id,
				// but different section depending on the selection
				tab = fDescriptorToTab.get(descriptor);

				if (tab != fCurrentTab) {
					hideTab(fCurrentTab);
				}

				Composite tabComposite = fTabToComposite.get(tab);

				if (tabComposite == null) {
					tabComposite = createTabComposite();
					tab.createControls(tabComposite, TabbedSheetPage.this);
					fTabToComposite.put(tab, tabComposite);
				}

				// store tab selection
				storeCurrentTabSelection(descriptor.getLabel());

				if (tab != fCurrentTab) {
					tab.setTabInput(fContext);
					showTab(tab);
				}
			}
			fTabbedComposite.getTabComposite().layout(true);
			fCurrentTab = tab;
			resizeScrolledComposite();

			if (descriptor != null) {
				handleTabSelection(descriptor);

				if (fActionBars != null) {
					// start with a clean slate; the managers are rebuilt from scratch
					fActionBars.getToolBarManager().removeAll();
					fActionBars.getMenuManager().removeAll();
					fActionBars.getStatusLineManager().removeAll();

					// Give each section in the tab an opportunity to customize the action bars
					List <ISection> sections = fCurrentTab.getSections();
					for (ISection section : sections) {
						section.setActionBarsForTab(fActionBars);
					}

					// Give the systemContributor an opportunity to contribute its page-wide
					// actions.
					if (fSystemContributor.getActionProvider() != null) {
						fSystemContributor.getActionProvider().setActionBars(fActionBars);
					}

					// Force all managers to show their new content
					fActionBars.getToolBarManager().update(true);
					fActionBars.getMenuManager().update(true);
					fActionBars.getStatusLineManager().update(true);
				}
			}
		}

		/**
		 * Shows the given tab.
		 */
		private void showTab(Tab target) {
			if (target != null) {
				Composite tabComposite = fTabToComposite.get(target);

				if (tabComposite != null) {
					/**
					 * the following method call order is important - do not
					 * change it or the widgets might be drawn incorrectly
					 */
					tabComposite.moveAbove(null);
					target.aboutToBeShown();
					tabComposite.setVisible(true);
					target.setFocus();
				}
			}
		}

		/**
		 * Hides the given tab.
		 */
		private void hideTab(Tab target) {
			if (target != null) {
				Composite tabComposite = fTabToComposite.get(target);

				if (tabComposite != null) {
					target.aboutToBeHidden();
					tabComposite.setVisible(false);
				}
			}
		}
	}

	/**
	 * An implementation of the tabbed page contributor backed by a system
	 * contributor instance
	 */
	static class SystemPageContributor implements ITabbedPageContributor {
		private ITabbedPage m_page;
		private SystemContributor m_systemContributor;

		/**
		 * Constructor.
		 * 
		 * @param system
		 *            the systemContributor
		 * @param page
		 *            the tabbed page instance
		 */
		public SystemPageContributor(SystemContributor system, ITabbedPage page) {
			m_systemContributor = system;
			m_page = page;
		}

		/* (non-Javadoc)
		 * @see com.freescale.system.browser.ui.tabbed.ITabbedPageContributor#getContributorId()
		 */
		@Override
		public String getContributorId() {
			return m_systemContributor.getId();
		}

		/* (non-Javadoc)
		 * @see com.freescale.system.browser.ui.tabbed.ITabbedPageContributor#getTabbedSheetPage()
		 */
		@Override
		public ITabbedPage getTabbedSheetPage() {
			return m_page;
		}
	}

	/**
	 * Create a new tabbed sheet page.
	 * 
	 * @param tabbedSheetPageContributor
	 *            the tabbed sheet page contributor.
	 */
	public TabbedSheetPage(final SystemContributor systemContributor) {
		final TabbedSheetPage page = this;
		final String id = systemContributor.getId();
		fPageContributor = new ITabbedPageContributor() {
			@Override
			public String getContributorId() {
				return id;
			}
			@Override
			public ITabbedPage getTabbedSheetPage() {
				return page;
			}
		};

		fTabToComposite = new HashMap<>();
		fSelectionQueue = new ArrayList<>(10);
		fTabSelectionListeners = new ArrayList<>();

		initContributor(systemContributor);
	}

	/**
	 * @see org.eclipse.ui.part.IPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {
		fWidgetFactory = new TabbedWidgetFactory();
		fTabbedComposite = new TabbedComposite(parent, fWidgetFactory, fSystemContributor.getLabelProvider() != null);
		fWidgetFactory.paintBordersFor(fTabbedComposite);
		fTabbedComposite.setLayout(new FormLayout());
		FormData formData = new FormData();

		formData.left = new FormAttachment(0, 0);
		formData.right = new FormAttachment(100, 0);
		formData.top = new FormAttachment(0, 0);
		formData.bottom = new FormAttachment(100, 0);
		fTabbedComposite.setLayoutData(formData);

		ImageDescriptor iconDescr = fSystemContributor.getIcon();
		fTabbedComposite.setOsNameAndImage(fSystemContributor.getName(),
				iconDescr != null ? iconDescr.createImage() : null);

		fTabbedSystemViewer = new TabbedSystemViewer(fTabbedComposite.getList());
		fTabbedSystemViewer.setContentProvider(fTabListContentProvider);
		fTabbedSystemViewer.setLabelProvider(new TabbedPageLabelProvider());
		fTabbedSystemViewer.addSelectionChangedListener(new SelectionChangedListener());

		fTabbedComposite.getScrolledComposite().addControlListener(new ControlAdapter() {
			@Override
			public void controlResized(ControlEvent e) {
				resizeScrolledComposite();
			}
		});

		/**
		 * Add a label provider change listener.
		 */
		if (fSystemContributor.getLabelProvider() != null) {
			fSystemContributor.getLabelProvider().addListener(this);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.Page#dispose()
	 */
	@Override
	public void dispose() {
		disposeContributor();

		if (fWidgetFactory != null) {
			fWidgetFactory.dispose();
			fWidgetFactory = null;
		}

		fSystemContributor = null;
		fPageContributor = null;
	}

	/**
	 * @see org.eclipse.ui.part.IPage#getControl()
	 */
	@Override
	public Control getControl() {
		return fTabbedComposite;
	}

	/**
	 * @see org.eclipse.ui.part.IPage#setActionBars(org.eclipse.ui.IActionBars)
	 */
	@Override
	public void setActionBars(IActionBars actionBars) {
		IActionProvider actionProvider = fSystemContributor.getActionProvider();
		if (actionProvider != null) {
			actionProvider.setActionBars(actionBars);
		}
		// Store the action bar object so that we can call the provider when
		// there's a change in the tab selection. He may want to add/remove
		// actions based on the active tab.
		fActionBars = actionBars;
	}

	/**
	 * Forward the focus directive to the active tab, or worst case,
	 * to the page composite.
	 * @see org.eclipse.ui.part.IPage#setFocus()
	 */
	@Override
	public void setFocus() {
		if (fCurrentTab != null) {
			fCurrentTab.setFocus();
		} else {
			getControl().setFocus();
		}
	}

	/**
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart,
	 *      org.eclipse.jface.viewers.ISelection)
	 * 
	 */
	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {// not usable in this context
		// setInput(part, selection);
	}

	/**
	 * Get the currently active tab.
	 * 
	 * @return the currently active tab.
	 */
	public Tab getCurrentTab() {
		return fCurrentTab;
	}

	/**
	 * Add a tab selection listener.
	 * 
	 * @param listener
	 *            a tab selection listener.
	 */
	public void addTabSelectionListener(ITabSelectionListener listener) {

		fTabSelectionListeners.add(listener);
	}

	/**
	 * Remove a tab selection listener.
	 * 
	 * @param listener
	 *            a tab selection listener.
	 */
	public void removeTabSelectionListener(ITabSelectionListener listener) {

		fTabSelectionListeners.remove(listener);
	}

	/**
	 * Get the widget factory.
	 * 
	 * @return the widget factory.
	 */
	public TabbedWidgetFactory getWidgetFactory() {

		return fWidgetFactory;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ILabelProviderListener#labelProviderChanged(org.eclipse.jface.viewers.LabelProviderChangedEvent)
	 */
	@Override
	public void labelProviderChanged(LabelProviderChangedEvent event) {// do nothing; not 
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabbedPage#getContributor()
	 */
	@Override
	public ITabbedPageContributor getContributor() {
		return (fPageContributor);
	}

	/**
	 * Gets the tab list content provider for the contributor.
	 * 
	 * @return the tab list content provider for the contributor.
	 */
	protected IStructuredContentProvider getTabListContentProvider() {
		return new TabListContentProvider(fSystemContributor);
	}

	/**
	 * Update the current tabs to represent the given input object. When tabs
	 * apply for both the old and new input they are reused otherwise they are
	 * disposed. If the current visible tab will not be reused (i.e. will be
	 * disposed) we have to send it an aboutToBeHidden() message.
	 * 
	 * @param descriptors an array of tab descriptors
	 */
	protected void updateTabs(final List<ITabDescriptor> descriptors) {

		Map<ITabDescriptor, Tab> newTabsMap = new HashMap<>(descriptors.size() * 2);
		boolean disposingCurrentTab = (fCurrentTab != null);

		for (final ITabDescriptor descriptor : descriptors) {
			Tab tab = fDescriptorToTab.remove(descriptor);
			if (tab != null && tab.controlsHaveBeenCreated()) {
				if (tab == fCurrentTab) {
					disposingCurrentTab = false;
				}
			} else {
				tab = descriptor.createTab();
			}

			newTabsMap.put(descriptor, tab);
		}
		if (disposingCurrentTab) {
			// If the current tab is about to be disposed we have to call
			// aboutToBeHidden
			fCurrentTab.aboutToBeHidden();
			fCurrentTab = null;
		}
		disposeTabs(fDescriptorToTab.values());
		fDescriptorToTab = Collections.synchronizedMap(newTabsMap);
	}
	
	/**
	 * Initialize the contributor with the provided contributor id.
	 * 
	 * @param contributorId
	 *            the contributor id.
	 */
	private void initContributor(SystemContributor systemContributor) {
		fDescriptorToTab = new HashMap<>();
		fSystemContributor = systemContributor;
		fTabListContentProvider = getTabListContentProvider();
	
		if (fTabbedSystemViewer != null) {
			fTabbedSystemViewer.setContentProvider(fTabListContentProvider);
		}

		/**
		 * Add a label provider change listener.
		 */
		if (fSystemContributor.getLabelProvider() != null) {
			fSystemContributor.getLabelProvider().addListener(this);
		}

	}

	/**
	 * Dispose the contributor with the provided contributor id. This happens on
	 * part close as well as when contributiors switch between the workbench
	 * part and contributor from a selection.
	 * 
	 * @param contributorId
	 *            the contributor id.
	 */
	private void disposeContributor() {

		/**
		 * If the current tab is about to be disposed we have to call
		 * aboutToBeHidden
		 */
		if (fCurrentTab != null) {
			fCurrentTab.aboutToBeHidden();
			fCurrentTab = null;
		}

		disposeTabs(fDescriptorToTab.values());
		fDescriptorToTab = new HashMap<>();

		/**
		 * Remove the label provider change listener.
		 */
		if (fSystemContributor != null && fSystemContributor.getLabelProvider() != null) {
			fSystemContributor.getLabelProvider().removeListener(this);
		}
	}

	/**
	 * Stores the current tab label in the selection queue. Tab labels are used
	 * to carry the tab context from one input object to another. The queue
	 * specifies the selection priority. So if the first tab in the queue is not
	 * available for the input we try the second tab and so on. If none of the
	 * tabs are available we default to the first tab available for the input.
	 * 
	 * @param label the label
	 */
	private void storeCurrentTabSelection(String label) {
		if (!fSelectionQueueLocked) {
			fSelectionQueue.remove(label);
			fSelectionQueue.add(0, label);
		}
	}

	/**
	 * Resizes the right hand side scrolled composite in the page.
	 */
	private void resizeScrolledComposite() {
		Point currentTabSize = new Point(0, 0);

		if (fCurrentTab != null) {
			Composite sizeReference = fTabToComposite.get(fCurrentTab);

			currentTabSize.y = (sizeReference != null) ? sizeReference.computeSize(SWT.DEFAULT, SWT.DEFAULT).y : 0;
			currentTabSize.x = (sizeReference != null) ? sizeReference.computeSize(SWT.DEFAULT, SWT.DEFAULT).x : 0;
		}
		if (fSystemContributor.getLabelProvider() != null) {
			int titleHeight = fTabbedComposite.getTitle().computeSize(SWT.DEFAULT, SWT.DEFAULT).y;
			currentTabSize.y += titleHeight;
		}
		fTabbedComposite.getScrolledComposite().setMinSize(currentTabSize.x, currentTabSize.y);
	}

	/**
	 * Proper cleanup for tabs.
	 * 
	 * @param tabs the collection of tabs
	 */
	private void disposeTabs(Collection<Tab> tabs) {
		for (Tab tab : tabs) {
			Composite composite = fTabToComposite.remove(tab);

			tab.dispose();
			if (composite != null) {
				composite.dispose();
			}
		}
	}

	/**
	 * Returns the most recently selected tab that is applicable to the given
	 * input (active debug context)
	 * 
	 * @param input
	 *            the active debug context
	 * @return a zero based index into the contributors list of tabs. Zero is
	 *         returned if the lookup yields no results.
	 * 
	 */
	private int getLastTabSelection(final Object input) {
		List<ITabDescriptor> descriptors = fSystemContributor.getTabDescriptors(input);
		if (descriptors.size() != 0) {
			for (String label : fSelectionQueue) {
				for (int i = 0; i < descriptors.size(); i++) {
					if (label.equals(descriptors.get(i).getLabel())) {
						return i;
					}
				}
			}
		}
		return 0;
	}

	/**
	 * Helper method for creating property tab composites.
	 */
	private Composite createTabComposite() {
		Composite result = fWidgetFactory.createComposite(fTabbedComposite.getTabComposite(), SWT.NO_FOCUS);

		result.setVisible(false);
		result.setLayout(new FillLayout());
		FormData data = new FormData();

		if (fSystemContributor.getLabelProvider() != null) {
			data.top = new FormAttachment(fTabbedComposite.getTitle(), 0);
		} else {
			data.top = new FormAttachment(0, 0);
		}
		data.bottom = new FormAttachment(100, 0);
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		result.setLayoutData(data);
		return result;
	}

	/**
	 * Set the context for this page; this is the active debug context (the selection
	 * in the Debug view).
	 * 
	 * @param selection the active debug context. The current page should update its 
	 * content accordingly.
	 */
	@Override
	public void setPageInput(final OSAwarenessData input) {
		//don't set input if control disposed
		if (fTabbedSystemViewer.getControl().isDisposed() && fContext != null) {
			shutdownTabs(fContext.getContext());

			ISimpleTableDataProvider provider = fContext.getLink();
			if (provider != null) {
				provider.getThreadSequenceHolder(null);
				SystemBrowserActivator.log(1, "setPageInput (shutdown) => " + provider.toString()); //$NON-NLS-1$
			}
			return;
		}

		IExecutionDMContext context = input.getContext();
		if (context == null) {
			if (fTabbedSystemViewer.getInput() != null) {
				Object obj = fContext.getContext().getSessionId();
				fContext = input;

				List<ITabDescriptor> descriptors = fSystemContributor.getTabDescriptors(obj);
				descriptors.forEach(descriptor -> {
					fDescriptorToTab.get(descriptor).setTabInput(fContext);
				});
				fTabbedSystemViewer.setInput(null);
				SystemBrowserActivator.log(1, "setPageInput => null"); //$NON-NLS-1$
			}
			return;
		}

		SystemBrowserActivator.log(1, "setPageInput => " + context); //$NON-NLS-1$
		fContext = input;
		final String obj = context.getSessionId();
		// We give the structured viewer its new context. This will end up invoking
		// TabbedSystemViewer.inputChanged() which will end up displaying the 
		// appropriate set of tabs (of our system contributor) based on the new context.
		fTabbedSystemViewer.setInput(obj);
		
		// If there are no tab descriptors for the given input we do not need to
		// touch the tab objects. We might reuse them for the next valid
		// input.
		List<ITabDescriptor> descriptors = fSystemContributor.getTabDescriptors(obj);
		if (!descriptors.isEmpty()) {
			updateTabs(descriptors);
		} else {
			// Hm. We're wasting our time. No tabs in this page. Bail...
			return;
		}
		
		final ITabDescriptor tabToSelect = fTabbedSystemViewer.getNthTab(getLastTabSelection(obj));

		fSelectionQueueLocked = true;
		try {
			if (tabToSelect == null) {
				fTabbedSystemViewer.setSelection(null);
			} else {
				fTabbedSystemViewer.setSelection(new StructuredSelection(tabToSelect));
			}
		} finally {
			fSelectionQueueLocked = false;
		}
		
		// Update the page (active tab) content using the new debug context...

		// First the page title and icon. This is the title that appears at the 
		// top of the active tab. It is NOT the name of the tab; the same string 
		// will display no matter what tab is the active one. The title 
		// displays the context for the page content. Each system contributor 
		// has a label provider. We pass the active debug context to it and 
		// display the string and icon it returns as the page title.
		TabbedTitle title = fTabbedComposite.getTitle();
		if (title != null) {
			if (fCurrentTab == null) {
				// No tabs are shown so hide the title bar, otherwise you see
				// "No systen info available" and a title bar for the selection.
				title.setTitle(null, null);
			} else {
				ILabelProvider labelProvider = fSystemContributor.getLabelProvider(); 
				if (labelProvider != null) {
					title.setTitle(labelProvider.getText(fContext), labelProvider.getImage(fContext));
				} else {
					title.setTitle(fContext.toString(), null);
				}
			}
		}

		// Then the content of the active tab itself. 
		Tab tab = fDescriptorToTab.get(tabToSelect);
		assert (tab != null);
		tab.setTabInput(fContext);
	}

	private void shutdownTabs(IExecutionDMContext context) {
		if (context!= null) {
			final String obj = context.getSessionId();
			final List<ITabDescriptor> descriptors = fSystemContributor.getTabDescriptors(obj);
			for (final ITabDescriptor descriptor: descriptors) {
				if (descriptor instanceof TasksTabDescriptor) {
					TasksTabDescriptor ttDescriptor = (TasksTabDescriptor)descriptor;
					ttDescriptor.getTableDataObject().getThreadSequenceHolder(null);
				}
			}
		}
	}

	/**
	 * Handle the tab selected change event.
	 * 
	 * @param tabDescriptor
	 *            the new selected tab.
	 */
	private void handleTabSelection(TabDescriptor tabDescriptor) {
		if (fSelectionQueueLocked) {
			// don't send tab selection events for non user changes.
			return;
		}
		for (ITabSelectionListener listener : fTabSelectionListeners) {
			listener.tabSelected(tabDescriptor);
		}
	}
	
	/**
	 * Force a refresh of all tabs. From a user perspective, this is what happens
	 * but under the covers, we refresh the currently showing tab and invalidate all
	 * others. Invalidated tabs will automatically refresh their content when they
	 * are brought to the forefront. 
	 */
	public void refresh() {
		Set<Tab> tabs = fTabToComposite.keySet();
		for (Tab tab : tabs) {
			if (tab == fCurrentTab) {
				tab.refresh();
			} else {
				tab.invalidate();
			}
		}
	}
}