/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *******************************************************************************/
package com.freescale.system.browser.internal.ui;


import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;

import com.freescale.system.browser.internal.Messages;
import com.freescale.system.browser.internal.SystemBrowserActivator;
import com.freescale.system.browser.internal.ui.tabbed.ISectionDescriptor;
import com.freescale.system.browser.internal.ui.tabbed.ITabDescriptor;
import com.ibm.icu.text.MessageFormat;


/**
 * Represents the default implementation of a tab descriptor on the tabbed tabs
 * extensions. Is a representation for the systemTabs extension point. It parses
 * the extension point and saves the attributes for further usage.
 */
public abstract class TabDescriptor
		implements ITabDescriptor {

	/**
	 * The label attribute of the systemTabs extension point.
	 */
	protected static final String ATT_TITLE = "title"; //$NON-NLS-1$

	/**
	 * The image attribute of the systemTabs extension point.
	 */
	protected static final String ATT_IMAGE = "image"; //$NON-NLS-1$

	/**
	 * The indented attribute of the systemTabs extension point.
	 */
	protected static final String ATT_INDENTED = "indented"; //$NON-NLS-1$

	/**
	 * The afterTab attribute of the systemTabs extension point.
	 */
	protected static final String ATT_AFTER_TAB = "afterTab"; //$NON-NLS-1$

	/**
	 * The top attribute of the systemTabs extension point.
	 */
	protected static final String TOP = "top"; //$NON-NLS-1$

	/**
	 * The indentation state.
	 */
	protected boolean fIndented;
	
	/**
	 * The name of the tab this tab is positioned after.
	 */
	protected String fAfterTab;

	/**
	 * The selection state.
	 */
	protected boolean fSelected;
	
	/**
	 * The label (text) in the tab
	 */
	protected String fTitle;

	/**
	 * The label (text) in the tab
	 */
	protected ImageDescriptor fIcon;
	
	/**
	 * The tab error string.
	 */
	private final static String TAB_ERROR = Messages.TabDescriptor_Tab_error;

	/**
	 * Constructor.
	 * 
	 * @param the XML that describes the tab
	 */
	protected TabDescriptor(IConfigurationElement configurationElement) {
		fAfterTab = configurationElement.getAttribute(ATT_AFTER_TAB);
		
		final String indentedString = configurationElement.getAttribute(ATT_INDENTED);

		fIndented = indentedString != null && indentedString.equals("true"); //$NON-NLS-1$
		
		fAfterTab = configurationElement.getAttribute(ATT_AFTER_TAB);
		if (fAfterTab == null) {
			fAfterTab = TOP;
		}
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabItem#isIndented()
	 */
	@Override
	public boolean isIndented() {
		return fIndented;
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabItem#isSelected()
	 */
	@Override
	public boolean isSelected() {
		return fSelected;
	}
	
	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabItem#getId()
	 */
	@Override
	public abstract String getId();

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabItem#getImage()
	 */
	@Override
	public ImageDescriptor getImage() {
		return fIcon;
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ITabItem#getLabel()
	 */
	@Override
	public String getLabel() {
		return fTitle;
	}

	/**
	 * Factory method. Subclass must instantiate the tab.
	 */
	@Override
	public final Tab createTab() {
		return createTabEx();		
	}
	
	protected abstract Tab createTabEx();
	/**
	 * Handle the tab error when an issue is found loading from the
	 * configuration element.
	 * 
	 * @param configurationElement
	 *            the configuration element
	 * @param exception
	 *            an optional CoreException
	 */
	protected void handleTabError(IConfigurationElement configurationElement,
			CoreException exception) {
		String pluginId = configurationElement.getDeclaringExtension().getNamespaceIdentifier();
		String message = MessageFormat.format(TAB_ERROR, new Object[] { pluginId});
		IStatus status = new Status(IStatus.ERROR, pluginId, 0, message, exception);

		SystemBrowserActivator.getPlugin().getLog().log(status);
	}
	
	/**
	 * A section can claim it is applicable for only select input. If all the
	 * sections of a tab are not applicable to the active input, then we hide
	 * that tab. Subclasses override this method if there is potential for
	 * selectivity
	 * 
	 * @param input
	 *            the active debug context
	 * @return wither the tab should be shown or not
	 */
	@Override
	public boolean showForInput(final Object input) {
		if (input != null) {
		    return true;
		} else {
			return false;
		}
	}
	
    /**
     * Link section to section descriptor.
     * @param sectionDescriptor
     *        
     * */ 
	protected abstract void addSectionDescriptor(final ISectionDescriptor sectionDescriptor);
}
