/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - heavily modified by and for Freescale. Changed name, too.
 *******************************************************************************/
package com.freescale.system.browser.internal.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import com.freescale.system.browser.internal.Messages;
import com.freescale.system.browser.internal.SystemBrowserActivator;
import com.freescale.system.browser.internal.ui.builtintab.tasks.TasksTabDescriptor;
import com.freescale.system.browser.internal.ui.tabbed.ISectionDescriptor;
import com.freescale.system.browser.internal.ui.tabbed.ITabDescriptor;
import com.freescale.system.browser.ui.IActionProvider;
import com.ibm.icu.text.MessageFormat;

/**
 * Represents a systemContributor extension. This class parses the
 * <code>systemExtension</code> extension declaration (in a plugin.xml) and all
 * other related extensions (e.g., <code>systemSections</code>) and holds the
 * configuration for further usage. Through static fields and methods, it also
 * acts as a registery of all such instances.
 * 
 */
public class SystemContributor {

	private static final String NO_TAB_ERROR = Messages.SystemContributor_Non_existing_tab;

	// Extenion point names, and the elements and attributes used in those
	// extension points
	private static final String EXTPT_SYSTEM_CONTRIBUTOR = "systemContributor"; //$NON-NLS-1$
	private static final String EXTPT_TABS = "systemTabs"; //$NON-NLS-1$
	private static final String EXTPT_SECTIONS = "systemSections"; //$NON-NLS-1$
	private static final String ELEMENT_TAB_CUSTOM = "systemTabCustom"; //$NON-NLS-1$
	private static final String ELEMENT_TAB_TASKS = "systemTabTasks"; //$NON-NLS-1$
	private static final String ELEMENT_SECTION = "systemSection"; //$NON-NLS-1$
	private static final String ELEMENT_SECTION_TASKS = "systemSectionTasks"; //$NON-NLS-1$
	private static final String ELEMENT_DEBUGGER = "debugger"; //$NON-NLS-1$
	private static final String ATT_DEBUGGER_ID = "debuggerId"; //$NON-NLS-1$
	public static final String ATT_SYSTEM_CONTRIBUTOR_ID = "contributorId"; //$NON-NLS-1$
	private static final String ATT_LABEL_PROVIDER = "labelProvider"; //$NON-NLS-1$
	private static final String ATT_ACTION_PROVIDER = "actionProvider"; //$NON-NLS-1$
	private static final String ATT_ID = "id"; //$NON-NLS-1$
	private static final String ATT_NAME = "name"; //$NON-NLS-1$
	private static final String ATT_ICON = "icon"; //$NON-NLS-1$

	/**
	 * The system contributor ID, as found in the XML
	 */
	protected String fSystemContributorId;

	/**
	 * A system contributor optionally provides a label provider that's
	 * exercised to convert the active context (the selection in the Debug view)
	 * to a displayable string. The string shows at the top-left of the tabbed
	 * page.
	 */
	protected ILabelProvider fLabelProvider;

	/**
	 * A system contributor optionally provides an action provider that it can
	 * use to contribute view actions when its tabbed page is the active one.
	 */
	protected IActionProvider fActionProvider;

	/**
	 * The system contributor's display name. Currently, this shows in the top
	 * left region of the view (directly above the tabs).
	 */
	protected String fName;

	/**
	 * The system contributor's icon (optional). This shows alongside the name.
	 */
	protected ImageDescriptor fIcon;

	/**
	 * The collection of tab descriptors associated with this system
	 * contributor. They are populated with their associated section descriptors
	 * prior to being put into this collection.
	 */
	protected volatile List<ITabDescriptor> fTabDescriptors;

	/**
	 * If a system contributor works with agent based debuggers (e.g., ones
	 * which use CW TRK), the it will list in its extension (XML) the ids of
	 * those debuggers. See the systemContributor extension schema description
	 * for further explanation.
	 */
	private Set<String> fDebuggers = new HashSet<>(3);

	/**
	 * The list of <i>instantiated</i> system contributors. There may be
	 * numerous in the Eclipse extension registry, but we only instantiate the
	 * ones that are requested at runtime.
	 */
	private static List<SystemContributor> sSystemContributors = new ArrayList<>(5);

	/**
	 * Returns the tab descriptors that apply to the given input (active debug
	 * context). A tab descriptor applies if one or more of its sections apply
	 * to the input.
	 */
	public List<ITabDescriptor> getTabDescriptors(Object input) {
		return filterTabDescriptors(getAllTabDescriptors(), input);
	}

	/**
	 * Accessor method. See fLabelProvider
	 */
	public ILabelProvider getLabelProvider() {
		return fLabelProvider;
	}

	/**
	 * Accessor method. See fName
	 */
	public String getName() {
		return fName;
	}

	/**
	 * Accessor method. See fIcon
	 */
	public ImageDescriptor getIcon() {
		return fIcon;
	}

	/**
	 * Accessor method. see fActionProvider
	 */
	public IActionProvider getActionProvider() {
		return fActionProvider;
	}

	/**
	 * Accessor method. See fSystemContributorId
	 */
	public String getId() {
		return fSystemContributorId;
	}

	/**
	 * Constructor. Protected as an instance is not supposed to be instantiated
	 * directly but through a <code>TabbedRegistryFactory</code>. // jjjjjjjjj
	 * 
	 * There is one singleton instance for each registered system contributor.
	 * Parses the extension points and gets all the attributes.
	 * 
	 * @param id
	 *            the ID of the system contributor
	 */
	protected SystemContributor(final IConfigurationElement configurationElement) {

		fSystemContributorId = configurationElement.getAttribute(ATT_ID);
		fName = configurationElement.getAttribute(ATT_NAME);

		// throw exception if required elements not found
		try {
			if (configurationElement.getAttribute(ATT_LABEL_PROVIDER) != null) {
				fLabelProvider = (ILabelProvider) configurationElement.createExecutableExtension(ATT_LABEL_PROVIDER);
			}

			if (configurationElement.getAttribute(ATT_ACTION_PROVIDER) != null) {
				fActionProvider = (IActionProvider) configurationElement.createExecutableExtension(ATT_ACTION_PROVIDER);
			}
		} catch (CoreException e) {
			SystemBrowserActivator.log(e.getStatus());
		}

		String iconStr = configurationElement.getAttribute(ATT_ICON);
		if (iconStr != null) {
			IExtension extension = configurationElement.getDeclaringExtension();
			fIcon = AbstractUIPlugin.imageDescriptorFromPlugin(extension.getNamespaceIdentifier(), iconStr);
		}

		IConfigurationElement[] debuggers = configurationElement.getChildren(ELEMENT_DEBUGGER);
		for (IConfigurationElement debugger : debuggers) {
			String debuggerId = debugger.getAttribute(ATT_DEBUGGER_ID);
			if (debuggerId != null) {
				fDebuggers.add(debuggerId);
			}
		}
	}

	/**
	 * Find the first system contributor associated with a given CDT-based
	 * debugger. Really, there should be one or none, but not multiple. But
	 * since we can't enforce that, we just ignore all but the first.
	 * 
	 * <p>
	 * A systemContributor extension lists debuggers only if it supports agent
	 * based (e.g., CW TRK) debuggers. For the other kind of debugger, JTAG, the
	 * OS contributor used in a debug session identifies the systemContributor
	 * and since the OS contributor is tied to a particular debugger, it makes
	 * no sense for the systemContributor to also identify the debugger
	 * association. For clarification, see the systemContributor extension
	 * schema description.
	 * 
	 * @param debuggerId
	 *            ID of the CDT-based debugger
	 * @return the SystemContributor instance or null if there's no registered
	 *         systemContributor extension for that debugger
	 */
	public static SystemContributor getSystemContributorForDebugger(String debuggerId) {
		// We look to see if we've already instantiated a system contributor
		// description that matches the input. If not, we look through all
		// system contributor extensions (XML) for a match, and instantiate
		// it if we find it.

		synchronized (sSystemContributors) {
			for (SystemContributor system : sSystemContributors) {
				if (system.fDebuggers.contains(debuggerId)) {
					return system;
				}
			}

			final IConfigurationElement[] systems = Platform.getExtensionRegistry()
					.getExtensionPoint(SystemBrowserActivator.getPluginId(), EXTPT_SYSTEM_CONTRIBUTOR).getConfigurationElements();
			for (IConfigurationElement system : systems) {
				IConfigurationElement[] debuggers = system.getChildren(ELEMENT_DEBUGGER);
				for (IConfigurationElement debugger : debuggers) {
					String thisDebuggerId = debugger.getAttribute(ATT_DEBUGGER_ID);
					if (thisDebuggerId != null //
							&& (thisDebuggerId.equals(debuggerId))) {
						SystemContributor syscontr = new SystemContributor(system);
						sSystemContributors.add(syscontr);
						return syscontr;
					}
				}
			}
		}

		return null;
	}

	/**
	 * Returns the ID of a the systemContributor associated with an
	 * osContributor. An osContributor specified one (and only one) system
	 * contributor.
	 * 
	 * @param osContributorId
	 *            the ID of the OS contributor
	 * @return the system contributor ID
	 */
	// public static String getSystemContributorIdForOS(String osContributorId)
	// {
	// final IConfigurationElement[] oses =
	// Platform.getExtensionRegistry().getExtensionPoint(CWPlugin.getPluginId(),
	// CWPlugin.OS_CONSTRIBUTOR_EXTENSION_POINT).getConfigurationElements();
	// for (IConfigurationElement os : oses) {
	// String thisOsContributorId = os.getAttribute(ATT_ID);
	// if (thisOsContributorId != null &&
	// thisOsContributorId.equals(osContributorId)) {
	// return
	// os.getAttribute(CWPlugin.OS_CONTRIBUTOR_OS_SYSTEM_CONTRIBUTOR_ID_ATTR);
	// }
	// }
	//
	// return null;
	// }

	/**
	 * Find the first system contributor with the given ID. Really, there should
	 * be one or none, but not multiple. But since we can't enforce that, we
	 * just ignore all but the first.
	 * 
	 * @param id
	 *            the system contributor ID
	 * @return the SystemContributor instance or null if there's no registered
	 *         systemContributor with that ID
	 */
	public static SystemContributor getSystemContributor(String id) {
		// We look to see if we've already instantiated a system contributor
		// description that matches the input. If not, we look through all
		// system contributor extensions (XML) for a match, and instantiate
		// it if we find it.

		synchronized (sSystemContributors) {
			for (SystemContributor system : sSystemContributors) {
				if (system.getId().equals(id)) {
					return system;
				}
			}

			final IConfigurationElement[] systems = Platform.getExtensionRegistry()
					.getExtensionPoint(SystemBrowserActivator.getPluginId(), EXTPT_SYSTEM_CONTRIBUTOR).getConfigurationElements();
			for (IConfigurationElement system : systems) {
				String systemContributorId = system.getAttribute(ATT_ID);
				if (systemContributorId != null && systemContributorId.equals(id)) {
					SystemContributor syscontr = new SystemContributor(system);
					sSystemContributors.add(syscontr);
					return syscontr;
				}
			}
		}
		return null;
	}

	/**
	 * Returns the section descriptors indirectly associated with this system
	 * contributor. I say "indirectly" because a section descriptor itself
	 * ("systemSection" in the XML) does not directly reference a system
	 * contributor. Rather, a higher level concept, an extension point called
	 * "systemExtensions" defines the set of section descriptors that apply to a
	 * particular system contributor. The same holds true for tab descriptors.
	 * So, the fact that we return a particular section descriptor here does not
	 * mean that the section will be shown to the user. It won't if the tab that
	 * section is directly associated with is not in turn associated with the
	 * system contributor via the "systemTabs" element).
	 * 
	 * An example: A System Browser plugin declares two sections: SecA and SecB.
	 * SecA is directly tied to a TabA, and SecB to a TabB. A "systemSections"
	 * extension declares that SecA and SecB are both tied to the Linux system
	 * contributor. Now a "systemTabs" extension declares that TabA is tied to
	 * the Linux system contributor, but that is not so for TabB. So, really, we
	 * can only expect the user to see SecA (in TabA). SecB will not see the
	 * light of day for a Linux KA session because its tab (TabB) will never see
	 * the light of day. Takes a little getting used to. So, in this scenario,
	 * we'll return SecA and SecB even though SecB won't ultimately apply. Of
	 * course, we're not being asked to make that determination; one of our
	 * callers presumably will.
	 */
	protected List<ISectionDescriptor> readSectionDescriptors() {
		final Set<ISectionDescriptor> result = new HashSet<>();
		final IConfigurationElement[] sectionGroups = getAssociatedExtensions(EXTPT_SECTIONS);
		for (final IConfigurationElement group : sectionGroups) {
			IConfigurationElement[] sections = group.getChildren(ELEMENT_SECTION);
			for (final IConfigurationElement section : sections) {
				result.add(new CustomSectionDescriptor(section));
			}
			sections = group.getChildren(ELEMENT_SECTION_TASKS);
			for (final IConfigurationElement section : sections) {
				result.add(new SimpleTableDataDescriptor(section));
			}

		}
		return new ArrayList<>(result);// .toArray(new
										// ISectionDescriptor[result.size()]);
	}

	/**
	 * Returns the extensions of a particular name that are associated with this
	 * system contributor. The association is assumed to be through an attribute
	 * called "contributorId" The elements are sorted by plugin prerequisite
	 * order.
	 * 
	 * @param extensionPointName
	 *            the simple identifier of a System Browser extension point
	 *            (e.g. "systemExtensions")
	 */
	protected IConfigurationElement[] getAssociatedExtensions(String extensionPointName) {
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(SystemBrowserActivator.getPluginId(), extensionPointName);

		IConfigurationElement[] extensions = point.getConfigurationElements();
		List<IConfigurationElement> elements = new ArrayList<>(extensions.length);

		for (IConfigurationElement extension : extensions) {
			if (extension.getName().equals(extensionPointName)) {
				String systemContributorId = extension.getAttribute(ATT_SYSTEM_CONTRIBUTOR_ID);
				if (systemContributorId != null) {
					if (fSystemContributorId.equals(systemContributorId)) {
						elements.add(extension);
					}
				}
			}
		}
		return elements.toArray(new IConfigurationElement[elements.size()]);
	}

	/**
	 * Filters out the tab descriptors that do not apply to the given input
	 * (active debug context).
	 * 
	 * @param descriptor
	 *            array of tab descriptors
	 * @param input
	 *            active debug context
	 * @return the updated tab descriptors array
	 */
	protected List<ITabDescriptor> filterTabDescriptors(final List<ITabDescriptor> descriptors, final Object input) {
		List<ITabDescriptor> result = new ArrayList<>();
		for (final ITabDescriptor descriptor : descriptors) {
			if (descriptor.showForInput(input)) {
				result.add(descriptor);
			}
		}
		return result;// .toArray(new TabDescriptor[result.size()]);
	}

	/**
	 * Reads property tab extensions. Returns all tab descriptors for the
	 * current contributor id or an empty array if none is found.
	 * 
	 * @return an array of tab descriptors
	 */
	protected List<ITabDescriptor> getAllTabDescriptors() {
		// Double-checked locking problem --
		// Please check out the following:
		// http://www.cs.umd.edu/~pugh/java/memoryModel/DoubleCheckedLocking.html
		// http://www.ibm.com/developerworks/library/j-jtp03304/
		// With the Java 5.0 memory model this can be fixed by making the
		// variable
		// volatile - it's not the best solution but is the one at hand
		if (fTabDescriptors == null) {
			synchronized (this) {
				if (fTabDescriptors == null) {
					List<ITabDescriptor> allTabs = readTabDescriptors();

					// There are two kinds of tabs: custom and built-in. The
					// built-in variety are somewhat of a black box. The System
					// Browser extension does not describe or let the user
					// dictate what sections are in a built in tab. So the
					// process of filling in a tab descriptor with section
					// descriptors applies to custom tabs only.
					// List<TabDescriptor> customTabs = new ArrayList<>(10);
					// for (TabDescriptor desc : allTabs) {
					// if (desc instanceof CustomTabDescriptor) {
					// customTabs.add(desc);
					// }
					// }
					if (allTabs.size() > 0) {
						populateWithSectionDescriptors(allTabs);
					}

					// Sort the tabs by their optional "afterTab" attribute
					List<ITabDescriptor> sortedList = allTabs;
					sortTabDescriptorsByAfterTab(sortedList);

					fTabDescriptors = sortedList;
				}
			}
		}
		return fTabDescriptors;
	}

	/**
	 * Returns a collection of all tab descriptors associated with this system
	 * contributor.
	 * 
	 * @return the collection of tab descriptors. The descriptors are NOT
	 *         singletons. Calling this a second time will instantiate a whole
	 *         new (equivalent) set of TabDescriptors
	 */
	protected List<ITabDescriptor> readTabDescriptors() {
		List<ITabDescriptor> result = new ArrayList<>();
		IConfigurationElement[] extensions = getAssociatedExtensions(EXTPT_TABS);

		for (IConfigurationElement tabs : extensions) {
			IConfigurationElement[] children = tabs.getChildren();
			for (IConfigurationElement child : children) {
				if (child.getName().equals(ELEMENT_TAB_CUSTOM)) {
					// A custom tab
					result.add(new CustomTabDescriptor(child));
				} else {
					// If it's not a custom tab, it has to be one of our built
					// in tabs (at this moment, we only have one--the Tasks tab)
					if (child.getName().equals(ELEMENT_TAB_TASKS)) {
						result.add(new TasksTabDescriptor(child));
					} // --------------------------------------------
						// add handling for future built in tabs here
						// --------------------------------------------
					else {// jjjjjjjjjjjjjjj give user an eror
					}
				}
			}
		}

		return Collections.synchronizedList(result);// .toArray(new
													// TabDescriptor[result.size()]);
	}

	/**
	 * Populates the given tab descriptors with the section descriptors that
	 * reference them
	 * 
	 * @param collection
	 *            of tab descriptors. Assumed the caller is giving us only tab
	 *            descriptors that are associated with this system contributor
	 * 
	 */
	protected void populateWithSectionDescriptors(final List<ITabDescriptor> tabDescriptors) {
		List<ISectionDescriptor> sections = readSectionDescriptors();
		for (final ISectionDescriptor section : sections) {
			appendSectionToApplicableTabs(section, tabDescriptors);
		}
	}

	/**
	 * Appends the specified section to the applicable tabs residing on the
	 * specified list. A section description directly references the tab it
	 * belongs to.
	 * 
	 * @param section
	 *            the section descriptor to be appended to the tab
	 * @param aTabDescriptors
	 *            the list of tab descriptors
	 */
	protected void appendSectionToApplicableTabs(final ISectionDescriptor section, final List<ITabDescriptor> tabDescriptors) {
		for (final ITabDescriptor tabDescriptor : tabDescriptors) {
			if (tabDescriptor.getId().equals(section.getTargetTab())) {
				((TabDescriptor) tabDescriptor).addSectionDescriptor(section);
				return;
			}
		}

		// The System Browser plugin specified sections that apply
		// to this system contributor but none of the tabs those sections refer
		// to
		// are associated with this system contributor! Almost certainly a goof.
		String message = MessageFormat.format(NO_TAB_ERROR, new Object[] { section.getId(), section.getTargetTab() });
		IStatus status = new Status(IStatus.ERROR, SystemBrowserActivator.getPluginId(), 0, message, null);
		SystemBrowserActivator.getPlugin().getLog().log(status);
	}

	/**
	 * Sorts the tab descriptors in the given list according to their (optional)
	 * "afterTab" attribute.
	 * 
	 * @param tabs
	 *            the list of tab descriptors, sorted in place
	 * @return
	 */
	protected void sortTabDescriptorsByAfterTab(List<ITabDescriptor> tabs) {
		if (tabs.size() > 0) {
			Collections.sort(tabs, new Comparator<Object>() {
				@Override
				public int compare(Object arg0, Object arg1) {
					TabDescriptor tab1 = (TabDescriptor) arg0;
					TabDescriptor tab2 = (TabDescriptor) arg1;

					if (tab2.fAfterTab.equals(tab1.getId())) {
						return -1;
					} else if (tab1.fAfterTab.equals(tab2.getId())) {
						return 1;
					} else {
						return 0;
					}
				}
			});
		}
	}
}
