/**
 * Copyright 2017-2019, 2021 NXP
 */
package com.nxp.swtools.periphs.gui.wizard;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.provider.configuration.sources.ISourceFileGenerationStatus.GenerationStatus;
import com.nxp.swtools.provider.configuration.sources.ISourceFileProviderWrapper;
import com.nxp.swtools.provider.configuration.sources.ISourceFileProviderWrapper.ISourceGenerateResult;
import com.nxp.swtools.provider.configuration.sources.SourceFileProviderFactory;
import com.nxp.swtools.provider.configuration.storage.ProcessorCore;
import com.nxp.swtools.utils.scripting.ScriptResult;
import com.nxp.swtools.utils.wizards.CannotExportWizardPage;
import com.nxp.swtools.utils.wizards.exportWizard.AExportSourcesWizard;

/**
 * Wizard used for exporting sources of the Peripherals tool.
 * @author Juraj Ondruska
 */
public class PeriphsExportSourcesWizard extends AExportSourcesWizard {

	/** Page to be displayed if it is not possible to export source files */
	private @Nullable CannotExportWizardPage cannotExportPage;

	/**
	 * Constructor.
	 */
	public PeriphsExportSourcesWizard() {
		super(UtilsText.safeString(Messages.get().PeriphsExportSourcesWizard_Title), "periphs_sources", SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL); //$NON-NLS-1$
		ScriptResult generatedCode = Controller.getInstance().waitAndGetGeneratedCode();
		if ((generatedCode != null) && generatedCode.containsAnyError()) {
			String wizardTitle = UtilsText.safeString(Messages.get().PeriphsExportSourcesWizard_CannotExportTitle);
			String wizardMessage = UtilsText.safeString(Messages.get().PeriphsExportSourcesWizard_CannotExportMessage);
			cannotExportPage = new CannotExportWizardPage(wizardTitle, wizardMessage);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.wizards.exportWizard.AExportWizard#generateFiles()
	 */
	@Override
	public GenerationResult generateFiles() {
		return GenerationResult.fail("This wizard should use generateCoreFiles() method."); //$NON-NLS-1$
		//@see generateCoreFiles()
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.wizards.exportWizard.AExportSourcesWizard#generateCoreFiles(com.nxp.swtools.provider.configuration.storage.common.StorageCore, com.nxp.swtools.provider.configuration.storage.ProcessorCore, java.lang.String)
	 */
	@Override
	protected GenerationResult generateCoreFiles(@NonNull ProcessorCore coreInfo, @NonNull String destination) {
		Controller controller = Controller.getInstance();
		controller.refreshIfChangeInOtherTool();
		controller.waitUntilRegenerated();

		// get also derived sources (currently for MQX App builder), so do not call directly the Peripherals tool
		ISourceFileProviderWrapper wrapper = SourceFileProviderFactory.getSourceFileProvidersWrapper(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL);
		if (wrapper != null) {
			ISourceGenerateResult wrapperResult = wrapper.generate(coreInfo.getID(), true);
			if (wrapperResult.getStatus() == GenerationStatus.FATAL) {
				return GenerationResult.fail(wrapperResult.getMessage());
			}
			return GenerationResult.success(wrapperResult.getAllGeneratedFiles());
		}
		assert false;
		return GenerationResult.fail("Internal error - no source file provider"); //$NON-NLS-1$
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.wizards.exportWizard.AExportWizard#addPages()
	 */
	@Override
	public void addPages() {
		ScriptResult generatedCode = Controller.getInstance().waitAndGetGeneratedCode();
		if ((generatedCode != null) && generatedCode.containsAnyError()) {
			addPage(cannotExportPage);
		} else {
			super.addPages();
		}
	}
}
