/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.provider;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.logging.Logger;

import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.PeripheralsProblem;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.gui.view.componentsettings.GlobalComponentSettingView;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IConfigSetConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IProblemOpenHandler;
import com.nxp.swtools.validation.engine.IProblemOpenProvider;

/**
 * Class provides open handlers for peripherals problems.
 * @author Juraj Ondruska
 */
public class PeripheralsProblemOpenProvider implements IProblemOpenProvider {
	/** Logger of the class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(PeripheralsProblemOpenProvider.class);
	/** The handlers */
	private static final @NonNull Collection<@NonNull IProblemOpenHandler> HANDLERS =
			CollectionsUtils.asList(new PeripheralsProblemOpenHandler());

	/* (non-Javadoc)
	 * @see com.nxp.swtools.validation.engine.IProblemOpenProvider#getOpenHandlers(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public @NonNull Collection<@NonNull IProblemOpenHandler> getOpenHandlers(@NonNull IBaseProblem problem) {
		if ((problem instanceof PeripheralsProblem) && (((PeripheralsProblem) problem).getComponentType() != null)) {
			// return handlers only for peripherals problem which has the component type
			return HANDLERS;
		} else {
			return CollectionsUtils.emptyList();
		}
	}

	/**
	 * Class used for opening view for peripherals problems.
	 * @author Juraj Ondruska
	 */
	private static class PeripheralsProblemOpenHandler implements IProblemOpenHandler {
		
		/**
		 * Constructor.
		 */
		PeripheralsProblemOpenHandler() {
			// Do not allow instantiating from outside
		}

		/* (non-Javadoc)
		 * @see com.nxp.swtools.validation.engine.IProblemOpenHandler#openProblem(com.nxp.swtools.validation.engine.IBaseProblem)
		 */
		@Override
		public void openProblem(@NonNull IBaseProblem problem) {
			if (problem instanceof PeripheralsProblem) {
				openTool(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL);
				PeripheralsProblem peripheralsProblem = (PeripheralsProblem) problem;
				IComponentInstanceConfig instance = peripheralsProblem.getChild().getChildContext().getComponentInstanceConfig();
				String componentType = peripheralsProblem.getComponentType();
				String componentName = peripheralsProblem.getComponentInstanceName();
				String functionalGroupName = peripheralsProblem.getFunctionalGroupName();
				if (componentType != null) {
					// Switch functional group when needed. Fires listeners for functional group change event
					IFunctionalGroup group = Controller.getInstance().getProfile().getFunctionalGroup(UtilsText.safeString(functionalGroupName));
					if (group != null) {
						if (!Controller.getInstance().getFunctionalGroup().getUiName().equals(functionalGroupName)) {
							// If current functional group is different then switch to group of this problem
							Controller.getInstance().setFunctionalGroup(group, this);
						}
					}
					// This opens required component setting view after functional group switches and everything is ready
					Display display = Display.getCurrent();
					assert display != null;
					display.asyncExec(new Runnable() {
						@Override
						public void run() {
							IWorkbenchWindow activeWorkbenchWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
							if (activeWorkbenchWindow != null) {
								IWorkbenchPage activePage = activeWorkbenchWindow.getActivePage();
								if (activePage != null) {
									if (isChildPresentInGlobalConfigSet(peripheralsProblem.getChild().getId())) {
										GlobalComponentSettingView.open(activePage, componentType, peripheralsProblem.getChild());
									} else {
										if (instance != null) {
											ComponentSettingView.open(activePage, instance.getUUID(), true, peripheralsProblem.getChild());
										} else {
											ComponentSettingView.open(activePage, componentType, UtilsText.safeString(componentName), componentName == null, true, peripheralsProblem.getChild());
										}
									}
								} else {
									LOGGER.warning("[TOOL] Cannot open component setting view because there is no active page"); //$NON-NLS-1$
								}
							} else {
								LOGGER.warning("[TOOL] Cannot open component setting view because there is no active workbench window"); //$NON-NLS-1$
							}
						}
					});
				}
			} else {
				LOGGER.warning(MessageFormat.format("[TOOL] {0} accepts only {1}", PeripheralsProblemOpenHandler.class.getSimpleName(), PeripheralsProblem.class.getSimpleName())); //$NON-NLS-1$
			}
		}

		/* (non-Javadoc)
		 * @see com.nxp.swtools.validation.engine.IProblemOpenHandler#getTargetToolId()
		 */
		@Override
		public @Nullable String getTargetToolId() {
			return SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL;
		}

		/**
		 * Method used to check if a setting is part of a global config set or not
		 * @param settingId - id of the setting to check if is a part of a global config set
		 * @return <code>true</code> if the setting with the given id is part of a global config set, <code>false</code> otherwise
		 */
		protected static boolean isChildPresentInGlobalConfigSet(String settingId) {
			Collection<IConfigSetConfig> globalConfigSets = Controller.getInstance().getProfile().getGlobalConfigSets().values();
			for (IConfigSetConfig cfgSet : globalConfigSets) {
				if (cfgSet.getAllStorableSettings().stream().anyMatch(setting -> setting.getId().equals(settingId))) {
					return true;
				}
			}
			return false;
		}
		
	}
	
}
