/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;


import java.util.ArrayList;
import java.util.logging.Level;

import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig.Type;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Class representing tabular representation of an array config with items represented as columns.
 * @author Juraj Ondruska
 * @author Marek Ciz
 */
public class ArrayControlTabularHorizontal extends AArrayControlTabular {

	/** Label for indices */
	private static final String INDICES_LABEL = "#"; //$NON-NLS-1$
	/** Offset for header column in column indexing */
	private static final int OFFSET_HEADER_COLUMN = 1;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlTabularHorizontal(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#tableViewerHandling(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void tableViewerHandling(@NonNull TableViewer tableViewerLoc) {
		final Table table = tableViewerLoc.getTable();
		setTableLayout(table, arrayConfig.getId(), false);
		table.setHeaderVisible(false);
		createTableViewerHeaderColumn(tableViewerLoc);
		for (int column = 0; column < children.size(); column++) {
			createTableViewerColumnHorizontal(column, tableViewerLoc);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#getMenuContext(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected @NonNull ArrayControlItemMenuContext getMenuContext(@NonNull TableViewer tableViewerLoc) {
		return new ArrayControlItemMenuContextHorizontal(this, tableViewerLoc);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#updateTableViewer(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void updateTableViewer(@NonNull TableViewer viewerLoc) {
		// check, if there are enough created columns to be populated with all the children
		int childrenSize = children.size();
		int columnCount = viewerLoc.getTable().getColumnCount() - OFFSET_HEADER_COLUMN;
		if (childrenSize > columnCount) {
			// create columns for newly added children
			for (int index = columnCount; index < childrenSize; index++) {
				createTableViewerColumnHorizontal(index, viewerLoc);
			}
		} else if (childrenSize < columnCount) {
			// dispose columns for newly added children
			for (int index = columnCount; index > childrenSize; index--) {
				viewerLoc.getTable().getColumn(index).dispose();
			}
		}
		// transform the children from horizontal to vertical lists
		ArrayControlTabularHorizontalProvider transformedChildrenProvider = new ArrayControlTabularHorizontalProvider(arrayConfig, children);
		// input updated data
		viewerLoc.setInput(transformedChildrenProvider.getHorizontalItems());
		updateColumnHeaders(viewerLoc);
		// update header size
		final TableColumn headerColumn = viewerLoc.getTable().getColumn(0);
		headerColumn.pack();
		// update table window size
		if (contentContainer != null) {
			contentContainer.requestLayout();
		}
	}

	/**
	 * Updates column headers with status in tooltip and status icon image
	 * @param viewer which should be updated
	 */
	private void updateColumnHeaders(TableViewer viewer) {
		for (int i = 0; i < children.size(); i++) {
			IChildControl childControl = children.get(i);
			IChild childOfArray = childControl.getChild();
			int level = -1;
			String status = null;
			if ((status = childOfArray.getError()) != null) {
				level = ErrorLevels.LEVEL_ERROR;
			} else if ((status = childOfArray.getWarning()) != null) {
				level = ErrorLevels.LEVEL_WARNING;
			} else if ((status = childOfArray.getInfo()) != null) {
				level = ErrorLevels.LEVEL_INFORMATION;
			}
			Image image = null;
			if (level >= 0) {
				image = ToolsImages.getStatusIcon(level);
			}
			int columnIndex = i + OFFSET_HEADER_COLUMN;
			if (viewer.getTable().getColumnCount() <= columnIndex) {
				LOGGER.log(Level.SEVERE, "[TOOL] Trying to update column {0} which does not exist", String.valueOf(columnIndex)); //$NON-NLS-1$
				continue;
			}
			TableColumn column = viewer.getTable().getColumn(columnIndex);
			column.setImage(image);
			column.setToolTipText(status);
		}
	}

	/**
	 * Provides current Setting of the cell
	 * @param itemLoc includes SettingsConfig, removeButton flag, and header names
	 * @param columnPosition column index in the setting table
	 * @return current Setting of the cell
	 */
	@Nullable IChild getChildFromItem(@NonNull TabularHorizontalItem itemLoc, int columnPosition){
		IChild currentChild = null;
		// current setting is extracted from list of @Nonnull ISettings, so there should not be a null ISetting
		if (itemLoc.getChildConfigs().size() > columnPosition) {
			currentChild = itemLoc.getChildConfigs().get(columnPosition);
		}
		return currentChild;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getMoveDownDescription()
	 */
	@Override
	protected @NonNull String getMoveDownDescription() {
		return Messages.get().ArrayControl_ItemMenu_MoveRight;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getMoveUpButtonDescription()
	 */
	@Override
	protected @NonNull String getMoveUpButtonDescription() {
		return Messages.get().ArrayControl_ItemMenu_MoveLeft;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getMoveUpIcon()
	 */
	@Override
	protected @Nullable Image getMoveUpIcon() {
		return ToolsImages.getImage(IToolsImages.ICON_LEFT);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getMoveDownIcon()
	 */
	@Override
	protected @Nullable Image getMoveDownIcon() {
		return ToolsImages.getImage(IToolsImages.ICON_RIGHT);
	}

	/**
	 * Create table column from given configuration in given Horizontal table
	 * @param columnPosition current position of the column
	 * @param tableViewerLoc tableViewer in which the new column will be created
	*/
	protected void createTableViewerColumnHorizontal(int columnPosition, @NonNull TableViewer tableViewerLoc) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewerLoc, SWT.NONE);
		viewerColumn.getColumn().addControlListener(new ControlAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				updateScrollSize();
			}
		});
		viewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String label = null;
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild != null) && (element instanceof TabularHorizontalItem)) {
					if (((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.INDICES_HEADER_IDENTIFIER)) {
						return currentChild.getUiName(); // Indices row -> index of setting in array
					}
					if (((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.STATE_HEADER_IDENTIFIER)) {
						return null; // State row -> Nothing
					}
				}
				if ((currentChild instanceof ScalarConfig)) {
					if (currentChild.isAvailable()) {
						ScalarConfig currentSettingLoc = (ScalarConfig) currentChild;
						label = AArrayControlTabular.getText(currentSettingLoc);
					} else {
						label = NOT_APPLICABLE;
					}
				}
				return label;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public Font getFont(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild != null) && (!currentChild.isAvailable())) {
					return italicFont;
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild != null) && (element instanceof TabularHorizontalItem)) {
					if (((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.INDICES_HEADER_IDENTIFIER)) {
						return null; // Indices row does not have status image
					}
				}
				if (currentChild != null) {
					Image image = getImageOfLinkedSettingState(currentChild);
					if (image != null) {
						return image;
					}
					return AArrayControlTabular.getImage(currentChild, getControlUpdateType());
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					if (!currentChild.isEnabled()) {
						return DISABLED_TEXT_COLOR;
					}
					if (currentChild.isOptionSet(SettingOptions.UI_SETTING_LINK)) {
						if (!currentChild.isOptionAvailable(SettingOptions.UI_SETTING_LINK)) {
							return DISABLED_TEXT_COLOR;
						}
						return LINK_COLOR;
					}
				}
				return super.getForeground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getBackground(java.lang.Object)
			 */
			@Override
			public Color getBackground(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					if ((element instanceof TabularHorizontalItem)) {
						if (((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.INDICES_HEADER_IDENTIFIER)) {
							return AArrayControlTabular.getBackgroundColor(currentChild);
						}
						if (((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.STATE_HEADER_IDENTIFIER)) {
							return super.getBackground(element);
						}
					}
					return AArrayControlTabular.getBackgroundColor(currentChild);
				}
				return super.getBackground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					String status = null;
					status = currentChild.getError();
					if (status == null) {
						status = currentChild.getWarning();
					}
					if (status == null) {
						status = currentChild.getInfo();
					}
					return status;
				}
				return tooltip;
			}
		});
		addEditingSupport(tableViewerLoc, viewerColumn, columnPosition);
		// set title and width to a column
		final TableColumn column = viewerColumn.getColumn();
		// hide or show column numbers in the top
		column.pack();
		setColumnWidth(column, columnPosition + 1);
		if (column.getWidth() > AArrayControlTabular.COLUMN_MAX_WIDTH) {
			column.setWidth(AArrayControlTabular.COLUMN_MAX_WIDTH);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportSelectChild(java.lang.Object, int)
	 */
	@Override
	protected IChild editingSupportSelectChild(Object element, int index) {
		setSelectedChild(getChildren().get(index));
		return getChild(element, index);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportPostAction(org.eclipse.jface.viewers.TableViewer, org.eclipse.jface.viewers.TableViewerColumn)
	 */
	@Override
	protected void editingSupportPostAction(@NonNull TableViewer viewer, @NonNull TableViewerColumn viewerColumn) {
		selectChildOfColumn(viewer, viewerColumn.getColumn());
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportCanEdit(java.lang.Object, int)
	 */
	@Override
	protected boolean editingSupportCanEdit(@NonNull Object element, int index) {
		boolean canEdit = true;
		if ((element instanceof TabularHorizontalItem)
				&& (((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.INDICES_HEADER_IDENTIFIER)
				|| ((TabularHorizontalItem) element).getHeaderName().equals(ArrayControlTabularHorizontalProvider.STATE_HEADER_IDENTIFIER))) {
			// For state and indices row do not allow editing of config it is pointing to
			return false;
		}
		IChild currentChild = getChild(element, index);
		if (currentChild != null) {
			canEdit = isCellEnabled(currentChild);
		}
		return canEdit;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportGetValue(java.lang.Object, int)
	 */
	@Override
	protected @Nullable Object editingSupportGetValue(@NonNull Object element, int index) {
		String selectedItem = null;
		IChild currentChild = getChild(element, index);
		if ((currentChild != null) && (currentChild.isAvailable())) {
			if (currentChild instanceof ScalarConfig) {
				selectedItem = AArrayControlTabular.getText((ScalarConfig) currentChild);
			}
		}
		return selectedItem;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportSetValue(org.eclipse.jface.viewers.TableViewer, java.lang.Object, int, java.lang.Object)
	 */
	@Override
	protected void editingSupportSetValue(@NonNull TableViewer tableViewerLoc, @NonNull Object element, int index, @Nullable Object userInputValue) {
		IChild currentChild = getChild(element, index);
		if (currentChild instanceof ScalarConfig) {
			ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
			// instant search list options
			if (currentSettingLoc.getType().equals(Type.ENUM)) {
				// Handled by listener of combo box in {@link #getCellEditor}
			} else if (currentSettingLoc.getType().equals(Type.INTEGER) || currentSettingLoc.getType().equals(Type.FLOAT)
					|| currentSettingLoc.getType().equals(Type.STRING)) {
				controllerWrapper.getController().setValue(currentSettingLoc,
						(userInputValue == null) ? UtilsText.EMPTY_STRING : UtilsText.safeString(userInputValue.toString()), tableViewerLoc);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportGetChildControl(java.lang.Object, int)
	 */
	@Override
	protected IChildControl editingSupportGetChildControl(@NonNull Object element, int index) {
		return getChildren().get(index);
	}

	/**
	 * Creates header Column
	 * @param tableViewerLoc table's table viewer
	 */
	protected void createTableViewerHeaderColumn(@NonNull TableViewer tableViewerLoc) {
		final TableViewerColumn headerViewerColumn = new TableViewerColumn(tableViewerLoc, SWT.NONE);
		final TableColumn headerColumn = headerViewerColumn.getColumn();
		headerColumn.addControlListener(new ControlAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				updateScrollSize();
			}
		});
		headerColumn.setAlignment(SWT.LEFT);
		headerViewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				if (element instanceof TabularHorizontalItem) {
					TabularHorizontalItem item = (TabularHorizontalItem) element;
					if (item.getHeaderName().equals(ArrayControlTabularHorizontalProvider.INDICES_HEADER_IDENTIFIER)) {
						return INDICES_LABEL; // Indices row - Indices to display
					}
					if (item.getHeaderName().equals(ArrayControlTabularHorizontalProvider.STATE_HEADER_IDENTIFIER)) {
						return UtilsText.EMPTY_STRING; // State row - Nothing to display
					}
					return item.getHeaderName();
				}
				return UtilsText.EMPTY_STRING;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public Font getFont(Object element) {
				return boldFont;
			}
		});
		if (headerColumn.getWidth() > AArrayControlTabular.COLUMN_MAX_WIDTH) {
			headerColumn.setWidth(AArrayControlTabular.COLUMN_MAX_WIDTH);
		}
	}

	/**
	 * Get proper child from the element.
	 * @param element from the table's input
	 * @param column of the child
	 * @return child or {@code null} if there is no child for the current column in the element
	 */
	protected @Nullable IChild getChild(@Nullable Object element, int column) {
		if (element instanceof TabularHorizontalItem) {
			TabularHorizontalItem item = (TabularHorizontalItem) element;
			if (item.getChildConfigs() instanceof ArrayList<?>) {
				return getChildFromItem(item, column);
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getSelection()
	 */
	@Override
	public @Nullable ISettingConfig getSelection() {
		IChildControl selectedChildLoc = getSelectedChild();
		if (selectedChildLoc != null) {
			return super.getChildFromSelection(new StructuredSelection(selectedChildLoc));
		}
		return null;
	}
}
