/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.logging.Logger;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.ui.utils.editors.FileSelectionCellEditor;
import com.nxp.swtools.common.ui.utils.editors.InstantSearchCellEditor;
import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.os.OSDetect;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.controller.UiValueIdHolder;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlFactory;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.gui.view.componentsettings.ScalarInfoControl;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig.Type;
import com.nxp.swtools.resourcetables.model.config.SetConfig.SetPresence;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.data.setting.DynamicEnumSetting.CustomValueSupport;
import com.nxp.swtools.resourcetables.model.data.setting.SetSetting.CustomItem;
import com.nxp.swtools.resourcetables.model.validation.ValidationHelper;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableImplementation;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;
import com.nxp.swtools.utils.view.ToolView;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.utils.ValidationProblemsMenuHelper;

/**
 * Class represents common base for each of the tabular array GUI representations (vertical/horizontal).
 * @author Juraj Ondruska
 * @author Marek Ciz
 */
public abstract class AArrayControlTabular extends AArrayControlGroup {
	/** Width of decorator column in pixels */
	private static final int DECORATOR_COLUMN_WIDTH = 20;
	/** Logger of the class */
	protected static final @NonNull Logger LOGGER = LogManager.getLogger(AArrayControlTabular.class);
	/** Specifies the minimum width of table column */
	protected static final int INDICES_COLUMN_MIN_WIDTH = 50;
	/** Specifies the maximum width of table column */
	protected static final int COLUMN_MAX_WIDTH = 180;
	/** string for cell value */
	protected static final @NonNull String NOT_APPLICABLE = "N/A"; //$NON-NLS-1$
	/** Index used in function {@link #getSettingAt(int, int)} for returning the item's setting */
	public static final int INDEX_FOR_RETURNING_THE_ITEMS_SETTING = -1;
	/** Icon for array menu */
	static final @Nullable Image ICON_ARRAY_MENU = ToolsImages.getImage(IToolsImages.ICON_ARRAY_MENU);
	/** Foreground color of uneditable cells */
	static final @Nullable Color DISABLED_TEXT_COLOR = SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
	/** Background color of disabled cells */
	static final @Nullable Color DISABLED_COLOR = SwToolsColors.getColor(SwToolsColors.DISABLED_BG);
	/** Foreground color for link simulation */
	static final @Nullable Color LINK_COLOR = SwToolsColors.getColor(SwToolsColors.LINK_FG);
	/** Error background decorator color */
	static final @Nullable Color ERROR_COLOR = SwToolsColors.getColor(SwToolsColors.ERROR_BG);
	/** Warning background decorator color */
	static final @Nullable Color WARNING_COLOR = SwToolsColors.getColor(SwToolsColors.WARNING_FG);
	/** Header cell font */
	protected @Nullable Font boldFont;
	/** "Not available" cell font */
	protected @Nullable Font italicFont;
	/** Table viewer showing settings table */
	protected @Nullable TableViewer tableViewer;
	/** Composite for containing whole tables */
	protected @Nullable Composite contentContainer = null;
	/** The scrolled composite in which the table is located */
	protected @Nullable ScrolledComposite scrolledComposite;
	/** Parent composite of table */
	protected @Nullable Composite tableComposite;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected AArrayControlTabular(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/**
	 * Get all non-structured setting configs from the child (if the child is non-structured then the child itself is returned).
	 * This method works recursively.
	 * @param rootChild to obtain non-structured setting configs from
	 * @return all non-structured setting configs
	 */
	public static @NonNull List<@NonNull IChild> getSettingsFlat(@NonNull IChild rootChild) {
		List<@NonNull IChild> result = CollectionsUtils.emptyList();
		if (rootChild instanceof IChildProvidable) {
			IChildProvidable childProvidable = (IChildProvidable) rootChild;
			if (childProvidable instanceof ArrayConfig) {
				LOGGER.severe("[DATA] Nested arrays are not allowed for displaying array in a table."); //$NON-NLS-1$
			} else {
				result = new ArrayList<>();
				for (IChild subChild : childProvidable.getChildren()) {
					result.addAll(getSettingsFlat(subChild));
				}
			}
		} else {
			// do not add Variable type ScalarConfigs
			if (!(rootChild instanceof ScalarConfig) || (((ScalarConfig) rootChild).getType() != Type.VARIABLE)) {
				result = CollectionsUtils.asList(rootChild);
			}
		}
		return result;
	}

	/**
	 * Get setting at the given position.
	 * @param itemNum numer of item
	 * @param idx number of setting in the item or -1 for the item's setting itself (typically a structure)
	 * @return the setting or {@code null} if any setting was found at the given position
	 */
	public @Nullable IChild getSettingAt(int itemNum, int idx) {
		if ((itemNum >= 0) && (itemNum < children.size())) {
			if (idx == INDEX_FOR_RETURNING_THE_ITEMS_SETTING) {
				return children.get(itemNum).getChild();
			}
			List<@NonNull IChild> itemSettings = getSettingsFlatVisible(children.get(itemNum).getChild());
			if ((idx >= 0) && (idx < itemSettings.size())) {
				return itemSettings.get(idx);
			}
		}
		 return null;
	}

	/**
	 * Returns flat list of settings in the given child that are available, enabled and not hidden
	 * @param rootChild from which the settings will be obtained
	 * @return list of settings that are visible to the user 
	 */
	public static @NonNull List<@NonNull IChild> getSettingsFlatVisible(@NonNull IChild rootChild) {
		return getSettingsFlat(rootChild).stream()
				.filter(x -> x.isAvailable())
				.filter(x -> x.isEnabled())
				.filter(x -> !x.isOptionAvailable(SettingOptions.UI_CONTROL_HIDDEN))
				.filter(x -> !x.isOptionAvailable(SettingOptions.UI_NOT_VISIBLE_PERMANENT))
				.collect(CollectorsUtils.toList());
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#setAllSettingsTo(com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlItemMenu, com.nxp.swtools.periphs.gui.view.componentsettings.internal.IArrayControlItemMenuControl)
	 */
	@Override
	public void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		IChild selectedSetting = control.getSelectedSettingHint();
		int idx = -1;
		List<@NonNull List<@NonNull IChild>> flattenedChildren = new ArrayList<>();
		for (IChildControl childControl : getChildren()) {
			List<@NonNull IChild> settingsFlat = getSettingsFlat(childControl.getChild());
			int idxOfSetting = settingsFlat.indexOf(selectedSetting);
			if (idxOfSetting >= 0) {
				idx = idxOfSetting;
			}
			flattenedChildren.add(settingsFlat);
		}
		if (idx >= 0) {
			List<@NonNull IChild> settings = new ArrayList<>();
			for (@NonNull List<@NonNull IChild> childrenOfItem : flattenedChildren) {
				if (idx < childrenOfItem.size()) {
					IChild setting = childrenOfItem.get(idx);
					if (setting.isAvailable() && setting.isEnabled()) {
						settings.add(setting);
					}
				}
			}
			if (selectedSetting != null) {
				ProgressUtils.run((m) -> {
					controllerWrapper.getController().setAllSettingsTo(UtilsText.safeToString(selectedSetting.getValue()), settings, getChild(), this);
				});
			}
		}
	}

	/**
	 * Creates column of Indices numbers
	 * @param viewer table viewer of the table
	 */
	protected void createDecoratorTableViewerColumn(@NonNull TableViewer viewer) {
		// first column in the table with left align
		final TableViewerColumn indicesViewerColumn = new TableViewerColumn(viewer, SWT.LEFT, 0);
		final TableColumn decoratorColumn = indicesViewerColumn.getColumn();
		decoratorColumn.setWidth(DECORATOR_COLUMN_WIDTH);
		decoratorColumn.setAlignment(SWT.LEFT);
		decoratorColumn.setText(UtilsText.EMPTY_STRING);
		indicesViewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				if (element instanceof IChildControl) {
					return AArrayControlTabular.getImage(((IChildControl) element).getChild(), getControlUpdateType());
				}
				return super.getImage(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				if (element instanceof IChildControl) {
					tooltip = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(((IChildControl) element).getChild()));
				}
				return tooltip;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(@NonNull Object element) {
				return null;
			}
		});
	}

	/**
	 * @param currentChild current child with settings
	 * @return {@code true}, if the table cell (element) is enabled
	 */
	static boolean isCellEnabled(@NonNull IChild currentChild) {
		return currentChild.isAvailable() && currentChild.isEnabled();
	}

	/**
	 * Paints cells background according to errors, warnings info and availability of a child.
	 * @param currentChild current setting in the current cell
	 * @return background color
	 */
	protected static @Nullable Color getBackgroundColor(@NonNull IChild currentChild) {
		// yellow cell background if there is a warning
		if (currentChild.getWarning() != null) {
			return WARNING_COLOR;
		}
		// red cell background if there is an error
		if (currentChild.getError() != null) {
			return ERROR_COLOR;
		}
		return null;
	}

	/**
	 * set Layout of table viewer's table
	 * @param table table from table viewer
	 * @param arrayConfigId Test ID for the table
	 * @param isVerticalTable flag, informs us if the table is in vertical table or not
	 */
	static void setTableLayout(@NonNull Table table, @NonNull String arrayConfigId, boolean isVerticalTable) {
		SWTFactoryProxy.INSTANCE.setTestId(table, TestIDs.PERIPHS_COMPONENT_SETTINGS_TABLE + UtilsText.UNDERSCORE + arrayConfigId);
		table.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, true));
		if (isVerticalTable) {
			table.setHeaderVisible(true);
		}
		table.setLinesVisible(true);
	}

	/**
	 * Creates checked or unchecked check box decorated icon with error, warning, and info
	 * add no decorator, if there are no errors, warnings, or info
	 * @param currentChild current setting of the table cell
	 * @param isChecked flag is check box is checked
	 * @param isEnabled flag if check box is enabled
	 * @return decorated check box icon
	 */
	static @Nullable Image getDecoratedCheckBoxIcon(@NonNull IChild currentChild, boolean isChecked, boolean isEnabled) {
		int problemLevel = ErrorLevels.LEVEL_SUCCESS;
		String iconName = null;
		if (isChecked) {
			iconName = isEnabled ? IToolsImages.ICON_CHECKED_LEFT_TRANSPARENT_PIXEL : IToolsImages.ICON_CHECKED_DISABLED_LEFT_TRANSPARENT_PIXEL;
		} else {
			iconName = isEnabled ? IToolsImages.ICON_UNCHECKED_LEFT_TRANSPARENT_PIXEL : IToolsImages.ICON_UNCHECKED_DISABLED_LEFT_TRANSPARENT_PIXEL;
		}
		if (currentChild.getError() != null) {
			problemLevel = ErrorLevels.LEVEL_ERROR;
		} else if (currentChild.getWarning() != null) {
			problemLevel = ErrorLevels.LEVEL_WARNING;
		} else if (currentChild.getInfo() != null) {
			problemLevel = ErrorLevels.LEVEL_INFORMATION;
		}
		return ToolView.getDecoratedImage(iconName, problemLevel);
	}

	/**
	 * Return image of linked setting state if given setting contains the link to other setting
	 * @param currentChild which may contain link to other setting
	 * @return status image when given setting contains linked setting or {@code null} when such linked setting does not have any state or no setting is linked
	 */
	protected @Nullable Image getImageOfLinkedSettingState(@NonNull IChild currentChild) {
		if (currentChild.isOptionSet(SettingOptions.UI_SETTING_LINK) && currentChild.isOptionAvailable(SettingOptions.UI_SETTING_LINK)) {
			Object optionValue = currentChild.getOptionValue(SettingOptions.UI_SETTING_LINK);
			if (optionValue instanceof IChild) {
				IChild linkedChild = (IChild) optionValue;
				return AArrayControlTabular.getImage(linkedChild, getControlUpdateType());
			}
		}
		return null;
	}

	/**
	 * Return image instead of check-box widget
	 * @param currentChild current child in table
	 * @param type control update type used for enabling/disabling the element
	 * @return result icon which will be displayed instead of check box
	 */
	protected static @Nullable Image getImage(@NonNull IChild currentChild, UpdateType type) {
		@Nullable Image shownIcon = null;
		boolean enabled = (type != UpdateType.FORCE_DISABLE) && currentChild.isEnabled();
		if (currentChild instanceof SetPresence) {
			SetPresence setPresence = (SetPresence) currentChild;
			shownIcon = getDecoratedCheckBoxIcon(currentChild, setPresence.getBoolValue(), enabled);
		} else if ((currentChild instanceof ScalarConfig) && ((ScalarConfig) currentChild).getType().equals(Type.BOOL)) {
			boolean isChecked = currentChild.getValue().equals(Boolean.TRUE) ? Boolean.TRUE.booleanValue() : Boolean.FALSE.booleanValue();
			shownIcon = getDecoratedCheckBoxIcon(currentChild, isChecked, enabled);
		} else {
			List<@NonNull IValidationProblem> errors = ValidationHelper.getSettingValidationProblems(currentChild, ErrorLevels.LEVEL_FATAL);
			errors.addAll(ValidationHelper.getSettingValidationProblems(currentChild, ErrorLevels.LEVEL_ERROR));
			List<@NonNull IValidationProblem> warnings = ValidationHelper.getSettingValidationProblems(currentChild, ErrorLevels.LEVEL_WARNING);
			List<@NonNull IBaseProblem> problems = new ArrayList<>( errors.size() + warnings.size());
			problems.addAll(warnings);
			problems.addAll(errors);
			List<@NonNull IBaseProblem> mergedProblems = new ArrayList<>(problems.size());
			ValidationProblemsMenuHelper.mergeProblems(problems, mergedProblems);
			String baseIcon = IToolsImages.ICON_EMPTY_21P;
			if (!mergedProblems.isEmpty()) {
				baseIcon = IToolsImages.ICON_AUTO_RESOLVE_PROBLEM;
			}
			if ((currentChild.getError() != null) || !errors.isEmpty()) {
				shownIcon = ToolView.getDecoratedImage(baseIcon, ErrorLevels.LEVEL_ERROR);
			} else if ((currentChild.getWarning() != null) || !warnings.isEmpty()) {
				shownIcon = ToolView.getDecoratedImage(baseIcon, ErrorLevels.LEVEL_WARNING);
			} else if (currentChild.getInfo() != null) {
				shownIcon = ToolView.getDecoratedImage(baseIcon, ErrorLevels.LEVEL_INFORMATION);
			}
		}
		return shownIcon;
	}

	/**
	 * Return tooltip of given child
	 * @param currentChild in table
	 * @return tooltip of given child
	 */
	protected static String getToolTip(IChild currentChild) {
		ToolTipableImplementation impl = new ToolTipableImplementation(currentChild);
		if (currentChild.isOptionSet(SettingOptions.UI_SETTING_LINK) && currentChild.isOptionAvailable(SettingOptions.UI_SETTING_LINK)) {
			Object optionValue = currentChild.getOptionValue(SettingOptions.UI_SETTING_LINK);
			if (optionValue instanceof IChild) {
				IChild linkedChild = (IChild) optionValue;
				String currentStatus[] = new String[ErrorLevels.LEVEL_ERROR + 1];
				int currentLevel = ErrorLevels.LEVEL_SUCCESS;
				if ((currentStatus[ErrorLevels.LEVEL_ERROR] = currentChild.getError()) != null) {
					currentLevel = ErrorLevels.LEVEL_ERROR;
				} else if ((currentStatus[ErrorLevels.LEVEL_WARNING] = currentChild.getWarning()) != null) {
					currentLevel = ErrorLevels.LEVEL_WARNING;
				} else if ((currentStatus[ErrorLevels.LEVEL_INFORMATION] = currentChild.getInfo()) != null) {
					currentLevel = ErrorLevels.LEVEL_INFORMATION;
				}
				String linkedStatus[] = new String[ErrorLevels.LEVEL_ERROR + 1];
				int linkedLevel = ErrorLevels.LEVEL_SUCCESS;
				if ((linkedStatus[ErrorLevels.LEVEL_ERROR] = linkedChild.getError()) != null) {
					linkedLevel = ErrorLevels.LEVEL_ERROR;
				} else if ((linkedStatus[ErrorLevels.LEVEL_WARNING] = linkedChild.getWarning()) != null) {
					linkedLevel = ErrorLevels.LEVEL_WARNING;
				} else if ((linkedStatus[ErrorLevels.LEVEL_INFORMATION] = linkedChild.getInfo()) != null) {
					linkedLevel = ErrorLevels.LEVEL_INFORMATION;
				}
				int displayedLevel = Math.max(currentLevel, linkedLevel);
				if (displayedLevel == ErrorLevels.LEVEL_ERROR) {
					impl.setError(combineStatus(currentStatus[ErrorLevels.LEVEL_ERROR], linkedStatus[ErrorLevels.LEVEL_ERROR]));
				} else if (displayedLevel == ErrorLevels.LEVEL_WARNING) {
					impl.setWarning(combineStatus(currentStatus[ErrorLevels.LEVEL_WARNING], linkedStatus[ErrorLevels.LEVEL_WARNING]));
				} else if (displayedLevel == ErrorLevels.LEVEL_INFORMATION) {
					impl.setInfo(combineStatus(currentStatus[ErrorLevels.LEVEL_INFORMATION], linkedStatus[ErrorLevels.LEVEL_INFORMATION]));
				}
			}
		}
		return ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(impl));
	}

	/**
	 * Combines states to create final status. {@code null} values will be omitted.
	 * @param firstState first state or {@code null}
	 * @param secondState second state or {@code null}
	 * @return When both states are not {@code null} then both states concatenated with new line separator, when one of states is {@code null} then the other is returned,
	 *  when both are {@code null} then {@code null} is returned
	 */
	private static @Nullable String combineStatus(@Nullable String firstState, @Nullable String secondState) {
		if ((firstState == null) && (secondState == null)) {
			return null;
		}
		if (firstState == null) {
			return secondState;
		}
		if (secondState == null) {
			return firstState;
		}
		return firstState + UtilsText.LF + secondState;
	}

	/**
	 * @param tableViewer tableViewer in which the new column will be created
	 * @param style of new font
	 * @return new font with requested style
	 */
	protected static Font createFont(@NonNull TableViewer tableViewer, int style) {
		final Table tableLoc = tableViewer.getTable();
		final Font fontLoc = tableLoc.getFont();
		assert fontLoc != null;
		final Font newFont = FontFactory.changeStyle(fontLoc, style);
		tableLoc.addDisposeListener(d -> newFont.dispose());
		return newFont;
	}

	/**
	 * Get value text to be displayed.
	 * @param scalarConfig the configuration containing the value
	 * @return textual representation of a value
	 */
	protected static @Nullable String getText(@NonNull ScalarConfig scalarConfig) {
		switch (scalarConfig.getType()) {
		case FLOAT:
		case INTEGER:
		case STRING:
			return scalarConfig.getStringValue();
		case FILE:
			return scalarConfig.getFileStateMessage();
		case INFO:
			return scalarConfig.getValue().toString();
		case ENUM:
			return scalarConfig.getEnumItemUiName();
		default:
			return UtilsText.EMPTY_STRING;
		}
	}

	/**
	 * Sets widths of table viewer columns as specified by {@link #getControlOptions}
	 * @param column in the table of settings
	 * @param index of width setting
	 */
	protected void setColumnWidth(@NonNull TableColumn column, int index) {
		final Table table = Objects.requireNonNull(column.getParent());
		final int unitSize = computeColumnWidthUnitSize(table);
		final List<@NonNull Double> widths = getControlOptions().getTableColumnWidths();
		if (widths != null) {
			if (index >= widths.size()) {
				// Index is over defined widths => use the last one
				index = widths.size() - 1;
			}
			final double columnWidthInUnits = widths.get(index).doubleValue();
			if (columnWidthInUnits >= 0) {
				column.setWidth((int) (columnWidthInUnits * unitSize));
			}
		}
	}

	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		int colSpan = 1;
		// create composite with the content
		Composite contentComposite = createComposite(composite, colSpan);
		contentContainer = contentComposite;
		createLabelWithControls(contentComposite);
		Composite tableCompositeLoc;
		int tableStyle;
		boolean scrolledCompositeRequired = OSDetect.isUnix(); // table scroll does not work properly with GTK3
		// Create scroll area
		if (scrolledCompositeRequired) {
			ScrolledComposite scrolledCompositeLoc = new ScrolledComposite(contentComposite, SWT.BORDER | SWT.H_SCROLL);
			scrolledComposite = scrolledCompositeLoc;
			tableCompositeLoc = scrolledCompositeLoc;
			tableStyle = SWT.SINGLE | SWT.FULL_SELECTION;
		} else {
			tableStyle = SWT.SINGLE | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.BORDER;
			tableCompositeLoc = new Composite(contentComposite, SWT.NONE);
			GridLayoutComponents layout = new GridLayoutComponents(colSpan, false);
			layout.marginWidth = 0;
			tableCompositeLoc.setLayout(layout);
		}
		tableCompositeLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false));
		tableComposite = tableCompositeLoc;
		// create a table viewer
		TableViewer tableViewerLoc = new TableViewer(tableCompositeLoc, tableStyle);
		tableViewerLoc.setContentProvider(ArrayContentProvider.getInstance());
		italicFont = createFont(tableViewerLoc, SWT.ITALIC);
		boldFont = createFont(tableViewerLoc, SWT.BOLD);
		tableViewer = tableViewerLoc;
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(tableViewerLoc);
		tableViewerHandling(tableViewerLoc);
		// create table context menu
		registerControl(getMenuContext(tableViewerLoc));
		final Table table = tableViewerLoc.getTable();
		addKeyboardListener(table);
		// initialize the scroll properties
		ScrolledComposite scrolledCompositeLoc = scrolledComposite;
		if (scrolledCompositeLoc != null) {
			scrolledCompositeLoc.setContent(table);
			scrolledCompositeLoc.setExpandVertical(true);
			scrolledCompositeLoc.setExpandHorizontal(true);
			updateScrollSize();
			contentComposite.addListener(SWT.Resize, e -> {
				updateScrollSize();
			});
		}
		Composite contentContainerLoc = contentContainer;
		if (contentContainerLoc != null) {
			addScrollListener(table, contentContainerLoc);
		}
		createInfoLabel(contentComposite, colSpan);
		restoreSelectionFromStorage();
		return contentComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateSelectionOfExpandGroupContent()
	 */
	@Override
	protected void updateSelectionOfExpandGroupContent() {
		// if no child in table, set the expand control instance null, otherwise set the tableComposite instance
		if (children.isEmpty()) {
			expandGroupContent = null;
		} else {
			expandGroupContent = tableComposite;
		}
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void updateMainContent(@NonNull Control mainControlLoc, @NonNull UpdateType updateType) {
		TableViewer viewerLoc = tableViewer;
		if (updateType == UpdateType.PROBLEM_DECORATION) {
			if (viewerLoc != null) {
				final Object input = viewerLoc.getInput();
				if (input instanceof List<?>) {
					viewerLoc.update(((List<?>) input).toArray(), null);
				} else {
					viewerLoc.refresh();
				}
			}
			return;
		}
		Iterator<IChildControl> childrenIterator = children.iterator();
		Iterator<ISettingConfig> arrayItemIterator = arrayConfig.getChildren().iterator();
		ISettingConfig arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
		// dispose controls of all removed items
		while (childrenIterator.hasNext()) {
			IChildControl childControl = childrenIterator.next();
			if (!childControl.getChild().equals(arrayItem)) {
				// Remove child
				childrenIterator.remove();
				childControl.dispose();
			} else { // match
				arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
			}
		}
		// update all the children
		for (IChildControl childControl : children) {
			childControl.update(updateType);
		}
		// add newly added items to the GUI
		Composite tableCompositeLoc = tableComposite;
		while (arrayItem != null) {
			IChildControl currentChildControl = ChildControlFactory.create(arrayItem, new ControlOptions().labelHidden(getControlOptions().isArrayIndicesHidden()), controllerWrapper);
			if (currentChildControl != null) {
				children.add(currentChildControl);
				currentChildControl.update(updateType);
			}
			arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
		}
		if (tableCompositeLoc != null) {
			Point size = tableCompositeLoc.getSize();
			boolean isChanged = !size.equals(new Point(0, 0));
			if (KEPreferences.isAnimationsEnabled() && isChanged && updateType == UpdateType.NORMAL && (children.isEmpty() || (children.size() == 1 && !tableCompositeLoc.isVisible()))) {
				setControlVisibleAnimation(tableCompositeLoc, null, !children.isEmpty());
			} else {
				setControlVisible(tableCompositeLoc, !children.isEmpty());
			}
		}
		if (viewerLoc != null) {
			updateTableViewer(viewerLoc);
		}
		IChildControl childToSelect = getChildToSelect();
		setSelectedChild(childToSelect);
	}

	/**
	 * Updates content of table viewer
	 * @param viewer to be updated
	 */
	protected abstract void updateTableViewer(@NonNull TableViewer viewer);

	/**
	 * Update size of the scroll area
	 */
	protected void updateScrollSize() {
		ScrolledComposite scrolledCompositeLoc = scrolledComposite;
		TableViewer tableViewerLoc = tableViewer;
		if ((scrolledCompositeLoc != null) && (tableViewerLoc != null)) {
			scrolledCompositeLoc.setMinSize(tableViewerLoc.getTable().computeSize(SWT.DEFAULT, SWT.DEFAULT));
		}
	}

	/**
	 * Returns menu context
	 * @param viewer for which the menu is created
	 * @return menu context
	 */
	protected abstract @NonNull ArrayControlItemMenuContext getMenuContext(@NonNull TableViewer viewer);

	/**
	 * Custom handling with table viewer
	 * @param viewer to be handled
	 */
	protected abstract void tableViewerHandling(@NonNull TableViewer viewer);

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#selectLastItem()
	 */
	@Override
	public void selectLastItem() {
		TableViewer tableViewerLoc = tableViewer;
		if ((tableViewerLoc != null) && !getChildren().isEmpty()) {
			IChildControl lastControl = getChildren().get(getChildren().size() - 1);
			tableViewerLoc.setSelection(new StructuredSelection(lastControl));
		}
	}

	/**
	 * @return the height of an {@link AArrayControlTabular} table item or 0 if its tableViewer is <code>null</code>
	 */
	public int getItemHeight() {
		if (tableViewer != null) {
			return tableViewer.getTable().getItemHeight();
		}
		return 0;
	}

	/**
	 * Provides child to be selected when user selects cell in table viewer
	 * @param element data in the cell
	 * @param index of child which requires the new editing support
	 * @return child that is mapped to the data from cell
	 */
	protected abstract IChild editingSupportSelectChild(@NonNull Object element, int index);

	/**
	 * Performs action after creating proper cell editor, just before returning the editor
	 * @param viewer table viewer in which the editing support is added
	 * @param viewerColumn viewer column to which the editing support is added
	 */
	protected abstract void editingSupportPostAction(@NonNull TableViewer viewer, @NonNull TableViewerColumn viewerColumn);

	/**
	 * Provides child control to be selected when cell editor is created in editing support
	 * @param element of the column
	 * @param index of child which requires the new editing support
	 * @return child control of selected child or {@code null} when there is some problem
	 */
	protected abstract @Nullable IChildControl editingSupportGetChildControl(@NonNull Object element, int index);

	/**
	 * {@link EditingSupport#setValue} implementation
	 * @param viewer the table viewer
	 * @param element of the column
	 * @param index of child which requires the new editing support
	 * @param userInputValue input value from user
	 */
	protected abstract void editingSupportSetValue(@NonNull TableViewer viewer, @NonNull Object element, int index, @Nullable Object userInputValue);

	/**
	 * {@link EditingSupport#getValue} implementation
	 * @param element of the column
	 * @param index of child which requires the new editing support
	 * @return value of the cell or {@code null} when cell does not contain content
	 */
	protected abstract @Nullable Object editingSupportGetValue(@NonNull Object element, int index);

	/**
	 * {@link EditingSupport#canEdit} implementation
	 * @param element of the column
	 * @param index of child which requires the new editing support
	 * @return {@code true} when cell can be edited, otherwise returns {@code false}
	 */
	protected abstract boolean editingSupportCanEdit(@NonNull Object element, int index);

	/**
	 * Adds editing support to provided table viewer column
	 * @param viewer table viewer in which the column is placed
	 * @param viewerColumn the column to which the editing support is added
	 * @param index of child which requires the new editing support
	 */
	protected void addEditingSupport(@NonNull TableViewer viewer, @NonNull TableViewerColumn viewerColumn, int index) {
		EditingSupport editingSupport = new EditingSupport(viewer) {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ASettingEditingSupport#getCellEditor(java.lang.Object)
			 */
			@Override
			protected @Nullable CellEditor getCellEditor(Object element) {
				CellEditor editor = null;
				IChild currentChild = editingSupportSelectChild(element, index);
				IChildControl childControl = editingSupportGetChildControl(element, index);
				if (childControl != null) {
					setSelectedChild(childControl);
				}
				if (currentChild instanceof ScalarConfig) {
					ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
					// check-boxes options
					if (currentSettingLoc.getType().equals(Type.BOOL)) {
						// change value of check box
						ProgressUtils.run((m) -> {
							controllerWrapper.getController().setValue(currentSettingLoc, currentSettingLoc.getValue().equals(Boolean.FALSE)
									? UtilsText.safeString(Boolean.TRUE.toString()) : UtilsText.safeString(Boolean.FALSE.toString()), viewer);
						});
					} else if (currentSettingLoc.getType().equals(Type.ENUM)) {
						List<@NonNull UiValueIdHolder> enumItems = controllerWrapper.getController().getEnumItems((ISettingConfig) currentChild);
						InstantSearchCellEditor editorCombo = new InstantSearchCellEditor(viewer.getTable(), SWT.NONE);
						for (UiValueIdHolder enumItem : enumItems) {
							editorCombo.add(enumItem.getUiValue());
							editorCombo.addItemToolTip(enumItem.getUiValue(), enumItem.getDescription());
						}
						Object value = currentChild.getValue();
						if (value instanceof String) {
							String valueString = (String) value;
							if (valueString.startsWith(CustomItem.PREFIX)) {
								// If custom value is selected then allow to enter custom value into combo box
								editorCombo.setAllowCustomValue(true);
							}
						}
						editorCombo.addSelectionListener(new SelectionAdapter() {
							@Override
							public void widgetSelected(@NonNull SelectionEvent e) {
								String text = editorCombo.getText();
								List<@NonNull UiValueIdHolder> items = controllerWrapper.getController().getEnumItems((ISettingConfig) currentChild);
								CustomItem customItem = currentSettingLoc.getCustomItem();
								for (UiValueIdHolder item : items) {
									if ((item.getId().equals(CustomItem.CUSTOM_ID)) && (text.equals(CustomItem.CUSTOM_LABEL))) { // Newly selected custom option
										if (!editorCombo.isAllowCustomValue()) { // Do not remove already given value
											editorCombo.setAllowCustomValue(true);
											CustomValueSupport customValueSupport = currentSettingLoc.getCustomValueSupport();
											if (customValueSupport != null) {
												text = customValueSupport.getDefaultValue();
											}
											editorCombo.setText(text);
										}
										changeModelValue(currentSettingLoc, CustomItem.getStorableFormat(text));
										// SELECTION OF CUSTOM VALUE (Custom...) IN COMBO ENDS HERE!!!
										return;
									}
									if ((item.getUiValue().equals(text)) && (!item.getId().equals(CustomItem.CUSTOM_ID))) { // When any other item than custom item
										editorCombo.setAllowCustomValue(false);
										if (customItem != null) {
											customItem.setValue(UtilsText.safeString(CustomItem.CUSTOM_LABEL)); // Roll-back to default value for custom item
										}
										if (!Objects.equals(currentSettingLoc.getValue(), (item.getId()))) {
											changeModelValue(currentSettingLoc, item.getId());
										}
										break;
									}
								}
								if (editorCombo.isAllowCustomValue()) {
									final String storableFormat = CustomItem.getStorableFormat(UtilsText.safeString(text));
									if (!currentSettingLoc.getStringValue().equals(storableFormat)) {
										changeModelValue(currentSettingLoc, storableFormat); // Change to flush cache
									}
								}
							}
						});
						editor = editorCombo;
					} else if (currentSettingLoc.getType().equals(Type.INTEGER) || currentSettingLoc.getType().equals(Type.FLOAT)
							|| currentSettingLoc.getType().equals(Type.STRING)) {
						editor = new TextCellEditor(viewer.getTable());
					} else if ((currentSettingLoc.getType() == Type.INFO) && currentSettingLoc.isOptionSet(SettingOptions.UI_SETTING_LINK)) {
						// Open link
						Object linkTo = currentChild.getOptionValue(SettingOptions.UI_SETTING_LINK);
						if (linkTo instanceof IChild) {
							handleLinkToSetting((IChild) linkTo, ScalarInfoControl.class); // FIXME Tomas R. v14 maintenance - check if the caller argument affects behavior of the tool, change to getClass() if not
						}
					} else if (currentSettingLoc.getType().equals(Type.FILE)) {
						Consumer<@Nullable String> listener = (pathToFile) -> {
							String newValue = currentSettingLoc.prepareFileValue(pathToFile);
							if (!newValue.equals(currentSettingLoc.getValue())) {
								controllerWrapper.getController().setValue(currentSettingLoc, newValue, ArrayControlTabularHorizontal.class);
							}
						};
						editor = new FileSelectionCellEditor(viewer.getTable(), Messages.get().FileControl_SelectFile, UtilsText.safeString(currentSettingLoc.getFilePath()), listener);
					}
				} else if (currentChild instanceof SetPresence) {
					SetPresence setPresence = (SetPresence) currentChild;
					// change value of check box
					ProgressUtils.run((m) -> {
						controllerWrapper.getController().setSetPresence(setPresence, !setPresence.getBoolValue(), viewer);
					});
				}
				editingSupportPostAction(viewer, viewerColumn);
				return editor;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#canEdit(java.lang.Object)
			 */
			@Override
			protected boolean canEdit(Object element) {
				return editingSupportCanEdit(element, index);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#getValue(java.lang.Object)
			 */
			@Override
			protected @Nullable Object getValue(Object element) {
				return editingSupportGetValue(element, index);
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#setValue(java.lang.Object, java.lang.Object)
			 */
			@Override
			protected void setValue(Object element, Object userInputValue) {
				editingSupportSetValue(viewer, element, index, userInputValue);
			}
		};
		viewerColumn.setEditingSupport(editingSupport);
	}

	/**
	 * Selects correct child based on selected column
	 * @param viewer table viewer that holds the given column 
	 * @param column column that is selected
	 */
	protected void selectChildOfColumn(@NonNull TableViewer viewer, @NonNull TableColumn column) {
		List<TableColumn> columnsList = Arrays.asList(viewer.getTable().getColumns());
		int indexOfColumn = columnsList.indexOf(column);
		if (indexOfColumn != -1) {
			IChildControl childToSet = getChildren().get(indexOfColumn - 1);
			setSelectedChild(childToSet);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#setEnabledStateToMainControl(boolean)
	 */
	@Override
	protected void setEnabledStateToMainControl(boolean enabled) {
		Control mainControlLoc = mainControl;
		if ((mainControlLoc != null) && !mainControlLoc.isDisposed()) {
			setControlEnabled(mainControlLoc, enabled, true);
		}
	}
}
