/**
 * Copyright 2018-2022 NXP
 * Created: Jul 11, 2018
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.data.setting.InfoSetting;

/**
 * Class representing the control of a scalar {@link InfoSetting}
 * @author Tomas Rudolf - nxf31690
 */
public class ScalarInfoControl extends ScalarControl {

	/** Control which changes. Different control types can be stored here */
	private @Nullable Control controlWhichChanges;

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarInfoControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		if (getChild().isOptionSet(SettingOptions.UI_SETTING_LINK)) {
			final Composite linkComposite = new Composite(composite, SWT.NONE);
			GridLayoutComponents linkCompositeLayout = new GridLayoutComponents();
			linkCompositeLayout.marginWidth = linkCompositeLayout.marginHeight = 0;
			linkComposite.setLayout(linkCompositeLayout);
			GridDataComponents linktCompositeLayoutData = new GridDataComponents(SWT.FILL, SWT.CENTER, true, false);
			Link link = new Link(linkComposite, SWT.NONE);
			link.setLayoutData(linktCompositeLayoutData);
			createErrorDecoration(link, SWT.TOP | SWT.LEFT);
			link.setBackground(composite.getBackground());
			link.addListener(SWT.Selection, new Listener() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
				 */
				@Override
				public void handleEvent(@NonNull Event event) {
					Object optionValue = getChild().getOptionValue(SettingOptions.UI_SETTING_LINK);
					if (optionValue instanceof IChild) {
						handleLinkToSetting((IChild) optionValue, ScalarInfoControl.class);
					}
				}
			});
			mainControlInternal = link;
			controlWhichChanges = link;
			addScrollListener(link, composite);
			return linkComposite;
		} else if (getControlOptions().isLineBreak()) {
			Composite lineBreakComposite = new Composite(composite, SWT.NONE);
			String valueString = getChild().getValue().toString();
			if (valueString.isEmpty()) {
				lineBreakComposite.setLayout(new GridLayout(1, false));
				Label line = new Label(lineBreakComposite, SWT.SEPARATOR | SWT.HORIZONTAL);
				line.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
				mainControlInternal = line;
			} else {
				lineBreakComposite.setLayout(new GridLayout(3, false));
				Label lineLeft = new Label(lineBreakComposite, SWT.SEPARATOR | SWT.HORIZONTAL);
				lineLeft.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
				Label label = new Label(lineBreakComposite, SWT.NONE);
				controlWhichChanges = label;
				Label lineRight = new Label(lineBreakComposite, SWT.SEPARATOR | SWT.HORIZONTAL);
				lineRight.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
				mainControlInternal = label;
			}
			return lineBreakComposite;
		}
		final Composite textComposite = new Composite(composite, SWT.NONE);
		GridLayoutComponents textCompositeLayout = new GridLayoutComponents();
		textCompositeLayout.marginWidth = textCompositeLayout.marginHeight = 0;
		textComposite.setLayout(textCompositeLayout);
		GridDataComponents textCompositeLayoutData = new GridDataComponents(SWT.FILL, SWT.CENTER, true, false);
		Text text = new Text(textComposite, SWT.READ_ONLY | SWT.WRAP);
		text.setLayoutData(textCompositeLayoutData);
		createErrorDecoration(text, SWT.TOP | SWT.LEFT);
		controlWhichChanges = text;
		mainControlInternal = text;
		addScrollListener(text, composite);
		return textComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			String valueString = child.getValue().toString();
			Control controlWhichChangesLoc = controlWhichChanges;
			if (controlWhichChangesLoc instanceof Text) {
				((Text) controlWhichChangesLoc).setText(ScalarConfig.replaceMarksWithMeanings(valueString));
			} else if (controlWhichChangesLoc instanceof Link) {
				boolean linkAvailable = getChild().isOptionAvailable(SettingOptions.UI_SETTING_LINK);
				Link link = (Link) controlWhichChangesLoc;
				link.setText(UtilsText.createHyperLinkString(valueString, null));
				link.setEnabled(linkAvailable && getChild().isEnabled());
			} else if (controlWhichChangesLoc instanceof Label) {
				((Label) controlWhichChangesLoc).setText(valueString);
			}
		}
		if (mainControlInternal != null) {
			updateErrorDecoration(mainControlInternal);
		}
		updateErrorDecoration(contentControl);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#getErrorLevel()
	 */
	@Override
	protected int getStatusLevel() {
		int currentChildLevel = super.getStatusLevel();
		if ((!child.isOptionSet(SettingOptions.UI_SETTING_LINK) || !child.isOptionAvailable(SettingOptions.UI_SETTING_LINK))) {
			return currentChildLevel;
		}
		Object optionValue = child.getOptionValue(SettingOptions.UI_SETTING_LINK);
		if (!(optionValue instanceof IChild)) {
			return currentChildLevel;
		}
		IChild linkedChild = (IChild) optionValue;
		int linkedChildLevel = ErrorLevels.LEVEL_SUCCESS;
		if (linkedChild.getError() != null) {
			linkedChildLevel = ErrorLevels.LEVEL_ERROR;
		} else if (linkedChild.getWarning() != null) {
			linkedChildLevel = ErrorLevels.LEVEL_WARNING;
		} else if (linkedChild.getInfo() != null) {
			linkedChildLevel = ErrorLevels.LEVEL_INFORMATION;
		}
		return Math.max(currentChildLevel, linkedChildLevel);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#getErrorStatus(int)
	 */
	@Override
	protected String getDecoratorStatus(int statusLevel) {
		String currentChildStatus = super.getDecoratorStatus(statusLevel);
		if ((!child.isOptionSet(SettingOptions.UI_SETTING_LINK) || !child.isOptionAvailable(SettingOptions.UI_SETTING_LINK))) {
			return currentChildStatus;
		}
		Object optionValue = child.getOptionValue(SettingOptions.UI_SETTING_LINK);
		if (!(optionValue instanceof IChild)) {
			return currentChildStatus;
		}
		IChild linkedChild = (IChild) optionValue;
		String linkedChildStatus = null;
		if (statusLevel >= ErrorLevels.LEVEL_ERROR) {
			linkedChildStatus = linkedChild.getError();
		} else if (statusLevel == ErrorLevels.LEVEL_WARNING) {
			linkedChildStatus = linkedChild.getWarning();
		} else if (statusLevel == ErrorLevels.LEVEL_INFORMATION) {
			linkedChildStatus = linkedChild.getInfo();
		}
		if (linkedChildStatus == null) {
			return currentChildStatus;
		}
		if (currentChildStatus == null) {
			return linkedChildStatus;
		}
		return currentChildStatus + UtilsText.LF + linkedChildStatus;
	}

}
