/**
 * Copyright 2017-2023 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.controller.UiValueIdHolder;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.data.setting.DynamicEnumSetting.CustomValueSupport;
import com.nxp.swtools.resourcetables.model.data.setting.SetSetting.CustomItem;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.text.TextBoxHelper;
import com.nxp.swtools.utils.text.TextBoxHelper.Status;
import com.nxp.swtools.utils.tooltip.ToolTipableImplementation;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

import animations.swt.ColorChanger;
import animations.swt.EaseSelector;

/**
 * Class representing control of a combo-box setting configuration.
 * @author Juraj Ondruska
 */
public class ScalarComboControl extends AScalarComboControl {
	/** Time how long the animation will be performed (ms) */
	private static final int ANIMATION_TIME = 1500;
	/** Main combo-box */
	@Nullable InstantSearchList combo;
	
	/** Check function for custom value */
	private @NonNull Function<@NonNull String, @NonNull Status> checkFunction = (newVal) -> {
		Object value = getChild().getValue();
		if (value instanceof String) {
			String valueString = (String) value;
			if (valueString.startsWith(CustomItem.PREFIX)) { // Only check if current value is custom
				final InstantSearchList comboLoc = combo;
				boolean isPredefined = (comboLoc != null) && !comboLoc.isDisposed() && comboLoc.getItems().contains(newVal); // allow to enter predefined value (MCUXCON-3666)
				if (!getChild().isCustomValueValid(newVal) && !isPredefined) {
					return Status.INVALID;
				}
			}
		}
		return Status.OK;
	};

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarComboControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		String valueName = String.valueOf(getChild().getValue());
		if (valueName.equals(CustomItem.CUSTOM_ID)) {
			// Selected by quick selection
			CustomItem customItem = getChild().getCustomItem();
			CustomValueSupport customValueSupport = getChild().getCustomValueSupport();
			if ((customItem != null) && (customValueSupport!= null)) {
				customItem.setValue(customValueSupport.getDefaultValue());
			}
		}
		super.update(updateType);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		InstantSearchList comboLoc = combo;
		if (comboLoc != null) {
			if (updateType != UpdateType.PROBLEM_DECORATION) {
				Text textBox = comboLoc.getTextBox();
				if (comboLoc.isAllowCustomValue()) {
					TextBoxHelper.removeListenersOfType(textBox, SWT.Modify);
					TextBoxHelper.removeListenersOfType(textBox, SWT.Deactivate);
				}
				List<@NonNull UiValueIdHolder> items = controllerWrapper.getController().getEnumItems(getChild());
				String[] arrayOfItems = items.stream().map(x -> x.getUiValue()).toArray(String[]::new);
				if (!Arrays.equals(comboLoc.getItems().toArray(), arrayOfItems))  {
					comboLoc.setData(KEY_INPUT, items);
					comboLoc.setItems(arrayOfItems);
					ToolTipableImplementation tooltip = new ToolTipableImplementation();
					ToolTipableMarkdownDescriptionDecorator decoratedTooltip = new ToolTipableMarkdownDescriptionDecorator(tooltip);
					for (UiValueIdHolder uiValueIdHolder : items) {
						tooltip.setDescription(uiValueIdHolder.getDescription());
						String itemDescription = UtilsText.safeString(decoratedTooltip.getDescription());
						if (!itemDescription.isEmpty()) {
							comboLoc.addItemToolTip(uiValueIdHolder.getUiValue(), itemDescription);
						}
					}
				}
				String valueName = String.valueOf(getChild().getValue());
				if (valueName.startsWith(CustomItem.PREFIX)) {
					valueName = CustomItem.CUSTOM_ID;
					comboLoc.setAllowCustomValue(true); // If given value is custom value, then allow to edit it
				}
				String value = controllerWrapper.getController().getEnumItemById(getChild(), valueName).getUiValue();
				if (!Objects.equals(comboLoc.getText(), value)) {
					comboLoc.select(value);
				}
				if (comboLoc.isAllowCustomValue()) {
					TextBoxHelper.attachModifyErrorListener(() -> CustomItem.parseValueFromStorableString(
							getChild().getStringValue()), textBox, checkFunction, () -> updateErrorDecoration(comboLoc));
				}
			}
			if (mainControlInternal != null) {
				updateErrorDecoration(mainControlInternal);
			}
			updateErrorDecoration(comboLoc);
		}
	}

	/**
	 * Update background of the control when error or warning occurred.
	 * @param controlWithDecoration instant search list to be decorated
	 */
	@Override
	protected void updateErrorDecoration(@NonNull Control controlWithDecoration) {
		super.updateErrorDecoration(controlWithDecoration);
		if (controlWithDecoration instanceof InstantSearchList) {
			InstantSearchList searchList = (InstantSearchList) controlWithDecoration;
			if (!searchList.isDisposed()) {	
				String text = UtilsText.safeString(searchList.getText());
				boolean errorFlag = false;
				Color background = SwToolsColors.getColor(SwToolsColors.DEFAULT_LIST_BG);
				Color foreground = SwToolsColors.getColor(SwToolsColors.DEFAULT_FG);
				searchList.setData(TextBoxHelper.STATE_PROPERTY_KEY, TextBoxHelper.STATE_OK);
				if ((child.getError() != null) || (checkFunction.apply(text) == Status.INVALID)) {
					background = SwToolsColors.getColor(SwToolsColors.ERROR_BG);
					foreground = SwToolsColors.getColor(SwToolsColors.ERROR_FG);
					searchList.setData(TextBoxHelper.STATE_PROPERTY_KEY, TextBoxHelper.STATE_ERROR);
					errorFlag = true;
				} else if (child.getWarning() != null) {
					background = SwToolsColors.getColor(SwToolsColors.WARNING_FG);
					foreground = SwToolsColors.getColor(SwToolsColors.BLACK_FG);
					searchList.setData(TextBoxHelper.STATE_PROPERTY_KEY, TextBoxHelper.STATE_WARNING);
					errorFlag = true;
				}
				
				if(KEPreferences.isAnimationsEnabled() && errorFlag) {
					ColorChanger cc = new ColorChanger(searchList, background, EaseSelector.SPIKE);
					cc.play(ANIMATION_TIME);
				} else {
					searchList.setBackground(background);
				}
				searchList.setForeground(foreground);
			}
		}
	}

	/**
	 * Method used to create composite for the controls of this class
	 * @param composite - parent composite
	 * @return created composite
	 */
	@Override
	protected @NonNull Composite createComposite(@NonNull Composite composite) {
		Composite comboComposite = new Composite(composite, SWT.NONE);
		GridLayoutComponents comboCompositeLayout = new GridLayoutComponents();
		comboCompositeLayout.marginWidth = comboCompositeLayout.marginHeight = 0;
		comboComposite.setLayout(comboCompositeLayout);
		return comboComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.AScalarComboControl#createControls(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createControls(@NonNull Composite composite) {
		combo = createComboBox(composite);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.AScalarComboControl#addComboSelectionListener()
	 */
	@Override
	protected void addComboSelectionListener() {
		InstantSearchList comboLoc = combo;
		if (comboLoc != null) {
			comboLoc.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetDefaultSelected(SelectionEvent e) {
					widgetSelected(e);
				}

				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					String text = comboLoc.getText();
					@SuppressWarnings("unchecked")
					List<@NonNull UiValueIdHolder> items = (List<@NonNull UiValueIdHolder>) Objects.requireNonNull(comboLoc.getData(KEY_INPUT));
					CustomItem customItem = getChild().getCustomItem();
					for (UiValueIdHolder item : items) {
						if ((item.getId().equals(CustomItem.CUSTOM_ID)) && (text.equals(CustomItem.CUSTOM_LABEL))) { // Newly selected custom option
							if (customItem != null && customItem.isInDefaultState()) { // Do not remove already given value
								comboLoc.setAllowCustomValue(true);
								customItem.setAsUsed();
								CustomValueSupport customValueSupport = getChild().getCustomValueSupport();
								if (customValueSupport != null) {
									text = customValueSupport.getDefaultValue();
								}
								comboLoc.setText(text);
							}
							changeModelValue(CustomItem.getStorableFormat(text));
							comboLoc.setSelection(new Point(0, text.length()));
							updateErrorDecoration(comboLoc);
							// SELECTION OF CUSTOM VALUE (Custom...) IN COMBO ENDS HERE!!!
							return;
						}
						if ((item.getUiValue().equals(text)) && (!item.getId().equals(CustomItem.CUSTOM_ID))) { // When any other item than custom item
							comboLoc.setAllowCustomValue(false);
							if (customItem != null) {
								customItem.setAsNotUsed();
								customItem.setValue(UtilsText.safeString(CustomItem.CUSTOM_LABEL)); // Roll-back to default value for custom item
							}
							if (!Objects.equals(getChild().getValue(), (item.getId()))) {
								changeModelValue(item.getId());
							}
							break;
						}
					}
					if (comboLoc.isAllowCustomValue()) {
						final String storableFormat = CustomItem.getStorableFormat(UtilsText.safeString(text));
						if (!getChild().getStringValue().equals(storableFormat)) {
							changeModelValue(storableFormat); // Change to flush cache
						}
					}
					updateErrorDecoration(comboLoc);
				}
			});
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.AScalarComboControl#addSpecificListeners()
	 */
	@Override
	protected void addSpecificListeners() {
		// no specific listeners for this class
	}
}
