/**
 * Copyright 2019-2021 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

import com.nxp.swtools.common.ui.utils.editors.FileSelectionCellEditor;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;

/**
 * Class representing control of a file picker with modality to persist file content on selection
 * @author Tomas Rudolf - nxf31690
 */
public class FilePersistentControl extends ScalarControl {

	/** Label with status */
	private @Nullable Label statusLabel;
	/** Button to select file */
	private @Nullable Button selectButton;
	/** Button to reset content */
	private @Nullable Button resetContentButton;

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public FilePersistentControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		Composite group = new Composite(composite, getSwtStyle());
		group.setLayout(new GridLayout(2, false));
		Label label = new Label(group, SWT.NONE);
		mainControlInternal = label;
		statusLabel = label;
		label.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		final Button selectButtonLoc = new Button(group, SWT.PUSH);
		selectButton = selectButtonLoc;
		selectButtonLoc.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
		selectButtonLoc.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				FileSelectionCellEditor.openFileDialog(selectButtonLoc.getShell(), Messages.get().FileControl_SelectFile, UtilsText.EMPTY_STRING,
						result -> controllerWrapper.getController().setValue(getChild(), getChild().prepareFileValue(result), FilePersistentControl.class));
			}
		});
		Button resetContentButtonLoc = new Button(group, SWT.PUSH);
		resetContentButtonLoc.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				controllerWrapper.getController().setValue(getChild(), UtilsText.EMPTY_STRING, FilePersistentControl.class);
			}
		});
		resetContentButton = resetContentButtonLoc;
		createErrorDecoration(group, SWT.LEFT | SWT.TOP);
		group.pack();
		return group;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			Label statusLabelLoc = statusLabel;
			Button selectButtonLoc = selectButton;
			Button resetContentButtonLoc = resetContentButton;
			String stateString = getChild().getFileStateMessage();
			String valueOfSetting = String.valueOf(getChild().getValue());
			boolean enabled = !UpdateType.FORCE_DISABLE.equals(updateType);
			if (valueOfSetting.isEmpty()) {
				setButtonVisibleState(selectButtonLoc, true, enabled);
				setButtonVisibleState(resetContentButtonLoc, false, enabled);
			} else {
				setButtonVisibleState(selectButtonLoc, false, enabled);
				setButtonVisibleState(resetContentButtonLoc, true, enabled);
			}
			if (statusLabelLoc != null) {
				statusLabelLoc.setText(stateString);
			}
			if (selectButtonLoc != null) {
				selectButtonLoc.setText(Messages.get().FileControl_SelectFile);
			}
			if (resetContentButtonLoc != null) {
				resetContentButtonLoc.setText(Messages.get().FileControl_Unlink);
			}
			contentControl.requestLayout();
		}
		if (mainControlInternal != null) {
			updateErrorDecoration(mainControlInternal);
		}
		updateErrorDecoration(contentControl);
	}

	/**
	 * Sets specified visibility state to given button
	 * @param button to be set
	 * @param visible flag
	 * @param enable flag
	 */
	private static void setButtonVisibleState(Button button, boolean visible, boolean enable) {
		if (button != null) {
			GridData data = (GridData) button.getLayoutData();
			if (data != null) {
				data.exclude = !visible;
			}
			button.setVisible(visible);
			button.setEnabled(enable);
		}
	}
}
