/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.text.MessageFormat;
import java.util.Objects;
import java.util.logging.Logger;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ConfigSetConfig;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IConfigSetConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.SetConfig;
import com.nxp.swtools.resourcetables.model.config.SetConfig.SetPresence;
import com.nxp.swtools.resourcetables.model.config.StructConfig;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.data.setting.ISetting;
import com.nxp.swtools.resourcetables.model.data.setting.VariableSetting;

/**
 * Class used for creating setting configuration controls.
 * @author Juraj Ondruska
 */
public class ChildControlFactory {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ChildControlFactory.class);
	
	/**
	 * Create control.
	 * @param child the setting configuration to create control for
	 * @param controllerWrapper containing the generic controller
	 * @param overrideOptions options that override options before creation process
	 * @return the newly created setting configuration control if visible in the UI, {@code null} otherwise
	 */
	public static @Nullable IChildControl create(@NonNull IChild child, @NonNull IControllerWrapper controllerWrapper, @Nullable ControlOptions overrideOptions) {
		if (child.isOptionSet(SettingOptions.UI_NOT_VISIBLE_PERMANENT) && Objects.equals(child.getOptionValue(SettingOptions.UI_NOT_VISIBLE_PERMANENT), Boolean.TRUE)) {
			return null; // Setting does not have GUI during whole setting's view life span
		}
		ControlOptions creationOptions = new ControlOptions(child);
		if (overrideOptions != null) {
			creationOptions.merge(overrideOptions);
		}
		if (child instanceof ScalarConfig) {
			ScalarConfig scalarConfig = (ScalarConfig) child;
			switch (scalarConfig.getType()) {
			case ENUM:
				// FIXME TomasR v99 - Consider to refactor this to work as delegate like arrays
				if (creationOptions.shouldLayoutAsRadioGroup()) {
					return new ScalarRadioGroupControl(scalarConfig, creationOptions, controllerWrapper);
				}
				if (child.isOptionSet(SettingOptions.FEATURE_COMPONENT_ADD)) {
					return new ScalarComboAddComponentControl(scalarConfig, creationOptions, controllerWrapper);
				}
				if (child.isOptionSet(SettingOptions.UI_SETTING_LINK)) {
					return new ScalarComboLinkControl(scalarConfig, creationOptions, controllerWrapper);
				}
				return new ScalarComboControl(scalarConfig, creationOptions, controllerWrapper);
			case BOOL:
				// FIXME TomasR v99 - Consider to refactor this to work as delegate like arrays
				if (creationOptions.shouldLayoutAsRadioGroup()) {
					return new ScalarRadioGroupControl(scalarConfig, creationOptions, controllerWrapper);
				}
				return new ScalarCheckboxControl(scalarConfig, creationOptions, controllerWrapper);
			case BUTTON:
				return new ScalarButtonControl(scalarConfig, creationOptions, controllerWrapper);
			case FLOAT:
				//$FALL-THROUGH$
			case INTEGER:
				//$FALL-THROUGH$
			case STRING:
				return new ScalarTextControl(scalarConfig, creationOptions, controllerWrapper);
			case FILE:
				if (creationOptions.isSelectionPersistent() && !scalarConfig.hasDirectoryType()) {
					return new FilePersistentControl(scalarConfig, creationOptions, controllerWrapper);
				}
				return new FileLinkControl(scalarConfig, creationOptions, controllerWrapper);
			case VARIABLE:
				return null;
			case INFO:
				if (child.isOptionSet(SettingOptions.UI_INFO_COPY_BUTTON)) {
					return new ScalarCopyControl(scalarConfig, creationOptions, controllerWrapper);
				}
				return new ScalarInfoControl((ScalarConfig) child, creationOptions, controllerWrapper);
			default:
				return new ScalarControl(scalarConfig, creationOptions, controllerWrapper);
			}
		}
		if (child instanceof StructConfig) {
			boolean invisible = true;
			StructConfig structConfig = (StructConfig) child;
			for (ISetting setting : structConfig.getModelData().getSettings().values()) {
				if (!(setting instanceof VariableSetting)) {
					invisible = false;
				}
			}
			if (invisible) {
				return new StructInvisibleControl(structConfig, creationOptions, controllerWrapper);
			}
			if (creationOptions.isMultiColumn()) {
				return new StructMultiColumnControl(structConfig, creationOptions, controllerWrapper);
			}
			return new StructControl(structConfig, creationOptions, controllerWrapper);
		}
		if (child instanceof ArrayConfig) {
			return new ArrayControl((ArrayConfig) child, creationOptions, controllerWrapper);
		}
		if (child instanceof SetConfig) {
			return new SetControl((SetConfig) child, creationOptions, controllerWrapper);
		}
		if (child instanceof IConfigSetConfig) {
			return new ConfigSetControl((ConfigSetConfig) child, creationOptions, controllerWrapper);
		}
		if (child instanceof SetPresence) {
			return new SetPresenceControl((SetPresence) child, creationOptions, controllerWrapper);
		}
		if (child instanceof IComponentInstanceConfig) {
			return new ComponentInstanceControl((IComponentInstanceConfig) child, creationOptions, controllerWrapper);
		}
		LOGGER.severe(MessageFormat.format("[TOOL] Unexpected class {0} of setting with ID {1}", child.getClass().getSimpleName(), child.getId())); //$NON-NLS-1$
		return new ChildControlBase(child, creationOptions, controllerWrapper);
	}

	/**
	 * Create control.
	 * @param child the setting configuration to create control for
	 * @param overwriteOptions UI options that are merged into options of this child created from model
	 * @param controllerWrapper containing the generic controller
	 * @return the newly created setting configuration control if visible in the UI, {@code null} otherwise
	 */
	public static @Nullable IChildControl create(@NonNull IChild child, @Nullable ControlOptions overwriteOptions, @NonNull IControllerWrapper controllerWrapper) {
		IChildControl childControl = create(child, controllerWrapper, overwriteOptions);
		if ((childControl != null) && (overwriteOptions != null)) {
			childControl.getControlOptions().merge(overwriteOptions);
		}
		return childControl;
	}
}
