/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.services.Rap;
import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.tileviewer.ITileFilter;
import com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider;
import com.nxp.swtools.common.ui.utils.tileviewer.ITileSelectionListenerData;
import com.nxp.swtools.common.ui.utils.tileviewer.TileCategoryData;
import com.nxp.swtools.common.ui.utils.tileviewer.TileItemData;
import com.nxp.swtools.common.ui.utils.tileviewer.TileViewer;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.expression.IContext;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.runtime.DelayedActionExecutorAdapter;
import com.nxp.swtools.common.utils.runtime.DelayedExecution;
import com.nxp.swtools.common.utils.runtime.IDelayedActionExecutor;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.dialogs.initializationorder.InitializationOrderDialog;
import com.nxp.swtools.periphs.gui.dialogs.initializationorder.InitializationOrderDialogProperties;
import com.nxp.swtools.periphs.gui.view.ComponentMenuCreator.ComponentMenuOptions;
import com.nxp.swtools.periphs.gui.view.ComponentMenuCreator.IComponentSelectionProvider;
import com.nxp.swtools.periphs.model.config.ComponentConfig;
import com.nxp.swtools.periphs.model.data.Categories;
import com.nxp.swtools.periphs.model.data.Components;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.ChildValidationHelper;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.data.Category;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.SWComponent;
import com.nxp.swtools.resourcetables.model.validation.ValidationHelper;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.engine.ValidationEngineFactory;
import com.nxp.swtools.validation.engine.ValidationProblemListenerAdapter;

/**
 * View responsible for displaying components and their settings.
 * @author Juraj Ondruska
 */
public class ComponentsView extends APeriphsViewBase {
	/** Id of "other" category */
	private static final @NonNull String CATEGORY_OTHER_ID = "other"; //$NON-NLS-1$
	/** UiName of "other" category */
	private static final @NonNull String CATEGORY_OTHER_UI_NAME = UtilsText.safeString(Messages.get().ComponentsView_CategoryOther_UiName);
	/** Description of "other" category */
	private static final @NonNull String CATEGORY_OTHER_DESCRIPTION = UtilsText.safeString(Messages.get().ComponentsView_CategoryOther_Description);
	/** ID of the view */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.componentsView"; //$NON-NLS-1$
	/** Main composite of the view */
	@Nullable Composite mainComposite;
	/**	String for filter to apply */
	@NonNull String filterString = UtilsText.EMPTY_STRING;
	/** Viewer for tiles */
	@Nullable TileViewer tileViewer;
	/** Text box with the filter text */
	private @Nullable Text filterText;
	/** Milliseconds to wait before delayed execution */
	public static final int DELAYED_EXECUTION_TIMEOUT = 1000;
	/** Suffix new constant */
	public static final String SUFFIX_COPY = UtilsText.UNDERSCORE + "copy"; //$NON-NLS-1$
	/** Listener for validations changes */
	private @NonNull ValidationProblemListenerAdapter validationsListener = new ValidationProblemListenerAdapter() {
		@Override
		public void validationProblemsChanged(Collection<@NonNull IValidationProblem> problems) {
			// Refresh when any dependency changes
			TileViewer tileViewerLoc = tileViewer;
			if ((tileViewerLoc != null) && !tileViewerLoc.getControl().isDisposed()) {
				tileViewerLoc.refresh();
			}
		}
	};

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parentComposite) {
		mainComposite = createDefaultComposite(parentComposite);
		Composite mainCompositeLoc = mainComposite;
		mainCompositeLoc.setLayout(new GridLayout(3, false));

		Text text = new Text(mainCompositeLoc, SWT.BORDER);
		text.setMessage(UtilsText.safeString(com.nxp.swtools.utils.Messages.get().RegistersView_TypeFilterText));
		text.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		SWTFactoryProxy.INSTANCE.setTestId(text, TestIDs.PERIPHS_FILTER_COMPONENTS_TEXT);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(text, Messages.get().ComponentsView_FilterTooltip);
		filterText = text;

		IDelayedActionExecutor actionExecutor = new DelayedActionExecutorAdapter() {
			@Override
			public void runAction() {
				mainCompositeLoc.getDisplay().asyncExec(new Runnable() {
					@Override
					public void run() {
						refresh(false);
					}
				});
			}
		};
		DelayedExecution delayedExecution = new DelayedExecution("FilteringComponents", actionExecutor, DELAYED_EXECUTION_TIMEOUT); //$NON-NLS-1$
		ModifyListener textModifyListener = new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				final String newFilterText = UtilsText.safeString(((Text)e.widget).getText());
				if (newFilterText.equals(filterString)) {
					return;
				}
				filterString = newFilterText;
				if (Rap.isActive()) {
					refresh(false);
				} else {
					delayedExecution.requestExecution();
				}
			}
		};
		text.addModifyListener(textModifyListener);

		Button button = new Button(mainCompositeLoc, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_ADD_COMPONENT_INSTANCE_BUTTON);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().ComponentsView_AddButtonTooltip);
		button.setImage(ToolsImages.getImage(IToolsImages.ICON_ADD_CIRCLE));
		button.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, false));
		button.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				AddComponentDialog.open(getViewSite(), null, controllerWrapper);
			}
		});
		button = new Button(mainCompositeLoc, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_OPEN_REORDER_DIALOG);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().InitializationOrderDialog_OpenDialogButtonTooltip);
		button.setImage(ToolsImages.getImage(IToolsImages.IMAGE_REORDER));
		button.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, false));
		button.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				Shell shell = getSiteNonNull().getShell();
				if (shell != null) {
					InitializationOrderDialog.open(new InitializationOrderDialogProperties(shell, controllerWrapper));
				}
			}
		});

		ArrayList<@NonNull TileCategoryData> tileCategoriesData = createTilesData();
		TileViewer tileViewerLoc = new TileViewer(mainCompositeLoc);
		tileViewerLoc.setLayoutData(GridDataFactory.fillDefaults()
				.align(SWT.FILL, SWT.FILL)
				.grab(true, true)
				.span(3, 1)
				.create());
		tileViewer = tileViewerLoc;

		tileViewerLoc.setSelectionListener(new ITileSelectionListenerData() {
			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileSelectionListenerData#itemSelected(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public void itemSelected(TileItemData selectedItem) {
				Object data = selectedItem.getData();
				if (data instanceof IComponentInstanceConfig) {
					String type = ((IComponentInstanceConfig) data).getType();
					String uiName = ((IComponentInstanceConfig) data).getUiName();
					IComponentInstanceConfig instance = Controller.getInstance().getComponentInstance(type, uiName);
					if (instance != null) {
						openView(instance);
					} else {
						openView(type, uiName, true);
					}
				}
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileSelectionListenerData#categorySelected(com.nxp.swtools.common.ui.utils.tileviewer.TileCategoryData)
			 */
			@Override
			public void categorySelected(TileCategoryData selectedCategory) {
				Object data = selectedCategory.getData();
				if (data instanceof Category) {
					Category category = (Category) data;
					List<@NonNull ConfigurationComponentTypeId> componentsForCategory;
					if (category.getId().equals(CATEGORY_OTHER_ID)) {
						Categories categories = controllerWrapper.getController().getCategories();
						if (categories != null) {
							List<@NonNull Category> categoriesList = categories.getCategories();
							// get components that do not match any category
							componentsForCategory = controllerWrapper.getController().getMcu().getAvailableComponents().getConfigCompsTypeIdsWithoutMatchingCategory(categoriesList);
						} else {
							AddComponentDialog.open(getViewSite(), null, controllerWrapper);
							return;
						}
					} else {
						// get components for given category
						componentsForCategory = controllerWrapper.getController().getMcu().getAvailableComponents().getConfigCompsTypeIdsForCategory(category.getId());
					}
					AddComponentDialogProperties properties = new AddComponentDialogProperties(getViewSite(), Controller.getInstance().getFunctionalGroup().getId(), controllerWrapper, true, true);
					properties.setConfigCompsTypeIds(componentsForCategory);
					AddComponentDialog.open(properties);
				}
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileSelectionListenerData#itemMenuSelected(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData, java.util.List)
			 */
			@Override
			public void itemMenuSelected(TileItemData tileItemData, List<@NonNull Control> controls) {
				IComponentSelectionProvider selectionProvider = new IComponentSelectionProvider() {
					/* (non-Javadoc)
					 * @see com.nxp.swtools.periphs.gui.view.ComponentsView.IChildSelectionProvider#getSelection()
					 */
					@Override
					public Collection<? extends @NonNull IChild> getSelection() {
						List<@NonNull IChild> result = new ArrayList<>();
						Object data = tileItemData.getData();
						if (data instanceof IChild) {
							result.add((IChild) data);
						}
						return result;
					}
				};
				new ComponentMenuCreator(selectionProvider, new ComponentMenuOptions(/* ask what to remove */ false), controllerWrapper).createMenu(controls, getViewSite(), null);
			}
		});

		Font categoryFont = FontFactory.changeStyle(parentComposite.getFont(), SWT.BOLD);
		parentComposite.addDisposeListener(e -> categoryFont.dispose());

		tileViewerLoc.setLabelProvider(new ITileLabelProvider() {
			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getText(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public String getText(TileItemData itemData) {
				Object data = itemData.getData();
				if (data instanceof IComponentInstanceConfig) {
					return ((IComponentInstanceConfig) data).getUiName();
				}
				return UtilsText.EMPTY_STRING;
			}

			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getFont(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public @Nullable Font getFont(@NonNull TileItemData tileItemData) {
				return null; // Use default font
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getText(com.nxp.swtools.common.ui.utils.tileviewer.TileCategoryData)
			 */
			@Override
			public String getText(TileCategoryData categoryData) {
				Object data = categoryData.getData();
				if (data instanceof Category) {
					return UtilsText.safeString(((Category) data).getUINameString());
				}
				return UtilsText.EMPTY_STRING;
			}

			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getFont(com.nxp.swtools.common.ui.utils.tileviewer.TileCategoryData)
			 */
			@Override
			public @Nullable Font getFont(@NonNull TileCategoryData tileCategoryData) {
				return categoryFont;
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getBackground(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public Color getBackground(TileItemData itemData) {
				Object data = itemData.getData();
				if (data instanceof IComponentInstanceConfig) {
					IComponentInstanceConfig instanceConfig = (IComponentInstanceConfig) data;
					if (!instanceConfig.isEnabled()) {
						return SwToolsColors.getColor(SwToolsColors.TILE_VIEWER_ITEM_BG_DISABLED);
					}
					final int problemLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(instanceConfig);
					if ((instanceConfig.getError() != null) || (problemLevel >= ErrorLevels.LEVEL_ERROR)) {
						return SwToolsColors.getColor(SwToolsColors.ERROR_BG);
					}
				}
				return SwToolsColors.getColor(SwToolsColors.TILE_VIEWER_ITEM_BG);
			}

			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getForeground(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public Color getForeground(TileItemData itemData) {
				Object data = itemData.getData();
				final Display display = tileViewerLoc.getControl().getDisplay();
				if (data instanceof IComponentInstanceConfig) {
					IComponentInstanceConfig instanceConfig = (IComponentInstanceConfig) data;
					if (!instanceConfig.isEnabled()) {
						return display.getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW);
					}
					final int problemLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(instanceConfig);
					if ((instanceConfig.getError() != null) || (problemLevel >= ErrorLevels.LEVEL_ERROR)) {
						return SwToolsColors.getColor(SwToolsColors.ERROR_FG);
					}
				}
				return display.getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getImage(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public Image getImage(TileItemData itemData) {
				Object data = itemData.getData();
				if (data instanceof IComponentInstanceConfig) {
					IComponentInstanceConfig instanceConfig = (IComponentInstanceConfig) data;
					if (!ValidationHelper.getDriverValidationProblems(instanceConfig, ErrorLevels.LEVEL_ERROR).isEmpty()) {
						return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
					}
					final int problemLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(instanceConfig);
					if ((instanceConfig.getError() != null) || (problemLevel >= ErrorLevels.LEVEL_ERROR)) {
						return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
					}
					if ((instanceConfig.getWarning() != null) || (problemLevel == ErrorLevels.LEVEL_WARNING)) {
						return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_WARNING);
					}
					if ((instanceConfig.getInfo() != null) || (problemLevel == ErrorLevels.LEVEL_INFORMATION)) {
						return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_INFORMATION);
					}
					if (!instanceConfig.getComment().isEmpty()) {
						return ToolsImages.getImage(IToolsImages.ICON_LEGEND);
					}
				}
				return null;
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getTooltipText(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public String getTooltipText(TileItemData itemData) {
				StringBuilder result = new StringBuilder();
				Object data = itemData.getData();
				if (data instanceof IComponentInstanceConfig) {
					IComponentInstanceConfig instanceConfig = (IComponentInstanceConfig) data;
					IComponentConfig componentConfig = instanceConfig.getChildContext().getComponentConfig();
					result.append(ComponentsView.getTooltipText(instanceConfig, controllerWrapper.getController().getCategories()));
					if (componentConfig != null) {
						result.append(UtilsText.XHTML_BR);
						result.append(UtilsText.XHTML_BR);
						result.append(ComponentsView.getTooltipText(componentConfig, controllerWrapper.getController().getCategories()));
					}
				}
				return (result.length() == 0) ? null : result.toString();
			}

			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileLabelProvider#getTooltipText(com.nxp.swtools.common.ui.utils.tileviewer.TileCategoryData)
			 */
			@Override
			public String getAddButtonTooltipText(@NonNull TileCategoryData tileCategoryData) {
				Object data = tileCategoryData.getData();
				if (data instanceof Category) {
					final String categoryName = UtilsText.safeString(((Category) data).getUINameString());
					return MessageFormat.format(Messages.get().ComponentsView_CategoryAddButtonTooltip, categoryName);
				}
				return UtilsText.EMPTY_STRING;
			}

			/**
			 * @param tileCategoryData
			 * @return
			 */
			@Override
			public String getTooltipText(TileCategoryData tileCategoryData) {
				Object data = tileCategoryData.getData();
				if (data instanceof Category) {
					return ((Category) data).getDescription();
				}
				return UtilsText.EMPTY_STRING;
			}
		});
		tileViewerLoc.setFilter(new ITileFilter() {
			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.tileviewer.ITileFilter#select(com.nxp.swtools.common.ui.utils.tileviewer.TileItemData)
			 */
			@Override
			public boolean select(TileItemData tileItemData) {
				Object data = tileItemData.getData();
				if (data instanceof IComponentInstanceConfig) {
					IComponentInstanceConfig componentInstanceConfig = (IComponentInstanceConfig) data;
					String filterStringLowerCase = filterString.toLowerCase();
					return componentInstanceConfig.getUiName().toLowerCase().contains(filterStringLowerCase);
				}
				return false;
			}
		});
		tileViewerLoc.setInput(tileCategoriesData);
		SWTFactoryProxy.INSTANCE.setTestId(tileViewerLoc.getControl(), TestIDs.PERIPHS_COMPONENT_TREE);

		registerListener(EventTypes.INITIALIZATION | EventTypes.CHANGE | EventTypes.SETTING_CHANGE, new IEventListener() {

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(java.util.Collection)
			 */
			@Override
			public void handle(@NonNull Collection<@NonNull ToolEvent> events) {
				boolean hasChangeEvent = events.stream()
						.anyMatch(e -> e.isType(EventTypes.CHANGE));
				refresh(hasChangeEvent);
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(@NonNull ToolEvent event) {
				if (event.isType(EventTypes.CHANGE)) {
					refresh(true); // perform deep refresh only in case of event type CHANGE
				} else {
					refresh(false); // perform shallow refresh (number of tiles was not modified)
				}
			}
		});
		ValidationEngineFactory.addListener(validationsListener);
	}

	/**
	 * Creates data for tileViewer from categories and component instance configs
	 * @return list of newly created TileCategoryData
	 */
	private @NonNull ArrayList<@NonNull TileCategoryData> createTilesData() {
		ArrayList<@NonNull TileCategoryData> tileCategoriesData = new ArrayList<>();
		Categories categories = controllerWrapper.getController().getCategories();
		List<@NonNull Category> categoriesList = new ArrayList<>();
		TileCategoryData tileCategoryData;

		if (categories != null) {
			categoriesList = categories.getCategories();
			for (Category category : categoriesList) {
				// create data only if there is any available component for given category
				final List<@NonNull ConfigurationComponentTypeId> configCompsForCategory = controllerWrapper.getController().getMcu().getAvailableComponents().getConfigCompsTypeIdsForCategory(category.getId());
				if (!configCompsForCategory.isEmpty()) {
					Collection<@NonNull IComponentInstanceConfig> instancesOfCategory = controllerWrapper.getController().getFunctionalGroup().getInstancesOfCategory(category.getId());
					tileCategoryData = createTileCategoryData(category, instancesOfCategory);
					tileCategoriesData.add(tileCategoryData);
				}
			}
		}

		final List<@NonNull ConfigurationComponentTypeId> configCompsTypeIdsWithoutMatchingCategory = controllerWrapper.getController().getMcu().getAvailableComponents().getConfigCompsTypeIdsWithoutMatchingCategory(categoriesList);
		AddComponentDialog.filterNonDisplayableComponents(configCompsTypeIdsWithoutMatchingCategory, controllerWrapper.getController().getProfile());
		if (!configCompsTypeIdsWithoutMatchingCategory.isEmpty()) {
			Category otherCategory = createOtherCategory();
			Collection<@NonNull IComponentInstanceConfig> instancesWithoutMatchingCategory = controllerWrapper.getController().getFunctionalGroup().getInstancesWithoutMatchingCategory(categoriesList);
			tileCategoryData = createTileCategoryData(otherCategory, instancesWithoutMatchingCategory);
			tileCategoriesData.add(tileCategoryData);
		}

		return tileCategoriesData;
	}

	/**
	 * Creates data for tileViewer for a category and its component instance configs.
	 * @param category category for which to create tile data
	 * @param componentInstanceConfigs list of component instance configs that belong to given category
	 * @return newly created TileCategoryData for tileViewer
	 */
	private static @NonNull TileCategoryData createTileCategoryData(@NonNull Category category, @NonNull Collection<@NonNull IComponentInstanceConfig> componentInstanceConfigs) {
		TileCategoryData tileCategoryData = new TileCategoryData(category);
		for (IComponentInstanceConfig componentInstanceConfig : componentInstanceConfigs) {
			TileItemData tileItemData = new TileItemData(componentInstanceConfig);
			tileCategoryData.addChild(tileItemData);
		}
		return tileCategoryData;
	}

	/**
	 * Creates tooltip for component or component instance
	 * @param child from which the tooltip would be created
	 * @param categories to get category from
	 * @return tooltip text
	 */
	@Nullable static String getTooltipText(@NonNull IChild child, Categories categories) {
		int problemLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(child);
		Set<@NonNull String> depDescs = new HashSet<>();
		List<@NonNull IValidationProblem> driverProblems = ValidationHelper.getDriverValidationProblems(child, ErrorLevels.LEVEL_ERROR);
		if (!driverProblems.isEmpty()) {
			problemLevel = ErrorLevels.LEVEL_ERROR;
			Set<@NonNull String> depDescsDrivers = driverProblems.stream().map(x -> x.getDependency().getDescription()).collect(CollectorsUtils.toSet());
			depDescs.addAll(depDescsDrivers);
		}
		Set<@NonNull String> depDescsComponent = ValidationHelper.getComponentValidationProblems(child, problemLevel).stream().map(
				p -> p.getDependency().getDescription()).collect(CollectorsUtils.toSet());
		depDescs.addAll(depDescsComponent);
		final String problemMsg = depDescs.isEmpty() ? null : ChildValidationHelper.mergeProblems(depDescs.toArray(new @Nullable String[depDescs.size()]));
		StringBuilder result = new StringBuilder();
		if (child instanceof IComponentInstanceConfig) {
			IComponentInstanceConfig instanceConfig = (IComponentInstanceConfig) child;
			if (instanceConfig.getType().equals(Components.COMPONENT_TYPE_USED_FOR_MARKING_AS_USED)) {
				result.append(MessageFormat.format(Messages.get().ComponentsView_ComponentIsMarkedAsUsed, instanceConfig.getPeripheral()));
			} else {
				// Name
				result.append("<u>"); //$NON-NLS-1$
				result.append(UtilsText.htmlB(instanceConfig.getUiName()));
				result.append(UtilsText.LEFT_BRACKET);
				// Peripheral(s)
				String peripheralsString = UtilsText.EMPTY_STRING;
				Collection<@NonNull String> peripherals = instanceConfig.getPeripherals();
				Collection<@NonNull String> erroneousPeripherals = instanceConfig.getErroneousPeripherals();
				if (!erroneousPeripherals.isEmpty()) {
					String erroneousPeripheralsText = CollectionsUtils.formatList(erroneousPeripherals, UtilsText.COMMA_SPACE, null);
					peripheralsString += ToolTipableFormatter.colorize(erroneousPeripheralsText, ToolsColors.HTML_TOOLTIP_ERROR_COLOR);
				}
				peripherals.removeAll(erroneousPeripherals);
				if (!peripheralsString.isEmpty() && !peripherals.isEmpty()) {
					peripheralsString += UtilsText.COMMA_SPACE;
				}
				peripheralsString += CollectionsUtils.formatList(peripherals, UtilsText.COMMA_SPACE, null);
				result.append(MessageFormat.format(UtilsText.safeString(Messages.get().ComponentEditor_Peripherals), peripheralsString));
				result.append(UtilsText.SEMICOLON_SPACE + Messages.get().ComponentsView_Tooltips_Component + UtilsText.COLON_SPACE
						+ instanceConfig.getComponent().getId());
				result.append(UtilsText.RIGHT_BRACKET + "</u>" + UtilsText.XHTML_BR); //$NON-NLS-1$

				// Description
				result.append(instanceConfig.getDescription());
				// Category
				String categoryString = UtilsText.EMPTY_STRING;
				String categoryIdFinal = UtilsText.safeString(instanceConfig.getComponent().getCategory());
				IContext contextFinal = instanceConfig.getExpressionContext();
				if (categories != null) {
					categoryString = categories.getCategoryLabel(categoryIdFinal, contextFinal);
				}
				if (categoryString != null) {
					result.append(UtilsText.LEFT_BRACKET).append(categoryString).append(UtilsText.RIGHT_BRACKET);
				}
				result.append(UtilsText.XHTML_BR);
				boolean showStatusSection = (instanceConfig.isEditingLocked() || !instanceConfig.isEnabled());
				if (showStatusSection) {
					result.append(UtilsText.XHTML_BR);
					result.append(UtilsText.htmlB(Messages.get().ComponentsView_Tooltips_Status + UtilsText.COLON_SPACE));
					// Locked
					if (instanceConfig.isEditingLocked()) {
						result.append(UtilsText.XHTML_BR);
						result.append(Messages.get().ComponentInstance_Locked);
					}
					// Enabled
					if (!instanceConfig.isEnabled()) {
						result.append(UtilsText.XHTML_BR);
						result.append(Messages.get().ComponentsView_Tooltips_Disabled);
					}
				}
				// State, error and warning
				String status = ToolTipableFormatter.createStatus(instanceConfig, problemMsg, problemLevel);
				if (!status.isEmpty()) {
					result.append(UtilsText.XHTML_BR);
					result.append(status);
				}
				String comment = instanceConfig.getComment();
				if (!comment.isEmpty()) {
					result.append(UtilsText.XHTML_BR);
					result.append(UtilsText.htmlB(Messages.get().ComponentSettingView_CommentLabel));
					result.append(UtilsText.XHTML_BR);
					comment = UtilsText.safeString(UtilsText.cutOffAfter(comment, CUTOFF_COMMENT_LINES))
							.replaceAll(UtilsText.REGULAR_ANY_EOL, UtilsText.XHTML_BR);
					result.append(comment);
				}
			}
		}
		if (child instanceof IComponentConfig) {
			ComponentConfig componentConfig = (ComponentConfig) child;
			// sw_comp_ref
			List<SWComponent> components = componentConfig.getComponent().getComponents();
			if (!components.isEmpty()) {
				result.append(UtilsText.htmlB(Messages.get().ComponentsView_Tooltips_SdkReferences + UtilsText.COLON_SPACE));
				result.append(UtilsText.XHTML_BR);
				for (SWComponent component : components) {
					result.append(component.getName() + UtilsText.COLON_SPACE + component.getVersion());
					result.append(UtilsText.XHTML_BR);
				}
			}
			// Full ID - directory with hash
			result.append(UtilsText.htmlB(Messages.get().ComponentsView_Tooltips_TypeId + UtilsText.COLON_SPACE));
			result.append(UtilsText.XHTML_BR);
			result.append(componentConfig.getComponentTypeId());
			// State, error and warning
			String status = ToolTipableFormatter.createStatus(componentConfig, problemMsg, problemLevel);
			if (!status.isEmpty()) {
				result.append(UtilsText.XHTML_BR);
				result.append(status);
			}
		}
		// Return result
		return result.length() == 0 ? null :
			ToolTipableFormatter.trimExcessBreakLine(UtilsText.safeString(result.toString()));
	}

	/**
	 * Refresh content.
	 */
	void refresh() {
		ArrayList<@NonNull TileCategoryData> tileCategoriesData = createTilesData();
		if (tileViewer != null) {
			TileViewer tileViewerLoc = tileViewer;
			tileViewerLoc.setInput(tileCategoriesData);
		}
	}

	/**
	 * Refresh content.
	 * @param fullRefresh {@code true} if whole content has to be recreated. {@code false} if only existing objects have to be updated
	 */
	void refresh(boolean fullRefresh) {
		if (fullRefresh) {
			refresh();
		} else if (tileViewer != null) {
			tileViewer.refresh();
		}
	}

	/**
	 * Create category "other".
	 * @return newly created category with pre-set id, ui name and description
	 */
	public static @NonNull Category createOtherCategory() {
		return new Category(CATEGORY_OTHER_ID, CATEGORY_OTHER_UI_NAME, null, CATEGORY_OTHER_DESCRIPTION);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.view.ToolView#resetView()
	 */
	@Override
	protected void resetView() {
		super.resetView();
		if (filterText != null) {
			filterText.setText(UtilsText.EMPTY_STRING);
		}
		if (tileViewer != null) {
			tileViewer.refresh();
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.APeriphsViewBase#dispose()
	 */
	@Override
	public void dispose() {
		ValidationEngineFactory.removeListener(validationsListener);
		super.dispose();
	}
}
