/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.dialogs.PatternFilter;
import org.osgi.framework.Version;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.ComparatorHelpers;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.kex.MfactConstants;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.PeriphsProfile;
import com.nxp.swtools.periphs.model.data.AvailableComponents;
import com.nxp.swtools.periphs.model.data.Categories;
import com.nxp.swtools.periphs.model.data.Components;
import com.nxp.swtools.periphs.model.data.ConfigurationComponent;
import com.nxp.swtools.periphs.model.templates.component.ComponentTemplate;
import com.nxp.swtools.periphs.model.templates.component.ComponentTemplateProperties;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.toolchainproject.ISdkComponentInProject;
import com.nxp.swtools.provider.toolchainproject.IToolchainProjectWithSdk;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.IConfigurationComponent;
import com.nxp.swtools.resourcetables.model.data.SWComponent;
import com.nxp.swtools.resourcetables.model.data.SWComponent.Match;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.storage.StorageHelper;

/**
 * Dialog used for creating new components.
 * @author Juraj Ondruska
 */
public class AddComponentDialog extends Dialog {
	/** Logger of the class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(AddComponentDialog.class);
	/** Associated view site */
	private final @NonNull IViewSite viewSite;
	/** The controller */
	final @NonNull IControllerWrapper controllerWrapper;
	/** Peripheral for which to create component */
	final @Nullable String peripheral;
	/** List of ConfigurationComponentTypeIds that were created by this dialog - used to be able to get result of the dialog */
	final @Nullable List<@NonNull ConfigurationComponentTypeId> configCompsTypeIds;
	/** Whether the filtering tool-chain project components is allowed and the options is visible in the GUI */
	boolean toolchainFilterAllowed;
	/** Add button */
	@Nullable Button addButton;
	/** Reference to table viewer */
	private @Nullable TreeViewer treeViewer;
	/** Whether to open view after component(s) creation */
	private final boolean openView;
	/** Functional group for which to add components */
	private final @NonNull String functionalGroup;
	/** Storage helper used for saving/restoring values to/from preferences */
	final @NonNull StorageHelper storageHelper = new StorageHelper(MfactConstants.TOOL_PERIPHERALS_ID);
	/** Result of the dialog - instance that was added by the dialog */
	private @Nullable IChildProvidable result;
	/** Combobox - provides which components should be displayed */
	private @Nullable InstantSearchList componentFilterCombo;
	/** Current filter type */
	protected FilteringType currentFilter;
	/** Collection of all components */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> allComponents = Collections.emptyList();
	/** Collection of latest components */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> latestComponents = Collections.emptyList();
	/** Collection of components that have SDK component in project */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> filteredComponents = Collections.emptyList();
	/** Collection of components that have SDK component in SDK package of project */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> sdkComponents = Collections.emptyList();
	/** This number is used to compute the table's height; number of rows that should be visible by default */
	private static final int DEFAULT_ROWS_NUM = 12;
	/** This number is used to compute the table's width given its height */
	private static final float DEFAULT_VIEWER_WIDTH_HEIGHT_RATIO = 16f / 9f;
	/** Key to the previously selected filter preference in the preference storage */
	static final @NonNull String FILTER_KEY = AddComponentDialog.class.getName() + "filter"; //$NON-NLS-1$
	/** Filter value - components in SDK package */
	private static final @NonNull String FILTER_VALUE_SDK = "sdk"; //$NON-NLS-1$
	/** Filter value - components in toolchain project */
	private static final @NonNull String FILTER_VALUE_PROJECT = "project"; //$NON-NLS-1$
	/** Filter value - latest components */
	private static final @NonNull String FILTER_VALUE_LATEST = "latest"; //$NON-NLS-1$
	/** Filter value - all components */
	private static final @NonNull String FILTER_VALUE_ALL = "all"; //$NON-NLS-1$
	/** Highest permitted value for part of version e.g. 2.7.99 */
	private static final int HIGHEST_PERMITED_VERSION_PART = 99;
	/** Score for not required driver that did not match */
	private static final @NonNull BigInteger SCORE_NOT_REQUIRED_NOT_MATCHING = BigInteger.valueOf(1_000_000).negate();
	/** Score for not detectable SDK component in toolchain project */
	private static final @NonNull BigInteger SCORE_UNDETECTABLE_COMPONENT = SCORE_NOT_REQUIRED_NOT_MATCHING;
	/** Error message when element in viewer is no entry object */
	private static final String INPUT_IS_NOT_ENTRY_MESSAGE = "[TOOL] [ADD_COMPONENT_DIALOG] Given element is not AddComponentDialogEntry: {0}"; //$NON-NLS-1$
	/** Number of columns in options composite */
	private static final int OPTIONS_COMPOSITE_COLUMNS = 2;
	/** ID of button that is used to open SDK manager */
	private static final int OPEN_SDK_MANAGER_BUTTON_ID = 20;
	/** Maximal length of use case description shown in the tree row */
	private static final int USE_CASE_DESCRIPTION_LENGTH_MAX = 50;

	/**
	 * Open a dialog.
	 * @param properties for creation of AddComponentDialog
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog open(@NonNull AddComponentDialogProperties properties) {
		AddComponentDialog dialog = new AddComponentDialog(properties);
		dialog.setBlockOnOpen(properties.getBlockOnOpen());
		dialog.updateAddButtonState();
		dialog.open();
		return dialog;
	}

	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param forPeripheral peripheral for which to add component
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param toolchainFilterAllowed whether the toolchain filter is allowed and should be visible
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog open(@NonNull IViewSite viewSite, @Nullable String forPeripheral,
			@NonNull String functionalGroup, boolean openView, boolean toolchainFilterAllowed, @NonNull IControllerWrapper controllerWrapper) {
		AddComponentDialogProperties properties = new AddComponentDialogProperties(viewSite, functionalGroup, controllerWrapper, openView, toolchainFilterAllowed);
		properties.setPeripheral(forPeripheral);
		return open(properties);
	}

	/**
	 * Enables/disables the "OK"/add button depending on the current selection
	 */
	protected void updateAddButtonState() {
		Button addButtonLoc = addButton;
		TreeViewer tableViewerLoc = treeViewer;
		if ((addButtonLoc != null) && (tableViewerLoc != null)) {
			setButtonState(addButtonLoc, tableViewerLoc.getSelection());
		}
	}

	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param forPeripheral peripheral for which to add component
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog open(@NonNull IViewSite viewSite, @Nullable String forPeripheral, @NonNull IControllerWrapper controllerWrapper) {
		AddComponentDialogProperties properties = new AddComponentDialogProperties(viewSite, Controller.getInstance().getFunctionalGroup().getName(), controllerWrapper, true, true);
		properties.setPeripheral(forPeripheral);
		return open(properties);
	}

	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param configCompsTypeIds list of ConfigurationComponentTypeIds that should be visible in dialog
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog openForComponents(@NonNull IViewSite viewSite, @Nullable List<@NonNull ConfigurationComponentTypeId> configCompsTypeIds, @NonNull IControllerWrapper controllerWrapper) {
		AddComponentDialogProperties properties = new AddComponentDialogProperties(viewSite, Controller.getInstance().getFunctionalGroup().getName(), controllerWrapper, true, true);
		properties.setConfigCompsTypeIds(configCompsTypeIds);
		return open(properties);
	}

	/**
	 * Constructor.
	 * @param properties of this dialog
	 */
	private AddComponentDialog(@NonNull AddComponentDialogProperties properties) {
		super(properties.getViewSite().getWorkbenchWindow().getShell());
		this.toolchainFilterAllowed = properties.getToolchainFilterAllowed();
		this.viewSite = properties.getViewSite();
		this.peripheral = properties.getPeripheral();
		this.configCompsTypeIds = properties.getConfigCompsTypeIds();
		this.functionalGroup = properties.getFunctionalGroupName();
		this.openView = properties.getOpenView();
		this.result = null;
		this.controllerWrapper = properties.getControllerWrapper();
		IToolchainProjectWithSdk toolchain = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		if ((toolchain != null) && toolchain.wasProjectDetected()) {
			this.currentFilter = toolchain.isGetAllSdkComponentsSupported() ? FilteringType.SDK_ONLY : FilteringType.PROJECT_ONLY;
		} else {
			this.currentFilter = FilteringType.LATEST;
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}

	/**
	 * Types of filtering options available for this dialog
	 * @author Tomas Rudolf - nxf31690
	 */
	public enum FilteringType {
		/** Indicates that all components should be returned */
		ALL,
		/** Indicates that only latest version of component should be returned */
		LATEST,
		/** Indicates that only components that are present in the associated project should be returned */
		PROJECT_ONLY,
		/** Indicates that only components which are available in manifest of the associated project should be returned */
		SDK_ONLY
	}

	/**
	 * Remove configuration components that are not active or worse match for current configuration.
	 * @param configComps Collection of configuration components which are being modified
	 * @param filteringType type of filtering which will be applied to list of components
	 * @param latestVersionOnly Whether to choose only the latest version of each component (set to {@code true}
	 * if choose latest, {@code false} otherwise)
	 * @param toolchainProject Toolchain project for the configuration
	 * @return new collection of configuration components
	 */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> filterComponents(
			@NonNull Collection<@NonNull ConfigurationComponentTypeId> configComps,
			@NonNull FilteringType filteringType,
			@Nullable IToolchainProjectWithSdk toolchainProject) {
		Map<@NonNull String, @NonNull List<@NonNull ConfigurationComponentTypeId>> compsByType = configComps.stream().collect(
				Collectors.groupingBy(ConfigurationComponentTypeId::getType, Collectors.toCollection(ArrayList::new)));
		List<@NonNull ConfigurationComponentTypeId> components = new ArrayList<>();
		for (Map.Entry<@NonNull String, @NonNull List<@NonNull ConfigurationComponentTypeId>> entry : compsByType.entrySet()) {
			assert entry.getKey() != null;
			String type = UtilsText.safeString(entry.getKey()); // using {@link UtilsText#safeString(String)} only to avoid dealing with null warning
			ConfigurationComponentTypeId confCompTypeId = null;
			// 1. add mandatory components from the project
			collectMandatoryComponents(components, entry.getValue());
			// Filtering type ALL is not handled here
			if (filteringType == FilteringType.PROJECT_ONLY) {
				// 2. use the same component already in the configuration if possible
				confCompTypeId = controllerWrapper.getController().getProfile().getActiveComponents().getComponentTypeIdByType(type);
				// 3. use the same component in the project if possible
				if ((confCompTypeId == null) && (toolchainProject != null)) {
					confCompTypeId = findComponentInProject(CollectionsUtils.safeList(entry.getValue()), toolchainProject);
				}
				// 4. use the best matching component available
				if ((confCompTypeId == null) && ((toolchainProject == null) || !toolchainProject.wasProjectDetected())) {
					confCompTypeId = controllerWrapper.getController().getMcu().getAvailableComponents().getBestMatchingConfigCompType(entry.getValue());
				}
			} else if (filteringType == FilteringType.SDK_ONLY) {
				// 2. use the same component already in the configuration if possible
				confCompTypeId = controllerWrapper.getController().getProfile().getActiveComponents().getComponentTypeIdByType(type);
				// 3. use the same component in the project if possible
				if ((confCompTypeId == null) && (toolchainProject != null)) {
					confCompTypeId = findComponentInSdk(CollectionsUtils.safeList(entry.getValue()), toolchainProject);
				}
				// 4. use the best matching component available
				if ((confCompTypeId == null) && ((toolchainProject == null) || !toolchainProject.wasProjectDetected())) {
					confCompTypeId = controllerWrapper.getController().getMcu().getAvailableComponents().getBestMatchingConfigCompType(entry.getValue());
				}
			} else if (filteringType == FilteringType.LATEST) {
				// 2. use the latest version of the component available
				confCompTypeId = Components.getLatestVersionConfigCompType(entry.getValue());
			}
			if (confCompTypeId != null) {
				if (!components.contains(confCompTypeId)) {
					components.add(confCompTypeId);
				}
			} else {
				LOGGER.severe(String.format("[TOOL] Could not find any matching component for type %1s", type)); //$NON-NLS-1$
			}
		}
		return components;
	}

	/**
	 * Collect mandatory (not referencing a SW Component and/or marked global-only) configuration components.
	 * @param mandatoryComps collection to receive mandatory components
	 * @param configComps configuration components of the same type ({@link ConfigurationComponentTypeId#getType()})
	 */
	public static void collectMandatoryComponents(
			@NonNull List<@NonNull ConfigurationComponentTypeId> mandatoryComps,
			@NonNull List<@NonNull ConfigurationComponentTypeId> configComps) {
		for (ConfigurationComponentTypeId configComp : configComps) {
			if (isMandatoryComponent(configComp) && !mandatoryComps.contains(configComp)) {
				mandatoryComps.add(configComp);
			}
		}
	}

	/**
	 * Search for the configuration component among the specified ones referencing an equal SW component in the toolchain project
	 * @param configComps configuration components of the same type ({@link ConfigurationComponentTypeId#getType()})
	 * @param toolchainProject Toolchain project for the configuration
	 * @return configuration component info or {@code null}
	 */
	private static @Nullable ConfigurationComponentTypeId findComponentInProject(
			@NonNull List<@NonNull ConfigurationComponentTypeId> configComps,
			@NonNull IToolchainProjectWithSdk toolchainProject) {
		BigInteger bestMatchScore = null;
		ConfigurationComponentTypeId bestMatch = null;
		for (ConfigurationComponentTypeId configComp : configComps) {
			if (isMandatoryComponent(configComp)) {
				// mandatory components are processed elsewhere
				continue;
			}
			List<SWComponent> swComps = configComp.getConfigurationComponent().getComponents();
			BigInteger componentMatchScore = null;
			for (SWComponent driver : swComps) {
				String driverName = driver.getName();
				ISdkComponentInProject prjComp = toolchainProject.getSdkComponent(driverName);
				Version sdkComponentVersion = null;
				if (prjComp == null || !prjComp.getID().equals(driverName)) {
					// Driver was not found or the found one does not have the required name
					componentMatchScore = null;
					break;
				}
				sdkComponentVersion = prjComp.getVersion();
				if (sdkComponentVersion == null) { // SDK component was found in project, but does not contain any version -> undetectable SDK component is always visible
					componentMatchScore = SCORE_UNDETECTABLE_COMPONENT;
					continue;
				}
				BigInteger localScore = Components.computeMatchScore(driver.getVersion(), sdkComponentVersion, driver.getMatchString(), driver.getUntilVersion());
				if (localScore == null && driver.getRequired()) {
					// If driver is required by the component and does not match then this component cannot be used
					componentMatchScore = null;
					break;
				} else {
					// Driver matches and it's score will be added to match sum of this component
					if (componentMatchScore == null) {
						componentMatchScore = localScore;
					} else {
						if (localScore != null) {
							componentMatchScore = componentMatchScore.add(localScore);
						} else {
							// Make the score of not required not matching driver worse than the matching one (even if the matching is not perfect e.g. score=-3)
							componentMatchScore = componentMatchScore.add(SCORE_NOT_REQUIRED_NOT_MATCHING);
						}
					}
				}
			}
			if ((componentMatchScore != null) && ((bestMatchScore == null) || (componentMatchScore.compareTo(bestMatchScore) > 0))) {
				// set only if the match score is defined (all drivers are available)
				bestMatch = configComp;
				bestMatchScore = componentMatchScore;
			}
		}
		return bestMatch;
	}

	/**
	 * Search for the configuration component among the specified ones referencing an equal SW component in SDK obtained from the toolchain project
	 * @param configComps configuration components of the same type ({@link ConfigurationComponentTypeId#getType()})
	 * @param toolchainProject Toolchain project for the configuration
	 * @return configuration component info or {@code null}
	 */
	private static @Nullable ConfigurationComponentTypeId findComponentInSdk(
			@NonNull List<@NonNull ConfigurationComponentTypeId> configComps,
			@NonNull IToolchainProjectWithSdk toolchainProject) {
		// FIXME TomasR v13 maintenance - Unify with findComponentInProject by changing argument from toolchain project to function accepting sdk component id and providing ISdkComponentInProject
		BigInteger bestMatchScore = null;
		ConfigurationComponentTypeId bestMatch = null;
		List<@NonNull ISdkComponentInProject> toolchainSdkComponents = toolchainProject.getAllSdkComponents();
		for (ConfigurationComponentTypeId configComp : configComps) {
			if (isMandatoryComponent(configComp)) {
				// mandatory components are processed elsewhere
				continue;
			}
			List<SWComponent> swComps = configComp.getConfigurationComponent().getComponents();
			BigInteger componentMatchScore = null;
			for (SWComponent driver : swComps) {
				String driverName = driver.getName();
				List<@NonNull ISdkComponentInProject> sdkComponentsWithWantedId = toolchainSdkComponents.stream().filter(x -> x.getID().equals(driverName)).collect(CollectorsUtils.toList());
				if (sdkComponentsWithWantedId.isEmpty()) {
					// SDK component with wanted id was not found in SDK attached to toolchain project
					break;
				}
				if (sdkComponentsWithWantedId.size() > 1) {
					LOGGER.log(Level.SEVERE, "[TOOL] SDK attached to toolchain project contains more than one SDK component with id {0}", driverName); //$NON-NLS-1$
				}
				ISdkComponentInProject prjComp = sdkComponentsWithWantedId.get(0);
				Version sdkComponentVersion = null;
				if (!prjComp.getID().equals(driverName)) {
					// Found SDK component does not have the required id
					componentMatchScore = null;
					break;
				}
				sdkComponentVersion = prjComp.getVersion();
				if (sdkComponentVersion == null) { // SDK component was found in project, but does not contain any version -> undetectable SDK component is always visible
					componentMatchScore = SCORE_UNDETECTABLE_COMPONENT;
					continue;
				}
				BigInteger localScore = Components.computeMatchScore(driver.getVersion(), sdkComponentVersion, driver.getMatchString(), driver.getUntilVersion());
				if (localScore == null && driver.getRequired()) {
					// If driver is required by the component and does not match then this component cannot be used
					componentMatchScore = null;
					break;
				} else {
					// Driver matches and it's score will be added to match sum of this component
					if (componentMatchScore == null) {
						componentMatchScore = localScore;
					} else {
						if (localScore != null) {
							componentMatchScore = componentMatchScore.add(localScore);
						} else {
							// Make the score of not required not matching driver worse than the matching one (even if the matching is not perfect e.g. score=-3)
							componentMatchScore = componentMatchScore.add(SCORE_NOT_REQUIRED_NOT_MATCHING);
						}
					}
				}
			}
			if ((componentMatchScore != null) && ((bestMatchScore == null) || (componentMatchScore.compareTo(bestMatchScore) > 0))) {
				// set only if the match score is defined (all drivers are available)
				bestMatch = configComp;
				bestMatchScore = componentMatchScore;
			}
		}
		return bestMatch;
	}

	/**
	 * Determines whether the component must be offered always.
	 * @param configComp configuration component
	 * @return {@code true} if mandatory, {@code false} otherwise
	 */
	private static boolean isMandatoryComponent(@NonNull ConfigurationComponentTypeId configComp) {
		return configComp.getConfigurationComponent().getComponents().isEmpty();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite dialogComposite = (Composite) super.createDialogArea(parent);
		dialogComposite.setLayout(new GridLayout());
		dialogComposite.getShell().setText(Messages.get().ComponentsView_SelectComponentDialog_Title);
		currentFilter = restoreFilterFromStorage();
		createOptionsComposite(dialogComposite);
		final TreeViewer viewer = createTreeViewer(dialogComposite);
		createTreeColumns(viewer);
		prepareTreeViewer(viewer);
		updateContent(viewer, peripheral);
		return dialogComposite;
	}

	/**
	 * Updates content of the viewer
	 * @param viewer to be updated
	 * @param peripheralInstance peripheral for which the dialog was opened
	 */
	public void updateContent(final @NonNull TreeViewer viewer, final @Nullable String peripheralInstance) {
		allComponents = getComponents(peripheralInstance);
		// remove component if its "hidden" flag is set to true and it is not used in project
		filterNonDisplayableComponents(allComponents, controllerWrapper.getController().getProfile());
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		latestComponents = filterComponents(allComponents, FilteringType.LATEST, toolchainProject);
		filteredComponents = filterComponents(allComponents, FilteringType.PROJECT_ONLY, toolchainProject);
		sdkComponents = filterComponents(allComponents, FilteringType.SDK_ONLY, toolchainProject);
		addListenersToFilterCombo(viewer);
		setTreeViewerInputComps(viewer);
		selectFirstItem(viewer);
		packColumns();
	}

	/**
	 * Restores previously selected filter from preference storage
	 * @return filtering type for stored id. When id is not known then returns ALL
	 */
	private FilteringType restoreFilterFromStorage() {
		String lastFilterState = storageHelper.loadString(FILTER_KEY, UtilsText.MISSING_MSG);
		switch (lastFilterState) {
			default:
				return currentFilter;
			case FILTER_VALUE_ALL: {
				return FilteringType.ALL;
			}
			case FILTER_VALUE_LATEST: {
				return FilteringType.LATEST;
			}
			case FILTER_VALUE_PROJECT: {
				return FilteringType.PROJECT_ONLY;
			}
			case FILTER_VALUE_SDK: {
				return FilteringType.SDK_ONLY;
			}
		}
	}

	/**
	 * @param peripheralInstance peripheral or {@code null}
	 * @return components of peripheral or all components when peripheralInstance is {@code null}
	 */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> getComponents(final @Nullable String peripheralInstance) {
		Collection<@NonNull ConfigurationComponentTypeId> components;
		// get components based on component IDs given on dialog creation
		if (configCompsTypeIds != null) {
			components = configCompsTypeIds;
		} else {
			// get components based on peripheral
			components = peripheralInstance != null ? controllerWrapper.getController().getComponentsOfPeripheral(peripheralInstance) : controllerWrapper.getController().getComponents();
		}
		return components;
	}

	/**
	 * Creates composite with options
	 * @param parentComposite parent composite
	 */
	private void createOptionsComposite(final @NonNull Composite parentComposite) {
		Composite optionsComposite = new Composite(parentComposite, parentComposite.getStyle());
		optionsComposite.setLayout(new GridLayout(OPTIONS_COMPOSITE_COLUMNS, true));
		Label label = new Label(optionsComposite, SWT.NONE);
		label.setText(Messages.get().AddComponentDialog_FilterCombo_Label);
		label.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
		InstantSearchList combo = new InstantSearchList(optionsComposite, SWT.BORDER);
		SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_FILTER_COMBO);
		componentFilterCombo = combo;
		ArrayList<String> itemsList = new ArrayList<>();
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		if (toolchainFilterAllowed && toolchainProject != null && toolchainProject.wasProjectDetected()) {
			itemsList.add(Messages.get().AddComponentDialog_FilterCombo_ProjectComponents);
			if (toolchainProject.isGetAllSdkComponentsSupported()) {
				itemsList.add(Messages.get().AddComponentDialog_FilterCombo_SdkComponents);
			}
		}
		itemsList.add(Messages.get().AddComponentDialog_FilterCombo_LatestComponents);
		itemsList.add(Messages.get().AddComponentDialog_FilterCombo_AllComponents);
		combo.setItems(itemsList.toArray(new @NonNull String[itemsList.size()]));
		combo.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
		String itemToSelect;
		if ((currentFilter == FilteringType.PROJECT_ONLY || currentFilter == FilteringType.SDK_ONLY) && (toolchainProject == null || !toolchainProject.wasProjectDetected())) {
			// When last preference was project only or SDK only and there is no toolchain project then set the preference to Latest and store that preference
			currentFilter = FilteringType.LATEST;
			setFilteringPreferenceStorage(currentFilter);
		}
		switch (currentFilter) {
			default:
			case ALL: {
				itemToSelect = Messages.get().AddComponentDialog_FilterCombo_AllComponents;
				break;
			}
			case LATEST: {
				itemToSelect = Messages.get().AddComponentDialog_FilterCombo_LatestComponents;
				break;
			}
			case PROJECT_ONLY: {
				itemToSelect = Messages.get().AddComponentDialog_FilterCombo_ProjectComponents;
				break;
			}
			case SDK_ONLY: {
				itemToSelect = Messages.get().AddComponentDialog_FilterCombo_SdkComponents;
				break;
			}
		}
		combo.select(itemToSelect);
	}

	/**
	 * Creates tree viewer
	 * @param parentComposite parent composite
	 * @return created tree viewer
	 */
	private @NonNull TreeViewer createTreeViewer(final @NonNull Composite parentComposite) {
		// create list viewer
		PatternFilter filter = new PatternFilter() {
			/* (non-Javadoc)
			 * @see org.eclipse.ui.dialogs.PatternFilter#isLeafMatch(org.eclipse.jface.viewers.Viewer, java.lang.Object)
			 */
			@Override
			protected boolean isLeafMatch(@NonNull Viewer viewer, @NonNull Object element) {
				TreeViewer treeViewerLoc = (TreeViewer) viewer;
				for (int i = 0; i < treeViewerLoc.getTree().getColumnCount(); i++) {
					ColumnLabelProvider columnLabelProvider = (ColumnLabelProvider) treeViewerLoc.getLabelProvider(i);
					if (columnLabelProvider != null) {
						String elementLabel = columnLabelProvider.getText(element);
						if (wordMatches(elementLabel)) {
							return true;
						}
					}
				}
				return false;
			}
		};
		filter.setIncludeLeadingWildcard(true);
		@SuppressWarnings("deprecation") // Required by RAP
		FilteredTree filteredTree = new FilteredTree(parentComposite, SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER, filter, true) {
					/* (non-Javadoc)
					 * @see org.eclipse.ui.dialogs.FilteredTree#doCreateFilterText(org.eclipse.swt.widgets.Composite)
					 */
					@Override
					protected Text doCreateFilterText(Composite panel) {
						Text text = super.doCreateFilterText(panel);
						if (text != null) {
							SWTFactoryProxy.INSTANCE.setHtmlTooltip(text, Messages.get().ComponentsView_AddComponent_FilterTooltip);
						}
						return text;
					}
				};
		final TreeViewer viewer = filteredTree.getViewer();
		viewer.setUseHashlookup(true);
		treeViewer = viewer;
		Tree tree = viewer.getTree();
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(viewer);
		tree.setLinesVisible(true);
		tree.setHeaderVisible(true);
		SWTFactoryProxy.INSTANCE.setTestId(tree, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_LIST);
		viewer.setContentProvider(new AddComponentTreeProvider());
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent e) {
				updateAddButtonState();
			}
		});
		filteredTree.setFocus();
		return viewer;
	}

	/**
	 * Selects first item in the viewer
	 * @param viewer
	 */
	private static void selectFirstItem(final @NonNull TreeViewer viewer) {
		Tree tree = viewer.getTree();
		if (tree.getItemCount() > 0) {
			TreeItem firstItem = tree.getItem(0);
			if (firstItem != null) {
				tree.select(firstItem);
			}
		}
	}

	/**
	 * Adds listeners to filter combo
	 * @param viewer
	 * @param allComponents all components
	 * @param latestComponents only latest components
	 * @param filteredComponents only components of project
	 * @param sdkComponents only component of SDK
	 */
	private void addListenersToFilterCombo(final @NonNull TreeViewer viewer) {
		InstantSearchList componentFilterComboLoc = componentFilterCombo;
		if (componentFilterComboLoc == null) {
			LOGGER.log(Level.SEVERE, "[TOOL] Filter combobox does not exist"); //$NON-NLS-1$
			return;
		}
		componentFilterComboLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				String selectedItem = e.text;
				if (Messages.get().AddComponentDialog_FilterCombo_AllComponents.equals(selectedItem)) {
					currentFilter = FilteringType.ALL;
				} else if (Messages.get().AddComponentDialog_FilterCombo_LatestComponents.equals(selectedItem)) {
					currentFilter = FilteringType.LATEST;
				} else if (Messages.get().AddComponentDialog_FilterCombo_ProjectComponents.equals(selectedItem)) {
					currentFilter = FilteringType.PROJECT_ONLY;
				} else if (Messages.get().AddComponentDialog_FilterCombo_SdkComponents.equals(selectedItem)) {
					currentFilter = FilteringType.SDK_ONLY;
				} else {
					LOGGER.log(Level.SEVERE, "[TOOL] Unknown filter type was selected \"{0}\"", selectedItem); //$NON-NLS-1$
					currentFilter = FilteringType.ALL;
				}
				setFilteringPreferenceStorage(currentFilter);
				setTreeViewerInputComps(viewer);
			}

		});
	}

	/**
	 * Sets filter preference to storage
	 * @param type of filtering
	 */
	void setFilteringPreferenceStorage(FilteringType type) {
		String filterStorage = FILTER_VALUE_ALL;
		switch (type) {
			case ALL: {
				filterStorage = FILTER_VALUE_ALL;
				break;
			}
			case LATEST: {
				filterStorage = FILTER_VALUE_LATEST;
				break;
			}
			case PROJECT_ONLY: {
				filterStorage = FILTER_VALUE_PROJECT;
				break;
			}
			case SDK_ONLY: {
				filterStorage = FILTER_VALUE_SDK;
				break;
			}
		}
		storageHelper.saveString(FILTER_KEY, filterStorage);
	}

	/**
	 * Prepares tree viewer.
	 * Sets filters, comparators etc.
	 * @param viewer to be prepared
	 */
	private void prepareTreeViewer(final @NonNull TreeViewer viewer) {
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		viewer.addFilter(new ViewerFilter() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public boolean select(Viewer viewerForSelect, Object parentElement, Object element) {
				AddComponentDialogEntry entry = (AddComponentDialogEntry) element;
				// filter components which are not in the current tool-chain project
				if (currentFilter == FilteringType.PROJECT_ONLY && (toolchainProject != null)) {
					for (SWComponent swComponent : entry.getComponent().getConfigurationComponent().getComponents()) {
						ISdkComponentInProject sdkComponent = toolchainProject.getSdkComponent(swComponent.getName());
						if (sdkComponent == null) {
							return false;
						}
					}
				}
				// filter out global-only components which are configured already
				IComponentConfig configuredComponent = controllerWrapper.getController().getConfiguredComponent(entry.getComponent().getConfigurationComponent().getId());
				if (configuredComponent != null) {
					return !configuredComponent.getComponent().isGlobalOnly();
				}
				return true;
			}
		});
		viewer.setComparator(new ViewerComparator() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerComparator#compare(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public int compare(Viewer viewerLoc, Object e1, Object e2) {
				if ((e1 instanceof AddComponentDialogEntry) && (e2 instanceof AddComponentDialogEntry)) {
					AddComponentDialogEntry entry1 = (AddComponentDialogEntry) e1;
					AddComponentDialogEntry entry2 = (AddComponentDialogEntry) e2;
					String name1Lower = UtilsText.safeString(entry1.getComponent().getConfigurationComponent().getUIName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext()).toLowerCase());
					String name2Lower = UtilsText.safeString(entry2.getComponent().getConfigurationComponent().getUIName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext()).toLowerCase());
					int comparisonResult = ComparatorHelpers.compareSignalNames(name1Lower, name2Lower);
					if ((comparisonResult == 0)
							&& (entry1.getComponent().getConfigurationComponent().getComponents().size() == 1)
							&& (entry2.getComponent().getConfigurationComponent().getComponents().size() == 1)) {
						Version ver1 = entry1.getComponent().getConfigurationComponent().getComponents().get(0).getVersion();
						Version ver2 = entry2.getComponent().getConfigurationComponent().getComponents().get(0).getVersion();
						if ((ver1 != null) && (ver2 != null)) {
							return ver1.compareTo(ver2);
						}
					}
					return comparisonResult;
				}
				return 0;
			}
		});
		Tree tree = viewer.getTree();
		GridData viewerLayoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewerLayoutData.heightHint = tree.getItemHeight() * DEFAULT_ROWS_NUM;
		viewerLayoutData.widthHint = (int) (viewerLayoutData.heightHint * DEFAULT_VIEWER_WIDTH_HEIGHT_RATIO);
		viewer.getControl().setLayoutData(viewerLayoutData);
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
			 */
			@Override
			public void doubleClick(DoubleClickEvent event) {
				StructuredSelection selection = (StructuredSelection) event.getSelection();
				Object data = selection.getFirstElement();
				assert (data instanceof AddComponentDialogEntry);
				AddComponentDialogEntry entry = (AddComponentDialogEntry) data;
				if (canAddComponent(entry.getComponent())) {
					createComponentFromSelection(selection, viewer, controllerWrapper);
					close();
				}
			}
		});
		tree.addListener(SWT.Expand, new Listener() {
			@Override
			public void handleEvent(@NonNull Event event) {
				tree.getDisplay().asyncExec(() -> {
					TreeItem treeItem = (TreeItem) event.item;
				    final TreeColumn[] treeColumns = treeItem.getParent().getColumns();
				    for (TreeColumn column : treeColumns) {
				    	column.pack();
				    }
				});
			}
		});
		viewer.setAutoExpandLevel(1);
	}

	/**
	 * Removes ConfigurationComponentTypeIds from list that should not be visible in UI.
	 * @param components list of ConfigurationComponentTypeId to filter out
	 * @param profile current profile
	 */
	public static void filterNonDisplayableComponents(@NonNull Collection<@NonNull ConfigurationComponentTypeId> components, @NonNull PeriphsProfile profile) {
		components.removeIf(c -> (c.getConfigurationComponent().isHidden()) && (!profile.getActiveComponents().getConfigCompTypeIds().contains(c)));
		components.removeIf(c -> (c.getConfigurationComponent().getId().equals(ConfigurationComponent.SYSTEM_COMPONENT_ID)));
	}

	/**
	 * Changes input for the specified viewer to the given components depending on the internal filter state.
	 * @param viewer component list viewer
	 * @param unfilteredComps an original collection of components
	 * @param latestComponents a collection of only the components that have highest supported version among the same type
	 * @param filteredComponents a collection of filtered components that are available in toolchain project
	 * @param sdkComponents a collection of filtered components that are available in SDK package used in toolchain project
	 */
	protected void setTreeViewerInputComps(@NonNull TreeViewer viewer) {
		switch (currentFilter) {
			default:
			case ALL: {
				viewer.setInput(prepareEntries(allComponents));
				break;
			}
			case LATEST: {
				viewer.setInput(prepareEntries(latestComponents));
				break;
			}
			case PROJECT_ONLY: {
				viewer.setInput(prepareEntries(filteredComponents));
				break;
			}
			case SDK_ONLY: {
				viewer.setInput(prepareEntries(sdkComponents));
				break;
			}
		}
	}

	/**
	 * Prepares entries from given components.
	 * Creates entry for the component and entries for all templates of that component.
	 * @param components collection of components
	 * @return List of all entries to be displayed
	 */
	private List<@NonNull AddComponentDialogEntry> prepareEntries(Collection<@NonNull ConfigurationComponentTypeId> components) {
		ArrayList<@NonNull AddComponentDialogEntry> allEntries = new ArrayList<>();
		for (ConfigurationComponentTypeId component : components) {
			AddComponentDialogEntry componentEntry = new AddComponentDialogEntry(component);
			allEntries.add(componentEntry);
			Map<@NonNull String, @NonNull ComponentTemplateProperties> componentTemplates = controllerWrapper.getController().getMcu().getAvailableComponents().getComponentTemplatesPropertiesByTypeId(component.getTypeId());
			for (ComponentTemplateProperties template : componentTemplates.values()) {
				componentEntry.addTemplateEntry(new AddComponentDialogEntry(componentEntry, template));
			}
		}
		return allEntries;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		if (toolchainProject != null && toolchainProject.canOpenSdkComponentManager()) {
			Button manageComponents = createButton(parent, OPEN_SDK_MANAGER_BUTTON_ID, Messages.get().AddComponentDialog_SdkManagerOpenButton_Label, true);
			manageComponents.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, false, false));
			manageComponents.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(@NonNull SelectionEvent e) {
					boolean closedProperly = toolchainProject.openSdkComponentsManager();
					TreeViewer treeViewerLoc = treeViewer;
					if (closedProperly && treeViewerLoc != null) {
						updateContent(treeViewerLoc, peripheral);
					}
				}
			});
		}
		addButton = createButton(parent, IDialogConstants.OK_ID, Messages.get().ComponentsView_SelectComponentDialog_OK, true);
		SWTFactoryProxy.INSTANCE.setTestId(addButton, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_OK_BUTTON);
		createButton(parent, IDialogConstants.CANCEL_ID, Messages.get().ComponentsView_RemoveComponentDialog_Cancel, false);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		if (treeViewer != null) {
			ISelection selection = treeViewer.getSelection();
			createComponentFromSelection(selection, AddComponentDialog.this, controllerWrapper);
		}
		super.okPressed();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#cancelPressed()
	 */
	@Override
	protected void cancelPressed() {
		close();
	}

	/**
	 * Get error text for a configuration component.
	 * @param configComp to get error of
	 * @param controllerWrapper containing the generic controller
	 * @param peripheral name
	 * @return error or {@code null} in case of no error
	 */
	public static @Nullable String getError(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper, @Nullable String peripheral) {
		APeriphController controller = controllerWrapper.getController();
		String peripheralLoc = peripheral;
		if ((peripheralLoc != null) && !controller.getMcu().isPeripheralAvailableForCore(controller.getFunctionalGroup().getCore(), peripheralLoc)) {
			return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_ComponentPeripheralUnavailableForCore_Tooltip), peripheralLoc, controller.getFunctionalGroup().getCore());
		}
		Collection<@NonNull String> periphs = controller.getAvailablePeripherals(controller.getFunctionalGroup(), configComp.getConfigurationComponent());
		if (!periphs.isEmpty()) {
			Collection<@NonNull String> corePeriphs = controller.filterPeripheralsForCore(controller.getFunctionalGroup().getCore(), periphs);
			if (corePeriphs.isEmpty()) {
				return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_ComponentPeripheralUnavailableForCore_Tooltip), String.join(UtilsText.COMMA_SPACE, periphs), controller.getFunctionalGroup().getCore());
			}
		}
		return null;
	}

	/**
	 * Get warning text for a configuration component.
	 * @param configComp to get warning of
	 * @param controllerWrapper containing the generic controller
	 * @return warning or {@code null} in case of no warning
	 */
	static @Nullable String getWarning(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper) {
		APeriphController controller = controllerWrapper.getController();
		StringJoiner resultJoiner = new StringJoiner(UtilsText.LF);
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		if ((toolchainProject != null) && toolchainProject.wasProjectDetected()) {
			for (SWComponent swComponent : configComp.getConfigurationComponent().getComponents()) {
				ISdkComponentInProject sdkComponent = toolchainProject.getSdkComponent(swComponent.getName());
				if (sdkComponent != null) {
					Version componentVersion = new Version(swComponent.getVersionStr());
					Version sdkComponentVersion = sdkComponent.getVersion();
					if (sdkComponentVersion == null) {
						resultJoiner.add(MessageFormat.format(com.nxp.swtools.periphs.controller.Messages.get().Controller_ProblemComponentWithUnknownVersions,
								controller.getMcu().getAvailableComponents().getSdkComponentName(swComponent.getName())));
					} else if (Components.computeMatchScore(componentVersion, sdkComponentVersion, swComponent.getMatchString(), swComponent.getUntilVersion()) == null) {
						resultJoiner.add(MessageFormat.format(
								UtilsText.safeString(com.nxp.swtools.periphs.controller.Messages.get().Controller_ProblemComponentVersionMismatchWithVersions),
								controller.getMcu().getAvailableComponents().getSdkComponentName(swComponent.getName()),
								componentVersion,
								sdkComponentVersion));
					}
				} else {
					Optional<@NonNull ISdkComponentInProject> sdkComp = toolchainProject.getAllSdkComponents().stream().filter(x -> {
							Version sdkVersion = x.getVersion();
							return x.getID().equals(swComponent.getName()) && (sdkVersion != null) &&
							(Components.computeMatchScore(swComponent.getVersion(), sdkVersion, swComponent.getMatchString(), swComponent.getUntilVersion()) != null);
					}).findAny();
					if (sdkComp.isPresent()) {
						resultJoiner.add(com.nxp.swtools.periphs.controller.Messages.get().Controller_ProblemComponentInSdkNotInProject);
					} else {
						// Component is not even in SDK with which the project was created
						resultJoiner.add(MessageFormat.format(
								UtilsText.safeString(com.nxp.swtools.periphs.controller.Messages.get().Controller_ProblemComponentMissing),
								controller.getMcu().getAvailableComponents().getSdkComponentName(swComponent.getName())));
					}
				}
			}
		}
		return resultJoiner.length() == 0 ? null : resultJoiner.toString();
	}

	/**
	 * Get info text for a configuration component.
	 * @param configComp to get info of
	 * @param controllerWrapper containing the generic controller
	 * @param functionalGroup name
	 * @return info or {@code null} in case of no info
	 */
	static @Nullable String getInfo(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper, @NonNull String functionalGroup) {
		APeriphController controller = controllerWrapper.getController();
		IConfigurationComponent configurationComponent = configComp.getConfigurationComponent();
		boolean canCreateInstances = controller.canCreateInstances(configurationComponent, functionalGroup);
		if (!canCreateInstances) {
			return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_InstancesLimitReached_Tooltip), configurationComponent.getId());
		}
		if (controller.hasModeWithMasterPeripheralNotDefined(configurationComponent, functionalGroup)) {
			return null;
		}
		boolean isPeripheralAvailable = controller.isAnyPeripheralAvailable(functionalGroup, configurationComponent);
		if (!isPeripheralAvailable) {
			Collection<@NonNull String> peripherals = controller.getUsablePeripherals(controller.getFunctionalGroup(), configurationComponent);
			return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_NoAvailablePeripherals_Tooltip), peripherals);
		}
		return null;
	}

	/**
	 * Get problem message for the configuration component.
	 * @param configComp to get message for
	 * @param controllerWrapper containing the generic controller
	 * @param functionalGroup name
	 * @param peripheral name
	 * @return message or {@code null} in case of no problem(s)
	 */
	static @Nullable String getProblemMessage(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper, @NonNull String functionalGroup, @Nullable String peripheral) {
		String tooltipText = getError(configComp, controllerWrapper, peripheral);
		if (tooltipText == null) {
			tooltipText = getWarning(configComp, controllerWrapper);
		}
		if (tooltipText == null) {
			tooltipText = getInfo(configComp, controllerWrapper, functionalGroup);
		}
		return tooltipText;
	}

	/**
	 * Create columns of the tree viewer.
	 * @param viewer to create columns in
	 */
	private void createTreeColumns(@NonNull TreeViewer viewer) {
		TreeViewerColumn configCompcolumn = new TreeViewerColumn(viewer, SWT.NONE);
		String localFunctionalGroup = functionalGroup;
		String localPeripheral = peripheral;
		configCompcolumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				AddComponentDialogEntry entry = (AddComponentDialogEntry) element;
				String label = entry.getName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext());
				if (entry.isComponentHidden()) {
					return label + Messages.get().AddComponentDialog_ComponentDeprecatedSuffix;
				}
				return label;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}

			/**
			 * Get error text for a configuration component.
			 * @param configComp to get error of
			 * @return error or {@code null} in case of no error
			 */
			private @Nullable String getError(@NonNull ConfigurationComponentTypeId configComp) {
				return AddComponentDialog.getError(configComp, controllerWrapper, localPeripheral);
			}

			/**
			 * Get warning text for a configuration component.
			 * @param configComp to get warning of
			 * @return warning or {@code null} in case of no warning
			 */
			private @Nullable String getWarning(@NonNull ConfigurationComponentTypeId configComp) {
				return AddComponentDialog.getWarning(configComp, controllerWrapper);
			}

			/**
			 * Get info text for a configuration component.
			 * @param configComp to get info of
			 * @return info or {@code null} in case of no info
			 */
			private @Nullable String getInfo(@NonNull ConfigurationComponentTypeId configComp) {
				return AddComponentDialog.getInfo(configComp, controllerWrapper, localFunctionalGroup);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				if (getError(configComp) != null) {
					return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
				} else if (getWarning(configComp) != null) {
					return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_WARNING);
				} else if (getInfo(configComp) != null) {
					return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_INFORMATION);
				}
				return ToolsImages.getImage(IToolsImages.ICON_EMPTY_16P);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				return AddComponentDialog.getProblemMessage(configComp, controllerWrapper, localFunctionalGroup, peripheral);
			}
		});
		TreeViewerColumn descriptionColumn = new TreeViewerColumn(viewer, SWT.NONE);
		descriptionColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				AddComponentDialogEntry entry = (AddComponentDialogEntry) element;
				String description = getDescription(entry);
				if (entry.isTemplateEntry() && description != null && description.length() > USE_CASE_DESCRIPTION_LENGTH_MAX) {
					return description.substring(0, USE_CASE_DESCRIPTION_LENGTH_MAX);
				}
				return description;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public @Nullable String getToolTipText(@NonNull Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				return getDescription((AddComponentDialogEntry) element);
			}

			private @Nullable String getDescription(AddComponentDialogEntry entry) {
				return entry.getDescription(controllerWrapper.getController().getFunctionalGroup().getExpressionContext());
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}
		});
		TreeViewerColumn categoryColumn = new TreeViewerColumn(viewer, SWT.NONE);
		categoryColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				String categoryId = UtilsText.safeString(configComp.getConfigurationComponent().getCategory());
				Categories categories = controllerWrapper.getController().getCategories();
				if (categories != null) {
					return categories.getCategoryLabel(categoryId, controllerWrapper.getController().getFunctionalGroup().getExpressionContext());
				}
				return categoryId;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}
		});
		TreeViewerColumn requiredSdkCompColumn = new TreeViewerColumn(viewer, SWT.NONE);
		requiredSdkCompColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				return buildDriverDependencyInfo(configComp, true);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}
		});
		TreeViewerColumn additionalSdkCompColumn = new TreeViewerColumn(viewer, SWT.NONE);
		additionalSdkCompColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				return buildDriverDependencyInfo(configComp, false);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (!(element instanceof AddComponentDialogEntry)) {
					LOGGER.log(Level.SEVERE, INPUT_IS_NOT_ENTRY_MESSAGE, element);
					return null;
				}
				ConfigurationComponentTypeId configComp = ((AddComponentDialogEntry) element).getComponent();
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
			}
		});
		configCompcolumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Name);
		descriptionColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_ComponentDescription);
		categoryColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Category);
		requiredSdkCompColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Version);
		additionalSdkCompColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_AdditionalVersion);
	}

	/**
	 * Builds string for version column e.g. platform.drivers.adc16[2.0.0-2.0.7]
	 * @param configComp component for which the string will be generated
	 * @param mustBeRequired {@code true} when only required drivers are included, {@code false} when only not required drivers are included
	 * @return version dependency string
	 */
	static String buildDriverDependencyInfo(ConfigurationComponentTypeId configComp, boolean mustBeRequired) {
		List<@NonNull SWComponent> sdkComponents = configComp.getConfigurationComponent().getComponents();
		return buildDriverDependencyInfo(mustBeRequired, sdkComponents);
	}

	/**
	 * Builds string for version column e.g. platform.drivers.adc16[2.0.0-2.0.7]
	 * @param mustBeRequired {@code true} when only required drivers are included, {@code false} when only not required drivers are included
	 * @param sdkComponents list of SDK components on which there is a dependency
	 * @return version dependency string
	 */
	public static String buildDriverDependencyInfo(boolean mustBeRequired, @NonNull List<@NonNull SWComponent> sdkComponents) {
		StringJoiner joiner = new StringJoiner(UtilsText.SEMICOLON_SPACE);
		for (SWComponent component : sdkComponents) {
			if (mustBeRequired != component.getRequired()) {
				// Current driver reference is not included in this column
				continue;
			}
			StringBuilder builder = new StringBuilder();
			Version version = component.getVersion();
			if (version == null) {
				continue;
			}
			Version untilVersion = component.getUntilVersion();
			Match match = component.getMatch();
			if (match == null) {
				match = Match.EQUIVALENT;
			}
			if (match == Match.PERFECT) {
				untilVersion = version;
			}
			if (untilVersion == null) {
				if (match == Match.COMPATIBLE) {
					untilVersion = new Version(version.getMajor(), HIGHEST_PERMITED_VERSION_PART, HIGHEST_PERMITED_VERSION_PART);
				}
				if (match == Match.GREATER_OR_EQUAL) {
					untilVersion = new Version(HIGHEST_PERMITED_VERSION_PART, HIGHEST_PERMITED_VERSION_PART, HIGHEST_PERMITED_VERSION_PART);
				}
			}
			builder.append(component.getName()).append(UtilsText.LEFT_BRACKET).append(version.toString());
			if ((!Objects.equals(version, untilVersion)) && (untilVersion != null)) {
				builder.append(UtilsText.MINUS_STRING).append(untilVersion.toString());
			}
			List<@NonNull String> revisionsList = component.getRevisionsList();
			if (!revisionsList.isEmpty()) {
				builder.append(UtilsText.COMMA_SPACE);
				builder.append(MessageFormat.format(Messages.get().AddComponentDialog_RevisionsList_Prefix, UtilsText.join(revisionsList, UtilsText.COMMA)));
			}
			builder.append(UtilsText.RIGHT_BRACKET);
			if (match == Match.EQUIVALENT) {
				String effectiveVersion = version.getMajor() + UtilsText.DOT + version.getMinor() + UtilsText.DOT + "x"; //$NON-NLS-1$
				builder.append(UtilsText.LEFT_PARENTHESIS).append(effectiveVersion).append(UtilsText.RIGHT_PARENTHESIS);
			}
			if (builder.length() != 0) {
				joiner.add(builder.toString());
			}
		}
		return joiner.toString();
	}

	/**
	 * Pack the columns. This method packs the columns' widths based on their content (omitting their titles)
	 */
	private void packColumns() {
		TreeViewer tableViewerLoc = treeViewer;
		if (tableViewerLoc != null) {
			for (TreeColumn col : tableViewerLoc.getTree().getColumns()) {
				col.pack();
			}
		}
	}

	/**
	 * Check whether the component can be added into the profile.
	 * @param configComp to check for
	 * @return {@code true} if the component can be added into profile, {@code false} otherwise
	 */
	static boolean canAddComponent(@NonNull ConfigurationComponentTypeId configComp) {
		String compType = configComp.getConfigurationComponent().getId();
		String configuredComponentTypeId = Controller.getInstance().getProfile().getConfiguredComponentTypeId(compType);
		return (configuredComponentTypeId == null) || configuredComponentTypeId.equals(configComp.getTypeId());
	}

	/**
	 * Creates component from the {@link ISelection}.
	 * @param selection the selected component types to create
	 * @param caller the method caller which will be used as an originator of a subsequent UI event.
	 * @param controllerWrapperLoc containing the generic controller
	 */
	protected void createComponentFromSelection(@Nullable ISelection selection, @NonNull Object caller, @NonNull IControllerWrapper controllerWrapperLoc) {
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			List<@NonNull AddComponentDialogEntry> comps = CollectionsUtils.getInstancesOf(
					(List<?>) structuredSelection.toList(),
					AddComponentDialogEntry.class
					).collect(Collectors.toList());
			assert (comps != null);
			assert (!comps.isEmpty());
			IChildProvidable createdComponent = createComponentsFromEntry(comps.get(0), peripheral, viewSite, functionalGroup, openView, caller, controllerWrapperLoc);
			setResult(createdComponent);
		}
	}

	/**
	 * Create given component and optionally open their views.
	 * @param entry from which to create component
	 * @param peripheral for which to create the components or {@code null} for the first available
	 * @param viewSite associated view site for which to create component setting view(s)
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param caller originator of the UI event(s) invoked by this method
	 * @param controllerWrapper containing the generic controller
	 * @return the last component or component instance that was created or {@code null} if there wasn't anything new created.
	 */
	public static @Nullable IChildProvidable createComponentsFromEntry(@NonNull AddComponentDialogEntry entry,
			@Nullable String peripheral, @NonNull IViewSite viewSite, @NonNull String functionalGroup, boolean openView,
			@NonNull Object caller, @NonNull IControllerWrapper controllerWrapper) {
		Controller controller = Controller.getInstance();
		IChildProvidable result = null;
		if (entry.isTemplateEntry()) {
			ComponentTemplateProperties templateProperties = entry.getTemplateProperties();
			if (templateProperties != null) {
				ComponentTemplate template = templateProperties.getTemplate();
				if (template != null) {
					result = createComponentFromTemplate(peripheral, viewSite, functionalGroup, openView, caller, controllerWrapper, controller, template);
				}
			}
		} else {
			ConfigurationComponentTypeId selectedConfigComp = entry.getComponent();
			result = createComponentFromTypeId(peripheral, viewSite, functionalGroup, openView, caller, controllerWrapper, controller, selectedConfigComp);
		}
		return result;
	}

	/**
	 * Create given component and optionally open their views.
	 * @param component to create instance from
	 * @param peripheral for which to create the components or {@code null} for the first available
	 * @param viewSite associated view site for which to create component setting view(s)
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param caller originator of the UI event(s) invoked by this method
	 * @param controllerWrapper containing the generic controller
	 * @return the last component or component instance that was created or {@code null} if there wasn't anything new created.
	 */
	public static @Nullable IChildProvidable createComponent(@NonNull ConfigurationComponentTypeId component,
			@Nullable String peripheral, @NonNull IViewSite viewSite, @NonNull String functionalGroup, boolean openView,
			@NonNull Object caller, @NonNull IControllerWrapper controllerWrapper) {
		Controller controller = Controller.getInstance();
		IChildProvidable result = null;
		result = createComponentFromTypeId(peripheral, viewSite, functionalGroup, openView, caller, controllerWrapper, controller, component);
		return result;
	}

	/**
	 * Creates component or component instance from given component typeid
	 * @param peripheral
	 * @param viewSite
	 * @param functionalGroup
	 * @param openView {@code true} if view should be opened, {@code false} otherwise
	 * @param caller
	 * @param controllerWrapper
	 * @param controller
	 * @param componentTypeId
	 * @return component config when component is global only, component instance when limit is not already reached or {@code null} otherwise
	 */
	private static IChildProvidable createComponentFromTypeId(@Nullable String peripheral, @NonNull IViewSite viewSite, @NonNull String functionalGroup, boolean openView, @NonNull Object caller,
			@NonNull IControllerWrapper controllerWrapper, @NonNull Controller controller, @NonNull ConfigurationComponentTypeId componentTypeId) {
		String componentType = componentTypeId.getConfigurationComponent().getId();
		IChildProvidable result = null;
		if (componentTypeId.getConfigurationComponent().isGlobalOnly()) {
			result = ProgressUtils.run(() -> controller.createOrGetComponent(componentTypeId.getTypeId(), caller));
			if (result != null) {
				if (openView) {
					controllerWrapper.getGUIController().openConfigurationSettingsView(viewSite, componentType, componentType, true, true);
				}
			} else {
				LOGGER.severe("[TOOL] Component configuration of type: " + componentType + " could not be created"); //$NON-NLS-1$ //$NON-NLS-2$
			}
		} else {
			if (controller.isInstancesLimitReached(functionalGroup)) {
				MessageDialog.openError(viewSite.getShell(), Messages.get().AddComponentDialog_LimitReached_Title,
						Messages.get().AddComponentDialog_LimitReached_Message);
			} else
			// if the component has the max_instances attribute defined, check if a new component instance can be added, if not, the user will be
			// informed about it
			if (!controller.canCreateInstances(componentTypeId.getConfigurationComponent(), functionalGroup)) {
				MessageDialog.openError(viewSite.getShell(), Messages.get().AddComponentDialog_LimitReached_Title, MessageFormat
						.format(Messages.get().AddComponentDialog_InstancesLimitReached_Message, componentTypeId.getConfigurationComponent().getId()));
			} else {
				IComponentInstanceConfig componentInstanceConfig = ProgressUtils.run(() -> controller.createComponentInstance(componentTypeId.getTypeId(),
						peripheral, functionalGroup, caller));
				result = componentInstanceConfig;
				if (componentInstanceConfig != null) {
					if (openView) {
						controllerWrapper.getGUIController().openViewOfInstance(viewSite, componentInstanceConfig, true);
					}
				} else {
					LOGGER.severe("[TOOL] Component instance of type: " + componentType + " could not be created"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
		return result;
	}

	/**
	 * Creates component instance from given component template
	 * @param peripheral
	 * @param viewSite
	 * @param functionalGroup
	 * @param openView
	 * @param caller
	 * @param controllerWrapper
	 * @param controller
	 * @param componentTemplate
	 * @return component instance when limit is not already reached or {@code null} otherwise
	 */
	private static @Nullable IChildProvidable createComponentFromTemplate(@Nullable String peripheral, @NonNull IViewSite viewSite, @NonNull String functionalGroup, boolean openView, @NonNull Object caller,
			@NonNull IControllerWrapper controllerWrapper, @NonNull Controller controller, @NonNull ComponentTemplate componentTemplate) {
		IChildProvidable result = null;
		String compTypeId = componentTemplate.getTypeId();
		ConfigurationComponentTypeId selectedConfigComp = controller.getMcu().getAvailableComponents().getConfigCompTypeId(compTypeId);
		if (selectedConfigComp == null) { // No component with given typeId is available
			return null;
		}
		String componentType = selectedConfigComp.getConfigurationComponent().getId();
		if (controller.isInstancesLimitReached(functionalGroup)) {
			MessageDialog.openError(viewSite.getShell(), Messages.get().AddComponentDialog_LimitReached_Title, Messages.get().AddComponentDialog_LimitReached_Message);
			return null;
		} else {
			// if the component has the max_instances attribute defined, check if a new component instance can be added, if not, the user will be
			// informed about it
			if (!controller.canCreateInstances(selectedConfigComp.getConfigurationComponent(), functionalGroup)) {
				MessageDialog.openError(viewSite.getShell(), Messages.get().AddComponentDialog_LimitReached_Title, MessageFormat
						.format(Messages.get().AddComponentDialog_InstancesLimitReached_Message, selectedConfigComp.getConfigurationComponent().getId()));
			} else {
				IComponentInstanceConfig componentInstanceConfig = ProgressUtils.run(() -> controller.createComponentInstanceFromTemplate(componentTemplate,
						peripheral, functionalGroup, caller));
				result = componentInstanceConfig;
				if (componentInstanceConfig != null) {
					if (openView) {
						controllerWrapper.getGUIController().openViewOfInstance(viewSite, componentInstanceConfig, true);
					}
				} else {
					LOGGER.severe("[TOOL] Component instance of type: " + componentType + " could not be created"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
		return result;
	}

	/**
	 * Set state of the button based on current selection
	 * @param button button to configure it's state
	 * @param iSelection selected item in a viewer
	 */
	static void setButtonState(@NonNull Button button, @Nullable ISelection iSelection) {
		if (iSelection == null) {
			return;
		}
		StructuredSelection selection = (StructuredSelection) iSelection;
		boolean setActive = !selection.isEmpty();
		for (Object data : selection.toList()) {
			assert (data instanceof AddComponentDialogEntry);
			AddComponentDialogEntry entry = (AddComponentDialogEntry) data;
			ConfigurationComponentTypeId configComp = entry.getComponent();
			String compType = configComp.getConfigurationComponent().getId();
			String configuredComponentTypeId = Controller.getInstance().getProfile().getConfiguredComponentTypeId(compType);
			if ((configuredComponentTypeId != null) && !configuredComponentTypeId.equals(configComp.getTypeId())) {
				setActive = false;
			}
		}
		button.setEnabled(setActive);
	}

	/**
	 * Get result of dialog.
	 * @return Component or ComponentInstance created by this dialog
	 */
	public @Nullable IChildProvidable getResult() {
		return result;
	}

	/**
	 * Set result of dialog.
	 * @param result Component or ComponentInstance created by this dialog
	 */
	private void setResult(@Nullable IChildProvidable result) {
		this.result = result;
	}

	/**
	 * Checks whether the given component has some templates
	 * @param component to be checked
	 * @param controllerWrapper controller wrapper
	 * @return @see {@link AvailableComponents#componentHasTemplates(ConfigurationComponentTypeId)}
	 */
	public static boolean componentHasTemplates(@NonNull ConfigurationComponentTypeId component, @NonNull IControllerWrapper controllerWrapper) {
		return controllerWrapper.getController().getMcu().getAvailableComponents().componentHasTemplates(component);
	}
}
