/**
 * Copyright 2017-2021 NXP
 */
package com.nxp.swtools.periphs.gui.perspective;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.ui.IFolderLayout;
import org.eclipse.ui.IPageLayout;

import com.nxp.swtools.common.ui.utils.perspectives.PerspectivesHelper;
import com.nxp.swtools.common.ui.utils.progress.UIJobHelper;
import com.nxp.swtools.common.ui.utils.services.Rap;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.view.ComponentsView;
import com.nxp.swtools.periphs.gui.view.DefaultComponentSettingsView;
import com.nxp.swtools.periphs.gui.view.DocumentationView;
import com.nxp.swtools.periphs.gui.view.PeripheralsView;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.gui.view.componentsettings.GlobalComponentSettingView;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.logging.UserLogView;
import com.nxp.swtools.utils.perspective.ToolsBasePerspective;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.registers.RegistersView;
import com.nxp.swtools.utils.sources.SourcesView;
import com.nxp.swtools.validation.engine.ValidationEngineFactory;

/**
 * Class representing perspective of a Peripherals tool
 * @author Juraj Ondruska
 * @author Tomas Rudolf - nxf31690
 */
public class PeripheralsPerspective extends ToolsBasePerspective {
	/** ID of the perspective */
	public static final String ID = "com.nxp.swtools.periphs.gui.perspective"; //$NON-NLS-1$
	/** ID of extension point providing views extending this perspective */
	public static final String EXTENSION_POINT_ID = "com.nxp.swtools.periphs.gui.perspective.additionalViews"; //$NON-NLS-1$
	/** ID of internal debug view. Used in placeholder */
	private static final String DEBUG_VIEW_ID = "com.nxp.swtools.periphs.internal_development.view.DebugView"; //$NON-NLS-1$
	/** Suffix for all folder IDs */
	private static final String FOLDER_ID_SUFFIX = "Folder"; //$NON-NLS-1$
	/** ID of category "editor" */
	private static final String EDITOR_FOLDER_CATEGORY = "editor"; //$NON-NLS-1$
	/** ID of folder for category "editor" */
	private static final String EDITOR_FOLDER_ID = EDITOR_FOLDER_CATEGORY + FOLDER_ID_SUFFIX;
	/** ID of category "main" */
	private static final String MAIN_FOLDER_CATEGORY = "main"; //$NON-NLS-1$
	/** ID of folder for category "main" */
	private static final String MAIN_FOLDER_ID = MAIN_FOLDER_CATEGORY + FOLDER_ID_SUFFIX;
	/** ID of category "bottom" */
	private static final String BOTTOM_FOLDER_CATEGORY = "bottom"; //$NON-NLS-1$
	/** ID of folder for category "bottom" */
	private static final String BOTTOM_FOLDER_ID = BOTTOM_FOLDER_CATEGORY + FOLDER_ID_SUFFIX;
	/** ID of category "shared" */
	private static final String SHARED_FOLDER_CATEGORY = "shared"; //$NON-NLS-1$
	/** ID of folder for category "shared" */
	private static final String SHARED_FOLDER_ID = SHARED_FOLDER_CATEGORY + FOLDER_ID_SUFFIX;

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.perspective.ToolsBasePerspective#createInitialLayout(org.eclipse.ui.IPageLayout)
	 */
	@Override
	public void createInitialLayout(IPageLayout layout) {
		super.createInitialLayout(layout);
		Map<String, Set<String>> externalViews = getRegisteredExtensionViews();
		layout.setEditorAreaVisible(false);
		String editorArea = layout.getEditorArea();
		// Create main folder as left column
		IFolderLayout mainFolder = layout.createFolder(MAIN_FOLDER_ID, IPageLayout.LEFT, 0.20f, editorArea);
		mainFolder.addView(ComponentsView.ID);
		mainFolder.addView(PeripheralsView.ID);
		addPlaceholdersToFolder(mainFolder, externalViews.get(MAIN_FOLDER_CATEGORY));
		// Create shared folder as right column
		IFolderLayout sharedFolder = layout.createFolder(SHARED_FOLDER_ID, IPageLayout.RIGHT, 0.68f, editorArea);
		sharedFolder.addView(TestIDs.DASHBOARD_VIEW_ID); //this should be visible by default, if new configuration is created so user can easily switch to different tool
		sharedFolder.addView(SourcesView.ID);
		sharedFolder.addPlaceholder(RegistersView.ID);
		sharedFolder.addPlaceholder(DocumentationView.ID + ":*"); //$NON-NLS-1$
		sharedFolder.addPlaceholder(DEBUG_VIEW_ID);
		addPlaceholdersToFolder(sharedFolder, externalViews.get(SHARED_FOLDER_CATEGORY));
		// Create bottom folder inside shared folder column
		IFolderLayout bottomFolder = layout.createFolder(BOTTOM_FOLDER_ID, IPageLayout.BOTTOM, 0.75f, SHARED_FOLDER_ID);
		bottomFolder.addView(ValidationEngineFactory.VIEW_ID);
		addPlaceholdersToFolder(bottomFolder, externalViews.get(BOTTOM_FOLDER_CATEGORY));
		addCheatSheetViewPlaceholder(layout);
		addConsoleViewPlaceholder(layout);
		bottomFolder.addPlaceholder(TestIDs.SNIPPET_VIEW_ID);
		// Create editor folder as center column
		IFolderLayout editorFolder = layout.createFolder(EDITOR_FOLDER_ID, IPageLayout.RIGHT, 0.5f, editorArea);
		editorFolder.addView(DefaultComponentSettingsView.ID);
		editorFolder.addPlaceholder(ComponentSettingView.ID + ":*"); //$NON-NLS-1$
		editorFolder.addPlaceholder(GlobalComponentSettingView.ID);
		addPlaceholdersToFolder(editorFolder, externalViews.get(EDITOR_FOLDER_CATEGORY));
		if (Rap.isActive()) {
			// one window can hold multiple view, each view being represented by a tab
			// it can happen that RAP will hide the tab of the first added view, if there is not enough space to show all tabs
			// but the content of the hidden tab is still visible in the window
			// workaround -> select some other (i.e. not the first one) view to make it active
			UIJobHelper.runNowOrInJob(() -> PerspectivesHelper.showView(ComponentsView.ID), "Selecting Components View", null); //$NON-NLS-1$
		}
	}

	/**
	 * @return map of extension view IDs grouped by target folder
	 */
	private static Map<String, Set<String>> getRegisteredExtensionViews() {
		IConfigurationElement[] registeredExtensions = Platform.getExtensionRegistry().getConfigurationElementsFor(EXTENSION_POINT_ID);
		Map<String, Set<String>> externalViews = new HashMap<>();
		for (IConfigurationElement extension : registeredExtensions) {
			String id = UtilsText.safeString(extension.getAttribute("id")); //$NON-NLS-1$
			String folder = UtilsText.safeString(extension.getAttribute("folder")); //$NON-NLS-1$
			Set<String> set = externalViews.computeIfAbsent(folder, x -> new HashSet<>());
			assert set != null; // Cannot be null
			set.add(id);
		}
		return externalViews;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.perspective.ToolsBasePerspective#getDefinedViewIDs()
	 */
	@Override
	public List<String> getDefinedViewIDs() {
		return CollectionsUtils.unmodifiableList(new ArrayList<String>() {
			/** serialVersionUID */
			private static final long serialVersionUID = 1L;
			{
				add(PeripheralsView.ID);
				add(ComponentsView.ID);
				add(SourcesView.ID);
				add(UserLogView.ID);
				add(ValidationEngineFactory.VIEW_ID);
				if (KEPreferences.showRegistersViewShortcut()) {
					add(RegistersView.ID);
				}
				if (!Rap.isActive()) {
					add(TestIDs.DASHBOARD_VIEW_ID);
					add(TestIDs.SNIPPET_VIEW_ID);
				}
			}
		});
	}

	/**
	 * Adds given placeholders to given folder
	 * @param folder to contain placeholders
	 * @param placeholders collection of view IDs
	 */
	protected void addPlaceholdersToFolder(IFolderLayout folder, @Nullable Collection<String> placeholders) {
		if (placeholders != null) {
			for (String placeholder : placeholders) {
				folder.addPlaceholder(placeholder);
			}
		}
	}
}
