/**
 * Copyright 2018-2021 NXP               
 * Created: 8. 2. 2018
 */
package com.nxp.swtools.periphs.gui.handler;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;

import com.nxp.swtools.common.ui.utils.dialogs.ShellProvider;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.expert.processordb.ISupportedDerivativesForSwTools;
import com.nxp.swtools.expert.processordb.ITemplateInfo;
import com.nxp.swtools.expert.processordb.SupportedDerivativesForSwTools;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.model.data.mcu.IMcu;
import com.nxp.swtools.provider.configuration.ISharedConfiguration;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsFuncGroup;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsTool;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.utils.ConfigurationUtils;
import com.nxp.swtools.utils.command.BoardResetDialog;

/**
 * Handler for reset board command
 * @author Marek Ciz
 *
 */
public class ResetToBoardDefaultHandler extends AbstractHandler {

	/** Logger of the class */
	private static final Logger LOGGER = LogManager.getLogger(ResetToBoardDefaultHandler.class);
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.commands.IHandler#execute(org.eclipse.core.commands.ExecutionEvent)
	 */
	@Override
	public @Nullable Object execute(ExecutionEvent event) throws ExecutionException {
		IFunctionalGroup changedFuncGroup = Controller.getInstance().getFunctionalGroup();
		Shell parentShell = ShellProvider.getAnyShellParent();
		IMcu mcu = Controller.getInstance().getMcu();
		String kitOrBoardName = UtilsText.nonEmpty(mcu.getKit()) ? mcu.getKit() : mcu.getBoard();
		ISharedConfiguration sharedConfigurationLoc = ConfigurationUtils.loadBoardConfiguration(kitOrBoardName, mcu.getSdkVersion(), mcu.getMexVariant());
		if ((sharedConfigurationLoc != null) && (kitOrBoardName != null)) {
			ISupportedDerivativesForSwTools derivatives = SupportedDerivativesForSwTools.getSupportedDerivativesForSwTools();
			ITemplateInfo template = UtilsText.isEmpty(mcu.getKit())
					? derivatives.getBoardById(kitOrBoardName, mcu.getSdkVersion())
					: derivatives.getKitById(kitOrBoardName, mcu.getSdkVersion());
			if (template == null) {
				LOGGER.log(Level.SEVERE, "[TOOL] Can't find board/kit {0} template to reset board defaults.", kitOrBoardName); //$NON-NLS-1$
				return null;
			}
			StoragePeriphsTool periphs = sharedConfigurationLoc.getTools().getPeripherals();
			if (periphs != null) {
				List<StoragePeriphsFuncGroup> functions = periphs.getFuncGroups();
				if (functions.isEmpty()) {
					informUserThatResetCannotBePerformed(parentShell);
					return null;
				}
				List<String> funcNames = functions.stream().map(StoragePeriphsFuncGroup::getConfigurationName).collect(CollectorsUtils.toList());
				String defaultMexVariant = BoardResetDialog.getDefaultMexVariant(sharedConfigurationLoc.getCommonConfig());
				BoardResetDialog<List<StoragePeriphsFuncGroup>> dialog = new BoardResetDialog<List<StoragePeriphsFuncGroup>>(parentShell, funcNames, changedFuncGroup.getName(),
						template.getTemplateVariants(), defaultMexVariant) {
					/* (non-Javadoc)
					 * @see com.nxp.swtools.utils.command.BoardResetDialog#getConfigs(java.lang.String)
					 */
					@Override
					public @Nullable Collection<@NonNull String> getConfigs(String mexVariant) {
						ISharedConfiguration tmpConfig = ConfigurationUtils.loadBoardConfiguration(kitOrBoardName, mcu.getSdkVersion(), mexVariant);
						if (tmpConfig != null) {
							StoragePeriphsTool tmpStorage = tmpConfig.getTools().getPeripherals();
							if (tmpStorage != null) {
								setCustomData(tmpStorage.getFuncGroups());
								return tmpStorage.getFuncGroups().stream().map(StoragePeriphsFuncGroup::getConfigurationName).collect(CollectorsUtils.toList());
							}
						}
						return Collections.emptyList();
					}
				};
				dialog.setCustomData(functions);
				if (dialog.open() == IDialogConstants.OK_ID) {
					Controller.getInstance().resetToStorageFuncGroup(Objects.requireNonNull(dialog.getCustomData()), UtilsText.safeString(dialog.getSelectedConfiguration()),
							changedFuncGroup.getStorageFuncGroup());
				}
			} else {
				informUserThatResetCannotBePerformed(parentShell);
			}
		} else {
			LOGGER.severe("[TOOL] Unable to load shared configuration to reset board defaults"); //$NON-NLS-1$
		}
		return null;
	}

	/**
	 * Opens information dialog to inform user that the rest to board defaults cannot be performed because there is no Peripherals tool configuration inside the board MEX
	 * @param parentShell shell in which the dialog will open
	 */
	private static void informUserThatResetCannotBePerformed(Shell parentShell) {
		MessageDialog.openInformation(parentShell, Messages.get().ResetToBoardDefaults_NoPeripherals_DialogTitle, Messages.get().ResetToBoardDefaults_NoPeripherals_DialogTitle);
	}
}
