/*
 * Copyright 2020-2021 NXP
 */
package com.nxp.swtools.periphs.gui.handler;

import java.text.MessageFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IViewSite;

import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.utils.httpserver.HttpRequest;
import com.nxp.swtools.utils.httpserver.Response;

/**
 * Handler that will open {@link ComponentSettingView} when there will be HTTP request
 * @author Tomas Rudolf - nxf31690
 */
public class HttpOpenComponentSettingViewHandler extends AHttpPeriphHandler {
	/** Wrapper that holds Peripherals controller and GUI controller */
	private static final PeriphControllerWrapper WRAPPER = PeriphControllerWrapper.getInstance();
	/** Id of regex group that matches the name of instance */
	private static final String REGEX_NAME_GROUP_ID = "name"; //$NON-NLS-1$
	/** This handler handles queries that starts with this prefix */
	private static final String PREFIX = "/periph/openComponentSettingView"; //$NON-NLS-1$
	/** Pattern to match specific peripheral request */
	private static final Pattern pattern = Pattern.compile(PREFIX + "/(?<" + REGEX_NAME_GROUP_ID + ">\\w+)"); //$NON-NLS-1$ //$NON-NLS-2$

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.httpserver.IHandler#canHandle(com.nxp.swtools.utils.httpserver.HttpRequest)
	 */
	@Override
	public boolean canHandle(HttpRequest query) {
		if (query.getLocation().startsWith(PREFIX)) {
			return true;
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.httpserver.IHandler#handle(com.nxp.swtools.utils.httpserver.HttpRequest)
	 */
	@Override
	public void handle(HttpRequest query) {
		Display display = WRAPPER.getController().getDisplay();
		if (display == null) {
			query.setResponse(new Response(Response.STATUS_CODE_INTERNAL_ERROR, "Could not retrieve display")); //$NON-NLS-1$
			return;
		}
		display.asyncExec(() -> {
			Matcher matcher = pattern.matcher(query.getLocation());
			String instanceName = UtilsText.EMPTY_STRING;
			if (matcher.matches()) {
				instanceName = UtilsText.safeString(matcher.group(REGEX_NAME_GROUP_ID));
			}
			IComponentInstanceConfig instance = WRAPPER.getController().getFunctionalGroup().getInstance(instanceName);
			if (instance == null) {
				query.setResponse(new Response(Response.STATUS_CODE_INTERNAL_ERROR, MessageFormat.format("Instance with name {0} was not found", instanceName))); //$NON-NLS-1$
				return;
			}
			putShellToTop();
			openPeripheralsPerspective();
			IViewSite currentViewSite = WRAPPER.getGUIController().getCurrentViewSite();
			if (currentViewSite == null) {
				query.setResponse(new Response(Response.STATUS_CODE_INTERNAL_ERROR, "Could not retrieve current view site")); //$NON-NLS-1$
				return;
			}
			ComponentSettingView.open(currentViewSite, instance.getComponent().getId(), instanceName, false, true);
			query.setResponse(new Response(Response.STATUS_CODE_OK, "OK")); //$NON-NLS-1$
		});
	}
}
