/**
 * Copyright 2020-2021 NXP
 * Created: 26 Aug 2020
 */
package com.nxp.swtools.periphs.gui.dialogs.initializationorder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.InitializationPrioritiesHelper;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.model.config.PeriphsProfile;
import com.nxp.swtools.periphs.model.data.InitializationPriorities;
import com.nxp.swtools.periphs.model.data.InitializationPrioritiesGroup;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.resourcetables.model.config.IRoot;
import com.nxp.swtools.resourcetables.model.data.Description;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Dialog for changing initialization order of component instances
 * @author Tomas Rudolf - nxf31690
 */
public class InitializationOrderDialog extends Dialog {
	/** Serialization ID */
	static final long serialVersionUID = 1L;
	/** Logger of this class */
	private static final Logger LOGGER = LogManager.getLogger(InitializationOrderDialog.class);
	/** The properties object */
	private final transient InitializationOrderDialogProperties properties;
	/** The tree viewer of this dialog */
	private @Nullable TreeViewer treeViewer;
	/** Functional group that is currently selected */
	private transient @Nullable IFunctionalGroup currentFunctionalGroup = null;
	/** Map of all instances in the result ordering in each functional group */
	private transient Map<IFunctionalGroup, LinkedList<IComponentInstanceConfig>> instancesOfGroups = new HashMap<>();
	/** Button up used in ordering */
	private @Nullable Button buttonUp;
	/** Button down used in ordering*/
	private @Nullable Button buttonDown;

	/**
	 * Constructor
	 * @param properties with which this dialog should open
	 */
	protected InitializationOrderDialog(InitializationOrderDialogProperties properties) {
		super(properties.getShell());
		this.properties = properties;
	}

	/**
	 * @return the properties
	 */
	private InitializationOrderDialogProperties getProperties() {
		return properties;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected @NonNull Control createDialogArea(@NonNull Composite parent) {
		Composite dialogComposite = (Composite) super.createDialogArea(parent);
		dialogComposite.setLayout(new GridLayout());
		dialogComposite.getShell().setText(Messages.get().InitializationOrderDialog_DialogHeaderText);
		Label groupLabel = new Label(dialogComposite, SWT.NONE);
		groupLabel.setText(Messages.get().InitializationOrderDialog_FunctionalGroupLabel);
		GridDataFactory.fillDefaults().align(SWT.LEFT, SWT.CENTER).grab(true, false).applyTo(groupLabel);
		InstantSearchList combo = new InstantSearchList(dialogComposite, SWT.BORDER);
		GridDataFactory.fillDefaults().grab(true, false).applyTo(combo);
		SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_REORDER_DIALOG_COMBO);
		
		List<IFunctionalGroup> groups = getProperties().getRoot().getFunctionalGroups();
		groups.forEach(g -> instancesOfGroups.put(g, new LinkedList<>(g.getInstances().values())));
		@NonNull String[] functionalGroupNames = groups.stream().map(IFunctionalGroup::getName).toArray(x -> new String[x]);
		combo.setItems(functionalGroupNames);
		
		Label viewerLabel = new Label(dialogComposite, SWT.NONE);
		viewerLabel.setText(Messages.get().InitializationOrderDialog_InitializationOrderLabel);
		GridDataFactory.fillDefaults().align(SWT.LEFT, SWT.CENTER).grab(true, false).applyTo(viewerLabel);
		Composite viewerComposite = new Composite(dialogComposite, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(viewerComposite);
		GridLayout layout = new GridLayout(2, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		viewerComposite.setLayout(layout);
		
		TreeViewer viewer = new TreeViewer(viewerComposite, SWT.SINGLE | SWT.BORDER);
		treeViewer = viewer;
		Tree tree = viewer.getTree();
		GridDataFactory.fillDefaults().grab(true, true).span(1, 2).applyTo(tree);
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(viewer);
		SWTFactoryProxy.INSTANCE.setTestId(tree, TestIDs.PERIPHS_REORDER_DIALOG_TREE);
		viewer.setContentProvider(new InitializationOrderProvider());
		viewer.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(Object element) {
				if (element instanceof InitializationOrderGroup) {
					return ((InitializationOrderGroup) element).getName();
				} else if (element instanceof InitializationOrderInstance) {
					InitializationOrderInstance instanceWrapper = (InitializationOrderInstance) element;
					IComponentInstanceConfig instance = instanceWrapper.getInstance();
					String componentId = instance.getComponent().getId().toUpperCase();
					return instance.getName() + UtilsText.LEFT_BRACKET + componentId + UtilsText.RIGHT_BRACKET;
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public @Nullable Font getFont(Object element) {
				return super.getFont(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public @Nullable Color getForeground(Object element) {
				return super.getForeground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public @Nullable Image getImage(Object element) {
				return super.getImage(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public @Nullable String getToolTipText(@NonNull Object element) {
				if (element instanceof InitializationOrderGroup) {
					return ((InitializationOrderGroup) element).getDescription();
				} else if (element instanceof InitializationOrderInstance) {
					InitializationOrderInstance instanceWrapper = (InitializationOrderInstance) element;
					IComponentInstanceConfig instance = instanceWrapper.getInstance();
					StringBuilder builder = new StringBuilder();
					builder.append(instance.getComment());
					Description descriptionObject = instance.getComponent().getDescription();
					if (descriptionObject != null) {
						if (builder.length() != 0) {
							builder.append(UtilsText.CRLF);
						}
						builder.append(UtilsText.LEFT_BRACKET).append(descriptionObject.getDescription(instance.getExpressionContext())).append(UtilsText.RIGHT_BRACKET);
					}
					return (builder.length() != 0) ? builder.toString() : null;
				}
				return null;
			}
		});
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			@Override
			public void selectionChanged(@Nullable SelectionChangedEvent event) {
				updateOrderingButtonsEnabledState();
			}
		});
		viewer.addFilter(new ViewerFilter() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				if (element instanceof InitializationOrderInstance) {
					return true;
				}
				if (element instanceof InitializationOrderGroup) {
					return !((InitializationOrderGroup)element).getInstances().isEmpty();
				}
				return false;
			}
		});

		createOrderingButtonUp(viewerComposite);
		createOrderingButtonDown(viewerComposite);
		addComboSelectionListener(combo);
		combo.setSelection(getProperties().getControllerWrapper().getController().getFunctionalGroup().getName());
		return dialogComposite;
	}

	/**
	 * Create ordering button up in given composite
	 * @param composite to create button in
	 */
	void createOrderingButtonUp(Composite composite) {
		Button buttonUpLoc = new Button(composite, SWT.PUSH);
		buttonUp = buttonUpLoc;
		buttonUpLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_UP));
		SWTFactoryProxy.INSTANCE.setTestId(buttonUpLoc, TestIDs.PERIPHS_REORDER_DIALOG_BUTTON_UP);
		GridDataFactory.fillDefaults().grab(false, true).span(1, 1).applyTo(buttonUpLoc);
		buttonUpLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TreeViewer treeViewerLoc = getTreeViewer();
				if (treeViewerLoc == null) {
					return;
				}
				InitializationOrderInstance instance = getCurrentlySelectedInstance();
				if (instance != null) {
					instance.getGroup().moveBefore(instance);
				}
				treeViewerLoc.refresh();
				updateOrderingButtonsEnabledState();
			}
		});
	}

	/**
	 * Create ordering button down in given composite
	 * @param composite to create button in
	 */
	void createOrderingButtonDown(Composite composite) {
		Button buttonDownLoc = new Button(composite, SWT.PUSH);
		buttonDown = buttonDownLoc;
		buttonDownLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_DOWN));
		SWTFactoryProxy.INSTANCE.setTestId(buttonDownLoc, TestIDs.PERIPHS_REORDER_DIALOG_BUTTON_DOWN);
		GridDataFactory.fillDefaults().grab(false, true).span(1, 1).applyTo(buttonDownLoc);
		buttonDownLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TreeViewer treeViewerLoc = getTreeViewer();
				if (treeViewerLoc == null) {
					return;
				}
				InitializationOrderInstance instance = getCurrentlySelectedInstance();
				if (instance != null) {
					instance.getGroup().moveAfter(instance);
				}
				treeViewerLoc.refresh();
				updateOrderingButtonsEnabledState();
			}
		});
	}

	/**
	 * @return the currently selected instance in tree viewer or {@code null} when selection is not an instance
	 */
	@Nullable InitializationOrderInstance getCurrentlySelectedInstance() {
		TreeViewer treeViewerLoc = getTreeViewer();
		if (treeViewerLoc == null) {
			return null;
		}
		ITreeSelection selection = treeViewerLoc.getStructuredSelection();
		Object firstElement = selection.getFirstElement();
		if (firstElement instanceof InitializationOrderInstance) {
			return (InitializationOrderInstance) firstElement;
		}
		return null;
	}

	/**
	 * Updates enabled state of the button down
	 */
	private void updateButtonDownEnabledState() {
		Button button = buttonDown;
		if (button == null) {
			return;
		}
		InitializationOrderInstance instance = getCurrentlySelectedInstance();
		if (instance != null) {
			button.setEnabled(instance.getGroup().canMoveAfter(instance));
		} else {
			button.setEnabled(false);
		}
	}

	/**
	 * Updates enabled state of the button up
	 */
	private void updateButtonUpEnabledState() {
		Button button = buttonUp;
		if (button == null) {
			return;
		}
		InitializationOrderInstance instance = getCurrentlySelectedInstance();
		if (instance != null) {
			button.setEnabled(instance.getGroup().canMoveBefore(instance));
		} else {
			button.setEnabled(false);
		}
	}

	/**
	 * Updates the enabled state of both ordering buttons
	 */
	void updateOrderingButtonsEnabledState() {
		updateButtonDownEnabledState();
		updateButtonUpEnabledState();
	}

	/**
	 * Store reordered instances
	 * @param functionalGroup which holds these instances
	 * @param instances list of instances
	 */
	void storeInstances(IFunctionalGroup functionalGroup, LinkedList<IComponentInstanceConfig> instances) {
		instancesOfGroups.put(functionalGroup, instances);
	}

	/**
	 * Get the instances in order which they appear in this dialog
	 * @param functionalGroup which holds those instances
	 * @return the list of instances in order specified by user
	 */
	LinkedList<IComponentInstanceConfig> loadInstances(IFunctionalGroup functionalGroup) {
		LinkedList<IComponentInstanceConfig> list = instancesOfGroups.computeIfAbsent(functionalGroup, x -> new LinkedList<>());
		assert list != null; // Cannot be null
		return list;
	}

	/**
	 * Adds the selection listener to functional group combo box
	 * @param combo to which the listener should be added
	 */
	public void addComboSelectionListener(InstantSearchList combo) {
		combo.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				String functionalGroupName = UtilsText.safeString(e.text);
				IFunctionalGroup currentFunctionalGroupLoc = getCurrentlySelectedFunctionalGroup();
				TreeViewer treeViewerLoc = getTreeViewer();
				if (treeViewerLoc == null) {
					return;
				}
				if (currentFunctionalGroupLoc != null) {
					// Store order of previously selected group
					storeInstances(currentFunctionalGroupLoc, getOrderedInstances(treeViewerLoc));
				}
				IFunctionalGroup functionalGroup = getFunctionalGroup(functionalGroupName);
				if (functionalGroup == null) {
					return;
				}
				setCurrentFunctionalGroup(functionalGroup);
				InitializationPriorities initializationPriorities = ((PeriphsProfile) functionalGroup.getChildContext().getRoot()).getMcu().getInitializationPriorities();
				if (initializationPriorities == null) {
					return;
				}
				LinkedList<@NonNull IComponentInstanceConfig> instances = new LinkedList<>(functionalGroup.getInstances().values());
				storeInstances(functionalGroup, instances);
				ArrayList<InitializationOrderGroup> groups = new ArrayList<>();
				InitializationOrderGroup defaultGroup = null;
				String defaultGroupName = initializationPriorities.getDefaultGroupName();
				for (InitializationPrioritiesGroup priorityGroup : initializationPriorities.getSortedGroups()) {
					InitializationOrderGroup group = new InitializationOrderGroup(priorityGroup);
					groups.add(group);
					if (priorityGroup.getId().equals(defaultGroupName)) {
						defaultGroup = group;
					}
					ArrayList<InitializationOrderInstance> instancesInGroup = new ArrayList<>();
					List<@NonNull IComponentInstanceConfig> instancesOfGroup = InitializationPrioritiesHelper.getComponentsOfGroup(instances, priorityGroup);
					for (IComponentInstanceConfig instance : instancesOfGroup) {
						instancesInGroup.add(new InitializationOrderInstance(instance, group));
					}
					instances.removeAll(instancesOfGroup);
					group.setInstances(instancesInGroup);
				}
				// Add the instances without mentioned types to the default group
				if (defaultGroup != null) {
					List<InitializationOrderInstance> mentionedInstances = defaultGroup.getInstances();
					List<InitializationOrderInstance> newList = new LinkedList<>(mentionedInstances);
					for (IComponentInstanceConfig instance : instances) {
						newList.add(new InitializationOrderInstance(instance, defaultGroup));
					}
					defaultGroup.setInstances(newList);
					if (!newList.isEmpty() && !groups.contains(defaultGroup)) {
						groups.add(defaultGroup);
					}
				}
				treeViewerLoc.setInput(groups);
				treeViewerLoc.refresh();
				treeViewerLoc.setExpandedElements(groups.toArray());
				updateOrderingButtonsEnabledState();
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(@NonNull Composite parent) {
		createButton(parent, IDialogConstants.CANCEL_ID, Messages.get().InitializationOrderDialog_CancelButton, true);
		createButton(parent, IDialogConstants.OK_ID, Messages.get().InitializationOrderDialog_SaveOrderButton, true);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		TreeViewer treeViewerLoc = getTreeViewer();
		if (treeViewerLoc == null) {
			LOGGER.log(Level.SEVERE, "[TOOL] TreeViewer was not yet created and OK button was pressed"); //$NON-NLS-1$
			super.okPressed();
			return;
		}
		IFunctionalGroup currentFunctionalGroupLoc = getCurrentlySelectedFunctionalGroup();
		if (currentFunctionalGroupLoc == null) {
			super.okPressed();
			return;
		}
		storeInstances(currentFunctionalGroupLoc, getOrderedInstances(treeViewerLoc));
		saveOrderingToProfile();
		getProperties().getControllerWrapper().getController().handleSettingChange(EventTypes.CHANGE, InitializationOrderDialog.class,
				Messages.get().InitializationOrderDialog_DirtyMessage, true, getProperties().getRoot(), true);
		super.okPressed();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#getInitialSize()
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(400, 400);
	}

	/**
	 * Saves the user specified order of instances to profile
	 * @return {@code true} when the saving was successful, {@code false} otherwise
	 */
	protected boolean saveOrderingToProfile() {
		IRoot root = getProperties().getRoot();
		for (IFunctionalGroup group : root.getFunctionalGroups()) {
			ArrayList<IComponentInstanceConfig> firstOnly = new ArrayList<>();
			ArrayList<IComponentInstanceConfig> secondOnly = new ArrayList<>();
			LinkedList<IComponentInstanceConfig> orderedInstances = loadInstances(group);
			LinkedList<IComponentInstanceConfig> instances = new LinkedList<>(group.getInstances().values());
			if (!CollectionsUtils.difference(instances, orderedInstances, firstOnly, secondOnly)) {
				LOGGER.log(Level.SEVERE, "[TOOL] Instances of the functional group {0} and the ordered list specified by user are not the same collection with different order", //$NON-NLS-1$
						group.getName());
				return false;
			}
			// Updates order in the functional group
			HashMap<String, Integer> orderMap = new HashMap<>();
			for (int i = 0; i < orderedInstances.size(); i++) {
				IComponentInstanceConfig instance = orderedInstances.get(i);
				orderMap.put(instance.getName(), Integer.valueOf(i));
			}
			// Set the new order
			group.setInstancesOrder(orderMap);
		}
		return true;
	}

	/**
	 * Returns the instances in same order as they appear in the tree
	 * @param viewer of the tree
	 * @return ordered list of instances or empty list in case of an error
	 */
	protected LinkedList<IComponentInstanceConfig> getOrderedInstances(TreeViewer viewer) {
		Object input = viewer.getInput();
		if (!(input instanceof List<?>)) {
			return new LinkedList<>();
		}
		@SuppressWarnings("unchecked") // When input is a list then it is a list of groups
		List<InitializationOrderGroup> groups = (List<InitializationOrderGroup>) input;
		return new LinkedList<>(groups.stream().flatMap(g -> g.getInstances().stream()).map(InitializationOrderInstance::getInstance).collect(CollectorsUtils.toList()));
	}

	/**
	 * Get functional group by it's name
	 * @param name of the group
	 * @return the functional group or {@code null} when the functional group is not found
	 */
	protected @Nullable IFunctionalGroup getFunctionalGroup(String name) {
		return getProperties().getRoot().getFunctionalGroup(name);
	}

	/**
	 * Set currently selected group
	 * @param functionalGroup that is currently selected
	 */
	void setCurrentFunctionalGroup(IFunctionalGroup functionalGroup) {
		currentFunctionalGroup = functionalGroup;
	}

	/**
	 * @return the tree viewer or {@code null} when it was not yet created
	 */
	@Nullable TreeViewer getTreeViewer() {
		return treeViewer;
	}

	/**
	 * @return the currently selected functional group or {@code null} when none is currently selected
	 */
	@Nullable IFunctionalGroup getCurrentlySelectedFunctionalGroup() {
		return currentFunctionalGroup;
	}

	/**
	 * Opens the initialization order dialog
	 * @param properties of the opened dialog
	 */
	public static void open(InitializationOrderDialogProperties properties) {
		InitializationOrderDialog dialog = new InitializationOrderDialog(properties);
		dialog.setBlockOnOpen(true);
		dialog.open();
	}
}
