 /**
 * Copyright 2019-2022 NXP
* Created: Jan 30, 2019
*/
package com.nxp.swtools.periphs.gui.controller;

import java.util.Arrays;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;

import com.nxp.swtools.common.ui.utils.perspectives.PerspectivesHelper;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.controller.AController;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingViewHelper;
import com.nxp.swtools.periphs.gui.view.componentsettings.GlobalComponentSettingView;
import com.nxp.swtools.resourcetables.model.config.ChildContext;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.view.ToolView;

/**
 * Abstract class used as generic GUI Controller
 * @author Rares-Mihai Gafton nxf38680
 *
 */
// FIXME Maria V99 - use this class for event management in GUI classes in the final code migration to common code
public abstract class APeriphGUIController extends AController {
	/** Logger of the class */
	protected static final Logger LOGGER = LogManager.getLogger(APeriphGUIController.class);

	/**
	 * Method that opens the proper Configuration Settings View
	 * @param viewSite
	 * @param componentType
	 * @param componentName
	 * @param global
	 * @param activate
	 * @return {@code true} when view was opened, otherwise returns {@code false}
	 */
	public abstract boolean openConfigurationSettingsView(IViewSite viewSite, String componentType, String componentName, boolean global, boolean activate);

	/**
	 * Method that opens the proper Configuration Settings View
	 * @param viewSite site in which the view should open
	 * @param uuid of the instance
	 * @param activate {@code true} when the view should be active right away, {@code false} otherwise
	 * @return {@code true} when view was opened, otherwise returns {@code false}
	 */
	public abstract boolean openConfigurationSettingsView(IViewSite viewSite, String uuid, boolean activate);

	/**
	 * Add one more permitted occurrence of view with the given identification base
	 * @param identificationBase base part of the view identification
	 */
	public abstract void addPermittedOccurrenceOfView(String identificationBase);

	/**
	 * Open view that should be opened for given instance
	 * @param viewSite eclipse view site
	 * @param instance
	 * @param activate
	 * @return {@code true} when view was opened, otherwise returns {@code false}
	 */
	public boolean openViewOfInstance(IViewSite viewSite, IComponentInstanceConfig instance, boolean activate) {
		String viewOnCreation = instance.getComponent().getDefaultViewId();
		if (!UtilsText.isEmpty(viewOnCreation)) {
			final String viewOnCreationFinal = UtilsText.safeString(viewOnCreation);
			// Open view specified in component
			// FIXME TomasR v13 maintenance - Unify with ComponentSettingView behavior in some common class
			String identificationBase = instance.getUUID();
			IWorkbenchPage workbenchPage = viewSite.getPage();
			Set<@NonNull Integer> permittedOccurrences = ComponentSettingViewHelper.getInstance().getPermittedOccurrences(identificationBase);
			Integer lastOpenOccurrence = Integer.valueOf(-1);
			for (Integer i : permittedOccurrences) {
				String identification = identificationBase + ComponentSettingView.SECONDARY_ID_IDENTIFICATION_OCCURENCE_SEPARATOR + i.toString();
				boolean viewPresent = Arrays.asList(workbenchPage.getViewReferences()).stream()
					.filter(x -> viewOnCreationFinal.equals(x.getId()))
					.anyMatch(x -> identification.equals(x.getSecondaryId()));
				if (!viewPresent) {
					return openView(UtilsText.safeString(viewOnCreation), identification, activate);
				}
				lastOpenOccurrence = i;
			}
			if (lastOpenOccurrence.intValue() != -1) {
				return openView(UtilsText.safeString(viewOnCreation), identificationBase + ComponentSettingView.SECONDARY_ID_IDENTIFICATION_OCCURENCE_SEPARATOR
						+ lastOpenOccurrence.toString(), activate);
			}
			return false;
		}
		return openConfigurationSettingsView(viewSite, instance.getUUID(), activate);
	}

	/**
	 * @return view site that could be used or {@code null} when current view site cannot be used
	 */
	public @Nullable IViewSite getCurrentViewSite() {
		IWorkbenchPart activePart = getActivePart();
		if (activePart == null) {
			return null;
		}
		IWorkbenchPartSite site = activePart.getSite();
		if (site instanceof IViewSite) {
			return (IViewSite) site;
		}
		return null;
	}

	/**
	 * @return default control of active ToolView or {@code null} when active view is not ToolView
	 */
	public @Nullable Control getDefaultControlOfActiveToolView() {
		IWorkbenchPart activePart = getActivePart();
		if (activePart instanceof ToolView) {
			ToolView toolView = (ToolView) activePart;
			return toolView.getToolMainControl();
		}
		return null;
	}

	/**
	 * Opens view of given child and scrolls to its control
	 * @param child whose view should be opened
	 */
	public void openViewOfChild(IChild child) {
		IViewSite currentViewSite = getCurrentViewSite();
		if (currentViewSite == null) {
			return;
		}
		ChildContext childContext = child.getChildContext();
		IComponentConfig componentConfig = childContext.getComponentConfig();
		if (componentConfig == null) {
			return;
		}
		IComponentInstanceConfig componentInstanceConfig = childContext.getComponentInstanceConfig();
		if (componentInstanceConfig != null) {
			final String uuid = componentInstanceConfig.getUUID();
			if (uuid.isEmpty()) {
				openViewOfInstance(currentViewSite, componentInstanceConfig, true);
				Control defaultControlOfActiveView = getDefaultControlOfActiveToolView();
				Composite composite;
				if (defaultControlOfActiveView instanceof Composite) {
					composite = (Composite) defaultControlOfActiveView;
					ComponentSettingView componentInstanceView = (ComponentSettingView) PerspectivesHelper.getViewById(ComponentSettingView.ID,
							componentInstanceConfig.getType() + ComponentSettingView.SECONDARY_ID_NAME_TYPE_SEPARATOR + componentInstanceConfig.getName());
					if (componentInstanceView != null) {
						ComponentSettingView.focusOnChild(componentInstanceView, child);
					} else {
						ScrolledCompositeHelper.scrollToControlWithId(composite, TestIDs.PERIPHS_SETTING_LABEL + child.getId());
					}
				} else {
			LOGGER.severe("[TOOL] Provided control of active tool view is either null or is not composite"); //$NON-NLS-1$
				}
			} else {
				ComponentSettingView.open(currentViewSite.getPage(), uuid, true, child);
			}
		} else {
			final String componentType = componentConfig.getType();
			final String componentName = componentConfig.getName();
			ComponentSettingView.open(currentViewSite.getPage(), componentType, componentName, true, true, child);
		}

	}

	/**
	 * Opens global settings view and scrolls to control of given child
	 * @param child to which it should be scrolled in global settings view
	 */
	public void openGlobalViewOfChild(IChild child) {
		IViewSite currentViewSite = getCurrentViewSite();
		if (currentViewSite == null) {
			return;
		}
		ChildContext childContext = child.getChildContext();
		IComponentConfig componentConfig = childContext.getComponentConfig();
		if (componentConfig == null) {
			return;
		}
		GlobalComponentSettingView.open(currentViewSite, componentConfig.getType(), true);
		Control defaultControlOfActiveView = getDefaultControlOfActiveToolView();
		if (defaultControlOfActiveView instanceof Composite) {
			IViewPart globalSettingsView = PerspectivesHelper.getViewById(GlobalComponentSettingView.ID, null);
			if (globalSettingsView != null) {
				ComponentSettingView.focusOnChild(globalSettingsView, child);
			} else {
				ScrolledCompositeHelper.scrollToControlWithId((Composite) defaultControlOfActiveView, TestIDs.PERIPHS_SETTING_LABEL + child.getId());
			}
		} else {
			LOGGER.severe("[TOOL] Provided control of active tool view is either null or is not composite"); //$NON-NLS-1$
		}
	}

	/**
	 * @return active part of eclipse workbench or {@code null} when no part is active
	 */
	private static @Nullable IWorkbenchPart getActivePart() {
		IWorkbenchPage activePage = getActivePage();
		if (activePage == null) {
			return null;
		}
		return activePage.getActivePart();
	}

	/**
	 * @return active workbench page or {@code null} when none is active
	 */
	private static @Nullable IWorkbenchPage getActivePage() {
		IWorkbenchWindow activeWorkbenchWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (activeWorkbenchWindow == null) {
			return null;
		}
		IWorkbenchPage activePage = activeWorkbenchWindow.getActivePage();
		return activePage;
	}

	/**
	 * Open view with given ID
	 * @param viewId id of view
	 * @return {@code true} when the view was opened, otherwise returns {@code false}
	 */
	public boolean openView(String viewId) {
		IWorkbenchPage activePage = getActivePage();
		if (activePage != null) {
			try {
				activePage.showView(viewId);
				return true;
			} catch (PartInitException e) {
				LOGGER.log(Level.SEVERE, "[TOOL] Requested view cannot be opened", e); //$NON-NLS-1$
				return false;
			}
		}
		return false;
	}

	/**
	 * Open view with given ID and secondary ID
	 * @param viewId id of view
	 * @param secondaryId secondary id
	 * @param activate {@code true} when view should be activated after creation, {@code false} just create it
	 * @return {@code true} when the view was opened, otherwise returns {@code false}
	 */
	public boolean openView(String viewId, String secondaryId, boolean activate) {
		IWorkbenchPage activePage = getActivePage();
		if (activePage != null) {
			try {
				activePage.showView(viewId, secondaryId, activate ? IWorkbenchPage.VIEW_ACTIVATE : IWorkbenchPage.VIEW_VISIBLE);
				return true;
			} catch (PartInitException e) {
				LOGGER.log(Level.SEVERE, "[TOOL] Requested view cannot be opened", e); //$NON-NLS-1$
				return false;
			}
		}
		return false;
	}
}
