/*******************************************************************************
 * Copyright (c) 2004, 2010 BitMethods Inc and others.
 * Copyright 2023 NXP
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BitMethods Inc - initial API and implementation
 *     Sascha Radike <sradike@ejectlag.com> - Support for workspace browsing and small improvements
 *     James Blackburn (Broadcom Corp.)
 *******************************************************************************/
package com.freescale.s32ds.ext.cdt.utils.epl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Stream;

import org.eclipse.cdt.ui.CDTSharedImages;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;

import com.nxp.s32ds.ext.rcp.jse.utils.FuncUtils;



public class ListEditorEpl {

	private static final String MOVEUP_STR = "Up";
	private static final String MOVEDOWN_STR = "Down";

	private ToolBar toolBar;
	private ToolItem moveUpItem, moveDownItem;
	private Label title;
	private final List list;

	private final Collection<Runnable> listeners = new ArrayList<>();

	private final Image IMG_MOVEUP = CDTSharedImages.getImage(CDTSharedImages.IMG_FILELIST_MOVEUP);
	private final Image IMG_MOVEDOWN = CDTSharedImages.getImage(CDTSharedImages.IMG_FILELIST_MOVEDOWN);

	public ListEditorEpl(Composite parent, String compTitle) {
		//file panel
		final Composite filePanel = new Composite(parent, SWT.NONE) {{
			final GridLayout form1 = new GridLayout();
			form1.numColumns = 1;
			form1.horizontalSpacing = 0;
			form1.verticalSpacing = 0;
			form1.marginHeight = 0;
			form1.marginWidth = 0;
			setLayout(form1);
			setLayoutData(new GridData(GridData.FILL_BOTH));
		}};
		
		// title panel
		final Composite titlePanel = new Composite(filePanel, SWT.BORDER) {{
			final GridLayout titleform = new GridLayout(2, false);
			titleform.horizontalSpacing = 0;
			titleform.verticalSpacing = 0;
			titleform.marginHeight = 0;
			titleform.marginWidth = 0;
			setLayout(titleform);
			final GridData tgdata = new GridData(GridData.FILL_HORIZONTAL);
			tgdata.heightHint = IDialogConstants.BUTTON_BAR_HEIGHT;
			setLayoutData(tgdata);
			title = new Label(this, SWT.NONE | SWT.BOLD);
			title.setText("  " + compTitle);
			final GridData grid2 = new GridData(GridData.FILL_HORIZONTAL);
			title.setLayoutData(grid2);
		}};
		
		//button panel
		new Composite(titlePanel, SWT.NONE) {{
			final GridLayout form2 = new GridLayout();
			form2.numColumns = 5;
			form2.horizontalSpacing = 0;
			form2.verticalSpacing = 0;
			form2.marginWidth = 0;
			form2.marginHeight = 0;
			setLayout(form2);
			
			// toolbar
			toolBar = new ToolBar(this, SWT.HORIZONTAL | SWT.RIGHT | SWT.FLAT);
			// moveup toolbar item
			moveUpItem = new ToolItem(toolBar, SWT.PUSH);
			moveUpItem.setImage(IMG_MOVEUP);
			moveUpItem.setToolTipText(MOVEUP_STR);
			moveUpItem.addListener(SWT.Selection, e -> upPressed());
			// movedown toolbar item
			moveDownItem = new ToolItem(toolBar, SWT.PUSH);
			moveDownItem.setImage(IMG_MOVEDOWN);
			moveDownItem.setToolTipText(MOVEDOWN_STR);
			moveDownItem.addListener(SWT.Selection, e -> downPressed());
			final GridData grid3 = new GridData(GridData.FILL_HORIZONTAL | GridData.HORIZONTAL_ALIGN_END);
			setLayoutData(grid3);			
		}};
		
		// list control
		list = new List(filePanel, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.MULTI);
		final GridData grid4 = new GridData(GridData.FILL_BOTH);
		// force the list to be no wider than the title bar
		final Point preferredSize = titlePanel.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		grid4.widthHint = preferredSize.x;
		grid4.heightHint = preferredSize.y * 3;
		grid4.horizontalSpan = 2;
		list.setLayoutData(grid4);
		list.addListener(SWT.Selection, e -> selectionChanged());

		selectionChanged();
	}

	public void setList(String[] listVal) {
		updateAndNotify(() -> {
			list.removeAll();
			Stream.of(listVal).forEach(list::add);
		});
	}

	public void addChangeListener(Runnable listener) {
		listeners.add(listener);
	}

	private void updateAndNotify(Runnable runnable) {
		final String[] before = list.getItems();
		runnable.run();
		if (!Arrays.equals(before, list.getItems())) {
			listeners.forEach(Runnable::run);
		}
	}
	
	public void removeAll() {
		updateAndNotify(list::removeAll);
	}

	public String[] getItems() {
		return list.getItems();
	}

	private void upPressed() {
		updateAndNotify(() -> {
			final int index = list.getSelectionIndex();
			final String curSelList = list.getItem(index);
			final String preList = list.getItem(index - 1);
			list.setItem(index - 1, curSelList);
			list.setItem(index, preList);
			list.setSelection(index - 1);
			selectionChanged();
		});
	}

	private void downPressed() {
		updateAndNotify(() -> {
			final int index = list.getSelectionIndex();
			final String curSelList = list.getItem(index);
			final String nextList = list.getItem(index + 1);
			list.setItem(index + 1, curSelList);
			list.setItem(index, nextList);
			list.setSelection(index + 1);
			selectionChanged();
		});
	}

	private void selectionChanged() {
		final int index = list.getSelectionIndex();
		final int size = list.getItemCount();
		final int count = list.getSelectionCount();
		moveUpItem.setEnabled(size > 1 && index > 0 && count == 1);
		moveDownItem.setEnabled(size > 1 && index >= 0 && index < size - 1 && count == 1);
	}

	public List getListControl() {
		return list;
	}

	public Label getLabelControl() {
		return title;
	}

	public void setEnabled(boolean enabled) {
		Stream.of(title, toolBar, list).forEach(FuncUtils.bindConsumer(Control::setEnabled, enabled));
	}

}
