/*******************************************************************************
 * Copyright (c) 2014 Liviu Ionescu.
 * Copyright 2023 NXP
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Liviu Ionescu - initial version
 *******************************************************************************/

package com.freescale.s32ds.ext.cdt.utils.epl;

import java.io.File;
import java.net.URL;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.Stream;

import org.eclipse.cdt.core.model.CoreModel;
import org.eclipse.cdt.core.settings.model.ICConfigurationDescription;
import org.eclipse.cdt.core.settings.model.ICExternalSetting;
import org.eclipse.cdt.core.settings.model.ICProjectDescription;
import org.eclipse.cdt.core.settings.model.ICSettingEntry;
import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.preferences.ConfigurationScope;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.internal.WorkbenchWindow;

import com.freescale.s32ds.ext.cdt.utils.epl.help.FSLCCorePluginEpl;
import com.freescale.s32ds.ext.cdt.utils.epl.help.Messages;
import com.freescale.s32ds.ext.cdt.utils.epl.help.OSFamilyEpl;
import com.nxp.s32ds.ext.rcp.jse.utils.FuncUtils;

/**
 * Various utilities that use Eclipse global classes, grouped together more like
 * a reference.
 * <ul>
 * <li>Platform</li>
 * <li>PlatformUI</li>
 * <li>ManagedBuildManager</li>
 * <li>ImmediateExecutor</li>
 * </ul>
 * 
 * Other interesting places to search for utility functions are:
 * <ul>
 * <li>Plugin</li>
 * </ul>
 *
 * For debugging, use
 * 
 * <pre>
 * private static final boolean DEBUG_TWO =
 *     ExamplesPlugin.getDefault().isDebugging() &&
 *        "true".equalsIgnoreCase(Platform.getDebugOption(
 *        "org.eclipse.faq.examples/debug/option2"));
 *  ...
 *  if (DEBUG_TWO)
 *     System.out.println("Debug statement two.");
 * </pre>
 * 
 * This will test two properties like
 * <ul>
 * <li>org.eclipse.faq.examples/debug=true</li>
 * <li>org.eclipse.faq.examples/debug/option2=true</li>
 * </ul>
 * These properties should be stored in a .option file in the plug-in root, or
 * in a custom file whose name is passed to the Eclipse -debug option.
 * <p>
 * See also the <a href=
 * "https://wiki.eclipse.org/FAQ_How_do_I_use_the_platform_debug_tracing_facility"
 * >Eclipse Wiki</a>.
 * 
 */
@SuppressWarnings("restriction")
public class EclipseUtilsEpl {

	public static final String PROPERTY_OS_NAME = "os.name"; //$NON-NLS-1$
	public static final String PROPERTY_OS_ARCH = "os.arch"; //$NON-NLS-1$
	
	// ------------------------------------------------------------------------
    public static final String FORBIDDEN_CHARS = "#;=,:@*?<>\\|\"\' {}[]()`&$%"; //$NON-NLS-1$

	public static final String EMPTY = ""; //$NON-NLS-1$

    private static final int STRING_MAX_LENGTH = 250;
	
    public static boolean isWindowsXP() {
		return System.getProperty(PROPERTY_OS_NAME).toLowerCase(Locale.getDefault()).equalsIgnoreCase("Windows XP"); //$NON-NLS-1$
	}

    public static boolean isWindows64() {
		if (System.getProperty(PROPERTY_OS_NAME).contains("Windows")) { //$NON-NLS-1$
		    return System.getenv("ProgramFiles(x86)") != null; //$NON-NLS-1$
		} else {
		    return System.getProperty(PROPERTY_OS_ARCH).indexOf("64") != -1; //$NON-NLS-1$
		}
	}
	
	// ------------------------------------------------------------------------

	/**
	 * Get the separator used to compose PATHs.
	 * 
	 * @return a string.
	 */
	public static String getPathSeparator() {
		return OSFamilyEpl.WINDOWS.isCurrent() ? ";" : ":"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	// ------------------------------------------------------------------------

	public static void openExternalBrowser(URL url) throws PartInitException {
		PlatformUI.getWorkbench().getBrowserSupport().getExternalBrowser().openURL(url);
	}

	public static void openExternalFile(IPath path) throws PartInitException {
		// Open external file. If the file is text, it will be opened
		// in the Eclipse editor, otherwise a system viewer is selected.
		IDE.openEditorOnFileStore(getActivePage(), EFS.getLocalFileSystem().getStore(path));
	}

	public static void openFileWithInternalEditor(IPath path) throws PartInitException {
		// Open a non-resource regular file with the Eclipse internal text
		// editor
		IDE.openInternalEditorOnFileStore(getActivePage(), EFS.getLocalFileSystem().getStore(path));
	}

	// ------------------------------------------------------------------------

	/**
	 * Find the project with the given project name.
	 * 
	 * @param name
	 *            a string with the project name
	 * @return the project or null, if not found
	 */
	public static IProject getProjectByName(String name) {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(name);
	}

	/**
	 * Find the project selected in the Project viewer. This works only if the
	 * project is really selected (i.e. the Project Explorer has the focus, and
	 * the project name is coloured in blue); if the focus is lost, the function
	 * returns null.
	 *
	 * @return the project or null, if not found
	 */
	public static IProject getSelectedProject() {

		IWorkbenchWindow window = getActiveWorkbenchWindow();
		if (window != null) {

			Object selection = window.getSelectionService().getSelection();
			if (selection instanceof IStructuredSelection
					&& ((IStructuredSelection) selection).size() == 1) {
				Object firstElement = ((IStructuredSelection) selection)
						.getFirstElement();
				if (firstElement instanceof IAdaptable) {
					IProject project = ((IAdaptable) firstElement)
							.getAdapter(IProject.class);
					return project;
				}
			}
		}

		return null;
	}

	/**
	 * Find the active page. Used, for example, to check if a part (like a view)
	 * is visible (page.ispartVisible(part)).
	 * <p>
	 * Preferably use getSite().getPage().
	 * 
	 * @return the active page.
	 */
	public static IWorkbenchPage getActivePage() {
		return getActiveWorkbenchWindow().getActivePage();
	}

	public static IWorkbenchPage getActivePage(IWorkbenchPart part) {
		return part.getSite().getWorkbenchWindow().getActivePage();
	}

	public static Shell getShell() {
		return getActiveWorkbenchWindow().getShell();
	}

	// ------------------------------------------------------------------------

	public static IConfiguration getConfigurationFromDescription(ICConfigurationDescription configDescription) {
		return ManagedBuildManager.getConfigurationForDescription(configDescription);
	}

	// ------------------------------------------------------------------------

	/**
	 * Helper function to open an error dialog.
	 * 
	 * @param title
	 * @param message
	 * @param e
	 */
	static public void openError(String title, String message, Exception e) {
		FSLCCorePluginEpl.log(title + " -> " + message); //$NON-NLS-1$
	}

	private static Runnable setStatusLineMessage(String message) {
		return () -> {
			IWorkbenchWindow window = getActiveWorkbenchWindow();
			if (window instanceof WorkbenchWindow) {
				((WorkbenchWindow) window).getStatusLineManager().setErrorMessage(message);
			}
		};
	}

	private static IWorkbenchWindow getActiveWorkbenchWindow() {
		Assert.isNotNull(Display.getCurrent(), "getActiveWorkbenchWindow() is called from a non-UI thread");
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
	}
	
	public static void clearStatusMessage() {
		Display.getDefault().syncExec(setStatusLineMessage(EMPTY));
	}

	/**
	 * Shows status message in RCP
	 * 
	 * @param message
	 *            message to be displayed
	 * @param isError
	 *            if its an error message or normal message
	 */
	public static void showStatusMessage(final String message) {
		FSLCCorePluginEpl.log(message);
		Display.getDefault().asyncExec(setStatusLineMessage(message));
	}

	public static void showStatusErrorMessage(final String message) {
		FSLCCorePluginEpl.log(new Status(IStatus.WARNING, FSLCCorePluginEpl.PLUGIN_ID, message));
		Display.getDefault().asyncExec(setStatusLineMessage("  " + message)); //$NON-NLS-1$
	}

	// ------------------------------------------------------------------------

	/**
	 * Search the given scopes and return the non empty trimmed string or the
	 * default.
	 * 
	 * @param pluginId
	 *            a string with the plugin id.
	 * @param key
	 *            a string with the key to search.
	 * @param defaultValue
	 *            a string with the default, possibly null.
	 * @param contexts
	 *            an array of IScopeContext.
	 * @return a trimmed string or the given default, possibly null.
	 */
	public static String getPreferenceValueForId(String pluginId, String key, String defaultValue, IScopeContext[] contexts) {
		return Stream.of(contexts).map(c -> c.getNode(pluginId).get(key, null)).filter(Objects::nonNull)
				.map(String::trim).filter(FuncUtils.not(String::isEmpty)).findAny().orElse(defaultValue);
	}

	/**
	 * Compute a maximum array of scopes where to search for.
	 * 
	 * @param project
	 *            the IProject reference to the project, possibly null.
	 * @return an array of IScopeContext.
	 */
	public static IScopeContext[] getPreferenceScopeContexts(IProject project) {
		// If the project is known, the contexts searched will include the
		// specific ProjectScope.
		return Stream.concat(
			project == null ? Stream.of() : Stream.of(new ProjectScope(project)),
			Stream.of(InstanceScope.INSTANCE, ConfigurationScope.INSTANCE, DefaultScope.INSTANCE)
		).toArray(IScopeContext[]::new);
	}

	/**
	 * Search all scopes and return the non empty trimmed string or the default.
	 * 
	 * @param pluginId
	 *            a string with the plugin id.
	 * @param key
	 *            a string with the key to search.
	 * @param defaultValue
	 *            a string with the default, possibly null.
	 * @param project
	 *            the IProject reference to the project, possibly null.
	 * @return a trimmed string or the given default, possibly null.
	 */
	public static String getPreferenceValueForId(String pluginId, String key, String defaultValue, IProject project) {
		return getPreferenceValueForId(pluginId, key, defaultValue, getPreferenceScopeContexts(project));
	}

	// ------------------------------------------------------------------------
	
	/**
	 * Unified check of file/project name for forbidden symbols 
	 * 
	 * @param name
	 * @param object
	 * @return
	 */
	private static String checkForSymbols(String name, String object) {
		if (name == null || name.trim().length() == 0) {
			return String.format(Messages.EclipseUtils_name_empty, object);
		}

		StringBuilder sb = new StringBuilder();
		for (char c : name.toCharArray()) {
			if (FORBIDDEN_CHARS.chars().anyMatch(x -> c == x)) {
				sb.append(c);
				break;
			}
		}
        if (sb.length() > 0) {
        	return String.format(Messages.EclipseUtils_bad_symbols, object, sb.toString());
        }
        
        if (name.length() > STRING_MAX_LENGTH) {
            return String.format(Messages.EclipseUtils_too_large, object);
        }
        
        return null;
	}
	
	/**
	 * Unified check of project or directory name
	 * 
	 * @param name - user-provided name
	 * @param full - true for project name, false for directory
	 * @return
	 */
	public static IStatus checkProjectName(String name, boolean full) {
				
		String object = full ? Messages.EclipseUtils_Project : Messages.EclipseUtils_Directory;
		
        String err = checkForSymbols(name, object);
        if (err != null) {
        	return new Status(IStatus.ERROR, FSLCCorePluginEpl.PLUGIN_ID, err);
        }
        
        //Don't know if we really need that projects would start from letters
		if (full) { // additional checks for project only, not for dir 
	        char first = name.charAt(0);
	        if (!(Character.isLetter(first) || Character.isDigit(first))) {
	        	return new Status(IStatus.ERROR, FSLCCorePluginEpl.PLUGIN_ID, 
	       			String.format(Messages.EclipseUtils_first_letter, object, first));
	        }

			IWorkspace wsp = ResourcesPlugin.getWorkspace();
			
			IStatus status = wsp.validateName(name, IResource.PROJECT);
			if (!status.isOK()) {
				return status;
			}
	        if (wsp.getRoot().getProject(name).exists()) {
	            return new Status(IStatus.ERROR, FSLCCorePluginEpl.PLUGIN_ID, Messages.EclipseUtils_name_equal);
	        }
	        
	        // ENGR00380964 - check projects, case-insensitive
	        for (IProject p : wsp.getRoot().getProjects()) {
	        	if (name.equalsIgnoreCase(p.getName())) {
		            return new Status(IStatus.ERROR, FSLCCorePluginEpl.PLUGIN_ID, String.format(Messages.EclipseUtils_name_similar, p.getName()));
	        	}
	        }
	       
	        // check case-insensitive for dirs or files, not exactly projects for d
	        File f = new File(wsp.getRoot().getLocationURI());
	        File[] foundFiles = f.listFiles(file -> name.equalsIgnoreCase(file.getName()));
	        if (foundFiles != null && foundFiles.length > 0) {
	        	File ff = foundFiles[0];
	        	return new Status(IStatus.ERROR, FSLCCorePluginEpl.PLUGIN_ID, String.format(String.format(Messages.EclipseUtils_already_exists, 
		            	ff.isDirectory() ? Messages.EclipseUtils_dir : Messages.EclipseUtils_file, ff.getName())));
	        }
		}
		
        return Status.OK_STATUS;
	}
	
	/**
	 * Reads external setting value either from selected configuration or from Active one.  
	 * 
	 * @param key      - external setting key 
	 * @param project  - affected project 
	 * @param cfg      - required configuration, or NULL for Active.
	 * 
	 * @return  value of external setting, or NULL in case when it's absent.
	 */
	public static String readExternalSetting(String key, IProject project, ICConfigurationDescription cfg) {
		ICProjectDescription desc = CoreModel.getDefault().getProjectDescription(project, false);
		cfg = cfg == null ? desc.getActiveConfiguration() : cfg;
		return Stream.of(cfg.getExternalSettings())
			.map(ICExternalSetting::getEntries).flatMap(Stream::of)
			.filter(e -> e.getName().equals(key)).map(ICSettingEntry::getValue)
			.findAny().orElse(null);
	}
	
	public static String readExternalSetting(String key, IProject project) { 
		return readExternalSetting(key, project, null); 
	}	
}
