/*
 * Copyright 2011-2017 Freescale
 * Copyright 2018,2021,2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 */

package com.nxp.sa.ui.common.epl;

import java.io.File;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;

import com.freescale.sa.util.Logger;

import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.debug.core.sourcelookup.MappingSourceContainer;
import org.eclipse.cdt.debug.internal.core.sourcelookup.MapEntrySourceContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.ISourceLocator;
import org.eclipse.debug.core.sourcelookup.AbstractSourceLookupDirector;
import org.eclipse.debug.core.sourcelookup.ISourceContainer;
import org.eclipse.debug.core.sourcelookup.ISourceLookupDirector;
import org.eclipse.swt.widgets.Display;

import com.freescale.sa.Messages;
import com.freescale.sa.SAPlugin;
//import com.freescale.sa.ui.common.sourcebrowser.SASourceLookupDirector;
import com.freescale.sa.util.LaunchConfigUtils;

@SuppressWarnings("restriction")
public class SASourceLookupUtils {

	private final static String foundMappingsContainerName = "Trace and Profile Mappings"; //$NON-NLS-1$
	/* States of SourceLookup window */
	public static final int NOT_DISPLAYED = 0x0;
	public static final int DISPLAYED = 0x1;
	private static int sourceLookupWindowDisplayed = NOT_DISPLAYED;
	public final static String PREF_DO_NOT_LOCATE_FILE = "PREF_DO_NOT_LOCATE_FILE"; //$NON-NLS-1$

	/**
	 * Return the state of SourceLookup Window (displayed or not displayed)
	 * 
	 * @param none
	 * @return state of SourceLookup Window
	 */
	public static int getSourceLookupWindowState() {
		return sourceLookupWindowDisplayed;
	}

	/**
	 * Set the state of SourceLookup Window (displayed or not displayed)
	 * 
	 * @param state
	 *            - state of SourceLookup Window
	 * @return none
	 */
	public static void setSourceLookupWindowState(int state) {
		sourceLookupWindowDisplayed = state;
	}

	/**
	 * Initialize a source lookup director from a ILaunchConfiguration. To be
	 * used for path mappings and source lookup.
	 * 
	 * @param config
	 *            the launch config
	 * @return a new source lookup director object
	 * @throws CoreException
	 */
	public static ISourceLookupDirector getSourceLookupDirector(
			ILaunchConfiguration config) throws CoreException {
		String srcLocMemento = null;
		String srcLocID = null;
		AbstractSourceLookupDirector director = null;

		if (config == null || !config.exists()) {
			String msg = MessageFormat.format(
					Messages.SourceNotFound_error_launchconfig,
					config != null ? config.getName() : "null"); //$NON-NLS-1$
			Status status = new Status(IStatus.ERROR,
					SAPlugin.getUniqueIdentifier(), SAPlugin.INTERNAL_ERROR,
					msg, null);
			throw new CoreException(status);
		}

		srcLocMemento = config
				.getAttribute(ILaunchConfiguration.ATTR_SOURCE_LOCATOR_MEMENTO,
						(String) null);
		srcLocID = config.getAttribute(
				ILaunchConfiguration.ATTR_SOURCE_LOCATOR_ID, (String) null);
		if (srcLocID == null) {
			srcLocID = config.getType().getSourceLocatorId();
		}

		ILaunchManager launchManager = DebugPlugin.getDefault()
				.getLaunchManager();
		ISourceLocator locator = launchManager.newSourceLocator(srcLocID);

		if (locator instanceof AbstractSourceLookupDirector) {
			director = (AbstractSourceLookupDirector) locator;
			if (srcLocMemento == null) {
				director.initializeDefaults(config);
			} else {
				director.initializeFromMemento(srcLocMemento, config);
			}
		} else {
			throw new CoreException(new Status(IStatus.ERROR,
					SAPlugin.getUniqueIdentifier(), SAPlugin.INTERNAL_ERROR,
					Messages.SourceNotFound_error_locator, null));
		}

		return director;
	}

	/**
	 * Initialize a SASourceLookupDirector from a results config file. To be
	 * used for path mappings and source lookup.
	 * 
	 * @param resultsConfigName
	 *            the results config file name
	 * @return a new SASourceLookupDirector object
	 * @throws CoreException
	 */
	public static SASourceLookupDirector getSourceLookupDirector(
			String resultsConfigName) throws CoreException {
		// String srcLocMemento = null;
		// String srcLocID = null;
		SASourceLookupDirector director = null;

		if (resultsConfigName == null
				|| !(new File(resultsConfigName).exists())) {
			String msg = MessageFormat.format(
					Messages.SourceNotFound_error_resultsconfig,
					resultsConfigName != null ? resultsConfigName : "null"); //$NON-NLS-1$
			Status status = new Status(IStatus.ERROR,
					SAPlugin.getUniqueIdentifier(), SAPlugin.INTERNAL_ERROR,
					msg, null);
			throw new CoreException(status);
		}

		director = new SASourceLookupDirector(resultsConfigName);

		return director;
	}

	/**
	 * Function which attempts to find a path mapping for a file path, using a
	 * source lookup director created for a given launch configuration object In
	 * case the mapping fails, prompts the user a dialog allowing him to browse
	 * for the file or edit the Source Path Mappings for the associated launch
	 * config resource. Updates the launch config resource if the Source locator
	 * attribute has changed.
	 * 
	 * @param config
	 *            the launch configuration object
	 * @param pathStr
	 *            the path of the file to be mapped
	 * @return the mapped path string, or null if it was not possible to find a
	 *         mapping
	 */
	public static IPath mapSource(final ILaunchConfiguration config,
			final String pathStr) {
		return mapSource(config, pathStr,
				new ArrayList<ISourceLookupDirector>());
	}

	/**
	 * Function which attempts to find a path mapping for a file path, using a
	 * source lookup director created for a given launch configuration object In
	 * case the mapping fails, prompts the user a dialog allowing him to browse
	 * for the file or edit the Source Path Mappings for the associated launch
	 * config resource. Updates the launch config resource if the Source locator
	 * attribute has changed.
	 * 
	 * @param config
	 *            the launch configuration object
	 * @param pathStr
	 *            the path of the file to be mapped
	 * @param lookupDirArray
	 *            used to get ISourceLookupDirector when the first search is
	 *            done
	 * @return the mapped path string, or null if it was not possible to find a
	 *         mapping
	 */
	public static IPath mapSource(final ILaunchConfiguration config,
			final String pathStr,
			ArrayList<ISourceLookupDirector> lookupDirArray) {

		if (pathStr.isEmpty()) {
			return null;
		}

		// Try to map the file internally.
		try {
			IPath mappedFilePath = getInternalMapping(config, pathStr,
					lookupDirArray);
			if (mappedFilePath != null) {
				return mappedFilePath;
			}
		} catch (CoreException e) {
			SAPlugin.getDefault().logError(e.getMessage());
			return null;
		}

		// If no internal map, ask the user.
		if (SAUILSPlugin.getDefault().getPreferenceStore()
				.getBoolean(PREF_DO_NOT_LOCATE_FILE)) {
			return null;
		}
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				setSourceLookupWindowState(DISPLAYED);
				SASourceLookupDialog dialog = new SASourceLookupDialog(Display
						.getCurrent().getActiveShell(), pathStr, config);
				dialog.open();
				SAUILSPlugin
						.getDefault()
						.getPreferenceStore()
						.setValue(
								PREF_DO_NOT_LOCATE_FILE,
								dialog.getDoNotPromptAgain());
				setSourceLookupWindowState(NOT_DISPLAYED);
			}
		});
		try {
			return getInternalMapping(config, pathStr, lookupDirArray);
		} catch (CoreException e) {
			SAPlugin.getDefault().logError(e.getMessage());
			return null;
		}
	}

	/**
	 * Helper method used to get a mapped path for a filename.
	 * 
	 * @param config
	 *            the launch configuration
	 * @param fileName
	 *            the file name to be mapped
	 * @param lookupDirArray
	 *            used to get ISourceLookupDirector when the first search is
	 *            done
	 * 
	 * @return the mapped file path, or null if it was not possible to find a
	 *         mapping
	 * @throws CoreException
	 */

	/**
	 * 
	 */
	private static IPath getInternalMapping(ILaunchConfiguration config,
			String fileName, ArrayList<ISourceLookupDirector> lookupDirArray)
			throws CoreException {

		IPath mappedFilePath = null;
		ISourceLookupDirector localDirLookup = null;
		if (config == null || fileName == null) {
			SAPlugin.getDefault().logInfo("[getInternalMapping] Null argument"); //$NON-NLS-1$
			return mappedFilePath;
		}

		// Get the object for the mapped file.
		if (lookupDirArray.isEmpty()) {
			localDirLookup = getSourceLookupDirector(config);
			lookupDirArray.add(localDirLookup);
		} else {
			// use only first element
			localDirLookup = lookupDirArray.get(0);
		}

		Object mappedObject = localDirLookup.getSourceElement(fileName);

		if (mappedObject instanceof ICElement) {
			// Local File Path mapping that was stored with a launch config
			// that is initially saved as a shared file.
			mappedFilePath = ((ICElement) mappedObject).getPath();
		} else if (mappedObject instanceof IResource) {
			// Mapped to default project path.
			mappedFilePath = ((IResource) mappedObject).getLocation();
		} else if (mappedObject instanceof IStorage) {
			// Local File Path mapping that was stored with a launch config
			// that is initially saved as a local file.
			mappedFilePath = ((IStorage) mappedObject).getFullPath();
		}

		return mappedFilePath;
	}

	/**
	 * Helper method used to get a mapped path for a filename.
	 * 
	 * @param config
	 *            the launch configuration
	 * @param fileName
	 *            the file name to be mapped
	 * @param lookupDirArray
	 *            used to get SASourceLookupDirector when the first search is
	 *            done
	 * 
	 * @return the mapped file path, or null if it was not possible to find a
	 *         mapping
	 * @throws CoreException
	 */
	private static IPath getInternalMapping(String resultsConfigName,
			String fileName, ArrayList<SASourceLookupDirector> lookupDirArray)
			throws CoreException {

		IPath mappedFilePath = null;
		SASourceLookupDirector localDirLookup = null;
		if (resultsConfigName == null || fileName == null) {
			SAPlugin.getDefault().logInfo("[getInternalMapping] Null argument"); //$NON-NLS-1$
			return mappedFilePath;
		}

		// Get the object for the mapped file.
		if (lookupDirArray.isEmpty()) {
			localDirLookup = getSourceLookupDirector(resultsConfigName);
			lookupDirArray.add(localDirLookup);
		} else {
			// use only first element
			localDirLookup = lookupDirArray.get(0);
		}

		if (localDirLookup != null) {
			String path = localDirLookup.getSourceElement(fileName);

			if (path != null) {
				mappedFilePath = new Path(path);
			}
		}

		return mappedFilePath;
	}

	/**
	 * Function which attempts to find a path mapping for a file path, using a
	 * SASourceLookupDirector object In case the mapping fails, prompts the user
	 * a dialog allowing him to browse for the file or edit the Source Path
	 * Mappings for the associated SASourceLookupDirector resource. Updates the
	 * SASourceLookupDirector resource if the Source locator attribute has
	 * changed.
	 * 
	 * @param resultsConfigFileName
	 *            the results config file name
	 * @param pathStr
	 *            the path of the file to be mapped
	 * @return the mapped path string, or null if it was not possible to find a
	 *         mapping
	 */
	public static IPath mapSource(String resultsConfigFileName,
			final String pathStr) {
		return mapSource(pathStr, new ArrayList<SASourceLookupDirector>(),
				resultsConfigFileName);
	}

	/**
	 * Function which attempts to find a path mapping for a file path, using a
	 * SASourceLookupDirector In case the mapping fails, prompts the user a
	 * dialog allowing him to browse for the file or edit the Source Path
	 * Mappings for the associated resultsConfig resource. Updates the
	 * resultsConfig resource if the Source locator attribute has changed.
	 * 
	 * @param pathStr
	 *            the path of the file to be mapped
	 * @param lookupDirArray
	 *            used to get ISourceLookupDirector when the first search is
	 *            done
	 * @param resultsConfigFileName
	 *            the results config file name
	 * 
	 * @return the mapped path string, or null if it was not possible to find a
	 *         mapping
	 */
	public static IPath mapSource(final String pathStr,
			ArrayList<SASourceLookupDirector> lookupDirArray,
			final String resultsConfigFileName) {

		if (pathStr.isEmpty()) {
			return null;
		}

		// Try to map the file internally.

		IPath mappedFilePath = null;
		Path mapPath = new Path(pathStr);
		if ((resultsConfigFileName != null) && (new File(resultsConfigFileName).exists())) {
			try {
				mappedFilePath = getInternalMapping(resultsConfigFileName, pathStr, lookupDirArray);
			} catch (CoreException e) {
				SAPlugin.getDefault().logError(e.getMessage());
				return null;
			}
		}
		
		try {
			if (mappedFilePath != null) {
				return mappedFilePath;
			} else if (mapPath.toFile().exists()) {
				//If the file exists on system 
				return mapPath;
			}
		} catch(Exception e) {
			//Path exceptions can be thrown
			SAPlugin.getDefault().logInfo(e.getMessage());
		}

		// If no internal map, ask the user.
		if (SAUILSPlugin.getDefault().getPreferenceStore()
				.getBoolean(PREF_DO_NOT_LOCATE_FILE)) {
			return null;
		}

		final ArrayList<IPath> newPaths = new ArrayList<IPath>(); // workaround to get data from the runnable
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				setSourceLookupWindowState(DISPLAYED);
				SASourceLookupDialog dialog = new SASourceLookupDialog(Display
						.getCurrent().getActiveShell(), pathStr,
						resultsConfigFileName);
				dialog.open();
				IPath newPath = dialog.getNewPath();
				if (newPath != null) {
					newPaths.add(newPath);
					SAUILSPlugin
					.getDefault()
					.getPreferenceStore()
					.setValue(
							PREF_DO_NOT_LOCATE_FILE,
							dialog.getDoNotPromptAgain());
			setSourceLookupWindowState(NOT_DISPLAYED);
				}
			}
		});

		if (newPaths.size() > 0) {
			return newPaths.get(0);
		}
		return null;
	}

	private static void addSourceMappingToDirector(IPath missingPath,
			IPath newSourcePath, ISourceLookupDirector director)
			throws CoreException {

		if ((missingPath == null) || (newSourcePath == null)
				|| (director == null)) {
			SAPlugin.getDefault().logInfo("[addSourceMappingToDirector] Null argument"); //$NON-NLS-1$
			return;
		}

		ArrayList<ISourceContainer> containerList = new ArrayList<ISourceContainer>(
				Arrays.asList(director.getSourceContainers()));

		boolean hasFoundMappings = false;

		MappingSourceContainer foundMappings = null;

		for (Iterator<ISourceContainer> iter = containerList.iterator(); iter
				.hasNext() && !hasFoundMappings;) {
			ISourceContainer container = (ISourceContainer) iter.next();

			if (container instanceof MappingSourceContainer) {
				hasFoundMappings = container.getName().equals(
						foundMappingsContainerName);
				if (hasFoundMappings) {
					foundMappings = (MappingSourceContainer) container;
				}
			}
		}

		if (!hasFoundMappings) {
			foundMappings = new MappingSourceContainer(
					foundMappingsContainerName);
			foundMappings.init(director);
			containerList.add(foundMappings);
		}

		
		/* We use reflection to determine the signature of the MapEntrySourceContainer constructor,
		 * which is MapEntrySourceContainer(IPath,IPath) on Indigo and MapEntrySourceContainer(String, IPath) on neon 
		 */
		  java.lang.reflect.Method method;
	        MapEntrySourceContainer obj = new MapEntrySourceContainer();
	        try {
	            obj.setLocalPath(newSourcePath);
	            method = obj.getClass().getMethod("setBackendPath", IPath.class); //$NON-NLS-1$
	            method.invoke(obj, missingPath);
	          } catch (NoSuchMethodException e) {  
	                try{
	                method = obj.getClass().getMethod("setBackendPath", String.class); //$NON-NLS-1$
	                method.invoke(obj, missingPath.toString());
	                } catch(Exception e2){
	                    SAPlugin.getDefault().logError(e2.getMessage());
	                    String msg = MessageFormat.format(
	                           Messages.SourceNotFound_error_locator, "null"); //$NON-NLS-1$
	                         Status status = new Status(IStatus.ERROR,
	                         SAPlugin.getUniqueIdentifier(), SAPlugin.INTERNAL_ERROR,
	                        msg, null);
	                    throw new CoreException(status);
	                }
	          } catch (Exception e) { 
	                SAPlugin.getDefault().logError(e.getMessage());
	                String msg = MessageFormat.format(
	                        Messages.SourceNotFound_error_locator, "null"); //$NON-NLS-1$
	                        Status status = new Status(IStatus.ERROR,
	                               SAPlugin.getUniqueIdentifier(), SAPlugin.INTERNAL_ERROR,
	                               msg, null);
	                throw new CoreException(status);
	                }    
	        foundMappings.addMapEntry(obj);
		director.setSourceContainers((ISourceContainer[]) containerList
				.toArray(new ISourceContainer[containerList.size()]));
	}

	/**
	 * Helper function that adds a path mapping to the launch configuration
	 * source locator memento, then saves it. Initializes and uses a source
	 * lookup director in the process.
	 * 
	 * @param config
	 *            the launch configuration object to be updated
	 * @param missingPath
	 *            the source file path
	 * @param newSourcePath
	 *            the mapped file path
	 * @throws CoreException
	 */
	public static void addSourceMappingToLaunch(ILaunchConfiguration config,
			IPath missingPath, IPath newSourcePath) throws CoreException {
		ISourceLookupDirector director = getSourceLookupDirector(config);

		addSourceMappingToDirector(missingPath, newSourcePath, director);
		LaunchConfigUtils.updateLaunchConfigurationFromDirector(config,
				director);
	}

	/**
	 * Helper function that adds a path mapping to the results config file, then
	 * saves it. Initializes and uses a SA source lookup director in the
	 * process.
	 * 
	 * @param resultsConfig
	 *            the results config file to be updated
	 * @param missingPath
	 *            the source file path
	 * @param newSourcePath
	 *            the mapped file path
	 * @throws CoreException
	 */
	public static void addSourceMappingToResultsConfig(String resultsConfig,
			IPath missingPath, IPath newSourcePath) throws CoreException {
		SASourceLookupDirector director = getSourceLookupDirector(resultsConfig);
		if (director != null) {
			director.setMappedFileName(missingPath.toString(),
					newSourcePath.toOSString());
		}
	}
}
