/*
 * Copyright: (c) 2004-2010 Mayo Foundation for Medical Education and
 * Research (MFMER). All rights reserved. MAYO, MAYO CLINIC, and the
 * triple-shield Mayo logo are trademarks and service marks of MFMER.
 *
 * Except as contained in the copyright notice above, or as used to identify
 * MFMER as the author of this software, the trade names, trademarks, service
 * marks, or product names of the copyright holder shall not be used in
 * advertising, promotion or otherwise in connection with this software without
 * prior written authorization of the copyright holder.
 *
 * Licensed under the Eclipse Public License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 		http://www.eclipse.org/legal/epl-v10.html
 *
 * Copyright 2019 NXP
 *
 * Contributors:
 *     Mayo Foundation for Medical Education and Research (MFMER) - Initial Release
 *     Libor Ukropec (Freescale) - code customized for Processor Expert
 *     Lukas Tyc (NXP) - code customized for MCUXpresso Config Tools
 */
package com.nxp.swtools.derivative.swt;
//package org.LexGrid.LexBIG.gui.displayResults;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Display;

/**
 * This class makes it easier to display formatted text in a SWT StyledText. You
 * can mark up your text with tags like <b> </b> - basic html tags - and it will
 * create the necessary style ranges for the StyledText object.
 *
 * Supported Tags: (pre is not supported - it is just here so you can read this
 * from the javadocs)
 *
 * <pre>
 * &lt;b&gt; - bold
 * &lt;i&gt; - italic
 * &lt;u&gt; - underline
 * &lt;so&gt; - strikeout
 * &lt;red&gt; - red font color
 * &lt;green&gt; - green font color
 * &lt;blue&gt; - blue font color
 * </pre>
 *
 * @author <A HREF="mailto:armbrust.daniel@mayo.edu">Dan Armbrust</A>
 * @version subversion $Revision: $ checked in on $Date: $
 */
public class TextContent {

	protected String content;

	/**
	 * @return Returns the content.
	 */
	public String getContent() {
		return content;
	}

	/**
	 * @param content
	 *            The content to set.
	 */
	public void setContent(String content) {
		this.content = content;
	}

	/**
     *
     */
	public TextContent(Display display) {
		// these don't have to be disposed because they are system colors.
		red = display.getSystemColor(SWT.COLOR_RED);
		green = display.getSystemColor(SWT.COLOR_GREEN);
		blue = display.getSystemColor(SWT.COLOR_BLUE);
		black = display.getSystemColor(SWT.COLOR_BLACK);
		white = display.getSystemColor(SWT.COLOR_WHITE);
	}

	/**
	 * Constructor
	 */
	public TextContent() {
		// nothing to do
	}

	/** counter for nesting elements */
	protected int nesting = 0;

	protected List<StyleRange> styleRanges = new ArrayList<>();

	/** current text color */
	protected Color currentColor;
	/** the system colors */
	protected Color anchorColor;
	/** custom font */
	protected Font customFont = null;
	/** custom value for rise - upper and down script */
	protected int customFontRise = 0;

	protected boolean inItalics = false, inBold = false, inUnderline = false,
			inStrikeout = false;
	/** are we in a href */
	protected boolean inAnchor;
	/** Flag to keep all characters, we are in <code>&ltcode&gt</code> or in <code>&ltpre&gt</code>*/
	protected boolean inRawFormat = false;
	boolean inRed = false, inBlue = false, inGreen = false;

	protected Color black, white, red, green, blue;

	protected StyleRange curRange;
	protected int rangeStyle = SWT.NORMAL;

	protected void startRange(int posn) {
		if (curRange != null) {
			endRange(posn);
		}
		curRange = new StyleRange();
		curRange.start = posn;
	}

	protected void endRange(int posn) {
		if (curRange != null) {
			curRange.length = posn - curRange.start;
			//System.out.println("state: " +  curRange.length + ": I=" + inItalics + ", B=" + inBold + ", U=" + inUnderline + ", S=" + inStrikeout + ":" + inRed + "x" + inGreen + "x" + inBlue);
			if (inItalics) {
				rangeStyle |= SWT.ITALIC;
			}
			else {
				rangeStyle &= ~SWT.ITALIC;
			}
			if (inBold) {
				rangeStyle |= SWT.BOLD;
			}
			else {
				rangeStyle &= ~SWT.BOLD;
			}
			curRange.fontStyle = rangeStyle;
			curRange.underline = inUnderline;
			curRange.strikeout = inStrikeout;
			curRange.foreground = currentColor;
			if (inAnchor) {
				curRange.foreground = anchorColor;
			}
			if (curRange.background == null) {
				//curRange.background = white; //defaultBackground;
			}
			// custom font can be set
			curRange.font = customFont;
			curRange.rise = customFontRise;
			//
			if (curRange.length > 0) {
				styleRanges.add(curRange);
				//System.out.println("added: " + Integer.toBinaryString(curRange.fontStyle) + "-" + curRange);
			}
		}
		curRange = null;
		if (nesting > 0) {
			startRange(posn);
		}
	}

	public String toPlainText() {
		int posn = 0;
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < content.length();) {
			char c = content.charAt(i);
			if (c == '<') {
				if ("<>".equals(content.substring(i, i + 2))) { //$NON-NLS-1$
					sb.append('<');
					i += 2;
					nesting++;
				} else if ("<i>".equals(content.substring(i, i + 3))) { //$NON-NLS-1$
					startRange(posn);
					inItalics = true;
					i += 3;
					nesting++;
				} else if ("<b>".equals(content.substring(i, i + 3))) { //$NON-NLS-1$
					startRange(posn);
					inBold = true;
					i += 3;
					nesting++;
				} else if ("<u>".equals(content.substring(i, i + 3))) { //$NON-NLS-1$
					startRange(posn);
					inUnderline = true;
					i += 3;
					nesting++;
				} else if ("<so>".equals(content.substring(i, i + 4))) { //$NON-NLS-1$
					startRange(posn);
					inStrikeout = true;
					i += 4;
					nesting++;
				} else if ("<red>".equals(content.substring(i, i + 5))) { //$NON-NLS-1$
					startRange(posn);
					inRed = true;
					i += 5;
					nesting++;
				} else if ("<green>".equals(content.substring(i, i + 7))) { //$NON-NLS-1$
					startRange(posn);
					inGreen = true;
					i += 7;
					nesting++;
				} else if ("<blue>".equals(content.substring(i, i + 6))) { //$NON-NLS-1$
					startRange(posn);
					inBlue = true;
					i += 6;
					nesting++;
				} else if ("</i>".equals(content.substring(i, i + 4))) { //$NON-NLS-1$
					endRange(posn);
					inItalics = false;
					i += 4;
					nesting--;
				} else if ("</b>".equals(content.substring(i, i + 4))) { //$NON-NLS-1$
					endRange(posn);
					inBold = false;
					i += 4;
					nesting--;
				} else if ("</u>".equals(content.substring(i, i + 4))) { //$NON-NLS-1$
					endRange(posn);
					inUnderline = false;
					i += 4;
					nesting--;
				} else if ("</so>".equals(content.substring(i, i + 5))) { //$NON-NLS-1$
					endRange(posn);
					inStrikeout = false;
					i += 5;
					nesting--;
				} else if ("</red>".equals(content.substring(i, i + 6))) { //$NON-NLS-1$
					endRange(posn);
					inRed = false;
					i += 6;
					nesting--;
				} else if ("</green>".equals(content.substring(i, i + 8))) { //$NON-NLS-1$
					endRange(posn);
					inGreen = false;
					i += 8;
					nesting--;
				} else if ("</blue>".equals(content.substring(i, i + 7))) { //$NON-NLS-1$
					endRange(posn);
					inBlue = false;
					i += 7;
					nesting--;
				} else {
					while (true) {
						if ((c = content.charAt(++i)) == '>') {
							break;
						}
					}
				}
			} else {
				sb.append(c);
				i++;
				posn++;
			}
		}
		endRange(posn);
		return sb.toString();
	}

	public StyleRange[] getStyleRanges() {
		return styleRanges.toArray(new StyleRange[styleRanges
				.size()]);
	}

	/**
	 * @return the custom font
	 */
	public Font getCustomFont() {
		return customFont;
	}

	/**
	 * @return the current color
	 */
	public Color getCurrentColor() {
		return currentColor;
	}

	/**
	 * Sets the current color
	 * @param currentColor
	 */
	public void setCurrentColor(Color currentColor) {
		this.currentColor = currentColor;
	}

	/**
	 * @return the current anchor color
	 */
	public Color getAnchorColor() {
		return anchorColor;
	}

	/**
	 * Sets the anchor color
	 * @param anchorColor
	 */
	public void setAnchorColor(Color anchorColor) {
		this.anchorColor = anchorColor;
	}

	/**
	 * Sets the custom font. Caller is responsible for font disposal.
	 * @param customFont the font or <code>null</code> for default font
	 */
	public void setCustomFont(Font customFont) {
		this.customFont = customFont;
	}
}
