/*******************************************************************************
 *  Copyright (c) 2002, 2012 IBM Corporation and others.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *  IBM Rational Software - Initial API and implementation
 *******************************************************************************/
package com.freescale.s32ds.cross.cdt.utils.epl;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.ErrorParserManager;
import org.eclipse.cdt.core.resources.IConsole;
import org.eclipse.cdt.internal.core.BuildRunnerHelper;
import org.eclipse.cdt.managedbuilder.buildmodel.BuildDescriptionManager;
import org.eclipse.cdt.managedbuilder.buildmodel.IBuildDescription;
import org.eclipse.cdt.managedbuilder.buildmodel.IBuildIOType;
import org.eclipse.cdt.managedbuilder.buildmodel.IBuildResource;
import org.eclipse.cdt.managedbuilder.buildmodel.IBuildStep;
import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.IManagedBuildInfo;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.cdt.managedbuilder.core.ManagedBuilderCorePlugin;
import org.eclipse.cdt.managedbuilder.internal.buildmodel.StepBuilder;
import org.eclipse.cdt.managedbuilder.internal.core.GeneratedMakefileBuilder;
import org.eclipse.cdt.managedbuilder.internal.core.ManagedMakeMessages;
import org.eclipse.cdt.utils.PathUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;

@SuppressWarnings("restriction")
public class GeneratedMakefileBuilder2EPL extends GeneratedMakefileBuilder {

	private List<IFile> outputBinaryFiles;

	public List<IFile> getOutputBinaryFiles() {
		return outputBinaryFiles;
	}

	private static final int PROGRESS_MONITOR_SCALE = 100;
	private static final int TICKS_STREAM_PROGRESS_MONITOR = 1 * PROGRESS_MONITOR_SCALE;
	private static final int TICKS_DELETE_MARKERS = 1 * PROGRESS_MONITOR_SCALE;

	private IConsole console;

	/*
	 * (non-Javadoc)
	 *
	 * @return
	 */
	private Vector<IStatus> getGenerationProblems() {
		if (generationProblems == null) {
			generationProblems = new Vector<IStatus>();
		}
		return generationProblems;
	}

	private Map<IProject, List<IFile>> arrangeFilesByProject(List<IFile> files) {
		Map<IProject, List<IFile>> projectMap = new HashMap<IProject, List<IFile>>();
		for (IFile file : files) {
			IProject project = file.getProject();
			List<IFile> filesInProject = projectMap.get(project);
			if (filesInProject == null) {
				filesInProject = new ArrayList<IFile>();
				projectMap.put(project, filesInProject);
			}
			filesInProject.add(file);
		}
		return projectMap;
	}

	/**
	 * Called to invoke the MBS Internal Builder for building the given resources
	 *
	 * @param files   - list of files to build.
	 * @param monitor - progress monitor to report progress to user.
	 * @return status of the operation. Can be {@link Status#OK_STATUS} or
	 *         {@link Status#CANCEL_STATUS}.
	 */
	public IStatus invokeInternalBuilder(List<IFile> files, IProgressMonitor monitor) {
		// Make sure there's a monitor to cancel the build
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}

		try {
			Map<IProject, List<IFile>> projectMap = arrangeFilesByProject(files);
			monitor.beginTask("", projectMap.size() * PROGRESS_MONITOR_SCALE); //$NON-NLS-1$

			outputBinaryFiles = new ArrayList<>();
			for (List<IFile> filesInProject : projectMap.values()) {
				IProject project = filesInProject.get(0).getProject();
				setCurrentProject(project);
				monitor.subTask(ManagedMakeMessages.getFormattedString("GeneratedMakefileBuilder.buildingProject", //$NON-NLS-1$
						project.getName()));
				invokeInternalBuilderForOneProject(filesInProject, new SubProgressMonitor(monitor,
						1 * PROGRESS_MONITOR_SCALE, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
			}
		} finally {
			if (monitor.isCanceled()) {
				return Status.CANCEL_STATUS;
			}
			monitor.done();
		}
		return Status.OK_STATUS;
	}

	private void invokeInternalBuilderForOneProject(List<IFile> files, IProgressMonitor monitor) {
		IProject project = files.get(0).getProject();
		BuildRunnerHelper buildRunnerHelper = new BuildRunnerHelper(project);

		try {
			monitor.beginTask(ManagedMakeMessages.getFormattedString("GeneratedMakefileBuilder.buildingProject", //$NON-NLS-1$
					project.getName()) + ':', TICKS_STREAM_PROGRESS_MONITOR + files.size() * PROGRESS_MONITOR_SCALE);

			// Get a build console for the project
			console = CCorePlugin.getDefault().getConsole();
			console.start(project);

			IManagedBuildInfo buildInfo = ManagedBuildManager.getBuildInfo(project);
			IConfiguration configuration = buildInfo.getDefaultConfiguration();

			String cfgName = configuration.getName();
			String toolchainName = configuration.getToolChain().getName();
			boolean isSupported = configuration.isSupported();

			IBuildDescription des = BuildDescriptionManager.createBuildDescription(configuration, null, 0);

			String[] errorParsers = configuration.getErrorParserList();
			ErrorParserManager epm = new ErrorParserManager(project, des.getDefaultBuildDirLocationURI(), this,
					errorParsers);

			buildRunnerHelper.prepareStreams(epm, null, console,
					new SubProgressMonitor(monitor, TICKS_STREAM_PROGRESS_MONITOR));
			OutputStream stdout = buildRunnerHelper.getOutputStream();
			OutputStream stderr = buildRunnerHelper.getErrorStream();

			buildRunnerHelper.greeting(
					ManagedMakeMessages.getResourceString("GeneratedMakefileBuilder.buildingSelectedFiles"), cfgName, //$NON-NLS-1$
					toolchainName, isSupported);
			buildRunnerHelper.printLine(
					ManagedMakeMessages.getResourceString("ManagedMakeBuilder.message.internal.builder.header.note")); //$NON-NLS-1$

			// Build artifacts for each file in the project
			for (IFile file : files) {
				if (monitor.isCanceled()) {
					break;
				}
				String filePath = file.getProjectRelativePath().toString();

				try {
					IBuildResource buildResource = des.getBuildResource(file);

					Set<IBuildStep> dependentSteps = new HashSet<IBuildStep>();
					for (IBuildIOType btype : buildResource.getDependentIOTypes()) {
						if (btype != null && btype.getStep() != null)
							dependentSteps.add(btype.getStep());
					}

					SubProgressMonitor monitor2 = new SubProgressMonitor(monitor, 1 * PROGRESS_MONITOR_SCALE,
							SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK);
					try {
						monitor2.beginTask("", TICKS_DELETE_MARKERS + dependentSteps.size() * PROGRESS_MONITOR_SCALE); //$NON-NLS-1$

						// Remove problem markers for the file
						monitor2.subTask(ManagedMakeMessages
								.getFormattedString("GeneratedMakefileBuilder.removingResourceMarkers", filePath)); //$NON-NLS-1$
						buildRunnerHelper.removeOldMarkers(file, new SubProgressMonitor(monitor2, TICKS_DELETE_MARKERS,
								SubProgressMonitor.SUPPRESS_SUBTASK_LABEL));

						// Build dependent steps
						for (IBuildStep step : dependentSteps) {
							if (monitor2.isCanceled()) {
								break;
							}

							monitor2.subTask(filePath);
							StepBuilder stepBuilder = new StepBuilder(step, null);
							stepBuilder.build(stdout, stderr, new SubProgressMonitor(monitor2,
									1 * PROGRESS_MONITOR_SCALE, SubProgressMonitor.SUPPRESS_SUBTASK_LABEL));

							monitor2.subTask(ManagedMakeMessages
									.getFormattedString("GeneratedMakefileBuilder.refreshingArtifacts", filePath)); //$NON-NLS-1$
							IBuildIOType[] outputIOTypes = step.getOutputIOTypes();
							for (IBuildIOType type : outputIOTypes) {
								for (IBuildResource outResource : type.getResources()) {
									IPath location = outResource.getLocation();
									IFile outFile = project.getFile(location);
									// Refresh the output resource without allowing the user to cancel.
									outFile.refreshLocal(IResource.DEPTH_INFINITE, null);
									IFile[] findFilesForLocation = PathUtil.getWorkspaceRoot()
											.findFilesForLocation(location);
									outputBinaryFiles.add(findFilesForLocation[0]);
								}
							}
						}
					} finally {
						monitor2.done();
					}

				} catch (Exception e) {
					ManagedBuilderCorePlugin.log(new CoreException(
							new Status(IStatus.ERROR, ManagedBuilderCorePlugin.PLUGIN_ID, "CDT Build Error", e))); //$NON-NLS-1$
				}

			}
			buildRunnerHelper.close();
			buildRunnerHelper.goodbye();

		} catch (Exception e) {
			ManagedBuilderCorePlugin.log(new CoreException(
					new Status(IStatus.ERROR, ManagedBuilderCorePlugin.PLUGIN_ID, "CDT Build Error", e))); //$NON-NLS-1$
			forgetLastBuiltState();
		} finally {
			getGenerationProblems().clear();
			try {
				buildRunnerHelper.close();
			} catch (IOException e) {
				ManagedBuilderCorePlugin.log(new CoreException(
						new Status(IStatus.ERROR, ManagedBuilderCorePlugin.PLUGIN_ID, "CDT Build Error", e))); //$NON-NLS-1$
			}
			monitor.done();
		}
	}
}
