/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package com.freescale.system.browser.epl;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.ISafeRunnable;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import com.freescale.s32ds.ui.S32dsUtils;
import com.freescale.system.browser2.SystemBrowserActivator2;
import com.freescale.system.browser.epl.TabbedSheetPageEpl;
import com.freescale.system.browser2.OSAwarenessData2;
import com.freescale.system.browser.epl.ISectionEpl;

/**
 * A tab is composed of one or more sections. This class effectively represents
 * the instantiation of a tab descriptor.
 */
public class TabEpl {

	/**
	 * The tab descriptor we are an instantiation of
	 */
	private TabDescriptorEpl fDescriptor;
	
	/**
	 * The collection of sections that are associated with us (via XML). Whether
	 * or not any particular section in this collection should actually be shown
	 * to the user is determined at runtime and can be dependent on the active
	 * context.
	 */
	private List<ISectionEpl> fSections;
	
	/**
	 * Listener we use to keep track of which control within the tab has focus
	 */
	private Listener fFocusListener;

	
	/**
	 * The control to last have focus in this tab. When the tab is sent to the
	 * background and then brought back, we attempt to restore focus to this
	 * control.
	 */
	private Control fLastControlWithFocus;
	
	/**
	 * The top level control (canvas, really) for the tabbed page.
	 */
	private Composite fPageComposite;

	/**
	 * Constructor
	 * 
	 * @param descriptor
	 *            the tab description we will be an instantiation of
	 */
	protected TabEpl(TabDescriptorEpl descriptor) {
		assert(descriptor != null);
		fDescriptor = descriptor;
		fSections = new ArrayList<>();
		// We set up a global focus listener so we can keep track of the 
		// control that last had focus in the tab. This allows us to 
		// restore focus to that control whenever the tab is shown again. We 
		// activate the listener when we're about to be shown and we deactivate 
		// it when we're about to be hidden. Global listeners have the ability
		// to impact performance across the entire workench; we'll get called
		// when ANY control in the workbench gets focus. Thus, we want to ensure
		// we don't have listeners plugged into the system any more than we need to. 
		fFocusListener = new Listener() {
			@Override
			public void handleEvent(Event event) {
				if (event.widget instanceof Control && fPageComposite != null) {
					Control control = (Control)event.widget;
					while (control != null && control != fPageComposite) {
						control = control.getParent();
					}
					if (control == fPageComposite) {
						fLastControlWithFocus = (Control)event.widget;
					}
				}
			}
		};
	}
	
	/**
	 * Constructor
	 * 
	 * @param descriptor
	 *            the tab description we will be an instantiation of
	 */
	public TabEpl(final TabDescriptorEpl tasksTabDescriptor, final List<ISectionEpl> sections) {
		this(tasksTabDescriptor);
		fSections = sections;
	}

	/**
	 * Accessor method. See fDescriptor 
	 */
	public TabDescriptorEpl getDescriptor() {
		return fDescriptor;
	}

	/**
	 * Accessor method. See fSections 
	 */
	public List<ISectionEpl> getSections() {
		return fSections;		
	}

	/**
	 * Creates the tab's GUI content.
	 * 
	 * @param parent the tab's parent composite
	 * @param page the tabbed sheet page (instance of IPageBookViewPage)
	 */
	public void createControls(Composite parent,
			final TabbedSheetPageEpl page) {

		// Create a top level canvas within our parent for us to put our content
		fPageComposite = page.getWidgetFactory().createComposite(parent, SWT.NO_FOCUS);
		fPageComposite.setLayout(S32dsUtils.makeTight(new GridLayout()));

		// Tell each of our sections to create its GUI content
		for (final ISectionEpl section : fSections) {
			// Create a top level canvas for each section			
			final Composite sectionComposite = page.getWidgetFactory().createComposite(fPageComposite, SWT.NO_FOCUS);
			sectionComposite.setLayout(new FillLayout());

			// A section's XML can ask that it use up all extra vertical space
			GridData data = new GridData(GridData.FILL, GridData.FILL, true, section.shouldUseExtraSpace()); 
			data.heightHint = section.getMinimumHeight();
			sectionComposite.setLayoutData(data);

			// Tell the section to create its GUI content. As this is untrusted client
			// code, use a safe runner. 
			ISafeRunnable runnable = new ISafeRunnable() {
				@Override
				public void run() throws Exception {
					section.createControls(sectionComposite, page);
				}
				@Override
				public void handleException(Throwable exc) {
					SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
				}
			};

			SafeRunner.run(runnable);
		}
	}

	/**
	 * Dispose of tab's GUI content
	 */
	public void dispose() {
		// Tell each section to dispose its GUI content. As this is untrusted client
		// code, use a safe runner. 
		for (final ISectionEpl section : fSections) {
			ISafeRunnable runnable = new ISafeRunnable() {
				@Override
				public void run() throws Exception {
					section.dispose();
				}
				@Override
				public void handleException(Throwable exc) {
					SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
				}
			};

			SafeRunner.run(runnable);
		}

		// Ensure we've unregistered our global focus filter. This is also done 
		// in done in aboutToBeHidden. We do it here in case we somehow get 
		// disposed without first being told we're about to be hidden. Leaving 
		// a pointless global focus listener in place is bad. See fFocusListener 
		// for explanation.
		Display display = Display.getCurrent();
		if (display != null) {
			display.removeFilter(SWT.FocusIn, fFocusListener);			
		}
	}

	/**
	 * We get called just prior to being displayed to the user, likely because
	 * we have been made the active tab. We in turn forward the notification to
	 * every section in the tab
	 */
	public void aboutToBeShown() {
		// Forward the notification to each of our sections. As that involves 
		// invoking untrusted client code, use a safe runner. 
		for (final ISectionEpl section : fSections) {
			ISafeRunnable runnable = new ISafeRunnable() {

				@Override
				public void run() throws Exception {
					section.aboutToBeShown();
				}

				@Override
				public void handleException(Throwable exc) {
					SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
				}
			};
			
			SafeRunner.run(runnable);
		}

		// register our global focus filter
		Display display = Display.getCurrent();
		if (display != null) {
			display.removeFilter(SWT.FocusIn, fFocusListener);	// trick to prevent listener build up			
			display.addFilter(SWT.FocusIn, fFocusListener);
		}
	}

	/**
	 * We get called just prior to being hidden from the user, likely because
	 * another tab has been made the active one. We forward this notification
	 * to every section in the tab so that each will stop reacting to context
	 * state changes. Imagine ten tabs, all of which expose information that can
	 * change on each suspend event. Since only one tab is showing, it would be
	 * very inefficient to have the other nine update their content every time
	 * the user does, say, a step operation.
	 */
	public void aboutToBeHidden() {
		// Forward the notification to each of our sections. As that involves 
		// invoking untrusted client code, use a safe runner. 
		for (final ISectionEpl section : fSections) {
			ISafeRunnable runnable = new ISafeRunnable() {

				@Override
				public void run() throws Exception {
					section.aboutToBeHidden();
				}

				@Override
				public void handleException(Throwable exc) {
					SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
				}
			};

			SafeRunner.run(runnable);
		}
		
		// Unregister our global focus filter. The newly selected tab will 
		// register its own		
		Display display = Display.getCurrent();
		if (display != null) {
			display.removeFilter(SWT.FocusIn, fFocusListener);			
		}
	}

	/**
	 * Give the sections in the tab a new context.
	 * 
	 * @param input
	 *            the new active debug context (selection in the Debug view)
	 */
	public void setTabInput(final OSAwarenessData2 input) {		
		if (input == null)
			return;
		SystemBrowserActivator2.log(1, "setTabInput =>"+input.toString()); //$NON-NLS-1$   
		// Forward the notification to each of our sections. As that involves 
		// invoking untrusted client code, use a safe runner. 
		for (final ISectionEpl section : fSections) {
			ISafeRunnable runnable = new ISafeRunnable() {
				@Override
				public void run() throws Exception {
					section.setSectionInput(input);
				}

				@Override
				public void handleException(Throwable exc) {
					SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
				}
			};

			SafeRunner.run(runnable);
		}
	}

	/**
	 * Reveals whether the controls have been created or not.
	 *
	 * @return <code>true</code> if controls have been created.
	 */
	public boolean controlsHaveBeenCreated() {
		return fPageComposite != null;
	}

	/**
	 * Refresh the tab content
	 */
	public void refresh() {
		// Forward the notification to each of our sections. As that involves 
		// invoking untrusted client code, use a safe runner. 
		if (controlsHaveBeenCreated()) {
			for (final ISectionEpl section : fSections) {
				ISafeRunnable runnable = new ISafeRunnable() {
					@Override
					public void run() throws Exception {
						section.refresh();
					}
					@Override
					public void handleException(Throwable exc) {
						SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
					}
				};
				SafeRunner.run(runnable);
			}
		}
	}

	/**
	 * Invalidate the tab content
	 */
	public void invalidate() {
		// Forward the notification to each of our sections. As that involves 
		// invoking untrusted client code, use a safe runner. 
		if (controlsHaveBeenCreated()) {
			for (final ISectionEpl section : fSections) {
				ISafeRunnable runnable = new ISafeRunnable() {
					@Override
					public void run() throws Exception {
						section.invalidate();
					}
					@Override
					public void handleException(Throwable exc) {
						SystemBrowserActivator2.log(new Status(IStatus.ERROR, SystemBrowserActivator2.getPluginId(), exc.getMessage()));
					}
				};
				SafeRunner.run(runnable);
			}
		}
	}
	
	/**
	 * This tab is being given focus. If some control within the myriad of
	 * controls that may exist in this tab had focus last time we were visisble,
	 * then attempt to restore focus to it. At the very least, set the focus to
	 * the page composite.
	 */
	public void setFocus() {
		if (fLastControlWithFocus != null && !fLastControlWithFocus.isDisposed() && fLastControlWithFocus.isVisible()) {
			fLastControlWithFocus.setFocus();
		}
		else {
			fPageComposite.setFocus();
		}
	}
}
