/*******************************************************************************
 * Copyright (c) 2000, 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Gary Duprex <Gary.Duprex@aspectstools.com> - bug 179213
 *     John Cortell (Freescale) - code customized for Processor Expert
 *******************************************************************************/
package com.processorexpert.derivative.npw;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;

import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.ide.IIDEHelpContextIds;
import org.eclipse.ui.internal.ide.dialogs.FileSystemSelectionArea;
import org.eclipse.ui.internal.ide.dialogs.IDEResourceInfoUtils;
import org.eclipse.ui.internal.ide.filesystem.FileSystemConfiguration;
import org.eclipse.ui.internal.ide.filesystem.FileSystemSupportRegistry;

import com.processorexpert.derivative.npw.CustomProjectContentsLocationArea.IErrorMessageReporter;


/**
 * This is a copy of the eclipse WizardNewProjectCreationPage page, with the addition
 * of being able to set the project name.
 * 
 */
@SuppressWarnings("restriction") // we use internal eclipse API and we're OK with it
public class ProjectNamePage extends WizardPage {

	// initial value stores
    private String initialProjectFieldValue;
    private String initialProjectLocation;
    private String helpID = null;
    
    // widgets
    Text projectNameField;

    private Listener nameModifyListener = new Listener() {
        public void handleEvent(Event e) {
        	setLocationForSelection();
            boolean valid = validatePage();
            setPageComplete(valid);
                
        }
    };

	private CustomProjectContentsLocationArea locationArea;

    // constants
    private static final int SIZING_TEXT_FIELD_WIDTH = 420;

    /**
     * Creates a new project creation wizard page.
     *
     * @param pageName the name of this page
     */
    public ProjectNamePage(String title, String description, String helpID) {
        super(title);
		setTitle(title);
		setDescription(description);
		this.helpID = helpID;
        setPageComplete(false);
    }

    public ProjectNamePage() {
        super(Messages.ProjectNamePage_title); //$NON-NLS-1$
		setTitle(Messages.ProjectNamePage_title); //$NON-NLS-1$
		setDescription(Messages.ProjectNamePage_Description); //$NON-NLS-1$
        setPageComplete(false);
    }

    /** (non-Javadoc)
     * Method declared on IDialogPage.
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        composite.setFont(parent.getFont());

        initializeDialogUnits(parent);

        if (helpID != null && (helpID.length() != 0)){
        	PlatformUI.getWorkbench().getHelpSystem().setHelp(composite,
        			helpID);
        }else{
        	PlatformUI.getWorkbench().getHelpSystem().setHelp(composite,
        			IIDEHelpContextIds.NEW_PROJECT_WIZARD_PAGE);
        }
        
        composite.setLayout(new GridLayout());
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));

        createProjectNameGroup(composite);
        locationArea = new CustomProjectContentsLocationArea(getErrorReporter(), composite);
        if(initialProjectFieldValue != null)
        	locationArea.updateProjectName(initialProjectFieldValue);
        if (initialProjectLocation != null) {
        	locationArea.updateProjectLocation(initialProjectLocation);
        	initialProjectLocation = null;
        }
		// Scale the button based on the rest of the dialog
		setButtonLayoutData(locationArea.getBrowseButton());
		
    	setLocationForSelection();
        setPageComplete(validatePage());
        // Show description on opening
        setErrorMessage(null);
        setMessage(null);
        setControl(composite);
        validatePage();
        if (helpID != null)
    		PlatformUI.getWorkbench().getHelpSystem().setHelp(composite, helpID);
    }
    
    /**
	 * Get an error reporter for the receiver.
	 * @return IErrorMessageReporter
	 */
	private IErrorMessageReporter getErrorReporter() {
		return new IErrorMessageReporter(){
			/* (non-Javadoc)
			 * @see org.eclipse.ui.internal.ide.dialogs.ProjectContentsLocationArea.IErrorMessageReporter#reportError(java.lang.String)
			 */
			public void reportError(String errorMessage) {
			
				setErrorMessage(errorMessage);
				boolean valid = errorMessage == null;
				if(valid)
					valid = validatePage();
				
				setPageComplete(valid);
			}
			
		};
	}

    /**
     * Creates the project name specification controls.
     *
     * @param parent the parent composite
     */
    private final void createProjectNameGroup(Composite 
    		parent) {
        // project specification group
        Composite projectGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        projectGroup.setLayout(layout);
        projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        // new project label
        Label projectLabel = new Label(projectGroup, SWT.NONE);
        projectLabel.setText(Messages.ProjectNamePage_label); //$NON-NLS-1$
        projectLabel.setFont(parent.getFont());

        // new project name entry field
        projectNameField = new Text(projectGroup, SWT.BORDER);
        GridData data = new GridData(GridData.FILL_HORIZONTAL);
        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        projectNameField.setLayoutData(data);
        projectNameField.setFont(parent.getFont());

        // Set the initial value first before listener
        // to avoid handling an event during the creation.
        if (initialProjectFieldValue != null)
            projectNameField.setText(initialProjectFieldValue);
        projectNameField.addListener(SWT.Modify, nameModifyListener);
    }


    /**
     * Returns the current project location path as entered by 
     * the user, or its anticipated initial value.
     * Note that if the default has been returned the path
     * in a project description used to create a project
     * should not be set.
     *
     * @return the project location path or its anticipated initial value.
     */
    public IPath getLocationPath() {
        return new Path(locationArea.getProjectLocation());
    }

    /**
	 * Creates a project resource handle using the project name currently set in
	 * the UI.
	 * 
	 * <p>
	 * This method does not materialize the project resource; that is done by
	 * calling <code>IProject::create()</code>. Of course, the project may
	 * already exist.
	 * 
	 * @return the new project resource handle
	 */
    public IProject getProjectHandle() {
    	return getProjectHandle(null);
    }

    /**
	 * Creates a project resource handle for the given project name, which may
	 * contain the substitution variable ${ProjName}. The variable is replaced
	 * with the project name currently set in the UI.
	 * 
	 * <p>
	 * This method does not materialize the project resource; that is done by
	 * calling <code>IProject::create()</code>. Of course, the project may
	 * already exist.
	 * 
	 * @return the new project resource handle
	 */
    public IProject getProjectHandle(String overrideName) {
    	String projectName = getProjectName();
    	if (overrideName!=null){
    		projectName = replaceVariableWithValue(overrideName, 
    				"${ProjName}", projectName); //$NON-NLS-1$
    	}
    	return ResourcesPlugin.getWorkspace().getRoot().getProject(projectName); 
    }
    
	/** Performs variable substitution */
	public static String replaceVariableWithValue(String string, String varName, String value) {
		// Not sure why String.replace() wasn't used, but I'm not messing with
		// this legacy code right now.
		int pos = string.indexOf(varName);
		while(pos != -1) {
			String left = string.substring(0, pos);
			String right = string.substring(pos + varName.length());
			string = left + value + right;
			pos = string.indexOf(varName, pos + value.length());
		}
		return string;
	}
	
    
    /**
     * Returns the current project name as entered by the user, or its anticipated
     * initial value.
     *
     * @return the project name, its anticipated initial value, or <code>null</code>
     *   if no project name is known
     */
    public String getProjectName() {
        if (projectNameField == null)
            return initialProjectFieldValue;

        return getProjectNameFieldValue();
    }

    /**
     * Returns the value of the project name field
     * with leading and trailing spaces removed.
     * 
     * @return the project name in the field
     */
    private String getProjectNameFieldValue() {
        if (projectNameField == null)
            return ""; //$NON-NLS-1$

        return projectNameField.getText().trim();
    }

    /**
     * Sets the initial project name that this page will use when
     * created. The name is ignored if the createControl(Composite)
     * method has already been called. Leading and trailing spaces
     * in the name are ignored.
     * Providing the name of an existing project will not necessarily 
     * cause the wizard to warn the user.  Callers of this method 
     * should first check if the project name passed already exists 
     * in the workspace.
     * 
     * @param name initial project name for this page
     * 
     * @see IWorkspace#validateName(String, int)
     * 
     */
    public void setInitialProjectName(String name) {
        if (name == null)
            initialProjectFieldValue = null;
        else {
        	initialProjectFieldValue = name;
            if(locationArea != null)
            	locationArea.updateProjectName(name);
        }
    }

    public void setProjectLocation(String location) {
    	if (location != null) {
            if(locationArea != null)
            	locationArea.updateProjectLocation(location);
            else
            	initialProjectLocation = location;
    	}
    }
    
    public void setProjectName(String name) {
        if (name != null)
        {
        	if (projectNameField == null)
        		setInitialProjectName(name);
        	else {
	            projectNameField.setText(name);
	            if(locationArea != null)
	            	locationArea.updateProjectName(name);
        	}
        }
    }
    
    /**
     * Set the location to the default location if we are set to useDefaults.
     */
    void setLocationForSelection() {
    	if (locationArea != null)
    		locationArea.updateProjectName(getProjectNameFieldValue());
    }

  
    /**
     * Returns whether this page's controls currently all contain valid 
     * values.
     *
     * @return <code>true</code> if all controls are valid, and
     *   <code>false</code> if at least one is invalid
     */
    protected boolean validatePage() {
        IWorkspace workspace = IDEWorkbenchPlugin.getPluginWorkspace();
        
        String projectFieldContents = getProjectNameFieldValue();
        if (projectFieldContents.equals("")) { //$NON-NLS-1$
            setErrorMessage(null);
            setMessage(null);
            return false;
        }

        IStatus nameStatus = workspace.validateName(projectFieldContents,
                IResource.PROJECT);
        if (!nameStatus.isOK()) {
            setErrorMessage(nameStatus.getMessage());
            return false;
        }

        IProject handle = getProjectHandle();
        if (handle.exists()) {
            setErrorMessage(Messages.ProjectNamePage_error); //$NON-NLS-1$
            return false;
        }
        
        String locationValidation = locationArea.checkValidLocation();
        if(locationValidation != null){
        	setErrorMessage(locationValidation);
            return false;   
        }
        
        String projectLocation = locationArea.getProjectLocation();
        if (initialProjectLocation != null)
        	projectLocation = initialProjectLocation;
        if (locationArea.isDefault())
        	projectLocation = projectLocation + File.separator + getProjectName();
        	
             
        /*
         * If not using the default value validate the location.
         */
        if (!locationArea.isDefault()) {
            IStatus locationStatus = workspace.validateProjectLocation(handle,
                    new Path(locationArea.getProjectLocation()));
            if (!locationStatus.isOK()) {
                setErrorMessage(locationStatus.getMessage());
                return false;
            }
        }

        if (!locationArea.isDefault()) {
        	File newFolder = new File(locationArea.getProjectLocation());
        	
	        if ( newFolder.exists() && (newFolder.list().length > 0) ){
				setMessage(Messages.ProjectLocationPage_warning,  IMessageProvider.WARNING);
				return true;
			}
        }
        
        setErrorMessage(null);
        setMessage(null);
        return true;
    }

    private boolean showSteps = false;
    private String title;
    
    public void setShowSteps(boolean show) {
    	showSteps = show;
    }
    
    public void setPageTitle(String title) {
    	this.title = title;
    }
    
    // Check if nextPage is a valid page and if not find the next one
    //
	private IWizardPage findNextPage(IWizardPage nextPage) {
		while (nextPage instanceof IPExWizardPage) {
			IPExWizardPage pexPage = (IPExWizardPage)nextPage;
			if (pexPage.isPageVisible())
				return nextPage;
			nextPage = nextPage.getNextPage();
		}
		return nextPage;
	}

	// Calculate the number of pages currently queued up in the wizard
	//
	private int calculatePageCount() {
		int count = 1;
		IWizardPage page = findNextPage(getNextPage());
		while (page != null) {
			count++;
			page = findNextPage(page.getNextPage());
		}
		return count;
	}
	
   /*
     * see @DialogPage.setVisible(boolean)
     */
    public void setVisible(boolean visible) {
        super.setVisible(visible);
        if (visible) {
        	// set the page title
        	if (title != null) {
    			setTitle(title);
    			
    			// show the steps if necessary
    			if (showSteps) {
    				String format = Messages.ProjectNamePage_stepFormat;
    				String title = String.format(format, getTitle(), 1, calculatePageCount());
    				setTitle(title);
    			}

        	}
            projectNameField.setFocus();
        }
    }

    /**
     * Returns the useDefaults.
     * @return boolean
     */
    public boolean useDefaults() {
        return locationArea.isDefault();
    }

	public boolean validProjectName(String name) {
        IProject handle = getProjectHandle(name);
        if (handle.exists()) {
            setErrorMessage(Messages.ProjectNamePage_error); //$NON-NLS-1$
            return false;
        }
        return true;
	}
	
	public boolean isCurrent(){
		return super.isCurrentPage();
	}
}

class CustomProjectContentsLocationArea
{
	public static final String EMPTY_STRING = ""; //$NON-NLS-1$
	

	/**
	 * IErrorMessageReporter is an interface for type that allow message
	 * reporting.
	 * 
	 */
	public interface IErrorMessageReporter {
		/**
		 * Report the error message
		 * 
		 * @param errorMessage
		 *            String or <code>null</code>. If the errorMessage is
		 *            null then clear any error state.
		 */
		public void reportError(String errorMessage);
	}

	@SuppressWarnings("restriction")
	private static String BROWSE_LABEL = IDEWorkbenchMessages.ProjectLocationSelectionDialog_browseLabel;

	private static final int SIZING_TEXT_FIELD_WIDTH = 250;

	private static final String FILE_SCHEME = "file"; //$NON-NLS-1$

	private Label locationLabel;

	private Text locationPathField;

	private Button browseButton;

	private IErrorMessageReporter errorReporter;

	private String projectName = EMPTY_STRING;

	private String userPath = EMPTY_STRING;

	private Button useDefaultsButton;

	private IProject existingProject;

	@SuppressWarnings("restriction")
	private FileSystemSelectionArea fileSystemSelectionArea;

	/**
	 * Create a new instance of the receiver.
	 * 
	 * @param reporter
	 * @param composite
	 * @param startProject
	 */
	public CustomProjectContentsLocationArea(IErrorMessageReporter reporter,
			Composite composite, IProject startProject) {

		errorReporter = reporter;
		projectName = startProject.getName();
		existingProject = startProject;

		boolean defaultEnabled = true;
		try {
			defaultEnabled = startProject.getDescription().getLocationURI() == null;
		} catch (CoreException e1) {
			// If we get a CoreException assume the default.
		}
		createContents(composite, defaultEnabled);
	}

	/**
	 * Create a new instance of a ProjectContentsLocationArea.
	 * 
	 * @param reporter
	 * @param composite
	 */
	public CustomProjectContentsLocationArea(IErrorMessageReporter reporter,
			Composite composite) {
		errorReporter = reporter;

		// If it is a new project always start enabled
		createContents(composite, true);
	}

	/**
	 * Create the contents of the receiver.
	 * 
	 * @param composite
	 * @param defaultEnabled
	 */
	@SuppressWarnings("restriction")
	private void createContents(Composite composite, boolean defaultEnabled) {

		int columns = 4;

		// project specification group
		Composite projectGroup = new Composite(composite, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = columns;
		projectGroup.setLayout(layout);
		projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		useDefaultsButton = new Button(projectGroup, SWT.CHECK | SWT.RIGHT);
		useDefaultsButton
				.setText(IDEWorkbenchMessages.ProjectLocationSelectionDialog_useDefaultLabel);
		useDefaultsButton.setSelection(defaultEnabled);
		GridData buttonData = new GridData();
		buttonData.horizontalSpan = columns;
		useDefaultsButton.setLayoutData(buttonData);

		createUserEntryArea(projectGroup, defaultEnabled);

		useDefaultsButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				boolean useDefaults = useDefaultsButton.getSelection();

				if (useDefaults) {
					userPath = locationPathField.getText();
					locationPathField.setText(TextProcessor
							.process(getDefaultPathDisplayString()));
				} else {
					locationPathField.setText(TextProcessor.process(userPath));
				}
				setUserAreaEnabled(!useDefaults);
			}
		});
		setUserAreaEnabled(!defaultEnabled);
	}

	/**
	 * Return whether or not we are currently showing the default location for
	 * the project.
	 * 
	 * @return boolean
	 */
	public boolean isDefault() {
		return useDefaultsButton.getSelection();
	}

	/**
	 * Create the area for user entry.
	 * 
	 * @param composite
	 * @param defaultEnabled
	 */
	@SuppressWarnings("restriction")
	private void createUserEntryArea(Composite composite, boolean defaultEnabled) {
		// location label
		locationLabel = new Label(composite, SWT.NONE);
		locationLabel
				.setText(IDEWorkbenchMessages.ProjectLocationSelectionDialog_locationLabel);

		// project location entry field
		locationPathField = new Text(composite, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = SIZING_TEXT_FIELD_WIDTH;
		data.horizontalSpan = 2;
		locationPathField.setLayoutData(data);

		// browse button
		browseButton = new Button(composite, SWT.PUSH);
		browseButton.setText(BROWSE_LABEL);
		browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				handleLocationBrowseButtonPressed();
			}
		});

		createFileSystemSelection(composite);

		if (defaultEnabled) {
			locationPathField.setText(TextProcessor
					.process(getDefaultPathDisplayString()));
		} else {
			if (existingProject == null) {
				locationPathField.setText(IDEResourceInfoUtils.EMPTY_STRING);
			} else {
				locationPathField.setText(TextProcessor.process(existingProject
						.getLocation().toString()));
			}
		}

		locationPathField.addModifyListener(new ModifyListener() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
			 */
			public void modifyText(ModifyEvent e) {
				errorReporter.reportError(checkValidLocation());
			}
		});
	}

	/**
	 * Create the file system selection area.
	 * 
	 * @param composite
	 */
	@SuppressWarnings("restriction")
	private void createFileSystemSelection(Composite composite) {

		// Always use the default if that is all there is.
		if (FileSystemSupportRegistry.getInstance().hasOneFileSystem()) {
			return;
		}

		new Label(composite, SWT.NONE);

		fileSystemSelectionArea = new FileSystemSelectionArea();
		fileSystemSelectionArea.createContents(composite);
	}

	/**
	 * Return the path we are going to display. If it is a file URI then remove
	 * the file prefix.
	 * 
	 * @return String
	 */
	private String getDefaultPathDisplayString() {

		URI defaultURI = null;
		if (existingProject != null) {
			defaultURI = existingProject.getLocationURI();
		}

		// Handle files specially. Assume a file if there is no project to query
		if (defaultURI == null || defaultURI.getScheme().equals(FILE_SCHEME)) {
			return Platform.getLocation().append(projectName).toString();
		}
		return defaultURI.toString();

	}

	/**
	 * Set the enablement state of the receiver.
	 * 
	 * @param enabled
	 */
	@SuppressWarnings("restriction")
	private void setUserAreaEnabled(boolean enabled) {

		locationLabel.setEnabled(enabled);
		locationPathField.setEnabled(enabled);
		browseButton.setEnabled(enabled);
		if (fileSystemSelectionArea != null) {
			fileSystemSelectionArea.setEnabled(enabled);
		}
	}

	/**
	 * Return the browse button. Usually referenced in order to set the layout
	 * data for a dialog.
	 * 
	 * @return Button
	 */
	public Button getBrowseButton() {
		return browseButton;
	}

	/**
	 * Open an appropriate directory browser
	 */
	@SuppressWarnings("restriction")
	private void handleLocationBrowseButtonPressed() {

		String selectedDirectory = null;
		String dirName = getPathFromLocationField();

		if (!dirName.equals(IDEResourceInfoUtils.EMPTY_STRING)) {
			IFileInfo info;
			info = IDEResourceInfoUtils.getFileInfo(dirName);

			if (info == null || !(info.exists()))
				dirName = IDEResourceInfoUtils.EMPTY_STRING;
		}

		FileSystemConfiguration config = getSelectedConfiguration();
		if (config== null || config.equals(
				FileSystemSupportRegistry.getInstance()
						.getDefaultConfiguration())) {
			DirectoryDialog dialog = new DirectoryDialog(locationPathField
					.getShell());
			dialog
					.setMessage(IDEWorkbenchMessages.ProjectLocationSelectionDialog_directoryLabel);

			dialog.setFilterPath(dirName);

			selectedDirectory = dialog.open();

		} else {
			URI uri = getSelectedConfiguration().getContributor()
					.browseFileSystem(dirName, browseButton.getShell());
			if (uri != null)
				selectedDirectory = uri.toString();
		}

		if (selectedDirectory != null)
			updateLocationField(selectedDirectory + File.separator + projectName);
	}

	/**
	 * Update the location field based on the selected path.
	 * 
	 * @param selectedPath
	 */
	private void updateLocationField(String selectedPath) {
		locationPathField.setText(TextProcessor.process(selectedPath));
	}

	/**
	 * Return the path on the location field.
	 * 
	 * @return String
	 */
	private String getPathFromLocationField() {
		URI fieldURI;
		try {
			fieldURI = new URI(locationPathField.getText());
		} catch (URISyntaxException e) {
			return locationPathField.getText();
		}
		return fieldURI.getPath();
	}

	/**
	 * Check if the entry in the widget location is valid. If it is valid return
	 * null. Otherwise return a string that indicates the problem.
	 * 
	 * @return String
	 */
	@SuppressWarnings("restriction")
	public String checkValidLocation() {

		String locationFieldContents = locationPathField.getText();
		if (locationFieldContents.length() == 0) {
			return (IDEWorkbenchMessages.WizardNewProjectCreationPage_projectLocationEmpty);
		}

		URI newPath = getProjectLocationURI();
		if (newPath == null) {
			return IDEWorkbenchMessages.ProjectLocationSelectionDialog_locationError;
		}

		//create a dummy project for the purpose of validation if necessary
		IProject project = existingProject;
		if (project == null) {
			String name = new Path(locationFieldContents).lastSegment();
			if (name != null && Path.EMPTY.isValidSegment(name)){
				project = ResourcesPlugin.getWorkspace().getRoot().getProject(name);
			}
			else {
				return IDEWorkbenchMessages.ProjectLocationSelectionDialog_locationError;
			}
		}
				
		if (locationFieldContents != null && new File(locationFieldContents).exists()) 
        	if (new File(locationFieldContents + File.separator + ".project").exists() ||
        			new File(locationFieldContents + File.separator + ".cproject").exists() ) {
        		
        		if (isDefault()) {
        			return Messages.ProjectLocationPage_Eclipse_error;
        		} else {
        			return Messages.ProjectLocationPage_error;
        		}
	        }
				
		if (existingProject != null) {
			URI projectPath = existingProject.getLocationURI();
			if (projectPath != null && URIUtil.equals(projectPath, newPath)) {
				return IDEWorkbenchMessages.ProjectLocationSelectionDialog_locationError;
			}
		}
		
		return null;
	}

	/**
	 * Get the URI for the location field if possible.
	 * 
	 * @return URI or <code>null</code> if it is not valid.
	 */
	@SuppressWarnings("restriction")
	public URI getProjectLocationURI() {

		FileSystemConfiguration configuration = getSelectedConfiguration();
		if (configuration == null) {
			return null;
		}

		return configuration.getContributor().getURI(
				locationPathField.getText());

	}

	/**
	 * Return the selected contributor
	 * 
	 * @return FileSystemConfiguration or <code>null</code> if it cannot be
	 *         determined.
	 */
	@SuppressWarnings("restriction")
	private FileSystemConfiguration getSelectedConfiguration() {
		if (fileSystemSelectionArea == null) {
			return FileSystemSupportRegistry.getInstance()
					.getDefaultConfiguration();
		}

		return fileSystemSelectionArea.getSelectedConfiguration();
	}

	/**
	 * Set the text to the default or clear it if not using the defaults.
	 * 
	 * @param useDefaults
	 * @param newName
	 *            the name of the project to use. If <code>null</code> use the
	 *            existing project name.
	 */
	public void updateProjectName(String newName) {
		projectName = newName;
		if (isDefault()) {
			locationPathField.setText(TextProcessor
					.process(getDefaultPathDisplayString()));
		}

	}

	/**
	 * Return the location for the project. If we are using defaults then return
	 * the workspace root so that core creates it with default values.
	 * 
	 * @return String
	 */
	public String getProjectLocation() {
		if (isDefault()) {
			return Platform.getLocation().toString();
		}
		return locationPathField.getText();
	}
	
	public void updateProjectLocation(String value) {
		useDefaultsButton.setSelection(false);
		locationPathField.setText(value);
		setUserAreaEnabled(true);
	}
	

}
