#!/bin/bash

# Extended regex pattern matching option on
shopt -s extglob

scriptdir=`dirname $0`
hnfull=`hostname`
hn=`basename $hnfull .am.freescale.net`
hn=`basename $hn .mtwk.freescale.net`
source $scriptdir/build.env-$hn
BLDSCRIPT=`basename $0`
BLDINVOKE="`pwd`/${BLDSCRIPT} $@"

BLDxscript=`readlink -e $0`
BLDxdir=`dirname $BLDxscript`
pushd $BLDxdir > /dev/null
BLDrevnum=`svn info | grep 'Revision' | awk '{print $2}'`
BLDrev="r$BLDrevnum"
popd > /dev/null

# Sub build dry run
BLDSUBDRY=""
TDY=`date +%Y%m%d`

BLDVERBOSE=0
BLDDRYRUN=0
BLDJOBS=()
BLDJOBSSTR=()

declare -A BLDDESC
BLDDESC=(
    ["dtuser"]=$USER
    ["cdeuser"]=$USER
    ["steps"]="all"
    ["targets"]="all"
    ["src"]="F462"
    ['flgs']="-O3"
)

declare -A BLDTARGETS
BLDTARGETS=(
    ["XLMe500v1_cde"]="XLMe500v1_cde"
    ["XLMe500v2_cde"]="XLMe500v2_cde"
    ["XLMe600_cde"]="XLMe600_cde"
    ["XLM603e_cde"]="XLM603e_cde"
    ["XLe300c2_cde"]="XLe300c2_cde"
    ["XLMe300c3_cde"]="XLMe300c3_cde"
    ["XLMe500mc_cde"]="XLMe500mc_cde"
    ["XLMe5500_cde"]="XLMe5500_cde"
    ["XLMe6500_cde"]="XLMe6500_cde"
    ["ELe500v1_cde"]="ELe500v1_cde"
    ["ELe500v2_cde"]="ELe500v2_cde"
    ["ELe600_cde"]="ELe600_cde"
    ["EL603e_cde"]="EL603e_cde"
    ["ELe300c2_cde"]="ELe300c2_cde"
    ["ELe300c3_cde"]="ELe300c3_cde"
    ["ELe500mc_cde"]="ELe500mc_cde"
    ["ELe5500_cde"]="ELe5500_cde"
    ["ELe6500_cde"]="ELe6500_cde"

    ["XLMe500v1"]="XLMe500v1"
    ["XLMe500v2"]="XLMe500v2"
    ["XLMe600"]="XLMe600"
    ["XLM603e"]="XLM603e"
    ["XLe300c2"]="XLe300c2"
    ["XLMe300c3"]="XLMe300c3"
    ["XLMe500mc"]="XLMe500mc"
    ["XLMe5500"]="XLMe5500"
    ["XLMe6500"]="XLMe6500"
    
    ["Xe500v1"]="Xe500v1"
    ["Xe500v2"]="Xe500v2"
    ["XMe600"]="XMe600"
    ["Xe300c2"]="Xe300c2"
    ["XMe300c3"]="XMe300c3"
    ["XMe500mc"]="XMe500mc"
    ["XMe5500"]="XMe5500"
    ["XMe6500"]="XMe6500"
    
    ["ELe500v1"]="ELe500v1"
    ["ELe500v2"]="ELe500v2"
    ["ELe600"]="ELe600"
    ["EL603e"]="EL603e"
    ["ELe300c2"]="ELe300c2"
    ["ELe300c3"]="ELe300c3"
    ["ELe500mc"]="ELe500mc"
    ["ELe5500"]="ELe5500"
    ["ELe6500"]="ELe6500"
    
    ["NLe600"]="NLe600"
    ["NLe500mc"]="NLe500mc"
    ["NLe5500"]="NLe5500"
    ["NLe6500"]="NLe6500"
    
    ["NLe500v2"]="NLe500v2"
    ["NSPE_targets"]="NLe500v2"
    
    ["cde-load1"]="cde-load1"
    ["cde-load2"]="cde-load2"
    ["cde-load3"]="cde-load3"
    ["cde-load4"]="cde-load4"
    ["cde-load5"]="cde-load5"
    ["cde-load6"]="cde-load6"
    ["cde-load7"]="cde-load7"
    ["cde-load8"]="cde-load8"
    ["cde-load9"]="cde-load9"
    ["cde-load10"]="cde-load10"
    ["cde-load11"]="cde-load11"
    ["cde-load12"]="cde-load12"
    ["x86-load1"]="x86-load1"
    ["x86-load2"]="x86-load2"
    ["x86-load3"]="x86-load3"
    ["x86-load4"]="x86-load4"
    ["x86-load5"]="x86-load5"
    ["x86-load6"]="x86-load6"
    ["nat-load1"]="nat-load1"
    ["nat-load2"]="nat-load2"
    ["all-nat"]="nat-load1 nat-load2"
    ["all"]="cde-load1 cde-load2 cde-load4 cde-load5
             cde-load7 cde-load8 cde-load9 cde-load10 cde-load11
             x86-load1 x86-load2 x86-load3 x86-load4 x86-load5 x86-load6
             nat-load1 nat-load2 NLe500v2"
)

# <BLDBASECMD id>, <host user [cde|dt]>,  <buildhost>, <target type>, <real target name>
declare -A BLDSPEC
BLDSPEC=(
    ["XLMe500v1_cde"]="base_cde_build         cde  lt03-tx30    Lcross  XLMe500v1"
    ["XLMe500v2_cde"]="base_cde_build         cde  lt03-tx30    Lcross  XLMe500v2"
    ["XLMe600_cde"]="  base_cde_build         cde  lt02-tx30    Lcross  XLMe600"
    ["XLM603e_cde"]="  base_cde_build         cde  lt02-tx30    Lcross  XLM603e"
    ["XLe300c2_cde"]=" base_cde_build         cde  lt02-tx30    Lcross  XLe300c2"
    ["XLMe300c3_cde"]="base_cde_build         cde  lt04-tx30    Lcross  XLMe300c3"
    ["XLMe500mc_cde"]="base_cde_build         cde  lt04-tx30    Lcross  XLMe500mc"
    ["XLMe5500_cde"]=" base_cde_build         cde  lt05-tx30    Lcross  XLMe5500"
    ["XLMe6500_cde"]=" base_cde_build         cde  lt05-tx30    Lcross  XLMe6500"
    					          
    ["ELe500v1_cde"]=" base_cde_build         cde  lds14-tx30   eabi    ELe500v1"
    ["ELe500v2_cde"]=" base_cde_build         cde  lds14-tx30   eabi    ELe500v2"
    ["ELe600_cde"]="   base_cde_build         cde  lds14-tx30   eabi    ELe600"
    ["EL603e_cde"]="   base_cde_build         cde  lds14-tx30   eabi    EL603e"
    ["ELe5500_cde"]="  base_cde_build         cde  lds14-tx30   eabi    ELe5500"
    ["ELe300c2_cde"]=" base_cde_build         cde  lds15-tx30   eabi    ELe300c2"
    ["ELe300c3_cde"]=" base_cde_build         cde  lds15-tx30   eabi    ELe300c3"
    ["ELe500mc_cde"]=" base_cde_build         cde  lds15-tx30   eabi    ELe500mc"
    ["ELe6500_cde"]="  base_cde_build         cde  lds15-tx30   eabi    ELe6500"
					          
    ["cde-load1"]="    base_cde_build         cde  lt03-tx30    Lcross  XLMe500v1"
    ["cde-load2"]="    base_cde_build         cde  lt04-tx30    Lcross  XLMe500v2"
    ["cde-load3"]="    base_cde_build         cde  lt05-tx30    Lcross  "
    ["cde-load4"]="    base_cde_build         cde  lds14-tx30   Lcross  XLMe500mc XLe300c2"
    ["cde-load5"]="    base_cde_build         cde  lds15-tx30   Lcross  XLMe600 XLM603e"
    ["cde-load6"]="    base_cde_build         cde  lt02-tx30    Lcross  "
    ["cde-load7"]="    base_cde_build         cde  lt07-tx30    Lcross  XLMe5500 XLMe300c3"
    ["cde-load8"]="    base_cde_build         cde  lt08-tx30    Lcross  XLMe6500"
    ["cde-load9"]="    base_cde_build         cde  lt09-tx30    eabi    ELe500mc ELe6500 ELe300c2"
    ["cde-load10"]="   base_cde_build         cde  lt10-tx30    eabi    ELe600 ELe5500 EL603e"
    ["cde-load11"]="   base_cde_build         cde  lt11-tx30    eabi    ELe500v1 ELe500v2 ELe300c3"
    ["cde-load12"]="   base_cde_build         cde  lt12-tx30    eabi    " 
   					          
    ["XLMe500v1"]="    base_cross_build       dt   perf3farm17  Lcross  XLMe500v1"
    ["XLMe500v2"]="    base_cross_build       dt   perf3farm17  Lcross  XLMe500v2"
    ["XLMe600"]="      base_cross_build       dt   perf3farm17  Lcross  XLMe600"
    ["XLM603e"]="      base_cross_build       dt   perf3farm17  Lcross  XLM603e"
    ["XLe300c2"]="     base_cross_build       dt   perf3farm17  Lcross  XLe300c2"
    ["XLMe300c3"]="    base_cross_build       dt   perf3farm18  Lcross  XLMe300c3"
    ["XLMe500mc"]="    base_cross_build       dt   perf3farm18  Lcross  XLMe500mc"
    ["XLMe5500"]="     base_cross_build       dt   perf3farm19  Lcross  XLMe5500"
    ["XLMe6500"]="     base_cross_build       dt   perf3farm19  Lcross  XLMe6500"
    					          
    ["Xe500v1"]="      base_cross_build       dt   perf3farm17  cross   Xe500v1"
    ["Xe500v2"]="      base_cross_build       dt   perf3farm17  cross   Xe500v2"
    ["XMe600"]="       base_cross_build       dt   perf3farm17  cross   XMe600"
    ["Xe300c2"]="      base_cross_build       dt   perf3farm17  cross   Xe300c2"
    ["XMe300c3"]="     base_cross_build       dt   perf3farm18  cross   XMe300c3"
    ["XMe500mc"]="     base_cross_build       dt   perf3farm18  cross   XMe500mc"
    ["XMe5500"]="      base_cross_build       dt   perf3farm19  cross   XMe5500"
    ["XMe6500"]="      base_cross_build       dt   perf3farm19  cross   XMe6500"
    					          
    ["ELe500v1"]="     base_eabi_build        dt   perf3farm18  eabi    ELe500v1"
    ["ELe500v2"]="     base_eabi_build        dt   perf3farm18  eabi    ELe500v2"
    ["ELe600"]="       base_eabi_build        dt   perf3farm18  eabi    ELe600"
    ["EL603e"]="       base_eabi_build        dt   perf3farm18  eabi    EL603e"
    ["ELe300c2"]="     base_eabi_build        dt   perf3farm18  eabi    ELe300c2"
    ["ELe300c3"]="     base_eabi_build        dt   perf3farm19  eabi    ELe300c3"
    ["ELe500mc"]="     base_eabi_build        dt   perf3farm19  eabi    ELe500mc"
    ["ELe5500"]="      base_eabi_build        dt   perf3farm19  eabi    ELe5500"
    ["ELe6500"]="      base_eabi_build        dt   perf3farm19  eabi    ELe6500"
					          
    ["x86-load1"]="    base_cross_build       dt   perf3farm17  Lcross  XLMe500v1 XLMe500v2 XLMe600 XLe300c2"
    ["x86-load2"]="    base_cross_build       dt   perf3farm18  Lcross  XLMe300c3 XLMe500mc XLM603e"
    ["x86-load3"]="    base_cross_build       dt   perf3farm19  Lcross  XLMe5500 XLMe6500"
    ["x86-load4"]="    base_eabi_build        dt   perf3farm17  eabi    ELe500v1 ELe500v2 ELe600"
    ["x86-load5"]="    base_eabi_build        dt   perf3farm18  eabi    ELe300c2 ELe300c3 ELe500mc"
    ["x86-load6"]="    base_eabi_build        dt   perf3farm19  eabi    ELe5500 ELe6500 EL603e"
    					          
    ["NLe5500"]="      base_native_build      dt   perf3farm5   native  NLe5500"
    ["NLe500mc"]="     base_native_build      dt   perf3farm6   native  NLe500mc"
    ["NLe600"]="       base_native_build      dt   perf3farm7   native  NLe600"
    ["NLe6500"]="      base_native_build      dt   perf3farm0   native  NLe6500"
    ["nat-load1"]="    base_native_build      dt   perf3farm10  native  NLe500mc NLe5500"
    ["nat-load2"]="    base_native_build      dt   perf3farm10  native  NLe600 NLe6500"
    
    ["NLe500v2"]="     base_native_build      dt   perf3farm3   native-spe  NLe500v2"    
)


BLDCMD="true && mkdir -p /tmp/${BLDDESC['dtuser']}/release && cd /tmp/${BLDDESC['dtuser']}/release && build_release.sh s=P_BLDDESC_SRC"
BLDCMD_CDE="true && mkdir -p /tmp/${BLDDESC['dtuser']}/release && cd /tmp/${BLDDESC['dtuser']}/release && build_release_CDE.sh s=P_BLDDESC_SRC"
NBLDCMD="true && cd / && build_release.sh s=P_BLDDESC_SRC"
NBLDCMD_P3F9="true && cd / && build_release_p3f9.sh s=P_BLDDESC_SRC"

#
# Caution keep BLDBASECMD and BLDBASECMD2 in synch
declare -A BLDBASECMD
BLDBASECMD=(
    ["base_cde_build"]="$BLDCMD_CDE P_BLDSUBDRY r=release-CDE-P_REV_P_BLDTARGETTYPE b=\"-i /pkg/fs-DTgnu-\" P_BLDREALTARGETNAME -rm-build -no-Wtar -no-deja"
    ["base_cross_build"]="$BLDCMD P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-deja"
    ["base_eabi_build"]="$BLDCMD P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-deja"
    ["base_native_build"]="$NBLDCMD P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-Wtar -no-deja"
    ["base_native_build2"]="$NBLDCMD P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-Wtar -no-deja"
    ["base_native_build_p3f9"]="$NBLDCMD_P3F9 P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-Wtar -no-deja"
)
declare -A BLDBASECMD2
BLDBASECMD2=(
    ["base_cde_build"]="echo DONE"
    ["base_cross_build"]="$BLDCMD P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-build -no-spec -no-tar -no-Wtar"
    ["base_eabi_build"]="echo DONE"
    ["base_native_build"]="$NBLDCMD P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-build -no-spec -no-tar -no-Wtar"
    ["base_native_build2"]="echo DONE"
    ["base_native_build_p3f9"]="$NBLDCMD_P3F9 P_BLDSUBDRY r=release-P_REV_P_BLDTARGETTYPE P_BLDREALTARGETNAME -no-build -no-spec -no-tar -no-Wtar"
)

trap kill_all_jobs INT

#
# Kill jobs if control-c hit
#
kill_all_jobs () {
    local i
    for (( i=0; i<50; i++ )); do
	if [ "${BLDJOBS[$i]}" != "" -a -d /proc/${BLDJOBS[$i]} ]; then
	    kill -9 ${BLDJOBS[$i]}
            echo "${BLDJOBSSTR[$i]}: (${BLDJOBS[$i]}) killed"
	fi
    done
}

#
# wait_for_build_jobs
#
wait_for_build_jobs () {
    # BLDJOBS is an array of PIDs for the 'sh' jobs started in the background and
    # are still running. Here, poll to see if the PID is still active. If it
    # has ended, issue a message and remove the job PID from the PID array.
    # Return if there are no more jobs to monitor.

    local i

    if [ ${#BLDJOBS[@]} -ne 0 ]; then
        echo "Waiting for jobs to complete..."
    fi
    while [ 1 ]; do
        for (( i=0; i<50; i++ )); do
            # Check process status
            if [ "${BLDJOBS[$i]}" != "" -a  ! -d /proc/${BLDJOBS[$i]} ]; then
                echo "${BLDJOBSSTR[$i]} (${BLDJOBS[$i]}) completed at `date`"
                # remove job from array
                unset BLDJOBS[$i]
            fi
        done
        # Break if no more jobs
        if [ ${#BLDJOBS[@]} -eq 0 ]; then 
           break
        fi
	sleep 5
    done
}

#
# issue_command - either issue command or print command if dry run only
#
issue_command () {
    local build_target=$1
    local build_phase=$2
    local targets
    local target
    local specs
    local user
    local command_spec
    local host
    local user_type
    local target_type
    local real_target_name
    local command
    local full_command
    local pid

    # Array of real targets
    targets=( $(echo ${BLDTARGETS[$build_target]}) )

    # For each real target
    for target in ${targets[@]}; do
        if [ "${BLDSPEC[$target]}" = "" ]; then
            echo "issue_command: no such BLDSPEC target: $target"
        fi
        specs=( $(echo ${BLDSPEC[$target]}) )
        command_spec=${specs[0]}
        user_type=${specs[1]}
        host=${specs[2]}
        target_type=${specs[3]}
        real_target_name="${specs[4]} ${specs[5]} ${specs[6]} ${specs[7]} ${specs[8]} ${specs[9]} ${specs[10]} ${specs[11]} ${specs[12]} ${specs[13]}"

        # This gets the base build command (background build)
	if [ "$build_phase" = "build" ]; then
            command="${BLDBASECMD[$command_spec]}"
	elif [ "$build_phase" = "deja" ]; then
            command="${BLDBASECMD2[$command_spec]}"
	else
	    echo "[issue_command] Invalid build_phase: $build_phase"
	    exit 1
	fi

        # cde or dt user name
        if [ "$user_type" = "cde" ]; then
            user=${BLDDESC['cdeuser']}
	else
	    user=${BLDDESC['dtuser']}
        fi

        full_command="ssh -l $user $host $command "
        # Replace "placeholder" vars BLDTARGETTYPE, BLDREALTARGETNAME parameters before execution
        # Note you can't use eval here, it strips the quotes from command, these are needed by ssh.
        full_command=${full_command/P_BLDTARGETTYPE/$target_type}
        full_command=${full_command/P_BLDSUBDRY/$BLDSUBDRY}
        full_command=${full_command/P_BLDREALTARGETNAME/$real_target_name}
        full_command=${full_command/P_REV/$BLDrev}
        full_command=${full_command/P_BLDDESC_SRC/${BLDDESC['src']}}

        [ ${BLDVERBOSE} -eq 1 -o ${BLDDRYRUN} -eq 1 ] && echo "$full_command"
	if [ ${BLDDRYRUN} -eq 0 ]; then
            $full_command &
            pid=$!
            BLDJOBS[${#BLDJOBS[@]}]=$pid
            BLDJOBSSTR[${#BLDJOBSSTR[@]}]="\"$real_target_name\""
            echo "\"$real_target_name\" ($pid) build on $host started at `date`"
	fi
    done
    if [ ${BLDDRYRUN} -eq 0 ]; then
	wait_for_build_jobs
    fi
}

#
# do_build - do build step
#
do_build () {
    # Turn targets string in an array
    local targets=( $(echo ${BLDDESC['targets']}) )
    local target 
    local i

    echo "do_build: ${targets[@]}"
    for target in ${targets[@]}; do
        # Check for legit target name
        if [ "${BLDTARGETS[$target]}" = "" ]; then 
            echo "do_build: unknown build target: $target"
            exit 1
        fi
	[ ${BLDVERBOSE} -eq 1 ] && echo "build_target: $target"
	issue_command $target build
    done
}

#
# do_deja - do deja step
#
do_deja () {
    # Turn targets string in an array
    local targets=( $(echo ${BLDDESC['targets']}) )
    local target 
    local i

    echo "do_deja: ${targets[@]}"
    for target in ${targets[@]}; do
        # Check for legit target name
        if [ "${BLDTARGETS[$target]}" = "" ]; then 
            echo "do_deja: unknown build target: $target"
            exit 1
        fi
	[ ${BLDVERBOSE} -eq 1 ] && echo "build_target: $target"
	issue_command $target deja
    done
}

#
# do_check - check result of build
#
do_check () {
    local targets=( $(echo ${BLDDESC['targets']}) )
    local specs=()
    local real_targets=()
    local x command_spec user_type host target_type real_target_names dtcde_str logfilename target real_target
    local has_error

    has_error=0
    [ "$targets" = "all" ] && targets=( $(echo ${BLDTARGETS['all']}) )
    echo "do_check:"
    for target in ${targets[@]}; do
        dtcde_str="-CDE"
        # Check for legit target name
        if [ "${BLDTARGETS[$target]}" = "" ]; then 
            echo "do_check: unknown target: $target"
            exit 1
        fi
	[ ${BLDVERBOSE} -eq 1 ] && echo " $target"

        # Get build specs to determine log file name
        specs=( $(echo ${BLDSPEC[$target]}) )
        command_spec=${specs[0]}
        user_type=${specs[1]}
        host=${specs[2]}
        target_type=${specs[3]}
        real_target_names="${specs[4]} ${specs[5]} ${specs[6]} ${specs[7]} ${specs[8]} ${specs[9]} ${specs[10]} ${specs[11]} ${specs[12]} ${specs[13]}"

        # The target could be a list of real targets
        real_targets=( $(echo $real_target_names) )
        
        [ "$user_type" = "dt" ] && dtcde_str=""

        # For each real target 
        for real_target in ${real_targets[@]}; do
            logfilename="$REPODIR/logs/release${dtcde_str}-${BLDrev}_${target_type}_${BLDDESC['src']}_${real_target}"
           # Now check for the file, error if not found
            if [ ! -e $logfilename ]; then 
                echo "Missing log file: $logfilename"
                has_error=1
            fi

           # CDE log files
            if [ "$user_type" = "cde" ]; then
                case $target_type in
                    Lcross)
	                x=`cat $logfilename | grep 'Setting up for mingwin canadian build'`
	                if [ "$x" = "" ]; then
	                    echo "Problem with log file: $logfilename"
	                    has_error=1
	                fi
                        ;;
                    eabi)
	                x=`cat $logfilename | grep 'Setting up for mingwin canadian build'`
	                if [ "$x" = "" ]; then
		            echo "Problem with log file: $logfilename"
		            has_error=1
	                fi
                        ;;
                    native)
                        echo "do_check: Invalid target type for CDE build: $target_type"
                        has_error=1
                        ;;
                    native-spe)
                        echo "do_check: Invalid target type for CDE build: $target_type"
                        has_error=1
                        ;;
                    *)
                        echo "do_check: Unknown target_type: $target_type ($target)"
                        has_error=1
                        ;;
                esac
                [ ${BLDVERBOSE} -eq 1 ] && echo "Log file $logfilename OK"
            else
                case $target_type in
                    Lcross)
	                x=`cat $logfilename | grep '^#DONE' | wc -l`
	                if [ "$x" != "4" ]; then
	                    echo "Problem with log file: $logfilename"
	                    has_error=1
	                fi
                        ;;
                    eabi)
	                x=`cat $logfilename | grep '^#DONE' | wc -l`
	                if [ "$x" != "3" ]; then
		            echo "Problem with log file: $logfilename"
		            has_error=1
	                fi
                        ;;
                    native)
	                x=`cat $logfilename | grep '^#DONE' | wc -l`
	                if [ "$x" != "3" ]; then
		            echo "Problem with log file: $logfilename"
		            has_error=1
	                fi
                        ;;
                    native-spe)
	                x=`cat $logfilename | grep '^#DONE' | wc -l`
	                if [ "$x" != "3" ]; then
		            echo "Problem with log file: $logfilename"
	                fi
                        ;;
                    *)
                        echo "do_check: Unknown target_type: $target_type ($target)"
                        has_error=1
                        ;;
                esac
                [ ${BLDVERBOSE} -eq 1 ] && echo "Log file $logfilename OK"
                [ ${BLDVERBOSE} -eq 1 ] && echo "Check Deja-Gnu results by hand for: $logfilename"
            fi
        done
    done
    [ ${has_error} -ne 0 ] && exit 1
}

#
# do_test - do build testing
#
fast_list="coremark auto-1 consumer-1 net-1 telecom-1 office-1 consumer-2 net-2 office-2 dhrystone whetstone smalltests linpack perfect"
all_part1="coremark auto-1 consumer-1 net-1 telecom-1 office-1 consumer-2 net-2 office-2 dhrystone whetstone smalltests linpack perfect stanford nas nbench spec2k"
all_part2="spec2k6"
TARFILE=""
TOOLDIR=""
HOSTDIR=""

parse_log () {
    # Argument is the log filename
    local f=$1
    
    TARFILE=`cat $f | grep 'Creating release tarball:' | awk '{print $4}'`

    local t1 t2 t3 t4 tp
    # Breakdown tarball file name, first with extension
    t1=`basename $TARFILE`
    t2=`basename $t1 .tar.gz`
    t3=`basename $t1 .tar.bz2`
    t4=`basename $t1 .tar.xz`
    tp=""
    if [ "$t2" != "$t1" ]; then
        tp=$t2
    elif [ "$t3" != "$t1" ]; then
        tp=$t3
    elif [ "$t4" != "$t1" ]; then
        tp=$t4
    else
        echo "I cannot process extension of this file: $t"
        exit 1
    fi

    # Calculate TOOLDIR
    TOOLDIR=`echo $tp | awk '{split($1,a,"-");i=index($1,a[5]);print(substr($1,1,i-2))}'`
    HOSTDIR=`echo $tp | awk '{split($1,a,"-");i=index($1,a[5]);j=index($1,a[7]);print(substr($1,i,j-i-1))}'`
}

dump_descriptor () {
    local tgt=$1
    local hn=$2
    local tn=$3
    local uprefix=$4
    local prefix=$5
    local core=$6
    local ml=$7
    local ct=$8
    cat > $REPODIR/build_gnu/f-bmk/TD-release-${BLDrev}_Lcross_$tgt <<EOF
BMKDESC=(
    ['extopts']=""
    ['boot_src']="perf3farm4"
    ['tar']="$RELEASEDIR/$TARFILE"
    ['label']="release-${BLDrev}_Lcross_$tgt"
    ['bd']=""
    ['hn']="$hn"
    ['hu']="tguru${uprefix}"
    ['tn']="$tn"
    ['tu']="tguru${uprefix}"
    ['Htools']=""
    ['Tlibs']=""
    ['tools']="$TOOLDIR/$HOSTDIR"
    ['prefix']="$prefix"
    ['core']="$core"
    ['ml']="$ml"
    ['ct']="$ct"
    ['flgs']="${BLDDESC['flgs']}"
)
EOF
}

do_test () {
    cat <<EOF
do_test:
Execute each "test_dispatch.sh" bellow as separated users for maximum parallelism

test_dispatch.sh TD-release-${BLDrev}_Lcross_aXLMe500mc TD-release-${BLDrev}_Lcross_aXLM603e  TD-release-${BLDrev}_Lcross_aXLMe5500
test_dispatch.sh TD-release-${BLDrev}_Lcross_bXLMe500mc TD-release-${BLDrev}_Lcross_bXLMe6500 TD-release-${BLDrev}_Lcross_bXLMe5500
test_dispatch.sh TD-release-${BLDrev}_Lcross_cXLMe300c3 TD-release-${BLDrev}_Lcross_cXLMe6500 TD-release-${BLDrev}_Lcross_cXLMe5500 
test_dispatch.sh TD-release-${BLDrev}_Lcross_dXLMe300c3 TD-release-${BLDrev}_Lcross_dXLM603e  TD-release-${BLDrev}_Lcross_dXLMe500v2
test_dispatch.sh TD-release-${BLDrev}_Lcross_eXLMe500v1 TD-release-${BLDrev}_Lcross_eXLe300c2 TD-release-${BLDrev}_Lcross_eXLMe600 
test_dispatch.sh TD-release-${BLDrev}_Lcross_fXLMe6500 TD-release-${BLDrev}_Lcross_fXLMe600 
EOF

    if [ -e  "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe500mc" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe500mc
        dump_descriptor aXLMe500mc perf3farm17 perf3board7 A powerpc-unknown-linux-gnu-     e500mc "Snof S32 nof 32" "$all_part1"
        dump_descriptor bXLMe500mc perf3farm17 perf3board7 B powerpc-unknown-linux-gnu-     e500mc "Snof S32 nof 32" "$all_part2"
    fi

    if [ -e  "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe300c3" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe300c3
        dump_descriptor dXLMe300c3 perf3farm17 perf3farm10 D powerpc-unknown-linux-gnu-     e300c3 "Snof S32 nof 32" "$all_part1"
        dump_descriptor cXLMe300c3 perf3farm17 perf3farm9 C powerpc-unknown-linux-gnu-      e300c3 "Snof S32 nof 32" "$all_part2"
    fi

    if [ -e "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe600" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe600
        dump_descriptor fXLMe600 perf3farm17 perf3farm9 F powerpc-unknown-linux-gnualtivec- e600   "Snof S32 nof 32" "$all_part1"
        dump_descriptor eXLMe600 perf3farm18 perf3farm10 E powerpc-unknown-linux-gnualtivec- e600   "Snof S32 nof 32" "$all_part2"
    fi

    if [ -e "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLM603e" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLM603e
        dump_descriptor aXLM603e perf3farm18 perf3farm9 A powerpc-unknown-linux-gnu-        603e   "Snof S32 nof 32" "$all_part1"
        dump_descriptor dXLM603e perf3farm18 perf3farm9 D powerpc-unknown-linux-gnu-        603e   "Snof S32 nof 32" "$all_part2"
    fi

    if [ -e "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe6500" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe6500
        dump_descriptor fXLMe6500 perf3farm18 perf3farm10 F powerpc-unknown-linux-gnu-      e6500  all               "$all_part1"
        dump_descriptor bXLMe6500 perf3farm18 perf3farm10 B powerpc-unknown-linux-gnu-      e6500  "Snof S32 nof 32" "$all_part2"
        dump_descriptor cXLMe6500 perf3farm18 perf3farm10 C powerpc-unknown-linux-gnu-      e6500  "S64 64"          "$all_part2"
    fi

    if [ -e "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe5500" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe5500
        dump_descriptor aXLMe5500 perf3farm19 perf3board8 A powerpc-unknown-linux-gnu-      e5500  all               "$all_part1"
        dump_descriptor bXLMe5500 perf3farm19 perf3board8 B powerpc-unknown-linux-gnu-      e5500  "Snof S32 nof 32" "$all_part2"
        dump_descriptor cXLMe5500 perf3farm19 perf3board8 C powerpc-unknown-linux-gnu-      e5500  "S64 64"          "$all_part2"
    fi

    if [ -e  "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLe300c2" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLe300c2
        dump_descriptor eXLe300c2 perf3farm17 perf3farm9 E powerpc-unknown-linux-gnu-       e300c2 "S32 32"          "$fast_list"
    fi

    if [ -e "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe500v1" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe500v1
        dump_descriptor eXLMe500v1 perf3farm19 perf3board1 E powerpc-unknown-linux-gnuspe-  e500v1 "Snof S32 nof 32" "$fast_list"
    fi

    if [ -e "$REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe500v2" ]; then
        parse_log $REPODIR/logs/release-${BLDrev}_Lcross_${BLDDESC['src']}_XLMe500v2
        dump_descriptor dXLMe500v2 perf3farm19 perf3board1 D powerpc-unknown-linux-gnuspe-  e500v2 "Snof S32 nof 32" "$fast_list"
    fi
}

#
# do_install_tr - do CDE TR install
#
do_install_tr () {
    local tarlist

    echo "do_install_tr"
    for f in $REPODIR/logs/release-CDE-${BLDrev}_Lcross_${BLDDESC['src']}_*; do
	t=`cat $f | grep 'Creating release tarball:' | awk '{print $4}'`
	tarlist="$tarlist $RELEASEDIR/$t"
    done
    for f in $REPODIR/logs/release-CDE-${BLDrev}_eabi_${BLDDESC['src']}_*; do
	t=`cat $f | grep 'Creating release tarball:' | awk '{print $4}'`
	tarlist="$tarlist $RELEASEDIR/$t"
    done

    if [ ${BLDDRYRUN} -eq 0 ]
    then      ssh -l ${BLDDESC['cdeuser']} wtc-ec1 "true && install_TR.sh DT $tarlist"
	      [ $? = 0 ] && echo "Toolchain $f installed Ok"
    else echo ssh -l ${BLDDESC['cdeuser']} wtc-ec1 "true && install_TR.sh DT $tarlist"
    fi
}

#
# do_steps - perform each specified step
#
do_steps () {
    # Turn steps string in an array of steps
    local steps=( $(echo ${BLDDESC['steps']}) )
    local step

    for step in ${steps[@]}; do
        case $step in
            all)
		do_build
		do_check
		do_test
		do_deja
		do_install_tr
                ;;
            build)
                do_build
                ;;
            check)
                do_check
                ;;
            deja)
                do_deja
                ;;
            test)
                do_test
                ;;
            install_tr)
                do_install_tr
                ;;
            *)
                echo "Unknown step: $step"
                exit 1
                ;;
         esac
    done
}

#
# do_help - help message
#
do_help () {
    local i
    local j
    local targets

    echo "Usage: $BLDSCRIPT  "
    echo "       [-h]           print this help"
    echo "       [-v]           verbose mode" 
    echo "       [-n]           dryrun, do not execute"
    echo "       src=<src>      build.sh source descriptor, default=F462" 
    echo "       rev=YYYYMMDD | rNNNN override revision tag, default is current svn rev number" 
    echo "       dtuser=<user>  user name for devtech linux network, default: $USER"
    echo "       cdeuser=<user> user name for CDE (Common Desktop Environment, default: $USER"
    echo "       flgs=<flags>   Passes <flags> to all descriptors"
    echo "       steps=all|build|check|test|install_tr"
    echo "         all:         do all steps, equivalent to steps=\"build check test install_tr\""
    echo "         build:       build specified targets"
    echo "         check:       check log files for errors"
    echo "         test:        prepare test dispatcher descriptors for specified targets"
    echo "         deja:        run dejagnu tests for specified targets"
    echo "         install_tr:  install build in CDE Tool Repository"
    echo "       targets=target_names ..."
    echo "          target_names are:"
    for i in ${!BLDTARGETS[@]}; do
        echo -n   "               $i ["
        targets=( $(echo ${BLDTARGETS[$i]}) )
        for j in ${targets[@]}; do
            echo -n "$j "
        done
        echo "]"
    done    
}

################################################################################
# Main 
################################################################################

#
# Scan all arguments
#
for arg in "$@"; do
    case $arg in 
        -h)
            do_help
            exit 1
            ;;
        -v)
            BLDVERBOSE=1 
            ;;
        -n)
            BLDDRYRUN=1 
            BLDSUBDRY='-n'
            ;;
	dtuser=*)
	    BLDDESC['dtuser']=${arg:7}
	    ;;
	cdeuser=*)
	    BLDDESC['cdeuser']=${arg:8}
	    ;;
        src=*)
            BLDDESC['src']=${arg:4}
            ;;
        rev=*)
            BLDrev=${arg:4}
            ;;
	targets=*)
	    BLDDESC['targets']=${arg:8}
	    ;;
	steps=*)
	    BLDDESC['steps']=${arg:6}
	    ;;
	flgs=*)
	    BLDDESC['flgs']=${arg:5}
	    ;;
        *)
            echo "Unknown option: $arg"
            exit 1
            ;;
    esac
done

#
# Show what we are going to do
#
if [ ${BLDVERBOSE} -eq 1 ]; then
    echo "===================="
    for i in ${!BLDDESC[@]}; do
        echo "$i: ${BLDDESC[$i]}"
    done
    echo "date: $TDY"
    echo "===================="
fi

#
# Execute it
#
do_steps
echo "#DONE"
