#!/bin/bash

print_usage () {
    # Display usage and exit

    me=`basename $0`
    cat <<EOF
$me is a tool to rework our collection of patches.

Arguments are:
   s=<source set> Set source descriptor to use
      This provides a list of tarballs and corresponding patches to use on them
      e.g.: s=F492
   t=<source tarball>
      Must be one of the sources listed in the descriptor/
      e.g.: t=binutils-2.25
   topo=[src | git | <list>]
      Order of branches to use in the commands:
         src -> Use original order found in <source set> (This is the default)
         git -> Let git compute topological order
         <list> -> Use given list
   <command>

Where <command> is one of:
   import
      This creates a new git repository in your current directory
      Will import the tarball of your choice.
      Apply all patches found in the descriptor.
      For each one will create a new branch with the same name as the patch
   export
   export=<path>
      Will make a diff for each branch found in the repository.
      The diff is saved inside <path> or directly into the build_gnu repository.
      Writes a file named from the branch name.
EOF
    exit 0
}

check_string () {
    # Arguments are
    local key=$1
    local choices=$2
    local k

    [ $verb -gt 1 ] && echo "check_string $key"
    for k in $choices; do
	if [ "$k" = "$key" ]; then
	    [ $verb -gt 1 ] && echo "check_string $key found"
	    return 1
	fi
    done
    [ $verb -gt 1 ] && echo "check_string FAIL"
    return 0
}

get_patch_prefix () {
    # Arguments are
    local t=$1

    # Set global variable PATCHPREFIX
    PATCHPREFIX=""
    case $t in
	gcc*)
	    PATCHPREFIX="gcc"
	    ;;
	gdb*)
	    PATCHPREFIX="gdb"
	    ;;
	binutils*)
	    PATCHPREFIX="bin"
	    ;;
	glibc*)
	    PATCHPREFIX="glibc"
	    ;;
	newlib*)
	    PATCHPREFIX="newlib"
	    ;;
    esac
}

gen_patch_list () {
    # Arguments are
    local list=$1

    local p

    # Uses global variable PATCHPREFIX
    # Sets global variable PATCHLIST
    PATCHLIST=""
    for p in $list; do
	k=`echo $p | awk '{split ($1,a,"."); print a[1]; exit}'`
	[ "$k" = "$PATCHPREFIX" ] && PATCHLIST="$PATCHLIST $p"
    done
    blist=$PATCHLIST
}

untar_any () {
    # Function to untar any file format

    # Arguments are:
    local tar=$1

    local b btgz bgz bz2 btar bxz btxz

    b=`basename $tar`
    btgz=`basename $b .tgz`
    bgz=`basename $b .gz`
    bz2=`basename $b .bz2`
    btar=`basename $b .tar`
    bxz=`basename $b .xz`
    btxz=`basename $b .txz`

    echo -n "   Expanding $tar  "
    if [ "$b" != "$btgz" -o "$b" != "$bgz" ];then
	tar xzfs $tar
	if [ $? != 0 ]; then echo "...Error"; return; else echo "...Ok"; fi
    elif [ "$b" != "$bz2" ]; then
	tar xjfs $tar
	if [ $? != 0 ]; then echo "...Error"; return; else echo "...Ok"; fi
    elif [ "$b" != "$btar" ]; then
	tar xfs $tar
	if [ $? != 0 ]; then echo "...Error"; return; else echo "...Ok"; fi
    elif [ "$b" != "$btxz" -o "$b" != "$xz" ];then
	tar xJfs $tar
	if [ $? != 0 ]; then echo "...Error"; return; else echo "...Ok"; fi
    else
	echo "Dont know how to extract: $tar"
	exit 1
    fi
}

insist_pkg () {
    # Function to try to find pkg in any possible compression type

    # Arguments are:
    # Package name
    local pkg=$1

    # Uses global variable REPODIR
    # Sets global variable PKG

    local e
    local extensions="tar.xz txz tar.gz tgz tar.bz2"

    PKG=""
    for e in $extensions; do
	if [ -f $REPODIR/pkg/$pkg.$e ]; then PKG="$REPODIR/pkg/$pkg.$e"; return 0; fi
    done
    return 1
}

prepare_sources () {
    # Function to get any sources

    # Arguments are:
    # Package name, if sucessfull a directory with same name is created
    local p=$1

    insist_pkg $p
    untar_any $PKG
}

check_arguments () {
    if [ "$SRCFRAG" = "" ]; then
	echo "Must set source fragment first"
	echo "##############################"
	print_usage
    fi
    if [ "$SRCTAR" = "" ]; then
	echo "Must pick source tarbal first"
	echo "#############################"
	echo "Pick one of:"
	echo -n "   "
	for f in ${SRCDESC[1]}; do echo -n -e "$f\t"; done
	echo ""
	exit 0
    fi
}

# Get our build repository
scriptdir=`dirname $0`
hnfull=`hostname`
hn=`basename $hnfull .am.freescale.net`
source $scriptdir/build.env-$hn
SRCFRAG=""
SRCTAR=""
PATCHPREFIX=""
PATCHLIST=""
verb=1
blist=""

# Parse command line
for a in "$@"; do
    case $a in
	-h)
	    print_usage
	    ;;
	-v)
	    let verb=$verb+1
	    echo "Verbose level: $verb"
	    ;;
	+v)
	    [ $verb -gt 0 ] && let verb=$verb-1 && echo "Verbose level: $verb"
	    ;;
	s=*)
	    SRCFRAG=${a:2}
	    source $REPODIR/build_gnu/f-sources/$SRCFRAG
	    [ $verb -gt 0 ] && echo "build_gnu at $REPODIR/build_gnu"
	    [ $verb -gt 0 ] && echo "Using source fragment:     ... $SRCFRAG"
	    [ $verb -gt 1 ] && for f in ${SRCDESC[1]}; do echo "   tarball:    $f"; done
	    [ $verb -gt 1 ] && for p in ${SRCDESC[2]}; do echo "   patch:      $p"; done
	    ;;
	t=*)
	    if [ "$SRCFRAG" = "" ]; then
		echo "Must set source fragment first"
		echo "##############################"
		print_usage
	    fi
	    # Check if given string is in source fragment
	    SRCTAR=${a:2}
	    check_string $SRCTAR "${SRCDESC[1]}"
	    [ "$?" != "1" ] && SRCTAR=""
	    if [ "$SRCTAR" = "" ]; then
		echo "Invalid tarball"
		echo "###############"
		echo "Pick one of:"
		echo -n "   "
		for f in ${SRCDESC[1]}; do echo -n -e "$f\t"; done
		echo ""
		exit 0
	    fi
	    get_patch_prefix $SRCTAR
	    if [ "$PATCHPREFIX" = "" ]; then
		echo "This tarball is not supported"
		echo "Pick one of:    gcc* gdb* binutils* glibc* newlib"
		exit 0
	    fi
	    gen_patch_list "${SRCDESC[2]}"
	    [ $verb -gt 0 ] && echo "Using source tarball:      ... $SRCTAR"
	    [ $verb -gt 0 ] && echo "Have patches:              ...$PATCHLIST"    
	    ;;
	import)
	    check_arguments
	    [ $verb -gt 0 ] && echo "Command import"
	    rm -rf $SRCTAR
	    prepare_sources $SRCTAR
	    cd $SRCTAR
	    git init -q
	    [ $? != 0 ] && echo "Error at git init" && exit 1
	    git add -f --all
	    [ $? != 0 ] && echo "Error at git initial add" && exit 1
	    git commit -q -m "Created $SRCTAR repo"
	    [ $? != 0 ] && echo "Error at git initial commit" && exit 1
	    # Apply patches one at a time, creating branches for each one"
	    for p in $PATCHLIST; do
		echo -n -e "   Creating branch for patch $p\t"
		git branch $p
		[ $? != 0 ] && echo "Error at git branch $p" && exit 1
		git checkout -q $p
		patch -p1 -s < $REPODIR/build_gnu/patches/$p
		[ $? != 0 ] && echo "Error at applying $p" && exit 1
		rm -f `find . -name '*orig'`
		[ $? != 0 ] && echo "Error at removing patch leftover" && exit 1
		git add --all
		[ $? != 0 ] && echo "Error at git add $p" && exit 1
		git commit -q -m "Import of patch $p"
		[ $? != 0 ] && echo "Error at git commit $p" && exit 1
		echo "...Ok"
	    done
	    cd ../
	    ;;
	topo=*)
	    if [ "$a" = "topo=src" ]; then
		echo "#### Using original order on source fragment ####"
		blist=$PATCHLIST
	    elif [ "$a" = "topo=git" ]; then
		echo "#### Using topological order git computed ####"
		blist=`git log --all --topo-order --reverse --format=%d | awk '{if ($1!="") if ($1!="(master)" && $3!="master)") {if ($1=="(HEAD") print substr($3,1,length($3)-1); else print substr($1,2,length($1)-2)}}'`
	    else
		echo "#### Using your topological order ####"
		blist=${a:5}
	    fi
	    ;;
	export | export=*)
	    check_arguments
	    if [ "$a" = "export" ]; then
		PATCHDIR="$REPODIR/build_gnu/patches"
	    else
		PATCHDIR="${a:7}"
	    fi
	    if [ ! -d $PATCHDIR ]; then
		echo "Not a directory: $PATCHDIR"
		echo "################"
		exit 1
	    fi
	    if [ "$blist" = "" ]; then
		echo "Must specify a topological order"
		echo "################################"
		exit 1
	    fi
	    [ $verb -gt 0 ] && echo "Command export              ... PATCHDIR=$PATCHDIR"
	    cd $SRCTAR

	    echo "Topological order:"
	    for b in $blist; do
		echo "   $b"
	    done
	    echo "List of diffs to be done:"
	    bprev="master"
	    for b in $blist; do
		echo -e "   git diff $bprev \t$b"
		bprev=$b
	    done
	    echo "Proceed ? (^C to abort)"
	    read -a LINE

	    git checkout -q master
	    bprev="master"
	    for b in $blist; do
		echo -n -e "git diff $bprev $b  \t"
		git diff $bprev $b > $PATCHDIR/$b
		if [ $? = 0 ]; then echo "...Ok"; else echo "...Error"; fi
		bprev=$b
	    done
	    cd ../
	    ;;
	rebase)
	    check_arguments
	    if [ "$blist" = "" ]; then
		echo "Must specify a topological order"
		echo "################################"
		exit 1
	    fi

	    this=`git branch | grep '*' | awk '{print $2}'`
	    echo "Topological order:"
	    for b in $blist; do
		echo "   $b"
	    done
	    echo "Current branch:"
	    echo "   $this"
	    echo "List of rebases to be done:"
	    found=NO
	    bprev="master"
	    for b in $blist; do
		if [ "$found" = "YES" -o "$this" = "master" ]; then
		    echo -e "   git rebase $bprev \t$b"
		fi
		if [ "$b" = "$this" ]; then
		    found=YES
		fi
		bprev=$b
	    done
	    echo "Proceed ? (^C to abort)"
	    read -a LINE

	    found=NO
	    bprev="master"
	    for b in $blist; do
		if [ "$found" = "YES" -o "$this" = "master" ]; then
		    echo -n "git rebase $bprev $b   "
		    git rebase -q $bprev $b
		    if [ $? = 0 ]; then echo "...Ok"; else echo "...Error"; exit 1; fi
		fi
		if [ "$b" = "$this" ]; then
		    found=YES
		fi
		bprev=$b
	    done
	    ;;
	*)
	    echo "Unkown command:            ... $a"
	    ;;
    esac
done
