#!/bin/bash

# TIPS:
# To push packages on remote STRs:
# put_pkg  /pkg/fs-DTgnu-/gcc-4.6.3-XMe6500-linux  -sites ZIN33,ONT20
#
# To find all tokens on MTR:
# /run/pkg/foundation-/frame/sbin/mtr -- cache -token fs-DTgnu-/gcc-4.6.3-XMe6500-linux
#
# To see the date in a token:
# echo <unixtime> | perl -e 'print localtime(<>) . "\n";'
# example:
# fs-DTgnu-/gcc-4.6.3-XMe5500-linux ra8797:tx32:cde-tx32.sps.mot.com:11800:1355166903
# unixtime is 1355166903
# date is     Mon Dec 10 13:15:03 2012

usage () {
    echo "Usage is: install_TR {CS|DT} <list of tar balls>"
    echo "      or: install_TR -remove {CS|DT} <list of tools>"
    echo "File name for tarball must have general format:"
    echo "    <prefix>-<host>-<date/revision>.tar.{gz|bz2|xz}"
    echo ""
    echo "If package that must be re-installed is already present in other STR:"
    echo "example: XMe6500 on ZIN33"
    echo "   TR support must remove it first, them:"
    echo "   ssh -l ra8797 log5-atx \"true && install_TR.sh DT  /proj/ppc/DT/labhome/edmar/build_repo/releases/CDE.gcc-4.6.3-XMe6500-linux-i686-linux-r721.tar.bz2\""
    echo "   push_more_sites fs-DTgnu-/gcc-4.6.3-XMe6500-linux -sites zin33,ont20,zil05"
}

remove_tool () {
    local typedir=$1
    local tooldir=$2

    echo -n "... Checking if tool still exists "
    if [ -e /pkg/$typedir/$tooldir ]; then echo "Yes"; else echo "No"; return; fi

    echo -n "... Getting real location of pkg  "
    linkdir=`readlink -e /pkg/$typedir/$tooldir`
    echo $linkdir

    echo -n "... Unregistering package         "
    sleep 10
    unregister_pkg $typedir/$tooldir > $WORKDIR/log.01.unregister_pkg 2>&1
    x1=`cat $WORKDIR/log.01.unregister_pkg | grep 'unregistered: '`
    if [ "$x1" = "unregistered: $typedir/$tooldir" ]; then echo OK; else echo Error; exit 1; fi

    echo -n "... Unregistering package (again) "
    sleep 10
    unregister_pkg $typedir/$tooldir > $WORKDIR/log.01.unregister_pkg 2>&1
    x1=`cat $WORKDIR/log.01.unregister_pkg | grep 'unregistered: '`
    if [ "$x1" = "unregistered: $typedir/$tooldir" ]; then echo OK; else echo Error; exit 1; fi

    echo -n "... Removing pkg from STR         "
    sleep 10
    delete_pkg $linkdir >  $WORKDIR/log.02.delete_pkg 2>&1
    x1=`cat $WORKDIR/log.02.delete_pkg | grep 'deleted: '`
    if [ "$x1" = "deleted: $linkdir" ]; then echo OK; else echo Error; exit 1; fi

    echo -n "... Removing pkg from MTR         "
    /run/pkg/foundation-/frame/sbin/mtr_delete_pkg $typedir/$tooldir > $WORKDIR/log.03.delete_mtr_pkg 2>&1
    x1=`cat $WORKDIR/log.03.delete_mtr_pkg | grep 'mtr_delete_pkg' | awk '{print $4}'`
    if [ "$x1" = "$typedir/$tooldir" ]; then echo OK; else echo Error; exit 1; fi

    sleep 30
}

TYPE=$1
shift
if [ "$TYPE" = "-remove" ]; then
    REMOVE="Y"
    TYPE=$1
    shift
fi
if [ "$TYPE" != "CS" -a "$TYPE" != "DT" ]; then
    usage
    exit 1
fi
if [ "$TYPE" = "DT" ]; then
    TYPEDIR=fs-DTgnu-
else
    TYPEDIR=fs-CSgnu-
fi
PATHORIG=$PATH

for t in "$@"; do
    echo "... Initializing temporary storage"
    cd ~/
    WORKDIR=/tmp/edmar/$TYPEDIR
    rm -rf $WORKDIR ~/installed_TR

    mkdir -p $WORKDIR
    cd $WORKDIR

    if [ "$REMOVE" = "Y" ]; then
	# Execute remotion of tools, and skip to next iteration
	remove_tool $TYPEDIR $t
	continue
    fi

    if [ ! -f $t ]; then
        echo "Cannot find file: $t"
        continue
    fi
    # Breakdown tarball file name, first with extension
    t1=`basename $t`
    t2=`basename $t1 .tar.gz`
    t3=`basename $t1 .tar.bz2`
    t4=`basename $t1 .tar.xz`
    tp=""
    if [ "$t2" != "$t1" ]; then
	# This is gzip
	TAR="xzf"
	tp=$t2
    elif [ "$t3" != "$t1" ]; then
	# This is bzip2
	TAR="xjf"
	tp=$t3
    elif [ "$t4" != "$t1" ]; then
	# This is xz
	TAR="xJf"
	tp=$t4
    else
	echo "I cannot process extension of this file: $t"
	continue
    fi
    echo "... Expanding: tar $TAR $t1"
    tar $TAR $t

    # Calculate TOOLDIR
    if [ "$TYPE" = "CS" ]; then
	# CS tools are named freescale-201x.xx-xx-<target1>-<target2>-<...>
	# We need the second to the fifth's parts of the array
	TOOLDIR=`echo $tp | awk '{split($1,a,"-");i=index($1,a[6]);print(substr($1,11,i-12))}'`
	HOSTDIR=i686-linux
	mkdir -p $TOOLDIR
	mv freescale-* $TOOLDIR/$HOSTDIR
    else
	# DT tools are named CDE.xxx
	TOOLDIR=`echo $tp | awk '{split($1,a,"-");i=index($1,a[5]);print(substr($1,5,i-6))}'`
	HOSTDIR=`echo $tp | awk '{split($1,a,"-");i=index($1,a[5]);j=index($1,a[7]);print(substr($1,i,j-i-1))}'`
        # Verify TOOLDIR
	if [ ! -d $TOOLDIR ]; then
	    echo "Could not extract tool name from tarball name $tp"
	    continue
	fi
        # Verify HOSTDIR
	if [ ! -d $TOOLDIR/$HOSTDIR ]; then
	    echo "Could not extract host dir from tarball name $tp"
	    continue
	fi
    fi
    echo "... Calculated: TOOLDIR=$TOOLDIR"
    echo "... Calculated: HOSTDIR=$HOSTDIR"

    # Calculate TOOLGCC
    TOOLGCC=`basename $TOOLDIR/*/bin/*-gcc`
    if [ "$TOOLGCC" = "*-gcc" ]; then
	echo "Could not find gcc binary: $TOOLDIR/*/bin/*-gcc"
	continue
    fi
    echo "... Calculated: TOOLGCC=$TOOLGCC"

    # Calculate TOOLPREFIX
    TOOLPREFIX=`echo $TOOLGCC | awk '{print(substr($1,1,length($1)-4))}'`
    echo "... Calculated: TOOLPREFIX=$TOOLPREFIX"

    echo "... Removing noise"
    rm -rf $TOOLDIR/i686-mingw32
    x=`find $TOOLDIR -name pt_chown`
    [ "$x" != "" ] && rm -f $x

    echo "... Creating env directory and files"
    mkdir -p $TOOLDIR/env
    if [ "$TYPE" = "DT" ]; then
    cat > $TOOLDIR/env/common.mtr <<EOF
# Generated by build.sh. Do not edit
prepend PATH <RUNROOT>/bin
prepend MANPATH <RUNROOT>/share/man
prepend INFOPATH <RUNROOT>/share/info
EOF
    else
    cat > $TOOLDIR/env/common.mtr <<EOF
# Generated by build.sh. Do not edit
prepend PATH <RUNROOT>/bin
prepend MANPATH <RUNROOT>/share/doc/freescale-$TOOLPREFIX/man
prepend INFOPATH <RUNROOT>/share/doc/freescale-$TOOLPREFIX/info
EOF
    fi
    touch $TOOLDIR/env/\#all

    echo -n "... Compiling environment         "
    cd $TOOLDIR
    compile_env . -o .
    if [ $? != 0 ]; then continue; fi
    cd ../

    echo -n "... Verifing tool installation    "
    check_pkg $WORKDIR/$TOOLDIR
    if [ $? = 0 ]; then echo "OK"; else echo "Error"; exit 1; fi

    cat <<EOF
    Calculated: TOOLDIR=$TOOLDIR
    Calculated: HOSTDIR=$HOSTDIR
    Calculated: TOOLGCC=$TOOLGCC
    Calculated: TOOLPREFIX=$TOOLPREFIX
Check all information above then press "return" for install_pkg
EOF
#   read DUMMY

    echo -n "... Check if re-installing        "
    x=`find_pkg -regex "^${TYPEDIR}/${TOOLDIR}\$"`
    if [ "$x" = "$TYPEDIR/$TOOLDIR" ]; then
	echo "Yes"
	REPLACE="--force"
	remove_tool $TYPEDIR $TOOLDIR
    else
	echo "No"
	REPLACE=""
    fi

    echo -n "... Installing package on STR     "
    install_pkg -replace $WORKDIR/$TOOLDIR > $WORKDIR/log.04.install_pkg 2>&1
    if [ $? = 0 ]; then echo "OK"; else echo "Error"; exit 1; fi

    echo -n "... Verifying log.04.install_pkg  "
    x1=`cat $WORKDIR/log.04.install_pkg | grep 'please wait while foundation permissions are updated...'`
    x2=`cat $WORKDIR/log.04.install_pkg | grep 'please wait while signatures are checked...'`
    x3=`cat $WORKDIR/log.04.install_pkg | grep 'generating certificate...'`
    x4=`cat $WORKDIR/log.04.install_pkg | grep 'installed: /pkg'`
    if [ "$x1" != "" -a "$x2" != "" -a "$x3" != "" -a "$x4" != "" ]; then
	echo "OK"
    else
	echo "Error"
	echo "x1=$x1"
	echo "x2=$x2"
	echo "x3=$x3"
	echo "x4=$x4"
	exit 1
    fi

    echo -n "... Registering package on STR    "
    register_pkg $TYPEDIR/$TOOLDIR > $WORKDIR/log.05.register_pkg 2>&1
    if [ $? = 0 ]; then echo "OK"; else echo "Error"; exit 1; fi

    echo -n "... Testing package on STR        "
    mkdir -p ~/installed_TR/$TYPEDIR/$TOOLDIR
    cd ~/installed_TR/$TYPEDIR/$TOOLDIR
    echo "$TYPEDIR/$TOOLDIR" > policy
    # Give time for server
    sleep 90
    collect ./policy > $WORKDIR/log.06.collect 2>&1
    x=`cat $WORKDIR/log.06.collect | grep 'Collection generation successful'`
    if [ "$x" != "" ]; then echo -n "OK "; else echo "Error"; exit 1; fi
    export PATH=~/installed_TR/$TYPEDIR/$TOOLDIR/$HOSTDIR/bin:$PATHORIG
    x=`$TOOLGCC -v 2>&1 | grep COLLECT_GCC`
    if [ "$x" = "COLLECT_GCC=/pkg/$TYPEDIR/$TOOLDIR/$HOSTDIR/bin/$TOOLGCC" ]; then 
	echo "OK"
    else 
	echo "Error"; exit 1
    fi

    echo -n "... Releasing package to MTR      "
    sleep 10
    put_pkg $REPLACE /pkg/$TYPEDIR/$TOOLDIR > $WORKDIR/log.07.put_pkg 2>&1
    if [ $? = 0 -o "$REPLACE" != "" ]; then echo "OK"; else echo "Error"; exit 1; fi

    off=`cat $WORKDIR/log.07.put_pkg | grep mtr_official | awk '{print substr($2,2),substr($3,1,length($3)-1)}'`
    if [ "$off" != "" ]; then
	echo "    [$off]"
	echo -n "... Making package official       "
	eval $off > $WORKDIR/log.08.mtr_official 2>&1
	if [ $? = 0 ]; then echo "OK"; else echo "Error"; exit 1; fi
    fi

    echo -n "... Cheking existance on MTR      "
    # Give time for server
    sleep 10
    x=`find_pkg -regex "^${TYPEDIR}/${TOOLDIR}\$"`
    if [ "$x" = "$TYPEDIR/$TOOLDIR" ]; then echo "OK"; else echo "Error"; exit 1; fi

    echo "DONE"
    echo ""
    echo ""
done
