	/* *** Copyright Freescale Semiconductor, Inc, 2010 *** */

/*****************************************************************************
*       Prototype emulation library.                                         *
*       Use of this is twofold:                                              *
*       1. verification of compiler prototypes                               *
*       2. host emulation of DSP functionality                               *
*****************************************************************************/

#include <prototype_sc3900.h>
#include <prototype_sc3900_emu.h>


#define TRUE 1
#define FALSE 0

/*****************************************************************************
*       Utility intrinsics must be defined in C when not building with       *
*       StarCore compiler                                                    *
*****************************************************************************/

#ifdef _ENTERPRISE_C_
/* These are defined in the compiler but need a newer build than we are using */
#define __l_put_msb(x)                ((__l_put_lsb(x)<<16)&0xFFFF0000)
#define __l_get_msb(x)                (short) ((x>>16))
#define __l_get_lsb(x)                (short) (x&0xffff)
#endif


#define __LL_get_lsb(x)               (Word32) (x&0xFFFFFFFF)
#define __LL_get_msb(x)               (Word32) ((x>>32)&0xFFFFFFFF)



/*****************************************************************************
*       Intrinsic Emulation Internals                                        *
*       DO NOT EDIT                                                          *
*****************************************************************************/

#define __ll_to_x(x) __x_set((x >> 32) & 0xff,(UWord32) x & 0xffffffff)

/*some declarations*/
Word32 do__extract_wh(Word40 __Da);
Word32 do__extract_wl(Word40 __Da);

/* Note: functions below are mapped to do_ versions for internal testing
   purposes                              */

/*set rounding mode*/
#define set2crm                           do__set2crm()
#define setcnvrm                          do__setcnvrm()
/*scaling off*/
#define setscoff                          do__setscoff()
/*scaling up 1 bit*/
#define setscup1                          do__setscup1()
/*scaling down 1 bit*/
#define setscdown1                        do__setscdown1()
/*scaling down 2 bits*/
#define setscdown2                        do__setscdown2()
/*scaling down 3 bits*/
#define setscdown3                        do__setscdown3()

#define __saturate(a)                     do__saturate(a)
#define __round_and_add_2t(a,b,c,d)       do__round_and_add_2t(a,b,c,d)
#define __round_and_add_2w(a,b,c,d)       do__round_and_add_2w(a,b,c,d)
#define __round_and_add_sat_2w(a,b,c,d)   do__round_and_add_sat_2w(a,b,c,d)
#define __sub_sod(a,b)                    do__sub_sod(a,b)
#define __add_sod(a,b)                    do__add_sod(a,b)
#define __ssodffcc_low(a,b,c)             do__ssodffcc_low(a,b,c)
#define __ssodffcc_high(a,b,c)            do__ssodffcc_high(a,b,c)
#define __x_to_ll(x)                      do__x_to_ll(x)
#define __set_WH(a,b)                     do__set_WH(a,b)
#define __set_WL(a,b)                     do__set_WL(a,b)
#define __X_scale(x)                      do__X_scale(x)
#define __X_scale_sat(x)                  do__X_scale_sat(x)
#define __x_round(x)                      do__x_round(x)
#define __swap_bits(x)                    do__swap_bits(x)
#define __set_u_WH(a,b)                   do__set_u_WH(a,b)
#define __set_u_WL(a,b)                   do__set_u_WL(a,b)
#define __extract_wh(x)                   do__extract_wh(x)
#define __extract_wl(x)                   do__extract_wl(x)
#define __neg_sat8(x)                     do__neg_sat8(x)


/*****************************************************************************
 * Status register modeling
 *****************************************************************************/

/* User Status Register mask */
/* Bits accessible in user privilege level. SC3850 privilege bits are moved to SR2 in SC3900*/
#define USR__MASK                0x003e3fff
/* Carry flag mask */
#define SR_C__MASK               0x1  /* same as that of SC3850 */
/* Rounding Mode mask */
#define SR_RM__MASK              0x8  /* same as that of SC3850 */
/* Scaling Mode mask */
#define SR_SCM__MASK             0x130 /* SCM bits are scattered in SC3900 */
#define SR_SCM__MASK_1           0x30
#define SR_SCM__MASK_2           0x100

/* 32-bit Saturation Mode mask */
#define SR_SM__MASK              0x4
/* 16-bit Saturation Mode mask */
#define SR_SM2__MASK             0x80
#define SR_W20__MASK             0x1000

#define SR_SCM_DOWN_1__MASK      0x10  /* same as that of SC3850 */
#define SR_SCM_UP_1__MASK        0x20  /* same as that of SC3850 */
#define SR_SCM_DOWN_2__MASK      0x30  /* added new for SC3900 */
#define SR_SCM_DOWN_3__MASK      0x100 /* added new for SC3900 */

/*W30 sm2 mask */
#define SM2_W20__MASK_1          0x80
#define SM2_W20__MASK_2          0x1000

/*Saturation flag mask*/
#define SAT_BIT_MASK			 0x800
/* Overflow flag index in Exception and Mode Register (EMR.DOVF) */
/* #define EMR_DOVF__INDEX          2*/
/* EMR register is removed in SC3900.In SC3900 it is replaced
   by a new bit in SR (the OVF bit, in SR.10), which for legacy instructions is set in the same
   conditions as the DOVF bit was set in SC3850 (with the exception of two cases described in
   Section 9.3.4.4, Differences in the DOVF status). SR.OVF however is cleared by writing  to it.
   The mapper will replace EMR accesses with an emulation sequence that in case of an EMR read returns a
   compatible result for these two bits alone; In case of a write, clear the SR.OVF bit in the same conditions
   it would have been cleared in EMR.*/

#define SR_OVF_INDEX 0xA

int SR = 0x00000008;         /* Status Register: In 3850, only RM and SM bits are set(0x0002000c). With 3900, RM is set and SM unset */

Flag Sat = 0;
Flag Overflow = 0;           /* emulates EMR.DOVF : Not present in SC3900 */
Flag Carry = 0;              /* emulates SR.C */
Flag SaturationMode = 1;     /* emulates SR.SM */
Flag shift_Overflow = 0;     /*Not related to SR. used for intermediate shift overflows in shift instr*/
int sm2_w20_mode =0;

enum SCALING_MODE
{
	NO_SCALING,
	DOWN_1,
	UP_1,
	DOWN_2,
	DOWN_3 /* Added for SC3900 */
};
enum SCALING_MODE ScalingMode = NO_SCALING;			/* default according to the ABI: Retained same for SC3900 */

enum ROUNDING_MODE
{
	CONVERGENT,
	_2S_COMPLEMENT
}; /* Retained same for SC3900 */

/*
 16-bit saturation and wide mode are linked in such a way that
 they cannot be set simultaneously; the valid modes are the following:
	W20(0) SM2(0)  - 16-bit with no saturation (Sat16WideMode = 0 = _16BITS_NOSAT)
	W20(0) SM2(1)  - 16-bit with saturation (Sat16WideMode = 1 = _16BITS_SAT)
	W20(1) SM2(0)  - 20-bit with no saturation (Sat16WideMode = 2 = _20BITS_NOSAT)
*/
enum SAT16_WIDE_MODE
{
	_16BITS_NOSAT,
	_16BITS_SAT,
	_20BITS_NOSAT
};
Flag Sat16WideMode = _16BITS_NOSAT;	/* emulates the {SR.W20, SR.SM2} pair of bits */

enum ROUNDING_MODE RoundingMode = _2S_COMPLEMENT;	/* default according to the ABI */

int AS0 = 0, AS1 = 0, AS2 = 0, AS3 = 0;	/* AS0-3 bits in GCR representing carry values set in divp0-3 prototypes , same as that of SC3850*/

void /*updateEMR()*/ updateSR_OVF()
{
  SR = (Overflow) ? (1 << SR_OVF_INDEX) : (0);
}

inline void update_carry()
{
	Carry = (SR & SR_C__MASK) ? 1: 0;
}

inline void update_scaling_mode()
{
	ScalingMode = (enum SCALING_MODE) ((SR & SR_SCM__MASK_1) >> 4);
	ScalingMode |= (enum SCALING_MODE) ((SR & SR_SCM__MASK_2) >> 6);

}

inline void update_rounding_mode()
{
	RoundingMode = (SR & SR_RM__MASK) ? _2S_COMPLEMENT : CONVERGENT;
}

void updateSR()
{
        /* update carry */
	if (Carry)
		SR |= SR_C__MASK;
	else
		SR &= ~SR_C__MASK;

	/* update saturation mode */
	if (SaturationMode)
		SR |= SR_SM__MASK;
	else
		SR &= ~SR_SM__MASK;

	/* update rounding mode */
	if (RoundingMode == _2S_COMPLEMENT)
		SR |= SR_RM__MASK;
	else
		SR &= ~SR_RM__MASK;

	/* update scaling mode */
	switch (ScalingMode)
	{
		case NO_SCALING:
			SR &= ~SR_SCM__MASK;
			break;
		case DOWN_1:
			SR = (SR | SR_SCM_DOWN_1__MASK) & (~SR_SCM_DOWN_1__MASK);
			break;
		case UP_1:
			SR = (SR | SR_SCM_UP_1__MASK) & (~SR_SCM_UP_1__MASK);
			break;
		case DOWN_2:
			SR = (SR | SR_SCM_DOWN_2__MASK) & (~SR_SCM_DOWN_2__MASK);
			break;
                case DOWN_3:
			SR = (SR | SR_SCM_DOWN_3__MASK) & (~SR_SCM_DOWN_3__MASK);
			break;

	}

	/*  update wide mode and 16-bit saturation mode */
	if (Sat16WideMode == _16BITS_NOSAT)
	{
		SR = SR & ~(SR_SM2__MASK | SM2_W20__MASK_1);
	}
	else if (Sat16WideMode == _16BITS_SAT)
	{
		SR = (SR | SR_SM2__MASK) & (~SM2_W20__MASK_1);
	}
	else if (Sat16WideMode == _20BITS_NOSAT)
	{
		SR = (SR | SM2_W20__MASK_2) & (~SM2_W20__MASK_2);
	}
}

void update_sat16_wide_mode()
{
	if (!(SR & SR_SM2__MASK) && !(SR & SR_W20__MASK))
	{
		Sat16WideMode = _16BITS_NOSAT;
	}
	else if (!(SR & SR_W20__MASK))
	{
		Sat16WideMode = _16BITS_SAT;
	}
	else if (!(SR & SR_SM2__MASK))
	{
		Sat16WideMode = _20BITS_NOSAT;
	}
}

void do__setnosat16() {
	SR = SR & (~ SR_SM2__MASK);
	update_sat16_wide_mode();
}

void do__setsat16() {
	SR = SR | SR_SM2__MASK;
	update_sat16_wide_mode();
}

void do__setwide20() {
	SR = SR | SR_W20__MASK;
	update_sat16_wide_mode();
}

void do__setnowide20() {
	SR = SR & (~SR_W20__MASK);
	update_sat16_wide_mode();
}

/*void clras0() { AS0 = 0; }
  void clras1() { AS1 = 0; }
  void clras2() { AS2 = 0; }
  void clras3() { AS3 = 0; } */

/*******************************************************************************************/
/*      SC3900 Helper Intrinsics                                                           */
/*******************************************************************************************/

#ifdef _HOST_EMULATION_

/* set to C equivalent */

inline Word40 __x_set(Word32 val1, UWord32 val2) {
    Word40 x;
  x.body=val2;
  x.ext=val1;
  return (x);
}

inline UWord40 __ux_set(UWord32 val1, UWord32 val2) {
  UWord40 x;
  x.body=val2;
  x.ext=val1;
  return (x);
}

inline Word32 __x_guard(Word40 val) {
  return(val.ext);
}
inline UWord32 __ux_guard(UWord40 val) {
  return (val.ext);
}

inline Word32 __x_to_l(Word40 val) {
  return (val.body);
}

inline UWord32 __ux_to_ul(UWord40 val) {
  return (val.body);
}

inline Word40 __l_to_x(Word32 val) {
int sign = val>>31;
  Word40 ret;

  if (sign && 0x1)
    ret.ext=0xF;
  else
    ret.ext=0x0;
  ret.body=val;
  return (ret);
}

inline Word32 __Db_get_msb(double Val)
{
	union {
		Int64 L;
		double D;
	} X;
	X.D = Val;

	return (X.L>>32);
}

inline UWord32 __Db_get_lsb(double Val)
{
	union {
		Int64 L;
		double D;
	} X;
	X.D = Val;
	return  (UWord32)(X.L);
}


#define __l_put_lsb(x)                (short) (x)
#define __l_get_lsb(x)                (short) (x&0xffff)
#define __l_put_msb(x)                ((__l_put_lsb(x)<<16)&0xFFFF0000)
#define __l_get_msb(x)                (short) ((x>>16))

extern UWord40 __x_to_ux(Word40 val);

extern Word40 __ux_to_x(UWord40 val);


/* if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_) */
/* Word32 __LL_get_msb(Int64 val); */
/* Word32 __LL_get_lsb(Int64 val); */
/* long long __Lli_set(Word32 val1, UWord32 val2); */
/* double __Db_set(Word32 val1, UWord32 val2); */
/* #endif */

#endif


/*******************************************************************************
 * Intrinsic Emulation Utility Functions
 *******************************************************************************/

Word16 do__saturate (Word32 L_var1)
{
    Word16 var_out;

    if (L_var1 > 0X00007fffL)
    {
		Overflow = 1;
        var_out = MAX_16;
    }
    else if (L_var1 < (Word32) 0xffff8000L)
    {
		Overflow = 1;
        var_out = MIN_16;
    }
    else
    {
        var_out = __l_get_lsb (L_var1);
    }

    return (var_out);
}


Word40 do__sat_x_hw(Word40 __Da)
{
  Word40 Dn_1;

  Dn_1 = do__cmpd_gt_x(__Da, __x_set(0x0, 0x0000007f));
   if ((__x_guard(Dn_1) == 0) && (__x_to_l(Dn_1) == 0))
    return (__x_set(0x00, 0x0000007F));

  Dn_1 =   do__cmpd_gt_x(__x_set(0xff, 0xffffff80), __Da);
   if ((__x_guard(Dn_1) == 0) && (__x_to_l(Dn_1) == 0))
    return (__x_set(0xFF, 0xFFFFFF80));

  return __Da;
}

Word40 do__round_and_add_2t(Word40 m1, Word40 m2, Word40 Da,Word8 hh)
{
	Word40 m;
	Word32 tmp, Da_WH, Da_WL;
	m = do__add_x(do__add_x(m1, m2) , __x_set(0, 0x8000));
	m = do__ash_rgt_x(16,m);
	if(hh)
	{
		m = do__add_x(__l_to_x(__extract_wh(Da)), m);
		tmp = (__x_to_l(m) & 0xFFFFF);
		tmp |= ((tmp >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
		Da_WH = (__x_to_l(Da) & 0xFFFF) | (tmp << 16) & 0xffff0000;
		Da = __x_set((__x_guard(Da) & 0x0F) | (((tmp & 0xF0000)>>12)&0xF0), Da_WH);
	}
	else
	{
		m = do__add_x(__l_to_x(__extract_wl(Da)), m);
		tmp = (__x_to_l(m) & 0xFFFFF);
		tmp |= ((tmp >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
		Da_WL = (__x_to_l(Da) & 0xFFFF0000) | (tmp & 0xffff);
		Da = __x_set((__x_guard(Da) & 0xF0) | (((tmp & 0xF0000)>>16)&0xF), Da_WL);
	}

	return Da;
}

Word32 do__round_and_add_sat_2w(Word40 m1, Word40 m2, Word32 Da,Word8 hh)
{
	Word40 m;
	Word16 Da_H, Da_L;
	m = do__add_x(do__add_x(m1, m2) , __x_set(0, 0x8000));
	m = do__ash_rgt_x(16,m);
	if(hh)
	{
		m = do__add_x(__l_to_x(__l_get_msb(Da)), m);
		/*Da_H = __saturate(__x_to_l(m)); */
		Da_H = __x_to_l(do__sat_x_w(m));
		Da = (Da & 0xffff) | ((Da_H << 16)& 0xffff0000);
	}
	else
	{
		m = do__add_x(__l_to_x(__l_get_lsb(Da)), m);
		/*Da_L = __saturate(__x_to_l(m)); */
		Da_L = __x_to_l(do__sat_x_w(m));
		Da = (Da & 0xffff0000) |(Da_L & 0xffff);;
	}
	return Da;
}


Word32 do__sub_sod ( Word32 a_ext , Word32 b_ext) {
    Word32 tmp , result ;
   tmp = result = b_ext - a_ext ;
   result = (Word32)(__saturate( result));
   if ( result == tmp ) {
       result = ( b_ext & 0x0ffff ) - ( a_ext & 0x0ffff ) ;
   }
   return result & 0xfffff;
}

Word32 do__add_sod ( Word32 a_ext , Word32 b_ext) {
  Word32 tmp , result ;

  tmp = result = a_ext + b_ext ;
  result = (Word32)(__saturate( result));
  if ( result == tmp ) {
    b_ext |= 0x0f0000 ;
    result = ( a_ext & 0x0ffff ) + b_ext ;
  }
  return result & 0xfffff;
}


Word16 do__ssodffcc_low ( Word32 __Da , Word32 __Db , char config ) {
  Word16 low_calc ;
  int always_sat = 1 ;
  switch ((config & 0x7)) {
    case 0 : low_calc = __add_sod ( (Word32)(__l_get_lsb(__Da)), (Word32)(__l_get_lsb(__Db))) ;
    break ;
    case 1 : low_calc = __add_sod ( (Word32)(__l_get_msb(__Da)), (Word32)(__l_get_lsb(__Db))) ;
    break ;
    case 2 : low_calc = __sub_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_lsb(__Db))  ) ;
    break ;
    case 3 : low_calc = __add_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_lsb(__Db))  ) ;
    break ;
    case 4 : low_calc = __add_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_lsb(__Db))  ) ;
    break ;
    case 5 : low_calc = __sub_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_lsb(__Db))  ) ;
    break ;
    case 6 : low_calc = __sub_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_lsb(__Db))  ) ;
    break ;
    case 7 : low_calc = __sub_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_lsb(__Db))  ) ;
    break ;
    default :
    break ;
  }
  return low_calc & 0xffff ;
}


Word16 do__ssodffcc_high ( Word32 __Da , Word32 __Db , char config  ) {
  Word16 high_calc ;
  int always_sat = 1 ;
  switch ( (config & 0x7)) {
    case 0 : high_calc = __add_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 1 : high_calc = __add_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 2 : high_calc = __add_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 3 : high_calc = __sub_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 4 : high_calc = __sub_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 5 : high_calc = __add_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 6 : high_calc = __sub_sod ( (Word32)(__l_get_msb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    case 7 : high_calc = __sub_sod ( (Word32)(__l_get_lsb(__Da)) , (Word32)(__l_get_msb(__Db))  ) ;
    break ;
    default :
    break ;
  }
  return high_calc & 0xffff;
}


Word8 do__neg_sat8(Word8 __Da)
{
	Word8 __Dn = 0;
		/*sat8 is only for 0x80 ->0x7F. since negative of 0x80 will be 0x00 which is overflow*/
	if((__Da&0xFF) == 0x80)return 0x7F;
	else
	return (__Dn - __Da);
}


Word32 do__extract_wh(Word40 __Da)
{
	Word32 __Dn = 0;

	__Dn = (((__x_to_l(__Da) & 0xFFFF0000) >> 16)&0xFFFF) | (((__x_guard(__Da) >> 4) << 16) & 0xf0000);
	if(__x_guard(__Da) & 0x80) __Dn = __Dn|0xfff00000;
	return __Dn;
}

Word32 do__extract_wl(Word40 __Da)
{
	Word32 __Dn = 0;

	__Dn = (__x_to_l(__Da) & 0xffff) | ((((__x_guard(__Da) << 4) >> 4) << 16) & 0xf0000);
	if(__x_guard(__Da) & 0x08) __Dn = __Dn|0xfff00000;
	return __Dn;
}

void do__set_u_WH(UWord32 __Da,UWord40 *__Dn)
{
	UWord32 body = 0;
	UWord8 ext=0;

	ext = (__ux_guard(*__Dn)&0x0f);
	ext = ext | (UWord8)(((__Da & 0xf0000)>>16)<<4);

	body = (__ux_to_ul(*__Dn)&0xffff);
	body = body | ((__Da & 0xffff)<<16);

	*__Dn = __ux_set(ext,body);
}

void do__set_u_WL(UWord32 __Da,UWord40 *__Dn)
{
	UWord32 body = 0;
	UWord8 ext=0;

	ext = (__ux_guard(*__Dn)&0xf0);
	ext = ext | (UWord8)(((__Da & 0xf0000)>>16));

	body = (__ux_to_ul(*__Dn)&0xffff0000);
	body = body | ((__Da & 0xffff));

	*__Dn = __ux_set(ext,body);
}


void do__set_WH(Word32 __Da,Word40 *__Dn)
{
	Word32 body = 0;
	Word8 ext=0;

	ext = (__x_guard(*__Dn)&0x0f);
	ext = ext | (Word8)(((__Da & 0xf0000)>>16)<<4);

	body = (__x_to_l(*__Dn)&0xffff);
	body = body | ((__Da & 0xffff)<<16);

	*__Dn = __x_set(ext,body);
}

void do__set_WL(Word32 __Da,Word40 *__Dn)
{
	Word32 body = 0;
	Word8 ext=0;

	ext = (__x_guard(*__Dn)&0xf0);
	ext = ext | (Word8)(((__Da & 0xf0000)>>16));

	body = (__x_to_l(*__Dn)&0xffff0000);
	body = body | ((__Da & 0xffff));

	*__Dn = __x_set(ext,body);
}

Word40 do__X_scale(Word40 X_src)
{
	Word40 X_scaled;

	switch(ScalingMode)
	{
		case UP_1:
			X_scaled = do__ash_lft_x(1,X_src);
			break;
		case DOWN_1:
			X_scaled = do__ash_rgt_x(1,X_src);
			break;
		case DOWN_2:
			X_scaled = do__ash_rgt_x(2,X_src);
			break;
		default:
			X_scaled = X_src;
	}
	return X_scaled;
}

static Word40 do__X_scale_sat(Word40 src)
{
	Word40 dest = src;
	Word32 shift_width;
	if (ScalingMode == UP_1)
	{
		// scale up one bit, careful with sign switching
		int lead_bits = (__x_guard(src) >> 6) & 0x03;
		if (lead_bits == 1)
			dest = __x_set(0, 0x7fffffff);
		else if (lead_bits == 2)
			dest = __x_set(0xFF, 0x80000000);
		else
			dest = do__ash_lft_x(1,src);
	}
	else if ((ScalingMode == DOWN_1) || (ScalingMode == DOWN_2) || (ScalingMode == DOWN_3))
	{
		// scale down one or two bits
		shift_width = (ScalingMode == DOWN_1) ? 1 :(ScalingMode == DOWN_2)? 2:3;
		dest = do__ash_rgt_x(shift_width , src);
	}
	return dest;
}

long long do__x_to_ll(Word40 __Dn)
{
	long long var1LL;
	var1LL  = __x_guard(__Dn);
	var1LL = (var1LL << 32) | (UWord32) __x_to_l(__Dn);
	return var1LL;
}

UWord8 do__swap_bits(UWord8 __Da)
{
	UWord8 result=0,i;
	/*now swap the bits*/
	for(i=0;i<8;i++)
	{
		result = result | ((__Da>>i & 1)<< (7-i));
	}
	return result;
}

Word40 do__x_round(Word40 __Dn)
{
	Word32 __Rn = __x_to_l(__Dn);
	Word40 Rounding_value; /*  used to change the rounding position based scaling mode later*/
	Word40 result;

	/*For now support for no scaling*/
	Rounding_value = __x_set(0,0x8000);
	if(RoundingMode == CONVERGENT)
	{
		/* Convergent rounding mode:
		  If Dn.L >1/2 then add 1 to Dn.H and clear Dn.L (round up)
		  If Dn.L <1/2 then add nothing to Dn.H and clear Dn.L(round down)
		  If Dn.L == 1/2 then if Dn.H is odd then round up else round down
		  */
	  if(__l_get_lsb(__Rn)>__x_to_l(Rounding_value))
	    result = do__add_x(__Dn,Rounding_value);/*check if this is correct*/
	  else
	    if(__l_get_lsb(__Rn)<__x_to_l(Rounding_value))
	      {
	      /* do nothing */
	      }
	    else
	      {
		result = (__Rn&0x00010000)?do__add_x(__Dn,Rounding_value):__Dn;
	      }
	}
	else
	{
		/*Two's compliement rounding mode
		  If Dn.L >=1/2 then add 1 to Dn.H(add 1 to rounding position,i.e 16th position) and clear Dn.L (round up)
		  If Dn.L <1/2 then add nothing to Dn.H(add 1 to rounding position ?) and clear Dn.L(round down)
		 */
	  /*if(__l_get_lsb(__Dn)>=0x8000U)__Rn = (__Dn + Rounding_value); -- check if this is correct*/
		result = do__add_x(__Dn,Rounding_value);

		/*  __Rn = (__Dn + 0x8000); check if this is correct*/

	}
	return result;
}

Word16 get_invsqrt_LUT_A(Word8 inp)
{
	Word16 INVSQRT_LUT_A = 0;
	switch(inp)
	{
       case 0 : INVSQRT_LUT_A = 0x05871;
       break;
       case 1 : INVSQRT_LUT_A = 0x05483;
       break;
       case 2 : INVSQRT_LUT_A = 0x050DD;
       break;
       case 3 : INVSQRT_LUT_A = 0x04D78;
       break;
       case 4 : INVSQRT_LUT_A = 0x04A4F;
       break;
       case 5 : INVSQRT_LUT_A = 0x0475B;
       break;
       case 6 : INVSQRT_LUT_A = 0x04498;
       break;
       case 7 : INVSQRT_LUT_A = 0x04201;
       break;
       case 8 : INVSQRT_LUT_A = 0x03F93;
       break;
       case 9 : INVSQRT_LUT_A = 0x03D4A;
       break;
       case 10 : INVSQRT_LUT_A = 0x03B24;
       break;
       case 11 : INVSQRT_LUT_A = 0x0391C;
       break;
       case 12 : INVSQRT_LUT_A = 0x03732;
       break;
       case 13 : INVSQRT_LUT_A = 0x03563;
       break;
       case 14 : INVSQRT_LUT_A = 0x033AD;
       break;
       case 15 : INVSQRT_LUT_A = 0x0320D;
       break;
       case 16 : INVSQRT_LUT_A = 0x03083;
       break;
       case 17 : INVSQRT_LUT_A = 0x02F0C;
       break;
       case 18 : INVSQRT_LUT_A = 0x02DA8;
       break;
       case 19 : INVSQRT_LUT_A = 0x02C55;
       break;
       case 20 : INVSQRT_LUT_A = 0x02B13;
       break;
       case 21 : INVSQRT_LUT_A = 0x029DF;
       break;
       case 22 : INVSQRT_LUT_A = 0x028B9;
       break;
       case 23 : INVSQRT_LUT_A = 0x027A1;
       break;
       case 24 : INVSQRT_LUT_A = 0x02695;
       break;
       case 25 : INVSQRT_LUT_A = 0x02594;
       break;
       case 26 : INVSQRT_LUT_A = 0x0249E;
       break;
       case 27 : INVSQRT_LUT_A = 0x023B3;
       break;
       case 28 : INVSQRT_LUT_A = 0x022D1;
       break;
       case 29 : INVSQRT_LUT_A = 0x021F9;
       break;
       case 30 : INVSQRT_LUT_A = 0x02129;
       break;
       case 31 : INVSQRT_LUT_A = 0x02061;
       break;
       case 32 : INVSQRT_LUT_A = 0x0FA27;
       break;
       case 33 : INVSQRT_LUT_A = 0x0EF08;
       break;
       case 34 : INVSQRT_LUT_A = 0x0E4B7;
       break;
       case 35 : INVSQRT_LUT_A = 0x0DB1E;
       break;
       case 36 : INVSQRT_LUT_A = 0x0D22C;
       break;
       case 37 : INVSQRT_LUT_A = 0x0C9D2;
       break;
       case 38 : INVSQRT_LUT_A = 0x0C202;
       break;
       case 39 : INVSQRT_LUT_A = 0x0BAB0;
       break;
       case 40 : INVSQRT_LUT_A = 0x0B3D0;
       break;
       case 41 : INVSQRT_LUT_A = 0x0AD5A;
       break;
       case 42 : INVSQRT_LUT_A = 0x0A745;
       break;
       case 43 : INVSQRT_LUT_A = 0x0A189;
       break;
       case 44 : INVSQRT_LUT_A = 0x09C1F;
       break;
       case 45 : INVSQRT_LUT_A = 0x09700;
       break;
       case 46 : INVSQRT_LUT_A = 0x09228;
       break;
       case 47 : INVSQRT_LUT_A = 0x08D91;
       break;
       case 48 : INVSQRT_LUT_A = 0x08935;
       break;
       case 49 : INVSQRT_LUT_A = 0x08512;
       break;
       case 50 : INVSQRT_LUT_A = 0x08123;
       break;
       case 51 : INVSQRT_LUT_A = 0x07D65;
       break;
       case 52 : INVSQRT_LUT_A = 0x079D4;
       break;
       case 53 : INVSQRT_LUT_A = 0x0766E;
       break;
       case 54 : INVSQRT_LUT_A = 0x0732F;
       break;
       case 55 : INVSQRT_LUT_A = 0x07016;
       break;
       case 56 : INVSQRT_LUT_A = 0x06D1F;
       break;
       case 57 : INVSQRT_LUT_A = 0x06A4A;
       break;
       case 58 : INVSQRT_LUT_A = 0x06793;
       break;
       case 59 : INVSQRT_LUT_A = 0x064F9;
       break;
       case 60 : INVSQRT_LUT_A = 0x0627B;
       break;
       case 61 : INVSQRT_LUT_A = 0x06017;
       break;
       case 62 : INVSQRT_LUT_A = 0x05DCB;
       break;
       case 63 : INVSQRT_LUT_A = 0x05B96;
       break;
       default :
       break;
	}
	return INVSQRT_LUT_A;
}

Word16 get_invsqrt_LUT_B(Word8 inp)
{
	Word16 INVSQRT_LUT_B = 0;
	switch(inp)
	{
       case 0 : INVSQRT_LUT_B = 0x086BA;
       break;
       case 1 : INVSQRT_LUT_B = 0x084B3;
       break;
       case 2 : INVSQRT_LUT_B = 0x082C3;
       break;
       case 3 : INVSQRT_LUT_B = 0x080E8;
       break;
       case 4 : INVSQRT_LUT_B = 0x07F21;
       break;
       case 5 : INVSQRT_LUT_B = 0x07D6C;
       break;
       case 6 : INVSQRT_LUT_B = 0x07BC8;
       break;
       case 7 : INVSQRT_LUT_B = 0x07A34;
       break;
       case 8 : INVSQRT_LUT_B = 0x078B0;
       break;
       case 9 : INVSQRT_LUT_B = 0x07739;
       break;
       case 10 : INVSQRT_LUT_B = 0x075D0;
       break;
       case 11 : INVSQRT_LUT_B = 0x07473;
       break;
       case 12 : INVSQRT_LUT_B = 0x07322;
       break;
       case 13 : INVSQRT_LUT_B = 0x071DC;
       break;
       case 14 : INVSQRT_LUT_B = 0x070A1;
       break;
       case 15 : INVSQRT_LUT_B = 0x06F70;
       break;
       case 16 : INVSQRT_LUT_B = 0x06E48;
       break;
       case 17 : INVSQRT_LUT_B = 0x06D2A;
       break;
       case 18 : INVSQRT_LUT_B = 0x06C13;
       break;
       case 19 : INVSQRT_LUT_B = 0x06B06;
       break;
       case 20 : INVSQRT_LUT_B = 0x069FF;
       break;
       case 21 : INVSQRT_LUT_B = 0x06900;
       break;
       case 22 : INVSQRT_LUT_B = 0x06809;
       break;
       case 23 : INVSQRT_LUT_B = 0x06718;
       break;
       case 24 : INVSQRT_LUT_B = 0x0662D;
       break;
       case 25 : INVSQRT_LUT_B = 0x06549;
       break;
       case 26 : INVSQRT_LUT_B = 0x0646A;
       break;
       case 27 : INVSQRT_LUT_B = 0x06391;
       break;
       case 28 : INVSQRT_LUT_B = 0x062BD;
       break;
       case 29 : INVSQRT_LUT_B = 0x061EF;
       break;
       case 30 : INVSQRT_LUT_B = 0x06126;
       break;
       case 31 : INVSQRT_LUT_B = 0x06061;
       break;
       case 32 : INVSQRT_LUT_B = 0x0BE88;
       break;
       case 33 : INVSQRT_LUT_B = 0x0BBAB;
       break;
       case 34 : INVSQRT_LUT_B = 0x0B8ED;
       break;
       case 35 : INVSQRT_LUT_B = 0x0B64D;
       break;
       case 36 : INVSQRT_LUT_B = 0x0B3C9;
       break;
       case 37 : INVSQRT_LUT_B = 0x0B15F;
       break;
       case 38 : INVSQRT_LUT_B = 0x0AF0E;
       break;
       case 39 : INVSQRT_LUT_B = 0x0ACD3;
       break;
       case 40 : INVSQRT_LUT_B = 0x0AAAD;
       break;
       case 41 : INVSQRT_LUT_B = 0x0A89B;
       break;
       case 42 : INVSQRT_LUT_B = 0x0A69C;
       break;
       case 43 : INVSQRT_LUT_B = 0x0A4AF;
       break;
       case 44 : INVSQRT_LUT_B = 0x0A2D3;
       break;
       case 45 : INVSQRT_LUT_B = 0x0A106;
       break;
       case 46 : INVSQRT_LUT_B = 0x09F48;
       break;
       case 47 : INVSQRT_LUT_B = 0x09D99;
       break;
       case 48 : INVSQRT_LUT_B = 0x09BF6;
       break;
       case 49 : INVSQRT_LUT_B = 0x09A61;
       break;
       case 50 : INVSQRT_LUT_B = 0x098D8;
       break;
       case 51 : INVSQRT_LUT_B = 0x0975A;
       break;
       case 52 : INVSQRT_LUT_B = 0x095E7;
       break;
       case 53 : INVSQRT_LUT_B = 0x0947F;
       break;
       case 54 : INVSQRT_LUT_B = 0x09320;
       break;
       case 55 : INVSQRT_LUT_B = 0x091CB;
       break;
       case 56 : INVSQRT_LUT_B = 0x09080;
       break;
       case 57 : INVSQRT_LUT_B = 0x08F3D;
       break;
       case 58 : INVSQRT_LUT_B = 0x08E02;
       break;
       case 59 : INVSQRT_LUT_B = 0x08CCF;
       break;
       case 60 : INVSQRT_LUT_B = 0x08BA4;
       break;
       case 61 : INVSQRT_LUT_B = 0x08A80;
       break;
       case 62 : INVSQRT_LUT_B = 0x08963;
       break;
       case 63 : INVSQRT_LUT_B = 0x0884D;
       break;
       default :
       break;
	}
	return INVSQRT_LUT_B;
}

Word32 get_recip_LUT_A(Word8 inp)
{
	Word32 recip_LUT_A = 0;
	switch(inp)
	{
	   case 0 : recip_LUT_A = 0x7E07E;
	   break;
	   case 1 : recip_LUT_A = 0x7A363;
	   break;
	   case 2 : recip_LUT_A = 0x76904;
	   break;
	   case 3 : recip_LUT_A = 0x73139;
	   break;
	   case 4 : recip_LUT_A = 0x6FBDA;
	   break;
	   case 5 : recip_LUT_A = 0x6C8C6;
	   break;
	   case 6 : recip_LUT_A = 0x697D9;
	   break;
	   case 7 : recip_LUT_A = 0x668F7;
	   break;
	   case 8 : recip_LUT_A = 0x63C02;
	   break;
	   case 9 : recip_LUT_A = 0x610DF;
	   break;
	   case 10 : recip_LUT_A = 0x5E776;
	   break;
	   case 11 : recip_LUT_A = 0x5BFB0;
	   break;
	   case 12 : recip_LUT_A = 0x59976;
	   break;
	   case 13 : recip_LUT_A = 0x574B4;
	   break;
	   case 14 : recip_LUT_A = 0x55158;
	   break;
	   case 15 : recip_LUT_A = 0x52F50;
	   break;
	   case 16 : recip_LUT_A = 0x50E8A;
	   break;
	   case 17 : recip_LUT_A = 0x4EEF7;
	   break;
	   case 18 : recip_LUT_A = 0x4D088;
	   break;
	   case 19 : recip_LUT_A = 0x4B32F;
	   break;
	   case 20 : recip_LUT_A = 0x496E0;
	   break;
	   case 21 : recip_LUT_A = 0x47B8D;
	   break;
	   case 22 : recip_LUT_A = 0x4612C;
	   break;
	   case 23 : recip_LUT_A = 0x447B1;
	   break;
	   case 24 : recip_LUT_A = 0x42F12;
	   break;
	   case 25 : recip_LUT_A = 0x41744;
	   break;
	   case 26 : recip_LUT_A = 0x40040;
	   break;
	   case 27 : recip_LUT_A = 0x3E9FC;
	   break;
	   case 28 : recip_LUT_A = 0x3D470;
	   break;
	   case 29 : recip_LUT_A = 0x3BF93;
	   break;
	   case 30 : recip_LUT_A = 0x3AB60;
	   break;
	   case 31 : recip_LUT_A = 0x397CE;
	   break;
	   case 32 : recip_LUT_A = 0x384D7;
	   break;
	   case 33 : recip_LUT_A = 0x37274;
	   break;
	   case 34 : recip_LUT_A = 0x360A0;
	   break;
	   case 35 : recip_LUT_A = 0x34F56;
	   break;
	   case 36 : recip_LUT_A = 0x33E8E;
	   break;
	   case 37 : recip_LUT_A = 0x32E45;
	   break;
	   case 38 : recip_LUT_A = 0x31E75;
	   break;
	   case 39 : recip_LUT_A = 0x30F1B;
	   break;
	   case 40 : recip_LUT_A = 0x30030;
	   break;
	   case 41 : recip_LUT_A = 0x2F1B2;
	   break;
	   case 42 : recip_LUT_A = 0x2E39B;
	   break;
	   case 43 : recip_LUT_A = 0x2D5E9;
	   break;
	   case 44 : recip_LUT_A = 0x2C897;
	   break;
	   case 45 : recip_LUT_A = 0x2BBA2;
	   break;
	   case 46 : recip_LUT_A = 0x2AF07;
	   break;
	   case 47 : recip_LUT_A = 0x2A2C2;
	   break;
	   case 48 : recip_LUT_A = 0x296D1;
	   break;
	   case 49 : recip_LUT_A = 0x28B30;
	   break;
	   case 50 : recip_LUT_A = 0x27FDD;
	   break;
	   case 51 : recip_LUT_A = 0x274D5;
	   break;
	   case 52 : recip_LUT_A = 0x26A15;
	   break;
	   case 53 : recip_LUT_A = 0x25F9B;
	   break;
	   case 54 : recip_LUT_A = 0x25565;
	   break;
	   case 55 : recip_LUT_A = 0x24B70;
	   break;
	   case 56 : recip_LUT_A = 0x241BA;
	   break;
	   case 57 : recip_LUT_A = 0x23842;
	   break;
	   case 58 : recip_LUT_A = 0x22F04;
	   break;
	   case 59 : recip_LUT_A = 0x22600;
	   break;
	   case 60 : recip_LUT_A = 0x21D33;
	   break;
	   case 61 : recip_LUT_A = 0x2149C;
	   break;
	   case 62 : recip_LUT_A = 0x20C39;
	   break;
	   case 63 : recip_LUT_A = 0x20408;
	   break;
	   default :
       break;
	}
	return recip_LUT_A;
}

Word32 get_recip_LUT_B(Word8 inp)
{
	Word32 recip_LUT_B = 0;
	switch(inp)
	{
	   case 0 : recip_LUT_B = 0x3F037;
       break;
       case 1 : recip_LUT_B = 0x3D130;
       break;
       case 2 : recip_LUT_B = 0x3B317;
       break;
       case 3 : recip_LUT_B = 0x395E3;
       break;
       case 4 : recip_LUT_B = 0x37989;
       break;
       case 5 : recip_LUT_B = 0x35E00;
       break;
       case 6 : recip_LUT_B = 0x3433F;
       break;
       case 7 : recip_LUT_B = 0x3293E;
       break;
       case 8 : recip_LUT_B = 0x30FF4;
       break;
       case 9 : recip_LUT_B = 0x2F75B;
       break;
       case 10 : recip_LUT_B = 0x2DF6A;
       break;
       case 11 : recip_LUT_B = 0x2C81C;
       break;
       case 12 : recip_LUT_B = 0x2B16A;
       break;
       case 13 : recip_LUT_B = 0x29B4E;
       break;
       case 14 : recip_LUT_B = 0x285C2;
       break;
       case 15 : recip_LUT_B = 0x270C1;
       break;
       case 16 : recip_LUT_B = 0x25C45;
       break;
       case 17 : recip_LUT_B = 0x2484A;
       break;
       case 18 : recip_LUT_B = 0x234CC;
       break;
       case 19 : recip_LUT_B = 0x221C4;
       break;
       case 20 : recip_LUT_B = 0x20F30;
       break;
       case 21 : recip_LUT_B = 0x1FD0C;
       break;
       case 22 : recip_LUT_B = 0x1EB52;
       break;
       case 23 : recip_LUT_B = 0x1DA01;
       break;
       case 24 : recip_LUT_B = 0x1C913;
       break;
       case 25 : recip_LUT_B = 0x1B887;
       break;
       case 26 : recip_LUT_B = 0x1A858;
       break;
       case 27 : recip_LUT_B = 0x19883;
       break;
       case 28 : recip_LUT_B = 0x18907;
       break;
       case 29 : recip_LUT_B = 0x179DF;
       break;
       case 30 : recip_LUT_B = 0x16B09;
       break;
       case 31 : recip_LUT_B = 0x15C83;
       break;
       case 32 : recip_LUT_B = 0x14E49;
       break;
       case 33 : recip_LUT_B = 0x1405B;
       break;
       case 34 : recip_LUT_B = 0x132B5;
       break;
       case 35 : recip_LUT_B = 0x12555;
       break;
       case 36 : recip_LUT_B = 0x11839;
       break;
       case 37 : recip_LUT_B = 0x10B60;
       break;
       case 38 : recip_LUT_B = 0x0FEC6;
       break;
       case 39 : recip_LUT_B = 0x0F26B;
       break;
       case 40 : recip_LUT_B = 0x0E64D;
       break;
       case 41 : recip_LUT_B = 0x0DA69;
       break;
       case 42 : recip_LUT_B = 0x0CEBE;
       break;
       case 43 : recip_LUT_B = 0x0C34B;
       break;
       case 44 : recip_LUT_B = 0x0B80E;
       break;
       case 45 : recip_LUT_B = 0x0AD06;
       break;
       case 46 : recip_LUT_B = 0x0A231;
       break;
       case 47 : recip_LUT_B = 0x0978D;
       break;
       case 48 : recip_LUT_B = 0x08D1A;
       break;
       case 49 : recip_LUT_B = 0x082D6;
       break;
       case 50 : recip_LUT_B = 0x078C0;
       break;
       case 51 : recip_LUT_B = 0x06ED7;
       break;
       case 52 : recip_LUT_B = 0x06519;
       break;
       case 53 : recip_LUT_B = 0x05B86;
       break;
       case 54 : recip_LUT_B = 0x0521C;
       break;
       case 55 : recip_LUT_B = 0x048DA;
       break;
       case 56 : recip_LUT_B = 0x03FC0;
       break;
       case 57 : recip_LUT_B = 0x036CC;
       break;
       case 58 : recip_LUT_B = 0x02DFD;
       break;
       case 59 : recip_LUT_B = 0x02553;
       break;
       case 60 : recip_LUT_B = 0x01CCD;
       break;
       case 61 : recip_LUT_B = 0x01469;
       break;
       case 62 : recip_LUT_B = 0x00C28;
       break;
       case 63 : recip_LUT_B = 0x00407;
       break;
       default :
       break;
	}
	return recip_LUT_B;
}

Word32 get_LOG2_LUT_A(Word8 inp)
{
	Word32 LOG2_LUT_A = 0;
	switch(inp)
	{
	   case 0 : LOG2_LUT_A = 0x2DCF3;
	   break;
	   case 1 : LOG2_LUT_A = 0x2D1C2;
	   break;
	   case 2 : LOG2_LUT_A = 0x2C6E7;
	   break;
	   case 3 : LOG2_LUT_A = 0x2BC5F;
	   break;
	   case 4 : LOG2_LUT_A = 0x2B226;
	   break;
	   case 5 : LOG2_LUT_A = 0x2A837;
	   break;
	   case 6 : LOG2_LUT_A = 0x29E91;
	   break;
	   case 7 : LOG2_LUT_A = 0x29530;
	   break;
	   case 8 : LOG2_LUT_A = 0x28C12;
	   break;
	   case 9 : LOG2_LUT_A = 0x28332;
	   break;
	   case 10 : LOG2_LUT_A = 0x27A90;
	   break;
	   case 11 : LOG2_LUT_A = 0x27228;
	   break;
	   case 12 : LOG2_LUT_A = 0x269F9;
	   break;
	   case 13 : LOG2_LUT_A = 0x26200;
	   break;
	   case 14 : LOG2_LUT_A = 0x25A3A;
	   break;
	   case 15 : LOG2_LUT_A = 0x252A7;
	   break;
	   case 16 : LOG2_LUT_A = 0x24B44;
	   break;
	   case 17 : LOG2_LUT_A = 0x2440F;
	   break;
	   case 18 : LOG2_LUT_A = 0x23D07;
	   break;
	   case 19 : LOG2_LUT_A = 0x2362A;
	   break;
	   case 20 : LOG2_LUT_A = 0x22F77;
	   break;
	   case 21 : LOG2_LUT_A = 0x228EC;
	   break;
	   case 22 : LOG2_LUT_A = 0x22287;
	   break;
	   case 23 : LOG2_LUT_A = 0x21C48;
	   break;
	   case 24 : LOG2_LUT_A = 0x2162E;
	   break;
	   case 25 : LOG2_LUT_A = 0x21036;
	   break;
	   case 26 : LOG2_LUT_A = 0x20A5F;
	   break;
	   case 27 : LOG2_LUT_A = 0x204AA;
	   break;
	   case 28 : LOG2_LUT_A = 0x1FF14;
	   break;
	   case 29 : LOG2_LUT_A = 0x1F99D;
	   break;
	   case 30 : LOG2_LUT_A = 0x1F443;
	   break;
	   case 31 : LOG2_LUT_A = 0x1EF06;
	   break;
	   case 32 : LOG2_LUT_A = 0x1E9E5;
	   break;
	   case 33 : LOG2_LUT_A = 0x1E4DE;
	   break;
	   case 34 : LOG2_LUT_A = 0x1DFF2;
	   break;
	   case 35 : LOG2_LUT_A = 0x1DB1F;
	   break;
	   case 36 : LOG2_LUT_A = 0x1D665;
	   break;
	   case 37 : LOG2_LUT_A = 0x1D1C2;
	   break;
	   case 38 : LOG2_LUT_A = 0x1CD37;
	   break;
	   case 39 : LOG2_LUT_A = 0x1C8C2;
	   break;
	   case 40 : LOG2_LUT_A = 0x1C463;
	   break;
	   case 41 : LOG2_LUT_A = 0x1C01A;
	   break;
	   case 42 : LOG2_LUT_A = 0x1BBE5;
	   break;
	   case 43 : LOG2_LUT_A = 0x1B7C3;
	   break;
	   case 44 : LOG2_LUT_A = 0x1B3B6;
	   break;
	   case 45 : LOG2_LUT_A = 0x1AFBB;
	   break;
	   case 46 : LOG2_LUT_A = 0x1ABD3;
	   break;
	   case 47 : LOG2_LUT_A = 0x1A7FD;
	   break;
	   case 48 : LOG2_LUT_A = 0x1A438;
	   break;
	   case 49 : LOG2_LUT_A = 0x1A084;
	   break;
	   case 50 : LOG2_LUT_A = 0x19CE1;
	   break;
	   case 51 : LOG2_LUT_A = 0x1994E;
	   break;
	   case 52 : LOG2_LUT_A = 0x195CA;
	   break;
	   case 53 : LOG2_LUT_A = 0x19256;
	   break;
	   case 54 : LOG2_LUT_A = 0x18EF1;
	   break;
	   case 55 : LOG2_LUT_A = 0x18B9A;
	   break;
	   case 56 : LOG2_LUT_A = 0x18852;
	   break;
	   case 57 : LOG2_LUT_A = 0x18517;
	   break;
	   case 58 : LOG2_LUT_A = 0x181EA;
	   break;
	   case 59 : LOG2_LUT_A = 0x17ECA;
	   break;
	   case 60 : LOG2_LUT_A = 0x17BB7;
	   break;
	   case 61 : LOG2_LUT_A = 0x178B0;
	   break;
	   case 62 : LOG2_LUT_A = 0x175B6;
	   break;
	   case 63 : LOG2_LUT_A = 0x172C8;
	   break;
	   default :
       break;
	}
	return LOG2_LUT_A;
}

Word32 get_LOG2_LUT_B(Word8 inp)
{
	Word32 LOG2_LUT_B = 0;
	switch(inp)
	{
	   case 0 : LOG2_LUT_B = 0x16E78;
	   break;
	   case 1 : LOG2_LUT_B = 0x168C9;
	   break;
	   case 2 : LOG2_LUT_B = 0x16331;
	   break;
	   case 3 : LOG2_LUT_B = 0x15DAD;
	   break;
	   case 4 : LOG2_LUT_B = 0x1583F;
	   break;
	   case 5 : LOG2_LUT_B = 0x152E4;
	   break;
	   case 6 : LOG2_LUT_B = 0x14D9E;
	   break;
	   case 7 : LOG2_LUT_B = 0x1486A;
	   break;
	   case 8 : LOG2_LUT_B = 0x14349;
	   break;
	   case 9 : LOG2_LUT_B = 0x13E39;
	   break;
	   case 10 : LOG2_LUT_B = 0x1393C;
	   break;
	   case 11 : LOG2_LUT_B = 0x1344F;
	   break;
	   case 12 : LOG2_LUT_B = 0x12F73;
	   break;
	   case 13 : LOG2_LUT_B = 0x12AA7;
	   break;
	   case 14 : LOG2_LUT_B = 0x125EA;
	   break;
	   case 15 : LOG2_LUT_B = 0x1213E;
	   break;
	   case 16 : LOG2_LUT_B = 0x11CA0;
	   break;
	   case 17 : LOG2_LUT_B = 0x11810;
	   break;
	   case 18 : LOG2_LUT_B = 0x1138F;
	   break;
	   case 19 : LOG2_LUT_B = 0x10F1C;
	   break;
	   case 20 : LOG2_LUT_B = 0x10AB6;
	   break;
	   case 21 : LOG2_LUT_B = 0x1065E;
	   break;
	   case 22 : LOG2_LUT_B = 0x10213;
	   break;
	   case 23 : LOG2_LUT_B = 0x0FDD4;
	   break;
	   case 24 : LOG2_LUT_B = 0x0F9A1;
	   break;
	   case 25 : LOG2_LUT_B = 0x0F57B;
	   break;
	   case 26 : LOG2_LUT_B = 0x0F160;
	   break;
	   case 27 : LOG2_LUT_B = 0x0ED51;
	   break;
	   case 28 : LOG2_LUT_B = 0x0E94D;
	   break;
	   case 29 : LOG2_LUT_B = 0x0E555;
	   break;
	   case 30 : LOG2_LUT_B = 0x0E167;
	   break;
	   case 31 : LOG2_LUT_B = 0x0DD84;
	   break;
	   case 32 : LOG2_LUT_B = 0x0D9AB;
	   break;
	   case 33 : LOG2_LUT_B = 0x0D5DC;
	   break;
	   case 34 : LOG2_LUT_B = 0x0D217;
	   break;
	   case 35 : LOG2_LUT_B = 0x0CE5C;
	   break;
	   case 36 : LOG2_LUT_B = 0x0CAAB;
	   break;
	   case 37 : LOG2_LUT_B = 0x0C702;
	   break;
	   case 38 : LOG2_LUT_B = 0x0C363;
	   break;
	   case 39 : LOG2_LUT_B = 0x0BFCD;
	   break;
	   case 40 : LOG2_LUT_B = 0x0BC40;
	   break;
	   case 41 : LOG2_LUT_B = 0x0B8BC;
	   break;
	   case 42 : LOG2_LUT_B = 0x0B540;
	   break;
	   case 43 : LOG2_LUT_B = 0x0B1CC;
	   break;
	   case 44 : LOG2_LUT_B = 0x0AE61;
	   break;
	   case 45 : LOG2_LUT_B = 0x0AAFD;
	   break;
	   case 46 : LOG2_LUT_B = 0x0A7A2;
	   break;
	   case 47 : LOG2_LUT_B = 0x0A44E;
	   break;
	   case 48 : LOG2_LUT_B = 0x0A102;
	   break;
	   case 49 : LOG2_LUT_B = 0x09DBD;
	   break;
	   case 50 : LOG2_LUT_B = 0x09A7F;
	   break;
	   case 51 : LOG2_LUT_B = 0x09749;
	   break;
	   case 52 : LOG2_LUT_B = 0x0941A;
	   break;
	   case 53 : LOG2_LUT_B = 0x090F2;
	   break;
	   case 54 : LOG2_LUT_B = 0x08DD1;
	   break;
	   case 55 : LOG2_LUT_B = 0x08AB6;
	   break;
	   case 56 : LOG2_LUT_B = 0x087A2;
	   break;
	   case 57 : LOG2_LUT_B = 0x08495;
	   break;
	   case 58 : LOG2_LUT_B = 0x0818E;
	   break;
	   case 59 : LOG2_LUT_B = 0x07E8D;
	   break;
	   case 60 : LOG2_LUT_B = 0x07B93;
	   break;
	   case 61 : LOG2_LUT_B = 0x0789E;
	   break;
	   case 62 : LOG2_LUT_B = 0x075B0;
	   break;
	   case 63 : LOG2_LUT_B = 0x072C7;
	   break;
	   default :
       break;
	}
	return LOG2_LUT_B;

}


/* intrinsics removed in ADL reduction which are used by other functions */

static inline Word40 do__mpy_i_x_hh(Word32 __Da_h, Word32 __Db_h)
{
  Word32 res;
  res=__l_get_msb(__Da_h)*__l_get_msb(__Db_h);
  if ((__l_get_msb(__Da_h) == MIN_16) && (__l_get_msb(__Db_h) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);
}

static inline Word40 do__mpy_i_x_hl(Word32 __Da_h, Word32 __Db_l)
{
  Word32 res;
  res=(__l_get_msb(__Da_h)*__l_get_lsb(__Db_l));
  if ((__l_get_msb(__Da_h) == MIN_16) && (__l_get_lsb(__Db_l) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);
}




/*set rounding mode*/
void do__set2crm()
{
 RoundingMode = _2S_COMPLEMENT;
 updateSR();
}

void do__setcnvrm()
{
 RoundingMode = CONVERGENT;
 updateSR();
}
/*scaling off*/
 void do__setscoff()
{
ScalingMode = NO_SCALING;
 updateSR();
}
/*scaling up 1 bit*/
 void do__setscup1()
{
ScalingMode = UP_1;
 updateSR();
}
/*scaling down 1 bit*/
 void do__setscdown1()
{
ScalingMode = DOWN_1;
 updateSR();
}
/*scaling down 2 bits*/
 void do__setscdown2()
{
ScalingMode = DOWN_2;
 updateSR();
}
/*scaling down 3 bits*/
void do__setscdown3()
{
ScalingMode = DOWN_3;
 updateSR();
}

/*****************************************************************************
*       Prototype emulation code. One function per intrinsic.
*****************************************************************************/


/*ID:    1, INST: abs.l Da,Dn*/
Word32 do__abs_l(Word32 __Da)
{
	if(__Da < 0)return(~__Da +1);
	else return __Da;
}

	/*ID:    2, INST: abs.2t Da,Dn*/
Word40 do__abs_2t(Word40 __Da)
{
	Word40 __Dn;
	Word32 __Da_WH,__Da_WL;
	__Da_WH = do__abs_l(do__extract_wh(__Da));
	__Da_WL = do__abs_l(do__extract_wl(__Da));
	do__set_WH(__Da_WH,&__Dn);
	do__set_WL(__Da_WL,&__Dn);
	return __Dn;
}

	/*ID:    3, INST: abs.4t Dab,Dmn*/
void do__abs_4t(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__abs_2t(__Da);
	*__Dn = do__abs_2t(__Db);
}

	/*ID:    4, INST: abs.2w Da,Dn*/
Word32 do__abs_2w(Word32 __Da)
{

	Word32 __Dn, __Da_2;
	Word32 __Dn_h, __Dn_l, __Da_h, __Da_l;

	/*__Da = __x_to_l(do__abs_s40_x(__l_to_x(__Da)));*/
#if 0
	__Da_2 = __x_to_l(__Da);
#else
	__Da_2 = (__Da) ;
#endif

	__Da_h = (__Da_2 & 0xffff0000) >> 16;
	if(__Da_h & 0x8000) __Da_h |= 0xffff0000;

	__Da_l = (__Da_2 & 0xffff);

	if(__Da_l & 0x8000) __Da_l |= 0xffff0000;

	__Dn_h = abs(__Da_h);
	__Dn_l = abs(__Da_l);

	__Dn = ((__Dn_h & 0xffff) << 16) | (__Dn_l & 0xffff);

#if 0
    return __x_set(0, __Dn);
#else
    return (__Dn);
#endif
}

	/*ID:    5, INST: abs.4w Dab,Dmn*/
void do__abs_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
	*__Dm = do__abs_2w(__Da);
	*__Dn = do__abs_2w(__Db);
}

	/*ID:    6, INST: abs.x Da,Dn*/
Word40 do__abs_x(Word40 __Da)
{
 if (__x_guard(__Da) & 0x80) {
   return do__neg_x(__Da);
 } else {
   return __Da;
 }
}

	/*ID:    7, INST: abs.2x Dab,Dmn*/
void do__abs_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__abs_x(__Da);
	*__Dn = do__abs_x(__Db);
}

	/*ID:    8, INST: abs.leg.x Da,Dn*/

	/*ID:    9, INST: abs.s.l Da,Dn*/
Word32 do__abs_s_l(Word32 __Da)
{
	return do__sat_x_l(__l_to_x(do__abs_l(__Da)));
}

	/*ID:   10, INST: abs.s.2w Da,Dn*/
Word32 do__abs_s_2w(Word32 __Da)
{
	Word32 __Dn, __Da_2;
	Word32 __Dn_h, __Dn_l, __Da_h, __Da_l;

	__Da_2 = (__Da) ;

	__Da_h = (__Da_2 & 0xffff0000) >> 16;
	if(__Da_h & 0x8000) __Da_h |= 0xffff0000;

	__Da_l = (__Da_2 & 0xffff);

	if(__Da_l & 0x8000) __Da_l |= 0xffff0000;

	__Dn_h = do__saturate(abs(__Da_h));
	__Dn_l = do__saturate(abs(__Da_l));


	__Dn = ((__Dn_h & 0xffff) << 16) | (__Dn_l & 0xffff);

	return __Dn;
}

	/*ID:   11, INST: abs.s.4w Dab,Dmn*/
void do__abs_s_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
	*__Dm = do__abs_s_2w(__Da);
	*__Dn = do__abs_s_2w(__Db);
}

	/*ID:   12, INST: abs.s.x Da,Dn*/
Word40 do__abs_s_x(Word40 __Da)
{
	Word40 __Dm;
	__Dm = do__abs_x(__Da);
	return __l_to_x(do__sat_x_l(__Dm));
}

	/*ID:   13, INST: abs.s.2x Dab,Dmn*/
void do__abs_s_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__abs_s_x(__Da);
	*__Dn = do__abs_s_x(__Db);
}

	/*ID:   14, INST: abs.s40.x Da,Dn*/
/*
  Function Name: do__abs_s40_x
  Purpose: Get the absolute of 40 bit data and saturate the result to 40 bit.
  Complexity Weight:
  Inputs: __Da is a 40 bit data whose value falls in the range 0x8000000000 to 0x7fffffffff
  Outputs:
  Return Value: It returns a 40 bit absolute value.
*/
Word40 do__abs_s40_x(Word40 __Da)
{
  if (__x_guard(__Da) & 0x80) {
   return do__neg_x(__Da);
 } else {
   return __Da;
 }
}

	/*ID:   15, INST: absa Ra,Rn*/
Word32 do__absa(Word32 __Ra)
{
	if(__Ra < 0)return(~__Ra +1);
	else return __Ra;
}

	/*ID:   16, INST: absa Ra,Rn*/

	/*ID:   17, INST: acator #u6_t4*/

	/*ID:   18, INST: acator.p #u6_t4,#u4_t6*/

	/*ID:   19, INST: acs.h.2w Da.h,Da.h,Db,Dn*/
Word32 do__acs_h_2w_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   20, INST: acs.h.2w Da.l,Da.l,Db,Dn*/
Word32 do__acs_h_2w_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   21, INST: acs.h.2w Da.h,Da.l,Db,Dn*/
Word32 do__acs_h_2w_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   22, INST: acs.h.2w Da.l,Da.h,Db,Dn*/
Word32 do__acs_h_2w_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   23, INST: acs.h.2w -Da.h,-Da.h,Db,Dn*/
Word32 do__acs_h_2w_ss_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   24, INST: acs.h.2w -Da.h,Da.h,Db,Dn*/
Word32 do__acs_h_2w_sa_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   25, INST: acs.h.2w Da.h,-Da.h,Db,Dn*/
Word32 do__acs_h_2w_as_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   26, INST: acs.h.2w -Da.l,-Da.l,Db,Dn*/
Word32 do__acs_h_2w_ss_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   27, INST: acs.h.2w -Da.l,Da.l,Db,Dn*/
Word32 do__acs_h_2w_sa_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   28, INST: acs.h.2w Da.l,-Da.l,Db,Dn*/
Word32 do__acs_h_2w_as_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   29, INST: acs.h.2w -Da.h,-Da.l,Db,Dn*/
Word32 do__acs_h_2w_ss_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   30, INST: acs.h.2w -Da.h,Da.l,Db,Dn*/
Word32 do__acs_h_2w_sa_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   31, INST: acs.h.2w -Da.l,-Da.h,Db,Dn*/
Word32 do__acs_h_2w_ss_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   32, INST: acs.h.2w -Da.l,Da.h,Db,Dn*/
Word32 do__acs_h_2w_sa_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   33, INST: acs.h.2w Da.h,-Da.l,Db,Dn*/
Word32 do__acs_h_2w_as_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   34, INST: acs.h.2w Da.l,-Da.h,Db,Dn*/
Word32 do__acs_h_2w_as_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   35, INST: acs.l.2w Da.h,Da.h,Db,Dn*/
Word32 do__acs_l_2w_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   36, INST: acs.l.2w Da.l,Da.l,Db,Dn*/
Word32 do__acs_l_2w_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   37, INST: acs.l.2w Da.h,Da.l,Db,Dn*/
Word32 do__acs_l_2w_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   38, INST: acs.l.2w Da.l,Da.h,Db,Dn*/
Word32 do__acs_l_2w_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   39, INST: acs.l.2w -Da.h,-Da.h,Db,Dn*/
Word32 do__acs_l_2w_ss_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   40, INST: acs.l.2w -Da.h,Da.h,Db,Dn*/
Word32 do__acs_l_2w_sa_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   41, INST: acs.l.2w Da.h,-Da.h,Db,Dn*/
Word32 do__acs_l_2w_as_hh(Word32 __Dn, Word32 __Da_h, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   42, INST: acs.l.2w -Da.l,-Da.l,Db,Dn*/
Word32 do__acs_l_2w_ss_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   43, INST: acs.l.2w -Da.l,Da.l,Db,Dn*/
Word32 do__acs_l_2w_sa_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   44, INST: acs.l.2w Da.l,-Da.l,Db,Dn*/
Word32 do__acs_l_2w_as_ll(Word32 __Dn, Word32 __Da_l, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   45, INST: acs.l.2w -Da.h,-Da.l,Db,Dn*/
Word32 do__acs_l_2w_ss_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   46, INST: acs.l.2w -Da.h,Da.l,Db,Dn*/
Word32 do__acs_l_2w_sa_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   47, INST: acs.l.2w -Da.l,-Da.h,Db,Dn*/
Word32 do__acs_l_2w_ss_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   48, INST: acs.l.2w -Da.l,Da.h,Db,Dn*/
Word32 do__acs_l_2w_sa_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   49, INST: acs.l.2w Da.h,-Da.l,Db,Dn*/
Word32 do__acs_l_2w_as_hl(Word32 __Dn, Word32 __Da_h, Word32 __Da_l_2, Word32 __Db)
{
}

	/*ID:   50, INST: acs.l.2w Da.l,-Da.h,Db,Dn*/
Word32 do__acs_l_2w_as_lh(Word32 __Dn, Word32 __Da_l, Word32 __Da_h_2, Word32 __Db)
{
}

	/*ID:   51, INST: add.ll Dab,Dcd,Dmn*/

	/*ID:   52, INST: add.t Da.h,Dn.h*/
Word40 do__add_t_hh(Word40 __Dn_h, Word40 __Da_h)
{
	Word32 Dn_WH;
	Dn_WH = __extract_wh(__Da_h) + __extract_wh(__Dn_h);
	__set_WH(Dn_WH,&__Da_h);
	return __Da_h;
}

	/*ID:   53, INST: add.t Da.l,Dn.l*/
Word40 do__add_t_ll(Word40 __Dn_l, Word40 __Da_l)
{
	Word32 Dn_WL;
	Dn_WL = __extract_wl(__Da_l) + __extract_wl(__Dn_l);
	__set_WL(Dn_WL,&__Da_l);
	return __Da_l;
}

	/*ID:   54, INST: add.t Da.h,Dn.l*/
Word40 do__add_t_hl(Word40 __Dn_l, Word40 __Da_h)
{
	Word32 Dn_WH;
	Dn_WH = __extract_wh(__Da_h) + __extract_wl(__Dn_l);
	__set_WH(Dn_WH,&__Da_h);
	return __Da_h;
}

	/*ID:   55, INST: add.t Da.l,Dn.h*/
Word40 do__add_t_lh(Word40 __Dn_h, Word40 __Da_l)
{
	Word32 Dn_WL;
	Dn_WL = __extract_wl(__Da_l) + __extract_wh(__Dn_h);
	__set_WL(Dn_WL,&__Da_l);
	return __Da_l;
}

	/*ID:   56, INST: add.2t Da,Db,Dn*/
Word40 do__add_2t(Word40 __Da, Word40 __Db)
{
	Word40 result;
	Word32 result_wh,result_wl,__Da_WH,__Da_WL,__Db_WH,__Db_WL;
	__Da_WH = do__extract_wh(__Da);
	__Da_WL = do__extract_wl(__Da);
	__Db_WH = do__extract_wh(__Db);
	__Db_WL = do__extract_wl(__Db);

	 result_wh = __x_to_l(do__add_x(__l_to_x(__Da_WH),__l_to_x(__Db_WH)));
	 result_wl = __x_to_l(do__add_x(__l_to_x(__Da_WL),__l_to_x(__Db_WL)));

	 do__set_WH(result_wh,&result);
	 do__set_WL(result_wl,&result);

	 return result;
}

	/*ID:   57, INST: add.x #u5,Da,Dn*/
Word40 do__add_x_imm(UWord32 __imm1, Word40 __Da)
{
	Word40 __imm1_x;
	Word32 __imm1_l = (Word32)__imm1;
	__imm1_x = __x_set(0x0,__imm1_l);
	return do__add_x(__imm1_x,__Da);
}

	/*ID:   58, INST: add.x Da,Db,Dn*/
Word40 do__add_x(Word40 __Da, Word40 __Db)
{
	Word40 X_result;
	long long resultLL = 0LL;
	long long var1LL = __x_guard(__Da);
	long long var2LL = __x_guard(__Db);
	long res ,Dbl,Dal,high;

	/* There is currently a simulator bug with long long. */
	var1LL = (var1LL << 32) | (UWord32) __x_to_l(__Da);
	var2LL = (var2LL << 32) | (UWord32) __x_to_l(__Db);

	resultLL = var1LL + var2LL;
	X_result  = __x_set((resultLL >> 32) & 0xff,(UWord32) resultLL & 0xffffffff);

	return X_result;
}

	/*ID:   59, INST: add.leg.x #u5,Da,Dn*/

	/*ID:   60, INST: add.leg.x Da,Db,Dn*/

	/*ID:   61, INST: add.s.x Da,Db,Dn*/
/*
  Function Name: do__add_s_x
  Purpose: Adds two 40 bit words and saturates the result to 32 bit.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x800 000 0000 - 0x7FF FFF FFFF
  Outputs:
  Return Value: It returns a 40 bit result.
*/
Word40 do__add_s_x(Word40 __Da, Word40 __Db)
{
  return (__l_to_x(do__sat_x_l(do__add_x(__Da,__Db))));
}

	/*ID:   62, INST: adda #s16_t7,Ra,Rn*/

	/*ID:   63, INST: adda #u5,Ra,Rn*/

	/*ID:   64, INST: adda Ra,Rb,Rn*/

	/*ID:   65, INST: adda.lin #s16,Ra,Rn*/
Word32 do__adda_lin(Word32 __imm1, Word32 __Ra)
{
}

	/*ID:   66, INST: adda.lin #s16_t7,Ra,Rn*/

	/*ID:   67, INST: adda.lin #s9,sp,Rn*/

	/*ID:   68, INST: adda.lin #s9,sp,Rn*/

	/*ID:   69, INST: adda.lin #s9_3,sp*/

	/*ID:   70, INST: adda.lin #u5,Ra,Rn*/

	/*ID:   71, INST: adda.lin #u5,Ra,Rn*/

	/*ID:   72, INST: adda.lin Ra,Rb,Rn*/

	/*ID:   73, INST: adda.lin Ra,Rb,Rn*/

	/*ID:   74, INST: addc.ro.x Da,Db,Dn*/

	/*ID:   75, INST: addc.rw.leg.l Da,Db,Dn*/

	/*ID:   76, INST: addc.wo.l #u5,Da,Dn*/

	/*ID:   77, INST: addc.wo.l Da,Db,Dn*/

	/*ID:   78, INST: addc.wo.x #u5,Da,Dn*/

	/*ID:   79, INST: addc.wo.x Da,Db,Dn*/

	/*ID:   80, INST: addc.wo.leg.x #u5,Da,Dn*/

	/*ID:   81, INST: addc.wo.leg.x Da,Db,Dn*/

	/*ID:   82, INST: addla.2 Ra,Rb,Rn*/

	/*ID:   83, INST: addla.4 Ra,Rb,Rn*/

	/*ID:   84, INST: addla.1 Ra,Rb,Rn*/

	/*ID:   85, INST: addla.3 Ra,Rb,Rn*/

	/*ID:   86, INST: addla.5 Ra,Rb,Rn*/

	/*ID:   87, INST: addla.6 Ra,Rb,Rn*/

	/*ID:   88, INST: addla.1.lin Ra,Rb,Rn*/
Word32 do__addla_1_lin(Word32 __Ra, Word32 __Rb)
{
	Word32 result,__Ra_2;
	__Ra_2 = __Ra <<1;
	return __Ra_2+__Rb;
}

	/*ID:   89, INST: addla.1.lin Ra,Rb,Rn*/

	/*ID:   90, INST: addla.2.lin Ra,Rb,Rn*/
Word32 do__addla_2_lin(Word32 __Ra, Word32 __Rb)
{
	Word32 result,__Ra_2;
	__Ra_2 = __Ra <<2;
	return __Ra_2+__Rb;
}

	/*ID:   91, INST: addla.2.lin Ra,Rb,Rn*/

	/*ID:   92, INST: addla.3.lin Ra,Rb,Rn*/
Word32 do__addla_3_lin(Word32 __Ra, Word32 __Rb)
{
	Word32 result,__Ra_2;
	__Ra_2 = __Ra <<3;
	return __Ra_2+__Rb;
}

	/*ID:   93, INST: addla.3.lin Ra,Rb,Rn*/

	/*ID:   94, INST: addla.4.lin Ra,Rb,Rn*/
Word32 do__addla_4_lin(Word32 __Ra, Word32 __Rb)
{
	Word32 result,__Ra_2;
	__Ra_2 = __Ra <<4;
	return __Ra_2+__Rb;
}

	/*ID:   95, INST: addla.4.lin Ra,Rb,Rn*/

	/*ID:   96, INST: addla.5.lin Ra,Rb,Rn*/
Word32 do__addla_5_lin(Word32 __Ra, Word32 __Rb)
{
	Word32 result,__Ra_2;
	__Ra_2 = __Ra <<5;
	return __Ra_2+__Rb;
}

	/*ID:   97, INST: addla.5.lin Ra,Rb,Rn*/

	/*ID:   98, INST: addla.6.lin Ra,Rb,Rn*/
Word32 do__addla_6_lin(Word32 __Ra, Word32 __Rb)
{
	Word32 result,__Ra_2;
	__Ra_2 = __Ra << 6;
	return __Ra_2+__Rb;
}

	/*ID:   99, INST: addla.6.lin Ra,Rb,Rn*/

	/*ID:  100, INST: addm.x Da.h,Db,Dn*/
Word40 do__addm_x_h(Word32 __Da_h, Word40 __Db)
{
	__Da_h = __Da_h >> 16;
	return do__add_x(__l_to_x(__Da_h) , __Db);
}

	/*ID:  101, INST: addm.x Da.l,Db,Dn*/
Word40 do__addm_x_l(Word32 __Da_l, Word40 __Db)
{
	Word32 __Da_l32;
	Word16 __Da_l16 = __Da_l & 0xffff;
	__Da_l32 = (Word32) __Da_l16;/*sign extension*/
	 return do__add_x(__l_to_x(__Da_l32) , __Db);
}

	/*ID:  102, INST: addm.hl.x Da,Db,Dn*/
Word40 do__addm_hl_x(Word32 __Da, Word40 __Db)
{
	Word32 __Da_h,__Da_l,result;
	Word16 temp;

	__Da_h = __Da >> 16;
	temp = __Da & 0xffff;
	__Da_l = (Word32)temp;/*sign extension*/
	result = __Da_h + __Da_l;
	return do__add_x(__l_to_x(result) , __Db);
}

	/*ID:  103, INST: addm.hl.2x Dab,Dcd,Dmn*/
void do__addm_hl_2x(Word32 __Da, Word32 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__addm_hl_x(__Da,__Dc);
	*__Dn = do__addm_hl_x(__Db,__Dd);
}

	/*ID:  104, INST: aextc.lft Da,Db*/

	/*ID:  105, INST: aextc.rgt Da,Db*/

	/*ID:  106, INST: and.l #u32,Da,Dn*/

	/*ID:  107, INST: and.x #s32,Da,Dn*/
Word40 do__and_x_imm(Word32 __imm1, Word40 __Da)
{
}

	/*ID:  108, INST: and.x Da,Db,Dn*/
Word40 do__and_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  109, INST: and.2x Dab,Dcd,Dmn*/
void do__and_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __x_set((__x_guard(__Da) & __x_guard(__Dc)), (__x_to_l(__Da) & __x_to_l(__Dc)));
	*__Dn = __x_set((__x_guard(__Db) & __x_guard(__Dd)), (__x_to_l(__Db) & __x_to_l(__Dd)));

}

	/*ID:  110, INST: anda Ra,Rb,Rn*/

	/*ID:  111, INST: anda Ra,Rb,Rn*/

	/*ID:  112, INST: ash.lft.l Da,Db,Dn*/

Word32 do__ash_lft_l(Word32 __Da, Word32 __Db)
{
	int shift_val;

	long long resultLL = 0LL;
	long long var1LL = __x_guard(__l_to_x(__Db));
	Word40 X_result;

	if (__Da & 0x40)
	  /* shift_val = __l_or_x(__Da,0xffffff80); */
		shift_val = __Da | 0xffffff80;
	else
		shift_val = __Da & 0x7f;

	if ((shift_val == 0) || (__Db == 0))
		return __Db;

	if(shift_val < 0)
		return do__ash_rgt_l(-shift_val,__Db);
	else
	{
	  var1LL = (var1LL << 32) | (UWord32)__Db;
	  resultLL = var1LL << shift_val;
	  X_result  = __x_set((resultLL >> 32) & 0xff,(UWord32) resultLL & 0xffffffff);
	  return __x_to_l(X_result);
	}
}


	/*ID:  113, INST: ash.lft.l #u5,Da,Dn*/

	/*ID:  114, INST: ash.lft.2l Da,Dcd,Dmn*/
void do__ash_lft_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  115, INST: ash.lft.2l #u5,Dab,Dmn*/

	/*ID:  116, INST: ash.lft.ll Da,Dcd,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__ash_lft_ll(Word32 __Da, Int64 __Dcd)
{
}
#endif

	/*ID:  117, INST: ash.lft.ll #u6_t3,Dab,Dmn*/

	/*ID:  118, INST: ash.lft.2t Da,Db,Dn*/
Word40 do__ash_lft_2t(Word32 __Da, Word40 __Db)
{
}

	/*ID:  119, INST: ash.lft.2t #u5,Da,Dn*/

	/*ID:  120, INST: ash.lft.4t Da,Dcd,Dmn*/
void do__ash_lft_4t(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  121, INST: ash.lft.4t #u5,Dab,Dmn*/

	/*ID:  122, INST: ash.lft.2w Da,Db,Dn*/
Word32 do__ash_lft_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  123, INST: ash.lft.2w #u5,Da,Dn*/

	/*ID:  124, INST: ash.lft.4w Da,Dcd,Dmn*/
void do__ash_lft_4w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  125, INST: ash.lft.4w #u4,Dab,Dmn*/

	/*ID:  126, INST: ash.lft.x Da,Db,Dn*/
Word40 do__ash_lft_x(Word32 __Da, Word40 __Db)
{

	int shift_val,right_shift_val,i=0;
	long long resultLL = 0LL;
	long long var1LL;
	Word40 X_result = __Db;

	if (__Da & 0x40)
		shift_val = __Da | 0xffffff80;
	else
		shift_val = __Da & 0x7f;

	if ((shift_val == 0) /*|| (__x_to_l(__Db) == 0)*/)
		 return X_result;

	var1LL = __x_to_ll(__Db);

	if(shift_val < 0)
	{
		right_shift_val = -shift_val;
		var1LL = var1LL >> right_shift_val;
		X_result  = __ll_to_x(var1LL);
	}
	else
	{
		var1LL = var1LL << shift_val;
		X_result  = __ll_to_x(var1LL);
	}
	return X_result;
}


	/*ID:  127, INST: ash.lft.x #ue5,Da,Dn*/

	/*ID:  128, INST: ash.lft.2x Da,Dcd,Dmn*/
void do__ash_lft_2x(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__ash_lft_x(__Da,__Dc);
	*__Dn = do__ash_lft_x(__Da,__Dd);
}

	/*ID:  129, INST: ash.lft.2x #ue5,Dab,Dmn*/

	/*ID:  130, INST: ash.lft.leg.x Da,Db,Dn*/

	/*ID:  131, INST: ash.lft.leg.x #u5,Da,Dn*/

	/*ID:  132, INST: ash.lft.s.l Da,Db,Dn*/
Word32 do__ash_lft_s_l(Word32 __Da, Word32 __Db)
{

	int shift_val,right_shift_val;

	if (__Da & 0x20)
		shift_val = __Da | 0xffffffc0;
	else
		shift_val = __Da & 0x3f;

	if ((shift_val == 0) || (__Db == 0))
		return __Db;

	if(shift_val < 0)
	{
		right_shift_val = -shift_val;
		/*I think right right shift will never result in saturation for 32 bit data.. is this right?*/
		__Db>>= right_shift_val;

	}
	else
	{

		for (; shift_val > 0; shift_val--)
		{

			if (__Db > (Word32) 0X3fffffffL)
			{
			  /*Overflow = 1; set Overflow when overflow occurs in 32-bit
			  				arithmetic saturation mode */
				__Db = MAX_32;
				break;
			}
			else
			{
				if (__Db < (Word32) 0xc0000000L)
				{
				  /* Overflow = 1; set Overflow when overflow occurs in 32-bit
								arithmetic saturation mode */
					__Db = MIN_32;
					break;
				}
			}
			__Db<<=1;
    	}
	}
	   return __Db;
	}

	/*ID:  133, INST: ash.lft.s.l #u5,Da,Dn*/

	/*ID:  134, INST: ash.lft.s.2l Da,Dcd,Dmn*/
void do__ash_lft_s_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  135, INST: ash.lft.s.2l #u5,Dab,Dmn*/

	/*ID:  136, INST: ash.lft.s.2t Da,Db,Dn*/
Word40 do__ash_lft_s_2t(Word32 __Da, Word40 __Db)
{
}

	/*ID:  137, INST: ash.lft.s.2t #u5,Da,Dn*/

	/*ID:  138, INST: ash.lft.s.4t Da,Dcd,Dmn*/
void do__ash_lft_s_4t(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  139, INST: ash.lft.s.4t #u5,Dab,Dmn*/

	/*ID:  140, INST: ash.lft.s.2w Da,Db,Dn*/
Word32 do__ash_lft_s_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  141, INST: ash.lft.s.2w #u5,Da,Dn*/

	/*ID:  142, INST: ash.lft.s.4w Da,Dcd,Dmn*/
void do__ash_lft_s_4w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  143, INST: ash.lft.s.4w #u4,Dab,Dmn*/

	/*ID:  144, INST: ash.lft.s.x Da,Db,Dn*/

Word40 do__ash_lft_s_x(Word32 __Da, Word40 __Db)
{
	int shift_val,right_shift_val,i=0;
	long long resultLL = 0LL;
	long long var1LL;
	Word40 X_result = __Db;
	shift_Overflow = 0;

	if (__Da & 0x40)
		shift_val = __Da | 0xffffff80;
	else
		shift_val = __Da & 0x7f;

	if ((shift_val == 0) || (__x_to_l(__Db) == 0))
	{
		__Db = __l_to_x(do__sat_x_l(__Db));
		if(shift_Overflow)return __Db;
		else return X_result;

	}

	var1LL = __x_to_ll(__Db);

	if(shift_val < 0)
	{
		right_shift_val = -shift_val;
	/*	for(i=0;i<right_shift_val;i++)
		{*/
			resultLL = var1LL >> right_shift_val;
			X_result  = __ll_to_x(resultLL);
			__Db = __l_to_x(do__sat_x_l(X_result));
			/*if(shift_Overflow)break; */
		/*}*/

	}
	else
		{
			for(i=0;i<shift_val;i++)
			{
				resultLL = var1LL << 1;
			X_result  = __ll_to_x(resultLL);
				__Db = __l_to_x(do__sat_x_l(X_result));
			if(shift_Overflow)break;
			}
		}
	if(shift_Overflow)return __Db;
		else return X_result;
	}

	/*ID:  145, INST: ash.lft.s.x #ue5,Da,Dn*/

	/*ID:  146, INST: ash.lft.s.2x Da,Dcd,Dmn*/
void do__ash_lft_s_2x(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  147, INST: ash.lft.s.2x #ue5,Dab,Dmn*/

	/*ID:  148, INST: ash.lft.s20.2t Da,Db,Dn*/
Word40 do__ash_lft_s20_2t(Word32 __Da, Word40 __Db)
{
}

	/*ID:  149, INST: ash.lft.s20.2t #u5,Da,Dn*/

	/*ID:  150, INST: ash.lft.s20.4t Da,Dcd,Dmn*/
void do__ash_lft_s20_4t(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  151, INST: ash.lft.s20.4t #u5,Dab,Dmn*/

	/*ID:  152, INST: ash.rgt.l Da,Db,Dn*/
Word32 do__ash_rgt_l(Word32 __Da, Word32 __Db)
{
	int shift_val;

	long long resultLL = 0LL;
	long long var1LL = __x_guard(__l_to_x(__Db));
	Word40 X_result;

	if (__Da & 0x40)
	  /*shift_val = __l_or_x(__Da,0xffffff80); */
		shift_val = __Da | 0xffffff80;
	else
		shift_val = __Da & 0x7f;

	if ((shift_val == 0) || (__Db == 0))
		return __Db;

	if(shift_val < 0)
		return do__ash_lft_l(-shift_val,__Db);
	else
	{
		var1LL = (var1LL << 32) |(UWord32)__Db;
		resultLL = var1LL >> shift_val;
		X_result  = __x_set((resultLL >> 32) & 0xff,(UWord32) resultLL & 0xffffffff);
		return __x_to_l(X_result);
	}
}

	/*ID:  153, INST: ash.rgt.l #u5,Da,Dn*/

	/*ID:  154, INST: ash.rgt.2l Da,Dcd,Dmn*/
void do__ash_rgt_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  155, INST: ash.rgt.2l #u5,Dab,Dmn*/

	/*ID:  156, INST: ash.rgt.ll Da,Dcd,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__ash_rgt_ll(Word32 __Da, Int64 __Dcd)
{
}
#endif

	/*ID:  157, INST: ash.rgt.ll #u6_t3,Dab,Dmn*/

	/*ID:  158, INST: ash.rgt.2t Da,Db,Dn*/
Word40 do__ash_rgt_2t(Word32 __Da, Word40 __Db)
{
}

	/*ID:  159, INST: ash.rgt.2t #u5,Da,Dn*/

	/*ID:  160, INST: ash.rgt.4t Da,Dcd,Dmn*/
void do__ash_rgt_4t(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  161, INST: ash.rgt.4t #u5,Dab,Dmn*/

	/*ID:  162, INST: ash.rgt.2w Da,Db,Dn*/
Word32 do__ash_rgt_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  163, INST: ash.rgt.2w #u5,Da,Dn*/

	/*ID:  164, INST: ash.rgt.4w Da,Dcd,Dmn*/
void do__ash_rgt_4w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  165, INST: ash.rgt.4w #u4,Dab,Dmn*/

	/*ID:  166, INST: ash.rgt.x Da,Db,Dn*/
Word40 do__ash_rgt_x(Word32 __Da, Word40 __Db)
{
	int shift_val,right_shift_val,i=0;
	long long resultLL = 0LL;
	long long var1LL;
	Word40 X_result = __Db;

	if (__Da & 0x40)
		shift_val = __Da | 0xffffff80;
	else
		shift_val = __Da & 0x7f;

	if ((shift_val == 0) /*|| (__x_to_l(__Db) == 0)*/)
		 return X_result;

	var1LL = __x_to_ll(__Db);

	if(shift_val < 0)
	{
		right_shift_val = -shift_val;
		var1LL = var1LL << right_shift_val;
		X_result  = __ll_to_x(var1LL);
	}
	else
	{
		var1LL = var1LL >> shift_val;
		X_result  = __ll_to_x(var1LL);
	}
	return X_result;

}

	/*ID:  167, INST: ash.rgt.x #ue5,Da,Dn*/

	/*ID:  168, INST: ash.rgt.2x Da,Dcd,Dmn*/
void do__ash_rgt_2x(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__ash_rgt_x(__Da,__Dc);
	*__Dn = do__ash_rgt_x(__Da,__Dd);
}

	/*ID:  169, INST: ash.rgt.2x #ue5,Dab,Dmn*/

	/*ID:  170, INST: ash.rgt.leg.x Da,Db,Dn*/

	/*ID:  171, INST: ash.rgt.leg.x #u5,Da,Dn*/

	/*ID:  172, INST: ash.rgt.s.l Da,Db,Dn*/
Word32 do__ash_rgt_s_l(Word32 __Da, Word32 __Db)
{

	int shift_val,left_shift_val;

	if (__Da & 0x20)
		shift_val = __Da | 0xffffffc0;
	else
		shift_val = __Da & 0x3f;

	if ((shift_val == 0) || (__Db == 0))
		return __Db;

	if(shift_val < 0)
	{
		left_shift_val = -shift_val;

		for (; left_shift_val > 0; left_shift_val--)
		{

			if (__Db > (Word32) 0X3fffffffL)
			{
			  /*Overflow = 1; set Overflow when overflow occurs in 32-bit
						arithmetic saturation mode */
				__Db = MAX_32;
				break;
			}
			else
			{
				if (__Db < (Word32) 0xc0000000L)
				{
				  /*Overflow = 1; /* set Overflow when overflow occurs in 32-bit */
				  /*			arithmetic saturation mode */
					__Db = MIN_32;
					break;
				}
			}
			__Db<<=1;
    	}

	}
	else
	{
		/*I think right right shift will never result in saturation for 32 bit data.. is this right?*/
		__Db>>= shift_val;
	}
		return __Db;
	}

	/*ID:  173, INST: ash.rgt.s.2l Da,Dcd,Dmn*/
void do__ash_rgt_s_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  174, INST: ash.rgt.s.2t Da,Db,Dn*/
Word40 do__ash_rgt_s_2t(Word32 __Da, Word40 __Db)
{
}

	/*ID:  175, INST: ash.rgt.s.2t #u5,Da,Dn*/

	/*ID:  176, INST: ash.rgt.s.4t Da,Dcd,Dmn*/
void do__ash_rgt_s_4t(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  177, INST: ash.rgt.s.4t #u5,Dab,Dmn*/

	/*ID:  178, INST: ash.rgt.s.2w Da,Db,Dn*/
Word32 do__ash_rgt_s_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  179, INST: ash.rgt.s.4w Da,Dcd,Dmn*/
void do__ash_rgt_s_4w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  180, INST: ash.rgt.s.x Da,Db,Dn*/

Word40 do__ash_rgt_s_x(Word32 __Da, Word40 __Db)
{

	int shift_val,left_shift_val,i=0;
	long long resultLL = 0LL;
	long long var1LL;
	Word40 X_result = __Db;
	shift_Overflow = 0;

	if (__Da & 0x40)
		shift_val = __Da | 0xffffff80;
	else
		shift_val = __Da & 0x7f;

	if ((shift_val == 0) || (__x_to_l(__Db) == 0))
	{
		__Db = __l_to_x(do__sat_x_l(__Db));
		if(shift_Overflow)return __Db;
		else return X_result;

	}

	var1LL = __x_to_ll(__Db);
	if(shift_val < 0)
	{

		left_shift_val = -shift_val;
		if(shift_Overflow == 0)
		{
			for(i=0;i<left_shift_val;i++)
			{
				resultLL = var1LL << 1;
				X_result  = __ll_to_x(resultLL);
				__Db = __l_to_x(do__sat_x_l(X_result));
				if(shift_Overflow)break;
			}
		}
	}
	else
	{
		if(shift_Overflow == 0)
		{
			resultLL = var1LL >> shift_val;
			X_result  = __ll_to_x(resultLL);
			__Db = __l_to_x(do__sat_x_l(X_result));
		}
	}
	if(shift_Overflow)return __Db;
		else return X_result;

}

	/*ID:  181, INST: ash.rgt.s.x #ue5,Da,Dn*/

	/*ID:  182, INST: ash.rgt.s.2x Da,Dcd,Dmn*/
void do__ash_rgt_s_2x(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  183, INST: ash.rgt.s.2x #ue5,Dab,Dmn*/

	/*ID:  184, INST: ash.rgt.s20.2t Da,Db,Dn*/
Word40 do__ash_rgt_s20_2t(Word32 __Da, Word40 __Db)
{
}

	/*ID:  185, INST: ash.rgt.s20.4t Da,Dcd,Dmn*/
void do__ash_rgt_s20_4t(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  186, INST: asha.lft Ra,Rb,Rn*/
Word32 do__asha_lft(Word32 __Ra, Word32 __Rb)
{
}

	/*ID:  187, INST: asha.lft #u5,Ra,Rn*/

	/*ID:  188, INST: asha.lft #u5,Ra,Rn*/

	/*ID:  189, INST: asha.rgt Ra,Rb,Rn*/
Word32 do__asha_rgt(Word32 __Ra, Word32 __Rb)
{
}

	/*ID:  190, INST: asha.rgt #u5,Ra,Rn*/

	/*ID:  191, INST: asha.rgt #u5,Ra,Rn*/

	/*ID:  192, INST: avg.u.4b Da,Db,Dn*/
UWord32 do__avg_u_4b(UWord32 __Da, UWord32 __Db)
{
}

	/*ID:  193, INST: bccas Ra*/
void do__bccas(Word32 __Ra)
{
}

	/*ID:  194, INST: bit.colpsh.b Dab,Dn.hh*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_b_hh(Word32 __Dn_hh, Int64 __Dab)
{
}
#endif

	/*ID:  195, INST: bit.colpsh.b Dab,Dn.hl*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_b_hl(Word32 __Dn_hl, Int64 __Dab)
{
}
#endif

	/*ID:  196, INST: bit.colpsh.b Dab,Dn.lh*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_b_lh(Word32 __Dn_lh, Int64 __Dab)
{
}
#endif

	/*ID:  197, INST: bit.colpsh.b Dab,Dn.ll*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_b_ll(Word32 __Dn_ll, Int64 __Dab)
{
}
#endif

	/*ID:  198, INST: bit.colpsh.rev.b Dab,Dn.hh*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_rev_b_hh(Word32 __Dn_hh, Int64 __Dab)
{
}
#endif

	/*ID:  199, INST: bit.colpsh.rev.b Dab,Dn.hl*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_rev_b_hl(Word32 __Dn_hl, Int64 __Dab)
{
}
#endif

	/*ID:  200, INST: bit.colpsh.rev.b Dab,Dn.lh*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_rev_b_lh(Word32 __Dn_lh, Int64 __Dab)
{
}
#endif

	/*ID:  201, INST: bit.colpsh.rev.b Dab,Dn.ll*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_rev_b_ll(Word32 __Dn_ll, Int64 __Dab)
{
}
#endif

	/*ID:  202, INST: bit.colpsh.sh.w Da,Dn*/
Word32 do__bit_colpsh_sh_w_da(Word32 __Dn, Word32 __Da)
{
}

	/*ID:  203, INST: bit.colpsh.sh.w Dab,Dn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsh_sh_w(Word32 __Dn, Int64 __Dab)
{
}
#endif

	/*ID:  204, INST: bit.colpsl.b Dab,Dn.ll*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsl_b(Word32 __Dn_ll, Int64 __Dab)
{
}
#endif

	/*ID:  205, INST: bit.colpsl.rev.b Dab,Dn.ll*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word32 do__bit_colpsl_rev_b(Word32 __Dn_ll, Int64 __Dab)
{
}
#endif

	/*ID:  206, INST: bit.dintlv2bi.l Da,Dn*/
Word32 do__bit_dintlv2bi_l(Word32 __Da)
{
}

	/*ID:  207, INST: bit.dintlv2bi.2l Dab,Dmn*/
void do__bit_dintlv2bi_2l(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  208, INST: bit.dintlv3bi.l Da,Dn*/
Word32 do__bit_dintlv3bi_l(Word32 __Da)
{
	Word32 __Dn = 0,i;
	for(i=0;i<8;i++)
	{
		__Dn =  __Dn | (((__Da & (1 << 3*i))>>3*i)<<i);
		__Dn = __Dn | (((__Da & (1 << 3*i+1))>>3*i+1)<<i+8);
		__Dn = __Dn | (((__Da & (1 << 3*i+2))>>3*i+2)<<i+16);
	}
	return __Dn;
}

	/*ID:  209, INST: bit.dintlv3bi.2l Dab,Dmn*/
void do__bit_dintlv3bi_2l(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  210, INST: bit.dintlv3bi.rev.l Da,Dn*/
Word32 do__bit_dintlv3bi_rev_l(Word32 __Da)
{
}

	/*ID:  211, INST: bit.dintlv3bi.rev.2l Dab,Dmn*/
void do__bit_dintlv3bi_rev_2l(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  212, INST: bit.expnd.8b Da.hh,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_8b_hh(Word32 __Da_hh)
{
}
#endif

	/*ID:  213, INST: bit.expnd.8b Da.hl,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_8b_hl(Word32 __Da_hl)
{
}
#endif

	/*ID:  214, INST: bit.expnd.8b Da.lh,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_8b_lh(Word32 __Da_lh)
{
}
#endif

	/*ID:  215, INST: bit.expnd.8b Da.ll,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_8b_ll(Word32 __Da_ll)
{
}
#endif

	/*ID:  216, INST: bit.expnd.2w Da,Dn*/
Word32 do__bit_expnd_2w(Word32 __Da)
{
}

	/*ID:  217, INST: bit.expnd.4w Da,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_4w(Word32 __Da)
{
}
#endif

	/*ID:  218, INST: bit.expnd.rev.8b Da.hh,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_rev_8b_hh(Word32 __Da_hh)
{
}
#endif

	/*ID:  219, INST: bit.expnd.rev.8b Da.hl,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_rev_8b_hl(Word32 __Da_hl)
{
}
#endif

	/*ID:  220, INST: bit.expnd.rev.8b Da.lh,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_rev_8b_lh(Word32 __Da_lh)
{
}
#endif

	/*ID:  221, INST: bit.expnd.rev.8b Da.ll,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__bit_expnd_rev_8b_ll(Word32 __Da_ll)
{
}
#endif

	/*ID:  222, INST: bit.intlv2bi.l Da,Dn*/
Word32 do__bit_intlv2bi_l(Word32 __Da)
{
}

	/*ID:  223, INST: bit.intlv2bi.2l Dab,Dmn*/
void do__bit_intlv2bi_2l(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  224, INST: bit.intlv3bi.l Da,Dn*/
Word32 do__bit_intlv3bi_l(Word32 __Da)
{
	Word32 __Dn = 0,i,j,k=0;
 	for(i=0;(i<9);i=i+3)
 	{
		for (j=0;j<3  && (k < 24);j++) {
 		__Dn =  __Dn | (((__Da & (1 << i+j))>>i+j)<<k++);
 		__Dn = __Dn | (((__Da & (1 << i+j+8))>>i+j+8)<<k++);
 		__Dn = __Dn | (((__Da & (1 << i+j+16))>>i+j+16)<<k++);
	   }
 	}
 	return __Dn;
}

	/*ID:  225, INST: bit.intlv3bi.2l Dab,Dmn*/
void do__bit_intlv3bi_2l(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  226, INST: bit.intlv3bi.rev.l Da,Dn*/
Word32 do__bit_intlv3bi_rev_l(Word32 __Da)
{
}

	/*ID:  227, INST: bit.intlv3bi.rev.2l Dab,Dmn*/
void do__bit_intlv3bi_rev_2l(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  228, INST: bit.rev.4b Da,Dn*/
Word32 do__bit_rev_4b(Word32 __Da)
{
	UWord8 temp;
	Word32 __Dn=0,i;
	/* 1st byte from left*/
	temp = (UWord8) (((UWord32)(__Da & 0xff000000))>>(31-7));
	temp = __swap_bits(temp);
	__Dn = __Dn | temp << (31-7);
	/*second byte from left*/
	temp = (UWord8) (((UWord32)(__Da & 0x00ff0000))>>(31-15));
	temp = __swap_bits(temp);
	__Dn = __Dn | temp << (31-15);
	/*third byte from left*/
	temp = (UWord8) (((UWord32)(__Da & 0x0000ff00))>>(31-23));
	temp = __swap_bits(temp);
	__Dn = __Dn | temp << (31-23);
	/*fourth byte from left*/
	temp = (UWord8) (__Da & 0x000000ff);
	temp = __swap_bits(temp);
	__Dn = __Dn | temp;
	return __Dn;
}

	/*ID:  229, INST: bit.rev.l Da,Dn*/
Word32 do__bit_rev_l(Word32 __Da)
{
}

	/*ID:  230, INST: bmchga #u16,C4.h*/

	/*ID:  231, INST: bmchga #u16,Rn.h*/
Word32 do__bmchga_h(Word32 __Rn_h, Word32 __imm1)
{
}

	/*ID:  232, INST: bmchga #u16,C4.l*/

	/*ID:  233, INST: bmchga #u16,Rn.l*/
Word32 do__bmchga_l(Word32 __Rn_l, Word32 __imm1)
{
}

	/*ID:  234, INST: bmclra #u16,C4.h*/

	/*ID:  235, INST: bmclra #u16,Rn.h*/
Word32 do__bmclra_h(Word32 __Rn_h, Word32 __imm1)
{
}

	/*ID:  236, INST: bmclra #u16,C4.l*/

	/*ID:  237, INST: bmclra #u16,Rn.l*/
Word32 do__bmclra_l(Word32 __Rn_l, Word32 __imm1)
{
}

	/*ID:  238, INST: bmseta #u16,C4.h*/

	/*ID:  239, INST: bmseta #u16,Rn.h*/
Word32 do__bmseta_h(Word32 __Rn_h, Word32 __imm1)
{
}

	/*ID:  240, INST: bmseta #u16,C4.l*/

	/*ID:  241, INST: bmseta #u16,Rn.l*/
Word32 do__bmseta_l(Word32 __Rn_l, Word32 __imm1)
{
}

	/*ID:  242, INST: bmtsta.c #u16,C4.h,Pmn*/

	/*ID:  243, INST: bmtsta.c #u16,C4.h,Pn*/

	/*ID:  244, INST: bmtsta.c #u16,Ra.h,Pmn*/
int do__bmtsta_c_h(Word32 __imm1, Word32 __Ra_h)
{
}

	/*ID:  245, INST: bmtsta.c #u16,Ra.h,Pn*/

	/*ID:  246, INST: bmtsta.c #u16,C4.l,Pmn*/

	/*ID:  247, INST: bmtsta.c #u16,C4.l,Pn*/

	/*ID:  248, INST: bmtsta.c #u16,Ra.l,Pmn*/
int do__bmtsta_c_l(Word32 __imm1, Word32 __Ra_l)
{
}

	/*ID:  249, INST: bmtsta.c #u16,Ra.l,Pn*/

	/*ID:  250, INST: bmtsta.s #u16,C4.h,Pmn*/

	/*ID:  251, INST: bmtsta.s #u16,C4.h,Pn*/

	/*ID:  252, INST: bmtsta.s #u16,Ra.h,Pmn*/
int do__bmtsta_s_h(Word32 __imm1, Word32 __Ra_h)
{
}

	/*ID:  253, INST: bmtsta.s #u16,Ra.h,Pn*/

	/*ID:  254, INST: bmtsta.s #u16,C4.l,Pmn*/

	/*ID:  255, INST: bmtsta.s #u16,C4.l,Pn*/

	/*ID:  256, INST: bmtsta.s #u16,Ra.l,Pmn*/
int do__bmtsta_s_l(Word32 __imm1, Word32 __Ra_l)
{
}

	/*ID:  257, INST: bmtsta.s #u16,Ra.l,Pn*/

	/*ID:  258, INST: bra RelAdd19_t2*/

	/*ID:  259, INST: bra.nobtb RelAdd19_t2*/

	/*ID:  260, INST: break.2 RelAdd19_t1*/

	/*ID:  261, INST: break.1 RelAdd19_t1*/

	/*ID:  262, INST: break.3 RelAdd19_t1*/

	/*ID:  263, INST: break.0 RelAdd19_t1*/

	/*ID:  264, INST: bsr RelAdd19_t2*/

	/*ID:  265, INST: bsr.nobtb RelAdd19_t2*/

	/*ID:  266, INST: cast.t.x Da.h,Dn*/
Word40 do__cast_t_x_h(Word40 __Da_h)
{
}

	/*ID:  267, INST: cast.t.x Da.l,Dn*/
Word40 do__cast_t_x_l(Word40 __Da_l)
{
}

	/*ID:  268, INST: cast.t.leg.x Da.h,Dn*/

	/*ID:  269, INST: cast.t.leg.x Da.l,Dn*/

	/*ID:  270, INST: cast.x.ll Da,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__cast_x_ll(Word40 __Da)
{
}
#endif

	/*ID:  271, INST: ccmd.resm*/
void do__ccmd_resm()
{
}

	/*ID:  272, INST: ccmd.susp*/
void do__ccmd_susp()
{
}

	/*ID:  273, INST: chcf #u1_t2,SR.RM*/

	/*ID:  274, INST: chcf #u1_t2,SR.SM*/

	/*ID:  275, INST: chcf #u2_t2,SR.SWM2*/

	/*ID:  276, INST: chcf #u2_t2,SR2.ASP*/

	/*ID:  277, INST: chcf #u3_t3,SR.SCM*/

	/*ID:  278, INST: chcf Ra,SR2.ASP*/

	/*ID:  279, INST: chcf Ra,SR2.IPM*/

	/*ID:  280, INST: clb.leg.x Da,Dn*/

	/*ID:  281, INST: clb.lft.x Da,Dn*/
Word40 do__clb_lft_x(Word40 __Da)
{
}

	/*ID:  282, INST: clba.lft Ra,Rn*/
Word32 do__clba_lft(Word32 __Ra)
{
}

	/*ID:  283, INST: clba.lft Ra,Rn*/

	/*ID:  284, INST: clip.t.u.2b #u5,Da,Dn*/
UWord32 do__clip_t_u_2b(UWord32 __imm1, UWord40 __Da)
{
}

	/*ID:  285, INST: clip.w.u.2b #u5,Da,Dn*/
UWord32 do__clip_w_u_2b(UWord32 __imm1, UWord32 __Da)
{
}

	/*ID:  286, INST: clip.x.u.b #u5,Da,Dn*/
UWord32 do__clip_x_u_b(UWord32 __imm1, UWord40 __Da)
{
}

	/*ID:  287, INST: clric #u5_t2*/
void do__clric(Word32 __imm1)
{
}

	/*ID:  288, INST: cmp.eq.l #s32,Da,Pmn*/

	/*ID:  289, INST: cmp.eq.l #s32,Da,Pn*/

	/*ID:  290, INST: cmp.eq.l #ux5,Da,Pmn*/

	/*ID:  291, INST: cmp.eq.l #ux5,Da,Pn*/

	/*ID:  292, INST: cmp.eq.l Da,Db,Pmn*/

	/*ID:  293, INST: cmp.eq.l Da,Db,Pn*/

	/*ID:  294, INST: cmp.eq.w Da.l,Db.l,Pmn*/

	/*ID:  295, INST: cmp.eq.w Da.l,Db.l,Pn*/

	/*ID:  296, INST: cmp.eq.x #s32,Da,Pmn*/
int do__cmp_eq_x_imm(Word32 __imm1, Word40 __Da)
{
}

	/*ID:  297, INST: cmp.eq.x #s32,Da,Pn*/

	/*ID:  298, INST: cmp.eq.x #ux5,Da,Pmn*/

	/*ID:  299, INST: cmp.eq.x #ux5,Da,Pn*/

	/*ID:  300, INST: cmp.eq.x Da,Db,Pmn*/
int do__cmp_eq_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  301, INST: cmp.eq.x Da,Db,Pn*/

	/*ID:  302, INST: cmp.ge.l Da,Db,Pmn*/

	/*ID:  303, INST: cmp.ge.l Da,Db,Pn*/

	/*ID:  304, INST: cmp.ge.w Da.l,Db.l,Pmn*/

	/*ID:  305, INST: cmp.ge.w Da.l,Db.l,Pn*/

	/*ID:  306, INST: cmp.ge.x Da,Db,Pmn*/
int do__cmp_ge_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  307, INST: cmp.ge.x Da,Db,Pn*/

	/*ID:  308, INST: cmp.ge.u.l Da,Db,Pmn*/

	/*ID:  309, INST: cmp.ge.u.l Da,Db,Pn*/

	/*ID:  310, INST: cmp.ge.u.w Da.l,Db.l,Pmn*/

	/*ID:  311, INST: cmp.ge.u.w Da.l,Db.l,Pn*/

	/*ID:  312, INST: cmp.ge.u.x Da,Db,Pmn*/
int do__cmp_ge_u_x(UWord40 __Da, UWord40 __Db)
{
}

	/*ID:  313, INST: cmp.ge.u.x Da,Db,Pn*/

	/*ID:  314, INST: cmp.gt.l #s32,Da,Pmn*/

	/*ID:  315, INST: cmp.gt.l #s32,Da,Pn*/

	/*ID:  316, INST: cmp.gt.l #ux5,Da,Pmn*/

	/*ID:  317, INST: cmp.gt.l #ux5,Da,Pn*/

	/*ID:  318, INST: cmp.gt.l Da,Db,Pmn*/

	/*ID:  319, INST: cmp.gt.l Da,Db,Pn*/

	/*ID:  320, INST: cmp.gt.w Da.l,Db.l,Pmn*/

	/*ID:  321, INST: cmp.gt.w Da.l,Db.l,Pn*/

	/*ID:  322, INST: cmp.gt.x #s32,Da,Pmn*/
int do__cmp_gt_x_imm(Word32 __imm1, Word40 __Da)
{
}

	/*ID:  323, INST: cmp.gt.x #s32,Da,Pn*/

	/*ID:  324, INST: cmp.gt.x #ux5,Da,Pmn*/

	/*ID:  325, INST: cmp.gt.x #ux5,Da,Pn*/

	/*ID:  326, INST: cmp.gt.x Da,Db,Pmn*/
int do__cmp_gt_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  327, INST: cmp.gt.x Da,Db,Pn*/

	/*ID:  328, INST: cmp.gt.u.l #u32,Da,Pmn*/

	/*ID:  329, INST: cmp.gt.u.l #u32,Da,Pn*/

	/*ID:  330, INST: cmp.gt.u.l #u5,Da,Pmn*/

	/*ID:  331, INST: cmp.gt.u.l #u5,Da,Pn*/

	/*ID:  332, INST: cmp.gt.u.l Da,Db,Pmn*/

	/*ID:  333, INST: cmp.gt.u.l Da,Db,Pn*/

	/*ID:  334, INST: cmp.gt.u.w Da.l,Db.l,Pmn*/

	/*ID:  335, INST: cmp.gt.u.w Da.l,Db.l,Pn*/

	/*ID:  336, INST: cmp.gt.u.x Da,Db,Pmn*/
int do__cmp_gt_u_x(UWord40 __Da, UWord40 __Db)
{
}

	/*ID:  337, INST: cmp.gt.u.x Da,Db,Pn*/

	/*ID:  338, INST: cmp.le.l #s32,Da,Pmn*/

	/*ID:  339, INST: cmp.le.l #s32,Da,Pn*/

	/*ID:  340, INST: cmp.le.l #ux5,Da,Pmn*/

	/*ID:  341, INST: cmp.le.l #ux5,Da,Pn*/

	/*ID:  342, INST: cmp.le.x #s32,Da,Pmn*/
int do__cmp_le_x(Word32 __imm1, Word40 __Da)
{
}

	/*ID:  343, INST: cmp.le.x #s32,Da,Pn*/

	/*ID:  344, INST: cmp.le.x #ux5,Da,Pmn*/

	/*ID:  345, INST: cmp.le.x #ux5,Da,Pn*/

	/*ID:  346, INST: cmp.le.u.l #u32,Da,Pmn*/

	/*ID:  347, INST: cmp.le.u.l #u32,Da,Pn*/

	/*ID:  348, INST: cmp.le.u.l #u5,Da,Pmn*/

	/*ID:  349, INST: cmp.le.u.l #u5,Da,Pn*/

	/*ID:  350, INST: cmp.ne.l #s32,Da,Pmn*/

	/*ID:  351, INST: cmp.ne.l #s32,Da,Pn*/

	/*ID:  352, INST: cmp.ne.l #ux5,Da,Pmn*/

	/*ID:  353, INST: cmp.ne.l #ux5,Da,Pn*/

	/*ID:  354, INST: cmp.ne.l Da,Db,Pmn*/

	/*ID:  355, INST: cmp.ne.l Da,Db,Pn*/

	/*ID:  356, INST: cmp.ne.w Da.l,Db.l,Pmn*/

	/*ID:  357, INST: cmp.ne.w Da.l,Db.l,Pn*/

	/*ID:  358, INST: cmp.ne.x #s32,Da,Pmn*/
int do__cmp_ne_x_imm(Word32 __imm1, Word40 __Da)
{
}

	/*ID:  359, INST: cmp.ne.x #s32,Da,Pn*/

	/*ID:  360, INST: cmp.ne.x #ux5,Da,Pmn*/

	/*ID:  361, INST: cmp.ne.x #ux5,Da,Pn*/

	/*ID:  362, INST: cmp.ne.x Da,Db,Pmn*/
int do__cmp_ne_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  363, INST: cmp.ne.x Da,Db,Pn*/

	/*ID:  364, INST: cmpa.eq #s16,Ra,Pmn*/

	/*ID:  365, INST: cmpa.eq #s16,Ra,Pn*/

	/*ID:  366, INST: cmpa.eq #s16_t7,Ra,Pmn*/

	/*ID:  367, INST: cmpa.eq #s16_t7,Ra,Pn*/

	/*ID:  368, INST: cmpa.eq #ux5,Ra,Pmn*/

	/*ID:  369, INST: cmpa.eq #ux5,Ra,Pmn*/

	/*ID:  370, INST: cmpa.eq #ux5,Ra,Pn*/

	/*ID:  371, INST: cmpa.eq #ux5,Ra,Pn*/

	/*ID:  372, INST: cmpa.eq Ra,Rb,Pmn*/

	/*ID:  373, INST: cmpa.eq Ra,Rb,Pmn*/

	/*ID:  374, INST: cmpa.eq Ra,Rb,Pn*/

	/*ID:  375, INST: cmpa.eq Ra,Rb,Pn*/

	/*ID:  376, INST: cmpa.ge Ra,Rb,Pmn*/

	/*ID:  377, INST: cmpa.ge Ra,Rb,Pmn*/

	/*ID:  378, INST: cmpa.ge Ra,Rb,Pn*/

	/*ID:  379, INST: cmpa.ge Ra,Rb,Pn*/

	/*ID:  380, INST: cmpa.gt #s16,Ra,Pmn*/

	/*ID:  381, INST: cmpa.gt #s16,Ra,Pn*/

	/*ID:  382, INST: cmpa.gt #s16_t7,Ra,Pmn*/

	/*ID:  383, INST: cmpa.gt #s16_t7,Ra,Pn*/

	/*ID:  384, INST: cmpa.gt #ux5,Ra,Pmn*/

	/*ID:  385, INST: cmpa.gt #ux5,Ra,Pmn*/

	/*ID:  386, INST: cmpa.gt #ux5,Ra,Pn*/

	/*ID:  387, INST: cmpa.gt #ux5,Ra,Pn*/

	/*ID:  388, INST: cmpa.gt Ra,Rb,Pmn*/

	/*ID:  389, INST: cmpa.gt Ra,Rb,Pmn*/

	/*ID:  390, INST: cmpa.gt Ra,Rb,Pn*/

	/*ID:  391, INST: cmpa.gt Ra,Rb,Pn*/

	/*ID:  392, INST: cmpa.le #s16,Ra,Pmn*/

	/*ID:  393, INST: cmpa.le #s16,Ra,Pn*/

	/*ID:  394, INST: cmpa.le #s16_t7,Ra,Pmn*/

	/*ID:  395, INST: cmpa.le #s16_t7,Ra,Pn*/

	/*ID:  396, INST: cmpa.le #ux5,Ra,Pmn*/

	/*ID:  397, INST: cmpa.le #ux5,Ra,Pmn*/

	/*ID:  398, INST: cmpa.le #ux5,Ra,Pn*/

	/*ID:  399, INST: cmpa.le #ux5,Ra,Pn*/

	/*ID:  400, INST: cmpa.ne #s16,Ra,Pmn*/

	/*ID:  401, INST: cmpa.ne #s16,Ra,Pn*/

	/*ID:  402, INST: cmpa.ne #s16_t7,Ra,Pmn*/

	/*ID:  403, INST: cmpa.ne #s16_t7,Ra,Pn*/

	/*ID:  404, INST: cmpa.ne #ux5,Ra,Pmn*/

	/*ID:  405, INST: cmpa.ne #ux5,Ra,Pmn*/

	/*ID:  406, INST: cmpa.ne #ux5,Ra,Pn*/

	/*ID:  407, INST: cmpa.ne #ux5,Ra,Pn*/

	/*ID:  408, INST: cmpa.ne Ra,Rb,Pmn*/

	/*ID:  409, INST: cmpa.ne Ra,Rb,Pmn*/

	/*ID:  410, INST: cmpa.ne Ra,Rb,Pn*/

	/*ID:  411, INST: cmpa.ne Ra,Rb,Pn*/

	/*ID:  412, INST: cmpa.ge.u Ra,Rb,Pmn*/

	/*ID:  413, INST: cmpa.ge.u Ra,Rb,Pmn*/

	/*ID:  414, INST: cmpa.ge.u Ra,Rb,Pn*/

	/*ID:  415, INST: cmpa.ge.u Ra,Rb,Pn*/

	/*ID:  416, INST: cmpa.gt.u #u16,Ra,Pmn*/

	/*ID:  417, INST: cmpa.gt.u #u16,Ra,Pn*/

	/*ID:  418, INST: cmpa.gt.u #u16_t7,Ra,Pmn*/

	/*ID:  419, INST: cmpa.gt.u #u16_t7,Ra,Pn*/

	/*ID:  420, INST: cmpa.gt.u Ra,Rb,Pmn*/

	/*ID:  421, INST: cmpa.gt.u Ra,Rb,Pmn*/

	/*ID:  422, INST: cmpa.gt.u Ra,Rb,Pn*/

	/*ID:  423, INST: cmpa.gt.u Ra,Rb,Pn*/

	/*ID:  424, INST: cmpa.le.u #u16,Ra,Pmn*/

	/*ID:  425, INST: cmpa.le.u #u16,Ra,Pn*/

	/*ID:  426, INST: cmpa.le.u #u16_t7,Ra,Pmn*/

	/*ID:  427, INST: cmpa.le.u #u16_t7,Ra,Pn*/

	/*ID:  428, INST: cmpd.eq.2t Da,Db,Dn*/
Word40 do__cmpd_eq_2t(Word40 __Da, Word40 __Db)
{
}

	/*ID:  429, INST: cmpd.eq.4t Dab,Dcd,Dmn*/
void do__cmpd_eq_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  430, INST: cmpd.eq.2w Da,Db,Dn*/
Word32 do__cmpd_eq_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  431, INST: cmpd.eq.4w Dab,Dcd,Dmn*/
void do__cmpd_eq_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  432, INST: cmpd.eq.x #s32,Da,Dn*/
Word40 do__cmpd_eq_x_imm(Word32 __imm1, Word40 __Da)
{
	Word40 Dn;
	Dn = (do__cmpd_eq_x(__l_to_x( __imm1), __Da));
	return Dn;
}

	/*ID:  433, INST: cmpd.eq.x Da,Db,Dn*/
/*
  Function Name: do__cmpd_eq_x
  Purpose: Compare two 40 bit words. If they are equal return -1 otheriwse 0.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns a 40 bit word having 0 or -1 value.
*/
Word40 do__cmpd_eq_x(Word40 __Da, Word40 __Db)
{
   Word40 __Dn;

   __Dn = ((__x_guard(__Da) == __x_guard(__Db)) && (__x_to_l(__Da) == __x_to_l(__Db))) ? __x_set(0xff, 0xffffffff) : __x_set(0,0);

   return __Dn;
}

	/*ID:  434, INST: cmpd.eq.2x Dab,Dcd,Dmn*/
void do__cmpd_eq_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  435, INST: cmpd.ge.2t Da,Db,Dn*/
Word40 do__cmpd_ge_2t(Word40 __Da, Word40 __Db)
{
}

	/*ID:  436, INST: cmpd.ge.4t Dab,Dcd,Dmn*/
void do__cmpd_ge_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  437, INST: cmpd.ge.2w Da,Db,Dn*/
Word32 do__cmpd_ge_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  438, INST: cmpd.ge.4w Dab,Dcd,Dmn*/
void do__cmpd_ge_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  439, INST: cmpd.ge.x #s32,Da,Dn*/
Word40 do__cmpd_ge_x_imm(Word32 __imm1, Word40 __Da)
{
	Word40 Dn;
	Dn = do__cmpd_ge_x(__l_to_x(__imm1), __Da);
	return Dn;

}

	/*ID:  440, INST: cmpd.ge.x Da,Db,Dn*/
/*
  Function Name: do__cmpd_ge_x
  Purpose: Compare two 40 bit words. If the first operand is greater than second , return 0, otherwise -1.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns a 40 bit word having 0 or -1 value.
*/
Word40 do__cmpd_ge_x(Word40 __Da, Word40 __Db)
{
   Word8 __Dn_1_ext, __Dn_2_ext;
   UWord32 __Dn_1, __Dn_2;
   Word40 __Dn;

   __Dn_1_ext = __x_guard(__Da);
   __Dn_2_ext = __x_guard(__Db);

   __Dn_1 = __x_to_l(__Da);
   __Dn_2 = __x_to_l(__Db);

   if ((__Dn_1_ext & 0x80) && (__Dn_2_ext & 0x80)) { /* both are negative */
     if (__Dn_1_ext == __Dn_2_ext) {
          __Dn = (__Dn_1 <= __Dn_2) ? __x_set(0,0) : __x_set(0xff, 0xffffffff);
	 } else {
       __Dn = (__Dn_1_ext <= __Dn_2_ext) ? __x_set(0,0) : __x_set(0xff, 0xffffffff);
	 }
   } else if ((__Dn_1_ext & 0x80) && ((__Dn_2_ext & 0x80) == 0)) { /* Da < Db */
     __Dn = __x_set(0xff, 0xffffffff);
   } else if ((__Dn_2_ext & 0x80) && ((__Dn_1_ext & 0x80) == 0)) { /* Da > Db */
     __Dn = __x_set(0x0, 0x0);
   } else {
     if (__Dn_1_ext == __Dn_2_ext) {
       __Dn = (__Dn_1 <= __Dn_2) ? __x_set(0xff, 0xffffffff) :__x_set(0,0);
	 } else {
       __Dn = (__Dn_1_ext <= __Dn_2_ext) ? __x_set(0xff, 0xffffffff) :__x_set(0,0);
	 }

   }
   return __Dn;
}

	/*ID:  441, INST: cmpd.ge.2x Dab,Dcd,Dmn*/
void do__cmpd_ge_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  442, INST: cmpd.gt.2t Da,Db,Dn*/
Word40 do__cmpd_gt_2t(Word40 __Da, Word40 __Db)
{
}

	/*ID:  443, INST: cmpd.gt.4t Dab,Dcd,Dmn*/
void do__cmpd_gt_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  444, INST: cmpd.gt.2w Da,Db,Dn*/
Word32 do__cmpd_gt_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  445, INST: cmpd.gt.4w Dab,Dcd,Dmn*/
void do__cmpd_gt_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  446, INST: cmpd.gt.x #s32,Da,Dn*/
Word40 do__cmpd_gt_x_imm(Word32 __imm1, Word40 __Da)
{
	Word40 Dn;
	Dn = do__cmpd_gt_x(__l_to_x(__imm1), __Da);
	return Dn;
}

	/*ID:  447, INST: cmpd.gt.x Da,Db,Dn*/
/*
  Function Name: do__cmpd_gt_x
  Purpose: Compare two 40 bit words. If the first operand is greater than second , return 0, otherwise -1.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns a 40 bit word having 0 or -1 value.
*/
Word40 do__cmpd_gt_x(Word40 __Da, Word40 __Db)
{
   Word8 __Dn_1_ext, __Dn_2_ext;
   Word32 __Dn_1, __Dn_2;
   Word40 __Dn;

   __Dn_1_ext = __x_guard(__Da);
   __Dn_2_ext = __x_guard(__Db);

   __Dn_1 = __x_to_l(__Da);
   __Dn_2 = __x_to_l(__Db);

   if ((__Dn_1_ext & 0x80) && (__Dn_2_ext & 0x80)) { /* both are negative */
     if (__Dn_1_ext == __Dn_2_ext) {
		/* for ex: 0xff-ffffff80 > 0xff-00000000 :true for all cases where one of the operand is positive*/
		 if(__Dn_1 < 0 && __Dn_2 < 0)
         	__Dn = (__Dn_1 < __Dn_2) ? __x_set(0xff, 0xffffffff):__x_set(0,0);
         else if(__Dn_1 < 0)
         	__Dn = __x_set(0,0);
         else if(__Dn_2 < 0)
         	__Dn = __x_set(0xff, 0xffffffff);
         else /*0xff-00000010 > 0xff-01000000*/
         	__Dn = (__Dn_1 < __Dn_2) ? __x_set(0,0) :__x_set(0xff, 0xffffffff);

	 } else {
       __Dn = (__Dn_1_ext < __Dn_2_ext) ?  __x_set(0xff, 0xffffffff):__x_set(0,0);
	 }
   } else if ((__Dn_1_ext & 0x80) && ((__Dn_2_ext & 0x80) == 0)) { /* Da < Db */
     __Dn = __x_set(0xff, 0xffffffff);
   } else if ((__Dn_2_ext & 0x80) && ((__Dn_1_ext & 0x80) == 0)) { /* Da > Db */
     __Dn = __x_set(0x0, 0x0);
   } else {
     if (__Dn_1_ext == __Dn_2_ext) {
       __Dn = ((UWord32)__Dn_1 < (UWord32)__Dn_2) ? __x_set(0xff, 0xffffffff) :__x_set(0,0);
	 } else {
       __Dn = (__Dn_1_ext < __Dn_2_ext) ? __x_set(0xff, 0xffffffff) :__x_set(0,0);
	 }

   }
   return __Dn;
}

	/*ID:  448, INST: cmpd.gt.2x Dab,Dcd,Dmn*/
void do__cmpd_gt_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  449, INST: cmpd.le.x #s32,Da,Dn*/
/*
  Function Name: do__cmpd_le_x
  Purpose: Compare two 40 bit words. If the first operand is less than or equal to second , return 0, otherwise -1.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns a 40 bit word having 0 or -1 value.
*/
Word40 do__cmpd_le_x(Word32 __imm1, Word40 __Da)
{
	Word40 result;
	result = do__cmpd_gt_x(__l_to_x(__imm1),__Da);
	if(__x_guard(result) == 0)
		return __x_set(0xff,0xffffffff);/*__imm1 > __Da */
	else
		return __x_set(0,0);
}

	/*ID:  450, INST: cmpd.lt.x #s32,Da,Dn*/
Word40 do__cmpd_lt_x(Word32 __imm1, Word40 __Da)
{
	Word40 result;
	result = do__cmpd_ge_x(__l_to_x(__imm1),__Da);
	if(__x_guard(result) == 0)
		return __x_set(0xff,0xffffffff);/*__imm1 > __Da */
	else
		return __x_set(0,0);
}

	/*ID:  451, INST: cmpd.ne.2t Da,Db,Dn*/
Word40 do__cmpd_ne_2t(Word40 __Da, Word40 __Db)
{
}

	/*ID:  452, INST: cmpd.ne.4t Dab,Dcd,Dmn*/
void do__cmpd_ne_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  453, INST: cmpd.ne.2w Da,Db,Dn*/
Word32 do__cmpd_ne_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID:  454, INST: cmpd.ne.4w Dab,Dcd,Dmn*/
void do__cmpd_ne_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  455, INST: cmpd.ne.x #s32,Da,Dn*/
Word40 do__cmpd_ne_x_imm(Word32 __imm1, Word40 __Da)
{
	Word40 Dn;
	Dn = do__cmpd_ne_x(__l_to_x(__imm1), __Da);
	return Dn;
}

	/*ID:  456, INST: cmpd.ne.x Da,Db,Dn*/
/*
  Function Name: do__cmpd_ne_x
  Purpose: Compare two 40 bit words. If they are not equal then return -1 otheriwse 0.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns a 40 bit word having 0 or -1 value.
*/
Word40 do__cmpd_ne_x(Word40 __Da, Word40 __Db)
{
   Word40 __Dn;

   __Dn = ((__x_guard(__Da) != __x_guard(__Db)) || (__x_to_l(__Da) != __x_to_l(__Db))) ? __x_set(0xff, 0xffffffff): __x_set(0, 0);

   return __Dn;
}

	/*ID:  457, INST: cmpd.ne.2x Dab,Dcd,Dmn*/
void do__cmpd_ne_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  458, INST: cneg.n.4b #u3_2,Da,Db,Dn*/
Word32 do__cneg_n_4b(Word32 __imm1, Word32 __Da, Word32 __Db)
{
	Word32 __Dn = 0;
	Word8 hh,hl,lh,ll,neg = 0;

	hh = (__Db>>24)&0xFF;
	hl = (__Db>>16)&0xFF;
	lh = (__Db>>8)&0xFF;
	ll = __Db & 0xFF;

	__Dn = (__Da & (1<<(__imm1+3)))? __Dn|((__neg_sat8(hh)<<24)&0xFF000000): __Dn|((hh<<24)&0xFF000000);
	__Dn = (__Da & (1<<(__imm1+2)))? __Dn|((__neg_sat8(hl)<<16)&0x00FF0000): __Dn|((hl<<16)&0x00FF0000);
	__Dn = (__Da & (1<<(__imm1+1)))? __Dn|((__neg_sat8(lh)<<8)&0xFF00): __Dn|((lh<<8)&0xFF00);
	__Dn = (__Da & (1<<(__imm1+0)))? __Dn|(__neg_sat8(ll)&0xFF): __Dn|(ll&0xFF);

	return __Dn;
}

	/*ID:  459, INST: cneg.n.4t #u3_2,Da,Dcd,Dmn*/
void do__cneg_n_4t(Word32 __imm1, Word40 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word32 __Dc_WH,__Dc_WL,__Dd_WH,__Dd_WL;

	__Dc_WH = __extract_wh(__Dc);
	__Dc_WL = __extract_wl(__Dc);
	__Dd_WH = __extract_wh(__Dd);
	__Dd_WL = __extract_wl(__Dd);

	__Dc_WH = (__x_to_l(__Da) & (1<<(__imm1+3)))? (0 - __Dc_WH):__Dc_WH;
	__Dc_WL = (__x_to_l(__Da) & (1<<(__imm1+2)))? (0 - __Dc_WL):__Dc_WL;
	__Dd_WH = (__x_to_l(__Da) & (1<<(__imm1+1)))? (0 - __Dd_WH):__Dd_WH;
	__Dd_WL = (__x_to_l(__Da) & (1<<(__imm1+0)))? (0 - __Dd_WL):__Dd_WL;

	 /*Store the result*/
	 __set_WH(__Dc_WH,__Dm);
	 __set_WL(__Dc_WL,__Dm);
	 __set_WH(__Dd_WH,__Dn);
	 __set_WL(__Dd_WL,__Dn);
}

	/*ID:  460, INST: cneg.n.4w #u3_2,Da,Dcd,Dmn*/
void do__cneg_n_4w(Word32 __imm1, Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  461, INST: cneg.n.rev.4b #u3_2,Da,Db,Dn*/
Word32 do__cneg_n_rev_4b(Word32 __imm1, Word32 __Da, Word32 __Db)
{
}

	/*ID:  462, INST: cneg.n.rev.4t #u3_2,Da,Dcd,Dmn*/
void do__cneg_n_rev_4t(Word32 __imm1, Word40 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  463, INST: cneg.n.rev.4w #u3_2,Da,Dcd,Dmn*/
void do__cneg_n_rev_4w(Word32 __imm1, Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  464, INST: cnegadd.n.2x #u3_2,Da,Dcd,Dmn*/
void do__cnegadd_n_2x(Word32 __imm1, Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  465, INST: cnegadd.n.rev.2x #u3_2,Da,Dcd,Dmn*/
void do__cnegadd_n_rev_2x(Word32 __imm1, Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  466, INST: cob.4b Da,Dn*/
Word32 do__cob_4b(Word32 __Da)
{
}

	/*ID:  467, INST: cob.l Da,Dn*/
Word32 do__cob_l(Word32 __Da)
{
}

	/*ID:  468, INST: cont.2 RelAdd19_t1,Ra*/

	/*ID:  469, INST: cont.1 RelAdd19_t1,Ra*/

	/*ID:  470, INST: cont.3 RelAdd19_t1,Ra*/

	/*ID:  471, INST: cont.0 RelAdd19_t1,Ra*/

	/*ID:  472, INST: dbar.eieio*/
void do__dbar_eieio()
{
}

	/*ID:  473, INST: dbar.hwsync*/
void do__dbar_hwsync()
{
}

	/*ID:  474, INST: dbar.ibll*/
void do__dbar_ibll()
{
}

	/*ID:  475, INST: dbar.ibsl*/
void do__dbar_ibsl()
{
}

	/*ID:  476, INST: dbar.l1sync*/
void do__dbar_l1sync()
{
}

	/*ID:  477, INST: dbar.scfg*/
void do__dbar_scfg()
{
}

	/*ID:  478, INST: dbars.ibss.l1*/
void do__dbars_ibss_l1()
{
}

	/*ID:  479, INST: dbars.ibss.l12*/
void do__dbars_ibss_l12()
{
}

	/*ID:  480, INST: dcm.flush.l1 (Rn)*/
void do__dcm_flush_l1(Word8* __p)
{
}

	/*ID:  481, INST: dcm.flush.l12 (Rn)*/
void do__dcm_flush_l12(Word8* __p)
{
}

	/*ID:  482, INST: dcm.inval.l1 (Rn)*/
void do__dcm_inval_l1(Word8* __p)
{
}

	/*ID:  483, INST: dcm.inval.l12 (Rn)*/
void do__dcm_inval_l12(Word8* __p)
{
}

	/*ID:  484, INST: dcm.sync.l1 (Rn)*/
void do__dcm_sync_l1(Word8* __p)
{
}

	/*ID:  485, INST: dcm.sync.l12 (Rn)*/
void do__dcm_sync_l12(Word8* __p)
{
}

	/*ID:  486, INST: dcmb.flush.l12 Ra,Rn*/
Word32 do__dcmb_flush_l12(Word32 __Ra)
{
}

	/*ID:  487, INST: dcmb.inval.l1 Ra,Rn*/
Word32 do__dcmb_inval_l1(Word32 __Ra)
{
}

	/*ID:  488, INST: dcmb.inval.l12 Ra,Rn*/
Word32 do__dcmb_inval_l12(Word32 __Ra)
{
}

	/*ID:  489, INST: dcmb.sync.l1 Ra,Rn*/
Word32 do__dcmb_sync_l1(Word32 __Ra)
{
}

	/*ID:  490, INST: dcmb.sync.l2 Ra,Rn*/
Word32 do__dcmb_sync_l2(Word32 __Ra)
{
}

	/*ID:  491, INST: debuge.2 #u10*/
void do__debuge_2(Word32 __imm1)
{
}

	/*ID:  492, INST: debuge.1 #u10*/
void do__debuge_1(Word32 __imm1)
{
}

	/*ID:  493, INST: debuge.3 #u10*/
void do__debuge_3(Word32 __imm1)
{
}

	/*ID:  494, INST: debuge.0 #u10*/
void do__debuge_0(Word32 __imm1)
{
}

	/*ID:  495, INST: debuge.0.fs #u10*/
void do__debuge_0_fs(Word32 __imm1)
{
}

	/*ID:  496, INST: debuge.1.fs #u10*/
void do__debuge_1_fs(Word32 __imm1)
{
}

	/*ID:  497, INST: debuge.2.fs #u10*/
void do__debuge_2_fs(Word32 __imm1)
{
}

	/*ID:  498, INST: debuge.3.fs #u10*/
void do__debuge_3_fs(Word32 __imm1)
{
}

	/*ID:  499, INST: debugev.2*/
void do__debugev_2()
{
}

	/*ID:  500, INST: debugev.4*/
void do__debugev_4()
{
}

	/*ID:  501, INST: debugev.1*/
void do__debugev_1()
{
}

	/*ID:  502, INST: debugev.3*/
void do__debugev_3()
{
}

	/*ID:  503, INST: debugev.0*/
void do__debugev_0()
{
}

	/*ID:  504, INST: debugev.5*/
void do__debugev_5()
{
}

	/*ID:  505, INST: debugev.6*/
void do__debugev_6()
{
}

	/*ID:  506, INST: debugev.7*/
void do__debugev_7()
{
}

	/*ID:  507, INST: debugm.2*/
void do__debugm_2()
{
}

	/*ID:  508, INST: debugm.1*/
void do__debugm_1()
{
}

	/*ID:  509, INST: debugm.3*/
void do__debugm_3()
{
}

	/*ID:  510, INST: debugm.0*/
void do__debugm_0()
{
}

	/*ID:  511, INST: dec.ge.x Dn,Pmn*/

	/*ID:  512, INST: dec.ge.x Dn,Pn*/

	/*ID:  513, INST: dec.ne.x Dn,Pmn*/

	/*ID:  514, INST: dec.ne.x Dn,Pn*/

	/*ID:  515, INST: deca.ge Rn,Pmn*/

	/*ID:  516, INST: deca.ge Rn,Pmn*/

	/*ID:  517, INST: deca.ge Rn,Pn*/

	/*ID:  518, INST: deca.ge Rn,Pn*/

	/*ID:  519, INST: deca.ne Rn,Pmn*/

	/*ID:  520, INST: deca.ne Rn,Pmn*/

	/*ID:  521, INST: deca.ne Rn,Pn*/

	/*ID:  522, INST: deca.ne Rn,Pn*/

	/*ID:  523, INST: decor #u4_t6*/

	/*ID:  524, INST: dfetch.l12 (Rn)*/
void do__dfetch_l12(Word8* __p)
{
}

	/*ID:  525, INST: dfetch.l2 (Rn)*/
void do__dfetch_l2(Word8* __p)
{
}

	/*ID:  526, INST: dfetchb.l12 Ra,Rn*/
Word32 do__dfetchb_l12(Word32 __Ra)
{
}

	/*ID:  527, INST: dfetchb.l2 Ra,Rn*/
Word32 do__dfetchb_l2(Word32 __Ra)
{
}

	/*ID:  528, INST: dfetchb.lck.l2 Ra,Rn*/
Word32 do__dfetchb_lck_l2(Word32 __Ra)
{
}

	/*ID:  529, INST: divp.2 Da,Dn*/
Word40 do__divp_2(Word40 __Dn, Word40 __Da)
{
	Word40 tmp;
	Word40 op1,op2;
	Word8 ext1,ext2;
	long long varLL;
	op1 = do__ash_lft_x(1,__Dn);
	varLL = do__x_to_ll(__Da);
	varLL = varLL & 0xFFFFFF0000ull;
	op2 = __ll_to_x(varLL);
	ext1 = __x_guard(__Dn);
	ext2 = __x_guard(__Da);

	if((UWord8)(ext1^ext2)>>7 == 1)
		tmp = do__add_x(do__add_x(op1,__l_to_x(AS2)),op2);
	else
		tmp = do__sub_x(op2,do__add_x(op1,__l_to_x(AS2)));

	return tmp;
}

	/*ID:  530, INST: divp.1 Da,Dn*/
Word40 do__divp_1(Word40 __Dn, Word40 __Da)
{
}

	/*ID:  531, INST: divp.3 Da,Dn*/
Word40 do__divp_3(Word40 __Dn, Word40 __Da)
{
}

	/*ID:  532, INST: divp.0 Da,Dn*/
Word40 do__divp_0(Word40 __Dn, Word40 __Da)
{
}

	/*ID:  533, INST: doalign.l Da,Db,Dn*/
Word32 do__doalign_l(Word32 __Da, Word32 __Db)
{
}

	/*ID:  534, INST: doen.2 #u16_t8*/

	/*ID:  535, INST: doen.2 Ra*/

	/*ID:  536, INST: doen.1 #u16_t8*/

	/*ID:  537, INST: doen.1 Ra*/

	/*ID:  538, INST: doen.3 #u16_t8*/

	/*ID:  539, INST: doen.3 Ra*/

	/*ID:  540, INST: doen.0 #u16_t8*/

	/*ID:  541, INST: doen.0 Ra*/

	/*ID:  542, INST: dqsync*/
void do__dqsync()
{
}

	/*ID:  543, INST: dquery.l1 (Rn),Rab*/
void do__dquery_l1(Word8* __p, Word32* __Ra, Word32* __Rb)
{
}

	/*ID:  544, INST: dquery.l12 (Rn),Rab*/
void do__dquery_l12(Word8* __p, Word32* __Ra, Word32* __Rb)
{
}

	/*ID:  545, INST: dunlockb.l2 Ra,Rn*/
Word32 do__dunlockb_l2(Word32 __Ra)
{
}

	/*ID:  546, INST: eor.l #u32,Da,Dn*/

	/*ID:  547, INST: eor.x Da,Db,Dn*/
Word40 do__eor_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  548, INST: eor.2x Dab,Dcd,Dmn*/
void do__eor_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  549, INST: eora Ra,Rb,Rn*/

	/*ID:  550, INST: eora Ra,Rb,Rn*/

	/*ID:  551, INST: extc.lft #u5,Da*/

	/*ID:  552, INST: extc.rgt #u5,Da*/

	/*ID:  553, INST: extins.ll Dab,Dcd,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__extins_ll(Int64 __Dmn, Int64 __Dab, Int64 __Dcd)
{
}
#endif

	/*ID:  554, INST: extins.x Dab,Dcd,Dn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word40 do__extins_x(Word40 __Dn, Int64 __Dab, Int64 __Dcd)
{
}
#endif

	/*ID:  555, INST: extract.2l Da,Dcd,Dmn*/
void do__extract_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  556, INST: extract.x #U6,#u6,Da,Dn*/
Word40 do__extract_x_imm(Word32 __imm1, Word32 __imm2, Word40 __Da)
{
	long long __Da_LL,temp2,temp3;
	unsigned long long  temp1;

	temp1 = ~0;
	temp2 = ~0;
	temp3 = ~0;
	__Da_LL = __x_to_ll(__Da);

	/*consider __imm1 : widtgh of bit field : __imm2 :offset*/
	/* mask and get required portion of bitfield*/

	if(__imm1 == 0)
		__imm1 = 1;

	if(__imm1+__imm2 > 40)
		return __x_set(0x0,0x0);

	temp2 = temp2 << __imm2;
	temp1 = temp1>>((sizeof(long long)*8) - (__imm1+__imm2));
	temp3 = temp1 & temp2;


	__Da_LL = __Da_LL & temp3;
	__Da_LL >>= __imm2;

	if(__Da_LL & (1 <<(__imm1-1)))
		__Da_LL = __Da_LL | ( (~(0x0ULL))<<__imm1);/*sign extend*/

	__Da = __ll_to_x(__Da_LL);

	return __Da;
}

	/*ID:  557, INST: extract.x Da,Db,Dn*/
Word40 do__extract_x(Word40 __Da, Word40 __Db)
{
}

	/*ID:  558, INST: extract.ll.x Da,Dcd,Dn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word40 do__extract_ll_x(Word32 __Da, Int64 __Dcd)
{
}
#endif

	/*ID:  559, INST: extract.ll.u.x Da,Dcd,Dn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
UWord40 do__extract_ll_u_x(UWord32 __Da, UInt64 __Dcd)
{
}
#endif

	/*ID:  560, INST: extract.u.x #U6,#u6,Da,Dn*/
UWord40 do__extract_u_x_imm(Word32 __imm1, UWord32 __imm2, UWord40 __Da)
{
}

	/*ID:  561, INST: extract.u.x Da,Db,Dn*/
UWord40 do__extract_u_x(UWord40 __Da, UWord40 __Db)
{
}

	/*ID:  562, INST: extracta #U5,#u5,Ra,Rn*/
Word32 do__extracta_imm(Word32 __imm1, Word32 __imm2, Word32 __Ra)
{
}

	/*ID:  563, INST: extracta.u #U5,#u5,Ra,Rn*/
UWord32 do__extracta_u_imm(Word32 __imm1, UWord32 __imm2, UWord32 __Ra)
{
}

	/*ID:  564, INST: fft.r2.s1.4t Da,Db,Dzw*/

	/*ID:  565, INST: fft.r3.ip2.8t Dab,Dc,Dd,Def,Dg,Dh,Duv,Dwx*/

	/*ID:  566, INST: fft.r3.isp1.4t Dab,Dcd,Duv*/

	/*ID:  567, INST: fft.r3.p2.8t Dab,Dc,Dd,Def,Dg,Dh,Duv,Dwx*/

	/*ID:  568, INST: fft.r3.sp1.4t Dab,Dcd,Duv*/

	/*ID:  569, INST: fft.r4.is1.8t Da,Dcd,Db,Dgh,Dxy,Dzw*/

	/*ID:  570, INST: fft.r4.isn.8t Da,Dcd,Def,Db,Dgh,Di,Dxy,Dmn*/

	/*ID:  571, INST: fft.r4.isn.8t Da,Dcd,Def,Db,Dgh,Di,Dxy,Dzw*/

	/*ID:  572, INST: fft.r4.s1.8t Da,Dcd,Db,Dgh,Dxy,Dzw*/

	/*ID:  573, INST: fft.r4.sn.8t Da,Dcd,Def,Db,Dgh,Di,Dxy,Dmn*/

	/*ID:  574, INST: fft.r4.sn.8t Da,Dcd,Def,Db,Dgh,Di,Dxy,Dzw*/

	/*ID:  575, INST: fft.r5.2t Dab,Dcd,Dn*/

	/*ID:  576, INST: fft.r5.4t Dab,Dcd,De,Dgh,Dij,Dk,Dm,Dn*/

	/*ID:  577, INST: fft.r5.i.2t Dab,Dcd,Dn*/

	/*ID:  578, INST: fft.r5.i.4t Dab,Dcd,De,Dgh,Dij,Dk,Dm,Dn*/

	/*ID:  579, INST: fft.r5.ip11.4t Dab,Dcd,Dmn*/

	/*ID:  580, INST: fft.r5.ip12.4t Dab,Dcd,Dmn*/

	/*ID:  581, INST: fft.r5.ip2.8t Dab,Dcd,De,Dgh,Dij,Dk,Duv,Dwx*/

	/*ID:  582, INST: fft.r5.p11.4t Dab,Dcd,Dmn*/

	/*ID:  583, INST: fft.r5.p12.4t Dab,Dcd,Dmn*/

	/*ID:  584, INST: fft.r5.p2.8t Dab,Dcd,De,Dgh,Dij,Dk,Duv,Dwx*/

	/*ID:  585, INST: fnd.max.4t #u2_t3,Da,Db,Dmn*/
/*
  Function Name: do__fnd_max_4t
  Purpose: Find maximum (not absolute) value and index between four signed 20 bit words and previous result.
  Complexity Weight:
  Inputs: __imm1 -32bit word and __Da and __Db 40 bit words
  Outputs:
  Return Value:
*/
void do__fnd_max_4t(Word32 __imm1, Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	Word32 local_max_index = 3,i=0 ;
	Word32 Da_WH,Da_WL,Db_WH,Db_WL,Dn_H,Dn_L,Dn_2;
	Word32 Dm_2;
	Word32 sign[4];/*0: positive 1: negative*/
	Word32 max[4];
	Word32 abs_max[4];

	Da_WH = __extract_wh(__Da);
	Da_WL = __extract_wl(__Da);
	Db_WH = __extract_wh(__Db);
	Db_WL = __extract_wl(__Db);

	max[0] = Da_WH;
	max[1] = Da_WL;
	max[2] = Db_WH;
	max[3] = Db_WL;


	sign[0] = (Da_WH & 0x80000000) ? 1: 0;
	sign[1] = (Da_WL & 0x80000000) ? 1: 0;
	sign[2] = (Db_WH & 0x80000000) ? 1: 0;
	sign[3] = (Db_WL & 0x80000000) ? 1: 0;

	Dm_2 = __x_to_l(*__Dm);

	/*comparision*/
	for(i=0;i<4;i++)
	{
		if(max[i] > max[local_max_index])
		{

			local_max_index = i;
		}
		else if(max[i] == max[local_max_index])
		{
			if( sign[i] ^ sign[local_max_index])
			{
				local_max_index = sign[i] ? local_max_index : i;
			}
			else
				local_max_index = (local_max_index >= i) ? i : local_max_index;
		}
	}

	/* storing results */


	Dn_2 = __x_to_l(*__Dn);
	Dn_H = __l_get_msb(Dn_2);
	Dn_L = __l_get_lsb(Dn_2);
	if(max[local_max_index] > Dm_2)
	{
		*__Dm = __l_to_x(max[local_max_index]);
		Dn_L = (Dn_H&0xfffffffc) + local_max_index;
	}
	else
		*__Dm = __l_to_x(Dm_2);
	Dn_H = Dn_H + ((__imm1+1)<<2);
	Dn_2 = ((Dn_H<<16)&0xFFFF0000)|(Dn_L & 0xFFFF);

	 *__Dn = __x_set(0,Dn_2);
}

	/*ID:  586, INST: fnd.max.4w #u2_t3,Da,Db,Dmn*/
/*
  Function Name: do__fnd_max_4w
  Purpose: Find maximum absolute value and index between four signed 16 bit words and previous result.
  Complexity Weight:
  Inputs: __imm1 -32bit word and __Da and __Db 32 bit words
  Outputs:
  Return Value:
*/
void do__fnd_max_4w(Word32 __imm1, Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
	Word32 local_max_index = 3,i=0 ;
	Word32 Da_H,Da_L,Db_H,Db_L,Dn_H,Dn_L,Dn_2;
	Word32 Dm_2;
	Word32 sign[4];/*0: positive 1: negative*/
	Word32 max[4];
	Word32 abs_max[4];

	Da_H = __l_get_msb(__Da);
	Da_L = __l_get_lsb(__Da);
	Db_H = __l_get_msb(__Db);
	Db_L = __l_get_lsb(__Db);

	max[0] = Da_H;
	max[1] = Da_L;
	max[2] = Db_H;
	max[3] = Db_L;

	sign[0] = (Da_H & 0x80000000) ? 1: 0;
	sign[1] = (Da_L & 0x80000000) ? 1: 0;
	sign[2] = (Db_H & 0x80000000) ? 1: 0;
	sign[3] = (Db_L & 0x80000000) ? 1: 0;

	Dm_2 = *__Dm;

	/*comparision*/
	for(i=0;i<4;i++)
	{
		if(max[i] > max[local_max_index])
		{

			local_max_index = i;
		}
		else if(max[i] == max[local_max_index])
		{
			if( sign[i] ^ sign[local_max_index])
			{
				local_max_index = sign[i] ? local_max_index : i;
			}
			else
				local_max_index = (local_max_index >= i) ? i : local_max_index;
		}
	}

	/* storing results */


	Dn_2 = *__Dn;
	Dn_H = __l_get_msb(Dn_2);
	Dn_L = __l_get_lsb(Dn_2);
	if(max[local_max_index] > Dm_2)
	{
		*__Dm = max[local_max_index];
		Dn_L = (Dn_H&0xfffffffc) + local_max_index;
	}
	else
		*__Dm = Dm_2;
	Dn_H = Dn_H + ((__imm1+1)<<2);
	*__Dn = ((Dn_H<<16)&0xFFFF0000)|(Dn_L & 0xFFFF);
}

	/*ID:  587, INST: fnd.maxm.4t #u2_t3,Da,Db,Dmn*/
/*
  Function Name: do__fnd_maxm_4t
  Purpose: Find maximum absolute value and index between four signed 20 bit words and previous result.
  Complexity Weight:
  Inputs: __imm1 -32bit word and __Da and __Db 40 bit words
  Outputs:
  Return Value:
*/
void do__fnd_maxm_4t(Word32 __imm1, Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	Word32 local_max_index = 3,i=0 ;
	Word32 Da_WH,Da_WL,Db_WH,Db_WL,Dn_H,Dn_L,Dn_2;
	UWord32 abs_Da_WH,abs_Da_WL,abs_Db_WH,abs_Db_WL,abs_Dm_2;
	Word32 Dm_2;
	Word32 sign[4];/*0: positive 1: negative*/
	Word32 max[4];
	Word32 abs_max[4];

	Da_WH = __extract_wh(__Da);
	Da_WL = __extract_wl(__Da);
	Db_WH = __extract_wh(__Db);
	Db_WL = __extract_wl(__Db);

	max[0] = Da_WH;
	max[1] = Da_WL;
	max[2] = Db_WH;
	max[3] = Db_WL;


	sign[0] = (Da_WH & 0x80000000) ? 1: 0;
	sign[1] = (Da_WL & 0x80000000) ? 1: 0;
	sign[2] = (Db_WH & 0x80000000) ? 1: 0;
	sign[3] = (Db_WL & 0x80000000) ? 1: 0;

	abs_Da_WH = do__abs_l(Da_WH);
	abs_Da_WL = do__abs_l(Da_WL);
	abs_Db_WH = do__abs_l(Db_WH);
	abs_Db_WL = do__abs_l(Db_WL);

	Dm_2 = __x_to_l(*__Dm);
	abs_Dm_2 = do__abs_l(Dm_2);


	abs_max[0] = abs_Da_WH;
	abs_max[1] = abs_Da_WL;
	abs_max[2] = abs_Db_WH;
	abs_max[3] = abs_Db_WL;

	/*comparision*/
	for(i=0;i<4;i++)
	{
		if(abs_max[i] > abs_max[local_max_index])
		{

			local_max_index = i;
		}
		else if(abs_max[i] == abs_max[local_max_index])
		{
			if( sign[i] ^ sign[local_max_index])
			{
				local_max_index = sign[i] ? local_max_index : i;
			}
			else
				local_max_index = (local_max_index >= i) ? i : local_max_index;
		}
	}

	/* storing results */


	Dn_2 = __x_to_l(*__Dn);
	Dn_H = __l_get_msb(Dn_2);
	Dn_L = __l_get_lsb(Dn_2);
	if(abs_max[local_max_index] > abs_Dm_2)
	{
		*__Dm = __l_to_x(max[local_max_index]);
		Dn_L = (Dn_H&0xfffffffc) + local_max_index;
	}
	else
		*__Dm = __l_to_x(Dm_2);
	Dn_H = Dn_H + ((__imm1+1)<<2);
	Dn_2 = ((Dn_H<<16)&0xFFFF0000)|(Dn_L & 0xFFFF);

	 *__Dn = __x_set(0,Dn_2);
}

	/*ID:  588, INST: fnd.maxm.4w #u2_t3,Da,Db,Dmn*/
/*
  Function Name: do__fnd_maxm_4w
  Purpose: Find maximum absolute value and index between four signed 16 bit words and previous result.
  Complexity Weight:
  Inputs: __imm1 -32bit word and __Da and __Db 32 bit words
  Outputs:
  Return Value:
*/
void do__fnd_maxm_4w(Word32 __imm1, Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
	Word32 local_max_index = 3,i=0 ;
	Word32 Da_H,Da_L,Db_H,Db_L,Dn_H,Dn_L,Dn_2;
	UWord32 abs_Da_H,abs_Da_L,abs_Db_H,abs_Db_L,abs_Dm_2;
	Word32 Dm_2;
	Word32 sign[4];/*0: positive 1: negative*/
	Word32 max[4];
	Word32 abs_max[4];

	Da_H = __l_get_msb(__Da);
	Da_L = __l_get_lsb(__Da);
	Db_H = __l_get_msb(__Db);
	Db_L = __l_get_lsb(__Db);

	max[0] = Da_H;
	max[1] = Da_L;
	max[2] = Db_H;
	max[3] = Db_L;

	sign[0] = (Da_H & 0x80000000) ? 1: 0;
	sign[1] = (Da_L & 0x80000000) ? 1: 0;
	sign[2] = (Db_H & 0x80000000) ? 1: 0;
	sign[3] = (Db_L & 0x80000000) ? 1: 0;

	abs_Da_H = do__abs_l(Da_H);
	abs_Da_L = do__abs_l(Da_L);
	abs_Db_H = do__abs_l(Db_H);
	abs_Db_L = do__abs_l(Db_L);

	Dm_2 = *__Dm;
	abs_Dm_2 = do__abs_l(Dm_2);

	abs_max[0] = abs_Da_H;
	abs_max[1] = abs_Da_L;
	abs_max[2] = abs_Db_H;
	abs_max[3] = abs_Db_L;

	/*comparision*/
	for(i=0;i<4;i++)
	{
		if(abs_max[i] > abs_max[local_max_index])
		{

			local_max_index = i;
		}
		else if(abs_max[i] == abs_max[local_max_index])
		{
			if( sign[i] ^ sign[local_max_index])
			{
				local_max_index = sign[i] ? local_max_index : i;
			}
			else
				local_max_index = (local_max_index >= i) ? i : local_max_index;
		}
	}

	/* storing results */


	Dn_2 = *__Dn;
	Dn_H = __l_get_msb(Dn_2);
	Dn_L = __l_get_lsb(Dn_2);
	if(abs_max[local_max_index] > abs_Dm_2)
	{
		*__Dm = max[local_max_index];
		Dn_L = (Dn_H&0xfffffffc) + local_max_index;
	}
	else
		*__Dm = Dm_2;
	Dn_H = Dn_H + ((__imm1+1)<<2);
	*__Dn = ((Dn_H<<16)&0xFFFF0000)|(Dn_L & 0xFFFF);

}

	/*ID:  589, INST: fnd.min.4t #u2_t3,Da,Db,Dmn*/
void do__fnd_min_4t(Word32 __imm1, Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID:  590, INST: fnd.min.4w #u2_t3,Da,Db,Dmn*/
void do__fnd_min_4w(Word32 __imm1, Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID:  591, INST: illegal*/
void do__illegal()
{
}

	/*ID:  592, INST: insert.x #U6,#u6,Da,Dn*/
Word40 do__insert_x_imm(Word40 __Dn, Word32 __imm1, Word32 __imm2, Word40 __Da)
{
}

	/*ID:  593, INST: insert.x Da,Db,Dn*/
Word40 do__insert_x(Word40 __Dn, Word40 __Da, Word40 __Db)
{
}

	/*ID:  594, INST: inserta #U5,#u5_t3,Ra,Rn*/
Word32 do__inserta(Word32 __Rn, Word32 __imm1, Word32 __imm2, Word32 __Ra)
{
}

	/*ID:  595, INST: invsqrt.2x Da,Dmn*/
void do__invsqrt_2x(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word8 inp;
	Word16 INVSQRT_LUT_A,INVSQRT_LUT_B;
	Word32 __Dm_2,__Dn_2;

	if(__Da &0x40000000)
		inp = (__Da &0x3e000000)>>25;
	else
	{
		inp =  ((0x20000000) | (__Da &0x1f000000)) >>24;
		inp = inp&0x3f;
	}

	INVSQRT_LUT_A = get_invsqrt_LUT_A(inp);
	INVSQRT_LUT_B = get_invsqrt_LUT_B(inp);

	__Dm_2 = INVSQRT_LUT_A << 16;
	__Dn_2 = INVSQRT_LUT_B << 16;

	*__Dm = __x_set(0x0,__Dm_2);
	*__Dn = __x_set(0x0,__Dn_2);
}

	/*ID:  596, INST: jmp AbsAdd31_t2*/

	/*ID:  597, INST: jmp Rn*/

	/*ID:  598, INST: jmp.nobtb AbsAdd31_t2*/

	/*ID:  599, INST: jmpnf AbsAdd31_t2,LSU0_NF,LSU1_NF*/

	/*ID:  600, INST: jmprf AbsAdd31_t2*/

	/*ID:  601, INST: jsr AbsAdd31_t2*/

	/*ID:  602, INST: jsr Rn*/

	/*ID:  603, INST: jsr.nobtb AbsAdd31_t2*/

	/*ID:  604, INST: ld.b (Rn)+,Da*/
void do__ld_b(const Word8* __p, Word32* __Da)
{
	*__Da = (*(__p));
}

	/*ID:  605, INST: ld.b (Rn)+,Ra*/

	/*ID:  606, INST: ld.b (Rn)+Rk,Da*/

	/*ID:  607, INST: ld.b (Rn)+Rk,Ra*/

	/*ID:  608, INST: ld.b (Rn),Da*/

	/*ID:  609, INST: ld.b (Rn),Ra*/

	/*ID:  610, INST: ld.b (Rn)-,Da*/

	/*ID:  611, INST: ld.b (Rn)-,Ra*/

	/*ID:  612, INST: ld.b (Rn+Rk),Da*/

	/*ID:  613, INST: ld.b (Rn+Rk),Ra*/

	/*ID:  614, INST: ld.b (Rn+s16_t7),Da*/

	/*ID:  615, INST: ld.b (Rn+s16_t7),Ra*/

	/*ID:  616, INST: ld.b (sp+s16_t7),Da*/

	/*ID:  617, INST: ld.b (sp+s16_t7),Ra*/

	/*ID:  618, INST: ld.b (sp-u9),Da*/

	/*ID:  619, INST: ld.b (sp-u9),Ra*/

	/*ID:  620, INST: ld.2b (Rn)+,Dab*/
void do__ld_2b(const Word8* __p, Word32* __Da, Word32* __Db)
{
	*__Da = (*(__p));
	*__Db = (*(__p+1));
}

	/*ID:  621, INST: ld.2b (Rn)+,Rab*/

	/*ID:  622, INST: ld.2b (Rn)+Rk,Dab*/

	/*ID:  623, INST: ld.2b (Rn)+Rk,Rab*/

	/*ID:  624, INST: ld.2b (Rn),Dab*/

	/*ID:  625, INST: ld.2b (Rn),Rab*/

	/*ID:  626, INST: ld.2b (Rn)-,Dab*/

	/*ID:  627, INST: ld.2b (Rn)-,Rab*/

	/*ID:  628, INST: ld.2b (Rn+Rk),Dab*/

	/*ID:  629, INST: ld.2b (Rn+Rk),Rab*/

	/*ID:  630, INST: ld.2b (Rn+s16_t7),Dab*/

	/*ID:  631, INST: ld.2b (Rn+s16_t7),Rab*/

	/*ID:  632, INST: ld.4b (Rn)+,Dabcd*/
void do__ld_4b(const Word8* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd)
{
	*__Da = (*(__p));
	*__Db = (*(__p+1));
	*__Dc = (*(__p+2));
	*__Dd = (*(__p+3));
}

	/*ID:  633, INST: ld.4b (Rn)+,Rabcd*/

	/*ID:  634, INST: ld.4b (Rn)+Rk,Dabcd*/

	/*ID:  635, INST: ld.4b (Rn)+Rk,Rabcd*/

	/*ID:  636, INST: ld.4b (Rn),Dabcd*/

	/*ID:  637, INST: ld.4b (Rn),Rabcd*/

	/*ID:  638, INST: ld.4b (Rn)-,Dabcd*/

	/*ID:  639, INST: ld.4b (Rn)-,Rabcd*/

	/*ID:  640, INST: ld.4b (Rn+Rk),Dabcd*/

	/*ID:  641, INST: ld.4b (Rn+Rk),Rabcd*/

	/*ID:  642, INST: ld.4b (Rn+s16_t7),Dabcd*/

	/*ID:  643, INST: ld.4b (Rn+s16_t7),Rabcd*/

	/*ID:  644, INST: ld.8b (Rn)+,Da__Dh*/
void do__ld_8b(const Word8* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd, Word32* __De, Word32* __Df, Word32* __Dg, Word32* __Dh)
{
	*__Da = (*(__p));
	*__Db = (*(__p+1));
	*__Dc = (*(__p+2));
	*__Dd = (*(__p+3));
	*__De = (*(__p+4));
	*__Df = (*(__p+5));
	*__Dg = (*(__p+6));
	*__Dh = (*(__p+7));
}

	/*ID:  645, INST: ld.8b (Rn)+Rk,Da__Dh*/

	/*ID:  646, INST: ld.8b (Rn),Da__Dh*/

	/*ID:  647, INST: ld.8b (Rn)-,Da__Dh*/

	/*ID:  648, INST: ld.8b (Rn+Rk),Da__Dh*/

	/*ID:  649, INST: ld.8b (Rn+s16_t7),Da__Dh*/

	/*ID:  650, INST: ld.bf (Rn)+,Da*/
void do__ld_bf(const Word8* __p, Word32* __Da)
{
	*__Da = (*(__p)<<24);
}

	/*ID:  651, INST: ld.bf (Rn)+Rk,Da*/

	/*ID:  652, INST: ld.bf (Rn),Da*/

	/*ID:  653, INST: ld.bf (Rn)-,Da*/

	/*ID:  654, INST: ld.bf (Rn+Rk),Da*/

	/*ID:  655, INST: ld.bf (Rn+s16_t7),Da*/

	/*ID:  656, INST: ld.bf (sp+s16_t7),Da*/

	/*ID:  657, INST: ld.bf (sp-u9),Da*/

	/*ID:  658, INST: ld.2bf (Rn)+,Dab*/
void do__ld_2bf(const Word8* __p, Word32* __Da, Word32* __Db)
{
	do__ld_bf(__p, __Da);
	do__ld_bf(__p+1, __Db);
}

	/*ID:  659, INST: ld.2bf (Rn)+Rk,Dab*/

	/*ID:  660, INST: ld.2bf (Rn),Dab*/

	/*ID:  661, INST: ld.2bf (Rn)-,Dab*/

	/*ID:  662, INST: ld.2bf (Rn+Rk),Dab*/

	/*ID:  663, INST: ld.2bf (Rn+s16_t7),Dab*/

	/*ID:  664, INST: ld.4bf (Rn)+,Dabcd*/
void do__ld_4bf(const Word8* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd)
{
	do__ld_2bf(__p, __Da, __Db);
	do__ld_2bf(__p+2, __Dc, __Dd);
}

	/*ID:  665, INST: ld.4bf (Rn)+Rk,Dabcd*/

	/*ID:  666, INST: ld.4bf (Rn),Dabcd*/

	/*ID:  667, INST: ld.4bf (Rn)-,Dabcd*/

	/*ID:  668, INST: ld.4bf (Rn+Rk),Dabcd*/

	/*ID:  669, INST: ld.4bf (Rn+s16_t7),Dabcd*/

	/*ID:  670, INST: ld.8bf (Rn)+,Da__Dh*/
void do__ld_8bf(const Word8* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd, Word32* __De, Word32* __Df, Word32* __Dg, Word32* __Dh)
{
	do__ld_4bf(__p, __Da, __Db, __Dc, __Dd);
	do__ld_4bf(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  671, INST: ld.8bf (Rn)+Rk,Da__Dh*/

	/*ID:  672, INST: ld.8bf (Rn),Da__Dh*/

	/*ID:  673, INST: ld.8bf (Rn)-,Da__Dh*/

	/*ID:  674, INST: ld.8bf (Rn+Rk),Da__Dh*/

	/*ID:  675, INST: ld.8bf (Rn+s16_t7),Da__Dh*/

	/*ID:  676, INST: ld.f (Rn)+,Da*/
void do__ld_f(const Word16* __p, Word32* __Da)
{
	*__Da = (*__p) << 16;
	*__Da &= 0xffff0000;
}

	/*ID:  677, INST: ld.f (Rn)+Rk,Da*/

	/*ID:  678, INST: ld.f (Rn),Da*/

	/*ID:  679, INST: ld.f (Rn)-,Da*/

	/*ID:  680, INST: ld.f (Rn+Rk),Da*/

	/*ID:  681, INST: ld.f (Rn+s16_t7),Da*/

	/*ID:  682, INST: ld.f (sp+s16_t7),Da*/

	/*ID:  683, INST: ld.f (sp-u9_1),Da*/

	/*ID:  684, INST: ld.2f (Rn)+,Dab*/
void do__ld_2f(const Word16* __p, Word32* __Da, Word32* __Db)
{
	do__ld_f(__p, __Da);
	do__ld_f(__p+1, __Db);
}

	/*ID:  685, INST: ld.2f (Rn)+Rk,Dab*/

	/*ID:  686, INST: ld.2f (Rn),Dab*/

	/*ID:  687, INST: ld.2f (Rn)-,Dab*/

	/*ID:  688, INST: ld.2f (Rn+Rk),Dab*/

	/*ID:  689, INST: ld.2f (Rn+s16_t7),Dab*/

	/*ID:  690, INST: ld.4f (Rn)+,Dabcd*/
void do__ld_4f(const Word16* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd)
{
	do__ld_2f(__p, __Da, __Db);
	do__ld_2f(__p+2, __Dc, __Dd);
}

	/*ID:  691, INST: ld.4f (Rn)+Rk,Dabcd*/

	/*ID:  692, INST: ld.4f (Rn),Dabcd*/

	/*ID:  693, INST: ld.4f (Rn)-,Dabcd*/

	/*ID:  694, INST: ld.4f (Rn+Rk),Dabcd*/

	/*ID:  695, INST: ld.4f (Rn+s16_t7),Dabcd*/

	/*ID:  696, INST: ld.8f (Rn)+,Da__Dh*/
void do__ld_8f(const Word16* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd, Word32* __De, Word32* __Df, Word32* __Dg, Word32* __Dh)
{
	do__ld_4f(__p, __Da, __Db, __Dc, __Dd);
	do__ld_4f(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  697, INST: ld.8f (Rn)+Rk,Da__Dh*/

	/*ID:  698, INST: ld.8f (Rn),Da__Dh*/

	/*ID:  699, INST: ld.8f (Rn)-,Da__Dh*/

	/*ID:  700, INST: ld.8f (Rn+Rk),Da__Dh*/

	/*ID:  701, INST: ld.8f (Rn+s16_t7),Da__Dh*/

	/*ID:  702, INST: ld.l (Rn)+,Da*/
void do__ld_l(const Word32* __p, Word32* __Da)
{
	*__Da = (*(__p));
}

	/*ID:  703, INST: ld.l (Rn)+,Ra*/

	/*ID:  704, INST: ld.l (Rn)+Rk,Da*/

	/*ID:  705, INST: ld.l (Rn)+Rk,Ra*/

	/*ID:  706, INST: ld.l (Rn),BTRN*/

	/*ID:  707, INST: ld.l (Rn),Da*/

	/*ID:  708, INST: ld.l (Rn),Ra*/

	/*ID:  709, INST: ld.l (Rn)-,Da*/

	/*ID:  710, INST: ld.l (Rn)-,Ra*/

	/*ID:  711, INST: ld.l (Rn+Rk),Da*/

	/*ID:  712, INST: ld.l (Rn+Rk),Ra*/

	/*ID:  713, INST: ld.l (Rn+s16_t7),Da*/

	/*ID:  714, INST: ld.l (Rn+s16_t7),Ra*/

	/*ID:  715, INST: ld.l (sp+s16_t7),Da*/

	/*ID:  716, INST: ld.l (sp+s16_t7),Ra*/

	/*ID:  717, INST: ld.l (sp-u9_2),Da*/

	/*ID:  718, INST: ld.l (sp-u9_2),Ra*/

	/*ID:  719, INST: ld.2l (Rn)+,Dab*/
void do__ld_2l(const Word32* __p, Word32* __Da, Word32* __Db)
{
	*__Da = (*(__p));
	*__Db = (*(__p+1));
}

	/*ID:  720, INST: ld.2l (Rn)+,Dac*/

	/*ID:  721, INST: ld.2l (Rn)+,Rab*/

	/*ID:  722, INST: ld.2l (Rn)+Rk,Dab*/

	/*ID:  723, INST: ld.2l (Rn)+Rk,Dac*/

	/*ID:  724, INST: ld.2l (Rn)+Rk,Rab*/

	/*ID:  725, INST: ld.2l (Rn),Dab*/

	/*ID:  726, INST: ld.2l (Rn),Dac*/

	/*ID:  727, INST: ld.2l (Rn),Rab*/

	/*ID:  728, INST: ld.2l (Rn)-,Dab*/

	/*ID:  729, INST: ld.2l (Rn)-,Dac*/

	/*ID:  730, INST: ld.2l (Rn)-,Rab*/

	/*ID:  731, INST: ld.2l (Rn+Rk),Dab*/

	/*ID:  732, INST: ld.2l (Rn+Rk),Dac*/

	/*ID:  733, INST: ld.2l (Rn+Rk),Rab*/

	/*ID:  734, INST: ld.2l (Rn+s16_t7),Dab*/

	/*ID:  735, INST: ld.2l (Rn+s16_t7),Dac*/

	/*ID:  736, INST: ld.2l (Rn+s16_t7),Rab*/

	/*ID:  737, INST: ld.2l (sp+s16_t7),Dab*/

	/*ID:  738, INST: ld.2l (sp+s16_t7),Rab*/

	/*ID:  739, INST: ld.4l (Rn)+,Dabcd*/
void do__ld_4l(const Word32* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd)
{
	*__Da = (*(__p));
	*__Db = (*(__p+1));
	*__Dc = (*(__p+2));
	*__Dd = (*(__p+3));
}

	/*ID:  740, INST: ld.4l (Rn)+,Dacbd*/

	/*ID:  741, INST: ld.4l (Rn)+,Daceg*/

	/*ID:  742, INST: ld.4l (Rn)+,Rabcd*/

	/*ID:  743, INST: ld.4l (Rn)+Rk,Dabcd*/

	/*ID:  744, INST: ld.4l (Rn)+Rk,Dacbd*/

	/*ID:  745, INST: ld.4l (Rn)+Rk,Daceg*/

	/*ID:  746, INST: ld.4l (Rn)+Rk,Rabcd*/

	/*ID:  747, INST: ld.4l (Rn),Dabcd*/

	/*ID:  748, INST: ld.4l (Rn),Dacbd*/

	/*ID:  749, INST: ld.4l (Rn),Daceg*/

	/*ID:  750, INST: ld.4l (Rn),Rabcd*/

	/*ID:  751, INST: ld.4l (Rn)-,Dabcd*/

	/*ID:  752, INST: ld.4l (Rn)-,Dacbd*/

	/*ID:  753, INST: ld.4l (Rn)-,Daceg*/

	/*ID:  754, INST: ld.4l (Rn)-,Rabcd*/

	/*ID:  755, INST: ld.4l (Rn+Rk),Dabcd*/

	/*ID:  756, INST: ld.4l (Rn+Rk),Dacbd*/

	/*ID:  757, INST: ld.4l (Rn+Rk),Daceg*/

	/*ID:  758, INST: ld.4l (Rn+Rk),Rabcd*/

	/*ID:  759, INST: ld.4l (Rn+s16_t7),Dabcd*/

	/*ID:  760, INST: ld.4l (Rn+s16_t7),Dacbd*/

	/*ID:  761, INST: ld.4l (Rn+s16_t7),Daceg*/

	/*ID:  762, INST: ld.4l (Rn+s16_t7),Rabcd*/

	/*ID:  763, INST: ld.8l (Rn)+,Da__Dh*/
void do__ld_8l(const Word32* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd, Word32* __De, Word32* __Df, Word32* __Dg, Word32* __Dh)
{
	do__ld_4l(__p, __Da, __Db, __Dc, __Dd);
	do__ld_4l(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  764, INST: ld.8l (Rn)+,Dacegbdfh*/

	/*ID:  765, INST: ld.8l (Rn)+,Daceghjln*/

	/*ID:  766, INST: ld.8l (Rn)+Rk,Da__Dh*/

	/*ID:  767, INST: ld.8l (Rn)+Rk,Dacegbdfh*/

	/*ID:  768, INST: ld.8l (Rn)+Rk,Daceghjln*/

	/*ID:  769, INST: ld.8l (Rn),Da__Dh*/

	/*ID:  770, INST: ld.8l (Rn),Dacegbdfh*/

	/*ID:  771, INST: ld.8l (Rn),Daceghjln*/

	/*ID:  772, INST: ld.8l (Rn)-,Da__Dh*/

	/*ID:  773, INST: ld.8l (Rn)-,Dacegbdfh*/

	/*ID:  774, INST: ld.8l (Rn)-,Daceghjln*/

	/*ID:  775, INST: ld.8l (Rn+Rk),Da__Dh*/

	/*ID:  776, INST: ld.8l (Rn+Rk),Dacegbdfh*/

	/*ID:  777, INST: ld.8l (Rn+Rk),Daceghjln*/

	/*ID:  778, INST: ld.8l (Rn+s16_t7),Da__Dh*/

	/*ID:  779, INST: ld.8l (Rn+s16_t7),Dacegbdfh*/

	/*ID:  780, INST: ld.8l (Rn+s16_t7),Daceghjln*/

	/*ID:  781, INST: ld.16l (Rn)+,Da__Dp*/
void do__ld_16l(const Word32* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd, Word32* __De, Word32* __Df, Word32* __Dg, Word32* __Dh, Word32* __Di, Word32* __Dj, Word32* __Dk, Word32* __Dl, Word32* __Dm, Word32* __Dn, Word32* __Do, Word32* __Dp)
{
	do__ld_8l(__p, __Da, __Db, __Dc, __Dd, __De, __Df, __Dg, __Dh);
	do__ld_8l(__p+8, __Di, __Dj, __Dk, __Dl, __Dm, __Dn, __Do, __Dp);
}

	/*ID:  782, INST: ld.16l (Rn)+Rk,Da__Dp*/

	/*ID:  783, INST: ld.16l (Rn),Da__Dp*/

	/*ID:  784, INST: ld.16l (Rn)-,Da__Dp*/

	/*ID:  785, INST: ld.16l (Rn+Rk),Da__Dp*/

	/*ID:  786, INST: ld.16l (Rn+s16_t7),Da__Dp*/

	/*ID:  787, INST: ld.w (Rn)+,Da*/
void do__ld_w(const Word16* __p, Word32* __Da)
{
	*__Da = (*(__p));
}

	/*ID:  788, INST: ld.w (Rn)+,Ra*/

	/*ID:  789, INST: ld.w (Rn)+Rk,Da*/

	/*ID:  790, INST: ld.w (Rn)+Rk,Ra*/

	/*ID:  791, INST: ld.w (Rn),Da*/

	/*ID:  792, INST: ld.w (Rn),Ra*/

	/*ID:  793, INST: ld.w (Rn)-,Da*/

	/*ID:  794, INST: ld.w (Rn)-,Ra*/

	/*ID:  795, INST: ld.w (Rn+Rk),Da*/

	/*ID:  796, INST: ld.w (Rn+Rk),Ra*/

	/*ID:  797, INST: ld.w (Rn+s16_t7),Da*/

	/*ID:  798, INST: ld.w (Rn+s16_t7),Ra*/

	/*ID:  799, INST: ld.w (sp+s16_t7),Da*/

	/*ID:  800, INST: ld.w (sp+s16_t7),Ra*/

	/*ID:  801, INST: ld.w (sp-u9_1),Da*/

	/*ID:  802, INST: ld.w (sp-u9_1),Ra*/

	/*ID:  803, INST: ld.2w (Rn)+,Dab*/
void do__ld_2w(const Word16* __p, Word32* __Da, Word32* __Db)
{
	*__Da = (*(__p));
	*__Db = (*(__p+1));
}

	/*ID:  804, INST: ld.2w (Rn)+,Rab*/

	/*ID:  805, INST: ld.2w (Rn)+Rk,Dab*/

	/*ID:  806, INST: ld.2w (Rn)+Rk,Rab*/

	/*ID:  807, INST: ld.2w (Rn),Dab*/

	/*ID:  808, INST: ld.2w (Rn),Rab*/

	/*ID:  809, INST: ld.2w (Rn)-,Dab*/

	/*ID:  810, INST: ld.2w (Rn)-,Rab*/

	/*ID:  811, INST: ld.2w (Rn+Rk),Dab*/

	/*ID:  812, INST: ld.2w (Rn+Rk),Rab*/

	/*ID:  813, INST: ld.2w (Rn+s16_t7),Dab*/

	/*ID:  814, INST: ld.2w (Rn+s16_t7),Rab*/

	/*ID:  815, INST: ld.4w (Rn)+,Dabcd*/
void do__ld_4w(const Word16* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd)
{
	do__ld_2w(__p, __Da, __Db);
	do__ld_2w(__p+2, __Dc, __Dd);
}

	/*ID:  816, INST: ld.4w (Rn)+,Rabcd*/

	/*ID:  817, INST: ld.4w (Rn)+Rk,Dabcd*/

	/*ID:  818, INST: ld.4w (Rn)+Rk,Rabcd*/

	/*ID:  819, INST: ld.4w (Rn),Dabcd*/

	/*ID:  820, INST: ld.4w (Rn),Rabcd*/

	/*ID:  821, INST: ld.4w (Rn)-,Dabcd*/

	/*ID:  822, INST: ld.4w (Rn)-,Rabcd*/

	/*ID:  823, INST: ld.4w (Rn+Rk),Dabcd*/

	/*ID:  824, INST: ld.4w (Rn+Rk),Rabcd*/

	/*ID:  825, INST: ld.4w (Rn+s16_t7),Dabcd*/

	/*ID:  826, INST: ld.4w (Rn+s16_t7),Rabcd*/

	/*ID:  827, INST: ld.8w (Rn)+,Da__Dh*/
void do__ld_8w(const Word16* __p, Word32* __Da, Word32* __Db, Word32* __Dc, Word32* __Dd, Word32* __De, Word32* __Df, Word32* __Dg, Word32* __Dh)
{
	do__ld_4w(__p, __Da, __Db, __Dc, __Dd);
	do__ld_4w(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  828, INST: ld.8w (Rn)+Rk,Da__Dh*/

	/*ID:  829, INST: ld.8w (Rn),Da__Dh*/

	/*ID:  830, INST: ld.8w (Rn)-,Da__Dh*/

	/*ID:  831, INST: ld.8w (Rn+Rk),Da__Dh*/

	/*ID:  832, INST: ld.8w (Rn+s16_t7),Da__Dh*/

	/*ID:  833, INST: ld.x (Rn)+,Da*/
void do__ld_x(const Word40* __p, Word40* __Da)
{
	*__Da = __x_set(__x_guard(*__p), __x_to_l(*__p));
}

	/*ID:  834, INST: ld.x (Rn)+Rk,Da*/

	/*ID:  835, INST: ld.x (Rn),Da*/

	/*ID:  836, INST: ld.x (Rn)-,Da*/

	/*ID:  837, INST: ld.x (Rn+Rk),Da*/

	/*ID:  838, INST: ld.x (Rn+s16_t7),Da*/

	/*ID:  839, INST: ld.x (sp+s16_t7),Da*/

	/*ID:  840, INST: ld.x (sp-u9_3),Da*/

	/*ID:  841, INST: ld.2x (Rn)+,Dab*/
void do__ld_2x(const Word40* __p, Word40* __Da, Word40* __Db)
{
	do__ld_x(__p, __Da);
	do__ld_x(__p+1, __Db);
}

	/*ID:  842, INST: ld.2x (Rn)+Rk,Dab*/

	/*ID:  843, INST: ld.2x (Rn),Dab*/

	/*ID:  844, INST: ld.2x (Rn)-,Dab*/

	/*ID:  845, INST: ld.2x (Rn+Rk),Dab*/

	/*ID:  846, INST: ld.2x (Rn+s16_t7),Dab*/

	/*ID:  847, INST: ld.4x (Rn)+,Dabcd*/
void do__ld_4x(const Word40* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd)
{
	do__ld_2x(__p, __Da, __Db);
	do__ld_2x(__p+2, __Dc, __Dd);
}

	/*ID:  848, INST: ld.4x (Rn)+Rk,Dabcd*/

	/*ID:  849, INST: ld.4x (Rn),Dabcd*/

	/*ID:  850, INST: ld.4x (Rn)-,Dabcd*/

	/*ID:  851, INST: ld.4x (Rn+Rk),Dabcd*/

	/*ID:  852, INST: ld.4x (Rn+s16_t7),Dabcd*/

	/*ID:  853, INST: ld.8x (Rn)+,Da__Dh*/
void do__ld_8x(const Word40* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd, Word40* __De, Word40* __Df, Word40* __Dg, Word40* __Dh)
{
	do__ld_4x(__p, __Da, __Db, __Dc, __Dd);
	do__ld_4x(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  854, INST: ld.8x (Rn)+Rk,Da__Dh*/

	/*ID:  855, INST: ld.8x (Rn),Da__Dh*/

	/*ID:  856, INST: ld.8x (Rn)-,Da__Dh*/

	/*ID:  857, INST: ld.8x (Rn+Rk),Da__Dh*/

	/*ID:  858, INST: ld.8x (Rn+s16_t7),Da__Dh*/

	/*ID:  859, INST: ld.par.w (Rn)+,Da.h*/
void do__ld_par_w_h(const Word16* __p, Word32* __Da_h)
{
	*__Da_h =  (*__p << 16) | ((*__Da_h)&0xffff);
}

	/*ID:  860, INST: ld.par.w (Rn)+Rk,Da.h*/

	/*ID:  861, INST: ld.par.w (Rn),Da.h*/

	/*ID:  862, INST: ld.par.w (Rn)-,Da.h*/

	/*ID:  863, INST: ld.par.w (Rn+Rk),Da.h*/

	/*ID:  864, INST: ld.par.w (Rn+s16_t7),Da.h*/

	/*ID:  865, INST: ld.par.w (sp+s16_t7),Da.h*/

	/*ID:  866, INST: ld.par.w (Rn)+,Da.l*/
void do__ld_par_w_l(const Word16* __p, Word32* __Da_l)
{
	*__Da_l =  ((*__p) & 0xffff) | ((*__Da_l)&0xffff0000);
}

	/*ID:  867, INST: ld.par.w (Rn)+Rk,Da.l*/

	/*ID:  868, INST: ld.par.w (Rn),Da.l*/

	/*ID:  869, INST: ld.par.w (Rn)-,Da.l*/

	/*ID:  870, INST: ld.par.w (Rn+Rk),Da.l*/

	/*ID:  871, INST: ld.par.w (Rn+s16_t7),Da.l*/

	/*ID:  872, INST: ld.par.w (sp+s16_t7),Da.l*/

	/*ID:  873, INST: ld.u.b (Rn)+,Da*/
void do__ld_u_b(const UWord8* __p, UWord32* __Da)
{
	*__Da = *__p;
}

	/*ID:  874, INST: ld.u.b (Rn)+,Ra*/

	/*ID:  875, INST: ld.u.b (Rn)+Rk,Da*/

	/*ID:  876, INST: ld.u.b (Rn)+Rk,Ra*/

	/*ID:  877, INST: ld.u.b (Rn),Da*/

	/*ID:  878, INST: ld.u.b (Rn),Ra*/

	/*ID:  879, INST: ld.u.b (Rn)-,Da*/

	/*ID:  880, INST: ld.u.b (Rn)-,Ra*/

	/*ID:  881, INST: ld.u.b (Rn+Rk),Da*/

	/*ID:  882, INST: ld.u.b (Rn+Rk),Ra*/

	/*ID:  883, INST: ld.u.b (Rn+s16_t7),Da*/

	/*ID:  884, INST: ld.u.b (Rn+s16_t7),Ra*/

	/*ID:  885, INST: ld.u.b (sp+s16_t7),Da*/

	/*ID:  886, INST: ld.u.b (sp+s16_t7),Ra*/

	/*ID:  887, INST: ld.u.b (sp-u9),Da*/

	/*ID:  888, INST: ld.u.b (sp-u9),Ra*/

	/*ID:  889, INST: ld.u.2b (Rn)+,Dab*/
void do__ld_u_2b(const UWord8* __p, UWord32* __Da, UWord32* __Db)
{
	do__ld_u_b(__p, __Da);
	do__ld_u_b(__p+1, __Db);
}

	/*ID:  890, INST: ld.u.2b (Rn)+,Rab*/

	/*ID:  891, INST: ld.u.2b (Rn)+Rk,Dab*/

	/*ID:  892, INST: ld.u.2b (Rn)+Rk,Rab*/

	/*ID:  893, INST: ld.u.2b (Rn),Dab*/

	/*ID:  894, INST: ld.u.2b (Rn),Rab*/

	/*ID:  895, INST: ld.u.2b (Rn)-,Dab*/

	/*ID:  896, INST: ld.u.2b (Rn)-,Rab*/

	/*ID:  897, INST: ld.u.2b (Rn+Rk),Dab*/

	/*ID:  898, INST: ld.u.2b (Rn+Rk),Rab*/

	/*ID:  899, INST: ld.u.2b (Rn+s16_t7),Dab*/

	/*ID:  900, INST: ld.u.2b (Rn+s16_t7),Rab*/

	/*ID:  901, INST: ld.u.4b (Rn)+,Dabcd*/
void do__ld_u_4b(const UWord8* __p, UWord32* __Da, UWord32* __Db, UWord32* __Dc, UWord32* __Dd)
{
	do__ld_u_2b(__p, __Da, __Db);
	do__ld_u_2b(__p+2, __Dc, __Dd);
}

	/*ID:  902, INST: ld.u.4b (Rn)+,Rabcd*/

	/*ID:  903, INST: ld.u.4b (Rn)+Rk,Dabcd*/

	/*ID:  904, INST: ld.u.4b (Rn)+Rk,Rabcd*/

	/*ID:  905, INST: ld.u.4b (Rn),Dabcd*/

	/*ID:  906, INST: ld.u.4b (Rn),Rabcd*/

	/*ID:  907, INST: ld.u.4b (Rn)-,Dabcd*/

	/*ID:  908, INST: ld.u.4b (Rn)-,Rabcd*/

	/*ID:  909, INST: ld.u.4b (Rn+Rk),Dabcd*/

	/*ID:  910, INST: ld.u.4b (Rn+Rk),Rabcd*/

	/*ID:  911, INST: ld.u.4b (Rn+s16_t7),Dabcd*/

	/*ID:  912, INST: ld.u.4b (Rn+s16_t7),Rabcd*/

	/*ID:  913, INST: ld.u.8b (Rn)+,Da__Dh*/
void do__ld_u_8b(const UWord8* __p, UWord32* __Da, UWord32* __Db, UWord32* __Dc, UWord32* __Dd, UWord32* __De, UWord32* __Df, UWord32* __Dg, UWord32* __Dh)
{
	do__ld_u_4b(__p, __Da, __Db, __Dc, __Dd);
	do__ld_u_4b(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  914, INST: ld.u.8b (Rn)+Rk,Da__Dh*/

	/*ID:  915, INST: ld.u.8b (Rn),Da__Dh*/

	/*ID:  916, INST: ld.u.8b (Rn)-,Da__Dh*/

	/*ID:  917, INST: ld.u.8b (Rn+Rk),Da__Dh*/

	/*ID:  918, INST: ld.u.8b (Rn+s16_t7),Da__Dh*/

	/*ID:  919, INST: ld.u.l (Rn)+,Da*/
void do__ld_u_l(const UWord32* __p, UWord32* __Da)
{
	*__Da = *__p;
}

	/*ID:  920, INST: ld.u.l (Rn)+Rk,Da*/

	/*ID:  921, INST: ld.u.l (Rn),Da*/

	/*ID:  922, INST: ld.u.l (Rn)-,Da*/

	/*ID:  923, INST: ld.u.l (Rn+Rk),Da*/

	/*ID:  924, INST: ld.u.l (Rn+s16_t7),Da*/

	/*ID:  925, INST: ld.u.l (sp+s16_t7),Da*/

	/*ID:  926, INST: ld.u.l (sp-u9_2),Da*/

	/*ID:  927, INST: ld.u.2l (Rn)+,Dab*/
void do__ld_u_2l(const UWord32* __p, UWord32* __Da, UWord32* __Db)
{
	do__ld_u_l(__p, __Da);
	do__ld_u_l(__p+1, __Db);
}

	/*ID:  928, INST: ld.u.2l (Rn)+Rk,Dab*/

	/*ID:  929, INST: ld.u.2l (Rn),Dab*/

	/*ID:  930, INST: ld.u.2l (Rn)-,Dab*/

	/*ID:  931, INST: ld.u.2l (Rn+Rk),Dab*/

	/*ID:  932, INST: ld.u.2l (Rn+s16_t7),Dab*/

	/*ID:  933, INST: ld.u.2l (sp+s16_t7),Dab*/

	/*ID:  934, INST: ld.u.4l (Rn)+,Dabcd*/
void do__ld_u_4l(const UWord32* __p, UWord32* __Da, UWord32* __Db, UWord32* __Dc, UWord32* __Dd)
{
	do__ld_u_2l(__p, __Da, __Db);
	do__ld_u_2l(__p+2, __Dc, __Dd);
}

	/*ID:  935, INST: ld.u.4l (Rn)+Rk,Dabcd*/

	/*ID:  936, INST: ld.u.4l (Rn),Dabcd*/

	/*ID:  937, INST: ld.u.4l (Rn)-,Dabcd*/

	/*ID:  938, INST: ld.u.4l (Rn+Rk),Dabcd*/

	/*ID:  939, INST: ld.u.4l (Rn+s16_t7),Dabcd*/

	/*ID:  940, INST: ld.u.8l (Rn)+,Da__Dh*/
void do__ld_u_8l(const UWord32* __p, UWord32* __Da, UWord32* __Db, UWord32* __Dc, UWord32* __Dd, UWord32* __De, UWord32* __Df, UWord32* __Dg, UWord32* __Dh)
{
	do__ld_u_4l(__p, __Da, __Db, __Dc, __Dd);
	do__ld_u_4l(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  941, INST: ld.u.8l (Rn)+Rk,Da__Dh*/

	/*ID:  942, INST: ld.u.8l (Rn),Da__Dh*/

	/*ID:  943, INST: ld.u.8l (Rn)-,Da__Dh*/

	/*ID:  944, INST: ld.u.8l (Rn+Rk),Da__Dh*/

	/*ID:  945, INST: ld.u.8l (Rn+s16_t7),Da__Dh*/

	/*ID:  946, INST: ld.u.w (Rn)+,Da*/
void do__ld_u_w(const UWord16* __p, UWord32* __Da)
{
	*__Da = *__p;
}

	/*ID:  947, INST: ld.u.w (Rn)+,Ra*/

	/*ID:  948, INST: ld.u.w (Rn)+Rk,Da*/

	/*ID:  949, INST: ld.u.w (Rn)+Rk,Ra*/

	/*ID:  950, INST: ld.u.w (Rn),Da*/

	/*ID:  951, INST: ld.u.w (Rn),Ra*/

	/*ID:  952, INST: ld.u.w (Rn)-,Da*/

	/*ID:  953, INST: ld.u.w (Rn)-,Ra*/

	/*ID:  954, INST: ld.u.w (Rn+Rk),Da*/

	/*ID:  955, INST: ld.u.w (Rn+Rk),Ra*/

	/*ID:  956, INST: ld.u.w (Rn+s16_t7),Da*/

	/*ID:  957, INST: ld.u.w (Rn+s16_t7),Ra*/

	/*ID:  958, INST: ld.u.w (sp+s16_t7),Da*/

	/*ID:  959, INST: ld.u.w (sp+s16_t7),Ra*/

	/*ID:  960, INST: ld.u.w (sp-u9_1),Da*/

	/*ID:  961, INST: ld.u.w (sp-u9_1),Ra*/

	/*ID:  962, INST: ld.u.2w (Rn)+,Dab*/
void do__ld_u_2w(const UWord16* __p, UWord32* __Da, UWord32* __Db)
{
	*__Da = *__p;
	*__Db = *(__p+1);
}

	/*ID:  963, INST: ld.u.2w (Rn)+,Rab*/

	/*ID:  964, INST: ld.u.2w (Rn)+Rk,Dab*/

	/*ID:  965, INST: ld.u.2w (Rn)+Rk,Rab*/

	/*ID:  966, INST: ld.u.2w (Rn),Dab*/

	/*ID:  967, INST: ld.u.2w (Rn),Rab*/

	/*ID:  968, INST: ld.u.2w (Rn)-,Dab*/

	/*ID:  969, INST: ld.u.2w (Rn)-,Rab*/

	/*ID:  970, INST: ld.u.2w (Rn+Rk),Dab*/

	/*ID:  971, INST: ld.u.2w (Rn+Rk),Rab*/

	/*ID:  972, INST: ld.u.2w (Rn+s16_t7),Dab*/

	/*ID:  973, INST: ld.u.2w (Rn+s16_t7),Rab*/

	/*ID:  974, INST: ld.u.4w (Rn)+,Dabcd*/
void do__ld_u_4w(const UWord16* __p, UWord32* __Da, UWord32* __Db, UWord32* __Dc, UWord32* __Dd)
{
	do__ld_u_2w(__p, __Da, __Db);
	do__ld_u_2w(__p+2, __Dc, __Dd);
}

	/*ID:  975, INST: ld.u.4w (Rn)+,Rabcd*/

	/*ID:  976, INST: ld.u.4w (Rn)+Rk,Dabcd*/

	/*ID:  977, INST: ld.u.4w (Rn)+Rk,Rabcd*/

	/*ID:  978, INST: ld.u.4w (Rn),Dabcd*/

	/*ID:  979, INST: ld.u.4w (Rn),Rabcd*/

	/*ID:  980, INST: ld.u.4w (Rn)-,Dabcd*/

	/*ID:  981, INST: ld.u.4w (Rn)-,Rabcd*/

	/*ID:  982, INST: ld.u.4w (Rn+Rk),Dabcd*/

	/*ID:  983, INST: ld.u.4w (Rn+Rk),Rabcd*/

	/*ID:  984, INST: ld.u.4w (Rn+s16_t7),Dabcd*/

	/*ID:  985, INST: ld.u.4w (Rn+s16_t7),Rabcd*/

	/*ID:  986, INST: ld.u.8w (Rn)+,Da__Dh*/
void do__ld_u_8w(const UWord16* __p, UWord32* __Da, UWord32* __Db, UWord32* __Dc, UWord32* __Dd, UWord32* __De, UWord32* __Df, UWord32* __Dg, UWord32* __Dh)
{
	do__ld_u_4w(__p, __Da, __Db, __Dc, __Dd);
	do__ld_u_4w(__p+4, __De, __Df, __Dg, __Dh);
}

	/*ID:  987, INST: ld.u.8w (Rn)+Rk,Da__Dh*/

	/*ID:  988, INST: ld.u.8w (Rn),Da__Dh*/

	/*ID:  989, INST: ld.u.8w (Rn)-,Da__Dh*/

	/*ID:  990, INST: ld.u.8w (Rn+Rk),Da__Dh*/

	/*ID:  991, INST: ld.u.8w (Rn+s16_t7),Da__Dh*/

	/*ID:  992, INST: ld2.2b (Rn)+,Da*/
void do__ld2_2b(const Word8* __p, Word40* __Da)
{
	Word32 Da_WH, Da_WL;
	Da_WH = *__p;
	Da_WL = *(__p+1);

	__set_WH(Da_WH, __Da);
	__set_WL(Da_WL, __Da);
}

	/*ID:  993, INST: ld2.2b (Rn)+Rk,Da*/

	/*ID:  994, INST: ld2.2b (Rn),Da*/

	/*ID:  995, INST: ld2.2b (Rn)-,Da*/

	/*ID:  996, INST: ld2.2b (Rn+Rk),Da*/

	/*ID:  997, INST: ld2.2b (Rn+s16_t7),Da*/

	/*ID:  998, INST: ld2.4b (Rn)+,Dab*/
void do__ld2_4b(const Word8* __p, Word40* __Da, Word40* __Db)
{
	do__ld2_2b(__p, __Da);
	do__ld2_2b(__p+2, __Db);
}

	/*ID:  999, INST: ld2.4b (Rn)+Rk,Dab*/

	/*ID: 1000, INST: ld2.4b (Rn),Dab*/

	/*ID: 1001, INST: ld2.4b (Rn)-,Dab*/

	/*ID: 1002, INST: ld2.4b (Rn+Rk),Dab*/

	/*ID: 1003, INST: ld2.4b (Rn+s16_t7),Dab*/

	/*ID: 1004, INST: ld2.8b (Rn)+,Dabcd*/
void do__ld2_8b(const Word8* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd)
{
	do__ld2_4b(__p, __Da, __Db);
	do__ld2_4b(__p+4, __Dc, __Dd);
}

	/*ID: 1005, INST: ld2.8b (Rn)+Rk,Dabcd*/

	/*ID: 1006, INST: ld2.8b (Rn),Dabcd*/

	/*ID: 1007, INST: ld2.8b (Rn)-,Dabcd*/

	/*ID: 1008, INST: ld2.8b (Rn+Rk),Dabcd*/

	/*ID: 1009, INST: ld2.8b (Rn+s16_t7),Dabcd*/

	/*ID: 1010, INST: ld2.16b (Rn)+,Da__Dh*/
void do__ld2_16b(const Word8* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd, Word40* __De, Word40* __Df, Word40* __Dg, Word40* __Dh)
{
	do__ld2_8b(__p, __Da, __Db, __Dc, __Dd);
	do__ld2_8b(__p+8, __De, __Df, __Dg, __Dh);
}

	/*ID: 1011, INST: ld2.16b (Rn)+Rk,Da__Dh*/

	/*ID: 1012, INST: ld2.16b (Rn),Da__Dh*/

	/*ID: 1013, INST: ld2.16b (Rn)-,Da__Dh*/

	/*ID: 1014, INST: ld2.16b (Rn+Rk),Da__Dh*/

	/*ID: 1015, INST: ld2.16b (Rn+s16_t7),Da__Dh*/

	/*ID: 1016, INST: ld2.2bf (Rn)+,Da*/
void do__ld2_2bf(const Word8* __p,Word40* __Da)
{
	Word32 __Dn_H,__Dn_L;
	__Dn_H = (*__p)<<8;
	__Dn_L = *(__p+1)<<8;

	__set_WH(__Dn_H,__Da);
	__set_WL(__Dn_L,__Da);
}

	/*ID: 1017, INST: ld2.2bf (Rn)+Rk,Da*/

	/*ID: 1018, INST: ld2.2bf (Rn),Da*/

	/*ID: 1019, INST: ld2.2bf (Rn)-,Da*/

	/*ID: 1020, INST: ld2.2bf (Rn+Rk),Da*/

	/*ID: 1021, INST: ld2.2bf (Rn+s16_t7),Da*/

	/*ID: 1022, INST: ld2.4bf (Rn)+,Dab*/
void do__ld2_4bf(const Word8* __p, Word40* __Da, Word40* __Db)
{
	do__ld2_2bf(__p,__Da);
	do__ld2_2bf(__p+2,__Db);
}

	/*ID: 1023, INST: ld2.4bf (Rn)+Rk,Dab*/

	/*ID: 1024, INST: ld2.4bf (Rn),Dab*/

	/*ID: 1025, INST: ld2.4bf (Rn)-,Dab*/

	/*ID: 1026, INST: ld2.4bf (Rn+Rk),Dab*/

	/*ID: 1027, INST: ld2.4bf (Rn+s16_t7),Dab*/

	/*ID: 1028, INST: ld2.8bf (Rn)+,Dabcd*/
void do__ld2_8bf(const Word8* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd)
{
	do__ld2_4bf(__p,__Da ,__Db);
	do__ld2_4bf(__p+4,__Dc ,__Dd);
}

	/*ID: 1029, INST: ld2.8bf (Rn)+Rk,Dabcd*/

	/*ID: 1030, INST: ld2.8bf (Rn),Dabcd*/

	/*ID: 1031, INST: ld2.8bf (Rn)-,Dabcd*/

	/*ID: 1032, INST: ld2.8bf (Rn+Rk),Dabcd*/

	/*ID: 1033, INST: ld2.8bf (Rn+s16_t7),Dabcd*/

	/*ID: 1034, INST: ld2.16bf (Rn)+,Da__Dh*/
void do__ld2_16bf(const Word8* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd, Word40* __De, Word40* __Df, Word40* __Dg, Word40* __Dh)
{
	do__ld2_8bf(__p,__Da,__Db,__Dc ,__Dd);
	do__ld2_8bf(__p+8,__De ,__Df,__Dg,__Dh);
}

	/*ID: 1035, INST: ld2.16bf (Rn)+Rk,Da__Dh*/

	/*ID: 1036, INST: ld2.16bf (Rn),Da__Dh*/

	/*ID: 1037, INST: ld2.16bf (Rn)-,Da__Dh*/

	/*ID: 1038, INST: ld2.16bf (Rn+Rk),Da__Dh*/

	/*ID: 1039, INST: ld2.16bf (Rn+s16_t7),Da__Dh*/

	/*ID: 1040, INST: ld2.2f (Rn)+,Da*/
void do__ld2_2f(const Word16* __p,Word40* __Da)
{
	Word32 __Dn_H,__Dn_L;
	__Dn_H = *__p;
	__Dn_L = *(__p+1);
	__set_WH(__Dn_H,__Da);
	__set_WL(__Dn_L,__Da);
}

	/*ID: 1041, INST: ld2.2f (Rn)+Rk,Da*/

	/*ID: 1042, INST: ld2.2f (Rn),Da*/

	/*ID: 1043, INST: ld2.2f (Rn)-,Da*/

	/*ID: 1044, INST: ld2.2f (Rn+Rk),Da*/

	/*ID: 1045, INST: ld2.2f (Rn+s16_t7),Da*/

	/*ID: 1046, INST: ld2.4f (Rn)+,Dab*/
void do__ld2_4f(const Word16* __p, Word40* __Da, Word40* __Db)
{
	do__ld2_2f(__p, __Da);
	do__ld2_2f(__p+2, __Db);
}

	/*ID: 1047, INST: ld2.4f (Rn)+Rk,Dab*/

	/*ID: 1048, INST: ld2.4f (Rn),Dab*/

	/*ID: 1049, INST: ld2.4f (Rn)-,Dab*/

	/*ID: 1050, INST: ld2.4f (Rn+Rk),Dab*/

	/*ID: 1051, INST: ld2.4f (Rn+s16_t7),Dab*/

	/*ID: 1052, INST: ld2.8f (Rn)+,Dabcd*/
void do__ld2_8f(const Word16* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd)
{
	do__ld2_4f(__p, __Da,__Db);
	do__ld2_4f(__p+4,__Dc,__Dd);
}

	/*ID: 1053, INST: ld2.8f (Rn)+,Dacdb*/

	/*ID: 1054, INST: ld2.8f (Rn)+Rk,Dabcd*/

	/*ID: 1055, INST: ld2.8f (Rn)+Rk,Dacdb*/

	/*ID: 1056, INST: ld2.8f (Rn),Dabcd*/

	/*ID: 1057, INST: ld2.8f (Rn),Dacdb*/

	/*ID: 1058, INST: ld2.8f (Rn)-,Dabcd*/

	/*ID: 1059, INST: ld2.8f (Rn)-,Dacdb*/

	/*ID: 1060, INST: ld2.8f (Rn+Rk),Dabcd*/

	/*ID: 1061, INST: ld2.8f (Rn+Rk),Dacdb*/

	/*ID: 1062, INST: ld2.8f (Rn+s16_t7),Dabcd*/

	/*ID: 1063, INST: ld2.8f (Rn+s16_t7),Dacdb*/

	/*ID: 1064, INST: ld2.16f (Rn)+,Da__Dh*/
void do__ld2_16f(const Word16* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd, Word40* __De, Word40* __Df, Word40* __Dg, Word40* __Dh)
{
	do__ld2_8f(__p, __Da,__Db,__Dc,__Dd);
	do__ld2_8f(__p+8,__De,__Df,__Dg,__Dh);
}

	/*ID: 1065, INST: ld2.16f (Rn)+,Da__DhMod8*/

	/*ID: 1066, INST: ld2.16f (Rn)+,Da__DhMod8p*/

	/*ID: 1067, INST: ld2.16f (Rn)+,Da__DhR3*/

	/*ID: 1068, INST: ld2.16f (Rn)+,Da__DhR5*/

	/*ID: 1069, INST: ld2.16f (Rn)+Rk,Da__Dh*/

	/*ID: 1070, INST: ld2.16f (Rn)+Rk,Da__DhMod8*/

	/*ID: 1071, INST: ld2.16f (Rn)+Rk,Da__DhMod8p*/

	/*ID: 1072, INST: ld2.16f (Rn)+Rk,Da__DhR3*/

	/*ID: 1073, INST: ld2.16f (Rn)+Rk,Da__DhR5*/

	/*ID: 1074, INST: ld2.16f (Rn),Da__Dh*/

	/*ID: 1075, INST: ld2.16f (Rn),Da__DhMod8*/

	/*ID: 1076, INST: ld2.16f (Rn),Da__DhMod8p*/

	/*ID: 1077, INST: ld2.16f (Rn),Da__DhR3*/

	/*ID: 1078, INST: ld2.16f (Rn),Da__DhR5*/

	/*ID: 1079, INST: ld2.16f (Rn)-,Da__Dh*/

	/*ID: 1080, INST: ld2.16f (Rn)-,Da__DhMod8*/

	/*ID: 1081, INST: ld2.16f (Rn)-,Da__DhMod8p*/

	/*ID: 1082, INST: ld2.16f (Rn)-,Da__DhR3*/

	/*ID: 1083, INST: ld2.16f (Rn)-,Da__DhR5*/

	/*ID: 1084, INST: ld2.16f (Rn+Rk),Da__Dh*/

	/*ID: 1085, INST: ld2.16f (Rn+Rk),Da__DhMod8*/

	/*ID: 1086, INST: ld2.16f (Rn+Rk),Da__DhMod8p*/

	/*ID: 1087, INST: ld2.16f (Rn+Rk),Da__DhR3*/

	/*ID: 1088, INST: ld2.16f (Rn+Rk),Da__DhR5*/

	/*ID: 1089, INST: ld2.16f (Rn+s16_t7),Da__Dh*/

	/*ID: 1090, INST: ld2.16f (Rn+s16_t7),Da__DhMod8*/

	/*ID: 1091, INST: ld2.16f (Rn+s16_t7),Da__DhMod8p*/

	/*ID: 1092, INST: ld2.16f (Rn+s16_t7),Da__DhR3*/

	/*ID: 1093, INST: ld2.16f (Rn+s16_t7),Da__DhR5*/

	/*ID: 1094, INST: ld2.32f (Rn)+,Da__Dp*/
void do__ld2_32f(const Word16* __p, Word40* __Da, Word40* __Db, Word40* __Dc, Word40* __Dd, Word40* __De, Word40* __Df, Word40* __Dg, Word40* __Dh, Word40* __Di, Word40* __Dj, Word40* __Dk, Word40* __Dl, Word40* __Dm, Word40* __Dn, Word40* __Do, Word40* __Dp)
{
	do__ld2_16f(__p, __Da,__Db,__Dc,__Dd,__De,__Df,__Dg,__Dh);
	do__ld2_16f(__p+16,__Di,__Dj,__Dk,__Dl,__Dm,__Dn,__Do,__Dp);
}

	/*ID: 1095, INST: ld2.32f (Rn)+,Da__DpMod16*/

	/*ID: 1096, INST: ld2.32f (Rn)+Rk,Da__Dp*/

	/*ID: 1097, INST: ld2.32f (Rn)+Rk,Da__DpMod16*/

	/*ID: 1098, INST: ld2.32f (Rn),Da__Dp*/

	/*ID: 1099, INST: ld2.32f (Rn),Da__DpMod16*/

	/*ID: 1100, INST: ld2.32f (Rn)-,Da__Dp*/

	/*ID: 1101, INST: ld2.32f (Rn)-,Da__DpMod16*/

	/*ID: 1102, INST: ld2.32f (Rn+Rk),Da__Dp*/

	/*ID: 1103, INST: ld2.32f (Rn+Rk),Da__DpMod16*/

	/*ID: 1104, INST: ld2.32f (Rn+s16_t7),Da__Dp*/

	/*ID: 1105, INST: ld2.32f (Rn+s16_t7),Da__DpMod16*/

	/*ID: 1106, INST: ld2.u.2b (Rn)+,Da*/
void  do__ld2_u_2b(const UWord8* __p,UWord40* __Da)
{
	UWord32 __Dn_WH,__Dn_WL;

	__Dn_WH = * __p;
	__Dn_WL = *(__p+1);
	__set_u_WH(__Dn_WH,__Da);
	__set_u_WL(__Dn_WL,__Da);
}

	/*ID: 1107, INST: ld2.u.2b (Rn)+Rk,Da*/

	/*ID: 1108, INST: ld2.u.2b (Rn),Da*/

	/*ID: 1109, INST: ld2.u.2b (Rn)-,Da*/

	/*ID: 1110, INST: ld2.u.2b (Rn+Rk),Da*/

	/*ID: 1111, INST: ld2.u.2b (Rn+s16_t7),Da*/

	/*ID: 1112, INST: ld2.u.4b (Rn)+,Dab*/
void do__ld2_u_4b(const UWord8* __p, UWord40* __Da, UWord40* __Db)
{
	do__ld2_u_2b(__p,__Da);
	do__ld2_u_2b(__p+2,__Db);
}

	/*ID: 1113, INST: ld2.u.4b (Rn)+Rk,Dab*/

	/*ID: 1114, INST: ld2.u.4b (Rn),Dab*/

	/*ID: 1115, INST: ld2.u.4b (Rn)-,Dab*/

	/*ID: 1116, INST: ld2.u.4b (Rn+Rk),Dab*/

	/*ID: 1117, INST: ld2.u.4b (Rn+s16_t7),Dab*/

	/*ID: 1118, INST: ld2.u.8b (Rn)+,Dabcd*/
void do__ld2_u_8b(const UWord8* __p, UWord40* __Da, UWord40* __Db, UWord40* __Dc, UWord40* __Dd)
{
	do__ld2_u_4b(__p,__Da,__Db);
	do__ld2_u_4b(__p+4,__Dc,__Dd);
}

	/*ID: 1119, INST: ld2.u.8b (Rn)+Rk,Dabcd*/

	/*ID: 1120, INST: ld2.u.8b (Rn),Dabcd*/

	/*ID: 1121, INST: ld2.u.8b (Rn)-,Dabcd*/

	/*ID: 1122, INST: ld2.u.8b (Rn+Rk),Dabcd*/

	/*ID: 1123, INST: ld2.u.8b (Rn+s16_t7),Dabcd*/

	/*ID: 1124, INST: ld2.u.16b (Rn)+,Da__Dh*/
void do__ld2_u_16b(const UWord8* __p, UWord40* __Da, UWord40* __Db, UWord40* __Dc, UWord40* __Dd, UWord40* __De, UWord40* __Df, UWord40* __Dg, UWord40* __Dh)
{
	do__ld2_u_8b(__p,__Da,__Db,__Dc,__Dd);
	do__ld2_u_8b(__p+8,__De,__Df,__Dg,__Dh);
}

	/*ID: 1125, INST: ld2.u.16b (Rn)+Rk,Da__Dh*/

	/*ID: 1126, INST: ld2.u.16b (Rn),Da__Dh*/

	/*ID: 1127, INST: ld2.u.16b (Rn)-,Da__Dh*/

	/*ID: 1128, INST: ld2.u.16b (Rn+Rk),Da__Dh*/

	/*ID: 1129, INST: ld2.u.16b (Rn+s16_t7),Da__Dh*/

	/*ID: 1130, INST: ldc.b (sp+s16_t7),Da*/

	/*ID: 1131, INST: ldc.b (sp+s16_t7),Ra*/

	/*ID: 1132, INST: ldc.bf (sp+s16_t7),Da*/

	/*ID: 1133, INST: ldc.f (sp+s16_t7),Da*/

	/*ID: 1134, INST: ldc.l (sp+s16_t7),Da*/

	/*ID: 1135, INST: ldc.l (sp+s16_t7),Ra*/

	/*ID: 1136, INST: ldc.w (sp+s16_t7),Da*/

	/*ID: 1137, INST: ldc.w (sp+s16_t7),Ra*/

	/*ID: 1138, INST: ldc.x (sp+s16_t7),Da*/

	/*ID: 1139, INST: ldc.u.b (sp+s16_t7),Da*/

	/*ID: 1140, INST: ldc.u.b (sp+s16_t7),Ra*/

	/*ID: 1141, INST: ldc.u.l (sp+s16_t7),Da*/

	/*ID: 1142, INST: ldc.u.w (sp+s16_t7),Da*/

	/*ID: 1143, INST: ldc.u.w (sp+s16_t7),Ra*/

	/*ID: 1144, INST: ldrstc*/

	/*ID: 1145, INST: ldsh2.dn.2f (Rn)+,Dab*/
void do__ldsh2_dn_2f(const Word16* __p, Word40* __Da, Word40* __Db)
{
	Word32 __Db_WL,__Da_WH;
	__Db_WL = *(__p+1);
	__Da_WH = __extract_wh(*__Da);
	__set_WH(__Da_WH,__Db);
	__set_WL(__Db_WL,__Db);
	do__ld2_2f(__p,__Da);
}

	/*ID: 1146, INST: ldsh2.dn.2f (Rn)+Rk,Dab*/

	/*ID: 1147, INST: ldsh2.dn.2f (Rn),Dab*/

	/*ID: 1148, INST: ldsh2.dn.2f (Rn)-,Dab*/

	/*ID: 1149, INST: ldsh2.dn.2f (Rn+Rk),Dab*/

	/*ID: 1150, INST: ldsh2.dn.2f (Rn+s16_t7),Dab*/

	/*ID: 1151, INST: ldsh2.dn.2f (sp+s16_t7),Dab*/

	/*ID: 1152, INST: ldsh2.up.2f (Rn)+,Dab*/
void do__ldsh2_up_2f(const Word16* __p, Word40* __Da, Word40* __Db)
{
	Word32 __Db_WH,__Da_WL;

	__Da_WL = __extract_wl(*__Da);
	__Db_WH = *(__p);

	__set_WH(__Db_WH,__Db);
	__set_WL(__Da_WL,__Db);
	do__ld2_2f(__p,__Da);
}

	/*ID: 1153, INST: ldsh2.up.2f (Rn)+Rk,Dab*/

	/*ID: 1154, INST: ldsh2.up.2f (Rn),Dab*/

	/*ID: 1155, INST: ldsh2.up.2f (Rn)-,Dab*/

	/*ID: 1156, INST: ldsh2.up.2f (Rn+Rk),Dab*/

	/*ID: 1157, INST: ldsh2.up.2f (Rn+s16_t7),Dab*/

	/*ID: 1158, INST: ldsh2.up.2f (sp+s16_t7),Dab*/

	/*ID: 1159, INST: lextc.lft.x Da,Db*/

	/*ID: 1160, INST: lextc.rgt.l Da,Db*/

	/*ID: 1161, INST: lextc.rgt.x Da,Db*/

	/*ID: 1162, INST: log2 Da,Dmn*/
void do__log2(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word8 inp;
	Word32 LOG2_LUT_A =0,__Dm_2 =0;
	Word32 LOG2_LUT_B=0,__Dn_2=0;
	inp = (__Da & 0x3F000000)>>24;
	LOG2_LUT_A = get_LOG2_LUT_A(inp);
	LOG2_LUT_B = get_LOG2_LUT_B(inp);
	__Dm_2 = LOG2_LUT_A << 14;
	__Dn_2 = (~LOG2_LUT_B) << 15;
	*__Dm = __x_set(0xFF,__Dm_2);
	*__Dn = __x_set(0x0,__Dn_2);
}

	/*ID: 1163, INST: lpend.2 RelAdd19_t2*/

	/*ID: 1164, INST: lpend.1 RelAdd19_t2*/

	/*ID: 1165, INST: lpend.3 RelAdd19_t2*/

	/*ID: 1166, INST: lpend.0 RelAdd19_t2*/

	/*ID: 1167, INST: lpend.sq0 RelAdd19_t2*/

	/*ID: 1168, INST: lpend.sq1 RelAdd19_t2*/

	/*ID: 1169, INST: lpend.sq2 RelAdd19_t2*/

	/*ID: 1170, INST: lpend.sq3 RelAdd19_t2*/

	/*ID: 1171, INST: lpskip.2 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1172, INST: lpskip.1 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1173, INST: lpskip.3 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1174, INST: lpskip.0 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1175, INST: lpskip.sq0 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1176, INST: lpskip.sq1 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1177, INST: lpskip.sq2 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1178, INST: lpskip.sq3 RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1179, INST: lpskip.0.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1180, INST: lpskip.1.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1181, INST: lpskip.2.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1182, INST: lpskip.3.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1183, INST: lpskip.sq0.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1184, INST: lpskip.sq1.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1185, INST: lpskip.sq2.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1186, INST: lpskip.sq3.u RelAdd19_t3,URRelAdd4_t3*/

	/*ID: 1187, INST: lpst.2 RelAdd19_t2*/

	/*ID: 1188, INST: lpst.1 RelAdd19_t2*/

	/*ID: 1189, INST: lpst.3 RelAdd19_t2*/

	/*ID: 1190, INST: lpst.0 RelAdd19_t2*/

	/*ID: 1191, INST: lpst.sq0 RelAdd19_t2*/

	/*ID: 1192, INST: lpst.sq1 RelAdd19_t2*/

	/*ID: 1193, INST: lpst.sq2 RelAdd19_t2*/

	/*ID: 1194, INST: lpst.sq3 RelAdd19_t2*/

	/*ID: 1195, INST: lsh.lft.l Da,Db,Dn*/
Word32 do__lsh_lft_l(Word32 __Da, Word32 __Db)
{
	UWord32 shift_data;
	/*Consider only 6:0, first 7 bits, 7th bit is for sign
	  can be changed to first 6 bits . 7 bits is taken from 3850 reference code*/

	if (__Da & 0x20)
	  /* __Da = __l_or_x(__Da,0xffffffc0);*/
		__Da = __Da | 0xffffff80;
	else
		__Da = __Da & 0x3f;

	/*if we call right shift in left shift and vice versa then it will go infinite loop for certain cases
	 if number of bits considered is same for both*/

	if(__Da<0) /*return do__lsh_rgt_l(-__Da,__Db); */
	{
		shift_data = (UWord32)__Db;
		shift_data >>= -__Da;
		return(Word32)shift_data;
	}
	else
	{
		__Db<<=__Da;
		return __Db;
	}
}

	/*ID: 1196, INST: lsh.lft.2l Da,Dcd,Dmn*/
void do__lsh_lft_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1197, INST: lsh.lft.ll Da,Dcd,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__lsh_lft_ll(Word32 __Da, Int64 __Dcd)
{
}
#endif

	/*ID: 1198, INST: lsh.lft.2w Da,Db,Dn*/
Word32 do__lsh_lft_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1199, INST: lsh.lft.4w Da,Dcd,Dmn*/
void do__lsh_lft_4w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1200, INST: lsh.lft.x Da,Db,Dn*/
Word40 do__lsh_lft_x(Word32 __Da, Word40 __Db)
{
}

	/*ID: 1201, INST: lsh.lft.2x Da,Dcd,Dmn*/
void do__lsh_lft_2x(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1202, INST: lsh.rgt.l Da,Db,Dn*/
Word32 do__lsh_rgt_l(Word32 __Da, Word32 __Db)
{
	UWord32 shift_data;
	/*Consider only 6:0, first 7 bits, 7th bit is for sign
	  can be changed to first 6 bits . 7 bits is taken from 3850 reference code*/
	if (__Da & 0x40)
	  /*__Da = __l_or_x(__Da,0xffffff80); */
		__Da = __Da | 0xffffff80;
	else
		__Da = __Da & 0x3f;


	if(__Da<0)/*return do__lsh_lft_l(-__Da,__Db); */
	{
		__Db <<= - __Da;
		return __Db;

	}
	else
	{
		shift_data = (UWord32)__Db;
		shift_data >>= __Da;
		return(Word32)shift_data;
	}

}

	/*ID: 1203, INST: lsh.rgt.l #u5,Da,Dn*/

	/*ID: 1204, INST: lsh.rgt.2l Da,Dcd,Dmn*/
void do__lsh_rgt_2l(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1205, INST: lsh.rgt.2l #u5,Dab,Dmn*/

	/*ID: 1206, INST: lsh.rgt.ll Da,Dcd,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__lsh_rgt_ll(Word32 __Da, Int64 __Dcd)
{
}
#endif

	/*ID: 1207, INST: lsh.rgt.ll #u6_t3,Dab,Dmn*/

	/*ID: 1208, INST: lsh.rgt.2w Da,Db,Dn*/
Word32 do__lsh_rgt_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1209, INST: lsh.rgt.2w #u5,Da,Dn*/

	/*ID: 1210, INST: lsh.rgt.4w Da,Dcd,Dmn*/
void do__lsh_rgt_4w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1211, INST: lsh.rgt.4w #u4,Dab,Dmn*/

	/*ID: 1212, INST: lsh.rgt.x Da,Db,Dn*/
Word40 do__lsh_rgt_x(Word32 __Da, Word40 __Db)
{
}

	/*ID: 1213, INST: lsh.rgt.x #ue5,Da,Dn*/

	/*ID: 1214, INST: lsh.rgt.2x Da,Dcd,Dmn*/
void do__lsh_rgt_2x(Word32 __Da, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1215, INST: lsh.rgt.2x #ue5,Dab,Dmn*/

	/*ID: 1216, INST: lsha.lft Ra,Rb,Rn*/
Word32 do__lsha_lft(Word32 __Ra, Word32 __Rb)
{
}

	/*ID: 1217, INST: lsha.rgt Ra,Rb,Rn*/
Word32 do__lsha_rgt(Word32 __Ra, Word32 __Rb)
{
}

	/*ID: 1218, INST: lsha.rgt #u5,Ra,Rn*/

	/*ID: 1219, INST: lsha.rgt #u5,Ra,Rn*/

	/*ID: 1220, INST: mac.2t Da,Db,Dn*/
Word40 do__mac_2t(Word40 __Dn, Word32 __Da, Word32 __Db)
{
      Word32 Dn_WH, Dn_WL;

      Dn_WH = __extract_wh(__Dn);
      Dn_WL = __extract_wl(__Dn);

      Dn_WH = Dn_WH + __x_to_l(do__ash_rgt_x(16,do__mpy_x_hh(__Da, __Db)));
      Dn_WL = Dn_WL + __x_to_l(do__ash_rgt_x(16,do__mpy_x_ll(__Da, __Db)));


      Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /*sign extend the result*/
	  Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;

	  return __x_set( ((Dn_WH>>16)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
}

	/*ID: 1221, INST: mac.2t -Da,Db,Dn*/
Word40 do__mac_2t_m(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1222, INST: mac.x #s16_t2,Da.h,Dn*/
Word40 do__mac_x_h(Word40 __Dn, Word32 __imm1, Word32 __Da_h)
{
}

	/*ID: 1223, INST: mac.x Da.h,Db.h,Dn*/
Word40 do__mac_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
}

	/*ID: 1224, INST: mac.x #s16_t2,Da.l,Dn*/
Word40 do__mac_x_l(Word40 __Dn, Word32 __imm1, Word32 __Da_l)
{
}

	/*ID: 1225, INST: mac.x Da.l,Db.l,Dn*/
Word40 do__mac_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1226, INST: mac.x Da.h,Db.l,Dn*/
Word40 do__mac_x_hl(Word40 __Dn, Word32 __Da_h, Word32 __Db_l)
{
}

	/*ID: 1227, INST: mac.2x Da,Db,Dmn*/
void do__mac_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
  Word40 mult1 = do__mpy_x_hh(__Da, __Db);
   if ((__Da >> 31) == 1) {
       *__Dm = do__sub_x(mult1, *__Dm);
   } else {
       *__Dm = do__add_x(mult1, *__Dm);
   }

   mult1 = do__mpy_x_ll(__Da, __Db);/*(DSrc1 .L *  DSrc2 .L) << 1; */
   if ((__Da >> 31) == 1) {
	   *__Dn = do__sub_x(mult1, *__Dn);
   } else {
       *__Dn = do__add_x(mult1, *__Dn);
   }
}

/*ID: 1228, INST: mac.h.2x Da,Db,Dmn */
void do__mac_h_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

/* ID: 1229, INST: mac.h.s.2x Da,Db,Dmn*/
void do__mac_h_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}
/* ID: 1230, INST: mac.hh.2x Dab,Dcd,Dmn*/
void do__mac_hh_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

/*ID: 1231, INST: mac.hh.s.2x Dab,Dcd,Dmn */
void do__mac_hh_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

/*ID: 1232, INST: mac.i.4w Dab,Dcd,Dmn */
void do__mac_i_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

/* ID: 1233, INST: mac.i.x Da.l,Db.l,Dn*/
Word40 do__mac_i_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

/* ID: 1234, INST: mac.i.x -Da.l,Db.l,Dn*/
Word40 do__mac_i_x_m_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1235, INST: mac.l.2x Da,Db,Dmn*/
void do__mac_l_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
	Word40 result;

	result = do__mpy_x_hl(__Db,__Da);
	*__Dm = do__add_x(*__Dm,result);
	result = do__mpy_x_ll(__Da,__Db);
	*__Dn = do__add_x(*__Dn,result);
}

	/*ID: 1236, INST: mac.l.s.2x Da,Db,Dmn*/
void do__mac_l_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1237, INST: mac.leg.x #s16_t2,Da.h,Dn*/

	/*ID: 1238, INST: mac.leg.x Da.h,Db.h,Dn*/

	/*ID: 1239, INST: mac.leg.x #s16_t2,Da.l,Dn*/

	/*ID: 1240, INST: mac.leg.x Da.l,Db.l,Dn*/

	/*ID: 1241, INST: mac.leg.x Da.h,Db.l,Dn*/

	/*ID: 1242, INST: mac.leg.x -Da.h,Db.h,Dn*/

	/*ID: 1243, INST: mac.leg.x -Da.l,Db.l,Dn*/

	/*ID: 1244, INST: mac.leg.x -Da.h,Db.l,Dn*/

	/*ID: 1245, INST: mac.ll.2x Dab,Dcd,Dmn*/
void do__mac_ll_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1246, INST: mac.ll.i.2x Dab,Dcd,Dmn*/
void do__mac_ll_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1247, INST: mac.ll.s.2x Dab,Dcd,Dmn*/
void do__mac_ll_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1248, INST: mac.r.2t Da,Db,Dn*/
Word40 do__mac_r_2t(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1249, INST: mac.r.2t -Da,Db,Dn*/
Word40 do__mac_r_2t_m(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1250, INST: mac.r.4t Dab,Dcd,Dmn*/
void do__mac_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1251, INST: mac.r.x Da.h,Db.h,Dn*/
Word40 do__mac_r_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
	Word40 result;
	result = do__mpy_x_hh(__Da_h,__Db_h);
	result = do__add_x(result,__Dn);
	result = __x_round(result);
	return (result);
}

	/*ID: 1252, INST: mac.r.x Da.l,Db.l,Dn*/
Word40 do__mac_r_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1253, INST: mac.r.x Da.l,Db.h,Dn*/
Word40 do__mac_r_x_lh(Word40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1254, INST: mac.rleg.x Da.h,Db.h,Dn*/
Word40 do__mac_rleg_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
}

	/*ID: 1255, INST: mac.rleg.x Da.l,Db.l,Dn*/
Word40 do__mac_rleg_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1256, INST: mac.rleg.x Da.l,Db.h,Dn*/
Word40 do__mac_rleg_x_lh(Word40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1257, INST: mac.rleg.x -Da.h,Db.h,Dn*/
Word40 do__mac_rleg_x_m_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
}

	/*ID: 1258, INST: mac.rleg.x -Da.l,Db.l,Dn*/
Word40 do__mac_rleg_x_m_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1259, INST: mac.rleg.x -Da.l,Db.h,Dn*/
Word40 do__mac_rleg_x_m_lh(Word40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1260, INST: mac.s.2w Da,Db,Dn*/
Word32 do__mac_s_2w(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1261, INST: mac.s.x #s16_t2,Da.h,Dn*/
Word40 do__mac_s_x_h(Word40 __Dn, Word32 __imm1, Word32 __Da_h)
{
}

	/*ID: 1262, INST: mac.s.x Da.h,Db.h,Dn*/
Word40 do__mac_s_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
	Word40 result;

	result = do__mpy_x_hh(__Da_h,__Db_h);
	result = do__add_x(result,__Dn);
	result = __l_to_x(do__sat_x_l(result));
	return (result);
}

	/*ID: 1263, INST: mac.s.x #s16_t2,Da.l,Dn*/
Word40 do__mac_s_x_l(Word40 __Dn, Word32 __imm1, Word32 __Da_l)
{
}

	/*ID: 1264, INST: mac.s.x Da.l,Db.l,Dn*/
Word40 do__mac_s_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
	Word40 result;

	result = do__mpy_x_ll(__Da_l,__Db_l);
	result = do__add_x(result,__Dn);
	result = __l_to_x(do__sat_x_l(result));
	return (result);
}

	/*ID: 1265, INST: mac.s.x Da.h,Db.l,Dn*/
Word40 do__mac_s_x_hl(Word40 __Dn, Word32 __Da_h, Word32 __Db_l)
{
	Word40 result;

	result = do__mpy_x_hl(__Da_h,__Db_l);
	result = do__add_x(result,__Dn);
	result = __l_to_x(do__sat_x_l(result));
	return (result);

}

	/*ID: 1266, INST: mac.s.2x Da,Db,Dmn*/
void do__mac_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1267, INST: mac.sr.2w Da,Db,Dn*/
Word32 do__mac_sr_2w(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1268, INST: mac.sr.4w Dab,Dcd,Dmn*/
void do__mac_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1269, INST: mac.sr.x Da.h,Db.h,Dn*/
Word40 do__mac_sr_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
	Word40 result;
	result = do__mpy_x_hh(__Da_h,__Db_h);
	result = do__add_x(result,__Dn);
	result = __l_to_x(do__sat_x_l(__x_round(result)));
	return (result);
}

	/*ID: 1270, INST: mac.sr.x Da.l,Db.l,Dn*/
Word40 do__mac_sr_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
	Word40 result;
	result = do__mpy_x_ll(__Da_l,__Db_l);
	result = do__add_x(result,__Dn);
	result = __l_to_x(do__sat_x_l(__x_round(result)));
	return (result);

}

	/*ID: 1271, INST: mac.sr.x Da.l,Db.h,Dn*/
Word40 do__mac_sr_x_lh(Word40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
	Word40 result;
	result = do__mpy_x_hl(__Db_h,__Da_l);
	result = do__add_x(result,__Dn);
	result = __l_to_x(do__sat_x_l(__x_round(result)));
	return (result);
}

	/*ID: 1272, INST: mac.su.i.2t Da,Db,Dn*/
Word40 do__mac_su_i_2t(Word40 __Dn, Word32 __Da, Word32 __Db)
{

	  Word32 Dn_WH, Dn_WL;
	  Word40 Dn;

	  Dn_WH = __extract_wh(__Dn) & 0xfffff;
	  Dn_WH = Dn_WH + ((__l_get_msb(__Da) * (UWord16)(__l_get_msb(__Db) & 0xffff)) & 0xfffff);

	  Dn_WL = __extract_wl(__Dn) & 0xfffff;
	  Dn_WL = Dn_WL + ((__l_get_lsb(__Da) * (UWord16)(__l_get_lsb(__Db) & 0x0000ffff)) & 0xfffff);

	  Dn_WH |= (((Dn_WH >> 19) & 0x1) == 1) ? 0xfff00000:0x0; /*sign extend the result */
	  Dn_WL |= (((Dn_WL >> 19) & 0x1) == 1) ? 0xfff00000:0x0;

	  Dn = __x_set(((Dn_WH>>16)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
	  return Dn;
}

	/*ID: 1273, INST: mac.us.i.x Da.l,Db.l,Dn*/
Word40 do__mac_us_i_x_ll(Word40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1274, INST: mac.uu.i.x Da.l,Db.l,Dn*/
UWord40 do__mac_uu_i_x_ll(UWord40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1275, INST: mac.uu.i.x -Da.l,Db.l,Dn*/
UWord40 do__mac_uu_i_x_m_ll(UWord40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1276, INST: mac32.il.l Da,Db,Dn*/
Word32 do__mac32_il_l(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1277, INST: mac32.il.l -Da,Db,Dn*/
Word32 do__mac32_il_l_m(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1278, INST: mac32.r.x Da,Db,Dn*/
Word40 do__mac32_r_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1279, INST: mac32.r.2x Da,Dcd,Dmn*/
void do__mac32_r_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1280, INST: mac32.r.2x Dab,Dcd,Dmn*/
void do__mac32_r_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1281, INST: mac32.sr.x Da,Db,Dn*/
Word40 do__mac32_sr_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1282, INST: mac32.sr.2x Da,Dcd,Dmn*/
void do__mac32_sr_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1283, INST: mac32.sr.2x Dab,Dcd,Dmn*/
void do__mac32_sr_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1284, INST: maccx.2x Da,Db,Dmn*/
void do__maccx_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
	Word40 result;

	result = do__sub_x(do__mpy_x_ll(__Da,__Db),do__mpy_x_hh(__Da,__Db));
	*__Dm = do__add_x(*__Dm,result);

	result = do__add_x(do__mpy_x_hl(__Da,__Db),do__mpy_x_hl(__Db,__Da));
	*__Dn = do__add_x(*__Dn,result);
}

	/*ID: 1285, INST: maccx.2x -Da,Db,Dmn*/
void do__maccx_2x_m(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1286, INST: maccx.c.2x Da,Db,Dmn*/
void do__maccx_c_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
	Word40 result;

	result = do__add_x(do__mpy_x_hh(__Da,__Db),do__mpy_x_ll(__Da,__Db));
	*__Dm = do__add_x(*__Dm,result);

	result = do__sub_x(do__mpy_x_hl(__Db,__Da),do__mpy_x_hl(__Da,__Db));
	*__Dn = do__add_x(*__Dn,result);
}

	/*ID: 1287, INST: maccx.c.2x -Da,Db,Dmn*/
void do__maccx_c_2x_m(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1288, INST: maccx.c.ir.4t Dab,Dcd,Dmn*/
void do__maccx_c_ir_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1289, INST: maccx.c.isr.4w Dab,Dcd,Dmn*/
void do__maccx_c_isr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1290, INST: maccx.c.r.2t Da,Db,Dn*/
Word40 do__maccx_c_r_2t(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1291, INST: maccx.c.r.4t Da,Dcd,Dmn*/
void do__maccx_c_r_4t_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1292, INST: maccx.c.r.4t Dab,Dcd,Dmn*/
void do__maccx_c_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1293, INST: maccx.c.r.4t -Dab,Dcd,Dmn*/
void do__maccx_c_r_4t_m(Word32 __Ra, Word32 __Rb, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1294, INST: maccx.c.s.2x Da,Db,Dmn*/
void do__maccx_c_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1295, INST: maccx.c.sr.2w Da,Db,Dn*/
Word32 do__maccx_c_sr_2w(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1296, INST: maccx.c.sr.4w Da,Dcd,Dmn*/
void do__maccx_c_sr_4w_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1297, INST: maccx.c.sr.4w Dab,Dcd,Dmn*/
void do__maccx_c_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1298, INST: maccx.i.2x Da,Db,Dmn*/
void do__maccx_i_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
  Word40 m1, m2, m3, m4;

  m1 = __l_to_x(__l_get_msb(__Da) * __l_get_msb(__Db));
  m2 = __l_to_x(__l_get_lsb(__Da) * __l_get_lsb(__Db));
  m3 = __l_to_x(__l_get_msb(__Da) * __l_get_lsb(__Db));
  m4 = __l_to_x(__l_get_lsb(__Da) * __l_get_msb(__Db));

  *__Dm = do__add_x(*__Dm, do__sub_x(m2,m1));
  *__Dn = do__add_x(*__Dn, do__add_x(m3,m4));
}

	/*ID: 1299, INST: maccx.ir.4t Dab,Dcd,Dmn*/
void do__maccx_ir_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1300, INST: maccx.is.2x Da,Db,Dmn*/
void do__maccx_is_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1301, INST: maccx.r.2t Da,Db,Dn*/
Word40 do__maccx_r_2t(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1302, INST: maccx.r.4t Da,Dcd,Dmn*/
void do__maccx_r_4t_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1303, INST: maccx.r.4t Dab,Dcd,Dmn*/
void do__maccx_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1304, INST: maccx.r.4t -Dab,Dcd,Dmn*/
void do__maccx_r_4t_m(Word32 __Ra, Word32 __Rb, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1305, INST: maccx.s.2x Da,Db,Dmn*/
void do__maccx_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1306, INST: maccx.sr.2w Da,Db,Dn*/
Word32 do__maccx_sr_2w(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1307, INST: maccx.sr.4w Da,Dcd,Dmn*/
void do__maccx_sr_4w_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1308, INST: maccx.sr.4w Dab,Dcd,Dmn*/
void do__maccx_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1309, INST: maccxd.cpn.2x Da,Dcd,Dmn*/
void do__maccxd_cpn_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1310, INST: maccxd.cpn.2x Dab,Dcd,Dmn*/
void do__maccxd_cpn_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;
	result1 = do__add_x(do__mpy_x_hh(__Da,__Dc),do__mpy_x_ll(__Da,__Dc));
	result2 = do__add_x(do__mpy_x_hh(__Db,__Dd),do__mpy_x_ll(__Db,__Dd));
	*__Dm = do__add_x(*__Dm,do__sub_x(result2,result1));

	result1 = do__sub_x(do__mpy_x_hl(__Dc,__Da),do__mpy_x_hl(__Da,__Dc));
	result2 = do__sub_x(do__mpy_x_hl(__Dd,__Db),do__mpy_x_hl(__Db,__Dd));
	*__Dn = do__add_x(*__Dn ,do__sub_x(result2,result1));
}

	/*ID: 1311, INST: maccxd.cpn.s.2x Da,Dcd,Dmn*/
void do__maccxd_cpn_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1312, INST: maccxd.cpn.s.2x Dab,Dcd,Dmn*/
void do__maccxd_cpn_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1313, INST: maccxd.cpn.sr.2w Dab,Dcd,Dn*/
void do__maccxd_cpn_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1314, INST: maccxd.cpp.2x Da,Dcd,Dmn*/
void do__maccxd_cpp_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1315, INST: maccxd.cpp.2x Dab,Dcd,Dmn*/
void do__maccxd_cpp_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
#if 0
	Word40 __Dm_1 = *__Dm;
	Word40 __Dn_1 = *__Dn;
	do__mpycxd_cpp_2x(__Da, __Db, __Dc, __Dd, &__Dm_1, &__Dn_1);
	*__Dm = do__add_x(__Dm_1, *__Dm);
	*__Dn = do__add_x(__Dn_1, *__Dn);
#else
   Word40 mult1 = do__mpy_x_hh(__Da, __Dc);
   Word40 mult2 = do__mpy_x_ll(__Da, __Dc);
   Word40 mult3 = do__mpy_x_hh(__Db, __Dd);
   Word40 mult4 = do__mpy_x_ll(__Db, __Dd);

   mult1 =  do__add_x(do__add_x(mult3, mult4), do__add_x(mult1, mult2));
   if ((__Da >> 31) == 1) {
      *__Dm = do__sub_x(mult1, *__Dm);
   } else {
     *__Dm = do__add_x(*__Dm, mult1);
   }

   mult1 = do__mpy_x_hl(__Da, __Dc);
   mult2 = do__mpy_x_hl(__Dc, __Da);
   mult3 = do__mpy_x_hl(__Db, __Dd);
   mult4 = do__mpy_x_hl(__Dd, __Db);

   mult1 = do__add_x(do__sub_x(mult4, mult3), do__sub_x(mult2, mult1));
   if ((__Da >> 31) == 1) {
      *__Dn = do__sub_x(mult1, *__Dn);
   } else {
     *__Dn = do__add_x(*__Dn, mult1);
   }

#endif
}

	/*ID: 1316, INST: maccxd.cpp.i.2x Dab,Dcd,Dmn*/
void do__maccxd_cpp_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;
	result1 = do__add_x(do__mpy_i_x_hh(__Da,__Dc),do__mpy_i_x_ll(__Da,__Dc));
	result2 = do__add_x(do__mpy_i_x_hh(__Db,__Dd),do__mpy_i_x_ll(__Db,__Dd));
	*__Dm = do__add_x(*__Dm,do__add_x(result1,result2));

	result1 = do__sub_x(do__mpy_i_x_hl(__Dc,__Da),do__mpy_i_x_hl(__Da,__Dc));
	result2 = do__sub_x(do__mpy_i_x_hl(__Dd,__Db),do__mpy_i_x_hl(__Db,__Dd));
	*__Dn = do__add_x(*__Dn,do__add_x(result1,result2));
}

	/*ID: 1317, INST: maccxd.cpp.ir.2t Dab,Dcd,Dn*/
void do__maccxd_cpp_ir_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1318, INST: maccxd.cpp.is.2x Dab,Dcd,Dmn*/
void do__maccxd_cpp_is_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1319, INST: maccxd.cpp.isr.2w Dab,Dcd,Dn*/
void do__maccxd_cpp_isr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1320, INST: maccxd.cpp.r.2t Dab,Dcd,Dn*/
void do__maccxd_cpp_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1321, INST: maccxd.cpp.s.2x Da,Dcd,Dmn*/
void do__maccxd_cpp_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1322, INST: maccxd.cpp.s.2x Dab,Dcd,Dmn*/
void do__maccxd_cpp_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1323, INST: maccxd.cpp.sr.2w Dab,Dcd,Dn*/
void do__maccxd_cpp_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1324, INST: maccxd.pn.2x Da,Dcd,Dmn*/
void do__maccxd_pn_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1325, INST: maccxd.pn.2x Dab,Dcd,Dmn*/
void do__maccxd_pn_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;
	result1 = do__sub_x(do__mpy_x_ll(__Da,__Dc),do__mpy_x_hh(__Da,__Dc));
	result2 = do__sub_x(do__mpy_x_ll(__Db,__Dd),do__mpy_x_hh(__Db,__Dd));
	*__Dm = do__add_x(do__sub_x(result2,result1),*__Dm);

	result1 = do__add_x(do__mpy_x_hl(__Da,__Dc),do__mpy_x_hl(__Dc,__Da));
	result2 = do__add_x(do__mpy_x_hl(__Db,__Dd),do__mpy_x_hl(__Dd,__Db));
	*__Dn = do__add_x(do__sub_x(result2,result1),*__Dn);
}

	/*ID: 1326, INST: maccxd.pn.i.2x Dab,Dcd,Dmn*/
void do__maccxd_pn_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1327, INST: maccxd.pn.is.2x Dab,Dcd,Dmn*/
void do__maccxd_pn_is_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1328, INST: maccxd.pn.isr.2w Dab,Dcd,Dn*/
void do__maccxd_pn_isr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1329, INST: maccxd.pn.s.2x Da,Dcd,Dmn*/
void do__maccxd_pn_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1330, INST: maccxd.pn.s.2x Dab,Dcd,Dmn*/
void do__maccxd_pn_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1331, INST: maccxd.pn.sr.2w Dab,Dcd,Dn*/
void do__maccxd_pn_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1332, INST: maccxd.pp.2x Da,Dcd,Dmn*/
void do__maccxd_pp_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1333, INST: maccxd.pp.2x Dab,Dcd,Dmn*/
void do__maccxd_pp_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
   Word40 mult1 = do__mpy_x_hh(__Da,__Dc);
   Word40 mult2 = do__mpy_x_ll(__Da,__Dc);
   Word40 mult3 = do__mpy_x_hh(__Db,__Dd);
   Word40 mult4 = do__mpy_x_ll(__Db,__Dd);

   mult1 = do__add_x(do__sub_x(mult2,mult1), do__sub_x(mult4,mult3));
   if ((__Da >> 31) == 1) {
      *__Dm = do__sub_x(mult1, *__Dm);
   } else {
     *__Dm = do__add_x(*__Dm, mult1);
   }

   mult1 = do__mpy_x_hl(__Da,__Dc);
   mult2 = do__mpy_x_hl(__Dc,__Da);
   mult3 = do__mpy_x_hl(__Db,__Dd);
   mult4 = do__mpy_x_hl(__Dd,__Db);

   mult1 = do__add_x(do__add_x(mult1, mult2), do__add_x(mult3, mult4));
   if ((__Da >> 31) == 1) {
      *__Dn = do__sub_x(mult1, *__Dn);
   } else {
     *__Dn = do__add_x(*__Dn, mult1);
   }
}

	/*ID: 1334, INST: maccxd.pp.i.2x Dab,Dcd,Dmn*/
void do__maccxd_pp_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1335, INST: maccxd.pp.ir.2t Dab,Dcd,Dn*/
void do__maccxd_pp_ir_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1336, INST: maccxd.pp.is.2x Dab,Dcd,Dmn*/
void do__maccxd_pp_is_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1337, INST: maccxd.pp.isr.2w Dab,Dcd,Dn*/
void do__maccxd_pp_isr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1338, INST: maccxd.pp.r.2t Dab,Dcd,Dn*/
void do__maccxd_pp_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1339, INST: maccxd.pp.s.2x Da,Dcd,Dmn*/
void do__maccxd_pp_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1340, INST: maccxd.pp.s.2x Dab,Dcd,Dmn*/
void do__maccxd_pp_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1341, INST: maccxd.pp.sr.2w Dab,Dcd,Dn*/
void do__maccxd_pp_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1342, INST: maccxd.ppx.2x Dab,Dcd,Dmn*/
void do__maccxd_ppx_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1343, INST: maccxd.ppx.s.2x Dab,Dcd,Dmn*/
void do__maccxd_ppx_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1344, INST: maccxd.ppx.sr.2w Dab,Dcd,Dn*/
void do__maccxd_ppx_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1345, INST: maccxm.c.r.2x Da,Dcd,Dmn*/
void do__maccxm_c_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1346, INST: maccxm.c.r.2x -Da,Dcd,Dmn*/
void do__maccxm_c_r_2x_m(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1347, INST: maccxm.c.sr.2x Da,Dcd,Dmn*/
void do__maccxm_c_sr_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1348, INST: maccxm.c.sr.2x -Da,Dcd,Dmn*/
void do__maccxm_c_sr_2x_m(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1349, INST: maccxm.r.2x Da,Dcd,Dmn*/
void do__maccxm_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1350, INST: maccxm.r.2x -Da,Dcd,Dmn*/
void do__maccxm_r_2x_m(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1351, INST: maccxm.sr.2x Da,Dcd,Dmn*/
void do__maccxm_sr_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1352, INST: maccxm.sr.2x -Da,Dcd,Dmn*/
void do__maccxm_sr_2x_m(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1353, INST: macd.x Da,Db,Dn*/
Word40 do__macd_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1354, INST: macd.2x Da,Dcd,Dmn*/
void do__macd_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1355, INST: macd.2x Dab,Dcd,Dmn*/
void do__macd_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word40 mult1 =  do__mpy_x_ll(__Da, __Dc);
    Word40 mult2 =  do__mpy_x_hh(__Da, __Dc);

   mult1 = do__add_x(mult1, mult2);
   if ((__Da >> 31) == 1) {
       *__Dm = do__sub_x(mult1, *__Dm);
   } else {
       *__Dm = do__add_x(mult1, *__Dm);
   }

   mult1 =  do__mpy_x_ll(__Db, __Dd);
   mult2 =  do__mpy_x_hh(__Db, __Dd);
   mult1 = do__add_x(mult1, mult2);
   if ((__Da >> 31) == 1) {
	   *__Dn = do__sub_x(mult1, *__Dn);

   } else {
	   *__Dn = do__add_x(mult1, *__Dn);

   }
}

	/*ID: 1356, INST: macd.2x -Dab,Dcd,Dmn*/
void do__macd_2x_m(Word32 __Ra, Word32 __Rb, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1357, INST: macd.cim.x Da,Db,Dn*/
Word40 do__macd_cim_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1358, INST: macd.cim.2x Dab,Dcd,Dmn*/
void do__macd_cim_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;

	result1 = do__sub_x(do__mpy_x_hl(__Da,__Dc),do__mpy_x_hl(__Dc,__Da));
	*__Dm = do__sub_x(result1,*__Dm);

	result2 = do__sub_x(do__mpy_x_hl(__Db,__Dd),do__mpy_x_hl(__Dd,__Db));
	*__Dn = do__sub_x(result2,*__Dn);
}

	/*ID: 1359, INST: macd.cim.i.2x Dab,Dcd,Dmn*/
void do__macd_cim_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;

	result1 = do__sub_x(do__mpy_i_x_hl(__Da,__Dc),do__mpy_i_x_hl(__Dc,__Da));
	*__Dm = do__sub_x(result1,*__Dm);

	result2 = do__sub_x(do__mpy_i_x_hl(__Db,__Dd),do__mpy_i_x_hl(__Dd,__Db));
	*__Dn = do__sub_x(result2,*__Dn);
}

	/*ID: 1360, INST: macd.cim.leg.x -Da,Db,Dn*/

	/*ID: 1361, INST: macd.cim.r.2t Dab,Dcd,Dn*/
void do__macd_cim_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1362, INST: macd.cim.s.x Da,Db,Dn*/
Word40 do__macd_cim_s_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1363, INST: macd.cim.s.2x Dab,Dcd,Dmn*/
void do__macd_cim_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1364, INST: macd.cim.sr.2w Dab,Dcd,Dn*/
void do__macd_cim_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1365, INST: macd.i.2x Dab,Dcd,Dmn*/
void do__macd_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1366, INST: macd.im.x Da,Db,Dn*/
Word40 do__macd_im_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1367, INST: macd.im.2x Dab,Dcd,Dmn*/
void do__macd_im_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word40 mult1 =  do__mpy_x_hl(__Dc, __Da);
    Word40 mult2 =  do__mpy_x_hl(__Da, __Dc);

   if ((__Da >> 31) == 1) {
     *__Dm = do__sub_x(do__add_x(mult1, mult2), *__Dm);/*(Word40) DSrcAcc1 - ((mult1 + mult2) << 1); */
   } else {
     *__Dm = do__add_x(do__add_x(mult1, mult2), *__Dm);/* (Word40) DSrcAcc1 + ((mult1 + mult2) << 1);*/
   }
   mult1 =  do__mpy_x_hl(__Dd, __Db);/*__l_get_lsb(__Db) *  __l_get_msb(__Dd); */
   mult2 =  do__mpy_x_hl(__Db, __Dd);/*__l_get_msb(__Db) *  __l_get_lsb(__Dd);*/
   if ((__Da >> 31) == 1) {
     *__Dn = do__sub_x(do__add_x(mult1, mult2), *__Dn);/* (Word40) DSrcAcc2 - ((mult1 + mult2) << 1);*/
   } else {
     *__Dn = do__add_x(do__add_x(mult1, mult2), *__Dn);/*(Word40) DSrcAcc2 + ((mult1 + mult2) << 1); */
   }
}

	/*ID: 1368, INST: macd.im.2x -Dab,Dcd,Dmn*/
void do__macd_im_2x_m(Word32 __Ra, Word32 __Rb, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1369, INST: macd.im.i.2x Dab,Dcd,Dmn*/
void do__macd_im_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1370, INST: macd.im.leg.x Da,Db,Dn*/

	/*ID: 1371, INST: macd.im.r.2t Dab,Dcd,Dn*/
void do__macd_im_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1372, INST: macd.im.s.x Da,Db,Dn*/
Word40 do__macd_im_s_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1373, INST: macd.im.s.2x Dab,Dcd,Dmn*/
void do__macd_im_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1374, INST: macd.im.sr.2w Dab,Dcd,Dn*/
void do__macd_im_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1375, INST: macd.leg.x Da,Db,Dn*/

	/*ID: 1376, INST: macd.leg.x -Da,Db,Dn*/

	/*ID: 1377, INST: macd.r.2t Dab,Dcd,Dn*/
void do__macd_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1378, INST: macd.re.x Da,Db,Dn*/
Word40 do__macd_re_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1379, INST: macd.re.2x Dab,Dcd,Dmn*/
void do__macd_re_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1380, INST: macd.re.2x -Dab,Dcd,Dmn*/
void do__macd_re_2x_m(Word32 __Ra, Word32 __Rb, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1381, INST: macd.re.i.2x Dab,Dcd,Dmn*/
void do__macd_re_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
 Word40 mult1 = __l_to_x(__l_get_msb(__Da) * __l_get_msb(__Dc));
 Word40 mult2 = __l_to_x(__l_get_lsb(__Da) * __l_get_lsb(__Dc));
 Word40 tmp;
   if ((__Da>>31) == 1) {
       *__Dm = do__sub_x(do__sub_x(mult2,mult1), *__Dm);
   } else {
     *__Dm = do__add_x(*__Dm, do__sub_x(mult2,mult1));/* (Word40) DSrcAcc1 + (mult1 - mult2);*/
   }
   mult1 = __l_to_x(__l_get_msb(__Db) * __l_get_msb(__Dd));
   mult2 = __l_to_x(__l_get_lsb(__Db) * __l_get_lsb(__Dd));
   if ((__Da>>31) == 1) {
       *__Dn= do__sub_x(do__sub_x(mult2,mult1), *__Dn);
   } else {
     *__Dn = do__add_x(*__Dn,  do__sub_x(mult2,mult1)); /* (Word40) DSrcAcc2 + (mult1 - mult2);*/
   }
}

	/*ID: 1382, INST: macd.re.leg.x Da,Db,Dn*/

	/*ID: 1383, INST: macd.re.r.2t Dab,Dcd,Dn*/
void do__macd_re_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1384, INST: macd.re.s.x Da,Db,Dn*/
Word40 do__macd_re_s_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1385, INST: macd.re.s.2x Dab,Dcd,Dmn*/
void do__macd_re_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1386, INST: macd.re.sr.2w Dab,Dcd,Dn*/
void do__macd_re_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1387, INST: macd.s.x Da,Db,Dn*/
Word40 do__macd_s_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1388, INST: macd.s.2x Da,Dcd,Dmn*/
void do__macd_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1389, INST: macd.s.2x Dab,Dcd,Dmn*/
void do__macd_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1390, INST: macd.sr.2w Dab,Dcd,Dn*/
void do__macd_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1391, INST: macdcf.2x Da,Dcd,Dmn*/
void do__macdcf_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 m1, m2, m3, m4;
	m1 = do__mpy_x_hl(__Dc, __Da);
	m2 = do__mpy_x_hh(__Da, __Dd);
	*__Dm = do__add_x(*__Dm, do__add_x(m1, m2));

	m3 = do__mpy_x_ll(__Da, __Dc);
	m4 = do__mpy_x_hl(__Da, __Dd);
	*__Dn = do__add_x(*__Dn, do__add_x(m3, m4));
}

	/*ID: 1392, INST: macdcf.h.r.4t Dab,Dcd,Dmn*/
void do__macdcf_h_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 m1, m2;
	Word32 Dm_WH, Dm_WL, Dn_WH, Dn_WL;
	m1 = do__mpy_x_hl(__Da, __Dc);
	m2 = do__mpy_x_hh(__Db, __Dc);
	*__Dm = __round_and_add_2t(m1, m2, *__Dm, 1);

	m1 = do__mpy_x_ll(__Da, __Dc);
	m2 = do__mpy_x_hl(__Dc, __Db);
	*__Dm = __round_and_add_2t(m1, m2, *__Dm, 0);

	m1 = do__mpy_x_hh(__Da, __Dd);
	m2 = do__mpy_x_hl(__Db, __Dc);
	*__Dn = __round_and_add_2t(m1, m2, *__Dn, 1);

	m1 = do__mpy_x_hl(__Dd, __Da);
	m2 = do__mpy_x_ll(__Db, __Dc);
	*__Dn = __round_and_add_2t(m1, m2, *__Dn, 0);
}

	/*ID: 1393, INST: macdcf.h.sr.4w Dab,Dcd,Dmn*/
void do__macdcf_h_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
	Word40 m1, m2,m;
	Word16 Dm_H, Dm_L, Dn_H, Dn_L;
	m1 = do__mpy_x_hl(__Da, __Dc);
	m2 = do__mpy_x_hh(__Db, __Dc);
	*__Dm = __round_and_add_sat_2w(m1, m2, *__Dm, 1);

	m1 = do__mpy_x_ll(__Da, __Dc);
	m2 = do__mpy_x_hl(__Dc, __Db);
	*__Dm = __round_and_add_sat_2w(m1, m2, *__Dm, 0);

	m1 = do__mpy_x_hh(__Da, __Dd);
	m2 = do__mpy_x_hl(__Db, __Dc);
	*__Dn = __round_and_add_sat_2w(m1, m2, *__Dn, 1);

	m1 = do__mpy_x_hl(__Dd, __Da);
	m2 = do__mpy_x_ll(__Db, __Dc);
	*__Dn = __round_and_add_sat_2w(m1, m2, *__Dn, 0);

}

	/*ID: 1394, INST: macdcf.inv.2x Da,Dcd,Dmn*/
void do__macdcf_inv_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result;
	result = do__add_x(do__mpy_x_hl(__Dd,__Da),do__mpy_x_hh(__Da,__Dc));
	*__Dm = do__add_x(*__Dm,result);

	result = do__add_x(do__mpy_x_ll(__Da,__Dd),do__mpy_x_hl(__Da,__Dc));
	*__Dn = do__add_x(*__Dn,result);
}

	/*ID: 1395, INST: macdcf.invh.r.4t Dab,Dcd,Dmn*/
void do__macdcf_invh_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 m,m1, m2;

	m1 = do__mpy_x_hh(__Da, __Dc);
	m2 = do__mpy_x_hl(__Db, __Dc);
	*__Dm = __round_and_add_2t(m1,m2,*__Dm,1);

	m1 = do__mpy_x_hl(__Dc,__Da);
	m2 = do__mpy_x_ll(__Db,__Dc);
	*__Dm = __round_and_add_2t(m1,m2,*__Dm,0);

	m1 = do__mpy_x_hl(__Da, __Dc);
	m2 = do__mpy_x_hh(__Db, __Dd);
	*__Dn = __round_and_add_2t(m1,m2,*__Dn,1);

	m1 = do__mpy_x_ll(__Da, __Dc);
	m2 = do__mpy_x_hl(__Dd, __Db);
	*__Dn = __round_and_add_2t(m1,m2,*__Dn,0);
}

	/*ID: 1396, INST: macdcf.invl.r.4t Dab,Dcd,Dmn*/
void do__macdcf_invl_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 m,m1, m2;
	Word32 Dm_WH, Dm_WL, Dn_WH, Dn_WL;

	m1 = do__mpy_x_hl(__Da, __Dc);
	m2 = do__mpy_x_hh(__Db, __Dd);
	*__Dm = __round_and_add_2t(m1,m2,*__Dm,1);

	m1 = do__mpy_x_ll(__Da,__Dc);
	m2 = do__mpy_x_hl(__Dd,__Db);
	*__Dm = __round_and_add_2t(m1,m2,*__Dm,0);

	m1 = do__mpy_x_hh(__Da, __Dd);
	m2 = do__mpy_x_hl(__Db, __Dd);
	*__Dn = __round_and_add_2t(m1,m2,*__Dn,1);

	m1 = do__mpy_x_hl(__Dd, __Da);
	m2 = do__mpy_x_ll(__Db, __Dd);
	*__Dn = __round_and_add_2t(m1,m2,*__Dn,0);
}

	/*ID: 1397, INST: macdcf.l.r.4t Dab,Dcd,Dmn*/
void do__macdcf_l_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 m1, m2;
	Word32 Dm_WH, Dm_WL, Dn_WH, Dn_WL;
	m1 = do__mpy_x_hh(__Da, __Dd);
	m2 = do__mpy_x_hl(__Db, __Dc);
	*__Dm = __round_and_add_2t(m1, m2, *__Dm, 1);

	m1 = do__mpy_x_hl(__Dd, __Da);
	m2 = do__mpy_x_ll(__Db, __Dc);
	*__Dm = __round_and_add_2t(m1, m2, *__Dm, 0);

	m1 = do__mpy_x_hl(__Da, __Dd);
	m2 = do__mpy_x_hh(__Db, __Dd);
	*__Dn = __round_and_add_2t(m1, m2, *__Dn, 1);

	m1 = do__mpy_x_ll(__Dd, __Da);
	m2 = do__mpy_x_hl(__Dd, __Db);
	*__Dn = __round_and_add_2t(m1, m2, *__Dn, 0);
}

	/*ID: 1398, INST: macdcf.l.sr.4w Dab,Dcd,Dmn*/
void do__macdcf_l_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
	Word40 m1, m2;
	Word16 Dm_H, Dm_L, Dn_H, Dn_L;
	m1 = do__mpy_x_hh(__Da, __Dd);
	m2 = do__mpy_x_hl(__Db, __Dc);
	*__Dm = __round_and_add_sat_2w(m1, m2, *__Dm, 1);

	m1 = do__mpy_x_hl(__Dd, __Da);
	m2 = do__mpy_x_ll(__Db, __Dc);
	*__Dm = __round_and_add_sat_2w(m1, m2, *__Dm, 0);

	m1 = do__mpy_x_hl(__Da, __Dd);
	m2 = do__mpy_x_hh(__Db, __Dd);
	*__Dn = __round_and_add_sat_2w(m1, m2, *__Dn, 1);

	m1 = do__mpy_x_ll(__Dd, __Da);
	m2 = do__mpy_x_hl(__Dd, __Db);
	*__Dn = __round_and_add_sat_2w(m1, m2, *__Dn, 0);
}

	/*ID: 1399, INST: macdrf.h.2x Da,Dcd,Dmn*/
void do__macdrf_h_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1400, INST: macdrf.h.r.2t Da,Dcd,Dn*/
void do__macdrf_h_r_2t(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1401, INST: macdrf.h.r.4t Dab,Dcd,Dmn*/
void do__macdrf_h_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1402, INST: macdrf.h.s.2x Da,Dcd,Dmn*/
void do__macdrf_h_s_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1403, INST: macdrf.h.sr.2w Da,Dcd,Dn*/
void do__macdrf_h_sr_2w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1404, INST: macdrf.h.sr.4w Dab,Dcd,Dmn*/
void do__macdrf_h_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1405, INST: macdrf.invh.2x Da,Dcd,Dmn*/
void do__macdrf_invh_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1406, INST: macdrf.invh.r.4t Dab,Dcd,Dmn*/
void do__macdrf_invh_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1407, INST: macdrf.invl.2x Da,Dcd,Dmn*/
void do__macdrf_invl_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1408, INST: macdrf.invl.r.4t Dab,Dcd,Dmn*/
void do__macdrf_invl_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1409, INST: macdrf.l.2x Da,Dcd,Dmn*/
void do__macdrf_l_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1410, INST: macdrf.l.r.2t Da,Dcd,Dn*/
void do__macdrf_l_r_2t(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
}

	/*ID: 1411, INST: macdrf.l.r.4t Dab,Dcd,Dmn*/
void do__macdrf_l_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1412, INST: macdrf.l.s.2x Da,Dcd,Dmn*/
void do__macdrf_l_s_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1413, INST: macdrf.l.sr.2w Da,Dcd,Dn*/
void do__macdrf_l_sr_2w(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1414, INST: macdrf.l.sr.4w Dab,Dcd,Dmn*/
void do__macdrf_l_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1415, INST: macem32.ss.x Da.h,Db.h,Dn*/
Word40 do__macem32_ss_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
	Word40 result;
	long long resultLL = 0LL;
	long long var1LL = __x_guard(__Dn);

	var1LL = (var1LL << 32) | (UWord32) __x_to_l(__Dn);

	resultLL = var1LL >>16;
	__Dn  = __x_set((resultLL >> 32) & 0xff,(UWord32) resultLL & 0xffffffff);

	result = do__add_x(__Dn,do__mpy_x_hh(__Da_h,__Db_h));
	return result;
}

	/*ID: 1416, INST: macem32.su.x Da.h,Db.l,Dn*/
Word40 do__macem32_su_x_hl(Word40 __Dn, Word32 __Da_h, Word32 __Db_l)
{
	Word40 result,mul40;
	Word32  mul;
	long long mul64;
	long long resultLL = 0LL;
	long long var1LL = __x_guard(__Dn);

	var1LL = (var1LL << 32) | (UWord32) __x_to_l(__Dn);

	resultLL = var1LL >>16;
	__Dn  = __x_set((resultLL >> 32) & 0xff,(UWord32) resultLL & 0xffffffff);

	mul64 = (long long)((__l_get_msb(__Da_h)*(unsigned short)(__l_get_lsb(__Db_l)))<<1);
	if((__l_get_msb(__Da_h)&0x8000) && (__Db_l & 0xFFFF))
		mul40 = __x_set(0xff,(UWord32) mul64 & 0xffffffff);
	else
		mul40 = __x_set(0,(UWord32) mul64 & 0xffffffff);

	result = do__add_x(__Dn,mul40);
	return result;
}

	/*ID: 1417, INST: macem32.us.x Da.l,Db.h,Dn*/
Word40 do__macem32_us_x_lh(Word40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1418, INST: macem32.us.i.x Da.l,Db.h,Dn*/
Word40 do__macem32_us_i_x_lh(Word40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1419, INST: macem32.uu.x Da.l,Db.l,Dn*/
UWord40 do__macem32_uu_x_ll(UWord40 __Dn, Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1420, INST: macem32.uu.i.x Da.l,Db.h,Dn*/
UWord40 do__macem32_uu_i_x_lh(UWord40 __Dn, Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1421, INST: macem32.xsu.x Da,Db,Dn*/
Word40 do__macem32_xsu_x(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1422, INST: macem32.xsu.x -Da,Db,Dn*/
Word40 do__macem32_xsu_x_m(Word40 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1423, INST: macm.x Da.h,Db,Dn*/
/*
  Function Name: do__macm_x_h
  Purpose: Mixed Precision Signed Fractional Multiply-Accumulate.
  Complexity Weight:
  Inputs: __Da_h and __Db are 32 bit signed integers whose value falls in the range 0x8000 0000 - 0x7FFF FFFF, __Dn a 40bit
  			Signed integer whose value falls in range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 40bit signed value
  Return Value: It returns a 40 bit word.
*/
Word40 do__macm_x_h(Word40 __Dn, Word32 __Da_h, Word32 __Db)
{
	unsigned long long tmp = 0x0LL;
	unsigned long long mul64 = 0x0LL;

	mul64 = ((unsigned long long)((__l_get_msb(__Da_h) )) * __Db) << 1;
	tmp = (((unsigned long long)((UWord32)__x_to_l(__Dn)) | ((unsigned long long)(__x_guard(__Dn)) << 32))) + (mul64 >> 16);
	__Dn = __x_set((tmp >> 32) & 0xff,(UWord32) tmp & 0xffffffff);
	return __Dn;
}

/*ID: 1424, INST: macm.r.x Da.h,Db,Dn */
/*
  Function Name: do__macm_r_x_h
  Purpose: Mixed Precision Signed Fractional Multiply-Accumulate
  Complexity Weight:
  Inputs: __Da_h, __Db are 32 bit signed integers whose value falls in the range 0x8000 0000 - 0x7FFF FFFF, __Dn is a 40bit
  			Signed integer whose value falls in range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: __Dn, 40bit signed value
  Return Value: __Dn
*/

	/*ID: 1424, INST: macm.r.x Da.h,Db,Dn*/
Word40 do__macm_r_x_h(Word40 __Dn, Word32 __Da_h, Word32 __Db)
{
	unsigned long long mul64, tmp;
	mul64 = (((unsigned long long)(__l_get_msb(__Da_h)) * __Db) << 1) + 0x8000;
	tmp = (((unsigned long long)((UWord32)__x_to_l(__Dn)) | ((unsigned long long)(__x_guard(__Dn)) << 32))) + (mul64 >> 16);
	__Dn = __x_set((tmp >> 32) & 0xff,(UWord32) tmp & 0xffffffff);
	return __Dn;
}

	/*ID: 1425, INST: macm.r.x Da.l,Db,Dn*/
Word40 do__macm_r_x_l(Word40 __Dn, Word32 __Da_l, Word32 __Db)
{
	unsigned long long mul64, tmp;
	mul64 = (((unsigned long long)(__l_get_lsb(__Da_l)) * __Db) << 1) + 0x8000;
	tmp = (((unsigned long long)((UWord32)__x_to_l(__Dn)) | ((unsigned long long)(__x_guard(__Dn)) << 32))) + (mul64 >> 16);
	__Dn = __x_set((tmp >> 32) & 0xff,(UWord32) tmp & 0xffffffff);
	return __Dn;

}

	/*ID: 1426, INST: macm.r.x -Da.h,Db,Dn*/
Word40 do__macm_r_x_m_h(Word40 __Dn, Word32 __Da_h, Word32 __Db)
{
	unsigned long long mul64, tmp;
	mul64 = (((unsigned long long)(__l_get_msb(__Da_h)) * __Db) << 1);
	mul64 = (0-mul64)+0x8000;
	tmp = (((unsigned long long)((UWord32)__x_to_l(__Dn)) | ((unsigned long long)(__x_guard(__Dn)) << 32))) + (mul64 >> 16);
	__Dn = __x_set((tmp >> 32) & 0xff,(UWord32) tmp & 0xffffffff);
	return __Dn;
}

/*
  Function Name: do__macm_r_2x
  Purpose: Mixed Precision signed fractional Multiply-Accumulate SIMD(2X40)
  Complexity Weight:
  Inputs: __Da, __Dc and __Dd are 32 bit signed integers whose value falls in the range 0x8000 0000 - 0x7FFF FFFF, __Dn and __Dm are 40bit
  			Signed integer whose value falls in range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: __Dm and __Dn, 40bit signed value
  Return Value: Register pair (fraction)__Dm and __Dn
*/

	/*ID: 1427, INST: macm.r.2x Da,Dcd,Dmn*/
void do__macm_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__macm_r_x_h(*__Dm, __Da, __Dc);
	*__Dn = do__macm_r_x_l(*__Dn, __Da, __Dd);
}

	/*ID: 1428, INST: macm.r.2x Da.h,Dcd,Dmn*/
void do__macm_r_2x_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__macm_r_x_h(*__Dm, __Da_h, __Dc);
	*__Dn = do__macm_r_x_h(*__Dn, __Da_h, __Dd);
}

	/*ID: 1429, INST: macm.r.2x Da.l,Dcd,Dmn*/
void do__macm_r_2x_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__macm_r_x_l(*__Dm, __Da_l, __Dc);
	*__Dn = do__macm_r_x_l(*__Dn, __Da_l, __Dd);
}

	/*ID: 1430, INST: macm.sr.x Da.h,Db,Dn*/
Word40 do__macm_sr_x_h(Word40 __Dn, Word32 __Da_h, Word32 __Db)
{
	Word32 __Dn_wh;
	__Dn_wh = do__sat_x_l(do__macm_r_x_h(__Dn, __Da_h, __Db));
	__Dn = __x_set(((__Dn_wh & 0x80000000) ? 0xff : 0), __Dn_wh);
	return __Dn;
}

	/*ID: 1431, INST: macm.sr.x Da.l,Db,Dn*/
Word40 do__macm_sr_x_l(Word40 __Dn, Word32 __Da_l, Word32 __Db)
{
	Word32 __Dn_wh;
	__Dn_wh = do__sat_x_l(do__macm_r_x_l(__Dn, __Da_l, __Db));
	__Dn = __x_set(((__Dn_wh & 0x80000000) ? 0xff : 0), __Dn_wh);
	return __Dn;
}

	/*ID: 1432, INST: macm.sr.2x Da,Dcd,Dmn*/
void do__macm_sr_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__macm_sr_x_h(*__Dm, __Da, __Dc);
	*__Dn = do__macm_sr_x_l(*__Dn, __Da, __Dd);
}

	/*ID: 1433, INST: macm.sr.2x Da.h,Dcd,Dmn*/
void do__macm_sr_2x_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__macm_sr_x_h(*__Dm, __Da_h, __Dc);
	*__Dn = do__macm_sr_x_h(*__Dn, __Da_h, __Dd);
}

	/*ID: 1434, INST: macm.sr.2x Da.l,Dcd,Dmn*/
void do__macm_sr_2x_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__macm_sr_x_l(*__Dm, __Da_l, __Dc);
	*__Dn = do__macm_sr_x_l(*__Dn, __Da_l, __Dd);
}

	/*ID: 1435, INST: macm.su.x Da.h,Db,Dn*/
Word40 do__macm_su_x_h(Word40 __Dn, Word32 __Da_h, Word32 __Db)
{
	unsigned long long mul64, tmp;
	mul64 = (((unsigned long long)(__l_get_msb(__Da_h)) * (UWord32)__Db) << 1);
	tmp = (((unsigned long long)((UWord32)__x_to_l(__Dn)) | ((unsigned long long)(__x_guard(__Dn)) << 32))) + (mul64 >> 16);
	__Dn = __x_set((tmp >> 32) & 0xff,(UWord32) tmp & 0xffffffff);
	return __Dn;
}

	/*ID: 1436, INST: macm.su.log2.x Da.h,Db,Dn*/
Word40 do__macm_su_log2_x_h(Word40 __Dn, Word32 __Da_h, Word32 __Db)
{
	unsigned long long mul64, tmp;
	mul64 = (((unsigned long long)(__l_get_msb(__Da_h)) * (UWord32)__Db) << 2);
	tmp = (((unsigned long long)((UWord32)__x_to_l(__Dn)) | ((unsigned long long)(__x_guard(__Dn)) << 32))) + (mul64 >> 16);
	__Dn = __x_set((tmp >> 32) & 0xff,(UWord32) tmp & 0xffffffff);
	return __Dn;
}

	/*ID: 1437, INST: macm.us.il.l Da.l,Db,Dn*/
Word32 do__macm_us_il_l_l(Word32 __Dn, Word32 __Da_l, Word32 __Db)
{
	unsigned long long mul64, tmp;
	mul64 = ((unsigned long long)(((Word32)__l_get_lsb(__Da_l)) & 0xffff) * (long long)__Db);
	tmp = (unsigned long long)(__Dn) + (long long)(mul64 & 0xffffffff);
	__Dn = (Word32)(tmp & 0xffffffff);
	return __Dn;
}


	/*ID: 1438, INST: macq.x Dab,Dcd,Dn*/
/*
  Function Name: do__macq_x
  Purpose: Fractional Quad Multiply-Accumulate to 1 register
  Complexity Weight:
  Inputs: __Da, __Db,__Dc and __Dd are 32 bit signed integers whose value falls in the range 0x8000 0000 - 0x7FFF FFFF, __Dn is 40bit
  			Signed integer whose value falls in range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: __Dn, 40bit signed value
  Return Value:
*/
void do__macq_x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
	Word40 m1, m2, m3, m4, result;

	m1 = do__mpy_x_hh(__Da, __Dc);
	m2 =do__mpy_x_ll(__Da ,__Dc);

	m3 = do__mpy_x_hh(__Db, __Dd);
	m4 = do__mpy_x_ll(__Db, __Dd);
	result = (do__add_x(do__add_x(m3, m4), do__add_x(m1, m2)));

	*__Dn = do__add_x(*__Dn, result);
}

	/*ID: 1439, INST: macq.i.x Dab,Dcd,Dn*/
/*
  Function Name: do__macq_i_x
  Purpose: Integer Quad Multiply-Accumulate to 1 register
  Complexity Weight:
  Inputs: __Da, __Db,__Dc and __Dd are 32 bit signed integers whose value falls in the range 0x8000 0000 - 0x7FFF FFFF, __Dn is 40bit
  			Signed integer whose value falls in range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: __Dn, 40bit signed value
  Return Value:
*/
void do__macq_i_x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dn)
{
	Word40 m1, m2, m3, m4, result;

	m1 = do__mpy_i_x_hh(__Da, __Dc);
	m2 =do__mpy_i_x_ll(__Da ,__Dc);

	m3 = do__mpy_i_x_hh(__Db, __Dd);
	m4 = do__mpy_i_x_ll(__Db, __Dd);
	result = (do__add_x(do__add_x(m3, m4), do__add_x(m1, m2)));

	*__Dn = do__add_x(*__Dn, result);
}

	/*ID: 1440, INST: macshl.s.x Da.h,Db.h,Dn*/
Word40 do__macshl_s_x_hh(Word40 __Dn, Word32 __Da_h, Word32 __Db_h)
{
}

	/*ID: 1441, INST: masksel.x Da,Db,Dn*/
Word40 do__masksel_x(Word40 __Dn, Word40 __Da, Word40 __Db)
{
}

	/*ID: 1442, INST: masksel.2x Dab,Dcd,Dmn*/
void do__masksel_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1443, INST: max.2t Da,Db,Dn*/
Word40 do__max_2t(Word40 __Da, Word40 __Db)
{
	Word32 Da_WH,Da_WL,Db_WH,Db_WL,Dn_WH,Dn_WL;
	Word40 Dn;

	Da_WH = do__extract_wh(__Da);
	Da_WL = do__extract_wl(__Da);
	Db_WH = do__extract_wh(__Db);
	Db_WL = do__extract_wl(__Db);

	Dn_WH = (Da_WH > Db_WH)?Da_WH:Db_WH;
	Dn_WL = (Da_WL > Db_WL)?Da_WL:Db_WL;

	do__set_WH(Dn_WH,&Dn);
	do__set_WL(Dn_WL,&Dn);

	return Dn;
}

	/*ID: 1444, INST: max.4t Dab,Dcd,Dmn*/
void do__max_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__max_2t(__Da,__Dc);
	*__Dn = do__max_2t(__Db,__Dd);
}

	/*ID: 1445, INST: max.2w Da,Db,Dn*/
Word32 do__max_2w(Word32 __Da, Word32 __Db)
{
	Word32 __Dn_1, __Dn_2, __Dn;
	Word16 __Dn_1_H, __Dn_1_L, __Dn_2_H, __Dn_2_L, __Dn_H, __Dn_L ;

#if 0
	__Dn_1 = __x_to_l(__Da);
	__Dn_2 = __x_to_l(__Db);
#else
	__Dn_1 = (__Da);
	__Dn_2 = (__Db);

#endif

	__Dn_1_H = (__Dn_1 >> 16);
	__Dn_1_L = (__Dn_1 & 0xffff);

	__Dn_2_H = (__Dn_2 >> 16);
	__Dn_2_L = (__Dn_2 & 0xffff);

	__Dn_H = ((Word16)(__Dn_1_H) > (Word16)(__Dn_2_H)) ? __Dn_1_H : __Dn_2_H;
    __Dn_L = ((Word16)(__Dn_1_L) > (Word16)(__Dn_2_L)) ? __Dn_1_L : __Dn_2_L;

    __Dn = (((__Dn_H & 0xFFFF) << 16) & 0xFFFF0000) | (__Dn_L & 0xFFFF) ;
#if 0
    return __x_set(0, __Dn);
#else
    return (__Dn);
#endif
}

	/*ID: 1446, INST: max.4w Dab,Dcd,Dmn*/
void do__max_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
	*__Dm = do__max_2w(__Da,__Dc);
	*__Dn = do__max_2w(__Db,__Dd);
}

	/*ID: 1447, INST: max.x Da,Db,Dn*/
/*
  Function Name: do__max_x
  Purpose: Compare two 40 bit words and return the maximum value.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns the maximum 40 bit word..
*/
Word40 do__max_x(Word40 __Da, Word40 __Db)
{
   Word8 __Dn_1_ext, __Dn_2_ext, __Dn_max_ext;
   Word32 __Dn_1, __Dn_2, __Dn_max;
   Word40 __Dn;

   __Dn_1_ext = __x_guard(__Da);
   __Dn_2_ext = __x_guard(__Db);

   __Dn_1 = __x_to_l(__Da);
   __Dn_2 = __x_to_l(__Db);

   if (((__Dn_1_ext & 0x80) == 0x80) && ((__Dn_2_ext & 0x80) == 0x80)) { /* Both are negative */
     if ((__Dn_1_ext & 0x7f) == (__Dn_2_ext & 0x7f)) { /* if both extensions are same compare magnitude, otherwise compare the extensions */
       __Dn = (__Dn_2 > __Dn_1) ? __x_set(__Dn_1_ext, __Dn_1) : __x_set(__Dn_2_ext, __Dn_2);
	 } else {
       __Dn = (__Dn_2_ext > __Dn_1_ext) ? __x_set(__Dn_1_ext, __Dn_1) : __x_set(__Dn_2_ext, __Dn_2);
	 }
   } else if (((__Dn_1_ext & 0x80) == 0x80) && ((__Dn_2_ext & 0x80) == 0x00)) { /* __Db is positive */
     __Dn = __x_set(__Dn_2_ext, __Dn_2);
   } else if (((__Dn_2_ext & 0x80) == 0x80) && ((__Dn_1_ext & 0x80) == 0x00)) { /* __Da is positive */
     __Dn = __x_set(__Dn_1_ext, __Dn_1);
   } else { /* both are positive */
     if (__Dn_1_ext == __Dn_2_ext) {
       __Dn= ((UWord32)(__Dn_1) > (UWord32)(__Dn_2)) ? __x_set(__Dn_1_ext, __Dn_1) : __x_set(__Dn_2_ext, __Dn_2);
	 } else { /* just compare the extension */
	   __Dn = (__Dn_1_ext > __Dn_2_ext) ? __x_set(__Dn_1_ext, __Dn_1):__x_set(__Dn_2_ext, __Dn_2);
	 }
   }
   return __Dn;
}

	/*ID: 1448, INST: max.2x Dab,Dcd,Dmn*/
void do__max_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
   Word40 res;
   res = do__cmpd_gt_x(__Da, __Dc); /* Da > Dc */

   if ((__x_guard(res))== 0 && (__x_to_l(res) == 0)) {
     *__Dm = __x_set(__x_guard(__Da), __x_to_l(__Da));
   } else {
     *__Dm = __x_set(__x_guard(__Dc), __x_to_l(__Dc));
   }

   res = do__cmpd_gt_x(__Db, __Dd); /* Db > Dd */

    if ((__x_guard(res))== 0 && (__x_to_l(res) == 0)) {
      *__Dn = __x_set(__x_guard(__Db), __x_to_l(__Db));
    } else {
      *__Dn = __x_set(__x_guard(__Dd), __x_to_l(__Dd));
    }
}

	/*ID: 1449, INST: maxa Ra,Rb,Rn*/
Word32 do__maxa(Word32 __Ra, Word32 __Rb)
{
	return (__Ra > __Rb)?__Ra:__Rb;
}

	/*ID: 1450, INST: maxa Ra,Rb,Rn*/

	/*ID: 1451, INST: maxm.x Da,Db,Dn*/
/*
  Function Name: do__maxm_x
  Purpose: Compare two 40 bit words and return the maximum absolute value.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs: 0 or -1
  Return Value: It returns the maximum absolute of two input 40 bit words.
*/

Word40 do__maxm_x(Word40 __Da, Word40 __Db)
{
   Word8 __Dn_1_ext, __Dn_2_ext;
   Word40 __Dn_1, __Dn_2;

   __Dn_1 = do__abs_s40_x(__Da);
   __Dn_2 = do__abs_s40_x(__Db);

   __Dn_1_ext = __x_guard(__Da);
   __Dn_2_ext = __x_guard(__Db);


	if(__x_to_l(do__cmpd_gt_x(__Dn_1, __Dn_2)) ==0 )
		return __Da;
	else if(__x_to_l(do__cmpd_gt_x(__Dn_2, __Dn_1)) == 0)
		return __Db;
    else if(((__Dn_1_ext & 0x80) != (__Dn_2_ext & 0x80))){
	    return __Dn_1;
   } else {
	   return __Da;
   }

}

	/*ID: 1452, INST: maxm.2x Dab,Dcd,Dmn*/
void do__maxm_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__maxm_x(__Da, __Dc);
	*__Dn = do__maxm_x(__Db, __Dd);
}

	/*ID: 1453, INST: min.2t Da,Db,Dn*/
Word40 do__min_2t(Word40 __Da, Word40 __Db)
{
	Word32 Da_WH,Da_WL,Db_WH,Db_WL,Dn_WH,Dn_WL;
	Word40 Dn;

	Da_WH = do__extract_wh(__Da);
	Da_WL = do__extract_wl(__Da);
	Db_WH = do__extract_wh(__Db);
	Db_WL = do__extract_wl(__Db);

	Dn_WH = (Da_WH > Db_WH)?Db_WH:Da_WH;
	Dn_WL = (Da_WL > Db_WL)?Db_WL:Da_WL;

	do__set_WH(Dn_WH,&Dn);
	do__set_WL(Dn_WL,&Dn);

	return Dn;
}

	/*ID: 1454, INST: min.4t Dab,Dcd,Dmn*/
void do__min_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__min_2t(__Da,__Dc);
	*__Dn = do__min_2t(__Db,__Dd);
}

	/*ID: 1455, INST: min.2w Da,Db,Dn*/
Word32 do__min_2w(Word32 __Da, Word32 __Db)
{
	Word32 __Dn_1, __Dn_2, __Dn;
	Word16 __Dn_1_H, __Dn_1_L, __Dn_2_H, __Dn_2_L, __Dn_H, __Dn_L ;

#if 0
	__Dn_1 = __x_to_l(__Da);
	__Dn_2 = __x_to_l(__Db);
#else
	__Dn_1 = (__Da);
	__Dn_2 = (__Db);
#endif

	__Dn_1_H = (__Dn_1 >> 16);
	__Dn_1_L = (__Dn_1 & 0xffff);

	__Dn_2_H = (__Dn_2 >> 16);
	__Dn_2_L = (__Dn_2 & 0xffff);

	__Dn_H = ((Word16)(__Dn_1_H) < (Word16)(__Dn_2_H)) ? __Dn_1_H : __Dn_2_H;
    __Dn_L = ((Word16)(__Dn_1_L) < (Word16)(__Dn_2_L)) ? __Dn_1_L : __Dn_2_L;

    __Dn = (((__Dn_H & 0xFFFF) << 16) & 0xFFFF0000) | (__Dn_L & 0xFFFF) ;
#if 0
    return __x_set(0, __Dn);
#else
    return (__Dn);

#endif
}

	/*ID: 1456, INST: min.4w Dab,Dcd,Dmn*/
void do__min_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
	*__Dm = do__min_2w(__Da,__Dc);
	*__Dn = do__min_2w(__Db,__Dd);
}

	/*ID: 1457, INST: min.x Da,Db,Dn*/
/*
  Function Name: do__min_x
  Purpose: Return the minim of two signed 40 bit words.
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs:
  Return Value: It returns a 40 bit word having the minimum value.
*/
Word40 do__min_x(Word40 __Da, Word40 __Db)
{
   Word8 __Dn_1_ext, __Dn_2_ext, __Dn_max_ext;
   Word32 __Dn_1, __Dn_2, __Dn_max;
   Word40 __Dn;

   __Dn_1_ext = __x_guard(__Da);
   __Dn_2_ext = __x_guard(__Db);

   __Dn_1 = __x_to_l(__Da);
   __Dn_2 = __x_to_l(__Db);

   if (((__Dn_1_ext & 0x80) == 0x80) && ((__Dn_2_ext & 0x80) == 0x80)) { /* Both are negative */
     if ((__Dn_1_ext & 0x7f) == (__Dn_2_ext & 0x7f)) { /* if both extensions are same compare magnitude, otherwise compare the extensions */
       __Dn = (__Dn_2 < __Dn_1) ? __x_set(__Dn_1_ext, __Dn_1) : __x_set(__Dn_2_ext, __Dn_2);
	 } else {
       __Dn = (__Dn_2_ext < __Dn_1_ext) ? __x_set(__Dn_1_ext, __Dn_1) : __x_set(__Dn_2_ext, __Dn_2);
	 }
   } else if (((__Dn_1_ext & 0x80) == 0x80) && ((__Dn_2_ext & 0x80) == 0x00)) { /* __Db is positive */
     __Dn = __x_set(__Dn_1_ext, __Dn_1);
   } else if (((__Dn_2_ext & 0x80) == 0x80) && ((__Dn_1_ext & 0x80) == 0x00)) { /* __Da is positive */
     __Dn = __x_set(__Dn_2_ext, __Dn_2);
   } else { /* both are positive */
     if (__Dn_1_ext == __Dn_2_ext) {
       __Dn= ((UWord32)(__Dn_1) < (UWord32)(__Dn_2)) ? __x_set(__Dn_1_ext, __Dn_1) : __x_set(__Dn_2_ext, __Dn_2);
	 } else { /* just compare the extension */
	   __Dn = (__Dn_1_ext < __Dn_2_ext) ? __x_set(__Dn_1_ext, __Dn_1):__x_set(__Dn_2_ext, __Dn_2);
	 }
   }
   return __Dn;
}

	/*ID: 1458, INST: min.2x Dab,Dcd,Dmn*/
void do__min_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__min_x(__Da,__Dc);
	*__Dn = do__min_x(__Db,__Dd);
}

	/*ID: 1459, INST: mina Ra,Rb,Rn*/
Word32 do__mina(Word32 __Ra, Word32 __Rb)
{
	return (__Ra>__Rb)?__Rb:__Ra;
}

	/*ID: 1460, INST: mina Ra,Rb,Rn*/

	/*ID: 1461, INST: move.l #s32_t6,Dn*/

	/*ID: 1462, INST: move.l #s32_t6,Dn*/

	/*ID: 1463, INST: move.l #s9,Dn*/

	/*ID: 1464, INST: move.l #s9,Dn*/

	/*ID: 1465, INST: move.l Da,Dn*/

	/*ID: 1466, INST: move.l Da,Rn*/

	/*ID: 1467, INST: move.l Da,Rn*/

	/*ID: 1468, INST: move.l Da,TMDAT*/

	/*ID: 1469, INST: move.l Ra,Dn*/

	/*ID: 1470, INST: move.l Ra,Dn*/

	/*ID: 1471, INST: move.2l Dab,Dmn*/

	/*ID: 1472, INST: move.2l Dab,Rm,Rn*/

	/*ID: 1473, INST: move.x Da,Dn*/

	/*ID: 1474, INST: move.2x Da,Dmn*/

	/*ID: 1475, INST: move.2x Dab,Dmn*/

	/*ID: 1476, INST: move.4x Da,Db,Dc,Dd,Dmnop*/

	/*ID: 1477, INST: move.par.w #u16_t7,Dn.h*/
Word32 do__move_par_w_h(Word32 __Dn_h, Word32 __imm1)
{
}

	/*ID: 1478, INST: move.par.w #u16_t7,Dn.l*/
Word32 do__move_par_w_l(Word32 __Dn_l, Word32 __imm1)
{
}

	/*ID: 1479, INST: mpy.2t Da,Db,Dn*/
Word40 do__mpy_2t(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1480, INST: mpy.x Da.h,Db.h,Dn*/
Word40 do__mpy_x_hh(Word32 __Da_h, Word32 __Db_h)
{
  Word32 res;
  res=__l_get_msb(__Da_h)*__l_get_msb(__Db_h)<<1;
  if ((__l_get_msb(__Da_h) == MIN_16) && (__l_get_msb(__Db_h) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);

}

	/*ID: 1481, INST: mpy.x Da.l,Db.l,Dn*/
Word40 do__mpy_x_ll(Word32 __Da_l, Word32 __Db_l)
{
  Word32 res;
  res=__l_get_lsb(__Da_l)*__l_get_lsb(__Db_l)<<1;
  if ((__l_get_lsb(__Da_l) == MIN_16) && (__l_get_lsb(__Db_l) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);

}

	/*ID: 1482, INST: mpy.x Da.h,Db.l,Dn*/
Word40 do__mpy_x_hl(Word32 __Da_h, Word32 __Db_l)
{
  Word32 res;
  res=(__l_get_msb(__Da_h)*__l_get_lsb(__Db_l))<<1;
  if ((__l_get_msb(__Da_h) == MIN_16) && (__l_get_lsb(__Db_l) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);
}

	/*ID: 1483, INST: mpy.2x Da,Db,Dmn*/
void do__mpy_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
   *__Dm = do__mpy_x_hh(__Da, __Db);
   *__Dn = do__mpy_x_ll(__Da, __Db);
}

	/*ID: 1484, INST: mpy.h.2x Da,Db,Dmn*/
void do__mpy_h_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1485, INST: mpy.h.s.2x Da,Db,Dmn*/
void do__mpy_h_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1486, INST: mpy.h.sr.2w Da,Db,Dn*/
Word32 do__mpy_h_sr_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1487, INST: mpy.hh.2x Dab,Dcd,Dmn*/
void do__mpy_hh_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1488, INST: mpy.hh.s.2x Dab,Dcd,Dmn*/
void do__mpy_hh_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1489, INST: mpy.hh.sr.2w Dab,Dcd,Dn*/
Word32 do__mpy_hh_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1490, INST: mpy.i.x #s16_t2,Da.l,Dn*/
Word40 do__mpy_i_x_l(Word32 __imm1, Word32 __Da_l)
{
}

	/*ID: 1491, INST: mpy.i.x Da.l,Db.l,Dn*/
Word40 do__mpy_i_x_ll(Word32 __Da_l, Word32 __Db_l)
{
  Word32 res;
  res=__l_get_lsb(__Da_l)*__l_get_lsb(__Db_l);
  if ((__l_get_lsb(__Da_l) == MIN_16) && (__l_get_lsb(__Db_l) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);
}

	/*ID: 1492, INST: mpy.l.2x Da,Db,Dmn*/
void do__mpy_l_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
  *__Dm = do__mpy_x_hl(__Db, __Da);
  *__Dn = do__mpy_x_ll(__Da, __Db);
}

	/*ID: 1493, INST: mpy.l.i.4t Dab,Dcd,Dmn*/
void do__mpy_l_i_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1494, INST: mpy.l.s.2x Da,Db,Dmn*/
void do__mpy_l_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1495, INST: mpy.l.sr.2w Da,Db,Dn*/
Word32 do__mpy_l_sr_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1496, INST: mpy.leg.x Da.h,Db.h,Dn*/

	/*ID: 1497, INST: mpy.leg.x Da.l,Db.l,Dn*/

	/*ID: 1498, INST: mpy.leg.x Da.h,Db.l,Dn*/

	/*ID: 1499, INST: mpy.ll.2x Dab,Dcd,Dmn*/
void do__mpy_ll_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
  *__Dm =  do__mpy_x_ll(__Da, __Dc);
  *__Dn =  do__mpy_x_ll(__Db, __Dd);

}

	/*ID: 1500, INST: mpy.ll.i.2x Dab,Dcd,Dmn*/
void do__mpy_ll_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
  *__Dm =  do__mpy_i_x_ll(__Da, __Dc);
  *__Dn =  do__mpy_i_x_ll(__Db, __Dd);
}

	/*ID: 1501, INST: mpy.ll.s.2x Dab,Dcd,Dmn*/
void do__mpy_ll_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1502, INST: mpy.ll.sr.2w Dab,Dcd,Dn*/
Word32 do__mpy_ll_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1503, INST: mpy.r.2t Da,Db,Dn*/
Word40 do__mpy_r_2t(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1504, INST: mpy.r.4t Da,Dcd,Dmn*/
void do__mpy_r_4t_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1505, INST: mpy.r.4t Dab,Dcd,Dmn*/
void do__mpy_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1506, INST: mpy.r.x Da.h,Db.h,Dn*/
Word40 do__mpy_r_x_hh(Word32 __Da_h, Word32 __Db_h)
{
	Word40 result;
	result = do__mpy_x_hh(__Da_h,__Db_h);
	result = __x_round(result);
	return (result);
}

	/*ID: 1507, INST: mpy.r.x Da.l,Db.l,Dn*/
Word40 do__mpy_r_x_ll(Word32 __Da_l, Word32 __Db_l)
{
	Word40 result;
	result = do__mpy_x_ll(__Da_l,__Db_l);
	result = __x_round(result);
	return (result);
}

	/*ID: 1508, INST: mpy.r.x Da.l,Db.h,Dn*/
Word40 do__mpy_r_x_lh(Word32 __Da_l, Word32 __Db_h)
{
	Word40 result;
	result = do__mpy_x_hl(__Db_h,__Da_l);
	result = __x_round(result);
	return (result);
}

	/*ID: 1509, INST: mpy.rleg.x Da.h,Db.h,Dn*/
Word40 do__mpy_rleg_x_hh(Word32 __Da_h, Word32 __Db_h)
{
}

	/*ID: 1510, INST: mpy.rleg.x Da.l,Db.l,Dn*/
Word40 do__mpy_rleg_x_ll(Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1511, INST: mpy.rleg.x Da.l,Db.h,Dn*/
Word40 do__mpy_rleg_x_lh(Word32 __Da_l, Word32 __Db_h)
{
}

	/*ID: 1512, INST: mpy.s.2w Da,Db,Dn*/
Word32 do__mpy_s_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1513, INST: mpy.s.x Da.h,Db.h,Dn*/
Word40 do__mpy_s_x_hh(Word32 __Da_h, Word32 __Db_h)
{
	Word40 result;
	result = do__mpy_x_hh(__Da_h,__Db_h);
	result = __l_to_x(do__sat_x_l(result));
	return (result);

}

	/*ID: 1514, INST: mpy.s.x Da.l,Db.l,Dn*/
Word40 do__mpy_s_x_ll(Word32 __Da_l, Word32 __Db_l)
{
	Word40 result;
	result = do__mpy_x_ll(__Da_l,__Db_l);
	result = __l_to_x(do__sat_x_l(result));
	return (result);
}

	/*ID: 1515, INST: mpy.s.x Da.h,Db.l,Dn*/
Word40 do__mpy_s_x_hl(Word32 __Da_h, Word32 __Db_l)
{
	Word40 result;
	result = do__mpy_x_hl(__Da_h,__Db_l);
	result = __l_to_x(do__sat_x_l(result));
	return (result);
}

	/*ID: 1516, INST: mpy.s.2x Da,Db,Dmn*/
void do__mpy_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1517, INST: mpy.sr.2w Da,Db,Dn*/
Word32 do__mpy_sr_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1518, INST: mpy.sr.4w Da,Dcd,Dmn*/
void do__mpy_sr_4w_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1519, INST: mpy.sr.4w Dab,Dcd,Dmn*/
void do__mpy_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1520, INST: mpy.su.i.2t Da,Db,Dn*/
Word40 do__mpy_su_i_2t(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1521, INST: mpy.us.i.x Da.l,Db.l,Dn*/
Word40 do__mpy_us_i_x_ll(Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1522, INST: mpy.uu.i.x Da.l,Db.l,Dn*/
UWord40 do__mpy_uu_i_x_ll(Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1523, INST: mpy32.ll Da,Db,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__mpy32_ll(Word32 __Da, Word32 __Db)
{
}
#endif

	/*ID: 1524, INST: mpy32.i.ll Da,Db,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__mpy32_i_ll(Word32 __Da, Word32 __Db)
{
}
#endif

	/*ID: 1525, INST: mpy32.il.l Da,Db,Dn*/
Word32 do__mpy32_il_l(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1526, INST: mpy32.pnx.r.2x Da,Dcd,Dmn*/
void do__mpy32_pnx_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1527, INST: mpy32.r.x Da,Db,Dn*/
Word40 do__mpy32_r_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1528, INST: mpy32.r.2x Dab,Dcd,Dmn*/
void do__mpy32_r_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1529, INST: mpy32.sr.x Da,Db,Dn*/
Word40 do__mpy32_sr_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1530, INST: mpy32.sr.2x Dab,Dcd,Dmn*/
void do__mpy32_sr_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1531, INST: mpy32.su.i.ll Da,Db,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__mpy32_su_i_ll(Word32 __Da, Word32 __Db)
{
}
#endif

	/*ID: 1532, INST: mpy32.uu.i.ll Da,Db,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
UInt64 do__mpy32_uu_i_ll(Word32 __Da, Word32 __Db)
{
}
#endif

	/*ID: 1533, INST: mpy32a.i #s16,Ra,Rn*/
Word32 do__mpy32a_i(Word32 __imm1, Word32 __Ra)
{
}

	/*ID: 1534, INST: mpy32a.i Ra,Rb,Rn*/

	/*ID: 1535, INST: mpyadda.i #u16_t6,Ra,Rb,Rn*/
Word32 do__mpyadda_i(Word32 __imm1, Word32 __Ra, Word32 __Rb)
{
}

	/*ID: 1536, INST: mpycx.2x Da,Db,Dmn*/
void do__mpycx_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
        *__Dm = do__sub_x( do__mpy_x_ll(__Da,__Db),do__mpy_x_hh(__Da,__Db));
	*__Dn = do__add_x( do__mpy_x_hl(__Db,__Da),do__mpy_x_hl(__Da,__Db));
}

	/*ID: 1537, INST: mpycx.c.2x Da,Db,Dmn*/
void do__mpycx_c_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x( do__mpy_x_hh(__Da,__Db),do__mpy_x_ll(__Da,__Db));
	*__Dn = do__sub_x( do__mpy_x_hl(__Db,__Da),do__mpy_x_hl(__Da,__Db));
}

	/*ID: 1538, INST: mpycx.c.r.2t Da,Db,Dn*/
Word40 do__mpycx_c_r_2t(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1539, INST: mpycx.c.r.4t Da,Dcd,Dmn*/
void do__mpycx_c_r_4t_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1540, INST: mpycx.c.r.4t Dab,Dcd,Dmn*/
void do__mpycx_c_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1541, INST: mpycx.c.s.2x Da,Db,Dmn*/
void do__mpycx_c_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1542, INST: mpycx.c.sr.2w Da,Db,Dn*/
Word32 do__mpycx_c_sr_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1543, INST: mpycx.c.sr.4w Da,Dcd,Dmn*/
void do__mpycx_c_sr_4w_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1544, INST: mpycx.c.sr.4w Dab,Dcd,Dmn*/
void do__mpycx_c_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
  Word16 Dm_H, Dm_L, Dn_H, Dn_L;
  long long mult1 = (__l_get_msb(__Da) *  __l_get_msb(__Dc));
  long long  mult2 = (__l_get_lsb(__Da) * __l_get_lsb(__Dc));
  mult1 = (mult1 + mult2) << 1;
  mult1 = (mult1 + 0x8000) >> 16;
  Dm_H = __saturate((Word32)(mult1));

  mult1 =  __l_get_lsb(__Da) *  __l_get_msb(__Dc);
  mult2 =  __l_get_msb(__Da) * __l_get_lsb(__Dc);
  mult1 = (mult2 - mult1) << 1;
  mult1 = (mult1 + 0x8000) >> 16;
  Dm_L = __saturate((Word32)(mult1));

  mult1 = __l_get_msb(__Db) * __l_get_msb(__Dd);
  mult2 = __l_get_lsb(__Db) * __l_get_lsb(__Dd);
  mult1 = (mult1 + mult2) << 1;
  mult1 = (mult1 + 0x8000) >> 16;
  Dn_H = __saturate((Word32)(mult1));

  mult1 =  __l_get_lsb(__Db) * __l_get_msb(__Dd);
  mult2 =  __l_get_msb(__Db) * __l_get_lsb(__Dd);
  mult1 = (mult2 - mult1) << 1;
  mult1 = (mult1 + 0x8000) >> 16;
  Dn_L = __saturate((Word32)(mult1));

  *__Dm = (((Dm_H << 16)& 0xffff0000) | (Dm_L & 0xffff));
  *__Dn = (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1545, INST: mpycx.i.2x Da,Db,Dmn*/
void do__mpycx_i_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
  *__Dm = do__sub_x(__l_to_x(__l_get_lsb(__Da) * __l_get_lsb(__Db)),__l_to_x(__l_get_msb(__Da) * __l_get_msb(__Db)));
  *__Dn = do__add_x(__l_to_x(__l_get_lsb(__Da) * __l_get_msb(__Db)), __l_to_x(__l_get_msb(__Da) * __l_get_lsb(__Db)));
}

	/*ID: 1546, INST: mpycx.ir.4t Dab,Dcd,Dmn*/
void do__mpycx_ir_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1547, INST: mpycx.r.2t Da,Db,Dn*/
Word40 do__mpycx_r_2t(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1548, INST: mpycx.r.4t Da,Dcd,Dmn*/
void do__mpycx_r_4t_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1549, INST: mpycx.r.4t Dab,Dcd,Dmn*/
void do__mpycx_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1550, INST: mpycx.s.2x Da,Db,Dmn*/
void do__mpycx_s_2x(Word32 __Da, Word32 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1551, INST: mpycx.sr.2w Da,Db,Dn*/
Word32 do__mpycx_sr_2w(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1552, INST: mpycx.sr.4w Da,Dcd,Dmn*/
void do__mpycx_sr_4w_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
	Word40 m1,m2,m;
	Word32 Da_H,Da_L;
	m = do__sub_x(do__mpy_x_ll(__Da,__Dc),do__mpy_x_hh(__Da,__Dc));
	m = do__add_x(m,__x_set(0x0,0x8000));
	m = do__ash_rgt_x(16,m);
	Da_H = __x_to_l(do__sat_x_w(m));
	*__Dm = (*__Dm & 0xffff) | ((Da_H << 16)& 0xffff0000);

	m = do__add_x(do__mpy_x_hl(__Da,__Dc),do__mpy_x_hl(__Dc,__Da));
	m = do__add_x(m,__x_set(0x0,0x8000));
	m = do__ash_rgt_x(16,m);
	Da_L = __x_to_l(do__sat_x_w(m));
	*__Dm = (*__Dm & 0xffff0000) |(Da_L & 0xffff);

	m = do__sub_x(do__mpy_x_ll(__Da,__Dd),do__mpy_x_hh(__Da,__Dd));
	m = do__add_x(m,__x_set(0x0,0x8000));
	m = do__ash_rgt_x(16,m);
	Da_H = __x_to_l(do__sat_x_w(m));
	*__Dn = (*__Dn & 0xffff) | ((Da_H << 16)& 0xffff0000);

	m = do__add_x(do__mpy_x_hl(__Dd,__Da),do__mpy_x_hl(__Da,__Dd));
	m = do__add_x(m,__x_set(0x0,0x8000));
	m = do__ash_rgt_x(16,m);
	Da_L = __x_to_l(do__sat_x_w(m));
	*__Dn = (*__Dn & 0xffff0000) |(Da_L & 0xffff);

}

	/*ID: 1553, INST: mpycx.sr.4w Dab,Dcd,Dmn*/
void do__mpycx_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1554, INST: mpycxd.cpn.2x Da,Dcd,Dmn*/
void do__mpycxd_cpn_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1555, INST: mpycxd.cpn.2x Dab,Dcd,Dmn*/
void do__mpycxd_cpn_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 m1,m2,m3,m4;

	m1 = do__mpy_x_hh(__Da,__Dc);
	m2=  do__mpy_x_ll(__Da, __Dc);
	m3=  do__mpy_x_hh(__Db, __Dd);
	m4 = do__mpy_x_ll(__Db, __Dd);

	*__Dm = do__sub_x(do__add_x(m3,m4), do__add_x(m1,m2));

	m1 = do__mpy_x_hl(__Da, __Dc);
	m2 = do__mpy_x_hl(__Dc, __Da);
	m3 = do__mpy_x_hl(__Db, __Dd);
	m4 = do__mpy_x_hl(__Dd, __Db);
	*__Dn = do__sub_x(do__sub_x(m4,m3), do__sub_x(m2,m1));
}

	/*ID: 1556, INST: mpycxd.cpn.s.2x Da,Dcd,Dmn*/
void do__mpycxd_cpn_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1557, INST: mpycxd.cpn.s.2x Dab,Dcd,Dmn*/
void do__mpycxd_cpn_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1558, INST: mpycxd.cpn.sr.2w Dab,Dcd,Dn*/
Word32 do__mpycxd_cpn_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1559, INST: mpycxd.cpp.2x Da,Dcd,Dmn*/
void do__mpycxd_cpp_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1560, INST: mpycxd.cpp.2x Dab,Dcd,Dmn*/
void do__mpycxd_cpp_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
   Word40 mult1 = do__mpy_x_hh(__Da, __Dc);
   Word40 mult2 = do__mpy_x_ll(__Da, __Dc);
   Word40 mult3 = do__mpy_x_hh(__Db, __Dd);
   Word40 mult4 = do__mpy_x_ll(__Db, __Dd);

   *__Dm = do__add_x(do__add_x(mult3, mult4), do__add_x(mult1, mult2));

   mult1 = do__mpy_x_hl(__Da, __Dc);
   mult2 = do__mpy_x_hl(__Dc, __Da);
   mult3 = do__mpy_x_hl(__Db, __Dd);
   mult4 = do__mpy_x_hl(__Dd, __Db);

   *__Dn = do__add_x(do__sub_x(mult4, mult3), do__sub_x(mult2, mult1));
}

	/*ID: 1561, INST: mpycxd.cpp.i.2x Dab,Dcd,Dmn*/
void do__mpycxd_cpp_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1562, INST: mpycxd.cpp.is.2x Dab,Dcd,Dmn*/
void do__mpycxd_cpp_is_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1563, INST: mpycxd.cpp.isr.2w Dab,Dcd,Dn*/
Word32 do__mpycxd_cpp_isr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1564, INST: mpycxd.cpp.s.2x Da,Dcd,Dmn*/
void do__mpycxd_cpp_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1565, INST: mpycxd.cpp.s.2x Dab,Dcd,Dmn*/
void do__mpycxd_cpp_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1566, INST: mpycxd.cpp.sr.2w Dab,Dcd,Dn*/
Word32 do__mpycxd_cpp_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1567, INST: mpycxd.pn.2x Da,Dcd,Dmn*/
void do__mpycxd_pn_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1568, INST: mpycxd.pn.2x Dab,Dcd,Dmn*/
void do__mpycxd_pn_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;
	result1 = do__sub_x(do__mpy_x_ll(__Da,__Dc),do__mpy_x_hh(__Da,__Dc));
	result2 = do__sub_x(do__mpy_x_ll(__Db,__Dd),do__mpy_x_hh(__Db,__Dd));
	*__Dm = do__sub_x(result2,result1);

	result1 = do__add_x(do__mpy_x_hl(__Da,__Dc),do__mpy_x_hl(__Dc,__Da));
	result2 = do__add_x(do__mpy_x_hl(__Db,__Dd),do__mpy_x_hl(__Dd,__Db));
	*__Dn = do__sub_x(result2,result1);
}

	/*ID: 1569, INST: mpycxd.pn.i.2x Dab,Dcd,Dmn*/
void do__mpycxd_pn_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;

	result1 = do__sub_x(do__mpy_i_x_ll(__Da,__Dc),do__mpy_i_x_hh(__Da,__Dc));
	result2 = do__sub_x(do__mpy_i_x_ll(__Db,__Dd),do__mpy_i_x_hh(__Db,__Dd));
	*__Dm = do__sub_x(result2,result1);

	result1 = do__add_x(do__mpy_i_x_hl(__Da,__Dc),do__mpy_i_x_hl(__Dc,__Da));
	result2 = do__add_x(do__mpy_i_x_hl(__Db,__Dd),do__mpy_i_x_hl(__Dd,__Db));
	*__Dn = do__sub_x(result2,result1);
}

	/*ID: 1570, INST: mpycxd.pn.is.2x Dab,Dcd,Dmn*/
void do__mpycxd_pn_is_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1571, INST: mpycxd.pn.isr.2w Dab,Dcd,Dn*/
Word32 do__mpycxd_pn_isr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1572, INST: mpycxd.pn.s.2x Da,Dcd,Dmn*/
void do__mpycxd_pn_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1573, INST: mpycxd.pn.s.2x Dab,Dcd,Dmn*/
void do__mpycxd_pn_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1574, INST: mpycxd.pn.sr.2w Dab,Dcd,Dn*/
Word32 do__mpycxd_pn_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1575, INST: mpycxd.pp.2x Da,Dcd,Dmn*/
void do__mpycxd_pp_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1576, INST: mpycxd.pp.2x Dab,Dcd,Dmn*/
void do__mpycxd_pp_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word40 result1,result2;

	result1 = do__sub_x(do__mpy_x_ll(__Da,__Dc),do__mpy_x_hh(__Da,__Dc));
	result2 = do__sub_x(do__mpy_x_ll(__Db,__Dd),do__mpy_x_hh(__Db,__Dd));
	*__Dm = do__add_x(result1,result2);

	result1 = do__add_x(do__mpy_x_hl(__Da,__Dc),do__mpy_x_hl(__Dc,__Da));
	result2 = do__add_x(do__mpy_x_hl(__Db,__Dd),do__mpy_x_hl(__Dd,__Db));
	*__Dn = do__add_x(result1,result2);
}

	/*ID: 1577, INST: mpycxd.pp.i.2x Dab,Dcd,Dmn*/
void do__mpycxd_pp_i_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word40 mult1 = __l_to_x(__l_get_msb(__Da) *  __l_get_msb(__Dc));
    Word40 mult2 = __l_to_x(__l_get_lsb(__Da) *  __l_get_lsb(__Dc));
    Word40 mult3 = __l_to_x(__l_get_msb(__Db) *  __l_get_msb(__Dd));
    Word40 mult4 = __l_to_x(__l_get_lsb(__Db) *  __l_get_lsb(__Dd));

    *__Dm = do__add_x(do__sub_x(mult2,mult1), do__sub_x(mult4, mult3));

   mult1 = __l_to_x(__l_get_msb(__Da) *  __l_get_lsb(__Dc));
   mult2 = __l_to_x(__l_get_lsb(__Da) *  __l_get_msb(__Dc));
   mult3 = __l_to_x(__l_get_msb(__Db) *  __l_get_lsb(__Dd));
   mult4 = __l_to_x(__l_get_lsb(__Db) *  __l_get_msb(__Dd));

   *__Dn = do__add_x(do__add_x(mult1, mult2), do__add_x(mult3, mult4));
}

	/*ID: 1578, INST: mpycxd.pp.is.2x Dab,Dcd,Dmn*/
void do__mpycxd_pp_is_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1579, INST: mpycxd.pp.s.2x Da,Dcd,Dmn*/
void do__mpycxd_pp_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1580, INST: mpycxd.pp.s.2x Dab,Dcd,Dmn*/
void do__mpycxd_pp_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1581, INST: mpycxd.pp.sr.2w Dab,Dcd,Dn*/
Word32 do__mpycxd_pp_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1582, INST: mpycxm.c.ir.2x Da,Dcd,Dmn*/
void do__mpycxm_c_ir_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1583, INST: mpycxm.c.r.2x Da,Dcd,Dmn*/
void do__mpycxm_c_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1584, INST: mpycxm.c.sr.2x Da,Dcd,Dmn*/
void do__mpycxm_c_sr_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1585, INST: mpycxm.ir.2x Da,Dcd,Dmn*/
void do__mpycxm_ir_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1586, INST: mpycxm.r.2x Da,Dcd,Dmn*/
void do__mpycxm_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1587, INST: mpycxm.sr.2x Da,Dcd,Dmn*/
void do__mpycxm_sr_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1588, INST: mpyd.x Da,Db,Dn*/
Word40 do__mpyd_x(Word32 __Da, Word32 __Db)
{
  return(  do__add_x(do__mpy_x_hh(__Da, __Db), do__mpy_x_ll(__Da,__Db)));
}

	/*ID: 1589, INST: mpyd.2x Da,Dcd,Dmn*/
void do__mpyd_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1590, INST: mpyd.2x Dab,Dcd,Dmn*/
void do__mpyd_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
   *__Dm = do__add_x(do__mpy_x_hh(__Da, __Dc), do__mpy_x_ll(__Da,__Dc));
   *__Dn = do__add_x(do__mpy_x_hh(__Db, __Dd), do__mpy_x_ll(__Db,__Dd));

}

	/*ID: 1591, INST: mpyd.cim.x Da,Db,Dn*/
Word40 do__mpyd_cim_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1592, INST: mpyd.cim.2x Dab,Dcd,Dmn*/
void do__mpyd_cim_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
   *__Dm = do__sub_x(do__mpy_x_hl(__Dc, __Da),do__mpy_x_hl(__Da, __Dc));
   *__Dn = do__sub_x(do__mpy_x_hl(__Dd, __Db),do__mpy_x_hl(__Db, __Dd));
}

	/*ID: 1593, INST: mpyd.cim.leg.x -Da,Db,Dn*/

	/*ID: 1594, INST: mpyd.cim.r.2t Dab,Dcd,Dn*/
Word40 do__mpyd_cim_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1595, INST: mpyd.cim.s.x Da,Db,Dn*/
Word40 do__mpyd_cim_s_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1596, INST: mpyd.cim.s.2x Dab,Dcd,Dmn*/
void do__mpyd_cim_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1597, INST: mpyd.cim.sr.2w Dab,Dcd,Dn*/
Word32 do__mpyd_cim_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1598, INST: mpyd.im.x Da,Db,Dn*/
Word40 do__mpyd_im_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1599, INST: mpyd.im.2x Dab,Dcd,Dmn*/
void do__mpyd_im_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x(do__mpy_x_hl(__Dc,__Da), do__mpy_x_hl(__Da, __Dc));
	*__Dn = do__add_x(do__mpy_x_hl(__Dd,__Db), do__mpy_x_hl(__Db, __Dd));
}

	/*ID: 1600, INST: mpyd.im.leg.x Da,Db,Dn*/

	/*ID: 1601, INST: mpyd.im.r.2t Dab,Dcd,Dn*/
Word40 do__mpyd_im_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1602, INST: mpyd.im.s.x Da,Db,Dn*/
Word40 do__mpyd_im_s_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1603, INST: mpyd.im.s.2x Dab,Dcd,Dmn*/
void do__mpyd_im_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1604, INST: mpyd.im.sr.2w Dab,Dcd,Dn*/
Word32 do__mpyd_im_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1605, INST: mpyd.leg.x Da,Db,Dn*/

	/*ID: 1606, INST: mpyd.leg.x -Da,Db,Dn*/

	/*ID: 1607, INST: mpyd.r.2t Dab,Dcd,Dn*/
Word40 do__mpyd_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1608, INST: mpyd.re.x Da,Db,Dn*/
Word40 do__mpyd_re_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1609, INST: mpyd.re.2x Dab,Dcd,Dmn*/
void do__mpyd_re_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
   *__Dm = do__sub_x(do__mpy_x_ll(__Da,__Dc), do__mpy_x_hh(__Da, __Dc));
   *__Dn = do__sub_x(do__mpy_x_ll(__Db,__Dd), do__mpy_x_hh(__Db, __Dd));
}

	/*ID: 1610, INST: mpyd.re.leg.x Da,Db,Dn*/

	/*ID: 1611, INST: mpyd.re.r.2t Dab,Dcd,Dn*/
Word40 do__mpyd_re_r_2t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1612, INST: mpyd.re.s.x Da,Db,Dn*/
Word40 do__mpyd_re_s_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1613, INST: mpyd.re.s.2x Dab,Dcd,Dmn*/
void do__mpyd_re_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1614, INST: mpyd.re.sr.2w Dab,Dcd,Dn*/
Word32 do__mpyd_re_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1615, INST: mpyd.s.x Da,Db,Dn*/
Word40 do__mpyd_s_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1616, INST: mpyd.s.2x Da,Dcd,Dmn*/
void do__mpyd_s_2x_da(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1617, INST: mpyd.s.2x Dab,Dcd,Dmn*/
void do__mpyd_s_2x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1618, INST: mpyd.sr.2w Dab,Dcd,Dn*/
Word32 do__mpyd_sr_2w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1619, INST: mpydcf.2x Da,Dcd,Dmn*/
void do__mpydcf_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1620, INST: mpydcf.h.r.4t Dab,Dcd,Dmn*/
void do__mpydcf_h_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1621, INST: mpydcf.inv.2x Da,Dcd,Dmn*/
void do__mpydcf_inv_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1622, INST: mpydcf.invh.r.4t Dab,Dcd,Dmn*/
void do__mpydcf_invh_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1623, INST: mpydcf.invl.r.4t Dab,Dcd,Dmn*/
void do__mpydcf_invl_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1624, INST: mpydcf.l.r.4t Dab,Dcd,Dmn*/
void do__mpydcf_l_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1625, INST: mpydem32.su.i.x Da,Db,Dn*/
Word40 do__mpydem32_su_i_x(Word32 __Da, Word32 __Db)
{
	Word40 result;
	Int64 __Da_h,__Db_h,resultLL;
	UInt64 __Da_l,__Db_l;
	__Db_h = (Int64)__l_get_msb(__Db);
	__Da_l = (UInt64)(__l_get_lsb(__Da));
	__Db_l = (UInt64)__l_get_lsb(__Db);
	__Da_h = (Int64)(__l_get_msb(__Da));
	resultLL = (( __Da_l * __Db_h) + (__Da_h*__Db_l))<<16;
	result = __ll_to_x(resultLL);
	return result;
}

	/*ID: 1626, INST: mpydem32.uu.i.x Da,Db,Dn*/
UWord40 do__mpydem32_uu_i_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1627, INST: mpydrf.h.2x Da,Dcd,Dmn*/
void do__mpydrf_h_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1628, INST: mpydrf.h.r.2t Da,Dcd,Dn*/
Word40 do__mpydrf_h_r_2t(Word32 __Da, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1629, INST: mpydrf.h.r.4t Dab,Dcd,Dmn*/
void do__mpydrf_h_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1630, INST: mpydrf.h.s.2x Da,Dcd,Dmn*/
void do__mpydrf_h_s_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1631, INST: mpydrf.h.sr.2w Da,Dcd,Dn*/
Word32 do__mpydrf_h_sr_2w(Word32 __Da, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1632, INST: mpydrf.h.sr.4w Dab,Dcd,Dmn*/
void do__mpydrf_h_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1633, INST: mpydrf.l.2x Da,Dcd,Dmn*/
void do__mpydrf_l_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1634, INST: mpydrf.l.r.2t Da,Dcd,Dn*/
Word40 do__mpydrf_l_r_2t(Word32 __Da, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1635, INST: mpydrf.l.r.4t Dab,Dcd,Dmn*/
void do__mpydrf_l_r_4t(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1636, INST: mpydrf.l.s.2x Da,Dcd,Dmn*/
void do__mpydrf_l_s_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1637, INST: mpydrf.l.sr.2w Da,Dcd,Dn*/
Word32 do__mpydrf_l_sr_2w(Word32 __Da, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1638, INST: mpydrf.l.sr.4w Dab,Dcd,Dmn*/
void do__mpydrf_l_sr_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1639, INST: mpyem32.su.i.x Da.h,Db.l,Dn*/
Word40 do__mpyem32_su_i_x_hl(Word32 __Da_h, Word32 __Db_l)
{
}

	/*ID: 1640, INST: mpyem32.us.x Da.l,Db.h,Dn*/
Word40 do__mpyem32_us_x_lh(Word32 __Da_l, Word32 __Db_h)
{
  Word32 res;
  res= ((UWord16)__l_get_lsb(__Da_l)) * ((Word16)__l_get_msb(__Db_h))<<1;
  if ((__l_get_msb(__Da_l) == MIN_16) && (__l_get_msb(__Db_h) == MIN_16))
    return __x_set(0,res); /* special case -1*-1 */
  else
    return __l_to_x(res);
}

	/*ID: 1641, INST: mpyem32.uu.x Da.l,Db.l,Dn*/
UWord40 do__mpyem32_uu_x_ll(Word32 __Da_l, Word32 __Db_l)
{
}

	/*ID: 1642, INST: mpyem32.uu.i.x Da.h,Db.l,Dn*/
UWord40 do__mpyem32_uu_i_x_hl(Word32 __Da_h, Word32 __Db_l)
{
}

	/*ID: 1643, INST: mpyem32.xsu.x Da,Db,Dn*/
Word40 do__mpyem32_xsu_x(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1644, INST: mpym.x Da.h,Db,Dn*/
Word40 do__mpym_x_h(Word32 __Da_h, Word32 __Db)
{
}

	/*ID: 1645, INST: mpym.i.ll Da.h,Db,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__mpym_i_ll_h(Word32 __Da_h, Word32 __Db)
{
}
#endif

	/*ID: 1646, INST: mpym.i.ll Da.l,Db,Dmn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Int64 do__mpym_i_ll_l(Word32 __Da_l, Word32 __Db)
{
}
#endif

	/*ID: 1647, INST: mpym.r.x Da.h,Db,Dn*/
Word40 do__mpym_r_x_h(Word32 __Da_h, Word32 __Db)
{
}

	/*ID: 1648, INST: mpym.r.x Da.l,Db,Dn*/
Word40 do__mpym_r_x_l(Word32 __Da_l, Word32 __Db)
{
}

	/*ID: 1649, INST: mpym.r.2x Da,Dcd,Dmn*/
void do__mpym_r_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1650, INST: mpym.r.2x Da.h,Dcd,Dmn*/
void do__mpym_r_2x_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1651, INST: mpym.r.2x Da.l,Dcd,Dmn*/
void do__mpym_r_2x_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
  long long m1,m2, result;

  m1 = (long long)(__l_get_lsb(__Da_l));
  m2 = (long long)(__Dc);
  result = ((((m1 * m2) << 1) + 0x8000)>> 16);

  *__Dm = __ll_to_x(result);

  m2 = (long long)(__Dd);
  result = ((((m1 * m2) << 1) + 0x8000)>> 16);
  *__Dn = __ll_to_x(result);
}

	/*ID: 1652, INST: mpym.sr.x Da.h,Db,Dn*/
Word40 do__mpym_sr_x_h(Word32 __Da_h, Word32 __Db)
{
}

	/*ID: 1653, INST: mpym.sr.x Da.l,Db,Dn*/
Word40 do__mpym_sr_x_l(Word32 __Da_l, Word32 __Db)
{
}

	/*ID: 1654, INST: mpym.sr.2x Da,Dcd,Dmn*/
void do__mpym_sr_2x(Word32 __Da, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1655, INST: mpym.sr.2x Da.h,Dcd,Dmn*/
void do__mpym_sr_2x_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1656, INST: mpym.sr.2x Da.l,Dcd,Dmn*/
void do__mpym_sr_2x_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1657, INST: mpym.us.il.l Da.l,Db,Dn*/
Word32 do__mpym_us_il_l_l(Word32 __Da_l, Word32 __Db)
{
}

	/*ID: 1658, INST: mpyq.x Dab,Dcd,Dn*/
Word40 do__mpyq_x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1659, INST: mpyq.i.x Dab,Dcd,Dn*/
Word40 do__mpyq_i_x(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
}

	/*ID: 1660, INST: mpyshl.s.x Da.h,Db.h,Dn*/
Word40 do__mpyshl_s_x_hh(Word32 __Da_h, Word32 __Db_h)
{
}

	/*ID: 1661, INST: msgsnd Ra*/
void do__msgsnd(Word32 __Ra)
{
}

	/*ID: 1662, INST: nand.x Da,Db,Dn*/
Word40 do__nand_x(Word40 __Da, Word40 __Db)
{
}

	/*ID: 1663, INST: nand.2x Dab,Dcd,Dmn*/
void do__nand_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __x_set(~(__x_guard(__Da) & __x_guard(__Dc)), ~(__x_to_l(__Da) & __x_to_l(__Dc)));
	*__Dn = __x_set(~(__x_guard(__Db) & __x_guard(__Dd)), ~(__x_to_l(__Db) & __x_to_l(__Dd)));
}

	/*ID: 1664, INST: neg.2t Da,Dn*/
Word40 do__neg_2t(Word40 __Da)
{
	Word32 __Da_WH,__Da_WL;
	Word40 __Dn;
	__Da_WH = do__extract_wh(__Da);
	__Da_WL = do__extract_wl(__Da);
	__Da_WH = - __Da_WH;
	__Da_WL = - __Da_WL;
	do__set_WH(__Da_WH,&__Dn);
	do__set_WL(__Da_WL,&__Dn);
	return __Dn;
}

	/*ID: 1665, INST: neg.4t Dab,Dmn*/
void do__neg_4t(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__neg_2t(__Da);
	*__Dn = do__neg_2t(__Db);
}

	/*ID: 1666, INST: neg.x Da,Dn*/

/*
  Function Name: do__neg_x
  Purpose: Return the negative of the input 40 bit Word.
  Complexity Weight:
  Inputs: __Da is a 40 bit signed integer whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs:
  Return Value: It returns a 40 bit word having the negative value..
*/
Word40 do__neg_x(Word40 __Da)
{
	Word40 tmp = __x_set(0,0);
	return do__sub_x(__Da, tmp);
}

	/*ID: 1667, INST: neg.2x Dab,Dmn*/
void do__neg_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__neg_x(__Da);
	*__Dn = do__neg_x(__Db);
}

	/*ID: 1668, INST: neg.leg.x Da,Dn*/

	/*ID: 1669, INST: neg.s.l Da,Dn*/
Word32 do__neg_s_l(Word32 __Da)
{
	Word32 __Dn;
	__Dn = -__Da;
	__Dn = do__sat_x_l(__l_to_x(__Dn));
	return __Dn;
}

	/*ID: 1670, INST: neg.s.2w Da,Dn*/
Word32 do__neg_s_2w(Word32 __Da)
{
	Word16 __Da_H,__Da_L;
	Word32 __Dn;
	__Da_H = (__Da&0xFFFF0000)>>16;
	__Da_L = (__Da&0xFFFF);

	__Da_H = (__Da_H == 0xffff8000)?0x7fff:-__Da_H;
	__Da_L = (__Da_L == 0xffff8000)?0x7fff:- __Da_L;
	__Da_H = __saturate((Word32)__Da_H);
	__Da_L = __saturate((Word32)__Da_L);
	__Dn = ((Word32)__Da_H<<16)|(__Da_L&0xFFFF);
	return __Dn;
}

	/*ID: 1671, INST: neg.s.4w Dab,Dmn*/
void do__neg_s_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
	*__Dm  = do__neg_s_2w(__Da);
	*__Dn  = do__neg_s_2w(__Db);
}

	/*ID: 1672, INST: neg.s.x Da,Dn*/
Word40 do__neg_s_x(Word40 __Da)
{
	Word40 __Dn;
	__Dn = __x_set(0x0,0x0);
	__Dn = do__sub_x(__Da,__Dn);
	__Dn = __l_to_x(do__sat_x_l(__Dn));
	return __Dn;
}

	/*ID: 1673, INST: neg.s.2x Dab,Dmn*/
void do__neg_s_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__neg_s_x(__Da);
	*__Dn = do__neg_s_x(__Db);
}

	/*ID: 1674, INST: nega Ra,Rn*/
Word32 do__nega(Word32 __Ra)
{
	return - __Ra;
}

	/*ID: 1675, INST: nega Ra,Rn*/

	/*ID: 1676, INST: neor.x Da,Db,Dn*/
Word40 do__neor_x(Word40 __Da, Word40 __Db)
{
}

	/*ID: 1677, INST: neor.2x Dab,Dcd,Dmn*/
void do__neor_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __x_set(~(__x_guard(__Da) ^ __x_guard(__Dc)), ~(__x_to_l(__Da) ^ __x_to_l(__Dc)));
	*__Dn = __x_set(~(__x_guard(__Db) ^ __x_guard(__Dd)), ~(__x_to_l(__Db) ^ __x_to_l(__Dd)));

}

	/*ID: 1678, INST: nop.agu*/
void do__nop_agu()
{
}

	/*ID: 1679, INST: nop.agu*/

	/*ID: 1680, INST: nop.dalu*/
void do__nop_dalu()
{
}

	/*ID: 1681, INST: nop.dalu*/

	/*ID: 1682, INST: nor.x Da,Db,Dn*/
/*
  Function Name: do__nor_x
  Purpose: Perform logical NOT-OR operation on two 40 bit words
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit signed integers whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs:
  Return Value: It returns a 40 bit result.
*/
Word40 do__nor_x(Word40 __Da, Word40 __Db)
{
  Word8 __Dn_1_ext, __Dn_2_ext;
  Word32 __Dn_1, __Dn_2;
  Word40 __Dn;

  __Dn_1_ext = __x_guard(__Da);
  __Dn_2_ext = __x_guard(__Db);
  __Dn_1 = __x_to_l(__Da);
  __Dn_2 = __x_to_l(__Db);

  __Dn = __x_set(~(__Dn_1_ext | __Dn_2_ext), ~(__Dn_1| __Dn_2));
  return __Dn;
}

	/*ID: 1683, INST: nor.2x Dab,Dcd,Dmn*/
void do__nor_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __x_set(~(__x_guard(__Da) | __x_guard(__Dc)), ~(__x_to_l(__Da) | __x_to_l(__Dc)));
	*__Dn = __x_set(~(__x_guard(__Db) | __x_guard(__Dd)), ~(__x_to_l(__Db) | __x_to_l(__Dd)));
}

	/*ID: 1684, INST: not.x Da,Dn*/
Word40 do__not_x(Word40 __Da)
{
}

	/*ID: 1685, INST: not.2x Dab,Dmn*/
void do__not_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __x_set(~(__x_guard(__Da)), ~(__x_to_l(__Da)));
	*__Dn = __x_set(~(__x_guard(__Db)), ~(__x_to_l(__Db)));

}

	/*ID: 1686, INST: nota Ra,Rn*/
Word32 do__nota(Word32 __Ra)
{
}

	/*ID: 1687, INST: nota Ra,Rn*/

	/*ID: 1688, INST: notify (Rn)*/
void do__notify(Word8* __p)
{
}

	/*ID: 1689, INST: or.l #u32,Da,Dn*/

	/*ID: 1690, INST: or.x Da,Db,Dn*/
Word40 do__or_x(Word40 __Da, Word40 __Db)
{
}

	/*ID: 1691, INST: or.2x Dab,Dcd,Dmn*/
void do__or_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __x_set((__x_guard(__Da) | __x_guard(__Dc)), (__x_to_l(__Da) | __x_to_l(__Dc)));
	*__Dn = __x_set((__x_guard(__Db) | __x_guard(__Dd)), (__x_to_l(__Db) | __x_to_l(__Dd)));
}

	/*ID: 1692, INST: ora Ra,Rb,Rn*/

	/*ID: 1693, INST: ora Ra,Rb,Rn*/

	/*ID: 1694, INST: pack.b.4b Da,Db,Dn*/
Word32 do__pack_b_4b(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1695, INST: pack.bh.8b Dab,Dcd,Dmn*/
void do__pack_bh_8b(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1696, INST: pack.bl.8b Dab,Dcd,Dmn*/
void do__pack_bl_8b(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1697, INST: pack.f.2t Da,Db,Dn*/
Word40 do__pack_f_2t(Word40 __Da, Word40 __Db)
{
  Word8 ext1, ext2;
  Word40 __Dn;

  ext1 = __x_guard(__Da);
  ext2 = __x_guard(__Db);

  __Dn = __x_set(((ext2 & 0x0f)<<4)|(ext1 & 0x0f), (__x_to_l(__Db) & 0xffff0000) | ((__x_to_l(__Da) >> 16) & 0x0000ffff));
  return __Dn;
}

	/*ID: 1698, INST: pack.h.4b Da,Db,Dn*/
Word32 do__pack_h_4b(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1699, INST: pack.h.8b Dab,Dcd,Dmn*/
void do__pack_h_8b(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1700, INST: pack.hhhh.4t Dab,Dcd,Dmn*/
void do__pack_hhhh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db) >> 16) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd) >> 16) & 0x0000ffff));
}

	/*ID: 1701, INST: pack.hhhl.4t Dab,Dcd,Dmn*/
void do__pack_hhhl_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db) >> 16) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|(ext2 & 0xf), (__x_to_l(__Dc) & 0xffff0000) | (__x_to_l(__Dd) & 0x0000ffff));
}

	/*ID: 1702, INST: pack.hhlh.4t Dab,Dcd,Dmn*/
void do__pack_hhlh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db) >> 16) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd) >> 16) & 0xffff));
}

	/*ID: 1703, INST: pack.hhll.4t Dab,Dcd,Dmn*/
void do__pack_hhll_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db) >> 16) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd)) & 0xffff));
}

	/*ID: 1704, INST: pack.hl.8b Da,Db,Dmn*/
void do__pack_hl_8b(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1705, INST: pack.hlhh.4t Dab,Dcd,Dmn*/
void do__pack_hlhh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db)) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd) >> 16) & 0x0000ffff));
}

	/*ID: 1706, INST: pack.hlhl.4t Dab,Dcd,Dmn*/
void do__pack_hlhl_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db)) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd)) & 0x0000ffff));
}

	/*ID: 1707, INST: pack.hllh.4t Dab,Dcd,Dmn*/
void do__pack_hllh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db)) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd) >> 16) & 0xffff));
}

	/*ID: 1708, INST: pack.hlll.4t Dab,Dcd,Dmn*/
void do__pack_hlll_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Da) & 0xffff0000) | ((__x_to_l(__Db)) & 0x0000ffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd)) & 0xffff));
}

	/*ID: 1709, INST: pack.ins0.h3.4w Da.h,Dcd,Dmn*/
void do__pack_ins0_h3_4w_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1710, INST: pack.ins0.h3.4w Da.l,Dcd,Dmn*/
void do__pack_ins0_h3_4w_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1711, INST: pack.ins0.l3.4w Da.h,Dcd,Dmn*/
void do__pack_ins0_l3_4w_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1712, INST: pack.ins0.l3.4w Da.l,Dcd,Dmn*/
void do__pack_ins0_l3_4w_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1713, INST: pack.ins1.h3.4w Da.h,Dcd,Dmn*/
void do__pack_ins1_h3_4w_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1714, INST: pack.ins1.h3.4w Da.l,Dcd,Dmn*/
void do__pack_ins1_h3_4w_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1715, INST: pack.ins2.h3.4w Da.h,Dcd,Dmn*/
void do__pack_ins2_h3_4w_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1716, INST: pack.ins2.h3.4w Da.l,Dcd,Dmn*/
void do__pack_ins2_h3_4w_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1717, INST: pack.ins3.h3.4w Da.h,Dcd,Dmn*/
void do__pack_ins3_h3_4w_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1718, INST: pack.ins3.h3.4w Da.l,Dcd,Dmn*/
void do__pack_ins3_h3_4w_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1719, INST: pack.ins3.l3.4w Da.h,Dcd,Dmn*/
void do__pack_ins3_l3_4w_h(Word32 __Da_h, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1720, INST: pack.ins3.l3.4w Da.l,Dcd,Dmn*/
void do__pack_ins3_l3_4w_l(Word32 __Da_l, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1721, INST: pack.l.4b Da,Db,Dn*/
Word32 do__pack_l_4b(Word32 __Da, Word32 __Db)
{
}

	/*ID: 1722, INST: pack.l.8b Dab,Dcd,Dmn*/
void do__pack_l_8b(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1723, INST: pack.l.s.2w Da,Db,Dn*/
Word40 do__pack_l_s_2w(Word32 __Da, Word32 __Db)
{
	Word16 __Da_16,__Db_16;
	Word8 __Dn_ext;
	Word40 __Dn;
	__Da_16 = __saturate(__Da);
	__Db_16 = __saturate(__Db);
	__Dn_ext = ((__Db_16 >> 15)&0x1)?0xf:0;
	__Dn_ext = ((__Da_16 >> 15)&0x1)?(__Dn_ext | 0xf0):(__Dn_ext | 0);
	__Dn = __x_set(__Dn_ext,( __Da_16<<16 | (UWord16)__Db_16));
	return __Dn;
}

	/*ID: 1724, INST: pack.l.s.4w Dab,Dcd,Dmn*/
void do__pack_l_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1725, INST: pack.lh.8b Da,Db,Dmn*/
void do__pack_lh_8b(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1726, INST: pack.lhhh.4t Dab,Dcd,Dmn*/
void do__pack_lhhh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db) >> 16) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd) >> 16) & 0x0000ffff));
}

	/*ID: 1727, INST: pack.lhhl.4t Dab,Dcd,Dmn*/
void do__pack_lhhl_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db) >> 16) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd)) & 0x0000ffff));
}

	/*ID: 1728, INST: pack.lhlh.4t Dab,Dcd,Dmn*/
void do__pack_lhlh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db) >> 16) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd) >> 16) & 0xffff));
}

	/*ID: 1729, INST: pack.lhll.4t Dab,Dcd,Dmn*/
void do__pack_lhll_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db) >> 16) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd)) & 0xffff));
}

	/*ID: 1730, INST: pack.llhh.4t Dab,Dcd,Dmn*/
void do__pack_llhh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db)) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2 >> 4) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd) >> 16) & 0x0000ffff));
}

	/*ID: 1731, INST: pack.llhl.4t Dab,Dcd,Dmn*/
void do__pack_llhl_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db)) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set((ext1 & 0xf0)|((ext2) & 0xf), (__x_to_l(__Dc) & 0xffff0000) | ((__x_to_l(__Dd)) & 0x0000ffff));
}

	/*ID: 1732, INST: pack.lllh.4t Dab,Dcd,Dmn*/
void do__pack_lllh_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db)) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2) >> 4) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd) >> 16) & 0xffff));
}

	/*ID: 1733, INST: pack.llll.4t Dab,Dcd,Dmn*/
void do__pack_llll_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word8 ext1, ext2;
	ext1 = __x_guard(__Da);
	ext2 = __x_guard(__Db);

	*__Dm = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Da) & 0xffff) << 16) | ((__x_to_l(__Db)) & 0xffff));
	ext1 = __x_guard(__Dc);
	ext2 = __x_guard(__Dd);

	*__Dn = __x_set(((ext1 & 0xf) << 4) |(((ext2)) & 0xf) , ((__x_to_l(__Dc) & 0xffff) << 16) | ((__x_to_l(__Dd)) & 0xffff));
}

	/*ID: 1734, INST: pack.t.2t Da.h,Db.h,Dn*/
Word40 do__pack_t_2t_hh(Word40 __Da_h, Word40 __Db_h)
{
	Word40 __Dn;
	Word32 __Dn32;
	Word8 __Dn_e = 0;
	__Dn32 = ((__x_to_l(__Da_h) & 0xFFFF0000) | ((__x_to_l(__Db_h) >> 16) & 0xFFFF));
	__Dn_e = ((__x_guard(__Da_h) & 0xF0) | ((__x_guard(__Db_h) & 0xF0) >> 4));
	__Dn = __x_set(__Dn_e, __Dn32);
	return __Dn;
}

	/*ID: 1735, INST: pack.t.2t Da.l,Db.l,Dn*/
Word40 do__pack_t_2t_ll(Word40 __Da_l, Word40 __Db_l)
{
	Word40 __Dn;
	Word32 __Dn32;
	Word8 __Dn_e = 0;
	__Dn32 = (((__x_to_l(__Da_l) & 0xFFFF) << 16) | (__x_to_l(__Db_l) & 0xFFFF));
	__Dn_e = (((__x_guard(__Da_l) & 0xF) << 4) | (__x_guard(__Db_l) & 0xF));
	__Dn = __x_set(__Dn_e, __Dn32);
	return __Dn;
}

	/*ID: 1736, INST: pack.t.2t Da.h,Db.l,Dn*/
Word40 do__pack_t_2t_hl(Word40 __Da_h, Word40 __Db_l)
{
	Word40 __Dn;
	Word32 __Dn32;
	Word8 __Dn_e = 0;
	__Dn32 = (((__x_to_l(__Da_h) & 0xFFFF0000)) | (__x_to_l(__Db_l) & 0xFFFF));
	__Dn_e = (((__x_guard(__Da_h) & 0xF0)) | (__x_guard(__Db_l) & 0xF));
	__Dn = __x_set(__Dn_e, __Dn32);
	return __Dn;
}

	/*ID: 1737, INST: pack.t.2t Da.l,Db.h,Dn*/
Word40 do__pack_t_2t_lh(Word40 __Da_l, Word40 __Db_h)
{
	Word40 __Dn;
	Word32 __Dn32;
	Word8 __Dn_e = 0;
	__Dn32 = (((__x_to_l(__Da_l) & 0xFFFF) << 16) | ((__x_to_l(__Db_h) >> 16) & 0xFFFF));
	__Dn_e = (((__x_guard(__Da_l) & 0xF) << 4) | ((__x_guard(__Db_h) & 0xF0) >> 4));
	__Dn = __x_set(__Dn_e, __Dn32);
	return __Dn;
}

	/*ID: 1738, INST: pack.w.2w Da.h,Db.h,Dn*/
Word40 do__pack_w_2w_hh(Word32 __Da_h, Word32 __Db_h)
{
	Word32 __Dn32;
	Word40 __Dn;
	Word8 __Dn_e = 0;
	__Dn32 = (((__l_get_msb(__Da_h) << 16) & 0xFFFF0000) | ((__l_get_msb(__Db_h)) & 0xFFFF));
	if(__Dn32 & 0x80000000) __Dn_e = 0xF0;
	if(__Dn32 & 0x8000) __Dn_e |= 0x0F;
	__Dn = __x_set(__Dn_e, __Dn32);
	return __Dn;
}

	/*ID: 1739, INST: pack.w.2w Da.l,Db.l,Dn*/
/*
  Function Name: do__pack_w_2w_ll
  Purpose: Pack two words (20 bit/16) in to 40/32 bit result
  Complexity Weight:
  Inputs: __Da and __Db are 32 bit signed integers whose value falls in the range 0x8000 0000 - 0x7FFF FFFF
  Outputs:
  Return Value: It returns a 32 bit result.
*/
Word40 do__pack_w_2w_ll(Word32 __Da_l, Word32 __Db_l)
{
	Word32 __Dn32;
	Word8 __Dn_e = 0;
   Word40 __Dn;

   __Dn32 = ((((__l_get_lsb(__Da_l)) << 16) & 0xFFFF0000) | ((__l_get_lsb(__Db_l)) & 0xFFFF));
   if(__Dn32 & 0x80000000) __Dn_e = 0xF0;
   if(__Dn32 & 0x8000) __Dn_e |= 0x0F;
   __Dn = __x_set(__Dn_e, __Dn32);
   return __Dn;

}

	/*ID: 1740, INST: pack.w.2w Da.h,Db.l,Dn*/
Word40 do__pack_w_2w_hl(Word32 __Da_h, Word32 __Db_l)
{
	Word32 __Dn32;
	Word8 __Dn_e =0;
	Word40 __Dn;

   __Dn32 = ((((__l_get_msb(__Da_h)) << 16) & 0xFFFF0000) | ((__l_get_lsb(__Db_l)) & 0xFFFF));
   if(__Dn32 & 0x80000000) __Dn_e = 0xF0;
   if(__Dn32 & 0x8000) __Dn_e |= 0x0F;
   __Dn = __x_set(__Dn_e, __Dn32);
   return __Dn;

}

	/*ID: 1741, INST: pack.w.2w Da.l,Db.h,Dn*/
Word40 do__pack_w_2w_lh(Word32 __Da_l, Word32 __Db_h)
{
	Word32 __Dn32;
	Word8 __Dn_e =0;
	Word40 __Dn;

   __Dn32 = ((((__l_get_lsb(__Da_l)) << 16) & 0xFFFF0000) | ((__l_get_msb(__Db_h)) & 0xFFFF));
   if(__Dn32 & 0x80000000) __Dn_e = 0xF0;
   if(__Dn32 & 0x8000) __Dn_e |= 0x0F;
   __Dn = __x_set(__Dn_e, __Dn32);
   return __Dn;

}

	/*ID: 1742, INST: pack.x.s.2f Da,Db,Dn*/
Word32 do__pack_x_s_2f(Word40 __Da, Word40 __Db)
{
}

	/*ID: 1743, INST: pack.x.s.4f Dab,Dcd,Dmn*/
void do__pack_x_s_4f(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1744, INST: pack.x.s.2w Da,Db,Dn*/
Word40 do__pack_x_s_2w(Word40 __Da, Word40 __Db)
{
}

	/*ID: 1745, INST: pack.x.s.4w Dab,Dcd,Dmn*/
void do__pack_x_s_4w(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1746, INST: pcalc #u8_t2,Pabc,Pmn*/

	/*ID: 1747, INST: pcalc #u8_t2,Pabc,Pn*/

	/*ID: 1748, INST: pcalca #u8_t2,Pabc,Pmn*/

	/*ID: 1749, INST: pcalca #u8_t2,Pabc,Pn*/

	/*ID: 1750, INST: pcmb.inval.l1 Ra,Rn*/
Word32 do__pcmb_inval_l1(Word32 __Ra)
{
}

	/*ID: 1751, INST: pcmb.inval.l12 Ra,Rn*/
Word32 do__pcmb_inval_l12(Word32 __Ra)
{
}

	/*ID: 1752, INST: pfetchb.l12 Ra,Rn*/
Word32 do__pfetchb_l12(Word32 __Ra)
{
}

	/*ID: 1753, INST: pfetchb.l2 Ra,Rn*/
Word32 do__pfetchb_l2(Word32 __Ra)
{
}

	/*ID: 1754, INST: pfetchb.lck.l2 Ra,Rn*/
Word32 do__pfetchb_lck_l2(Word32 __Ra)
{
}

	/*ID: 1755, INST: pinita #uena,#uval*/

	/*ID: 1756, INST: pop.l Ca*/

	/*ID: 1757, INST: pop.l Da*/

	/*ID: 1758, INST: pop.l Ra*/

	/*ID: 1759, INST: pop.2l Deo*/

	/*ID: 1760, INST: pop.2l Reo*/

	/*ID: 1761, INST: pop.4l Cxyzw*/

	/*ID: 1762, INST: pop.4l Rxyzw*/

	/*ID: 1763, INST: pop.4x Dxyzw*/

	/*ID: 1764, INST: popc.l Ca*/

	/*ID: 1765, INST: popc.l Da*/

	/*ID: 1766, INST: popc.l Ra*/

	/*ID: 1767, INST: popc.2l Deo*/

	/*ID: 1768, INST: popc.2l Reo*/

	/*ID: 1769, INST: popc.4l Cxyzw*/

	/*ID: 1770, INST: popc.4l Rxyzw*/

	/*ID: 1771, INST: popc.4x Dxyzw*/

	/*ID: 1772, INST: prda.rst #uena,#u12_t1,Ra*/

	/*ID: 1773, INST: prda.swp #uena,#u12_t1,Rn*/

	/*ID: 1774, INST: punlockb.l2 Ra,Rn*/
Word32 do__punlockb_l2(Word32 __Ra)
{
}

	/*ID: 1775, INST: push.l Ca*/

	/*ID: 1776, INST: push.l Da*/

	/*ID: 1777, INST: push.l Ra*/

	/*ID: 1778, INST: push.2l Deo*/

	/*ID: 1779, INST: push.2l Reo*/

	/*ID: 1780, INST: push.4l Cxyzw*/

	/*ID: 1781, INST: push.4l Rxyzw*/

	/*ID: 1782, INST: push.4x Dxyzw*/

	/*ID: 1783, INST: pushc.l Ca*/

	/*ID: 1784, INST: pushc.l Da*/

	/*ID: 1785, INST: pushc.l Ra*/

	/*ID: 1786, INST: pushc.2l Deo*/

	/*ID: 1787, INST: pushc.2l Reo*/

	/*ID: 1788, INST: pushc.4l Cxyzw*/

	/*ID: 1789, INST: pushc.4l Rxyzw*/

	/*ID: 1790, INST: pushc.4x Dxyzw*/

	/*ID: 1791, INST: recip Da,Dmn*/
void do__recip(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word8 inp;
	Word16 tmp13;
	Word8 tmp1 = 1;
	Word32 recip_LUT_A,__Dm_2;
	Word32 recip_LUT_B,__Dn_2;
	inp = (__Da & 0x3F000000)>>24;
	recip_LUT_A = get_recip_LUT_A(inp);
	recip_LUT_B = get_recip_LUT_B(inp);
	__Dm_2 = recip_LUT_A << 13;
	__Dn_2 = (recip_LUT_B << 13)|(1<<31);
	*__Dm = __x_set(0x0,__Dm_2);
	*__Dn = __x_set(0x0,__Dn_2);
}

	/*ID: 1792, INST: rnd.leg.x Da,Dn*/

	/*ID: 1793, INST: rol.x Da,Dn*/

	/*ID: 1794, INST: ror.x Da,Dn*/

	/*ID: 1795, INST: rte*/

	/*ID: 1796, INST: rte.cic #u5_t2*/

	/*ID: 1797, INST: rte.stp*/

	/*ID: 1798, INST: rts*/

	/*ID: 1799, INST: rts.nobtb*/

	/*ID: 1800, INST: rts.stk*/

	/*ID: 1801, INST: sad.4b Da,Db,Dn*/
Word32 do__sad_4b(Word32 __Dn, Word32 __Da, Word32 __Db)
{
}

	/*ID: 1802, INST: sad.8b Dab,Dcd,Dn*/
void do__sad_8b(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dn)
{
}

	/*ID: 1803, INST: sat.l.w Da,Dn*/
Word40 do__sat_l_w(Word32 __Da)
{
}

	/*ID: 1804, INST: sat.l.u.w Da,Dn*/
UWord32 do__sat_l_u_w(UWord32 __Da)
{
}

	/*ID: 1805, INST: sat.ll.x Dab,Dn*/
#if defined(_CW_X_SLLD_) || !defined(_ENTERPRISE_C_)
Word40 do__sat_ll_x(Int64 __Dab)
{
}
#endif

	/*ID: 1806, INST: sat.sc.f Da,Dn*/
Word40 do__sat_sc_f(Word40 __Da)
{
	long long __Da_ll,__Db_ll;
	Word32 __Da_32,__Db_32;
	Word40 __Dm;
	if(SaturationMode == 1)
	{
		__Dm = do__ash_rgt_x(16,__Da);
		__Dm = do__sat_x_w(__Dm);
		__Dm = do__ash_lft_x(16,__Dm);
		return __Dm;
	}
	else
	{
		if(ScalingMode == NO_SCALING)
			__Dm = __Da;
		else if(ScalingMode == DOWN_1)
			__Dm = do__ash_rgt_x(1,__Da);
		else if(ScalingMode == UP_1)
			__Dm = do__ash_lft_x(1,__Da);

		__Dm = do__ash_rgt_x(16,__Dm);
		__Dm = do__sat_x_w(__Dm);
		__Dm = do__ash_lft_x(16,__Dm);
		return __Dm;
		}

    }

	/*ID: 1807, INST: sat.sc.2f Dab,Dmn*/
void do__sat_sc_2f(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm =  do__sat_sc_f(__Da);
	*__Dn =  do__sat_sc_f(__Db);
}

	/*ID: 1808, INST: sat.t.2w Da,Dn*/
Word40 do__sat_t_2w(Word40 __Da)
{
   Word40 __Dn;
   Word8 __Da_WH_ext, __Da_WL_ext;
   Word32 __Da_WH, __Da_WL, __Dn_WH, __Dn_WL;


	__Da_WH_ext = __x_guard(__Da) >> 4;
	__Da_WH =(__x_to_l(__Da) & 0xFFFF0000) >> 16 | ((__Da_WH_ext << 16) & 0xf0000);

	__Da_WL_ext = (char)(__x_guard(__Da) << 4) >> 4;
	__Da_WL = (__x_to_l(__Da) & 0xffff) | ((__Da_WL_ext << 16) & 0xf0000);

	__Da_WH |= ((__Da_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result*/
	__Da_WL |= ((__Da_WL >> 19) == 1) ? 0xfff00000:0x0;

    if (__Da_WL > 0x7FFF) {
      __Dn_WL = 0x07ffF;
    } else if ( __Da_WL < (Word32)(0xffff8000)) {
      __Dn_WL = 0xf8000;
    } else {
     __Dn_WL = __Da_WL;
    }

    if (__Da_WH > 0x07FFF)
      __Dn_WH = 0x07FFF;
    else if (__Da_WH < (Word32)(0xffff8000))
      __Dn_WH = 0xF8000;
    else __Dn_WH = __Da_WH;

	__Dn = __x_set( ((__Dn_WH>>16)<<4)| (__Dn_WL>>16 & 0x0f), (__Dn_WH << 16) | (__Dn_WL & 0xFFFF));

	return __Dn;
}

	/*ID: 1809, INST: sat.t.4w Dab,Dmn*/
void do__sat_t_4w(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 1810, INST: sat.t.u.2w Da,Dn*/
UWord32 do__sat_t_u_2w(UWord40 __Da)
{
   Word32 Da_WH, Da_WL;
   UWord16 Dn_WH, Dn_WL;

   Da_WH = (((__ux_guard(__Da) & 0xF0) >> 4) << 16) | ((__ux_to_ul(__Da) & 0xffff0000)>> 16);
   Da_WL = (((__ux_guard(__Da) & 0xF)) << 16) | (__ux_to_ul(__Da) & 0xffff);

   if ((Da_WH >> 19) == 1) { /* neg sat */
	 Dn_WH = 0;
   } else if (Da_WH > 0xffff) { /* pos sat */
     Dn_WH = 0xffff;
   } else {
     Dn_WH = Da_WH;
   }

   if ((Da_WL >> 19) == 1) {
	 Dn_WL = 0;
   } else if (Da_WL > 0xffff) {
     Dn_WL = 0xffff;
   } else {
     Dn_WL = Da_WL;
   }
   return ((Dn_WH << 16)& 0xffff0000) | (Dn_WL);
}

	/*ID: 1811, INST: sat.w.2b Da,Dn*/
Word32 do__sat_w_2b(Word32 __Da)
{
}

	/*ID: 1812, INST: sat.w.u.4b Dab,Dmn*/
void do__sat_w_u_4b(UWord32 __Da, UWord32 __Db, UWord32* __Dm, UWord32* __Dn)
{
}

	/*ID: 1813, INST: sat.x.f Da,Dn*/
Word32 do__sat_x_f(Word40 __Da)
{
   Word8 ext = __x_guard(__Da);
   Word32 __Dn = __x_to_l(__Da);
   Word40 Dn_1;

   Dn_1 = do__cmpd_gt_x(__Da, __x_set(0x0, 0x7fffffff));
   if ((__x_guard(Dn_1) == 0) && (__x_to_l(Dn_1) == 0))
     return __x_to_l(__x_set(0x00, 0x7FFF0000));

   Dn_1 =   do__cmpd_gt_x(__x_set(0xff, 0x80000000), __Da);
   if ((__x_guard(Dn_1) == 0) && (__x_to_l(Dn_1) == 0))
     return __x_to_l(__x_set(0xFF, 0x80000000));

   return __x_to_l(__x_set(ext & 0xFF, __Dn &0xFFFF0000));
}

/* ID: 1814, INST: sat.x.l Da,Dn */
/*
  Function Name: do__sat_x_l
  Purpose: Saturate the input 40 bit data to 32 bit.
  Complexity Weight:
  Inputs: __Da is a 40 bit signed integer whose value falls in the range 0x80 0000 0000 - 0x7F FFFF FFFF
  Outputs:
  Return Value: It returns a 32 bit result.
*/

	/*ID: 1814, INST: sat.x.l Da,Dn*/
Word32 do__sat_x_l(Word40 __Da)
{
 if (( ( __x_guard(__Da) == 0)  && ((__x_to_l(__Da) & 0x80000000) == 0) )
  || ( (__x_guard(__Da) == -1) && (__x_to_l(__Da) & 0x80000000) ))
 {
  return  __x_to_l(__Da);
 }
 else
 {
   shift_Overflow = 1; /* don't comment this ..as this is used in shift intrinsics*/
  if (__x_guard(__Da) >= 0)
   return MAX_32;
  else
   return MIN_32;
 }
}

	/*ID: 1815, INST: sat.x.2l Dab,Dmn*/
void do__sat_x_2l(Word40 __Da, Word40 __Db, Word32* __Dm, Word32* __Dn)
{
 if (( ( __x_guard(__Da) == 0)  && ((__x_to_l(__Da) & 0x80000000) == 0) )
  || ( (__x_guard(__Da) == -1) && (__x_to_l(__Da) & 0x80000000) ))
 {
	  *__Dm =  __x_to_l(__Da);
 }
 else
 {
   /*   Overflow = 1;  handle later.*/
  if (__x_guard(__Da) >= 0)
   *__Dm =  MAX_32;
  else
   *__Dm = MIN_32;
 }
  if (( ( __x_guard(__Db) == 0)  && ((__x_to_l(__Db) & 0x80000000) == 0) )
   || ( (__x_guard(__Db) == -1) && (__x_to_l(__Db) & 0x80000000) ))
  {
 	  *__Dn =  __x_to_l(__Db);
  }
  else
  {
    /*  Overflow = 1; handle later.*/
   if (__x_guard(__Db) >= 0)
    *__Dn =  MAX_32;
   else
    *__Dn = MIN_32;
 }
}

	/*ID: 1816, INST: sat.x.w Da,Dn*/
Word40 do__sat_x_w(Word40 __Da)
{
  Word40 Dn_1;

  Dn_1 = do__cmpd_gt_x(__Da, __x_set(0x0, 0x00007fff));
  if ((__x_guard(Dn_1) == 0) && (__x_to_l(Dn_1) == 0))
    return (__x_set(0x00, 0x00007FFF));

  Dn_1 =   do__cmpd_gt_x(__x_set(0xff, 0xffff8000), __Da);
  if ((__x_guard(Dn_1) == 0) && (__x_to_l(Dn_1) == 0))
    return (__x_set(0xFF, 0xFFFF8000));

  return __Da;
}

	/*ID: 1817, INST: sat.x.u.l Da,Dn*/
UWord32 do__sat_x_u_l(UWord40 __Da)
{
}

	/*ID: 1818, INST: sat.x.u.2w Dab,Dmn*/
/*
  Function Name: do__sat_x_u_2w
  Purpose: Saturate the input unsigned 40 bit data to unsigned 16 bit.
  Complexity Weight:
  Inputs: __Da is a 40 bit unsigned integer whose value falls in the range 0x00 0000 0000 - 0xFF FFFF FFFF
  		  __Db is a 40 bit unsigned integer whose value falls in the range 0x00 0000 0000 - 0xFF FFFF FFFF
  Outputs:
  Return Value:
*/
void do__sat_x_u_2w(UWord40 __Da, UWord40 __Db, UWord32* __Dm, UWord32* __Dn)
{
  if(__ux_guard(__Da) == 0)
  {
	if(__ux_to_ul(__Da) > 0xffff) * __Dm = 0xffff;
	else
		*__Dm = __ux_to_ul(__Da);
  }
  else
  	 * __Dm = 0xffff;

  if(__ux_guard(__Db) == 0)
  {
	if(__ux_to_ul(__Db) > 0xffff) * __Dn = 0xffff;
	else
		*__Dn = __ux_to_ul(__Db);
  }
  else
  	 * __Dn = 0xffff;
}

	/*ID: 1819, INST: setalign Ra,GCR.BAM*/
void do__setalign(Word32 __Ra)
{
}

	/*ID: 1820, INST: sign.4w Dab,Dmn*/
void do__sign_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1821, INST: skip.2 RelAdd19_t1*/

	/*ID: 1822, INST: skip.1 RelAdd19_t1*/

	/*ID: 1823, INST: skip.3 RelAdd19_t1*/

	/*ID: 1824, INST: skip.0 RelAdd19_t1*/

	/*ID: 1825, INST: skip.0.u RelAdd19_t1*/

	/*ID: 1826, INST: skip.1.u RelAdd19_t1*/

	/*ID: 1827, INST: skip.2.u RelAdd19_t1*/

	/*ID: 1828, INST: skip.3.u RelAdd19_t1*/

	/*ID: 1829, INST: sod.aaaai.4t Dab,Dcd,Dmn*/
void do__sod_aaaai_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word32 Da_WH,Db_WH,Da_WL,Db_WL,Dm_WH,Dm_WL,Dn_WH,Dn_WL;

	Da_WH = __extract_wh(__Da);
	Db_WH = __extract_wh(__Db);
	Da_WL = __extract_wl(__Da);
	Db_WL = __extract_wl(__Db);

	Dm_WH = __extract_wh(__Dc) + Da_WH;
	Dm_WL = __extract_wl(__Dc) + Da_WL;
	Dn_WH = __extract_wh(__Dd) + Db_WH;
	Dn_WL = __extract_wl(__Dd) + Db_WL;

	__set_WH(Dm_WH,__Dm);
	__set_WL(Dm_WL,__Dm);
	__set_WH(Dn_WH,__Dn);
	__set_WL(Dn_WL,__Dn);
}

	/*ID: 1830, INST: sod.aaaai.s.4w Dab,Dcd,Dmn*/
void do__sod_aaaai_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_msb(__Da);
    src2 = __l_get_lsb(__Da);
    src3 = __l_get_msb(__Db);
    src4 = __l_get_lsb(__Db);

	src1 = __l_get_msb(__Dc) + src1 ;
    src2 = __l_get_lsb(__Dc) + src2 ;
    src3 = __l_get_msb(__Dd) + src3 ;
    src4 = __l_get_lsb(__Dd) + src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);
}

	/*ID: 1831, INST: sod.aaaax.4t Dab,Dcd,Dmn*/
	/*ADL :SOD_ffffc_4T_Da_Db_Dc_Dd_Dm_Dn_family*/
void do__sod_aaaax_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;
    Word32 Res1WH , Res1WL , Res2WH , Res2WL ;

    src1 = do__extract_wl(__Da);
    src2 = do__extract_wh(__Da);
    src3 = do__extract_wl(__Db);
    src4 = do__extract_wh(__Db);

    Res1WH = do__extract_wh(__Dc) + src1 ;
    Res1WL = do__extract_wl(__Dc) + src2 ;
    Res2WH = do__extract_wh(__Dd) + src3 ;
    Res2WL = do__extract_wl(__Dd) + src4 ;

	do__set_WH(Res1WH,__Dm);
	do__set_WL(Res1WL,__Dm);
	do__set_WH(Res2WH,__Dn);
	do__set_WL(Res2WL,__Dn);
}

	/*ID: 1832, INST: sod.aaaax.s.4w Dab,Dcd,Dmn*/
void do__sod_aaaax_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_lsb(__Da);
	src2 = __l_get_msb(__Da);
	src3 = __l_get_lsb(__Db);
	src4 = __l_get_msb(__Db);

    src1 = __l_get_msb(__Dc) + src1 ;
    src2 = __l_get_lsb(__Dc) + src2 ;
    src3 = __l_get_msb(__Dd) + src3 ;
    src4 = __l_get_lsb(__Dd) + src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);
}

	/*ID: 1833, INST: sod.aaii.2t Da,Db,Dn*/
Word40 do__sod_aaii_2t(Word40 __Da, Word40 __Db)
{
   Word40 result40;
   Word32 result32;
   Word16 Dn_H, Dn_L;

   Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
   Word40 __Dn;
   Da_WH = __extract_wh(__Da);
   Db_WH = __extract_wh(__Db);
   Da_WL = __extract_wl(__Da);
   Db_WL = __extract_wl(__Db);

   Dn_WH = Db_WH+Da_WH;
   Dn_WL = Db_WL+Da_WL;

   Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
   Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
   __Dn = __x_set( ((Dn_WH>>16 & 0x0f)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
   return __Dn;
}

	/*ID: 1834, INST: sod.aaii.2x Dab,Dcd,Dmn*/
void do__sod_aaii_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x(__Da,__Dc);
	*__Dn = do__add_x(__Db,__Dd);
}

	/*ID: 1835, INST: sod.aaii.s.2w Da,Db,Dn*/
Word32 do__sod_aaii_s_2w(Word32 __Da, Word32 __Db)
{
  Word16 Dn_H, Dn_L;

  Dn_H = __saturate(__l_get_msb(__Da) + __l_get_msb(__Db));
  Dn_L = __saturate(__l_get_lsb(__Da) + __l_get_lsb(__Db));

  return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1836, INST: sod.aaii.s.2x Dab,Dcd,Dmn*/
void do__sod_aaii_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__add_x(__Da,__Dc)));
	*__Dn = __l_to_x(do__sat_x_l(do__add_x(__Db,__Dd)));
}

	/*ID: 1837, INST: sod.aassi.4t Da,Db,Dmn*/
void do__sod_aassi_4t(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	Word32 Da_WH,Db_WH,Da_WL,Db_WL,Dm_WH,Dm_WL,Dn_WH,Dn_WL;

	Da_WH = __extract_wh(__Da);
	Db_WH = __extract_wh(__Db);
	Da_WL = __extract_wl(__Da);
	Db_WL = __extract_wl(__Db);

	Dm_WH = Db_WH + Da_WH;
	Dm_WL = Db_WL + Da_WL;
	Dn_WH = Db_WH - Da_WH;
	Dn_WL = Db_WL - Da_WL;

	__set_WH(Dm_WH,__Dm);
	__set_WL(Dm_WL,__Dm);
	__set_WH(Dn_WH,__Dn);
	__set_WL(Dn_WL,__Dn);
}

	/*ID: 1838, INST: sod.aassi.s.4w Da,Db,Dmn*/
void do__sod_aassi_s_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{

}

	/*ID: 1839, INST: sod.aaxx.2t Da,Db,Dn*/
Word40 do__sod_aaxx_2t(Word40 __Da, Word40 __Db)
{
   Word40 result40;
   Word32 result32;
   Word16 Dn_H, Dn_L;
   Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
   Word40 __Dn;
   Da_WH = __extract_wh(__Da);
   Db_WH = __extract_wh(__Db);
   Da_WL = __extract_wl(__Da);
   Db_WL = __extract_wl(__Db);

   Dn_WH = Db_WH+Da_WL;
   Dn_WL = Db_WL+Da_WH;

   Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
   Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
  __Dn = __x_set( ((Dn_WH>>16 & 0x0f)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
   return __Dn;
}

	/*ID: 1840, INST: sod.aaxx.2x Dab,Dcd,Dmn*/
void do__sod_aaxx_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x(__Dc,__Db);
	*__Dn = do__add_x(__Dd,__Da);
}

	/*ID: 1841, INST: sod.aaxx.s.2w Da,Db,Dn*/
Word32 do__sod_aaxx_s_2w(Word32 __Da, Word32 __Db)
{
  Word16 Dn_H, Dn_L;

  Dn_H = __saturate(__l_get_lsb(__Da) + __l_get_msb(__Db));
  Dn_L = __saturate(__l_get_msb(__Da) + __l_get_lsb(__Db));

  return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1842, INST: sod.aaxx.s.2x Dab,Dcd,Dmn*/
void do__sod_aaxx_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__add_x(__Dc,__Db)));
	*__Dn = __l_to_x(do__sat_x_l(do__add_x(__Dd,__Da)));
}

	/*ID: 1843, INST: sod.as.2x Da,Db,Dmn*/
void do__sod_as_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x(__Db,__Da);
	*__Dn = do__sub_x(__Da,__Db);
}

	/*ID: 1844, INST: sod.as.s.2x Da,Db,Dmn*/
void do__sod_as_s_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__add_x(__Db,__Da)));
	*__Dn = __l_to_x(do__sat_x_l(do__sub_x(__Da,__Db)));
}

	/*ID: 1845, INST: sod.asasi.4t Dab,Dcd,Dmn*/
void do__sod_asasi_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
   Word32 src1 , src2 , src3 , src4 ;
   Word32 Res1WH , Res1WL , Res2WH , Res2WL ;

   src1 = do__extract_wh(__Da);
   src2 = do__extract_wl(__Da);
   src3 = do__extract_wh(__Db);
   src4 = do__extract_wl(__Db);

   Res1WH = do__extract_wh(__Dc) + src1 ;
   Res1WL = do__extract_wl(__Dc) - src2 ;
   Res2WH = do__extract_wh(__Dd) + src3 ;
   Res2WL = do__extract_wl(__Dd) - src4 ;

	do__set_WH(Res1WH,__Dm);
	do__set_WL(Res1WL,__Dm);
	do__set_WH(Res2WH,__Dn);
	do__set_WL(Res2WL,__Dn);
}

	/*ID: 1846, INST: sod.asasi.s.4w Dab,Dcd,Dmn*/
void do__sod_asasi_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_msb(__Da);
    src2 = __l_get_lsb(__Da);
    src3 = __l_get_msb(__Db);
    src4 = __l_get_lsb(__Db);

	src1 = __l_get_msb(__Dc) + src1 ;
    src2 = __l_get_lsb(__Dc) - src2 ;
    src3 = __l_get_msb(__Dd) + src3 ;
    src4 = __l_get_lsb(__Dd) - src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);
}

	/*ID: 1847, INST: sod.asasx.4t Dab,Dcd,Dmn*/
void do__sod_asasx_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;
    Word32 Res1WH , Res1WL , Res2WH , Res2WL ;

    src1 = do__extract_wl(__Da);
    src2 = do__extract_wh(__Da);
    src3 = do__extract_wl(__Db);
    src4 = do__extract_wh(__Db);

	Res1WH = do__extract_wh(__Dc) + src1 ;
    Res1WL = do__extract_wl(__Dc) - src2 ;
    Res2WH = do__extract_wh(__Dd) + src3 ;
    Res2WL = do__extract_wl(__Dd) - src4 ;

	do__set_WH(Res1WH,__Dm);
	do__set_WL(Res1WL,__Dm);
	do__set_WH(Res2WH,__Dn);
	do__set_WL(Res2WL,__Dn);
}

	/*ID: 1848, INST: sod.asasx.s.4w Dab,Dcd,Dmn*/
void do__sod_asasx_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_lsb(__Da);
    src2 = __l_get_msb(__Da);
    src3 = __l_get_lsb(__Db);
    src4 = __l_get_msb(__Db);

    src1 = __l_get_msb(__Dc) + src1 ;
    src2 = __l_get_lsb(__Dc) - src2 ;
    src3 = __l_get_msb(__Dd) + src3 ;
    src4 = __l_get_lsb(__Dd) - src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);
}

	/*ID: 1849, INST: sod.asii.2t Da,Db,Dn*/
Word40 do__sod_asii_2t(Word40 __Da, Word40 __Db)
{
   Word40 result40;
   Word32 result32;
   Word16 Dn_H, Dn_L;

   Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
   Word40 __Dn;
   Da_WH = __extract_wh(__Da);
   Db_WH = __extract_wh(__Db);
   Da_WL = __extract_wl(__Da);
   Db_WL = __extract_wl(__Db);

   Dn_WH = Db_WH+Da_WH;
   Dn_WL = Db_WL-Da_WL;

   Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /*sign extend the result */
   Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
   __Dn = __x_set( ((Dn_WH>>16 & 0x0f)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
	return __Dn;
}

	/*ID: 1850, INST: sod.asii.2x Dab,Dcd,Dmn*/
void do__sod_asii_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x(__Dc,__Da);
	*__Dn = do__sub_x(__Db,__Dd);
}

	/*ID: 1851, INST: sod.asii.s.2w Da,Db,Dn*/
Word32 do__sod_asii_s_2w(Word32 __Da, Word32 __Db)
{
 Word16 Dn_H, Dn_L;

  Dn_H = __saturate(__l_get_msb(__Da) + __l_get_msb(__Db));
  Dn_L = __saturate(__l_get_lsb(__Db) - __l_get_lsb(__Da));

  return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1852, INST: sod.asii.s.2x Dab,Dcd,Dmn*/
void do__sod_asii_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__add_x(__Da,__Dc)));
	*__Dn = __l_to_x(do__sat_x_l(do__sub_x(__Db,__Dd)));
}

	/*ID: 1853, INST: sod.asxx.2t Da,Db,Dn*/
Word40 do__sod_asxx_2t(Word40 __Da, Word40 __Db)
{
   Word40 result40;
   Word32 result32;
   Word16 Dn_H, Dn_L;

   Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
   Word40 __Dn;
   Da_WH = __extract_wh(__Da);
   Db_WH = __extract_wh(__Db);
   Da_WL = __extract_wl(__Da);
   Db_WL = __extract_wl(__Db);

   Dn_WH = Db_WH+Da_WL;
   Dn_WL = Db_WL-Da_WH;

   Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result*/
   Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
   __Dn = __x_set( ((Dn_WH>>16 & 0x0f)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
	return __Dn;
}

	/*ID: 1854, INST: sod.asxx.2x Dab,Dcd,Dmn*/
void do__sod_asxx_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__add_x(__Dc,__Db);
	*__Dn = do__sub_x(__Da,__Dd);
}

	/*ID: 1855, INST: sod.asxx.s.2w Da,Db,Dn*/
Word32 do__sod_asxx_s_2w(Word32 __Da, Word32 __Db)
{
Word16 Dn_H, Dn_L;

Dn_H = __ssodffcc_high(__Da, __Db, 5);
Dn_L = __ssodffcc_low(__Da, __Db, 5);

return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1856, INST: sod.asxx.s.2x Dab,Dcd,Dmn*/
void do__sod_asxx_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__add_x(__Dc,__Db)));
	*__Dn = __l_to_x(do__sat_x_l(do__sub_x(__Da,__Dd)));
}

	/*ID: 1857, INST: sod.sa.2x Da,Db,Dmn*/
void do__sod_sa_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__sub_x(__Da,__Db);
	*__Dn = do__add_x(__Db,__Da);
}

	/*ID: 1858, INST: sod.sa.s.2x Da,Db,Dmn*/
void do__sod_sa_s_2x(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__sub_x(__Da,__Db)));
	*__Dn = __l_to_x(do__sat_x_l(do__add_x(__Db,__Da)));
}

	/*ID: 1859, INST: sod.saii.2t Da,Db,Dn*/
Word40 do__sod_saii_2t(Word40 __Da, Word40 __Db)
{
      Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
      Word40 __Dn;
      Da_WH = __extract_wh(__Da);
      Db_WH = __extract_wh(__Db);
      Da_WL = __extract_wl(__Da);
      Db_WL = __extract_wl(__Db);

      Dn_WH = Db_WH-Da_WH;
      Dn_WL = Db_WL+Da_WL;

      Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
	  Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
	  __Dn = __x_set(((Dn_WH>>16)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));

  	  return __Dn;
}

	/*ID: 1860, INST: sod.saii.2x Dab,Dcd,Dmn*/
void do__sod_saii_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__sub_x(__Da,__Dc);
	*__Dn = do__add_x(__Dd,__Db);
}

	/*ID: 1861, INST: sod.saii.s.2w Da,Db,Dn*/
Word32 do__sod_saii_s_2w(Word32 __Da, Word32 __Db)
{
  Word16 Dn_H, Dn_L;

  Dn_H = __saturate(__l_get_msb(__Db) - __l_get_msb(__Da));
  Dn_L = __saturate(__l_get_lsb(__Da) + __l_get_lsb(__Db));

  return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1862, INST: sod.saii.s.2x Dab,Dcd,Dmn*/
void do__sod_saii_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__sub_x(__Da,__Dc)));
	*__Dn = __l_to_x(do__sat_x_l(do__add_x(__Dd,__Db)));
}

	/*ID: 1863, INST: sod.sasai.4t Dab,Dcd,Dmn*/
void do__sod_sasai_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;
    Word32 Res1WH , Res1WL , Res2WH , Res2WL ;

    src1 = do__extract_wh(__Da);
    src2 = do__extract_wl(__Da);
    src3 = do__extract_wh(__Db);
    src4 = do__extract_wl(__Db);

 	Res1WH = do__extract_wh(__Dc) - src1 ;
    Res1WL = do__extract_wl(__Dc) + src2 ;
    Res2WH = do__extract_wh(__Dd) - src3 ;
    Res2WL = do__extract_wl(__Dd) + src4 ;

 	do__set_WH(Res1WH,__Dm);
	do__set_WL(Res1WL,__Dm);
	do__set_WH(Res2WH,__Dn);
	do__set_WL(Res2WL,__Dn);
}

	/*ID: 1864, INST: sod.sasai.s.4w Dab,Dcd,Dmn*/
void do__sod_sasai_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_msb(__Da);
    src2 = __l_get_lsb(__Da);
    src3 = __l_get_msb(__Db);
    src4 = __l_get_lsb(__Db);

    src1 = __l_get_msb(__Dc) - src1 ;
    src2 = __l_get_lsb(__Dc) + src2 ;
    src3 = __l_get_msb(__Dd) - src3 ;
    src4 = __l_get_lsb(__Dd) + src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);
}

	/*ID: 1865, INST: sod.sasax.4t Dab,Dcd,Dmn*/
void do__sod_sasax_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;
    Word32 Res1WH , Res1WL , Res2WH , Res2WL ;

 	src1 = do__extract_wl(__Da);
    src2 = do__extract_wh(__Da);
    src3 = do__extract_wl(__Db);
    src4 = do__extract_wh(__Db);

 	Res1WH = do__extract_wh(__Dc) - src1 ;
    Res1WL = do__extract_wl(__Dc) + src2 ;
    Res2WH = do__extract_wh(__Dd) - src3 ;
    Res2WL = do__extract_wl(__Dd) + src4 ;

 	do__set_WH(Res1WH,__Dm);
	do__set_WL(Res1WL,__Dm);
	do__set_WH(Res2WH,__Dn);
	do__set_WL(Res2WL,__Dn);
}

	/*ID: 1866, INST: sod.sasax.s.4w Dab,Dcd,Dmn*/
void do__sod_sasax_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 1867, INST: sod.saxx.2t Da,Db,Dn*/
Word40 do__sod_saxx_2t(Word40 __Da, Word40 __Db)
{
      Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
      Word40 __Dn;
      Da_WH = __extract_wh(__Da);
      Db_WH = __extract_wh(__Db);
      Da_WL = __extract_wl(__Da);
      Db_WL = __extract_wl(__Db);

      Dn_WH = Db_WH - Da_WL;
      Dn_WL = Db_WL + Da_WH;

      Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
	  Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
	  __Dn = __x_set(((Dn_WH>>16)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));

  	  return __Dn;
}

	/*ID: 1868, INST: sod.saxx.2x Dab,Dcd,Dmn*/
void do__sod_saxx_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__sub_x(__Db,__Dc);
	*__Dn = do__add_x(__Dd,__Da);
}

	/*ID: 1869, INST: sod.saxx.s.2w Da,Db,Dn*/
Word32 do__sod_saxx_s_2w(Word32 __Da, Word32 __Db)
{
	Word16 Dn_H, Dn_L;

Dn_H = __ssodffcc_high(__Da, __Db, 3);
Dn_L = __ssodffcc_low(__Da, __Db, 3);

return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1870, INST: sod.saxx.s.2x Dab,Dcd,Dmn*/
void do__sod_saxx_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__sub_x(__Db,__Dc)));
	*__Dn = __l_to_x(do__sat_x_l(do__add_x(__Dd,__Da)));
}

	/*ID: 1871, INST: sod.ssaai.4t Da,Db,Dmn*/
void do__sod_ssaai_4t(Word40 __Da, Word40 __Db, Word40* __Dm, Word40* __Dn)
{

}

	/*ID: 1872, INST: sod.ssaai.s.4w Da,Db,Dmn*/
void do__sod_ssaai_s_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{

}

	/*ID: 1873, INST: sod.ssii.2t Da,Db,Dn*/
Word40 do__sod_ssii_2t(Word40 __Da, Word40 __Db)
{

      Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
      Word40 __Dn;
      Da_WH = __extract_wh(__Da);
      Db_WH = __extract_wh(__Db);
      Da_WL = __extract_wl(__Da);
      Db_WL = __extract_wl(__Db);

      Dn_WH = Db_WH - Da_WH;
      Dn_WL = Db_WL - Da_WL;

      Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /* sign extend the result */
	  Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
	  __Dn = __x_set(((Dn_WH>>16)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));

  	  return __Dn;
}

	/*ID: 1874, INST: sod.ssii.2x Dab,Dcd,Dmn*/
void do__sod_ssii_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__sub_x(__Da,__Dc);
	*__Dn = do__sub_x(__Db,__Dd);
}

	/*ID: 1875, INST: sod.ssii.s.2w Da,Db,Dn*/
Word32 do__sod_ssii_s_2w(Word32 __Da, Word32 __Db)
{
  Word16 Dn_H, Dn_L;

  Dn_H = __saturate(__l_get_msb(__Db) - __l_get_msb(__Da));
  Dn_L = __saturate(__l_get_lsb(__Db) - __l_get_lsb(__Da));

  return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1876, INST: sod.ssii.s.2x Dab,Dcd,Dmn*/
void do__sod_ssii_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__sub_x(__Da,__Dc)));
	*__Dn = __l_to_x(do__sat_x_l(do__sub_x(__Db,__Dd)));
}

	/*ID: 1877, INST: sod.ssssi.4t Dab,Dcd,Dmn*/
void do__sod_ssssi_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word32 Da_WH,Db_WH,Da_WL,Db_WL,Dm_WH,Dm_WL,Dn_WH,Dn_WL;

	Da_WH = __extract_wh(__Da);
	Db_WH = __extract_wh(__Db);
	Da_WL = __extract_wl(__Da);
	Db_WL = __extract_wl(__Db);

	Dm_WH = __extract_wh(__Dc) - Da_WH;
	Dm_WL = __extract_wl(__Dc) - Da_WL;
	Dn_WH = __extract_wh(__Dd) - Db_WH;
	Dn_WL = __extract_wl(__Dd) - Db_WL;

	__set_WH(Dm_WH,__Dm);
	__set_WL(Dm_WL,__Dm);
	__set_WH(Dn_WH,__Dn);
	__set_WL(Dn_WL,__Dn);
}

	/*ID: 1878, INST: sod.ssssi.s.4w Dab,Dcd,Dmn*/
void do__sod_ssssi_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_msb(__Da);
    src2 = __l_get_lsb(__Da);
    src3 = __l_get_msb(__Db);
    src4 = __l_get_lsb(__Db);

	src1 = __l_get_msb(__Dc) - src1 ;
    src2 = __l_get_lsb(__Dc) - src2 ;
    src3 = __l_get_msb(__Dd) - src3 ;
    src4 = __l_get_lsb(__Dd) - src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);

}

	/*ID: 1879, INST: sod.ssssx.4t Dab,Dcd,Dmn*/
void do__sod_ssssx_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;
    Word32 Res1WH , Res1WL , Res2WH , Res2WL ;

        src1 = do__extract_wl(__Da);
        src2 = do__extract_wh(__Da);
        src3 = do__extract_wl(__Db);
        src4 = do__extract_wh(__Db);

		Res1WH = do__extract_wh(__Dc) - src1 ;
        Res1WL = do__extract_wl(__Dc) - src2 ;
        Res2WH = do__extract_wh(__Dd) - src3 ;
        Res2WL = do__extract_wl(__Dd) - src4 ;

		do__set_WH(Res1WH,__Dm);
		do__set_WL(Res1WL,__Dm);
		do__set_WH(Res2WH,__Dn);
		do__set_WL(Res2WL,__Dn);
}

	/*ID: 1880, INST: sod.ssssx.s.4w Dab,Dcd,Dmn*/
void do__sod_ssssx_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
    Word32 src1 , src2 , src3 , src4 ;

    src1 = __l_get_lsb(__Da);
    src2 = __l_get_msb(__Da);
    src3 = __l_get_lsb(__Db);
    src4 = __l_get_msb(__Db);

	src1 = __l_get_msb(__Dc) - src1 ;
    src2 = __l_get_lsb(__Dc) - src2 ;
    src3 = __l_get_msb(__Dd) - src3 ;
    src4 = __l_get_lsb(__Dd) - src4 ;

	*__Dm = (do__saturate(src1)&0x0000ffff)<<16;
	*__Dm = (*__Dm&0xffff0000) | (do__saturate(src2)&0xffff);
	*__Dn = (do__saturate(src3)&0x0000ffff)<<16;
	*__Dn = (*__Dn&0xffff0000) | (do__saturate(src4)&0xffff);
}

	/*ID: 1881, INST: sod.ssxx.2t Da,Db,Dn*/
Word40 do__sod_ssxx_2t(Word40 __Da, Word40 __Db)
{
   Word40 result40;
   Word32 result32;
   Word16 Dn_H, Dn_L;

   Word32 Da_WH, Da_WL, Db_WH,Db_WL , Dn_WH, Dn_WL;
   Word40 __Dn;
   Da_WH = __extract_wh(__Da);
   Db_WH = __extract_wh(__Db);
   Da_WL = __extract_wl(__Da);
   Db_WL = __extract_wl(__Db);

   Dn_WH = Db_WH - Da_WL;
   Dn_WL = Db_WL - Da_WH;

   Dn_WH |= ((Dn_WH >> 19) == 1) ? 0xfff00000:0x0; /*sign extend the result */
   Dn_WL |= ((Dn_WL >> 19) == 1) ? 0xfff00000:0x0;
   __Dn = __x_set( ((Dn_WH>>16 & 0x0f)<<4)| (Dn_WL>>16 & 0x0f), (Dn_WH << 16) | (Dn_WL & 0xFFFF));
	return __Dn;
}

	/*ID: 1882, INST: sod.ssxx.2x Dab,Dcd,Dmn*/
void do__sod_ssxx_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = do__sub_x(__Db,__Dc);
	*__Dn = do__sub_x(__Da,__Dd);
}

	/*ID: 1883, INST: sod.ssxx.s.2w Da,Db,Dn*/
Word32 do__sod_ssxx_s_2w(Word32 __Da, Word32 __Db)
{
  Word16 Dn_H, Dn_L;

  Dn_H = __saturate(__l_get_msb(__Db) - __l_get_lsb(__Da));
  Dn_L = __saturate(__l_get_lsb(__Db) - __l_get_msb(__Da));

  return (((Dn_H << 16)& 0xffff0000) | (Dn_L & 0xffff));
}

	/*ID: 1884, INST: sod.ssxx.s.2x Dab,Dcd,Dmn*/
void do__sod_ssxx_s_2x(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	*__Dm = __l_to_x(do__sat_x_l(do__sub_x(__Db,__Dc)));
	*__Dn = __l_to_x(do__sat_x_l(do__sub_x(__Da,__Dd)));
}

	/*ID: 1885, INST: sodhl.asas.4t Dab,Dcd,Dmn*/
void do__sodhl_asas_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{
	Word32 Da_WH,Db_WH,Dc_WL,Dd_WL,Dm_WH,Dm_WL,Dn_WH,Dn_WL;

	Da_WH = __extract_wh(__Da);
	Db_WH = __extract_wh(__Db);
	Dc_WL = __extract_wl(__Dc);
	Dd_WL = __extract_wl(__Dd);
	Dm_WH = Da_WH + Dc_WL;
	Dm_WL = Da_WH - Dc_WL;
	Dn_WH = Db_WH + Dd_WL;
	Dn_WL = Db_WH - Dd_WL;

	__set_WH(Dm_WH,__Dm);
	__set_WL(Dm_WL,__Dm);
	__set_WH(Dn_WH,__Dn);
	__set_WL(Dn_WL,__Dn);
}

	/*ID: 1886, INST: sodhl.asas.s.4w Dab,Dcd,Dmn*/
void do__sodhl_asas_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
/*     sbits < 40 > DRes1WH = ( sbits < 40 >  ) D ( Src1 ) . H + ( sbits < 40 >  ) D ( Src3 ) . L ;
     sbits < 40 > DRes1WL = ( sbits < 40 >  ) D ( Src1 ) . H - ( sbits < 40 >  ) D ( Src3 ) . L ;
     sbits < 40 > DRes2WH = ( sbits < 40 >  ) D ( Src2 ) . H + ( sbits < 40 >  ) D ( Src4 ) . L ;
     sbits < 40 > DRes2WL = ( sbits < 40 >  ) D ( Src2 ) . H - ( sbits < 40 >  ) D ( Src4 ) . L ;
    D ( Res1 ) . WH = zeroExtend ( SignedSaturation16 ( DRes1WH ) , 20 ) ;
    D ( Res1 ) . WL = zeroExtend ( SignedSaturation16 ( DRes1WL ) , 20 ) ;
    D ( Res2 ) . WH = zeroExtend ( SignedSaturation16 ( DRes2WH ) , 20 ) ;
    D ( Res2 ) . WL = zeroExtend ( SignedSaturation16 ( DRes2WL ) , 20 ) ;*/
   Word40 DRes1WH,DRes1WL,DRes2WH,DRes2WL;
   Word32 __Da_h,__Da_l,__Db_h,__Db_l,__Dc_h,__Dc_l,__Dd_h,__Dd_l;
	__Da_h = __l_get_msb(__Da);
	__Da_l = __l_get_lsb(__Da);
	__Db_h = __l_get_msb(__Db);
	__Db_l = __l_get_lsb(__Db);
	__Dc_h = __l_get_msb(__Dc);
	__Dc_l = __l_get_lsb(__Dc);
	__Dd_h = __l_get_msb(__Dd);
	__Dd_l = __l_get_lsb(__Dd);

	DRes1WH =  __l_to_x(__Da_h + __Dc_l);
	DRes1WL =  __l_to_x(__Da_h - __Dc_l);
	DRes2WH =  __l_to_x(__Db_h + __Dd_l);
	DRes2WL =  __l_to_x(__Db_h - __Dd_l);

	//do__set_WH(__x_to_l(do__sat_x_w(DRes1WH))&0xffff ,__Dm);
	*__Dm = *__Dm | (__x_to_l(do__sat_x_w(DRes1WH))&0xffff)<<16;
	*__Dm = *__Dm | (__x_to_l(do__sat_x_w(DRes1WL))&0xffff);
	*__Dn = *__Dn | (__x_to_l(do__sat_x_w(DRes2WH))&0xffff)<<16;
	*__Dn = *__Dn | (__x_to_l(do__sat_x_w(DRes2WL))&0xffff);
}

	/*ID: 1887, INST: sodhl.sasa.4t Dab,Dcd,Dmn*/
void do__sodhl_sasa_4t(Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40* __Dm, Word40* __Dn)
{

	    Word32 Res1WH,Res1WL,Res2WH,Res2WL;
	    Word32 __Da_WH,__Da_WL,__Db_WH,__Db_WL,__Dc_WH,__Dc_WL,__Dd_WH,__Dd_WL;
	    __Da_WH = do__extract_wh(__Da);
	    __Da_WL = do__extract_wl(__Da);
	    __Db_WH = do__extract_wh(__Db);
	    __Db_WL = do__extract_wl(__Db);
	    __Dc_WH = do__extract_wh(__Dc);
	    __Dc_WL = do__extract_wl(__Dc);
	    __Dd_WH = do__extract_wh(__Dd);
	    __Dd_WL = do__extract_wl(__Dd);

	    Res1WH = __Da_WH - __Dc_WL;
	    Res1WL = __Da_WH + __Dc_WL;
	    Res2WH = __Db_WH - __Dd_WL;
	    Res2WH = __Db_WH + __Dd_WL;

		do__set_WH(Res1WH,__Dm);
		do__set_WL(Res1WL,__Dm);
		do__set_WH(Res2WH,__Dn);
		do__set_WL(Res2WL,__Dn);
}

	/*ID: 1888, INST: sodhl.sasa.s.4w Dab,Dcd,Dmn*/
void do__sodhl_sasa_s_4w(Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32* __Dm, Word32* __Dn)
{
  /*     sbits < 40 > DRes1WH = ( sbits < 40 >  ) D ( Src1 ) . H - ( sbits < 40 >  ) D ( Src3 ) . L ;
     sbits < 40 > DRes1WL = ( sbits < 40 >  ) D ( Src1 ) . H + ( sbits < 40 >  ) D ( Src3 ) . L ;
     sbits < 40 > DRes2WH = ( sbits < 40 >  ) D ( Src2 ) . H - ( sbits < 40 >  ) D ( Src4 ) . L ;
     sbits < 40 > DRes2WL = ( sbits < 40 >  ) D ( Src2 ) . H + ( sbits < 40 >  ) D ( Src4 ) . L ;
    D ( Res1 ) . WH = zeroExtend ( SignedSaturation16 ( DRes1WH ) , 20 ) ;
    D ( Res1 ) . WL = zeroExtend ( SignedSaturation16 ( DRes1WL ) , 20 ) ;
    D ( Res2 ) . WH = zeroExtend ( SignedSaturation16 ( DRes2WH ) , 20 ) ;
    D ( Res2 ) . WL = zeroExtend ( SignedSaturation16 ( DRes2WL ) , 20 ) ;
   */
   Word40 DRes1WH,DRes1WL,DRes2WH,DRes2WL;
   Word32 __Da_h,__Da_l,__Db_h,__Db_l,__Dc_h,__Dc_l,__Dd_h,__Dd_l;
	__Da_h = __l_get_msb(__Da);
	__Da_l = __l_get_lsb(__Da);
	__Db_h = __l_get_msb(__Db);
	__Db_l = __l_get_lsb(__Db);
	__Dc_h = __l_get_msb(__Dc);
	__Dc_l = __l_get_lsb(__Dc);
	__Dd_h = __l_get_msb(__Dd);
	__Dd_l = __l_get_lsb(__Dd);

	DRes1WH =  __l_to_x(__Da_h - __Dc_l);
	DRes1WL =  __l_to_x(__Da_h + __Dc_l);
	DRes2WH =  __l_to_x(__Db_h - __Dd_l);
	DRes2WL =  __l_to_x(__Db_h + __Dd_l);

	//do__set_WH(__x_to_l(do__sat_x_w(DRes1WH))&0xffff ,__Dm);
	*__Dm = *__Dm | (__x_to_l(do__sat_x_w(DRes1WH))&0xffff)<<16;
	*__Dm = *__Dm | (__x_to_l(do__sat_x_w(DRes1WL))&0xffff);
	*__Dn = *__Dn | (__x_to_l(do__sat_x_w(DRes2WH))&0xffff)<<16;
	*__Dn = *__Dn | (__x_to_l(do__sat_x_w(DRes2WL))&0xffff);

}

	/*ID: 1889, INST: st.b Da,(Rn)*/
void do__st_b(Word8* __p, Word32 __Da)
{
	*__p = (__Da & 0xff);
}

	/*ID: 1890, INST: st.b Da,(Rn)+*/

	/*ID: 1891, INST: st.b Da,(Rn)+Rk*/

	/*ID: 1892, INST: st.b Da,(Rn)-*/

	/*ID: 1893, INST: st.b Da,(Rn+Rk)*/

	/*ID: 1894, INST: st.b Da,(Rn+s16_t7)*/

	/*ID: 1895, INST: st.b Da,(sp+s16_t7)*/

	/*ID: 1896, INST: st.b Da,(sp-u9)*/

	/*ID: 1897, INST: st.b Ra,(Rn)*/

	/*ID: 1898, INST: st.b Ra,(Rn)+*/

	/*ID: 1899, INST: st.b Ra,(Rn)+Rk*/

	/*ID: 1900, INST: st.b Ra,(Rn)-*/

	/*ID: 1901, INST: st.b Ra,(Rn+Rk)*/

	/*ID: 1902, INST: st.b Ra,(Rn+s16_t7)*/

	/*ID: 1903, INST: st.b Ra,(sp+s16_t7)*/

	/*ID: 1904, INST: st.b Ra,(sp-u9)*/

	/*ID: 1905, INST: st.2b Dab,(Rn)*/
void do__st_2b(Word8* __p, Word32 __Da, Word32 __Db)
{
	do__st_b(__p, __Da);
	do__st_b(__p+1, __Db);
}

	/*ID: 1906, INST: st.2b Dab,(Rn)+*/

	/*ID: 1907, INST: st.2b Dab,(Rn)+Rk*/

	/*ID: 1908, INST: st.2b Dab,(Rn)-*/

	/*ID: 1909, INST: st.2b Dab,(Rn+Rk)*/

	/*ID: 1910, INST: st.2b Dab,(Rn+s16_t7)*/

	/*ID: 1911, INST: st.2b Rab,(Rn)*/

	/*ID: 1912, INST: st.2b Rab,(Rn)+*/

	/*ID: 1913, INST: st.2b Rab,(Rn)+Rk*/

	/*ID: 1914, INST: st.2b Rab,(Rn)-*/

	/*ID: 1915, INST: st.2b Rab,(Rn+Rk)*/

	/*ID: 1916, INST: st.2b Rab,(Rn+s16_t7)*/

	/*ID: 1917, INST: st.4b Dabcd,(Rn)*/
void do__st_4b(Word8* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
	do__st_2b(__p, __Da,__Db);
	do__st_2b(__p+2,__Dc,__Dd);
}

	/*ID: 1918, INST: st.4b Dabcd,(Rn)+*/

	/*ID: 1919, INST: st.4b Dabcd,(Rn)+Rk*/

	/*ID: 1920, INST: st.4b Dabcd,(Rn)-*/

	/*ID: 1921, INST: st.4b Dabcd,(Rn+Rk)*/

	/*ID: 1922, INST: st.4b Dabcd,(Rn+s16_t7)*/

	/*ID: 1923, INST: st.4b Rabcd,(Rn)*/

	/*ID: 1924, INST: st.4b Rabcd,(Rn)+*/

	/*ID: 1925, INST: st.4b Rabcd,(Rn)+Rk*/

	/*ID: 1926, INST: st.4b Rabcd,(Rn)-*/

	/*ID: 1927, INST: st.4b Rabcd,(Rn+Rk)*/

	/*ID: 1928, INST: st.4b Rabcd,(Rn+s16_t7)*/

	/*ID: 1929, INST: st.8b Da__Dh,(Rn)*/
void do__st_8b(Word8* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32 __De, Word32 __Df, Word32 __Dg, Word32 __Dh)
{
	do__st_4b(__p, __Da,__Db,__Dc,__Dd);
	do__st_4b(__p+4,__De,__Df,__Dg,__Dh);

}

	/*ID: 1930, INST: st.8b Da__Dh,(Rn)+*/

	/*ID: 1931, INST: st.8b Da__Dh,(Rn)+Rk*/

	/*ID: 1932, INST: st.8b Da__Dh,(Rn)-*/

	/*ID: 1933, INST: st.8b Da__Dh,(Rn+Rk)*/

	/*ID: 1934, INST: st.8b Da__Dh,(Rn+s16_t7)*/

	/*ID: 1935, INST: st.bf BTR1,(Rn)*/

	/*ID: 1936, INST: st.bf BTR1,(Rn)+*/

	/*ID: 1937, INST: st.bf BTR1,(Rn)-*/

	/*ID: 1938, INST: st.bf Da,(Rn)*/
void do__st_bf(Word8* __p, Word32 __Da)
{
	*__p = ((__Da >> 24)&0xff);
}

	/*ID: 1939, INST: st.bf Da,(Rn)+*/

	/*ID: 1940, INST: st.bf Da,(Rn)+Rk*/

	/*ID: 1941, INST: st.bf Da,(Rn)-*/

	/*ID: 1942, INST: st.bf Da,(Rn+Rk)*/

	/*ID: 1943, INST: st.bf Da,(Rn+s16_t7)*/

	/*ID: 1944, INST: st.bf Da,(sp+s16_t7)*/

	/*ID: 1945, INST: st.bf Da,(sp-u9)*/

	/*ID: 1946, INST: st.2bf Dab,(Rn)*/
void do__st_2bf(Word8* __p, Word32 __Da, Word32 __Db)
{
	do__st_bf(__p,__Da);
	do__st_bf(__p+1,__Db);
}

	/*ID: 1947, INST: st.2bf Dab,(Rn)+*/

	/*ID: 1948, INST: st.2bf Dab,(Rn)+Rk*/

	/*ID: 1949, INST: st.2bf Dab,(Rn)-*/

	/*ID: 1950, INST: st.2bf Dab,(Rn+Rk)*/

	/*ID: 1951, INST: st.2bf Dab,(Rn+s16_t7)*/

	/*ID: 1952, INST: st.4bf Dabcd,(Rn)*/
void do__st_4bf(Word8* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
	do__st_2bf(__p,__Da,__Db);
	do__st_2bf(__p+2,__Dc,__Dd);
}

	/*ID: 1953, INST: st.4bf Dabcd,(Rn)+*/

	/*ID: 1954, INST: st.4bf Dabcd,(Rn)+Rk*/

	/*ID: 1955, INST: st.4bf Dabcd,(Rn)-*/

	/*ID: 1956, INST: st.4bf Dabcd,(Rn+Rk)*/

	/*ID: 1957, INST: st.4bf Dabcd,(Rn+s16_t7)*/

	/*ID: 1958, INST: st.8bf Da__Dh,(Rn)*/
void do__st_8bf(Word8* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32 __De, Word32 __Df, Word32 __Dg, Word32 __Dh)
{
	do__st_4bf(__p,__Da,__Db,__Dc,__Dd);
	do__st_4bf(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 1959, INST: st.8bf Da__Dh,(Rn)+*/

	/*ID: 1960, INST: st.8bf Da__Dh,(Rn)+Rk*/

	/*ID: 1961, INST: st.8bf Da__Dh,(Rn)-*/

	/*ID: 1962, INST: st.8bf Da__Dh,(Rn+Rk)*/

	/*ID: 1963, INST: st.8bf Da__Dh,(Rn+s16_t7)*/

	/*ID: 1964, INST: st.f BTR1,(Rn)*/

	/*ID: 1965, INST: st.f BTR1,(Rn)+*/

	/*ID: 1966, INST: st.f BTR1,(Rn)-*/

	/*ID: 1967, INST: st.f Da,(Rn)*/
void do__st_f(Word16* __p, Word32 __Da)
{
   *__p = __l_get_msb(__Da);
}

	/*ID: 1968, INST: st.f Da,(Rn)+*/

	/*ID: 1969, INST: st.f Da,(Rn)+Rk*/

	/*ID: 1970, INST: st.f Da,(Rn)-*/

	/*ID: 1971, INST: st.f Da,(Rn+Rk)*/

	/*ID: 1972, INST: st.f Da,(Rn+s16_t7)*/

	/*ID: 1973, INST: st.f Da,(sp+s16_t7)*/

	/*ID: 1974, INST: st.f Da,(sp-u9_1)*/

	/*ID: 1975, INST: st.2f Dab,(Rn)*/
void do__st_2f(Word16* __p, Word32 __Da, Word32 __Db)
{
	do__st_f(__p,__Da);
	do__st_f(__p+1,__Db);
}

	/*ID: 1976, INST: st.2f Dab,(Rn)+*/

	/*ID: 1977, INST: st.2f Dab,(Rn)+Rk*/

	/*ID: 1978, INST: st.2f Dab,(Rn)-*/

	/*ID: 1979, INST: st.2f Dab,(Rn+Rk)*/

	/*ID: 1980, INST: st.2f Dab,(Rn+s16_t7)*/

	/*ID: 1981, INST: st.4f Dabcd,(Rn)*/
void do__st_4f(Word16* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
	do__st_2f(__p,__Da,__Db);
	do__st_2f(__p+2,__Dc,__Dd);
}

	/*ID: 1982, INST: st.4f Dabcd,(Rn)+*/

	/*ID: 1983, INST: st.4f Dabcd,(Rn)+Rk*/

	/*ID: 1984, INST: st.4f Dabcd,(Rn)-*/

	/*ID: 1985, INST: st.4f Dabcd,(Rn+Rk)*/

	/*ID: 1986, INST: st.4f Dabcd,(Rn+s16_t7)*/

	/*ID: 1987, INST: st.8f Da__Dh,(Rn)*/
void do__st_8f(Word16* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32 __De, Word32 __Df, Word32 __Dg, Word32 __Dh)
{
	do__st_4f(__p,__Da,__Db,__Dc,__Dd);
	do__st_4f(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 1988, INST: st.8f Da__Dh,(Rn)+*/

	/*ID: 1989, INST: st.8f Da__Dh,(Rn)+Rk*/

	/*ID: 1990, INST: st.8f Da__Dh,(Rn)-*/

	/*ID: 1991, INST: st.8f Da__Dh,(Rn+Rk)*/

	/*ID: 1992, INST: st.8f Da__Dh,(Rn+s16_t7)*/

	/*ID: 1993, INST: st.l BTR1,(Rn)+*/

	/*ID: 1994, INST: st.l BTR1,(Rn)-*/

	/*ID: 1995, INST: st.l BTRN,(Rn)*/

	/*ID: 1996, INST: st.l Da,(Rn)*/
void do__st_l(Word32* __p, Word32 __Da)
{
	*__p = __Da;
}

	/*ID: 1997, INST: st.l Da,(Rn)+*/

	/*ID: 1998, INST: st.l Da,(Rn)+Rk*/

	/*ID: 1999, INST: st.l Da,(Rn)-*/

	/*ID: 2000, INST: st.l Da,(Rn+Rk)*/

	/*ID: 2001, INST: st.l Da,(Rn+s16_t7)*/

	/*ID: 2002, INST: st.l Da,(sp+s16_t7)*/

	/*ID: 2003, INST: st.l Da,(sp-u9_2)*/

	/*ID: 2004, INST: st.l Ra,(Rn)*/

	/*ID: 2005, INST: st.l Ra,(Rn)+*/

	/*ID: 2006, INST: st.l Ra,(Rn)+Rk*/

	/*ID: 2007, INST: st.l Ra,(Rn)-*/

	/*ID: 2008, INST: st.l Ra,(Rn+Rk)*/

	/*ID: 2009, INST: st.l Ra,(Rn+s16_t7)*/

	/*ID: 2010, INST: st.l Ra,(sp+s16_t7)*/

	/*ID: 2011, INST: st.l Ra,(sp-u9_2)*/

	/*ID: 2012, INST: st.2l BTR0:BTR1,(Rn)*/

	/*ID: 2013, INST: st.2l BTR0:BTR1,(Rn)+*/

	/*ID: 2014, INST: st.2l BTR0:BTR1,(Rn)-*/

	/*ID: 2015, INST: st.2l Dab,(Rn)*/
void do__st_2l(Word32* __p, Word32 __Da, Word32 __Db)
{
	do__st_l(__p,__Da);
	do__st_l(__p+1,__Db);
}

	/*ID: 2016, INST: st.2l Dab,(Rn)+*/

	/*ID: 2017, INST: st.2l Dab,(Rn)+Rk*/

	/*ID: 2018, INST: st.2l Dab,(Rn)-*/

	/*ID: 2019, INST: st.2l Dab,(Rn+Rk)*/

	/*ID: 2020, INST: st.2l Dab,(Rn+s16_t7)*/

	/*ID: 2021, INST: st.2l Dab,(sp+s16_t7)*/

	/*ID: 2022, INST: st.2l Dac,(Rn)*/

	/*ID: 2023, INST: st.2l Dac,(Rn)+*/

	/*ID: 2024, INST: st.2l Dac,(Rn)+Rk*/

	/*ID: 2025, INST: st.2l Dac,(Rn)-*/

	/*ID: 2026, INST: st.2l Dac,(Rn+Rk)*/

	/*ID: 2027, INST: st.2l Dac,(Rn+s16_t7)*/

	/*ID: 2028, INST: st.2l Rab,(Rn)*/

	/*ID: 2029, INST: st.2l Rab,(Rn)+*/

	/*ID: 2030, INST: st.2l Rab,(Rn)+Rk*/

	/*ID: 2031, INST: st.2l Rab,(Rn)-*/

	/*ID: 2032, INST: st.2l Rab,(Rn+Rk)*/

	/*ID: 2033, INST: st.2l Rab,(Rn+s16_t7)*/

	/*ID: 2034, INST: st.2l Rab,(sp+s16_t7)*/

	/*ID: 2035, INST: st.4l Dabcd,(Rn)*/
void do__st_4l(Word32* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
	do__st_2l(__p,__Da,__Db);
	do__st_2l(__p+2,__Dc,__Dd);
}

	/*ID: 2036, INST: st.4l Dabcd,(Rn)+*/

	/*ID: 2037, INST: st.4l Dabcd,(Rn)+Rk*/

	/*ID: 2038, INST: st.4l Dabcd,(Rn)-*/

	/*ID: 2039, INST: st.4l Dabcd,(Rn+Rk)*/

	/*ID: 2040, INST: st.4l Dabcd,(Rn+s16_t7)*/

	/*ID: 2041, INST: st.4l Rabcd,(Rn)*/

	/*ID: 2042, INST: st.4l Rabcd,(Rn)+*/

	/*ID: 2043, INST: st.4l Rabcd,(Rn)+Rk*/

	/*ID: 2044, INST: st.4l Rabcd,(Rn)-*/

	/*ID: 2045, INST: st.4l Rabcd,(Rn+Rk)*/

	/*ID: 2046, INST: st.4l Rabcd,(Rn+s16_t7)*/

	/*ID: 2047, INST: st.8l Da__Dh,(Rn)*/
void do__st_8l(Word32* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32 __De, Word32 __Df, Word32 __Dg, Word32 __Dh)
{
	do__st_4l(__p,__Da,__Db,__Dc,__Dd);
	do__st_4l(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 2048, INST: st.8l Da__Dh,(Rn)+*/

	/*ID: 2049, INST: st.8l Da__Dh,(Rn)+Rk*/

	/*ID: 2050, INST: st.8l Da__Dh,(Rn)-*/

	/*ID: 2051, INST: st.8l Da__Dh,(Rn+Rk)*/

	/*ID: 2052, INST: st.8l Da__Dh,(Rn+s16_t7)*/

	/*ID: 2053, INST: st.16l Da__Dp,(Rn)*/
void do__st_16l(Word32* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32 __De, Word32 __Df, Word32 __Dg, Word32 __Dh, Word32 __Di, Word32 __Dj, Word32 __Dk, Word32 __Dl, Word32 __Dm, Word32 __Dn, Word32 __Do, Word32 __Dp)
{
	do__st_8l(__p,__Da,__Db,__Dc,__Dd,__De,__Df,__Dg,__Dh);
	do__st_8l(__p+8,__Di,__Dj,__Dk,__Dl,__Dm,__Dn,__Do,__Dp);
}

	/*ID: 2054, INST: st.16l Da__Dp,(Rn)+*/

	/*ID: 2055, INST: st.16l Da__Dp,(Rn)+Rk*/

	/*ID: 2056, INST: st.16l Da__Dp,(Rn)-*/

	/*ID: 2057, INST: st.16l Da__Dp,(Rn+Rk)*/

	/*ID: 2058, INST: st.16l Da__Dp,(Rn+s16_t7)*/

	/*ID: 2059, INST: st.w Da,(Rn)*/
void do__st_w(Word16* __p, Word32 __Da)
{
	*__p = __l_get_lsb(__Da);
}

	/*ID: 2060, INST: st.w Da,(Rn)+*/

	/*ID: 2061, INST: st.w Da,(Rn)+Rk*/

	/*ID: 2062, INST: st.w Da,(Rn)-*/

	/*ID: 2063, INST: st.w Da,(Rn+Rk)*/

	/*ID: 2064, INST: st.w Da,(Rn+s16_t7)*/

	/*ID: 2065, INST: st.w Da,(sp+s16_t7)*/

	/*ID: 2066, INST: st.w Da,(sp-u9_1)*/

	/*ID: 2067, INST: st.w Ra,(Rn)*/

	/*ID: 2068, INST: st.w Ra,(Rn)+*/

	/*ID: 2069, INST: st.w Ra,(Rn)+Rk*/

	/*ID: 2070, INST: st.w Ra,(Rn)-*/

	/*ID: 2071, INST: st.w Ra,(Rn+Rk)*/

	/*ID: 2072, INST: st.w Ra,(Rn+s16_t7)*/

	/*ID: 2073, INST: st.w Ra,(sp+s16_t7)*/

	/*ID: 2074, INST: st.w Ra,(sp-u9_1)*/

	/*ID: 2075, INST: st.2w Dab,(Rn)*/
void do__st_2w(Word16* __p, Word32 __Da, Word32 __Db)
{
	do__st_w(__p,__Da);
	do__st_w(__p+1,__Db);
}

	/*ID: 2076, INST: st.2w Dab,(Rn)+*/

	/*ID: 2077, INST: st.2w Dab,(Rn)+Rk*/

	/*ID: 2078, INST: st.2w Dab,(Rn)-*/

	/*ID: 2079, INST: st.2w Dab,(Rn+Rk)*/

	/*ID: 2080, INST: st.2w Dab,(Rn+s16_t7)*/

	/*ID: 2081, INST: st.2w Rab,(Rn)*/

	/*ID: 2082, INST: st.2w Rab,(Rn)+*/

	/*ID: 2083, INST: st.2w Rab,(Rn)+Rk*/

	/*ID: 2084, INST: st.2w Rab,(Rn)-*/

	/*ID: 2085, INST: st.2w Rab,(Rn+Rk)*/

	/*ID: 2086, INST: st.2w Rab,(Rn+s16_t7)*/

	/*ID: 2087, INST: st.4w Dabcd,(Rn)*/
void do__st_4w(Word16* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd)
{
	do__st_2w(__p,__Da,__Db);
	do__st_2w(__p+2,__Dc,__Dd);
}

	/*ID: 2088, INST: st.4w Dabcd,(Rn)+*/

	/*ID: 2089, INST: st.4w Dabcd,(Rn)+Rk*/

	/*ID: 2090, INST: st.4w Dabcd,(Rn)-*/

	/*ID: 2091, INST: st.4w Dabcd,(Rn+Rk)*/

	/*ID: 2092, INST: st.4w Dabcd,(Rn+s16_t7)*/

	/*ID: 2093, INST: st.4w Rabcd,(Rn)*/

	/*ID: 2094, INST: st.4w Rabcd,(Rn)+*/

	/*ID: 2095, INST: st.4w Rabcd,(Rn)+Rk*/

	/*ID: 2096, INST: st.4w Rabcd,(Rn)-*/

	/*ID: 2097, INST: st.4w Rabcd,(Rn+Rk)*/

	/*ID: 2098, INST: st.4w Rabcd,(Rn+s16_t7)*/

	/*ID: 2099, INST: st.8w Da__Dh,(Rn)*/
void do__st_8w(Word16* __p, Word32 __Da, Word32 __Db, Word32 __Dc, Word32 __Dd, Word32 __De, Word32 __Df, Word32 __Dg, Word32 __Dh)
{
	do__st_4w(__p,__Da,__Db,__Dc,__Dd);
	do__st_4w(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 2100, INST: st.8w Da__Dh,(Rn)+*/

	/*ID: 2101, INST: st.8w Da__Dh,(Rn)+Rk*/

	/*ID: 2102, INST: st.8w Da__Dh,(Rn)-*/

	/*ID: 2103, INST: st.8w Da__Dh,(Rn+Rk)*/

	/*ID: 2104, INST: st.8w Da__Dh,(Rn+s16_t7)*/

	/*ID: 2105, INST: st.x Da,(Rn)*/
void do__st_x(Word40* __p, Word40 __Da)
{
	*__p = __x_set(__x_guard(__Da),__x_to_l(__Da));
}

	/*ID: 2106, INST: st.x Da,(Rn)+*/

	/*ID: 2107, INST: st.x Da,(Rn)+Rk*/

	/*ID: 2108, INST: st.x Da,(Rn)-*/

	/*ID: 2109, INST: st.x Da,(Rn+Rk)*/

	/*ID: 2110, INST: st.x Da,(Rn+s16_t7)*/

	/*ID: 2111, INST: st.x Da,(sp+s16_t7)*/

	/*ID: 2112, INST: st.x Da,(sp-u9_3)*/

	/*ID: 2113, INST: st.2x Dab,(Rn)*/
void do__st_2x(Word40* __p, Word40 __Da, Word40 __Db)
{
	do__st_x(__p,__Da);
	do__st_x(__p+1,__Db);
}

	/*ID: 2114, INST: st.2x Dab,(Rn)+*/

	/*ID: 2115, INST: st.2x Dab,(Rn)+Rk*/

	/*ID: 2116, INST: st.2x Dab,(Rn)-*/

	/*ID: 2117, INST: st.2x Dab,(Rn+Rk)*/

	/*ID: 2118, INST: st.2x Dab,(Rn+s16_t7)*/

	/*ID: 2119, INST: st.4x Dabcd,(Rn)*/
void do__st_4x(Word40* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st_2x(__p,__Da,__Db);
	do__st_2x(__p+2,__Dc,__Dd);
}

	/*ID: 2120, INST: st.4x Dabcd,(Rn)+*/

	/*ID: 2121, INST: st.4x Dabcd,(Rn)+Rk*/

	/*ID: 2122, INST: st.4x Dabcd,(Rn)-*/

	/*ID: 2123, INST: st.4x Dabcd,(Rn+Rk)*/

	/*ID: 2124, INST: st.4x Dabcd,(Rn+s16_t7)*/

	/*ID: 2125, INST: st.8x Da__Dh,(Rn)*/
void do__st_8x(Word40* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st_4x(__p,__Da,__Db,__Dc,__Dd);
	do__st_4x(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 2126, INST: st.8x Da__Dh,(Rn)+*/

	/*ID: 2127, INST: st.8x Da__Dh,(Rn)+Rk*/

	/*ID: 2128, INST: st.8x Da__Dh,(Rn)-*/

	/*ID: 2129, INST: st.8x Da__Dh,(Rn+Rk)*/

	/*ID: 2130, INST: st.8x Da__Dh,(Rn+s16_t7)*/

	/*ID: 2131, INST: st.srs.bf Da,(Rn)*/
/*
  Function Name: do__st_srs_bf
  Purpose: Scale,round and saturate a 40 bit register(__Da.HH) data and store the result to 8 bit memory
  Complexity Weight:
  Inputs: __Da is 40 bit and __p is pointer to 8 bit memory.
  Outputs:
  Return Value:
*/
void do__st_srs_bf(Word8* __p, Word40 __Da)
{
	Word40 scaled, rounded;
	UWord32 discarded;

	scaled = __X_scale_sat(__Da);

	if(RoundingMode == CONVERGENT)
	{
		discarded = (__x_to_l(scaled) & 0x00ffffff);
		if((discarded > 0x00800000) || ((discarded == 0x00800000) && (__x_to_l(scaled)&0x01000000)))
			rounded = do__add_x(scaled, __x_set(0, 0x00800000));
		else
			rounded = scaled;
	}
	else
		rounded = do__add_x(scaled, __x_set(0, 0x00800000));

	rounded = do__ash_rgt_x(24,rounded);
	rounded = do__sat_x_hw(rounded);
	*__p = __x_to_l(rounded)&0xff;
}

	/*ID: 2132, INST: st.srs.bf Da,(Rn)+*/

	/*ID: 2133, INST: st.srs.bf Da,(Rn)+Rk*/

	/*ID: 2134, INST: st.srs.bf Da,(Rn)-*/

	/*ID: 2135, INST: st.srs.bf Da,(Rn+Rk)*/

	/*ID: 2136, INST: st.srs.bf Da,(Rn+s16_t7)*/

	/*ID: 2137, INST: st.srs.2bf Dab,(Rn)*/
/*
  Function Name: do__st_srs_2bf
  Purpose: Scale,round and saturate a pair of 40 bit registers(__Da.HH,__Db.hh) data and store the result to 8 bit memory
  Complexity Weight:
  Inputs: __Da and __Db are 40 bit and __p is pointer to 8 bit memory.
  Outputs:
  Return Value:
*/
void do__st_srs_2bf(Word8* __p, Word40 __Da, Word40 __Db)
{
	do__st_srs_bf(__p,__Da);
	do__st_srs_bf(__p+1,__Db);
}

	/*ID: 2138, INST: st.srs.2bf Dab,(Rn)+*/

	/*ID: 2139, INST: st.srs.2bf Dab,(Rn)+Rk*/

	/*ID: 2140, INST: st.srs.2bf Dab,(Rn)-*/

	/*ID: 2141, INST: st.srs.2bf Dab,(Rn+Rk)*/

	/*ID: 2142, INST: st.srs.2bf Dab,(Rn+s16_t7)*/

	/*ID: 2143, INST: st.srs.4bf Dabcd,(Rn)*/
/*
  Function Name: do__st_srs_4bf
  Purpose: Scale,round and saturate a 40 bit register quad(__Da.HH,__Db.HH,__Dc.HH,__Dd.HH) and store the result to 8 bit memory
  Complexity Weight:
  Inputs: __Da ,__Db,__Dc and __Dd are 40 bit and __p is pointer to 8 bit memory.
  Outputs:
  Return Value:
*/
void do__st_srs_4bf(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st_srs_2bf(__p,__Da,__Db);
	do__st_srs_2bf(__p+2,__Dc,__Dd);

}

	/*ID: 2144, INST: st.srs.4bf Dabcd,(Rn)+*/

	/*ID: 2145, INST: st.srs.4bf Dabcd,(Rn)+Rk*/

	/*ID: 2146, INST: st.srs.4bf Dabcd,(Rn)-*/

	/*ID: 2147, INST: st.srs.4bf Dabcd,(Rn+Rk)*/

	/*ID: 2148, INST: st.srs.4bf Dabcd,(Rn+s16_t7)*/

	/*ID: 2149, INST: st.srs.8bf Da__Dh,(Rn)*/
/*
  Function Name: do__st_srs_8bf
  Purpose: Scale,round and saturate a 40 bit register octet(__Da.HH,__Db.HH,__Dc.HH,__Dd.HH,__De.HH,__Df.HH,__Dg.HH,__Dh.HH) and store the result to 8 bit memory
  Complexity Weight:
  Inputs: __Da ,__Db,__Dc,__Dd,__De,__Df,__Dg,__Dh are 40 bit and __p is pointer to 8 bit memory.
  Outputs:
  Return Value:
*/
void do__st_srs_8bf(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st_srs_4bf(__p,__Da,__Db,__Dc,__Dd);
	do__st_srs_4bf(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 2150, INST: st.srs.8bf Da__Dh,(Rn)+*/

	/*ID: 2151, INST: st.srs.8bf Da__Dh,(Rn)+Rk*/

	/*ID: 2152, INST: st.srs.8bf Da__Dh,(Rn)-*/

	/*ID: 2153, INST: st.srs.8bf Da__Dh,(Rn+Rk)*/

	/*ID: 2154, INST: st.srs.8bf Da__Dh,(Rn+s16_t7)*/

	/*ID: 2155, INST: st.srs.f Da,(Rn)*/
/*
  Function Name: do__st_srs_f
  Purpose: move a data register(scale,round and saturate) to 16 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 16 bit memory and __Da - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_f(Word16* __p, Word40 __Da)
{
    Word40 scaled, rounded;
    UWord32 discarded;

	scaled = __X_scale_sat(__Da);

	if(RoundingMode == CONVERGENT)
	{
		discarded = (__x_to_l(scaled) & 0x0000ffff);
		if((discarded > 0x00008000) || ((discarded == 0x00008000) && (__x_to_l(scaled)&0x00010000)))
			rounded = do__add_x(scaled, __x_set(0, 0x00008000));
		else
			rounded = scaled;
	}
	else
		rounded = do__add_x(scaled, __x_set(0, 0x00008000));
	rounded = do__ash_rgt_x(16,rounded);
	rounded = do__sat_x_w(rounded);
	*__p = (__x_to_l(rounded))&0xffff;
}

	/*ID: 2156, INST: st.srs.f Da,(Rn)+*/

	/*ID: 2157, INST: st.srs.f Da,(Rn)+Rk*/

	/*ID: 2158, INST: st.srs.f Da,(Rn)-*/

	/*ID: 2159, INST: st.srs.f Da,(Rn+Rk)*/

	/*ID: 2160, INST: st.srs.f Da,(Rn+s16_t7)*/

	/*ID: 2161, INST: st.srs.2f Dab,(Rn)*/
/*
  Function Name: do__st_srs_2f
  Purpose: move a pair of data registers(scale,round and saturate) to 16 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 16 bit memory and __Da,__Db - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_2f(Word16* __p, Word40 __Da, Word40 __Db)
{
	do__st_srs_f(__p,__Da);
	do__st_srs_f(__p+1,__Db);
}

	/*ID: 2162, INST: st.srs.2f Dab,(Rn)+*/

	/*ID: 2163, INST: st.srs.2f Dab,(Rn)+Rk*/

	/*ID: 2164, INST: st.srs.2f Dab,(Rn)-*/

	/*ID: 2165, INST: st.srs.2f Dab,(Rn+Rk)*/

	/*ID: 2166, INST: st.srs.2f Dab,(Rn+s16_t7)*/

	/*ID: 2167, INST: st.srs.4f Dabcd,(Rn)*/
/*
  Function Name: do__st_srs_4f
  Purpose: move data register quad(scale,round and saturate) to 16 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 16 bit memory and __Da,__Db ,__Dc,__Dd - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_4f(Word16* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st_srs_2f(__p,__Da,__Db);
	do__st_srs_2f(__p+2,__Dc,__Dd);
}

	/*ID: 2168, INST: st.srs.4f Dabcd,(Rn)+*/

	/*ID: 2169, INST: st.srs.4f Dabcd,(Rn)+Rk*/

	/*ID: 2170, INST: st.srs.4f Dabcd,(Rn)-*/

	/*ID: 2171, INST: st.srs.4f Dabcd,(Rn+Rk)*/

	/*ID: 2172, INST: st.srs.4f Dabcd,(Rn+s16_t7)*/

	/*ID: 2173, INST: st.srs.8f Da__Dh,(Rn)*/
/*
  Function Name: do__st_srs_8f
  Purpose: move data register quad(scale,round and saturate) to 16 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 16 bit memory and __Da,__Db ,__Dc,__Dd,__De,__Df,__Dg,__Dh - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_8f(Word16* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st_srs_4f(__p,__Da,__Db,__Dc,__Dd);
	do__st_srs_4f(__p+4,__De,__Df,__Dg,__Dh);

}

	/*ID: 2174, INST: st.srs.8f Da__Dh,(Rn)+*/

	/*ID: 2175, INST: st.srs.8f Da__Dh,(Rn)+Rk*/

	/*ID: 2176, INST: st.srs.8f Da__Dh,(Rn)-*/

	/*ID: 2177, INST: st.srs.8f Da__Dh,(Rn+Rk)*/

	/*ID: 2178, INST: st.srs.8f Da__Dh,(Rn+s16_t7)*/

	/*ID: 2179, INST: st.srs.l Da,(Rn)*/
/*
  Function Name: do__st_srs_l
  Purpose: move a data register(scale,round and saturate) to 32 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 32 bit memory and __Da - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_l(Word32* __p, Word40 __Da)
{
	Word40 scaled, rounded;
	Word32 result;
	scaled = __X_scale_sat(__Da);
	result = do__sat_x_l(scaled);
	*__p = result;
}

	/*ID: 2180, INST: st.srs.l Da,(Rn)+*/

	/*ID: 2181, INST: st.srs.l Da,(Rn)+Rk*/

	/*ID: 2182, INST: st.srs.l Da,(Rn)-*/

	/*ID: 2183, INST: st.srs.l Da,(Rn+Rk)*/

	/*ID: 2184, INST: st.srs.l Da,(Rn+s16_t7)*/

	/*ID: 2185, INST: st.srs.2l Dab,(Rn)*/
/*
  Function Name: do__st_srs_2l
  Purpose: move a pair of data registers(scale,round and saturate) to 32 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 32 bit memory and __Da,__Db - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_2l(Word32* __p, Word40 __Da, Word40 __Db)
{
	do__st_srs_l(__p,__Da);
	do__st_srs_l(__p+1,__Db);
}

	/*ID: 2186, INST: st.srs.2l Dab,(Rn)+*/

	/*ID: 2187, INST: st.srs.2l Dab,(Rn)+Rk*/

	/*ID: 2188, INST: st.srs.2l Dab,(Rn)-*/

	/*ID: 2189, INST: st.srs.2l Dab,(Rn+Rk)*/

	/*ID: 2190, INST: st.srs.2l Dab,(Rn+s16_t7)*/

	/*ID: 2191, INST: st.srs.4l Dabcd,(Rn)*/
/*
  Function Name: do__st_srs_4l
  Purpose: move a data registers quad(scale,round and saturate) to 32 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 32 bit memory and __Da,__Db, __Dc, __Dd - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_4l(Word32* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st_srs_2l(__p,__Da,__Db);
	do__st_srs_2l(__p+2,__Dc,__Dd);
}

	/*ID: 2192, INST: st.srs.4l Dabcd,(Rn)+*/

	/*ID: 2193, INST: st.srs.4l Dabcd,(Rn)+Rk*/

	/*ID: 2194, INST: st.srs.4l Dabcd,(Rn)-*/

	/*ID: 2195, INST: st.srs.4l Dabcd,(Rn+Rk)*/

	/*ID: 2196, INST: st.srs.4l Dabcd,(Rn+s16_t7)*/

	/*ID: 2197, INST: st.srs.8l Da__Dh,(Rn)*/
/*
  Function Name: do__st_srs_8l
  Purpose: move a data registers octet(scale,round and saturate) to 32 bit memory.
  Complexity Weight:
  Inputs: __p - pointer to 32 bit memory and __Da,__Db, __Dc,__Dd,__De,__Df, __Dg, __Dh - 40 bit register
  Outputs:
  Return Value:
*/
void do__st_srs_8l(Word32* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st_srs_4l(__p,__Da,__Db,__Dc,__Dd);
	do__st_srs_4l(__p+4,__De,__Df,__Dg,__Dh);
}

	/*ID: 2198, INST: st.srs.8l Da__Dh,(Rn)+*/

	/*ID: 2199, INST: st.srs.8l Da__Dh,(Rn)+Rk*/

	/*ID: 2200, INST: st.srs.8l Da__Dh,(Rn)-*/

	/*ID: 2201, INST: st.srs.8l Da__Dh,(Rn+Rk)*/

	/*ID: 2202, INST: st.srs.8l Da__Dh,(Rn+s16_t7)*/

	/*ID: 2203, INST: st.srs.wh.8f Da__Dh,(Rn)*/
void do__st_srs_wh_8f(Word16* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{

	Word32 i=0,temp[8];
	Word16 temp1[8];
	temp[0] = __extract_wh(__Da);
	temp[1] = __extract_wh(__Db);
	temp[2] = __extract_wh(__Dc);
	temp[3] = __extract_wh(__Dd);
	temp[4] = __extract_wh(__De);
	temp[5] = __extract_wh(__Df);
	temp[6] = __extract_wh(__Dg);
	temp[7] = __extract_wh(__Dh);

	for(i=0;i<8;i++)
	{
		temp[i] = __x_to_l(__X_scale(__l_to_x(temp[i])));
		temp1[i] = (__x_to_l(do__sat_x_w(__l_to_x(temp[i])))) & 0xFFFF;
	}

	for(i=0;i<8;i++)
	{
	 *__p = temp1[i];
	 __p++;
	}
}

	/*ID: 2204, INST: st.srs.wh.8f Da__Dh,(Rn)+*/

	/*ID: 2205, INST: st.srs.wh.8f Da__Dh,(Rn)+Rk*/

	/*ID: 2206, INST: st.srs.wh.8f Da__Dh,(Rn)-*/

	/*ID: 2207, INST: st.srs.wh.8f Da__Dh,(Rn+Rk)*/

	/*ID: 2208, INST: st.srs.wh.8f Da__Dh,(Rn+s16_t7)*/

	/*ID: 2209, INST: st.srs.wl.8f Da__Dh,(Rn)*/
void do__st_srs_wl_8f(Word16* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	Word32 i=0,temp[8];
	Word16 temp1[8];
	temp[0] = __extract_wl(__Da);
	temp[1] = __extract_wl(__Db);
	temp[2] = __extract_wl(__Dc);
	temp[3] = __extract_wl(__Dd);
	temp[4] = __extract_wl(__De);
	temp[5] = __extract_wl(__Df);
	temp[6] = __extract_wl(__Dg);
	temp[7] = __extract_wl(__Dh);

	for(i=0;i<8;i++)
	{
		temp[i] = __x_to_l(__X_scale(__l_to_x(temp[i])));
		temp1[i] = (__x_to_l(do__sat_x_w(__l_to_x(temp[i])))) & 0xFFFF;
	}

	for(i=0;i<8;i++)
	{
	 *__p = temp1[i];
	 __p++;
	}
}

	/*ID: 2210, INST: st.srs.wl.8f Da__Dh,(Rn)+*/

	/*ID: 2211, INST: st.srs.wl.8f Da__Dh,(Rn)+Rk*/

	/*ID: 2212, INST: st.srs.wl.8f Da__Dh,(Rn)-*/

	/*ID: 2213, INST: st.srs.wl.8f Da__Dh,(Rn+Rk)*/

	/*ID: 2214, INST: st.srs.wl.8f Da__Dh,(Rn+s16_t7)*/

	/*ID: 2215, INST: st2.2b Da,(Rn)*/
void do__st2_2b(Word8* __p, Word40 __Da)
{
	Word8 __Da_HL,__Da_LL;

	__Da_HL = (__x_to_l(__Da)>>16)&0xff;
	__Da_LL = (__x_to_l(__Da))&0xff;

	*__p = __Da_HL;
	*(__p+1) = __Da_LL;
}

	/*ID: 2216, INST: st2.2b Da,(Rn)+*/

	/*ID: 2217, INST: st2.2b Da,(Rn)+Rk*/

	/*ID: 2218, INST: st2.2b Da,(Rn)-*/

	/*ID: 2219, INST: st2.2b Da,(Rn+Rk)*/

	/*ID: 2220, INST: st2.2b Da,(Rn+s16_t7)*/

	/*ID: 2221, INST: st2.4b Dab,(Rn)*/
void do__st2_4b(Word8* __p, Word40 __Da, Word40 __Db)
{
	do__st2_2b(__p,__Da);
	do__st2_2b(__p+2,__Db);
}

	/*ID: 2222, INST: st2.4b Dab,(Rn)+*/

	/*ID: 2223, INST: st2.4b Dab,(Rn)+Rk*/

	/*ID: 2224, INST: st2.4b Dab,(Rn)-*/

	/*ID: 2225, INST: st2.4b Dab,(Rn+Rk)*/

	/*ID: 2226, INST: st2.4b Dab,(Rn+s16_t7)*/

	/*ID: 2227, INST: st2.8b Dabcd,(Rn)*/
void do__st2_8b(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st2_4b(__p,__Da,__Db);
	do__st2_4b(__p+4,__Dc,__Dd);
}

	/*ID: 2228, INST: st2.8b Dabcd,(Rn)+*/

	/*ID: 2229, INST: st2.8b Dabcd,(Rn)+Rk*/

	/*ID: 2230, INST: st2.8b Dabcd,(Rn)-*/

	/*ID: 2231, INST: st2.8b Dabcd,(Rn+Rk)*/

	/*ID: 2232, INST: st2.8b Dabcd,(Rn+s16_t7)*/

	/*ID: 2233, INST: st2.16b Da__Dh,(Rn)*/
void do__st2_16b(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st2_8b(__p,__Da,__Db,__Dc,__Dd);
	do__st2_8b(__p+8,__De,__Df,__Dg,__Dh);
}

	/*ID: 2234, INST: st2.16b Da__Dh,(Rn)+*/

	/*ID: 2235, INST: st2.16b Da__Dh,(Rn)+Rk*/

	/*ID: 2236, INST: st2.16b Da__Dh,(Rn)-*/

	/*ID: 2237, INST: st2.16b Da__Dh,(Rn+Rk)*/

	/*ID: 2238, INST: st2.16b Da__Dh,(Rn+s16_t7)*/

	/*ID: 2239, INST: st2.2bf Da,(Rn)*/
void do__st2_2bf(Word8* __p, Word40 __Da)
{
	Word8 __Da_HH,__Da_LH;

	__Da_HH = (__x_to_l(__Da)>>24)&0xff;
	__Da_LH = (__x_to_l(__Da)>>8)&0xff;

	*__p = __Da_HH;
	*(__p+1) = __Da_LH;
}

	/*ID: 2240, INST: st2.2bf Da,(Rn)+*/

	/*ID: 2241, INST: st2.2bf Da,(Rn)+Rk*/

	/*ID: 2242, INST: st2.2bf Da,(Rn)-*/

	/*ID: 2243, INST: st2.2bf Da,(Rn+Rk)*/

	/*ID: 2244, INST: st2.2bf Da,(Rn+s16_t7)*/

	/*ID: 2245, INST: st2.4bf Dab,(Rn)*/
void do__st2_4bf(Word8* __p, Word40 __Da, Word40 __Db)
{
	do__st2_2bf(__p,__Da);
	do__st2_2bf(__p+2,__Db);
}

	/*ID: 2246, INST: st2.4bf Dab,(Rn)+*/

	/*ID: 2247, INST: st2.4bf Dab,(Rn)+Rk*/

	/*ID: 2248, INST: st2.4bf Dab,(Rn)-*/

	/*ID: 2249, INST: st2.4bf Dab,(Rn+Rk)*/

	/*ID: 2250, INST: st2.4bf Dab,(Rn+s16_t7)*/

	/*ID: 2251, INST: st2.8bf Dabcd,(Rn)*/
void do__st2_8bf(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st2_4bf(__p,__Da,__Db);
	do__st2_4bf(__p+4,__Dc,__Dd);
}

	/*ID: 2252, INST: st2.8bf Dabcd,(Rn)+*/

	/*ID: 2253, INST: st2.8bf Dabcd,(Rn)+Rk*/

	/*ID: 2254, INST: st2.8bf Dabcd,(Rn)-*/

	/*ID: 2255, INST: st2.8bf Dabcd,(Rn+Rk)*/

	/*ID: 2256, INST: st2.8bf Dabcd,(Rn+s16_t7)*/

	/*ID: 2257, INST: st2.16bf Da__Dh,(Rn)*/
void do__st2_16bf(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st2_8bf(__p,__Da,__Db,__Dc,__Dd);
	do__st2_8bf(__p+8,__De,__Df,__Dg,__Dh);
}

	/*ID: 2258, INST: st2.16bf Da__Dh,(Rn)+*/

	/*ID: 2259, INST: st2.16bf Da__Dh,(Rn)+Rk*/

	/*ID: 2260, INST: st2.16bf Da__Dh,(Rn)-*/

	/*ID: 2261, INST: st2.16bf Da__Dh,(Rn+Rk)*/

	/*ID: 2262, INST: st2.16bf Da__Dh,(Rn+s16_t7)*/

	/*ID: 2263, INST: st2.srs.2bf Da,(Rn)*/
void do__st2_srs_2bf(Word8* __p, Word40 __Da)
{
	Word40 __Da_HH,__Da_LH;

	__Da_HH = __l_to_x(__extract_wh(__Da));
	__Da_LH = __l_to_x(__extract_wl(__Da));
	__Da_HH = do__ash_lft_x(16,__Da_HH);
	__Da_LH = do__ash_lft_x(16,__Da_LH);
	do__st_srs_bf(__p,__Da_HH);
	do__st_srs_bf(__p+1,__Da_LH);
}

	/*ID: 2264, INST: st2.srs.2bf Da,(Rn)+*/

	/*ID: 2265, INST: st2.srs.2bf Da,(Rn)+Rk*/

	/*ID: 2266, INST: st2.srs.2bf Da,(Rn)-*/

	/*ID: 2267, INST: st2.srs.2bf Da,(Rn+Rk)*/

	/*ID: 2268, INST: st2.srs.2bf Da,(Rn+s16_t7)*/

	/*ID: 2269, INST: st2.srs.4bf Dab,(Rn)*/
void do__st2_srs_4bf(Word8* __p, Word40 __Da, Word40 __Db)
{
	do__st2_srs_2bf(__p,__Da);
	do__st2_srs_2bf(__p+2,__Db);
}

	/*ID: 2270, INST: st2.srs.4bf Dab,(Rn)+*/

	/*ID: 2271, INST: st2.srs.4bf Dab,(Rn)+Rk*/

	/*ID: 2272, INST: st2.srs.4bf Dab,(Rn)-*/

	/*ID: 2273, INST: st2.srs.4bf Dab,(Rn+Rk)*/

	/*ID: 2274, INST: st2.srs.4bf Dab,(Rn+s16_t7)*/

	/*ID: 2275, INST: st2.srs.8bf Dabcd,(Rn)*/
void do__st2_srs_8bf(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st2_srs_4bf(__p,__Da,__Db);
	do__st2_srs_4bf(__p+4,__Dc,__Dd);
}

	/*ID: 2276, INST: st2.srs.8bf Dabcd,(Rn)+*/

	/*ID: 2277, INST: st2.srs.8bf Dabcd,(Rn)+Rk*/

	/*ID: 2278, INST: st2.srs.8bf Dabcd,(Rn)-*/

	/*ID: 2279, INST: st2.srs.8bf Dabcd,(Rn+Rk)*/

	/*ID: 2280, INST: st2.srs.8bf Dabcd,(Rn+s16_t7)*/

	/*ID: 2281, INST: st2.srs.16bf Da__Dh,(Rn)*/
void do__st2_srs_16bf(Word8* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st2_srs_8bf(__p,__Da,__Db,__Dc,__Dd);
	do__st2_srs_8bf(__p+8,__De,__Df,__Dg,__Dh);
}

	/*ID: 2282, INST: st2.srs.16bf Da__Dh,(Rn)+*/

	/*ID: 2283, INST: st2.srs.16bf Da__Dh,(Rn)+Rk*/

	/*ID: 2284, INST: st2.srs.16bf Da__Dh,(Rn)-*/

	/*ID: 2285, INST: st2.srs.16bf Da__Dh,(Rn+Rk)*/

	/*ID: 2286, INST: st2.srs.16bf Da__Dh,(Rn+s16_t7)*/

	/*ID: 2287, INST: st2.srs.2f Da,(Rn)*/
void do__st2_srs_2f(Word16* __p, Word40 __Da)
{
	Word40 __Da_WH,__Da_WL;

	__Da_WH = __l_to_x(__extract_wh(__Da));
	__Da_WL = __l_to_x(__extract_wl(__Da));
	__Da_WH = do__ash_lft_x(16,__Da_WH);
	__Da_WL = do__ash_lft_x(16,__Da_WL);
	do__st_srs_f(__p,__Da_WH);
	do__st_srs_f(__p+1,__Da_WL);
}

	/*ID: 2288, INST: st2.srs.2f Da,(Rn)+*/

	/*ID: 2289, INST: st2.srs.2f Da,(Rn)+Rk*/

	/*ID: 2290, INST: st2.srs.2f Da,(Rn)-*/

	/*ID: 2291, INST: st2.srs.2f Da,(Rn+Rk)*/

	/*ID: 2292, INST: st2.srs.2f Da,(Rn+s16_t7)*/

	/*ID: 2293, INST: st2.srs.4f Dab,(Rn)*/
void do__st2_srs_4f(Word16* __p, Word40 __Da, Word40 __Db)
{
	do__st2_srs_2f(__p,__Da);
	do__st2_srs_2f(__p+2,__Db);
}

	/*ID: 2294, INST: st2.srs.4f Dab,(Rn)+*/

	/*ID: 2295, INST: st2.srs.4f Dab,(Rn)+Rk*/

	/*ID: 2296, INST: st2.srs.4f Dab,(Rn)-*/

	/*ID: 2297, INST: st2.srs.4f Dab,(Rn+Rk)*/

	/*ID: 2298, INST: st2.srs.4f Dab,(Rn+s16_t7)*/

	/*ID: 2299, INST: st2.srs.8f Dabcd,(Rn)*/
void do__st2_srs_8f(Word16* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd)
{
	do__st2_srs_4f(__p,__Da,__Db);
	do__st2_srs_4f(__p+4,__Dc,__Dd);
}

	/*ID: 2300, INST: st2.srs.8f Dabcd,(Rn)+*/

	/*ID: 2301, INST: st2.srs.8f Dabcd,(Rn)+Rk*/

	/*ID: 2302, INST: st2.srs.8f Dabcd,(Rn)-*/

	/*ID: 2303, INST: st2.srs.8f Dabcd,(Rn+Rk)*/

	/*ID: 2304, INST: st2.srs.8f Dabcd,(Rn+s16_t7)*/

	/*ID: 2305, INST: st2.srs.16f Da__Dh,(Rn)*/
void do__st2_srs_16f(Word16* __p, Word40 __Da, Word40 __Db, Word40 __Dc, Word40 __Dd, Word40 __De, Word40 __Df, Word40 __Dg, Word40 __Dh)
{
	do__st2_srs_8f(__p,__Da,__Db,__Dc,__Dd);
	do__st2_srs_8f(__p+8,__De,__Df,__Dg,__Dh);
}

	/*ID: 2306, INST: st2.srs.16f Da__Dh,(Rn)+*/

	/*ID: 2307, INST: st2.srs.16f Da__Dh,(Rn)+Rk*/

	/*ID: 2308, INST: st2.srs.16f Da__Dh,(Rn)-*/

	/*ID: 2309, INST: st2.srs.16f Da__Dh,(Rn+Rk)*/

	/*ID: 2310, INST: st2.srs.16f Da__Dh,(Rn+s16_t7)*/

	/*ID: 2311, INST: st2.srs.16f Da__DhMod8,(Rn)*/

	/*ID: 2312, INST: st2.srs.16f Da__DhMod8,(Rn)+*/

	/*ID: 2313, INST: st2.srs.16f Da__DhMod8,(Rn)+Rk*/

	/*ID: 2314, INST: st2.srs.16f Da__DhMod8,(Rn)-*/

	/*ID: 2315, INST: st2.srs.16f Da__DhMod8,(Rn+Rk)*/

	/*ID: 2316, INST: st2.srs.16f Da__DhMod8,(Rn+s16_t7)*/

	/*ID: 2317, INST: st2.srs.16f Da__DhMod8p,(Rn)*/

	/*ID: 2318, INST: st2.srs.16f Da__DhMod8p,(Rn)+*/

	/*ID: 2319, INST: st2.srs.16f Da__DhMod8p,(Rn)+Rk*/

	/*ID: 2320, INST: st2.srs.16f Da__DhMod8p,(Rn)-*/

	/*ID: 2321, INST: st2.srs.16f Da__DhMod8p,(Rn+Rk)*/

	/*ID: 2322, INST: st2.srs.16f Da__DhMod8p,(Rn+s16_t7)*/

	/*ID: 2323, INST: stc.b Da,(sp+s16_t7)*/

	/*ID: 2324, INST: stc.b Ra,(sp+s16_t7)*/

	/*ID: 2325, INST: stc.bf Da,(sp+s16_t7)*/

	/*ID: 2326, INST: stc.f Da,(sp+s16_t7)*/

	/*ID: 2327, INST: stc.l Da,(sp+s16_t7)*/

	/*ID: 2328, INST: stc.l Ra,(sp+s16_t7)*/

	/*ID: 2329, INST: stc.w Da,(sp+s16_t7)*/

	/*ID: 2330, INST: stc.w Ra,(sp+s16_t7)*/

	/*ID: 2331, INST: stc.x Da,(sp+s16_t7)*/

	/*ID: 2332, INST: sub.ll Dab,Dcd,Dmn*/

	/*ID: 2333, INST: sub.t Da.h,Dn.h*/
Word40 do__sub_t_hh(Word40 __Dn_h, Word40 __Da_h)
{
	Word32 Dn_WH;
	Dn_WH = __extract_wh(__Dn_h) - __extract_wh(__Da_h);
	__set_WH(Dn_WH,&__Dn_h);
	return __Dn_h;
}

	/*ID: 2334, INST: sub.t Da.l,Dn.l*/
Word40 do__sub_t_ll(Word40 __Dn_l, Word40 __Da_l)
{
	Word32 Dn_WL;
	Dn_WL = __extract_wl(__Dn_l) - __extract_wl(__Da_l);
	__set_WL(Dn_WL,&__Dn_l);
	return __Dn_l;
}

	/*ID: 2335, INST: sub.t Da.h,Dn.l*/
Word40 do__sub_t_hl(Word40 __Dn_l, Word40 __Da_h)
{
	Word32 __Dn_WL,__Da_WH;
	__Dn_WL = do__extract_wl(__Dn_l);
	__Da_WH = do__extract_wh(__Da_h);
	__Dn_WL = __Dn_WL - __Da_WH;
	do__set_WL(__Dn_WL,&__Dn_l);
	return __Dn_l;
}

	/*ID: 2336, INST: sub.t Da.l,Dn.h*/
Word40 do__sub_t_lh(Word40 __Dn_h, Word40 __Da_l)
{
	Word32 __Dn_WH,__Da_WL;
	__Dn_WH = do__extract_wh(__Dn_h);
	__Da_WL = do__extract_wl(__Da_l);
	__Dn_WH = __Dn_WH - __Da_WL;
	do__set_WH(__Dn_WH,&__Dn_h);
	return __Dn_h;

}

	/*ID: 2337, INST: sub.2t Da,Db,Dn*/
Word40 do__sub_2t(Word40 __Da, Word40 __Db)
{
	/*     bool ovf ;
     bits < 21 > wh = signExtend ( D ( Src2 ) . WH , 21 ) - signExtend ( D ( Src1 ) . WH , 21 ) ;
     bits < 21 > wl = signExtend ( D ( Src2 ) . WL , 21 ) - signExtend ( D ( Src1 ) . WL , 21 ) ;
    ovf = overflow20 ( wh ) ;
    ovf = overflow20 ( wl ) ;
    D ( Res1 ) . WH = wh ( 19 , 0 ) ;
    D ( Res1 ) . WL = wl ( 19 , 0 ) ;
    */
    Word40 __Dn;
	Word32 __Da_WH,__Da_WL,__Db_WH,__Db_WL,__Dn_WH,__Dn_WL;
	__Da_WH = do__extract_wh(__Da);
	__Da_WL = do__extract_wl(__Da);
	__Db_WH = do__extract_wh(__Db);
	__Db_WL = do__extract_wl(__Db);
	__Dn_WH = __Db_WH - __Da_WH;
	__Dn_WL = __Db_WL - __Da_WL;
	do__set_WH(__Dn_WH,&__Dn);
	do__set_WL(__Dn_WL,&__Dn);
	return __Dn;
}

	/*ID: 2338, INST: sub.x #u5,Da,Dn*/
Word40 do__sub_x_imm(UWord32 __imm1, Word40 __Da)
{
	Word40 __imm;
	Word32 __imm2 = (Word32)__imm1;
	__imm = __x_set(0x0,__imm2);
	return do__sub_x(__Da,__imm);
}

	/*ID: 2339, INST: sub.x Da,Db,Dn*/
Word40 do__sub_x(Word40 __Da, Word40 __Db)
{

	Word40 X_result;
	long long resultLL = 0LL;
	long long var1LL = __x_guard(__Da);
	long long var2LL = __x_guard(__Db);
	Word32 high,low;

	var1LL = (var1LL << 32) | (UWord32)__x_to_l(__Da);
	var2LL = (var2LL << 32) | (UWord32)__x_to_l(__Db);
	resultLL = var2LL - var1LL;
	X_result  = __x_set((resultLL >> 32) & 0xff, resultLL & 0xffffffff);
	return X_result;
}

	/*ID: 2340, INST: sub.leg.x #u5,Da,Dn*/

	/*ID: 2341, INST: sub.leg.x Da,Db,Dn*/

	/*ID: 2342, INST: sub.s.x Da,Db,Dn*/
Word40 do__sub_s_x(Word40 __Da, Word40 __Db)
{
  return (__l_to_x(do__sat_x_l(do__sub_x(__Da,__Db))));
}

	/*ID: 2343, INST: suba #u5,Ra,Rn*/

	/*ID: 2344, INST: suba Ra,Rb,Rn*/

	/*ID: 2345, INST: suba.lin #u5,Ra,Rn*/
Word32 do__suba_lin(Word32 __imm1, Word32 __Ra)
{
}

	/*ID: 2346, INST: suba.lin #u5,Ra,Rn*/

	/*ID: 2347, INST: suba.lin Ra,Rb,Rn*/

	/*ID: 2348, INST: suba.lin Ra,Rb,Rn*/

	/*ID: 2349, INST: subc.ro.x Da,Db,Dn*/

	/*ID: 2350, INST: subc.rw.leg.l Da,Db,Dn*/

	/*ID: 2351, INST: subc.wo.l #u5,Da,Dn*/

	/*ID: 2352, INST: subc.wo.l Da,Db,Dn*/

	/*ID: 2353, INST: subc.wo.x #u5,Da,Dn*/

	/*ID: 2354, INST: subc.wo.x Da,Db,Dn*/

	/*ID: 2355, INST: subc.wo.leg.x #u5,Da,Dn*/

	/*ID: 2356, INST: subc.wo.leg.x Da,Db,Dn*/

	/*ID: 2357, INST: swap.dx.4b Da,Dn*/
Word32 do__swap_dx_4b(Word32 __Da)
{
}

	/*ID: 2358, INST: swap.dx.8b Da,Db,Dmn*/
void do__swap_dx_8b(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 2359, INST: swap.ixi.4w Da,Db,Dmn*/
void do__swap_ixi_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 2360, INST: swap.sx.4b Da,Dn*/
Word32 do__swap_sx_4b(Word32 __Da)
{
}

	/*ID: 2361, INST: swap.sx.8b Da,Db,Dmn*/
void do__swap_sx_8b(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 2362, INST: swap.sx.4w Da,Db,Dmn*/
void do__swap_sx_4w(Word32 __Da, Word32 __Db, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 2363, INST: swb*/
void do__swb()
{
}

	/*ID: 2364, INST: swbe*/
void do__swbe()
{
}

	/*ID: 2365, INST: sxt.b.2t Da,Dn*/

	/*ID: 2366, INST: sxt.b.4t Dab,Dmn*/

	/*ID: 2367, INST: sxt.b.x Da,Dn*/

	/*ID: 2368, INST: sxt.b.2x Dab,Dmn*/

	/*ID: 2369, INST: sxt.l.x Da,Dn*/

	/*ID: 2370, INST: sxt.l.2x Dab,Dmn*/

	/*ID: 2371, INST: sxt.w.2t Da,Dn*/

	/*ID: 2372, INST: sxt.w.4t Dab,Dmn*/

	/*ID: 2373, INST: sxt.w.x Da,Dn*/

	/*ID: 2374, INST: sxt.w.2x Dab,Dmn*/

	/*ID: 2375, INST: sxta.b.l Ra,Rn*/

	/*ID: 2376, INST: sxta.b.l Ra,Rn*/

	/*ID: 2377, INST: sxta.w.l Ra,Rn*/

	/*ID: 2378, INST: sxta.w.l Ra,Rn*/

	/*ID: 2379, INST: sync.b*/
void do__sync_b()
{
}

	/*ID: 2380, INST: sync.d*/
void do__sync_d()
{
}

	/*ID: 2381, INST: tfetch Ra,URelAdd7_t3*/

	/*ID: 2382, INST: tfetch.rel RelAdd19_t1,URelAdd7_t2*/

	/*ID: 2383, INST: tfetch.rts URelAdd7_t3*/

	/*ID: 2384, INST: tfetch.sbr Ra,URelAdd7_t3,URRelAdd4_t4*/

	/*ID: 2385, INST: tfetch.sbr.rel RelAdd19_t1,URelAdd7_t2,URRelAdd4_t3*/

	/*ID: 2386, INST: tfr.x #s32,Dn*/

	/*ID: 2387, INST: tfr.x Da,Dn*/

	/*ID: 2388, INST: tfr.2x #s32,Dmn*/

	/*ID: 2389, INST: tfr.2x Da,Db,Dmn*/

	/*ID: 2390, INST: tfra C8,Rn*/

	/*ID: 2391, INST: tfra Ra,C7*/

	/*ID: 2392, INST: tfra Ra,sp*/

	/*ID: 2393, INST: tfra sp,Rn*/

	/*ID: 2394, INST: tfra.l #s32_t5,Rn*/

	/*ID: 2395, INST: tfra.l #s32_t5,Rn*/

	/*ID: 2396, INST: tfra.l #s9,Rn*/

	/*ID: 2397, INST: tfra.l #s9,Rn*/

	/*ID: 2398, INST: tfrca Ra,sp*/

	/*ID: 2399, INST: tfrca sp,Rn*/

	/*ID: 2400, INST: trap.1 #u10*/

	/*ID: 2401, INST: trap.0 #u10*/

	/*ID: 2402, INST: tstbm.c.l #u32,Da,Pmn*/
int do__tstbm_c_l(Word32 __imm1, Word32 __Da)
{
	return ((__imm1 & __Da) == 0) ? 1 : 0;
}

	/*ID: 2403, INST: tstbm.c.l #u32,Da,Pn*/

	/*ID: 2404, INST: tstbm.s.l #u32,Da,Pmn*/
int do__tstbm_s_l(Word32 __imm1, Word32 __Da)
{
	return ((__imm1 & ~__Da) == 0) ? 1 : 0;
}

	/*ID: 2405, INST: tstbm.s.l #u32,Da,Pn*/

	/*ID: 2406, INST: unpack.b.4t Da,Dmn*/
void do__unpack_b_4t(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 2407, INST: unpack.b.4w Da,Dmn*/
void do__unpack_b_4w(Word32 __Da, Word32* __Dm, Word32* __Dn)
{
}

	/*ID: 2408, INST: unpack.b.u.4t Da,Dmn*/
void do__unpack_b_u_4t(UWord32 __Da, UWord40* __Dm, UWord40* __Dn)
{
}

	/*ID: 2409, INST: unpack.bf.4t Da,Dmn*/
void do__unpack_bf_4t(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
}

	/*ID: 2410, INST: unpack.bh.u.2t Da,Dn*/
UWord40 do__unpack_bh_u_2t(UWord32 __Da)
{
}

	/*ID: 2411, INST: unpack.bl.u.2t Da,Dn*/
UWord40 do__unpack_bl_u_2t(UWord32 __Da)
{
}

	/*ID: 2412, INST: unpack.thl.2f Da,Dmn*/
void do__unpack_thl_2f(Word40 __Da, Word40* __Dm, Word40* __Dn)
{
	Word32 __Da_WH, __Da_WL;
	Word8 __Dm_e, __Dn_e;

	__Da_WH = (__x_to_l(__Da) & 0xffff0000);
	__Da_WL = (__x_to_l(__Da) & 0xffff) << 16;
	__Dm_e = (__x_guard(__Da) & 0xf0) >> 4;
	__Dn_e = (__x_guard(__Da) & 0xf);
	if(__Dn_e & 0x8) __Dn_e |= 0xf0;
	if(__Dm_e & 0x8) __Dm_e |= 0xf0;
	*__Dm = __x_set(__Dm_e,__Da_WH);
	*__Dn = __x_set(__Dn_e,__Da_WL);
}

	/*ID: 2413, INST: unpack.thl.2x Da,Dmn*/
void do__unpack_thl_2x(Word40 __Da, Word40* __Dm, Word40* __Dn)
{
	Word32 __Da_WH, __Da_WL;
	Word8 __Dm_e=0, __Dn_e=0;

	__Da_WH = __extract_wh(__Da);
	__Da_WL = __extract_wl(__Da);
	if(__Da_WH & 0x80000)
	{
		__Da_WH |= 0xfff00000;
		__Dm_e = 0xff;
	}

	if(__Da_WL & 0x80000)
	{
		__Da_WL |= 0xfff00000;
		__Dn_e = 0xff;
	}

	*__Dm = __x_set(__Dm_e,__Da_WH);
	*__Dn = __x_set(__Dn_e,__Da_WL);
}

	/*ID: 2414, INST: unpack.tlh.2f Da,Dmn*/
void do__unpack_tlh_2f(Word40 __Da, Word40* __Dm, Word40* __Dn)
{
	Word32 WH,WL;
	Word8 ext_wh,ext_wl,ext;
	ext = __x_guard(__Da);
	ext_wh = (ext&0xF0)>>4;
	ext_wl = ext&0x0F;
	if(ext_wl & 0x08)ext_wl = ext_wl|0xF0;/*sign_extend*/
	WH = __x_to_l(__Da) & 0xFFFF0000;
	WL = (__x_to_l(__Da) & 0xFFFF)<<16;
	*__Dm = __x_set(ext_wl,WL);
	*__Dn = __x_set(ext_wh,WH);
}

	/*ID: 2415, INST: unpack.tlh.2x Da,Dmn*/
void do__unpack_tlh_2x(Word40 __Da, Word40* __Dm, Word40* __Dn)
{
	Word32 WH,WL;
	WH = __extract_wh(__Da);
	WL = __extract_wl(__Da);
	*__Dm = __l_to_x(WL);
	*__Dn = __l_to_x(WH);
}

	/*ID: 2416, INST: unpack.whl.2f Da,Dmn*/
void do__unpack_whl_2f(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word8 SE = 0;
	Word32 __Da_H, __Da_L;
	__Da_L = ((__Da & 0xffff) << 16) & 0xffff0000;
	__Da_H = (__Da & 0xffff0000);

	if(__Da_L & 0x80000000) SE = 0xff;
	*__Dn = __x_set(SE, __Da_L);
	SE = 0;
	if(__Da_H & 0x80000000) SE = 0xff;
	*__Dm = __x_set(SE, __Da_H);
}

	/*ID: 2417, INST: unpack.whl.2x Da,Dmn*/
void do__unpack_whl_2x(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word32 WH,WL;
	WH = __l_get_msb(__Da);
	WL = __l_get_lsb(__Da);
	*__Dn = __l_to_x(WL);
	*__Dm = __l_to_x(WH);
}

	/*ID: 2418, INST: unpack.wlh.2f Da,Dmn*/
void do__unpack_wlh_2f(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word8 SE = 0;
	Word32 __Da_H, __Da_L;
	__Da_L = ((__Da & 0xffff) << 16) & 0xffff0000;
	__Da_H = (__Da & 0xffff0000);

	if(__Da_L & 0x80000000) SE = 0xff;
	*__Dm = __x_set(SE, __Da_L);
	SE = 0;
	if(__Da_H & 0x80000000) SE = 0xff;
	*__Dn = __x_set(SE, __Da_H);
}

	/*ID: 2419, INST: unpack.wlh.2x Da,Dmn*/
void do__unpack_wlh_2x(Word32 __Da, Word40* __Dm, Word40* __Dn)
{
	Word32 __Da_WH, __Da_WL;
	Word8 SE = 0;
	__Da_WH = (__Da >> 16);
	__Da_WL = ((__Da & 0xFFFF) << 16) >> 16;

	if(__Da_WH & 0x80000000) SE = 0xFF;
		*__Dn = __x_set(SE,__Da_WH);

	SE = 0;

	if(__Da_WL & 0x80000000) SE = 0xFF;
		*__Dm = __x_set(SE, __Da_WL);
}

	/*ID: 2420, INST: vtrace.l #0,#2,Da,Dn*/
Word32 do__vtrace_l_0_2(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2421, INST: vtrace.l #0,#3,Da,Dn*/
Word32 do__vtrace_l_0_3(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2422, INST: vtrace.l #0,#4,Da,Dn*/
Word32 do__vtrace_l_0_4(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2423, INST: vtrace.l #1,#4,Da,Dn*/
Word32 do__vtrace_l_1_4(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2424, INST: vtrace.l #2,#4,Da,Dn*/
Word32 do__vtrace_l_2_4(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2425, INST: vtrace.l #3,#4,Da,Dn*/
Word32 do__vtrace_l_3_4(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2426, INST: vtrace.l #4,#4,Da,Dn*/
Word32 do__vtrace_l_4_4(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2427, INST: vtrace.l #5,#4,Da,Dn*/
Word32 do__vtrace_l_5_4(Word32 __Dn, Word32 __Da)
{
}

	/*ID: 2428, INST: zxt.b.x Da,Dn*/

	/*ID: 2429, INST: zxt.l.x Da,Dn*/

	/*ID: 2430, INST: zxt.w.x Da,Dn*/

	/*ID: 2431, INST: zxta.b.l Ra,Rn*/

	/*ID: 2432, INST: zxta.b.l Ra,Rn*/

	/*ID: 2433, INST: zxta.w.l Ra,Rn*/

	/*ID: 2434, INST: zxta.w.l Ra,Rn*/



