#ifndef CORTEXM_H
#define CORTEXM_H

#include "typedefs.h"

#define REG32(addr) (*(uint32_t *)(addr))

/* Globally disable all interrupts */
#define __disable_irq() asm("CPSID i")

/* Globally enable all interrupts */
#define __enable_irq() asm("CPSIE i")

/*
 * Enables an interrupt.
 *
 * Allows the corresponding interrupt source to interrupt the cpu
 */
void nvic_enable_irq(uint8_t irqn);

/*
 * Disables an interrupt.
 *
 * Prevents the corresponding interrupt source from interrupting the cpu
 */
void nvic_disable_irq(uint8_t irqn);

/*
 * Checks if an interrupt is pending.
 *
 * Returns 1 if the interrupt is pending; 0 otherwise.
 */
int nvic_get_pending_irq(uint8_t irqn);

/*
 * Sets an interrupt to pending.
 */
void nvic_set_pending_irq(uint8_t irqn);

/*
 * Clears a pending interrupt.
 */
void nvic_clear_pending_irq(uint8_t irqn);

/*
 * Sets the priority of an interrupt.
 */
void nvic_set_priority(uint8_t irqn, uint8_t prio);

/*
 * Checks the priority of an interrupt.
 */
uint8_t nvic_get_priority(uint8_t irqn);

/*
 * Turns on the SysTick
 */
void systick_start(void);

/*
 * Turns off the SysTick
 */
void systick_stop(void);

/*
 * Set the SysTick reload value
 *
 * Returns the new reload value. Use a reload value of 0 to simply return the
 * current reload value.
 */
uint32_t systick_reload(uint32_t reload);

/*
 * Get the current SysTick value
 *
 * Returns the current timer value. If 'clear' is not 0, clear the current timer
 * value.
 */
uint32_t systick_current(uint32_t clear);

/*
 * Clear SysTick count flag
 */
void systick_clear_flag(void);

#endif
