/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.v8.tracecommander.controller;

import java.io.File;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.config.ConfigurationUtils;
import com.freescale.sa.configurator.ls.jni.JniConfigurationUtils;
import com.freescale.sa.configurator.ls.v8.platform.PlatformTypeV8;
import com.freescale.sa.configurator.ui.tree.TreeDataNode;
import com.freescale.sa.ls.v8.launch.ui.ConfigureWindow;
import com.freescale.sa.ls.v8.launch.ui.LsV8ConfigurationContents;
import com.freescale.sa.ls.v8.launch.ui.LsV8TargetSettingsManager;
import com.freescale.sa.model.BoardType;
import com.freescale.sa.trace.configandcollect.TraceStreamManager;
import com.freescale.sa.trace.configandcollect.model.ITraceStream;
import com.freescale.sa.tracecommander.controller.StartStopTraceCollectionHandler;
import com.freescale.sa.tracecommander.controller.TraceConfigurationHandler;
import com.freescale.sa.tracecommander.model.PlatformConfiguration;
import com.freescale.sa.tracecommander.view.SAConfigDialog;
import com.freescale.sa.tracecommander.view.ViewConstants;
import com.freescale.sa.ui.v8.tracecommander.view.ArmV8ConfigDialog;

public class ArmV8TraceConfigurationHandler extends TraceConfigurationHandler {

	private static Logger LOGGER = Logger
			.getLogger(ArmV8TraceConfigurationHandler.class);
	public static final String TEMPLATE_CONFIG_FILE_LS_ARMv8 = "TEMPLATEConfigFile%s.xml"; //$NON-NLS-1$
	private static final String NO_NEW_BOARD = ""; //$NON-NLS-1$

	/**
	 * Handles the 'Configure trace' command for the case when it is called for
	 * an active launch session (e.g., from the 'Configure Trace' button in the
	 * debug menu). The difference compared to 'handleConfigCommandOffline' is
	 * that this version updates the state and icons of the SA commands in the
	 * debug menu (e,g,. start/stop trace, upload).
	 * 
	 * @param activeLaunchSession
	 *            the active launch session
	 * @param parentShell
	 *            parent shell to use for displaying the SA configuration dialog
	 */
	public static void handleConfigCommandOnline(File platformConfig,
			ILaunch activeLaunchSession, Shell parentShell) {
		SAConfigDialog sad = null;
		ILaunchConfiguration launchConfig = null;
		ITraceStream traceStream = null;
		int saDialogStatus = IDialogConstants.CANCEL_ID;

		if (activeLaunchSession != null) {
			launchConfig = activeLaunchSession.getLaunchConfiguration();
			if (!activeLaunchSession.isTerminated()) {
				traceStream = TraceStreamManager.getDefault().getTraceStream(
						activeLaunchSession);
			}
		} else {
			return;
		}

		try {
			sad = new ArmV8ConfigDialog(parentShell, launchConfig,
					platformConfig);
		} catch (Exception e) {
			LOGGER.error("Error opening SA config ", e); //$NON-NLS-1$
		}
		if (sad != null) {
			saDialogStatus = sad.open();
		}

		if (saDialogStatus == IDialogConstants.CANCEL_ID) {
			// Nothing to do.
			return;
		}

		if (traceStream == null) {// cannot create a SA stream, so make SA
									// disabled
			StartStopTraceCollectionHandler.setEnableAnalysisStateFromLaunch(
					activeLaunchSession, false);
			return;
		}

	}

	/**
	 * Handles the 'Configure trace' command for LTA, the case when it no
	 * launch. The difference compared to 'handleConfigCommandOnline' is that
	 * this version works 'offline'; i.e. the launch session does not have to be
	 * active.
	 * 
	 * @param platformCfgFile
	 *            the platform configuration file
	 * @param parentShell
	 *            parent shell to use for displaying the SA configuration dialog
	 */
	public static void handleConfigCommandOffline(File platformCfgFile,
			Shell parentShell) {
		SAConfigDialog sad = null;
		ILaunchConfiguration launchConfig = null;
		int saDialogStatus = IDialogConstants.CANCEL_ID;

		try {
			sad = new ArmV8ConfigDialog(parentShell, launchConfig,
					platformCfgFile);
		} catch (Exception e) {
			LOGGER.error("Error opening SA config ", e); //$NON-NLS-1$
		}
		if (sad != null) {
			saDialogStatus = sad.open();
		}

		if (saDialogStatus == IDialogConstants.CANCEL_ID) {
			// Nothing to do.
			return;
		}
	}

	public static void handleConfigAction(PlatformConfiguration platformCfg) {
		if (platformCfg != null) {
			ILaunch launch = platformCfg.getLaunch();
			String targetAcces = platformCfg.getTargetAccess();
			if ((targetAcces.equals(ViewConstants.TARGET_ACCESS_GTA))
					&& (launch != null)) {
				handleConfigCommandOnline(platformCfg.getPlatCfgFile(), launch,
						Display.getDefault().getActiveShell());
			} else {
				handleConfigCommandOffline(platformCfg.getPlatCfgFile(),
						Display.getDefault().getActiveShell());
			}
		}
	}

	public static void handleTraceConfigBlockAction(
			PlatformConfiguration platformCfg, String nodeName) {

		Composite comp = (Composite) (Display.getDefault().getActiveShell());
		LsV8TargetSettingsManager lsV8TargetSettingsMan = new LsV8TargetSettingsManager(
				null);
		lsV8TargetSettingsMan.setPlatformConfigFile(platformCfg
				.getPlatCfgFile());
		lsV8TargetSettingsMan.readConfigurationAttributes();

		LsV8ConfigurationContents saContents = new LsV8ConfigurationContents();
		if (saContents instanceof LsV8ConfigurationContents) {
			saContents.setParent(comp);
			saContents.createConfigControls(comp);
		}

		ConfigureWindow confWindow = new ConfigureWindow(Display.getDefault()
				.getActiveShell(), null, lsV8TargetSettingsMan, saContents);
		confWindow.create();

		// get from ConfigureWindow the node that has to be selected
		Vector<TreeDataNode> allElements = confWindow.getAllTreeElements();
		TreeDataNode nodeToSelect = null;
		for (TreeDataNode node : allElements) {
			// There are cases (like STM) where the trace generator name
			// contains also the description of the trace generator type (like
			// System Trace Macrocell)
            if ((node.getName().equals(nodeName)) || (node.getName().contains(nodeName))
                || (node.getUiName().equals(nodeName)) || (node.getUiName().contains(nodeName))) {
                nodeToSelect = node;
                break;
            }
		}

		// fire tree selection event
		if (nodeToSelect != null) {
			confWindow.getTree().setSelection(
					new StructuredSelection(nodeToSelect), true);
			confWindow.open();
		}
	}

	public static PlatformConfiguration handleNewPlatformConfig(String newName,
			String boardName) {
		PlatformConfiguration newPlatfConfig = null;
		IPath platCfgFolder = ConfigurationUtils
				.getDefaultPlatformConfigStoragePath();

		// add .xml extension, if it doesn't exist
		String newFileName;
		if (newName.endsWith("." + SaConstants.PLATFORM_CONFIG_EXTENSION)) { //$NON-NLS-1$
			newFileName = newName;
		} else {
			newFileName = newName + "." //$NON-NLS-1$
					+ SaConstants.PLATFORM_CONFIG_EXTENSION;
		}

		IPath newPath = platCfgFolder.append(newFileName);
		if (!newPath.toFile().exists() && !boardName.equals(NO_NEW_BOARD)) {
			String platformConfigTemplate;
			platformConfigTemplate = BoardType.getPlatformConfigTemplate(
					boardName, PlatformTypeV8.ARMV8_OPTIONAL_PREFIX);
			if (platformConfigTemplate == null) {
				platformConfigTemplate = String.format(
						TEMPLATE_CONFIG_FILE_LS_ARMv8, boardName);
			}
			// If the file does not exist, create it using the default settings.
			File platCfgFile = JniConfigurationUtils.loadAndStoreDefaultConfig(
					newPath, platformConfigTemplate);
			newPlatfConfig = new PlatformConfiguration(platCfgFile);
		}

		return newPlatfConfig;
	}
}
