/*******************************************************************************
 * Copyright (c) 2015-2016: Freescale Semiconductor, Inc. All Rights Reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.v8.tracecommander.controller;

import java.lang.reflect.InvocationTargetException;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.IProgressService;

import com.freescale.sa.Messages;
import com.freescale.sa.ls.v8.applyconfig.jni.LSV8ApplyConfig_JNI;
import com.freescale.sa.tracecommander.controller.IHardwareConfigurator;
import com.freescale.sa.ui.v8.tracecommander.Activator;
import com.freescale.sa.util.CommonConstants;

/**
 * This class manages the configuration of ARMv8 targets. This class is a
 * sigleton.
 * 
 * @author R70188
 *
 */
public class ArmV8HardwareConfigurator implements IHardwareConfigurator {

	private LSV8ApplyConfig_JNI m_configurator = null;
	private static ArmV8HardwareConfigurator m_instance = new ArmV8HardwareConfigurator();
	private boolean m_connected = false;
	private static Logger LOGGER = Logger
			.getLogger(ArmV8HardwareConfigurator.class);
	private boolean m_configJobDone = false;

	/**
	 * Protected constructor to implement singleton
	 */
	protected ArmV8HardwareConfigurator() {
	}

	/**
	 * Retrieves the class sigleton instance
	 * 
	 * @return singleton instance
	 */
	public static ArmV8HardwareConfigurator getInstance() {
		return m_instance;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.freescale.sa.tracecommander.controller.IHardwareConfigurator#configure
	 * (java.lang.String)
	 */
	@Override
	public boolean connect(String platCfgFile) {

		final String in_platCfgFile = platCfgFile;

		IProgressService progressService = PlatformUI.getWorkbench()
				.getProgressService();

		try {
			progressService.run(true, true, new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor) {
					m_configJobDone = false;
					monitor.beginTask(
							Messages.ArmV8HardwareConfigurator_config,
							IProgressMonitor.UNKNOWN); //$NON-NLS-1$
					m_configurator = new LSV8ApplyConfig_JNI(in_platCfgFile);

					Job applyConfigJob = new Job("Writing trace configuration ") { //$NON-NLS-1$

						@Override
						protected IStatus run(IProgressMonitor monitor) {
							if (m_configurator.applyConfig()) {
								m_connected = true;
							} else {
								m_connected = false;
							}
							Status status = new Status(IStatus.OK,
									Activator.PLUGIN_ID, CommonConstants.EMPTY_STRING);
							m_configJobDone = true;
							return status;
						}

					};

					applyConfigJob.schedule();

					while (!m_configJobDone) {
						try {
							if (monitor.isCanceled()) {
								m_configurator.cancelApplying();
								m_configJobDone = true;
							}
							Thread.sleep(1000);
						} catch (InterruptedException e) {
							m_configJobDone = true;
							LOGGER.error(e);
						}
					}

				}

			});
		} catch (InvocationTargetException e) {
			LOGGER.error("Error while writing trace configuration", e); //$NON-NLS-1$
			m_connected = false;
		} catch (InterruptedException e) {
			LOGGER.warn("Trace configuration has been interrupted ", e); //$NON-NLS-1$
			m_connected = false;
		}

		return m_connected;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.freescale.sa.tracecommander.controller.IHardwareConfigurator#
	 * controlSource(java.lang.String, boolean)
	 */
	@Override
	public boolean controlSource(String sourceName, boolean startStopSwitch, boolean force) {
		if (m_configurator != null && m_connected) {
			return m_configurator.controlTraceSource(sourceName,
					startStopSwitch, force);
		} else {
			return true;
		}

	}

	@Override
	public boolean getTraceBufferMonitorStatus() {
		if (m_configurator != null) {
			return m_configurator.useTraceBufferMonitoring();
		} else {
			return false;
		}
	}

	@Override
	public int bufferFillLevel(String dataStreamName) {
		if (m_configurator != null && m_connected) {
			return m_configurator.getFillLevelTrace(dataStreamName);
		} else {
			return 0;
		}
	}

	@Override
	public void disconnect() {
		m_connected = false;

	}

}
