/*******************************************************************************
 * Copyright (c) 2014-2016: Freescale Semiconductor, Inc. All Rights Reserved.
 * Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.v8.tracecommander.controller;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.ls.jni.LSConfigLoader_JNI;
import com.freescale.sa.configurator.ls.v8.platform.PlatformTypeV8;
import com.freescale.sa.ls.v8.applyconfig.jni.LSV8JNIConfigurationUtils;
import com.freescale.sa.ls.v8.launch.trace_stream.LaunchContext;
import com.freescale.sa.ls.v8.launch.trace_stream.LsV8TraceStream;
import com.freescale.sa.ls.v8.launch.trace_stream.TCCContext;
import com.freescale.sa.ls.v8.launch.ui.LsV8LaunchUtils;
import com.freescale.sa.trace.configandcollect.TraceStreamManager;
import com.freescale.sa.trace.configandcollect.model.ITraceStream;
import com.freescale.sa.tracecommander.controller.ConnectDisconnectTraceCollectionHandler;
import com.freescale.sa.tracecommander.model.PlatformConfiguration;
import com.freescale.sa.ui.v8.tracecommander.utils.TraceCommanderUtils;
import com.freescale.sa.util.Utils;

/**
 * 
 * Starts trace collection for the trace stream associated with the currently
 * selected debug target.
 */
public class ArmV8ConnectDisconnectTraceCollectionHandler
        extends ConnectDisconnectTraceCollectionHandler {

    private static Logger LOGGER = Logger.getLogger(ArmV8ConnectDisconnectTraceCollectionHandler.class);
    private static String ARMv8_COLLECTOR_CDTC = "C-DTC"; //$NON-NLS-1$

    private static String CONFIG_PLATFORM_CONFIG = "CONFIG_PLATFORM_CONFIG"; //$NON-NLS-1$
    private static String PROBE_PLATFORM_CONFIG = "PROBE_PLATFORM_CONFIG"; //$NON-NLS-1$
    private static String PROBE_EXECUTABLE = "PROBE_EXECUTABLE"; //$NON-NLS-1$
    private static String OUTPUT_FILE = "OUTPUT_FILE"; //$NON-NLS-1$
    private static String SA_CALLBACK_INIT_FILE = "saCallback.ini"; //$NON-NLS-1$

    private static String KA_CALL_SET = "ka-callback-set "; //$NON-NLS-1$

    static private class UBootProperties {
        private HashMap<String, String> properties = new HashMap<String, String>();

        public UBootProperties() {
        }

        public void setProperty(String name, String value) {
            properties.put(name, value);
        }

        public void store(String file) {
            BufferedWriter bufWriter = null;
            try {
                FileWriter fileWriter = new FileWriter(file);
                bufWriter = new BufferedWriter(fileWriter);

                // Write section
                bufWriter.write("[Default]\n");
                for (Entry<String, String> entry : properties.entrySet()) {
                    bufWriter.write(entry.getKey() + "=" + entry.getValue() + "\n");
                }
            } catch (IOException e) {
                LOGGER.error(e.getLocalizedMessage());
            } finally {
                if (bufWriter != null) {
                    try {
                        bufWriter.close();
                    } catch (IOException e) {
                        LOGGER.error(e.getLocalizedMessage());
                    }
                }
            }
        }

    }

    private static String registerUbootCallback(LSConfigLoader_JNI configurator,
                                                File platCfgFile) throws Exception {

        // Create a new Platform configuration file with
        // TraceLocation = DTC
        String configFilePath = LSV8JNIConfigurationUtils.createUbootConfigFile(platCfgFile.getPath());
        TCCContext context = new TCCContext(platCfgFile);

        // Save init data for sa callback
        UBootProperties uBootProperties = new UBootProperties();
        uBootProperties.setProperty(SaConstants.SA_HOME_DIR_VAR, Utils.getCWSAHome());
        uBootProperties.setProperty(SaConstants.SA_CONFIG_DIR_VAR, Utils.getCWSAConfig());
        uBootProperties.setProperty(PROBE_PLATFORM_CONFIG, configFilePath);
        uBootProperties.setProperty(CONFIG_PLATFORM_CONFIG, platCfgFile.getPath());
        uBootProperties.setProperty(PROBE_EXECUTABLE,
                                    LsV8LaunchUtils.getProbeExecutable(configurator));
        uBootProperties.setProperty(OUTPUT_FILE,
                                    LsV8LaunchUtils.getOutputTraceFileLocation(configurator,
                                                                               context.getBaseName(),
                                                                               "_" + LsV8LaunchUtils.KA_UBOOT_RELOC, //$NON-NLS-1$
                                                                               null));
        File uBootCallback = new File(LsV8LaunchUtils.SA_UBOOT_PYTHON_CALLBACK);

        if (uBootCallback.exists()) {
            File uBootPropertiesFile = new File(uBootCallback.getParentFile(),
                                                SA_CALLBACK_INIT_FILE);
            uBootProperties.store(uBootPropertiesFile.getAbsolutePath());
        } else {
            LOGGER.error("[registerUbootCallback] Missing `" + uBootCallback.getAbsolutePath() //$NON-NLS-1$
                         + "`"); //$NON-NLS-1$
        }

        // Remove previously defined callback
        LsV8LaunchUtils.removeUbootCallback();

        // Register Python callback for U-Boot relocation
        LsV8LaunchUtils.queueConsoleCommand(KA_CALL_SET + LsV8LaunchUtils.KA_UBOOT_RELOC + " " //$NON-NLS-1$
                                            + LsV8LaunchUtils.SA_UBOOT_PYTHON_CALLBACK.replaceAll("\\\\", //$NON-NLS-1$
                                                                                                  "/"));

        return configFilePath;
    }

    public static boolean connect(PlatformConfiguration platformCfgNode) {
        // on connect, if target access is TCC, set selected platform
        // configuration file as active config
        boolean statusOk = true;
        ArmV8HardwareConfigurator hwConfig = ArmV8HardwareConfigurator.getInstance();

        if (platformCfgNode != null) {
            // Apply config producer
            File platCfgFile = platformCfgNode.getPlatCfgFile();
            if (platCfgFile == null) {
                return false;
            }

            String configFilePath = platCfgFile.getPath();

            /* Check if U-Boot support is enabled */
            LSConfigLoader_JNI configurator = new LSConfigLoader_JNI(platCfgFile.getPath());

            if (LSV8JNIConfigurationUtils.isUbootSupportEnabled(configurator)
                && !TraceCommanderUtils.isUBootRelocated()) {
                try {
                    configFilePath = registerUbootCallback(configurator, platCfgFile);
                } catch (Exception e) {
                    LOGGER.error("Failed to register U-Boot callback : " //$NON-NLS-1$
                                 + e.getLocalizedMessage());
                }
            }

            try {
                statusOk = hwConfig.connect(configFilePath);
            } catch (Exception e) {
                LOGGER.error("Error at appling configuration!", e); //$NON-NLS-1$
            }

        }

        return statusOk;
    }

    public static void disconnect(PlatformConfiguration platformCfgNode) {
        ArmV8HardwareConfigurator hwConfig = ArmV8HardwareConfigurator.getInstance();
        if (hwConfig != null) {
            hwConfig.disconnect();
        }
    }

    /**
     * Get the fill level of the trace buffer in percentage for the current
     * platform configuration
     * 
     * @param platformCfgNode
     *            the platform configuration
     * @return int fill level of the trace buffer in percentage
     */
    public static int getBufferFillLevel(PlatformConfiguration platformCfgNode) {

        // on connect, if target access is TCC, get buffer fill level
        int fillLevel = 0;
        ArmV8HardwareConfigurator hwConfig = ArmV8HardwareConfigurator.getInstance();
        if (platformCfgNode != null) {
            // use config producer to read buffer fill level via JNI
            try {
                fillLevel = hwConfig.bufferFillLevel(ARMv8_COLLECTOR_CDTC);
            } catch (Exception e) {
                LOGGER.error("Error at getting fill level for buffer collection trace!", e); //$NON-NLS-1$
            }
        }

        return fillLevel;
    }

    /**
     * Creates a trace stream and configures it.
     * 
     * @param launch
     *            The current launch.
     * @throws CoreException
     */
    public static void createTraceStream(PlatformConfiguration platfCfgNode) throws CoreException {

        if (platfCfgNode == null) {
            LOGGER.error("[createTraceStream] Error at creating a trace stream based on platfom configuration file!"); //$NON-NLS-1$
            return;
        }

        ILaunch launch = platfCfgNode.getLaunch();
        LaunchContext launchContext = null;
        if (launch != null) {
            ILaunchConfiguration launchConfig = launch.getLaunchConfiguration();
            launchContext = new LaunchContext(launchConfig);
        } else {
            LOGGER.error("[createTraceStream] No active launch!"); //$NON-NLS-1$
            return;
        }

        if (TraceStreamManager.getDefault().getTraceStream(launch) != null) {
            // If a stream is already associated with the launch, no need to
            // create a new one.
            return;
        }

        ITraceStream v8Stream = null;
        if (launchContext.getPlatformType() == PlatformTypeV8.TYPE_LSv8_HW) {
            v8Stream = (ITraceStream) new LsV8TraceStream(launch, launchContext);
        }
        if (v8Stream == null) {
            LOGGER.error("[createTraceStream] Error - ARMv8 trace stream is null."); //$NON-NLS-1$
            return;
        }
        TraceStreamManager.getDefault().addTraceStream(launch, v8Stream);
    }
}
