/*******************************************************************************
 * Copyright (c) 2011-2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ls.traceviewer.editors;


import java.net.URL;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.config.AbstractUiBindingConfiguration;
import org.eclipse.nebula.widgets.nattable.grid.GridRegion;
import org.eclipse.nebula.widgets.nattable.ui.NatEventData;
import org.eclipse.nebula.widgets.nattable.ui.binding.UiBindingRegistry;
import org.eclipse.nebula.widgets.nattable.ui.matcher.MouseEventMatcher;
import org.eclipse.nebula.widgets.nattable.ui.menu.IMenuItemProvider;
import org.eclipse.nebula.widgets.nattable.ui.menu.MenuItemProviders;
import org.eclipse.nebula.widgets.nattable.ui.menu.PopupMenuAction;
import org.eclipse.nebula.widgets.nattable.ui.menu.PopupMenuBuilder;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.ISharedImages;

import com.freescale.sa.ls.traceviewer.Activator;
import com.freescale.sa.ls.traceviewer.editors.TraceContentDataProvider.ColumnType;
import com.freescale.sa.ls.traceviewer.editors.TreeViewCommand.TreeCommand;

/**
 * Class that implements the context menu on the column header
 * @author R70188
 *
 */
public class TraceViewerHeaderMenuConfiguration extends AbstractUiBindingConfiguration {

	protected MenuItem m_liveHexDisplay;
	protected MenuItem m_liveDeltaDisplay;
	protected Menu m_colHeaderMenu;
	protected TraceContentDataProvider m_traceDataProvider;
	protected boolean m_treeView;
	
	/**
	 * Constructor
	 * @param natTable parent Nattable
	 * @param traceDataProvider data provider
	 * @param treeView true if the TreeLayer is included and false otherwise
	 */
	public TraceViewerHeaderMenuConfiguration(NatTable natTable, TraceContentDataProvider traceDataProvider, boolean treeView) {
		this.m_treeView = treeView;
		this.m_colHeaderMenu = createColumnHeaderMenu(natTable).build(); 
		this.m_traceDataProvider = traceDataProvider;

        
		//ensure that the menus will be disposed when the NatTable is disposed 
		natTable.addDisposeListener(new DisposeListener() { 
			@Override
		    public void widgetDisposed(DisposeEvent e) { 
		        if (m_colHeaderMenu != null) 
		        	m_colHeaderMenu.dispose(); 
		    } 
		});      
	}

	/**
	 * Creates the popup menu
	 * @param natTable parent NatTable component
	 * @return an instance of PopupMenuBuilder
	 */
	protected PopupMenuBuilder createColumnHeaderMenu(NatTable natTable) {
		PopupMenuBuilder builder = new PopupMenuBuilder(natTable)
				.withHideColumnMenuItem()
				.withShowAllColumnsMenuItem()
				.withColumnRenameDialog()
				.withMenuItemProvider(hexadecimalDisplayMenuProvider())
				.withMenuItemProvider(deltaDisplayMenuProvider());
		
		if (m_treeView) {
			builder = builder.withMenuItemProvider(collapseAllMenuProvider())
					.withMenuItemProvider(expandAllMenuProvider());
		}
		return builder;
	}
	
    /* (non-Javadoc) 
     * @see org.eclipse.nebula.widgets.nattable.config.IConfiguration#configureUiBindings(org.eclipse.nebula.widgets.nattable.ui.binding.UiBindingRegistry) 
     */
    @Override
    public void configureUiBindings(UiBindingRegistry uiBindingRegistry) { 
        if (this.m_colHeaderMenu != null) { 
            uiBindingRegistry.registerMouseDownBinding( 
                    new MouseEventMatcher(SWT.NONE, GridRegion.COLUMN_HEADER, MouseEventMatcher.RIGHT_BUTTON), 
                    new PopupMenuAction(this.m_colHeaderMenu) { 
                        @Override 
                        public void run(NatTable natTable, MouseEvent event) {
                        	if (event.data instanceof NatEventData) {
	                        	NatEventData natData = (NatEventData)event.data;
	                        	
	                        	// We need the base data layer column number, not the transformed one
	                        	int realColumnIndex = natTable.getColumnIndexByPosition(natData.getColumnPosition());
	                        	if (m_traceDataProvider.getColumnType(realColumnIndex) != ColumnType.NUMERIC) {
	                        		m_liveHexDisplay.setEnabled(false);
	                        		m_liveDeltaDisplay.setEnabled(false);
	                        		m_liveHexDisplay.setSelection(false);
	                        		m_liveDeltaDisplay.setSelection(false);
	                        	}
	                        	else {
	                        		m_liveHexDisplay.setEnabled(true);
	                        		m_liveDeltaDisplay.setEnabled(true);
	                        		m_liveHexDisplay.setSelection(m_traceDataProvider.getColumnHexDisplay(realColumnIndex));
	                        		m_liveDeltaDisplay.setSelection(m_traceDataProvider.getColumnDeltaDisplay(realColumnIndex));
	                        	}
                        	}
                        	super.run(natTable, event); 
                        }                        
                    }); 
        } 
  
    } 
  
	private IMenuItemProvider hexadecimalDisplayMenuProvider() {
		
		return new IMenuItemProvider() {

			public void addMenuItem(final NatTable natTable, final Menu popupMenu) {
				m_liveHexDisplay = new MenuItem(popupMenu, SWT.CHECK);
				m_liveHexDisplay.setText(Messages.TraceViewerHeaderMenuConfiguration_HexadecimalDisplay);
				m_liveHexDisplay.setEnabled(true);

				m_liveHexDisplay.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent event) {
						int columnIndex = MenuItemProviders.getNatEventData(event).getColumnPosition();
						natTable.doCommand(new HexadecimalDisplayCommand(natTable, columnIndex, m_liveHexDisplay.getSelection()));
					}
				});
			};
		};
	}
	
	private IMenuItemProvider deltaDisplayMenuProvider() {
		
		return new IMenuItemProvider() {

			public void addMenuItem(final NatTable natTable, final Menu popupMenu) {
				m_liveDeltaDisplay = new MenuItem(popupMenu, SWT.CHECK);
				m_liveDeltaDisplay.setText(Messages.TraceViewerHeaderMenuConfiguration_DeltaDisplay);
				m_liveDeltaDisplay.setEnabled(true);

				m_liveDeltaDisplay.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent event) {
						int columnIndex = MenuItemProviders.getNatEventData(event).getColumnPosition();
						natTable.doCommand(new DeltaDisplayCommand(natTable, columnIndex, m_liveDeltaDisplay.getSelection()));
					}
				});
			};
		};
	}
	
	private IMenuItemProvider collapseAllMenuProvider() {
		
		return new IMenuItemProvider() {

			public void addMenuItem(final NatTable natTable, final Menu popupMenu) {
				MenuItem collapseAllMenuItem = new MenuItem(popupMenu, SWT.PUSH);
				collapseAllMenuItem.setText(Messages.TraceViewerHeaderMenuConfiguration_CollapseAll);
                ISharedImages sharedImages = Activator.getDefault().getWorkbench().getSharedImages(); 
                collapseAllMenuItem.setImage(sharedImages.getImage(ISharedImages.IMG_ELCL_COLLAPSEALL));
                collapseAllMenuItem.setEnabled(true);


				collapseAllMenuItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent event) {
						natTable.doCommand(new TreeViewCommand(TreeCommand.COLLAPSE_ALL));
					}
				});
			};
		};
	}
	
	private IMenuItemProvider expandAllMenuProvider() {
		
		return new IMenuItemProvider() {

			public void addMenuItem(final NatTable natTable, final Menu popupMenu) {
				MenuItem expandAllMenuItem = new MenuItem(popupMenu, SWT.PUSH);
				expandAllMenuItem.setText(Messages.TraceViewerHeaderMenuConfiguration_ExpandAll);
        		URL expandAllUrl = FileLocator.find(Platform.getBundle("com.freescale.sa.ui.traceviewer"), //$NON-NLS-1$
        				new Path("icons/expandall.gif"), null); //$NON-NLS-1$
        		expandAllMenuItem.setImage(ImageDescriptor.createFromURL(expandAllUrl).createImage());
        		expandAllMenuItem.setEnabled(true);

                expandAllMenuItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent event) {
						natTable.doCommand(new TreeViewCommand(TreeCommand.EXPAND_ALL));
					}
				});
			};
		};
	}		
	
}
