/*******************************************************************************
 * Copyright (c) 2014-2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ls.traceviewer.editors;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.window.DefaultToolTip;
import org.eclipse.jface.window.ToolTip;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.config.DefaultNatTableStyleConfiguration;
import org.eclipse.nebula.widgets.nattable.grid.GridRegion;
import org.eclipse.nebula.widgets.nattable.grid.cell.AlternatingRowConfigLabelAccumulator;
import org.eclipse.nebula.widgets.nattable.grid.data.DefaultCornerDataProvider;
import org.eclipse.nebula.widgets.nattable.grid.layer.ColumnHeaderLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.CornerLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.GridLayer;
import org.eclipse.nebula.widgets.nattable.grid.layer.RowHeaderLayer;
import org.eclipse.nebula.widgets.nattable.hideshow.ColumnHideShowLayer;
import org.eclipse.nebula.widgets.nattable.layer.DataLayer;
import org.eclipse.nebula.widgets.nattable.layer.ILayer;
import org.eclipse.nebula.widgets.nattable.layer.LayerUtil;
import org.eclipse.nebula.widgets.nattable.layer.cell.ILayerCell;
import org.eclipse.nebula.widgets.nattable.layer.config.DefaultColumnHeaderStyleConfiguration;
import org.eclipse.nebula.widgets.nattable.reorder.ColumnReorderLayer;
import org.eclipse.nebula.widgets.nattable.resize.command.ColumnResizeCommand;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer;
import org.eclipse.nebula.widgets.nattable.selection.config.DefaultSelectionBindings;
import org.eclipse.nebula.widgets.nattable.selection.config.DefaultSelectionStyleConfiguration;
import org.eclipse.nebula.widgets.nattable.tree.action.TreeExpandCollapseAction;
import org.eclipse.nebula.widgets.nattable.tree.config.DefaultTreeLayerConfiguration;
import org.eclipse.nebula.widgets.nattable.tree.painter.TreeImagePainter;
import org.eclipse.nebula.widgets.nattable.ui.binding.UiBindingRegistry;
import org.eclipse.nebula.widgets.nattable.ui.matcher.CellPainterMouseEventMatcher;
import org.eclipse.nebula.widgets.nattable.ui.matcher.KeyEventMatcher;
import org.eclipse.nebula.widgets.nattable.ui.matcher.MouseEventMatcher;
import org.eclipse.nebula.widgets.nattable.util.GUIHelper;
import org.eclipse.nebula.widgets.nattable.viewport.ViewportLayer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.FileStoreEditorInput;
import org.eclipse.ui.part.FileEditorInput;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.ls.jni.LSConfigLoader_JNI;
import com.freescale.sa.ls.traceviewer.editors.TraceContentDataProvider.ColumnType;
import com.freescale.sa.ls.traceviewer.search.TraceViewerSearchAction;
import com.freescale.sa.relocation.SourceCodeCorrelation;
import com.freescale.sa.sourcebrowser.ls.internal.SourceBrowserManager;
import com.freescale.sa.sourcebrowser.ls.jni.LSSourceBrowser_JNI;
import com.freescale.sa.sourcebrowser.ls.jni.SourceBrowserData;
import com.freescale.sa.ui.common.AbstractSaEditor;
import com.freescale.sa.ui.common.SAUIUtils;
import com.freescale.sa.ui.common.utils.IOkraDecoder;
import com.freescale.sa.ui.common.utils.SaUiLsUtils;
import com.freescale.sa.ui.common.utils.StorageUtils;
import com.freescale.sa.util.CommonConstants;

/**
 * Class that implements the TraceViewer Eclipse Editor
 * 
 * @author R70188
 * 
 */
public class TraceViewer extends AbstractSaEditor {

	private static Logger LOGGER = Logger.getLogger(TraceViewer.class);
	protected NatTable m_natTable;
	private final int PERFORMANCE_THRESHOLD_LINE_COUNT = 100000;
	private final int TEXT_COLUMN_LARGE_WIDTH = 250;
	private final int FIRST_COLUMN = 1;
	private static final String IOKRA_ID = "com.freescale.sa.ui.traceviewer.okraTraceDecoder"; //$NON-NLS-1$	
	private String m_traceFileName;
	private String m_csvFileName;
	private String baseDir;
	private String xmlFile;
	private boolean m_canceled;
	private boolean m_decoded;
	private TraceContentDataProvider m_traceDataProviderCached;
	private TraceViewerDataLayer m_dataLayer;
	private TraceViewerTreeLayer m_treeLayer;
    private ArrayList<Object> m_okraDecoderExtensions;
	private static final String CSV_EXTENSION = ".csv"; //$NON-NLS-1$
	private String m_baseName = CommonConstants.EMPTY_STRING;
	
	private static final Map<String, String> tooltipMap = new HashMap<String, String>(){

		private static final long serialVersionUID = 7312647360913585303L;

		{
			put("TC", "Task create event");
			put("TS", "Task terminate event");
			put("TT", "Task terminate event");
			put("Core", "Core number");
			put("Task", "Task number");
			put("Prev_Task", "Previous task number");
			put("Next_Task", "Next task number");
			put("PC", "Program Counter");
			put("AccId", "The accelerator ID that triggers a task switch");

			put("TMAN", "Timer Manager calls");
			put("MFLU", "Multifield lookup unit calls");
			put("CTLU", "Clasifier and table lookup unit calls");
			put("TLU", "Table lookup unit");
			put("FPMA", "Frame Presentation DMA calls");
			put("CDMA", "Context DMA calls");
			put("FOMA", "Frame Output DMA calls");
			put("STE", "Statistics engine calls");

			put("EPID", "Entry point ID");
			put("ICount", "Instructions counter");
			put("CCount", "Cycles counter estimated value");
			put("DMA", "Frame DMA calls");
			put("SRAM_R", "Shared RAM reads");
			put("SRAM_RB", "Shared RAM read bytes");

			put("STATS", "Statistics engine calls");
		}
	};

	/**
	 * <p>
	 * The unique ID of this editor.
	 * </p>
	 */
	public static final String ID = "com.freescale.sa.ui.traceviewer"; //$NON-NLS-1$
	
	/**
	 * Constructor
	 */
	public TraceViewer() {
		super(ID, IOKRA_ID);
        m_okraDecoderExtensions = SaUiLsUtils.createOkraObjectsVector(IOKRA_ID);
	}

	/*
	 * (non-Javadoc)
	 * @see com.freescale.sa.ls.traceviewer.editors.AbstractSaLsEditor#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();

		if (m_natTable != null) {
			m_natTable.dispose();
		}

		// delete csv file if the decoding has been canceled
		if (m_canceled) {
			File csvFile = new File(m_csvFileName);
			boolean deleted = csvFile.delete();
			if (!deleted) {
				LOGGER.error("Cannot delete cvs file!"); //$NON-NLS-1$
			}
		}
	}

	/*
	 * (non-Javadoc)
     * @see com.freescale.sa.ls.traceviewer.editors.AbstractSaLsEditor#
     * createPartControl0 (org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl0(Composite parent) {
		//If the file has already CSV content do not decode
		IOkraDecoder.DecoderReturn ret = IOkraDecoder.DecoderReturn.DECODER_OK;
		if (StorageUtils.fileHasCsvContent(StorageUtils.getFileExtension(m_traceFileName))) {
			m_decoded = true;
			m_csvFileName = m_traceFileName;
		} else {
			ret = decodeTrace();
		}
	
		//Show partial decoded trace.
        if ((ret == IOkraDecoder.DecoderReturn.DECODER_OK
             || ret == IOkraDecoder.DecoderReturn.DECODER_CANCELLED)
            && m_decoded && m_csvFileName != null) {
			/* Identify the folder generated after decoding an archived trace */
            if ((m_traceFileName != null)
                && (m_traceFileName.endsWith(SaConstants.USER_SPACE_TRACE_ARCHIVE_EXTENSION)
					|| m_traceFileName.endsWith(SaConstants.KERNEL_SPACE_TRACE_ARCHIVE_EXTENSION)
					|| m_traceFileName.endsWith(SaConstants.SYSTEM_TRACE_ARCHIVE_EXTENSION))) {
				final String csvFile = new File(m_csvFileName).getName();

				String parentFolder = null;
				if (new Path(m_traceFileName).segmentCount() != 0) {
					parentFolder = new Path(m_traceFileName).removeLastSegments(1).toString();
				}

				File[] tempFolders = new File(parentFolder).listFiles(new FileFilter() {
                    @Override
                    public boolean accept(File file) {
                        // Match Linux storage folders
                        return StorageUtils.matchLinuxStorageFolder(file);
                        // digits
                    }
				});
				
				// Sort files by last modification date and pick the last one
				Arrays.sort(tempFolders, new Comparator<File>() {
					@Override
					public int compare(File arg0, File arg1) {
						return Long.valueOf(arg1.lastModified() - arg0.lastModified()).intValue();
					}
				});
				
				if (tempFolders.length != 0) {
					m_csvFileName = tempFolders[0].getAbsolutePath()  + Path.SEPARATOR + csvFile;
					if (!new File(m_csvFileName).exists()) {
						for (File file : tempFolders[0].listFiles()) {
							if (file.getAbsolutePath().endsWith(CSV_EXTENSION)) {
								m_csvFileName = file.getAbsolutePath();
							}
						}
					}
					StorageUtils.addCustomFolder(tempFolders[0].getAbsolutePath());
				}
			}
			
			m_baseName = new Path(m_csvFileName).removeFileExtension().lastSegment();
			
			// First line represents the names of table columns
            m_traceDataProviderCached = new TraceContentDataProvider(m_csvFileName);

            final String[] columnNames = m_traceDataProviderCached.getColumnNames();
			if (columnNames == null) {
				return;
			}

			// Adapt functionality according to the file size
			// Tree View will be available only for smaller csv files
			SelectionLayer selectionLayer = null;
			m_treeLayer = null;
			TraceTreeRowModel traceTreeRowModel = null;
			m_dataLayer = new TraceViewerDataLayer(m_traceDataProviderCached);

			// Set cell label accumulator
            m_dataLayer.setConfigLabelAccumulator(new TraceViewerLabelAccumulator(m_dataLayer));

			if (m_traceDataProviderCached.getRowCount() <= PERFORMANCE_THRESHOLD_LINE_COUNT) {
                ColumnReorderLayer columnReorderLayer = new ColumnReorderLayer(m_dataLayer);
                ColumnHideShowLayer columnHideShowLayer = new ColumnHideShowLayer(columnReorderLayer);

                traceTreeRowModel = new TraceTreeRowModel(m_traceDataProviderCached);
                m_treeLayer = new TraceViewerTreeLayer(columnHideShowLayer, traceTreeRowModel,
                                                       false);
				selectionLayer = new SelectionLayer(m_treeLayer);
			} else {
                selectionLayer = new SelectionLayer(new ColumnHideShowLayer(new ColumnReorderLayer(m_dataLayer)));
			}

			ViewportLayer viewportLayer = new ViewportLayer(selectionLayer);

            CustomizableColumnHeaderDataProvider columnHeaderDataProvider = new CustomizableColumnHeaderDataProvider(columnNames);
            DataLayer columnHeaderDataLayer = new DataLayer(columnHeaderDataProvider);
            ILayer columnHeaderLayer = new ColumnHeaderLayer(columnHeaderDataLayer, viewportLayer,
                                                             selectionLayer);

            TraceRowHeaderDataProvider rowHeaderDataProvider = new TraceRowHeaderDataProvider(m_traceDataProviderCached);
			DataLayer rowHeaderDataLayer = new DataLayer(rowHeaderDataProvider);
            ILayer rowHeaderLayer = new RowHeaderLayer(rowHeaderDataLayer, viewportLayer,
                                                       selectionLayer);

            ILayer cornerLayer = new CornerLayer(new DataLayer(new DefaultCornerDataProvider(columnHeaderDataProvider,
                                                                                             rowHeaderDataProvider)),
                                                 rowHeaderLayer, columnHeaderLayer);

            GridLayer compositeLayer = new GridLayer(viewportLayer, columnHeaderLayer,
                                                     rowHeaderLayer, cornerLayer);
			
			// For NatTable 1.1.0 
            // Set alternating colors for rows for both collapsed and expanded
            // tree modes
            // This is done because the change "Bug 432127 corrected even/odd
            // row label calculation
            // to be index based instead of position based" produces confusing
            // line coloring scheme
			// when Trace Events are collapsed
			compositeLayer.setConfigLabelAccumulatorForRegion(GridRegion.BODY, 
					new AlternatingRowConfigLabelAccumulator(null));

			m_natTable = new NatTable(parent, compositeLayer, false);
            m_natTable.addConfiguration(new TraceViewerHeaderMenuConfiguration(m_natTable,
                                                                               m_traceDataProviderCached,
							m_treeLayer != null));
			m_natTable.addConfiguration(new DefaultNatTableStyleConfiguration());
			// Change default bold on selection behavior
			DefaultSelectionStyleConfiguration selectionStyleConfiguration = new DefaultSelectionStyleConfiguration();
			DefaultColumnHeaderStyleConfiguration defaultColumnHeaderStyle = new DefaultColumnHeaderStyleConfiguration();
			FontData selectionFontData = GUIHelper.DEFAULT_FONT.getFontData()[0];
            FontData selectionHeaderFontData = defaultColumnHeaderStyle.font.getFontData()[0];
			selectionFontData.setStyle(SWT.NORMAL);
            selectionStyleConfiguration.selectionFont = GUIHelper.getFont(selectionFontData);
            selectionStyleConfiguration.selectedHeaderFont = GUIHelper.getFont(selectionHeaderFontData);
			m_natTable.addConfiguration(selectionStyleConfiguration);

			// Cell style for label configuration
			m_natTable.addConfiguration(new TraceViewerRegistryConfiguration());

            // Remove default behavior for CTRL + f and add a new one (custom
            // find dialog).
            selectionLayer.addConfiguration(new DefaultSelectionBindings() {
                public void configureUiBindings(UiBindingRegistry uiBindingRegistry) {
                    uiBindingRegistry.unregisterKeyBinding(new KeyEventMatcher(SWT.CTRL, 'f'));
                    uiBindingRegistry.registerKeyBinding(new KeyEventMatcher(SWT.CTRL, 'f'),
                                                         new TraceViewerSearchAction());
                }
            });
			
			// Workaround for RedHat missing mouseUp events
			// Expanding and collapsing TreeView will happen on mouseDown events

			if (m_treeLayer != null) {
                m_natTable.addConfiguration(new DefaultTreeLayerConfiguration(m_treeLayer) {
					/*
					 * (non-Javadoc)
					 * @see
					 * org.eclipse.nebula.widgets.nattable.config.IConfiguration
					 * #
					 * configureUiBindings(org.eclipse.nebula.widgets.nattable.
					 * ui .binding.UiBindingRegistry)
					 */
                    public void configureUiBindings(UiBindingRegistry uiBindingRegistry) {
						TreeExpandCollapseAction treeExpandCollapseAction = new TreeExpandCollapseAction();

                        CellPainterMouseEventMatcher treeImagePainterMouseEventMatcher = new CellPainterMouseEventMatcher(GridRegion.BODY,
                                                                                                                          MouseEventMatcher.LEFT_BUTTON,
								TreeImagePainter.class);

                        uiBindingRegistry.registerFirstMouseDownBinding(treeImagePainterMouseEventMatcher,
								treeExpandCollapseAction);
					}
				});
			}

			m_natTable.configure();

			m_natTable.addMouseListener(new MouseAdapter() {

                @Override
                public void mouseDoubleClick(MouseEvent e) {

                    try {
                        SourceBrowserManager sourceBrowser = new SourceBrowserManager();

                        int rowPosition = LayerUtil.convertRowPosition(m_natTable, m_natTable.getRowPositionByY(e.y), m_dataLayer);

                        String[] rowData = m_traceDataProviderCached.readLineFromCache(rowPosition);

                        if (rowData == null) {
                            return;
                        }

                        String instructionAddress = sourceBrowser.getAddress(rowData);
                        if (instructionAddress == null) {
                            return;
                        }

                        if (xmlFile == null) {
                            return;
                        }

                        LSConfigLoader_JNI configLoader = new LSConfigLoader_JNI(xmlFile);

                        ArrayList<SourceCodeCorrelation> elfPaths = sourceBrowser.getBinariesList(configLoader, getCoreNumber(rowPosition));

                        for (SourceCodeCorrelation elfPath : elfPaths) {
                            File parentElfPathDir = new File(elfPath.getNewAppPathName());
                            boolean status = sourceBrowser.checkIfElfExists(new File(elfPath.getNewAppPathName()), new File (parentElfPathDir.getParent()));

                            if (!status) {
                                LOGGER.error("XML file does not exist in :" + parentElfPathDir.getParent()); //$NON-NLS-1$
                                return;
                            }

                            String elfFilePath = elfPath.getNewAppPathName();
                            String loadAddress = elfPath.getLoadAddress();

                            getSourceCorelation(elfFilePath, loadAddress, instructionAddress);
                        }
                    } catch (NumberFormatException ex) {
                        LOGGER.error(e);
                    }
                }
            });

			// collapse all only if tree layer is available
			if (m_treeLayer != null) {
				m_treeLayer.collapseAll();
			}

			// resize the large fields
			for (int i = 0; i < columnNames.length; i++) {
				if (m_traceDataProviderCached.getColumnType(i) == ColumnType.LONG_TEXT) {
                    m_natTable.doCommand(new ColumnResizeCommand(m_dataLayer, i,
                                                                 TEXT_COLUMN_LARGE_WIDTH));
				}
			}
			
			attachToolTip(m_natTable);			
        } else {
			Label lbl = new Label(parent, SWT.HORIZONTAL | SWT.CENTER);
			lbl.setText(Messages.TraceViewer_Error_Decoding);
		}
	}
	
	private void attachToolTip(NatTable natTable) {
        DefaultToolTip toolTip = new TraceViewerNatTableToolTip(natTable);
        toolTip.setBackgroundColor(natTable.getDisplay().getSystemColor(SWT.COLOR_INFO_BACKGROUND));
        toolTip.setPopupDelay(500);
        toolTip.activate();
        toolTip.setShift(new Point(10, 10));
    }

	/**
	 * Get line specific core number.
	 * @param rowPosition  Selected row position.
	 * @return The core number.
	 */
	private int getCoreNumber(int rowPosition) throws NumberFormatException {
	    if (m_dataLayer != null) {
	        if (m_dataLayer.getDataValue(FIRST_COLUMN, new TraceTreeRowModel(m_traceDataProviderCached).getParentIndex(rowPosition)) instanceof String) {
                String coreInfo = m_dataLayer.getDataValue(FIRST_COLUMN, new TraceTreeRowModel(m_traceDataProviderCached).getParentIndex(rowPosition)).toString();
                if (coreInfo.isEmpty()) {
                    return -1;
                }

                if (coreInfo.split(" ").length == 0) { //$NON-NLS-1$
                    return -1;
                }

                // When the core name is splitted generates two components, the second one being the core number
                String coreIdString = coreInfo.split(" ")[1]; //$NON-NLS-1$
                if (coreIdString == null || coreIdString.isEmpty()) {
                    return -1;
                }

                return Integer.parseInt(coreIdString);
	        }
	    }

        return -1;
	}

	/**
	 * Correlates a trace event with his source code line.
	 * @param elfAndLoadAddress    Provides the elf location and loadAddress if there is one.
	 * @param instructionAddress   Provides the trace event instruction address to look for in the elf.
	 */
	private void getSourceCorelation(String elfFilePath, String loadAddress, String instructionAddress) {
	    if (elfFilePath == null || loadAddress == null) {
	        return;
	    }

        LSSourceBrowser_JNI sourceLoader = new LSSourceBrowser_JNI();
        SourceBrowserData sourceData = new SourceBrowserData();
        SourceBrowserManager sourceBrowser = new SourceBrowserManager();

        sourceLoader.sourceLocation(sourceData, elfFilePath, loadAddress, instructionAddress,
                                    sourceBrowser.getArchName());

        SAUIUtils.getSourceBrowser(m_traceFileName, m_baseName).showSource(sourceData.getName(), sourceData.getValue());
	}

    private class TraceViewerNatTableToolTip extends DefaultToolTip {

        private NatTable m_NatTable;

        /*
         * Constructor for a TraceViewerNatTableToolTip object taking the 
         * NatTable which it is associated to as parameter;
         */
        public TraceViewerNatTableToolTip(NatTable natTable) {
            super(natTable, ToolTip.NO_RECREATE, false);
            m_NatTable = natTable;
        }

        /*
         * (non-Javadoc)
         * @see org.eclipse.jface.window.ToolTip#getToolTipArea(org.eclipse.swt.
         * widgets .Event) Implementation here means the tooltip is not redrawn
         * unless mouse hover moves outside of the current cell (the combination
         * of ToolTip.NO_RECREATE style and override of this method).
         */
        @Override
        protected Object getToolTipArea(Event event) {
            int col = m_NatTable.getColumnPositionByX(event.x);
            int row = m_NatTable.getRowPositionByY(event.y);

            if(row < 0 || col < 0) {
            	return null;
            }
            
            if(m_traceDataProviderCached == null) {
            	return null;
            }
            
            int numColumns =  m_traceDataProviderCached.getColumnNames().length;
            
            if (col > numColumns) {
            	return null;
            }
            
            int numRows = m_NatTable.getRowCount();
            
            if (row > numRows) {
            	return null;
            }
            
            return new Point(col, row);
        }

        @Override
        protected String getText(Event event) {
            int col = m_NatTable.getColumnIndexByPosition(m_NatTable.getColumnPositionByX(event.x));
            int rowPosition = m_NatTable.getRowPositionByY(event.y);
            int row = m_NatTable.getRowIndexByPosition(rowPosition);

        	String columnName = m_traceDataProviderCached.getColumnNames()[col];
        	ILayerCell cell = m_NatTable.getCellByPosition(1, rowPosition);
        	String cellText = null;
        	if (cell != null && cell.getDataValue() != null) {
        		cellText = cell.getDataValue().toString();
        	}
        	boolean foundCell = tooltipMap.containsKey(cellText);
        	boolean isColumnHeader = (row == 0);
        	
            if(isColumnHeader || (!isColumnHeader && !foundCell)) {
	            if(tooltipMap.containsKey(columnName)) {
	            	return tooltipMap.get(columnName);
	            }
            }
            if( !isColumnHeader && foundCell) {
	           	return tooltipMap.get(cellText);
            }
            
            return columnName;
        }

        @Override
        protected Composite createToolTipContentArea(Event event, Composite parent) {
            return super.createToolTipContentArea(event, parent);
        }
    }


	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.
	 * IProgressMonitor)
	 */
	@Override
	public void doSave(IProgressMonitor monitor) {
		// nothing to do
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs() {
		// nothing to do
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#init(org.eclipse.ui.IEditorSite,
	 * org.eclipse.ui.IEditorInput)
	 */
	@Override
    public void init(IEditorSite site, IEditorInput input) throws PartInitException {

		setSite(site);
		setInput(input);
		
		setPartName(StorageUtils.getTraceViewerTitle(input.getName()));
		
		if (StorageUtils.hasCustomImage(input.getName())) {
			setTitleImage(StorageUtils.getCustomImage(input.getName()));
		}

		m_canceled = false;
		m_traceFileName = null;
		
		if (input instanceof FileStoreEditorInput) {
			// This happens when the editor is opened by drag-and-drop of the
			// trace file in CW.
			// For this case the trace file will not be associated with any
			// project. The effect
			// is that the content of the editor will not get updated based on
			// target events
			// (e.g., suspend, terminate). This is expected behavior since the
			// trace is not
			// associated with a project.
			FileStoreEditorInput fileStoreEditorInput = (FileStoreEditorInput) input;
            m_traceFileName = fileStoreEditorInput.getURI().getPath().toString();
		} else if (input instanceof FileEditorInput) {
			// This happens when the editor is opened from the CodeWarrior
			// Projects view.
			FileEditorInput fileEditorInput = (FileEditorInput) input;
			m_traceFileName = fileEditorInput.getPath().toString();
			setProject(fileEditorInput);

		} else if (input instanceof IPathEditorInput) {
			// This happens when the editor is opened from the Open Path dialog
			IPathEditorInput pathEditorInput = (IPathEditorInput) input;
			m_traceFileName = pathEditorInput.getPath().toString();
		}

		if (m_traceFileName == null) {
			// This shouldn't happen
			LOGGER.error("[init]: Invalid trace file"); //$NON-NLS-1$
		}

		baseDir = new File(m_traceFileName).getParent();
        if (baseDir == null) {
            return;
        }

        SourceBrowserManager sourceBrowser = new SourceBrowserManager();

        xmlFile = sourceBrowser.searchForXMLFile(new File(baseDir));
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isDirty()
	 */
	@Override
	public boolean isDirty() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		// nothing to do

	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#setInput(org.eclipse.ui.IEditorInput)
	 */
	@Override
	protected void setInput(IEditorInput input) {
		super.setInput(input);
	}
	
	@Override
	public String getHeaderText() {
		return null;
	}

    public IOkraDecoder.DecoderReturn decodeTrace() {
		// call the Okra decoder for the proper architecture
		// if csv image is older than the binary file
		IOkraDecoder.DecoderReturn ret = IOkraDecoder.DecoderReturn.DECODER_OK;
		
		if (m_traceFileName != null) {

			int extIndex = m_traceFileName.lastIndexOf('.');
            m_csvFileName = m_traceFileName.substring(0, extIndex) + CSV_EXTENSION;
			IPath configFilePath = new Path(m_traceFileName);			
			if (configFilePath.isValidPath(m_traceFileName)) {
				m_baseName = configFilePath.lastSegment();
			}
			
			File csvFile = new File(m_csvFileName);
			File binFile = new File(m_traceFileName);

			// Check if csv file exists and it is actual (binary file is older
			// than the decoded
			// csv file. If Yes, the decoding is not needed: binary file is
			// cached in the csv
            if (!csvFile.exists() || csvFile.lastModified() < binFile.lastModified()) {

				// Iterate through decoder extension points
                for (Object decoder : this.m_okraDecoderExtensions) {
                    if (decoder instanceof IOkraDecoder) {
                        ret = ((IOkraDecoder) decoder).decodeTrace(m_traceFileName, CSV_EXTENSION,
                                                                   getSite().getShell());
					if (ret == IOkraDecoder.DecoderReturn.DECODER_CANNOT_DECODE) {
						continue;
					} else if (ret == IOkraDecoder.DecoderReturn.DECODER_CANCELLED) {
						m_canceled = true;
						m_decoded = true;
						break;
					} else if (ret == IOkraDecoder.DecoderReturn.DECODER_OK) {
						m_canceled = false;
						m_decoded = true;
						break;
					}
				}
                }
			} else {
				m_decoded = true;
				m_canceled = false;
			}

		}
		
		SAUIUtils.refreshProject(m_project);
		
		return ret;
	}

    /*
     * public static ArrayList<IOkraDecoder> createOkraDecodersVector() {
     * IExtensionPoint okraDecoder = Platform.getExtensionRegistry()
     * .getExtensionPoint(IOKRA_ID); ArrayList<IOkraDecoder>
     * okraDecoderExtensions = new ArrayList<IOkraDecoder>(); if (okraDecoder !=
     * null) { IConfigurationElement[] config = Platform.getExtensionRegistry()
     * .getConfigurationElementsFor(IOKRA_ID); try { for (IConfigurationElement
     * e : config) { final Object o = e.createExecutableExtension("class");
     * //$NON-NLS-1$ if (o instanceof IOkraDecoder) {
     * okraDecoderExtensions.add(((IOkraDecoder) o)); } } } catch (CoreException
     * ex) { LOGGER.error("Cannot create extension points " + IOKRA_ID, ex);
     * //$NON-NLS-1$ } } return okraDecoderExtensions; }
     */

	@Override
	public void refresh() {
	}

	@Override
	public String getBaseName() {
		return m_baseName;
	}


	@Override
	public String getEditorType() {
		return null;
	}

	@Override
	public Image getHeaderImage() {
		return null;
	}

	@Override
	public String getStoragePath() {
		return new Path(m_csvFileName).removeLastSegments(1).toOSString();
	}

}
