/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ls.traceviewer.editors;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.nebula.widgets.nattable.data.IDataProvider;

/**
 * Class that generates the data to be displayed into the Trace Viewer body
 * 
 * @author R70188
 */
public class TraceContentDataProvider implements IDataProvider {

	private static Logger LOGGER = Logger
			.getLogger(TraceContentDataProvider.class);

	public enum ColumnType {
		NUMERIC, TEXT, LONG_TEXT
	}

	private final static int INDEXING_WIDTH = 50;

	private String m_csvFilePath;

	/**
	 * Constructor
	 * 
	 * @param csvFile
	 *            Path to the csv file to be displayed
	 */
	public TraceContentDataProvider(String csvFilePath) {

		// read the csv file into an internal memory structure
		try {
			m_csvFilePath = csvFilePath;
			m_csvFile = new BufferedRandomAccessFile(new File(csvFilePath), "r"); //$NON-NLS-1$
			m_primaryCacheArray = new HashMap<Integer, String[]>();
			m_secondaryCacheArray = new HashMap<Integer, String[]>();

			// First line represents the names of table columns and the Type
			// (Numeric, Text and Long Text)
			String dataRow = m_csvFile.readLine();
			if (dataRow == null) {
				// empty file
				m_csvFile.close();
				return;
			}
			m_columnHeader = splitCellData(dataRow);
			m_columnDataInfo = new ColumnDataInfo[m_columnHeader.length];
			for (int i = 0; i < m_columnHeader.length; i++) {
				ColumnDataInfo info = new ColumnDataInfo();
				info.name = m_columnHeader[i].replaceAll("\\(.*\\)", ""); //$NON-NLS-1$ //$NON-NLS-2$
				if (m_columnHeader[i].contains("(n)")) { //$NON-NLS-1$
					info.columnType = ColumnType.NUMERIC;
					info.hexDisplay = false;
				} else if (m_columnHeader[i].contains("(nh)")) { //$NON-NLS-1$
					info.columnType = ColumnType.NUMERIC;
					info.hexDisplay = true;
				} else if (m_columnHeader[i].contains("(lt)")) { //$NON-NLS-1$
					info.columnType = ColumnType.LONG_TEXT;
				} else {
					info.columnType = ColumnType.TEXT;
				}

				info.deltaDisplay = false;
				m_columnDataInfo[i] = info;
			}

			m_lineFileOffset = new HashMap<Integer, Long>();
			m_lineFileOffset.put(0, m_csvFile.getFilePointer());

			// Index the file to be able to provide a caching mechanism for its
			// contents
			// This may take a while
			// Do this in main thread for the moment
			dataRow = m_csvFile.readLine();
			int line = 0;
			while (dataRow != null && !dataRow.equals("")) {
				if ((line + 1) % INDEXING_WIDTH == 0) {
					m_lineFileOffset.put(line + 1, m_csvFile.getFilePointer());
				}

				// Also cache the first part of the file
				if (line < INDEXING_WIDTH) {
					String[] dataArray = splitCellData(dataRow);
					m_primaryCacheArray.put(line, dataArray);
				}
				//
				dataRow = m_csvFile.readLine();
				line++;
			}

			m_lineCnt = line;

			m_csvFile.close();

		} catch (FileNotFoundException e) {
			LOGGER.error("Cannot find file " + csvFilePath, e); //$NON-NLS-1$
		} catch (IOException e) {
			LOGGER.error("I/O Exception " + csvFilePath, e); //$NON-NLS-1$
		} finally {
			try{
				if(m_csvFile != null)
					m_csvFile.close();
			} catch (IOException e) {
				LOGGER.error("I/O Exception " + csvFilePath, e); //$NON-NLS-1$
			}
		}
	}

	public String[] readLineFromCache(int rowIndex) {
		String[] cachedValue = m_primaryCacheArray.get(rowIndex);
		if (cachedValue == null) {
			// try secondary buffer - to avoid cache miss cycles
			cachedValue = m_secondaryCacheArray.get(rowIndex);
			if (cachedValue == null) {
				// Cache miss - load the page corresponding to the new required
				// data in Secondary Array
				handleCacheMiss(rowIndex);
				cachedValue = m_primaryCacheArray.get(rowIndex);
			}
		}
		return cachedValue;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.nebula.widgets.nattable.data.IDataProvider#getDataValue(int,
	 * int)
	 */
	@Override
	public Object getDataValue(int columnIndex, int rowIndex) {

		String[] cachedValue = readLineFromCache(rowIndex);
		if (cachedValue == null || columnIndex >= cachedValue.length)
			return null;

		String ret = null;
		String rawValue = cachedValue[columnIndex];
		if (rawValue == null)
			return null;

		if (m_columnDataInfo[columnIndex].deltaDisplay) {
			int previousRowIndex = rowIndex - 1;
			boolean foundPrevious = false;
			String[] previousRow = null;
			while (previousRowIndex >= 0 && foundPrevious == false) {
				previousRow = readLineFromCache(previousRowIndex);
				if (previousRow != null && columnIndex < previousRow.length
						&& !previousRow[columnIndex].equals("")) { //$NON-NLS-1$
					foundPrevious = true;
					break;
				}
				previousRowIndex--;
			}
			if (foundPrevious) {
				try {

					if (!rawValue.contains("0x")) { //$NON-NLS-1$
						BigInteger firstValue = new BigInteger(rawValue);
						BigInteger secondValue = new BigInteger(
								previousRow[columnIndex]);
						rawValue = firstValue.subtract(secondValue).toString();
					} else {
						String rawHex = rawValue.replace("0x", ""); //$NON-NLS-1$ //$NON-NLS-2$
						String previousRowValue = previousRow[columnIndex]
								.replace("0x", ""); //$NON-NLS-1$ //$NON-NLS-2$
						BigInteger firstValue = new BigInteger(rawHex, 16);
						BigInteger secondValue = new BigInteger(
								previousRowValue, 16);

						rawValue = "0x" + firstValue.subtract(secondValue).toString(16); //$NON-NLS-1$ 
					}
				} catch (java.lang.NumberFormatException e) {
					// just leave the raw value as it is
				} catch (java.lang.ArrayIndexOutOfBoundsException e) {
					// just leave the raw value as it is
				}
			}
		}
			
		if (columnIndex < cachedValue.length) {
			if (m_columnDataInfo[columnIndex].columnType == ColumnType.NUMERIC) {
				
				if (!m_columnDataInfo[columnIndex].hexDisplay) {
					if (!rawValue.contains("0x")) { //$NON-NLS-1$
						ret = rawValue;
					}
					else {
						// hex to dec convert
						String hexString = rawValue.replace("0x", ""); //$NON-NLS-1$ //$NON-NLS-2$
						try {
							BigInteger big = new BigInteger(hexString, 16);
							ret = big.toString();
						}
						catch (java.lang.NumberFormatException e){
							// just leave the raw value as it is
							ret = rawValue;
						}
					}
				}
				else {
					if (rawValue.contains("0x")) { //$NON-NLS-1$
						ret = rawValue;
					}
					else {
						// dec to hex convert
						try {
							BigInteger big = new BigInteger(rawValue);
							ret = "0x" + big.toString(16); //$NON-NLS-1$
						}
						catch (java.lang.NumberFormatException e) {
							ret = rawValue;
						}
					}
				}
			}
			else
				ret = rawValue;
		}

		return ret;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.nebula.widgets.nattable.data.IDataProvider#setDataValue(int,
	 * int, java.lang.Object)
	 */
	@Override
	public void setDataValue(int columnIndex, int rowIndex, Object newValue) {
		// nothing to do
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.nebula.widgets.nattable.data.IDataProvider#getColumnCount()
	 */
	@Override
	public int getColumnCount() {
		if (m_columnHeader != null) {
			return m_columnHeader.length;
		} else {
			return 0;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.nebula.widgets.nattable.data.IDataProvider#getRowCount()
	 */
	@Override
	public int getRowCount() {
		return m_lineCnt;
	}

	private void handleCacheMiss(int missRowNumber) {
		// Switch the primary with secondary buffer
		Map<Integer, String[]> tmp = m_secondaryCacheArray;
		m_secondaryCacheArray = m_primaryCacheArray;
		m_primaryCacheArray = tmp;
		m_primaryCacheArray.clear();

		// the new data will be in primary cache and the older data in secondary
		// cache
		int firstLineNumber = (missRowNumber / INDEXING_WIDTH) * INDEXING_WIDTH;
		try {
			m_csvFile = new BufferedRandomAccessFile(new File(m_csvFilePath),
					"r"); //$NON-NLS-1$

			Long offset = m_lineFileOffset.get(firstLineNumber);
			if (offset == null) {
				m_csvFile.close();
				return;
			}
			m_csvFile.seek(offset);
			String dataRow = m_csvFile.readLine();
			int cachedLineCnt = 0;
			while (dataRow != null && cachedLineCnt < INDEXING_WIDTH) {
				// Also cache the first part of the file
				String[] dataArray = splitCellData(dataRow);
				m_primaryCacheArray.put(firstLineNumber + cachedLineCnt,
						dataArray);
				dataRow = m_csvFile.readLine();
				cachedLineCnt++;
			}
			m_csvFile.close();
		} catch (IOException e) {
			LOGGER.error("Error handling line cache miss ", e); //$NON-NLS-1$
		}
	}

	private String[] splitCellData(String line) {

		// Do not split string from within the quotes
		if (line.contains("\"")) { //$NON-NLS-1$
			// replace double separators
			line = line.replace(",\"", "\""); //$NON-NLS-1$
			line = line.replace("\",", "\""); //$NON-NLS-1$
			ArrayList<String> list = new ArrayList<String>();
			String[] sections = line.split("\""); //$NON-NLS-1$

			// even sections are outside the quotes and
			// odd sections are inside the quotes
			for (int i = 0; i < sections.length; i++) {
				if (i % 2 == 0) {
					list.addAll(Arrays.asList(sections[i].split(",", -1))); //$NON-NLS-1$
				} else {
					list.add(sections[i]);
				}
			}
			return list.toArray(new String[0]);
		} else {
			return line.split(","); //$NON-NLS-1$
		}
	}

	/**
	 * Returns the column type
	 * 
	 * @param columnIndex
	 *            index of the column
	 * @return enum describing the column type NUMERIC, TEXT and LONG_TEXT
	 */
	public ColumnType getColumnType(int columnIndex) {
		if (columnIndex < m_columnDataInfo.length) {
			return m_columnDataInfo[columnIndex].columnType;
		} else {
			return null;
		}
	}

	/**
	 * Sets the representation mode for numeric columns. For text columns this
	 * method has no effect
	 * 
	 * @param columnIndex
	 *            index of the column
	 * @param hexMode
	 *            true to display the numeric data in hexadecimal format and
	 *            false for decimal display format
	 */
	public void setColumnHexDisplay(int columnIndex, boolean hexMode) {
		if (columnIndex < m_columnDataInfo.length
				&& m_columnDataInfo[columnIndex].columnType == ColumnType.NUMERIC) {
			m_columnDataInfo[columnIndex].hexDisplay = hexMode;
		}
	}

	/**
	 * Gets the numeric representation of data column, hexadecimal or decimal
	 * 
	 * @param columnIndex
	 *            index of the column
	 * @return true if display the numeric data in hexadecimal format and false
	 *         for decimal display format
	 */
	public boolean getColumnHexDisplay(int columnIndex) {
		if (columnIndex < m_columnDataInfo.length) {
			return m_columnDataInfo[columnIndex].hexDisplay;
		} else {
			return false;
		}

	}

	/**
	 * Sets the representation delta mode for numeric columns. For text columns
	 * this method has no effect
	 * 
	 * @param columnIndex
	 *            index of the column
	 * @param deltaMode
	 *            true to display the difference between the current and
	 *            previous column value
	 */
	public void setColumnDeltaDisplay(int columnIndex, boolean deltaMode) {
		if (columnIndex < m_columnDataInfo.length
				&& m_columnDataInfo[columnIndex].columnType == ColumnType.NUMERIC) {
			m_columnDataInfo[columnIndex].deltaDisplay = deltaMode;
		}
	}

	/**
	 * Gets the numeric representation of data column, hexadecimal or decimal
	 * 
	 * @param columnIndex
	 *            index of the column
	 * @return true if display the difference between the current and previous
	 *         column value
	 */
	public boolean getColumnDeltaDisplay(int columnIndex) {
		if (columnIndex < m_columnDataInfo.length) {
			return m_columnDataInfo[columnIndex].deltaDisplay;
		} else {
			return false;
		}

	}

	/**
	 * Gets the column names without the type information
	 * 
	 * @return
	 */
	public String[] getColumnNames() {
		if (m_columnDataInfo != null) {
			String[] ret = new String[m_columnDataInfo.length];
			for (int i = 0; i < m_columnDataInfo.length; i++) {
				ret[i] = m_columnDataInfo[i].name;
			}
			return ret;
		} else {
			return null;
		}
	}

	/**
	 * Class that stores information about table columns
	 * 
	 * @author R70188
	 * 
	 */
	private static class ColumnDataInfo {	
		public ColumnDataInfo() {
			// constructor
		}

		/**
		 * Column Name
		 */
		public String name;

		/**
		 * Column Type
		 */
		public ColumnType columnType;

		/**
		 * Flag to enable hexadecimal display for numeric data
		 */
		public boolean hexDisplay;

		/**
		 * Flag to enable
		 */
		public boolean deltaDisplay;
	}

	private BufferedRandomAccessFile m_csvFile;
	private Map<Integer, String[]> m_primaryCacheArray;
	private Map<Integer, String[]> m_secondaryCacheArray;
	private String[] m_columnHeader;
	private Map<Integer, Long> m_lineFileOffset;
	private int m_lineCnt;
	private ColumnDataInfo[] m_columnDataInfo;
}
