/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ls.traceviewer.editors;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * <p> Class that provides the same functionality as {@link java.io.RandomAccessFile}
 * but with faster readLine function </p>
 * @author R70188
 *
 */
public class BufferedRandomAccessFile {
	
	private final static int MAX_BUFFERRED_BYTES_CNT = 8192; 
	private final static int DEFAULT_LINE_SIZE = 100; 
	private char[] m_readBuffer;
	private int m_crtPosInBuffer;
	private int m_bufferedBytesCnt;
	private RandomAccessFile m_randomAccessFile;
	private Long m_filePointer;
	private long m_bufferStartFilePointer;
	
    /**
    * <p> See {@link RandomAccessFile#RandomAccessFile(File,String)} </p>
    *
    */
   public BufferedRandomAccessFile(File file, String mode)
           throws FileNotFoundException {
       this.m_randomAccessFile = new RandomAccessFile(file, mode);
       m_filePointer = null;
       m_readBuffer = new char[MAX_BUFFERRED_BYTES_CNT];
   }
	
    /**
     * <p> Has the same functionality as {@link java.io.RandomAccessFile#readLine()}
     * but it's about 100 times faster </p>
     *
     * @throws IOException
     */
    public synchronized String readLine() throws IOException {
    	
    	StringBuilder sb = null;
    	while (true) {
    		boolean haveData = true;
    		// see if we are at the end of buffered part
    		if (m_crtPosInBuffer >= m_bufferedBytesCnt){
    			haveData = cacheNextDataChunk();
    		}
    		
    		// If no data has been read due to EOF return what we have 
    		// until now
    		if (!haveData) { 
    			if (sb != null && sb.length() > 0) {
    				return sb.toString();
    			}
    			else {
    				return null;
    			}
    		}
    		
    		// If we have data read until the next EOL
    		int eolIndex = 0;
    		char c = 0;
    		boolean endOfLine = false;
    		for (eolIndex = m_crtPosInBuffer; eolIndex < m_bufferedBytesCnt; eolIndex ++) {
    			c = m_readBuffer[eolIndex];
    			if (c == '\n' || c == '\r') {
    				endOfLine = true;
    				break;
    			}
    		}
    		
    		int startIndex = m_crtPosInBuffer;
    		m_crtPosInBuffer = eolIndex;
    		
    		if (endOfLine) { // End of line found in this buffer
    			String retString;
    			if (sb == null) { // No continuation
    				retString = new String(m_readBuffer, startIndex, eolIndex - startIndex);
    			} else { // Start of string was on the previous cached buffer
    				sb.append(m_readBuffer, startIndex, eolIndex - startIndex);
    				retString = sb.toString();
    			}
    			
    			// skip the trailing line feed on Win
    			m_crtPosInBuffer ++;
    			if (c == '\r') {
    	    		if (m_crtPosInBuffer >= m_bufferedBytesCnt){
    	    			haveData = cacheNextDataChunk();
    	    		}
    	    		if (haveData && m_readBuffer[m_crtPosInBuffer] == '\n') {
    	    			m_crtPosInBuffer ++;
    	    		}
    			}
    			m_filePointer = m_bufferStartFilePointer + m_crtPosInBuffer;
    			return retString;
    		}
    		else { // end of line not found in the current cache
    			if (sb == null) {
    				sb = new StringBuilder(DEFAULT_LINE_SIZE);
    			}
    			sb.append(m_readBuffer, startIndex, eolIndex - startIndex);
    		}
    		
    	}
    }
    
    
    /**
    * <p>See {@link RandomAccessFile#getFilePointer()}</p>
    *
    * @throws IOException
    */
    public synchronized long getFilePointer() throws IOException {
        if (m_filePointer == null) {
            return m_randomAccessFile.getFilePointer();
        } else {
            return this.m_filePointer;
        }
    }

    /**
    * <p>See {@link RandomAccessFile#seek(long pos)}</p>
    *
    * @throws IOException
    */
    public synchronized void seek(long pos) throws IOException {
    	m_filePointer = null;
        m_randomAccessFile.seek(pos);
        m_crtPosInBuffer = 0;
        m_bufferedBytesCnt = 0;
    }
    
    /**
    * <p>See {@link RandomAccessFile#close()}</p>
    *
    * @throws IOException
    */
    public void close() throws IOException {
			m_randomAccessFile.close();
    }

    
    private boolean cacheNextDataChunk() throws IOException {
    	m_bufferStartFilePointer = m_randomAccessFile.getFilePointer();
        m_filePointer = m_bufferStartFilePointer;
        byte[] buffer = new byte[MAX_BUFFERRED_BYTES_CNT];
        int bytesReadCnt = m_randomAccessFile.read(buffer);
        if (bytesReadCnt > 0) {
        	m_bufferedBytesCnt = bytesReadCnt;
        	m_crtPosInBuffer = 0;
            for (int i = 0; i < bytesReadCnt; i++) {
            	m_readBuffer[i] = (char) buffer[i];
            }
        	return true;
        }
        else { // no more data EOF
        	return false;
        }

    }
        
}
