/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.hierarchicalprofiler.editors;

import java.io.File;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

import com.freescale.sa.model.IAnalyzer;
import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.common.ISection;
import com.freescale.sa.ui.common.UIConstants;
import com.freescale.sa.ui.common.UIMessages;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerColumn;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerConfigureTable;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerFileInput;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerTab;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerTableCell;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SubsectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;
import com.freescale.sa.ui.common.utils.MetricsGroupDescription;
import com.freescale.sa.ui.common.utils.MetricsUtils;
import com.freescale.sa.ui.common.utils.TimeUnitUtils;
import com.freescale.sa.ui.hierarchicalprofiler.chart.CallerCalleePieChartViewer;

public class HierarchicalProfilerTab extends FlatProfilerTab {

	private CCombo mAnds_combo;
	private CCombo mOrs_combo;
	private String previousMOrSComboSel = null;
	private boolean showPerformanceStatistics = false;
	private boolean summaryMetricsColsToShow[];
	private boolean summaryStatisticsColsToShow[];
	private CallerCalleePieChartViewer pieView;
	private int step;

	public static final int callerNameIndex = 0;
	public static final int calleeNameIndex = 1;
	public static final int percentCalleeIndex = 7;
	public static final int percentCallerIndex = 8;

	// Internal identifiers within .flatprofiler file.
	private static final String CHOF_STALLS = "Chof Stalls"; //$NON-NLS-1$
	private static final String INTERLOCK_STALLS = "Interlock Stalls"; //$NON-NLS-1$
	private static final String MEM_STALLS = "Memory Stalls"; //$NON-NLS-1$
	private static final String PROG_STALLS = "Program Stalls"; //$NON-NLS-1$
	private static final String REWIND_STALLS = "Rewind Stalls"; //$NON-NLS-1$

	private static final Vector<String> STATISTICS_NAMES = new Vector<String>() {
		private static final long serialVersionUID = -2220210240919917174L;
		{
			add(UIMessages.Performance_Inclusive);
			add(UIMessages.Performance_Min_Inclusive);
			add(UIMessages.Performance_Max_Inclusive);
			add(UIMessages.Performance_Avg_Inclusive);
			add(UIMessages.Performance_Percent_Inclusive);
			add(UIMessages.Performance_Exclusive);
			add(UIMessages.Performance_Min_Exclusive);
			add(UIMessages.Performance_Max_Exclusive);
			add(UIMessages.Performance_Avg_Exclusive);
			add(UIMessages.Performance_Percent_Exclusive);
		}
	};

	private static final Map<String, String> COLUMN_TOOLTIP_MAP = new HashMap<String, String>() {
		private static final long serialVersionUID = -2220210236958487174L;
		{
			put(UIConstants.Performance_Function_Name, UIMessages.Performance_Tooltip_Function_Name);
			put(UIConstants.Performance_Num_Calls, UIMessages.Performance_Tooltip_Num_Calls);
			put(UIConstants.Performance_Num_Samples, UIMessages.Performance_Tooltip_Num_Samples);
			put(UIConstants.Performance_Inclusive, UIMessages.Performance_Tooltip_Inclusive);
			put(UIConstants.Performance_Min_Inclusive, UIMessages.Performance_Tooltip_Min_Inclusive);
			put(UIConstants.Performance_Max_Inclusive, UIMessages.Performance_Tooltip_Max_Inclusive);
			put(UIConstants.Performance_Avg_Inclusive, UIMessages.Performance_Tooltip_Avg_Inclusive);
			put(UIConstants.Performance_Percent_Inclusive, UIMessages.Performance_Tooltip_Percent_Inclusive);
			put(UIConstants.Performance_Exclusive, UIMessages.Performance_Tooltip_Exclusive);
			put(UIConstants.Performance_Min_Exclusive, UIMessages.Performance_Tooltip_Min_Exclusive);
			put(UIConstants.Performance_Max_Exclusive, UIMessages.Performance_Tooltip_Max_Exclusive);
			put(UIConstants.Performance_Avg_Exclusive, UIMessages.Performance_Tooltip_Avg_Exclusive);
			put(UIConstants.Performance_Percent_Exclusive, UIMessages.Performance_Tooltip_Percent_Exclusive);
			put(UIConstants.Performance_Percent_Total_Calls, UIMessages.Performance_Tooltip_Percent_Total_Calls);
			put(UIConstants.Performance_Percent_Samples, UIMessages.Performance_Tooltip_Percent_Samples);
			put(UIConstants.Performance_Code_Size, UIMessages.Performance_Tooltip_Code_Size);

			put(UIConstants.Performance_Details_Caller, UIMessages.Performance_Details_Caller);
			put(UIConstants.Performance_Details_Callee, UIMessages.Performance_Details_Callee);
			put(UIConstants.Performance_Details_Num_Calls, UIMessages.Performance_Details_Num_Calls);
			put(UIConstants.Performance_Details_Inclusive, UIMessages.Performance_Details_Inclusive);
			put(UIConstants.Performance_Details_Min_Inclusive, UIMessages.Performance_Details_Min_Inclusive);
			put(UIConstants.Performance_Details_Max_Inclusive, UIMessages.Performance_Details_Max_Inclusive);
			put(UIConstants.Performance_Details_Avg_Inclusive, UIMessages.Performance_Details_Avg_Inclusive);
			put(UIConstants.Performance_Details_Percent_Callee, UIMessages.Performance_Details_Percent_Callee);
			put(UIConstants.Performance_Details_Percent_Caller, UIMessages.Performance_Details_Percent_Caller);
			put(UIConstants.Performance_Details_Call_Site, UIMessages.Performance_Details_Call_Site);
			put(UIConstants.Performance_Details_Samples, UIMessages.Performance_Details_Samples);
			put(UIConstants.Performance_Details_Percent_Samples, UIMessages.Performance_Details_Percent_Samples);

			put(CHOF_STALLS, UIMessages.FlatProfilerInput_bubble_flow);
			put(INTERLOCK_STALLS, UIMessages.FlatProfilerInput_bubble_interlocks);
			put(MEM_STALLS, UIMessages.FlatProfilerInput_bubble_hold);
			put(PROG_STALLS, UIMessages.FlatProfilerInput_bubble_starvation);
			put(REWIND_STALLS, UIMessages.FlatProfilerInput_bubble_rewind);
		}
	};

	private static int NUMBER_OF_PERFORMANCE_STATISTICS = 14;
	private static int NUMBER_OF_TRIAD_ELEMENTS = 3;
	private static int NUMBER_OF_PERF_DETAILS_COLUMNS = 10;
	private static int DETAILS_TABLE_TYPE = 222;
	private static int SUMMARY_TABLE_TYPE = 111;
	private static int INCLUSIVE_COLUMN_INDEX = 2;
	private static int PERCENT_EXCLUSIVE_INDEX = 11;
	private static int INCLUISIVE_CALEE_COLUMN_INDEX = 3;
	private static int PERCENT_CALLER_COLUMN_INDEX = 8;

	public HierarchicalProfilerTab(FlatProfilerFileInput input, String section, Composite parent, IAnalyzer analyzer) {
		super(input, section, parent, analyzer);
	}

	@Override
	protected void drawSummary(Composite parent) {
		super.drawSummary(parent);
	}
	
	@Override
	public Composite draw() {
		super.draw();

		// Shortcuts for changing to performance statistic or metrics view.
		if (analyzer != null && analyzer.supports(IAnalyzer.Feature.METRICS_PERFORMANCE)) {
			m_composite.getShell().getDisplay().addFilter(SWT.KeyDown, new Listener() {
				private boolean shiftKeyPressed = false;

				@Override
				public void handleEvent(Event arg0) {
					if (arg0.keyCode == SWT.SHIFT) {
						shiftKeyPressed = true;
					} else {
						Control children[] = m_composite.getChildren();
						if (m_sash != null) {
							children = m_sash.getChildren();
						}
						if (shiftKeyPressed) {
							if (arg0.character == 'S') {
								changeViewToStatistics(children, 0);
							} else {
								if (arg0.character == 'M') {
									changeViewToMetrics(children, 0);
								}
							}

							m_summaryViewer.setInput(m_summaryData);
							reloadTree(m_summaryViewer);
						}
					}
				}
			});
		}

		// Summary table columns
		FlatProfilerColumn[] summaryTableColumns = getSummaryConfiguration();
		summaryStatisticsColsToShow = new boolean[summaryTableColumns.length];
		summaryMetricsColsToShow = new boolean[summaryTableColumns.length];

		for (int i = 0; i < summaryTableColumns.length; i++) {
			TreeColumn column = m_summary.getColumn(i);

			String columnName = summaryTableColumns[i].getName();
			// Width for summary columns.
			summaryColsWidth[i] = summaryTableColumns[i].getWidth();
			
			if (columnName.equals(UIConstants.Performance_Function_Name)
					|| columnName.equals(UIConstants.Performance_Num_Calls)
					|| columnName.equals(UIConstants.Performance_Percent_Total_Calls)
					|| columnName.equals(UIConstants.Performance_Code_Size)
					|| columnName.equals(UIConstants.Performance_Num_Samples)
					|| columnName.equals(UIConstants.Performance_Percent_Samples)) {

				// The columns that are showed when metrics or statistics are
				// chosen in Performance view.
				summaryMetricsColsToShow[i] = true;
				summaryStatisticsColsToShow[i] = true;

				// When Performance view is first shown then metrics for cycles
				// should be in the table.
				column.setWidth(summaryTableColumns[i].getWidth());
			} else {
				if (columnName.equals(UIConstants.Performance_Inclusive)
						|| columnName.equals(UIConstants.Performance_Min_Inclusive)
						|| columnName.equals(UIConstants.Performance_Max_Inclusive)
						|| columnName.equals(UIConstants.Performance_Avg_Inclusive)
						|| columnName.equals(UIConstants.Performance_Percent_Inclusive)
						|| columnName.equals(UIConstants.Performance_Exclusive)
						|| columnName.equals(UIConstants.Performance_Min_Exclusive)
						|| columnName.equals(UIConstants.Performance_Max_Exclusive)
						|| columnName.equals(UIConstants.Performance_Avg_Exclusive)
						|| columnName.equals(UIConstants.Performance_Percent_Exclusive)) {

					// The columns that are showed when metrics are chosen in
					// Performance view.
					summaryMetricsColsToShow[i] = true;
				} else {
					if (analyzer != null && analyzer.supports(IAnalyzer.Feature.METRICS_PERFORMANCE)) {
						int metricIndex = Integer.parseInt(columnName.substring(columnName.length() - 1,
								columnName.length()));
						if (columnName.contains("Variable")) { //$NON-NLS-1$
							// The columns that are showed when statistics are
							// chosen in Performance view.
							summaryStatisticsColsToShow[i] = true;
							if (metricIndex < metricsNames.size()) {
								summaryTableColumns[i].setName(metricsNames.get(metricIndex));
							}
							summaryExcludedColumns.add(summaryTableColumns[i].getName());
						} else {
							// The columns that are showed when statistics are
							// chosen in Performance view.
							summaryStatisticsColsToShow[i] = false;
							// Those columns that aren't visible anytime will be
							// excluded from configure table.
							if (metricIndex < metricsNames.size()) {
								summaryTableColumns[i].setName(metricsNames.get(metricIndex)
										+ Integer.toString(metricIndex));
							}
							summaryExcludedColumns.add(summaryTableColumns[i].getName());
						}
					} else {
						summaryStatisticsColsToShow[i] = false;
						summaryExcludedColumns.add(summaryTableColumns[i].getName());
					}
				}
			}

			// Hide excluded columns in Performance viewer
			if (!summaryMetricsColsToShow[i]) {
				column.setWidth(0);
				column.setResizable(false);
				column.setMoveable(false);
			}
		}

		if (analyzer != null && analyzer.supports(IAnalyzer.Feature.METRICS_PERFORMANCE)) {
			// First view of Performance will be with metrics for first combo
			// box and cycles in second combo box.
			// Summary and details table view should contain data corresponding
			// to those selections.
			showPerformanceStatistics = false;
			previousMAndSComboSel = UIMessages.Performance_ComboBox_Text_Metrics;
			previousMOrSComboSel = UIMessages.Performance_Cycles_Metric_Name;

			for (int i = 0; i < m_summary.getColumnCount(); i++) {
				TreeColumn column = m_summary.getColumn(i);

				if (summaryMetricsColsToShow[i]) {
					column.setWidth(summaryColsWidth[i]);
					column.setResizable(true);
					// Those columns should be visible in configure table.
					if (summaryExcludedColumns.contains(column.getText())) {
						summaryExcludedColumns.remove(column.getText());
					}
				} else {
					column.setWidth(0);
					column.setResizable(false);
					// Those columns shouldn't be visible in configure table.
					if (!summaryExcludedColumns.contains(column.getText())) {
						summaryExcludedColumns.add(column.getText());
					}
				}

				// Set columns that aren't visible anytime with width 0 and not
				// resizable.
				if (i > metricsNames.size() + NUMBER_OF_PERFORMANCE_STATISTICS) {
					column.setWidth(0);
					column.setResizable(false);
				}
			}

			// Update summary viewer in order to show data for cycles.
			int startingIndex = step + NUMBER_OF_PERFORMANCE_STATISTICS;
			updateMetricsSummaryView(startingIndex, 0, step);

			mAnds_combo.setEnabled(true);
			mOrs_combo.setEnabled(true);
		}

		return m_composite;
	}
	
	@Override
	protected boolean showColumn(int i) {
		if (!summaryMetricsColsToShow[i])
			return false;
		return true;
	}

	@Override
	protected ConfigureDetailsAction initConfigureDetailsAction() {
		class LConfigureDetailsAction extends ConfigureDetailsAction {
			public LConfigureDetailsAction(IConfigurableDisplay parentTraceTree, ISection parentSection) {
				super(parentTraceTree, parentSection);
				super.setEnabled(!showPerformanceStatistics);
			}

			@Override
			public void run() {
				List<String> excludedColumns = detailsExcludedColumns;

				m_configuredDetailsTable = new FlatProfilerConfigureTable(m_details, getDetailsConfiguration(),
						m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection(), excludedColumns);
				m_configuredDetailsTable.open();
			}

		}

		return new LConfigureDetailsAction(null, null);
	}

	@Override
	protected ConfigureSummaryAction initConfigureSummaryAction() {
		class LConfigureSummaryAction extends ConfigureSummaryAction {
			public LConfigureSummaryAction(IConfigurableDisplay parentTraceTree, ISection parentSection) {
				super(parentTraceTree, parentSection);
			}

			@Override
			public void run() {
				List<String> excludedColumns = summaryExcludedColumns;

				m_configuredSummaryTable = new FlatProfilerConfigureTable(m_summary, getSummaryConfiguration(),
						m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection(), excludedColumns);
				m_configuredSummaryTable.open();
			}
		}

		return new LConfigureSummaryAction(null, null);
	}

	@Override
	protected void initDetailsViewer(Composite parent) {
		SashForm callPairForm = null;

		callPairForm = new SashForm(parent, SWT.HORIZONTAL);
		callPairForm.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		pieView = new CallerCalleePieChartViewer(callPairForm, this);

		m_details = new Tree(callPairForm, SWT.SINGLE | SWT.FULL_SELECTION | SWT.BORDER);
		m_details.setLayout(new GridLayout());
		m_details.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		callPairForm.setWeights(new int[] { 1, 10 });
	}

	protected FlatProfilerColumn[] getDetailsConfiguration() {
		return getTableConfiguration(new String[] { UIConstants.PERF_CALL_PAIR });
	}

	@Override
	public void createDetailsTable(Composite parent) {
		super.createDetailsTable(parent);

		if (m_details == null) {
			return;
		}
		
		// Details table columns
		FlatProfilerColumn[] detailsTableColumns = getDetailsConfiguration();

		for (int i = 0; i < detailsTableColumns.length; i++) {
			TreeColumn column = m_details.getColumn(i);
			String columnName = detailsTableColumns[i].getName();
			detailsColsWidth[i] = detailsTableColumns[i].getWidth();

			if (columnName.equals(UIConstants.Performance_Details_Caller)
					|| columnName.equals(UIConstants.Performance_Details_Callee)
					|| columnName.equals(UIConstants.Performance_Details_Num_Calls)
					|| columnName.equals(UIConstants.Performance_Details_Inclusive)
					|| columnName.equals(UIConstants.Performance_Details_Min_Inclusive)
					|| columnName.equals(UIConstants.Performance_Details_Max_Inclusive)
					|| columnName.equals(UIConstants.Performance_Details_Avg_Inclusive)
					|| columnName.equals(UIConstants.Performance_Details_Percent_Callee)
					|| columnName.equals(UIConstants.Performance_Details_Percent_Caller)
					|| columnName.equals(UIConstants.Performance_Details_Call_Site)
					|| columnName.equals(UIConstants.Performance_Details_Samples)
					|| columnName.equals(UIConstants.Performance_Details_Percent_Samples)) {

				// Columns that are always viewed in details table.
				detailsColsToShow[i] = true;
			} else {
				// Columns that are not viewed in details table.
				detailsColsToShow[i] = false;
				// Which columns don't appear in configure table for details
				// tree.
				detailsExcludedColumns.add(columnName);
			}

			if (detailsColsToShow[i]) {
				// Show those columns.
				column.setWidth(detailsTableColumns[i].getWidth());
				column.setResizable(true);
			} else {
				// Don't show those columns.
				column.setWidth(0);
				column.setResizable(false);
			}
		}

		m_details.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TreeItem item = (TreeItem) e.item;
				if (!(item.getData() instanceof FlatProfilerTableCell[])) {
					return;
				}

				FlatProfilerTableCell[] data = (FlatProfilerTableCell[]) item.getData();
				if (data == null || data.length == 0) {
					return;
				}

				TableRecordData record = data[0].tableDataReference;
				pieView.selectRecord(record);
			}
		});
		
		privateRegisterDetailsColumnsListeners();
	}
	
	@Override
	protected void registerDetailsColumnsListeners() {
	}
	
	private void privateRegisterDetailsColumnsListeners() {
		super.registerDetailsColumnsListeners();
	}

	/**
	 * Selects an element from sumaryTable indicated by the given position. It
	 * will select the nearest parent (Section. subsection, table) if the
	 * element is not found.
	 * 
	 * @param position
	 *            The element's position
	 */
	private void setSummarySelection(ItemPosition position) {
		for (TreeItem subsectionItem : m_summary.getItems()) {
			if (!ItemPosition.getItemPosition(subsectionItem).getSubsection().equals(position.getSubsection()))
				continue;

			for (TreeItem rowItem : subsectionItem.getItems()) {

				if (!ItemPosition.getItemPosition(rowItem).equals(position))
					continue;

				m_summary.setSelection(rowItem);
				return;
			}

			m_summary.setSelection(subsectionItem);
		}
	}

	/**
	 * Select row from details table indicated by an index
	 * 
	 * @param index
	 *            The row index from details table
	 */
	public void selectCallPair(int index) {
		m_details.select(m_details.getItem(index));
	}

	public void selectFunction(ItemPosition item) {
		if (m_summaryData == null)
			return;

		Vector<Long> indexes = new Vector<Long>();
		indexes.add(item.getTable());

		item.setTable(0L); /* Select item from main table */
		setSummarySelection(item);

		updateDetailsTable(indexes, true);
	}

	@Override
	protected void privateUpdateDetailsTable(Vector<Long> summaryLines, boolean updateHistory) {

		ItemPosition itemPosition = ItemPosition.getItemPosition(m_summary.getSelection()[0]);
		ItemPosition linkPosition = new ItemPosition(itemPosition.getSection(), itemPosition.getSubsection(),
				summaryLines.get(0), null);
		FlatProfilerTableCell[][] cells = null;

		Object o = m_detailsViewer.getInput();
		if (o instanceof FlatProfilerTableCell[][]) {
			cells = (FlatProfilerTableCell[][]) o;
		}

		if (cells == null) {
			return;
		}

		if (pieView != null) {
			if (summaryLines.size() == 1) {
				pieView.loadRecords(fileInput, itemPosition, linkPosition);
			}
		}

		m_detailsData = fileInput.getDetailsTableData(itemPosition.getSection(), itemPosition.getSubsection(),
				summaryLines.get(0));

		if (analyzer != null && analyzer.supports(IAnalyzer.Feature.METRICS_PERFORMANCE)) {
			int selectionIndex = mOrs_combo.getSelectionIndex();
			int startingIndex;

			startingIndex = 3; // getStartingIndex(selectionIndex, DETAILS_TABLE_TYPE);
			if (startingIndex != -1) {
				for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
					for (int columnIdx = 0; columnIdx < m_details.getColumnCount(); columnIdx++) {
						if (columnIdx >= 3 && columnIdx < 9) {
							String cellText = cells[lineIdx][startingIndex + selectionIndex + (columnIdx - 3) * step].text;

							cellText = TimeUnitUtils.convertCycles(cellText, m_timeUnitGroup.getTimeUnitSelection(),
									m_timeUnitGroup.getFreqSelection());

							cells[lineIdx][columnIdx].text = cellText;
						}
					}
				}
			}
		}

		m_detailsViewer.setInput(cells);
	}

	@Override
	protected ToolBar createSummaryToolBar(Composite parent) {
		ToolBar toolBar = super.createSummaryToolBar(parent);

		if (analyzer != null && analyzer.supports(IAnalyzer.Feature.METRICS_PERFORMANCE)) {
			new ToolItem(toolBar, SWT.SEPARATOR);
			drawMetricsAndStatisticsButton(toolBar);
			drawMetricsOrStatisticsButton(toolBar);
		}

		toolBar.pack();

		return toolBar;
	}

	/**
	 * Draws button with selection between metrics or statistics.
	 * 
	 * @param _toolBar
	 *            . Toolbar that contains buttons.
	 */
	private void drawMetricsAndStatisticsButton(ToolBar _toolBar) {
		final ToolBar toolBar = _toolBar;

		// Get the names of metrics and of statistics.
		metricsNames = new Vector<String>();
		metricsNames.add(UIMessages.Performance_Cycles_Metric_Name);

		String metricsDescrPath = fileInput.getMetricsDescrFilePath();
		List<MetricsGroupDescription> metricsList = MetricsUtils.loadMetricsDescription(new Path(metricsDescrPath));
		for (int i = 0; i < metricsList.size(); i++) {
			String[] metricsLabels = metricsList.get(i).getMetricsLabels();
			for (int j = 0; j < metricsLabels.length; j++) {
				metricsNames.add(metricsLabels[j]);
			}
		}

		// Check if we have a correct number of metrics. For example, if we have
		// an imported project
		// then we can choose one or two triads but for profiling clock cycles
		// we can collect only triad A metrics.
		checkNumberOfMetrics();

		ToolItem separator = new ToolItem(toolBar, SWT.SEPARATOR);
		mAnds_combo = new CCombo(toolBar, SWT.READ_ONLY);
		mAnds_combo.setToolTipText(UIMessages.Performance_MetricsView_ToolTip);
		mAnds_combo.add(UIMessages.Performance_ComboBox_Text_Metrics);
		mAnds_combo.add(UIMessages.Performance_ComboBox_Text_Statistics);

		mAnds_combo.select(0);
		mAnds_combo.pack();

		separator.setWidth(mAnds_combo.getSize().x);
		separator.setControl(mAnds_combo);

		previousMAndSComboSel = UIMessages.Performance_ComboBox_Text_Metrics;

		mAnds_combo.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				String selection = mAnds_combo.getItem(mAnds_combo.getSelectionIndex());
				Control children[] = m_composite.getChildren();
				if (m_sash != null) {
					children = m_sash.getChildren();
				}

				// Check if we choose the same metric as the current selection
				// from combo box.
				// In this case we don't have to do nothing.
				if (selection.equals(UIMessages.Performance_ComboBox_Text_Metrics)
						&& !previousMAndSComboSel.equals(UIMessages.Performance_ComboBox_Text_Metrics)) {

					changeViewToMetrics(children, 0);
				}

				// Check if we choose the same metric as the current selection
				// from combo box.
				// In this case we don't have to do nothing.
				if (selection.equals(UIMessages.Performance_ComboBox_Text_Statistics)
						&& !previousMAndSComboSel.equals(UIMessages.Performance_ComboBox_Text_Statistics)) {

					changeViewToStatistics(children, 0);
				}

				m_summaryViewer.setInput(m_summaryData);
				reloadTree(m_summaryViewer);
			}

		});
	}

	/**
	 * Verifies the correct number of metrics by comparing data from .perf and
	 * .metricsdesc file.
	 */
	private void checkNumberOfMetrics() {
		FlatProfilerColumn[] summaryTableColumns = getSummaryConfiguration();

		int maxNumberOfMetrics = 0;

		for (int i = 0; i < summaryTableColumns.length; i++) {
			String columnName = summaryTableColumns[i].getName();
			if (columnName.startsWith(UIMessages.Performance_ColumnName_Variable)) {
				int metricIndex = Integer.parseInt(columnName.substring(columnName.length() - 1, columnName.length()));

				if (metricIndex > maxNumberOfMetrics)
					maxNumberOfMetrics = metricIndex;
			}
		}

		step = maxNumberOfMetrics + 1;

		if (maxNumberOfMetrics < metricsNames.size()) {
			// Remove unnecessary elements from metrics names vector.
			int removeIndex = metricsNames.size() - 1;
			while (removeIndex > maxNumberOfMetrics) {
				metricsNames.remove(removeIndex);
				removeIndex--;
			}
		}
	}

	/**
	 * Draws button with selection between all metrics or all statistics.
	 * 
	 * @param _toolBar
	 *            . Toolbar that contains buttons.
	 */
	private void drawMetricsOrStatisticsButton(ToolBar _toolBar) {
		final ToolBar toolBar = _toolBar;

		ToolItem separator = new ToolItem(toolBar, SWT.SEPARATOR);
		mOrs_combo = new CCombo(toolBar, SWT.READ_ONLY);
		mOrs_combo.setToolTipText(UIMessages.Performance_StatisticsView_ToolTip);
		if (mAnds_combo.getText().equals(UIMessages.Performance_ComboBox_Text_Metrics)) {
			mOrs_combo.removeAll();
			for (int i = 0; i < metricsNames.size(); i++) {
				mOrs_combo.add(metricsNames.get(i));
			}
		} else {
			mOrs_combo.removeAll();
			for (int i = 0; i < STATISTICS_NAMES.size(); i++) {
				mOrs_combo.add(STATISTICS_NAMES.get(i));
			}
		}

		mOrs_combo.select(0);
		mOrs_combo.pack();

		separator.setWidth(mOrs_combo.getSize().x);
		separator.setControl(mOrs_combo);

		previousMOrSComboSel = UIMessages.Performance_Cycles_Metric_Name;

		mOrs_combo.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				String selectionName = mOrs_combo.getText();

				// If the current selection is choosed then nothing is done
				// otherwise updates should be made.
				if (!selectionName.equals(previousMOrSComboSel)) {
					updateMetricsOrStatisticsView(selectionName, step);
				}
			}
		});
	}

	private void changeViewToStatistics(Control childs[], int selectionIndex) {

		int startingIndex;
		showPerformanceStatistics = true;
		previousMAndSComboSel = UIMessages.Performance_ComboBox_Text_Statistics;
		previousMOrSComboSel = UIMessages.Performance_Inclusive;
		initializeSummaryData();

		mAnds_combo.setRedraw(false);
		mOrs_combo.setRedraw(false);
		mAnds_combo.select(1);

		// Update elements from metrics or statistics combo box.
		mOrs_combo.removeAll();
		for (int i = 0; i < STATISTICS_NAMES.size(); i++) {
			mOrs_combo.add(STATISTICS_NAMES.get(i));
		}

		// In this case the details table shouldn't be visible.
		childs[1].setVisible(false);
		// Select the first element from the list to be shown in metrics or
		// statistics combo box text.
		mOrs_combo.select(selectionIndex);

		m_summary.setRedraw(false);

		for (int i = 0; i < metricsNames.size() + NUMBER_OF_PERFORMANCE_STATISTICS; i++) {
			TreeColumn column = m_summary.getColumn(i);

			if (summaryStatisticsColsToShow[i]) {
				column.setWidth(summaryColsWidth[i]);
				column.setResizable(true);
				// Those columns should be visible in configure table.
				if (summaryExcludedColumns.contains(column.getText())) {
					summaryExcludedColumns.remove(column.getText());
				}
			} else {
				column.setWidth(0);
				column.setResizable(false);
				// Those columns shouldn't be visible in configure table.
				if (!summaryExcludedColumns.contains(column.getText())) {
					summaryExcludedColumns.add(column.getText());
				}
			}
		}

		// Update summary viewer in order to show data for inclusive statistic.
		startingIndex = getStartingIndex(selectionIndex, SUMMARY_TABLE_TYPE);
		if (startingIndex != -1) {
			updateStatisticsView(startingIndex, selectionIndex, step);
		}

		m_summary.setRedraw(true);
		mAnds_combo.setRedraw(true);
		mOrs_combo.setRedraw(true);
	}

	private void changeViewToMetrics(Control childs[], int selectionIndex) {

		int startingIndex;
		showPerformanceStatistics = false;
		previousMAndSComboSel = UIMessages.Performance_ComboBox_Text_Metrics;
		previousMOrSComboSel = UIMessages.Performance_Cycles_Metric_Name;
		initializeSummaryData();

		mAnds_combo.setRedraw(false);
		mOrs_combo.setRedraw(false);
		mAnds_combo.select(0);

		// Update elements from metrics or statistics combo box.
		mOrs_combo.removeAll();
		for (int i = 0; i < metricsNames.size(); i++) {
			mOrs_combo.add(metricsNames.get(i));
		}

		// In this case the details table should be visible.
		childs[1].setVisible(true);
		// Select the first element from the list to be shown in metrics or
		// statistics combo box text.
		mOrs_combo.select(selectionIndex);

		m_summary.setRedraw(false);

		for (int i = 0; i < metricsNames.size() + NUMBER_OF_PERFORMANCE_STATISTICS; i++) {
			TreeColumn column = m_summary.getColumn(i);

			if (summaryMetricsColsToShow[i]) {
				column.setWidth(summaryColsWidth[i]);
				column.setResizable(true);
				// Those columns should be visible in configure table.
				if (summaryExcludedColumns.contains(column.getText())) {
					summaryExcludedColumns.remove(column.getText());
				}
			} else {
				column.setWidth(0);
				column.setResizable(false);
				// Those columns shouldn't be visible in configure table.
				if (!summaryExcludedColumns.contains(column.getText())) {
					summaryExcludedColumns.add(column.getText());
				}
			}
		}

		// Update summary viewer in order to show data for cycles.
		startingIndex = getStartingIndex(selectionIndex, SUMMARY_TABLE_TYPE);
		if (startingIndex != -1) {
			updateMetricsSummaryView(startingIndex, selectionIndex, step);
		}

		if (m_detailsData != null) {
			// Remove elements from details table.
			FlatProfilerTableCell[][] cells = null;
			m_detailsViewer.setInput(cells);
		}

		m_summary.setRedraw(true);
		mAnds_combo.setRedraw(true);
		mOrs_combo.setRedraw(true);
	}

	/**
	 * Returns the starting index from where to begin switching table cells.
	 * 
	 * @param selectionIndex
	 *            . Current metric or statistic.
	 * @param tableType
	 *            . If tableType is 111 then starting index is for summary
	 *            table. If it is 222 then it is for details table.
	 * @return starting index.
	 */
	private int getStartingIndex(int selectionIndex, int tableType) {
		int startingIndex = -1;

		if (step > metricsNames.size() && selectionIndex != 0) {
			if (tableType == SUMMARY_TABLE_TYPE)
				startingIndex = step + NUMBER_OF_PERFORMANCE_STATISTICS + NUMBER_OF_TRIAD_ELEMENTS;
			if (tableType == DETAILS_TABLE_TYPE)
				startingIndex = NUMBER_OF_PERF_DETAILS_COLUMNS - 1 + NUMBER_OF_TRIAD_ELEMENTS;
		} else {
			if (tableType == SUMMARY_TABLE_TYPE)
				startingIndex = step + NUMBER_OF_PERFORMANCE_STATISTICS;
			if (tableType == DETAILS_TABLE_TYPE)
				startingIndex = NUMBER_OF_PERF_DETAILS_COLUMNS - 1;
		}

		return startingIndex;
	}

	/**
	 * Switches summary table (metrics view) cells from startingIndex for
	 * current metric or statistic.
	 * 
	 * @param startingIndex
	 *            . Start switching from this index.
	 * @param selectionIndex
	 *            . Current metric or statistic.
	 * @param step
	 *            . Number of metrics in .perf file.
	 */
	private void updateMetricsSummaryView(int startingIndex, int selectionIndex, int step) {

		int cellIndex;

		for (SubsectionData subsection : m_completeSummaryData.values()) {
			for (TableData completeTable : subsection.getTables()) {
				TableData summaryTable = m_summaryData.get(subsection.getName()).get(completeTable.id);

				for (int i = 0; i < completeTable.getRecords().size(); i++) {
					// Elements to update are stored in columns with index from
					// 2 to 14:
					// Inclusive, Min Inclusive ... Percent Exclusive.
					for (int j = INCLUSIVE_COLUMN_INDEX; j <= PERCENT_EXCLUSIVE_INDEX; j++) {
						// Swap the cells values depending on the selection.
						cellIndex = startingIndex + selectionIndex + (j - INCLUSIVE_COLUMN_INDEX) * step;

						summaryTable.getRecord(i).cells.elementAt(j).text = completeTable.getRecord(i).cells
								.elementAt(cellIndex).text;

						summaryTable.getRecord(i).cells.elementAt(j).value = completeTable.getRecord(i).cells
								.elementAt(cellIndex).value;
					}
				}
			}
		}
	}

	/**
	 * Updates metrics or statistic view.
	 * 
	 * @param selectionName
	 *            . "Metrics" or "Statistic".
	 * @param step
	 *            . Number of metrics in .perf file.
	 */
	public void updateMetricsOrStatisticsView(String selectionName, int step) {
		int selectionIndex = mOrs_combo.getSelectionIndex();
		int startingIndex;

		// If current view is for metrics.
		if (mAnds_combo.getText().equals(UIMessages.Performance_ComboBox_Text_Metrics)) {
			if (m_exportDetails != null)
				m_exportDetails.setEnabled(m_details.getItemCount() > 0);
			previousMOrSComboSel = selectionName;

			startingIndex = getStartingIndex(selectionIndex, SUMMARY_TABLE_TYPE);
			if (startingIndex != -1) {
				updateMetricsSummaryView(startingIndex, selectionIndex, step);
			}

			// Update details table only if it is open.
			if (m_detailsData != null) {

				startingIndex = getStartingIndex(selectionIndex, DETAILS_TABLE_TYPE);
				if (startingIndex != -1) {
					updateMetricsDetailsView(startingIndex, selectionIndex, step);
				}
			}
		}

		// If current view is for statistics.
		if (mAnds_combo.getText().equals(UIMessages.Performance_ComboBox_Text_Statistics)) {
			if (m_exportDetails != null)
				m_exportDetails.setEnabled(false);
			previousMOrSComboSel = selectionName;

			startingIndex = getStartingIndex(selectionIndex, SUMMARY_TABLE_TYPE);
			if (startingIndex != -1) {
				updateStatisticsView(startingIndex, selectionIndex, step);
			}
		}

		// Reload fresh data into summary viewer.
		m_summaryViewer.setInput(m_summaryData);
		reloadTree(m_summaryViewer);
	}

	/**
	 * Switches details table cells from startingIndex for current metric or
	 * statistic.
	 * 
	 * @param startingIndex
	 *            . Start switching from this index.
	 * @param selectionIndex
	 *            . Current metric or statistic.
	 * @param step
	 *            . Number of metrics in .perf file.
	 */
	private void updateMetricsDetailsView(int startingIndex, int selectionIndex, int step) {

		int cellIndex;
		String cellText;
		FlatProfilerTableCell[][] cells;

		cells = new FlatProfilerTableCell[getDetailsTableItemsCount()][];
		for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
			cells[lineIdx] = new FlatProfilerTableCell[m_details.getColumnCount()];
			for (int columnIdx = 0; columnIdx < m_details.getColumnCount(); columnIdx++) {
				cells[lineIdx][columnIdx] = getDetailsTableItem(lineIdx, columnIdx);
			}
		}

		for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
			for (int columnIdx = 0; columnIdx < m_details.getColumnCount(); columnIdx++) {
				if (columnIdx >= INCLUISIVE_CALEE_COLUMN_INDEX && columnIdx <= PERCENT_CALLER_COLUMN_INDEX) {

					cellIndex = startingIndex + selectionIndex + (columnIdx - INCLUISIVE_CALEE_COLUMN_INDEX) * step;
					cellText = cells[lineIdx][cellIndex].text;

					cellText = TimeUnitUtils.convertCycles(cellText, m_timeUnitGroup.getTimeUnitSelection(),
							m_timeUnitGroup.getFreqSelection());

					cells[lineIdx][columnIdx].text = cellText;
					cells[lineIdx][columnIdx].value = cells[lineIdx][cellIndex].value;
				}
			}
		}

		// Reload fresh data into details viewer.
		m_detailsViewer.setInput(cells);
	}

	/**
	 * Switches summary table (statistic view) cells from startingIndex for
	 * current metric or statistic.
	 * 
	 * @param startingIndex
	 *            . Start switching from this index.
	 * @param selectionIndex
	 *            . Current metric or statistic.
	 * @param step
	 *            . Number of metrics in .perf file.
	 */
	private void updateStatisticsView(int startingIndex, int selectionIndex, int step) {

		int cellIndex;

		for (SubsectionData subsection : m_completeSummaryData.values()) {
			for (TableData completeTable : subsection.getTables()) {
				TableData summaryTable = m_summaryData.get(subsection.getName()).get(completeTable.id);

				for (int i = 0; i < completeTable.getRecords().size(); i++) {
					// Elements to update are stored in columns with index from
					// 14 to
					// 14+(number of metrics): Cycles, Metric 1, Metric2 ...
					// Metric n.
					for (int j = NUMBER_OF_PERFORMANCE_STATISTICS; j < NUMBER_OF_PERFORMANCE_STATISTICS + step; j++) {
						// Swap the cells values depending on the selection.
						cellIndex = startingIndex + (j - NUMBER_OF_PERFORMANCE_STATISTICS) + selectionIndex * step;

						summaryTable.getRecord(i).cells.elementAt(j).text = completeTable.getRecord(i).cells
								.elementAt(cellIndex).text;

						summaryTable.getRecord(i).cells.elementAt(j).value = completeTable.getRecord(i).cells
								.elementAt(cellIndex).value;
					}
				}
			}
		}
	}

	@Override
	public void refresh() {
		super.refresh();

		if (analyzer != null && analyzer.supports(IAnalyzer.Feature.METRICS_PERFORMANCE)) {
			// If call is made from Performance view then update should keep
			// track of metrics or/and statistics combo selection.
			String selectionName = mOrs_combo.getText();
			updateMetricsOrStatisticsView(selectionName, step);
		}
	}

	/**
	 * Tells if the given table column measures time for the tree sent as
	 * argument.
	 * 
	 * @param columnNum
	 *            the column index
	 * @return true if the column measures time, false otherwise
	 */
	protected boolean isTimeColumn(String columnName) {
		if (columnName.startsWith(UIMessages.Performance_ColumnName_Variable)
				|| columnName.equals(UIConstants.Performance_Inclusive)
				|| columnName.equals(UIConstants.Performance_Min_Inclusive)
				|| columnName.equals(UIConstants.Performance_Max_Inclusive)
				|| columnName.equals(UIConstants.Performance_Avg_Inclusive)
				|| columnName.equals(UIConstants.Performance_Details_Inclusive)
				|| columnName.equals(UIConstants.Performance_Details_Min_Inclusive)
				|| columnName.equals(UIConstants.Performance_Details_Max_Inclusive)
				|| columnName.equals(UIConstants.Performance_Details_Avg_Inclusive)
				|| columnName.equals(UIConstants.Performance_Exclusive)
				|| columnName.equals(UIConstants.Performance_Min_Exclusive)
				|| columnName.equals(UIConstants.Performance_Max_Exclusive)
				|| columnName.equals(UIConstants.Performance_Avg_Exclusive)) {
			return true;
		}

		return false;
	}

	protected FlatProfilerTableCell getTableItem(Tree tree, TableData tableData, int lineNum, int columnNum) {
		FlatProfilerTableCell ret = new FlatProfilerTableCell();

		TableRecordData recordData = tableData.getRecord(lineNum);
		ret.background = recordData.background;
		ret.tableDataReference = recordData;

		if (recordData.cells.elementAt(columnNum).readSourceFile == null) {
			ret.text = recordData.cells.elementAt(columnNum).value;
		} else {
			ret.text = recordData.cells.elementAt(columnNum).readSourceFile;
		}
		ret.lineType = recordData.recordConfigurationName;
		String lineName = recordData.recordConfigurationName;
		TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);
		TableCellConfiguration cell = config.cells.get(columnNum);
		ret.columnType = cell.name;

		if (cell.type.equals("string")) { //$NON-NLS-1$
			ret.text = ret.text.replaceAll("\t", "    "); //$NON-NLS-1$  //$NON-NLS-2$
			ret.value = ret.text;
			
			if (tableData.getRecord(lineNum).cells.elementAt(columnNum).linkToAnotherTable >= 0) {
				ret.text = tableData.getRecord(lineNum).cells.elementAt(columnNum).value;
			}
		} else if (cell.type.equals("integer")) { //$NON-NLS-1$
			ret.value = getLong(ret.text);
			ret.graphicsValue = (Long) ret.value;
			ret.hasGraphics = true;
		} else if (cell.type.equals("float") || cell.type.equals("long")) { //$NON-NLS-1$ //$NON-NLS-2$
			ret.value = getDouble(ret.text);
			ret.graphicsValue = (Double) ret.value;
			ret.hasGraphics = true;
			String text;
			if ((Double) ret.value > 99.99 && (Double) ret.value < 100.0) {
				text = "99.99"; //$NON-NLS-1$
			} else {
				text = m_doubleFormatter.format((Double) ret.value);
			}

			ret.text = text;

			if (isTimeColumn(cell.name)) {
				ret.text = TimeUnitUtils.convertCycles(ret.text, m_timeUnitGroup.getTimeUnitSelection(),
						m_timeUnitGroup.getFreqSelection());
			}
		} else if (cell.type.equals("bigint")) { //$NON-NLS-1$
			Integer base = 10;
			String text = ret.text;
			text = text.toLowerCase();
			if (text.startsWith("0x")) {
				text = text.substring(2);
				base = 16;
			}
			final BigInteger bigIntRet = new BigInteger(text, base);
			ret.value = new Comparable<BigInteger>() {
				@Override
				public int compareTo(BigInteger arg0) {
					return bigIntRet.compareTo(arg0);
				}
				
				@Override
				public boolean equals(Object arg0) {
					if (arg0 == null) {
						return false;
					}
					
					if (!(arg0 instanceof Comparable)) {
						return false;
					}

					return super.equals(arg0);
				}
				
				// Fixes an FindBugs issue
				@Override
				public int hashCode() {
					return super.hashCode();
				}
			};
		}

		StringBuffer ident = new StringBuffer();
		for (int i = 0; i < cell.ident; i++) {
			ident.append(" "); //$NON-NLS-1$
		}
		ret.text = ident + ret.text;

		if (config.courier && !config.boldFont) {
			ret.font = m_courierFont;
		} else if (config.courier && config.boldFont) {
			ret.font = m_courierBoldFont;
		} else if (config.boldFont) {
			ret.font = m_normalBoldFont;
		} else {
			ret.font = m_normalFont;
		}

		return ret;
	}

	protected void setSummaryTableItem(TreeItem item) {
		if (!(item.getData() instanceof TableRecordData)) {
			return;
		}

		TableRecordData recordData = (TableRecordData) item.getData();

		if ((recordData == null) || (!recordData.isVisible())) {
			return;
		}

		String lineName = recordData.recordConfigurationName;
		TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);

		// Parse through recordData/cells[] and config/cells[] to update data
		for (int columnNum = 0; columnNum < recordData.cells.size(); columnNum++) {
			TableCellData tableCell = recordData.cells.elementAt(columnNum);
			TableCellConfiguration cellConfig = config.cells.elementAt(columnNum);

			if (cellConfig.type.equals("float") && !tableCell.value.isEmpty()) { //$NON-NLS-1$
				if (isTimeColumn(cellConfig.name)) {
					if (!m_timestampAvailable) {
						tableCell.text = UIMessages.FlatProfilerInput_N_A;
					} else {
						tableCell.text = TimeUnitUtils.convertCycles(tableCell.value,
								m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection());
					}
					continue;
				}

				if (tableCell.value.startsWith(UIMessages.FlatProfilerInput_N_A)) { //$NON-NLS-1$
					tableCell.value = UIMessages.FlatProfilerInput_N_A; //$NON-NLS-1$
					continue;
				}
				
				// Remove % if necessary
				if (tableCell.value.endsWith("%")) { //$NON-NLS-1$
					tableCell.value = tableCell.value.replace("%", "").trim(); //$NON-NLS-1$ //$NON-NLS-2$
				}

				if ((getDouble(tableCell.value) > 99.99) && (getDouble(tableCell.value) < 100.0)) {
					tableCell.value = "99.99"; //$NON-NLS-1$
				} else {
					tableCell.value = m_doubleFormatter.format(getDouble(tableCell.value));
				}
			} else if (cellConfig.type.equals("long") && !tableCell.value.isEmpty()) { //$NON-NLS-1$
				if (isTimeColumn(cellConfig.name)) {
					if (!m_timestampAvailable) {
						tableCell.text = UIMessages.FlatProfilerInput_N_A;
						tableCell.value = UIMessages.FlatProfilerInput_N_A;
					} else {
						tableCell.value = TimeUnitUtils.convertCycles(tableCell.text,
								m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection());
					}
				}
			}

			if (lineName.equals(FILE)
					&& (cellConfig.name.equals(UIMessages.Metrics_CofLatency)
							|| cellConfig.name.equals(UIMessages.Metrics_InterlockStall)
							|| cellConfig.name.equals(UIMessages.Metrics_MemStall)
							|| cellConfig.name.equals(UIMessages.Metrics_DispatchStarvation) || cellConfig.name
								.equals(UIMessages.Metrics_RewindLatency))) {

				tableCell.text = UIMessages.FlatProfilerInput_N_A;
			}
		}
	}

	@Override
	protected void loadPropFile(File file) {

		// Summary table columns
		FlatProfilerColumn[] summaryTableColumns = getSummaryConfiguration();
		int columnsCount = m_summary.getColumnCount();

		if (file.exists()) {
			int columnsOrder[] = null;
			boolean showColumns[] = null;
			boolean loadedProperties = false;

			columnsOrder = new int[columnsCount];
			showColumns = new boolean[columnsCount];
			loadedProperties = loadPropertiesTable(file.getAbsolutePath(), columnsOrder, showColumns);

			if (loadedProperties) {
				m_summary.setColumnOrder(columnsOrder);
				for (int i = 0; i < columnsCount; i++) {
					if (showColumns[i] == false) {
						m_summary.getColumn(i).setWidth(0);
					} else {
						m_summary.getColumn(i).setWidth(summaryTableColumns[i].getWidth());
					}
				}
			}
		}
	}

	@Override
	protected String getDetailsPropertiesFile() {
		IPath fileLocation = new Path(getFileName()).removeLastSegments(1);
		return (fileLocation.append(UIConstants.DETAILES_PERFORMANCE_PROPERTIES_NAME)).toString();
	}

	protected String getSummaryPropertiesFile() {
		IPath fileLocation = new Path(getFileName()).removeLastSegments(1);
		return (fileLocation.append(UIConstants.SUMMARY_PERFORMANCE_PROPERTIES_NAME)).toString();

	}

	@Override
	protected Map<String, String> getToolTipMap() {
		return COLUMN_TOOLTIP_MAP;
	}

	@Override
	public String getDisplayId() {
		return "Hierarchical Tab";
	}

	@Override
	protected void paintItem(Tree tree, Event event) {
		// Nothing to do
	}

	@Override
	protected String getInstructionAddress(TableRecordData recordData) {
	    return recordData.cells.lastElement().text;
	}
}
