/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.hierarchicalprofiler.chart;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.progress.UIJob;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.entity.PieSectionEntity;
import org.jfree.chart.labels.PieToolTipGenerator;
import org.jfree.chart.plot.PiePlot;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.general.PieDataset;
import org.jfree.experimental.chart.swt.ChartComposite;

public class PieChart extends ChartComposite {
	private static final Logger LOGGER = Logger.getLogger(PieChart.class);
	private PiePlot pieplot;
	private DefaultPieDataset dataset;
	private HashMap<SliceData, PieSlice> slices;
	private List<UIJob> jobs;
	private List<ProducerThread> producers;

	private static Double MAX_EXPLODE_PERCENT = 0.2D;

	/**
	 * This class will plan a complete pie animation by submitting UIJobs which
	 * correspond to one animation step
	 * 
	 * @author B46903
	 * 
	 */
	public static class ProducerThread implements Runnable {
		private PiePlot plot;
		private Integer endAngle;
		private HashMap<SliceData, PieSlice> slices;
		private List<UIJob> jobs;
		private boolean running;

		private static final Logger LOGGER = Logger.getLogger(ProducerThread.class);

		public ProducerThread(HashMap<SliceData, PieSlice> slices, PiePlot plot, Double endAngle, List<UIJob> jobs) {
			this.endAngle = endAngle.intValue();
			this.plot = plot;
			this.slices = slices;
			this.jobs = jobs;
			this.running = true;
		}

		@Override
		public void run() {
			Double explodePercent;
			Integer sign = endAngle > 0 ? 1 : (-1);
			endAngle = Math.abs(endAngle);

			for (Integer i = 1; i <= endAngle; i++) {
				if (!isRunning()) {
					break;
				}

				explodePercent = ((double)i) / endAngle * MAX_EXPLODE_PERCENT;

				if (i == endAngle.intValue())
					explodePercent = MAX_EXPLODE_PERCENT;

				ConsumerThread consumer = new ConsumerThread(slices, explodePercent, plot, 1 * sign, this);
				synchronized (jobs) {
					jobs.add(consumer);
				}

				consumer.schedule();
				try {
					synchronized (plot) {
						if (consumer != null) {
							plot.wait();
						}
					}
					if (isRunning()) {
						Thread.sleep(5);
					}
				} catch (InterruptedException e) {
					LOGGER.debug(e.getLocalizedMessage());
				}

				synchronized (jobs) {
					jobs.remove(consumer);
				}
			}

			if (!isRunning()) {
				synchronized (jobs) {
					for (UIJob job : jobs) {
						job.cancel();
					}

					jobs.clear();
				}
			}

			stopRunning();
		}

		public synchronized void stopRunning() {
			if (this.isRunning()) {
				synchronized (plot) {
					plot.notify();
				}
				running = false;
			}
		}

		private synchronized boolean isRunning() {
			return running;
		}
	}

	/**
	 * This class describes one animation step of the pie
	 * 
	 * @author B46903
	 * 
	 */
	public static class ConsumerThread extends UIJob {
		private PiePlot plot;
		private Integer angle;
		private Double explodePercent;
		private HashMap<SliceData, PieSlice> slices;
		private boolean running;

		public ConsumerThread(HashMap<SliceData, PieSlice> slices, Double explodePercent, PiePlot plot,
				Integer angleDiff, ProducerThread producerThread) {
			super("ConsumerThread"); //$NON-NLS-1$
			this.plot = plot;
			this.angle = angleDiff;
			this.explodePercent = explodePercent;
			this.slices = slices;
			running = true;
		}
		
		public synchronized void stopRunning() {
			super.canceling();
			if (this.isRunning()) {
				synchronized (plot) {
					plot.notify();
				}
				running = false;
			}
		}

		private synchronized boolean isRunning() {
			return running;
		}

		@Override
		public IStatus runInUIThread(IProgressMonitor monitor) {
			Double plotAngle = angle + plot.getStartAngle();
			plotAngle %= 360;

			try {
				for (Object key : plot.getDataset().getKeys()) {
					if (!isRunning()) {
						break;
					}

					PieSlice pslice = slices.get(key);

					if (pslice.isImploding()) {
						plot.setExplodePercent((SliceData) key, MAX_EXPLODE_PERCENT - explodePercent);
						continue;
					}

					if (pslice.isExploding()) {
						plot.setExplodePercent((SliceData) key, explodePercent);
						continue;
					}
				}

				if (explodePercent.intValue() == MAX_EXPLODE_PERCENT) {

					for (SliceData slice : slices.keySet()) {
						PieSlice pslice = slices.get(slice);

						if (pslice.isImploding()) {
							pslice.setImplode(false);
							pslice.setExplode(false);
							plot.setExplodePercent(slice, 0D);
						}
					}
				}

				plot.setStartAngle(plotAngle);

				synchronized (plot) {
					plot.notify();
				}

				return Status.OK_STATUS;
			} catch (Exception e) {
				LOGGER.debug(e.getLocalizedMessage());
				return Status.OK_STATUS;
			}
		}
	}

	/**
	 * Checks if there are running animation
	 * 
	 * @return true if at least one animation is running, false otherwise
	 */
	protected boolean activeAnimation() {
		synchronized (producers) {
			for (ProducerThread producer : producers) {
				if (producer.isRunning()) {
					return true;
				}
			}

			producers.clear();
		}

		synchronized (jobs) {
			if (jobs.isEmpty()) {
				for (PieSlice slice : slices.values()) {
					slice.setImplode(false);
				}
			}
		}

		return !jobs.isEmpty();
	}

	/**
	 * Selects a slice from pie an trigger an animation.
	 * 
	 * @param slice
	 *            The slice to be selected.
	 */
	public void selectSlice(SliceData slice) {

		for (int i = 0; i < pieplot.getDataset().getItemCount(); i++) {
			if (slice.equals(pieplot.getDataset().getKey(i))) {
				selectSlice(slice, i);
				return;
			}
		}
	}

	/**
	 * Selects a slice from pie an trigger an animation.
	 * 
	 * @param slice
	 *            The slice to be selected.
	 * @param sliceIndex
	 *            Slice's index in pie data set
	 */
	private void selectSlice(SliceData slice, Integer sliceIndex) {
		/* Interrupt all active animations */
		interruptAnimation();

		for (PieSlice ps : slices.values()) {
			if (ps.isExploding()) {
				ps.setExplode(false);
				ps.setImplode(true);
			}
		}

		slices.get(slice).setImplode(false);
		slices.get(slice).setExplode(true);
		planAnimation(slices.get(slice), slice, sliceIndex);
	}

	/**
	 * Interrupts current animation
	 */
	private void interruptAnimation() {
		// Cancel all planned jobs
		if (producers != null) {
			synchronized (producers) {
				for (ProducerThread producer : producers) {
					producer.stopRunning();
				}
				producers.clear();
			}
			
			/* Cancel all UI jobs */
			synchronized (jobs) {
				for (UIJob job : jobs) {
					if (job instanceof ConsumerThread) {
						((ConsumerThread)job).stopRunning();
					}
					job.cancel();
				}

				jobs.clear();
			}
		}
	}

	public PieChart(Composite parent, int style) {
		super(parent, style, null, 60,// DEFAULT_WIDTH
				60, // DEFAULT_HEIGHT
				60, // DEFAULT_MINIMUM_DRAW_WIDTH
				60, // DEFAULT_MINIMUM_DRAW_HEIGHT
				600,// DEFAULT_MAXIMUM_DRAW_WIDTH
				600,// DEFAULT_MAXIMUM_DRAW_HEIGHT
				false, false, // Properties
				false, // Save
				false, // Print
				false, // Zoom
				true // Tooltips
		);

		reset();
		PieDataset piedataset = new DefaultPieDataset();
		initPie(piedataset);
		setVisible(false);

		addChartMouseListener(new ChartMouseListener() {

			@Override
			public void chartMouseMoved(ChartMouseEvent event) {
				// Nothing to do
			}

			@Override
			public void chartMouseClicked(ChartMouseEvent event) {
				if (!(event.getEntity() instanceof PieSectionEntity)) {
					return;
				}
				PieSectionEntity section = (PieSectionEntity) event.getEntity();
				SliceData slice = (SliceData) section.getSectionKey();

				selectSlice(slice, section.getSectionIndex());
			}
		});
	}

	/**
	 * Plan pie chart animation for the given slice. Slice will be rotated until
	 * the middle of it hits 0 degrees.
	 * 
	 * @param pieSlice
	 * @param slice
	 */
	private void planAnimation(PieSlice pieSlice, SliceData slice, Integer sliceIndex) {
		Double middle;
		Double startAngle = pieplot.getStartAngle();
		ProducerThread producer;

		for (int i = 0; i <= sliceIndex; i++) {
			startAngle -= ((SliceData) dataset.getKey(i)).getValue().doubleValue() * 360 / 100;
		}

		startAngle %= 360;
		if (startAngle < 0) {
			startAngle = 360 + startAngle;
		}
		pieSlice.setStart(startAngle);
		middle = pieSlice.getMiddle();

		// Rotate anticlockwise
		if (middle > 360 - middle) {
			producer = new ProducerThread(slices, pieplot, (360 - middle), jobs);
		} else {
			// Clockwise rotation
			producer = new ProducerThread(slices, pieplot, -middle, jobs);
		}

		synchronized (producers) {
			producers.add(producer);
		}
		new Thread(producer).start();
	}

	private void initPie(PieDataset piedataset) {
		JFreeChart jfreechart = ChartFactory.createPieChart("", piedataset, false, true, false); //$NON-NLS-1$
		setChart(jfreechart);
		pieplot = (PiePlot) jfreechart.getPlot();

		pieplot.setBackgroundPaint(null);
		pieplot.setCircular(true);
		pieplot.setInteriorGap(0.04);
		pieplot.setOutlineVisible(false);
		pieplot.setBaseSectionOutlinePaint(Color.WHITE);
		pieplot.setSectionOutlinesVisible(true);

		// Customize the section label appearance
		pieplot.setLabelGenerator(null);
		pieplot.setNoDataMessage("No data available"); //$NON-NLS-1$
		pieplot.setToolTipGenerator(new PieToolTipGenerator() {

			@Override
			@SuppressWarnings("rawtypes") //$NON-NLS-1$
			public String generateToolTip(PieDataset dataset, Comparable key) {
				SliceData data = (SliceData) key;
				return data.getName() + " (" + data.getValue() + "%)"; //$NON-NLS-1$  //$NON-NLS-2$
			}
		});
	}

	@Override
	public void dispose() {
		reset();
	}

	private static Integer colorId = 0;
	private static Color[] colors = new Color[] { new Color(249, 188, 2), new Color(251, 153, 2),
			new Color(253, 83, 8), new Color(254, 39, 18), // Red
			new Color(167, 25, 75), new Color(134, 1, 176), // Magenta
			new Color(62, 1, 164), new Color(2, 71, 254), // Blue
			new Color(3, 146, 206), new Color(102, 176, 51), // Green
			new Color(209, 234, 44), new Color(255, 254, 52) // Yellow
	};

	synchronized private void resetPalette() {
		colorId = 0;
	}

	synchronized private Color getColor() {
		Color color = colors[colorId];
		colorId++;
		colorId %= colors.length;
		return color;
	}

	/**
	 * Restores all pie components to default values
	 */
	protected void reset() {
		interruptAnimation();

		resetPalette();

		slices = new HashMap<SliceData, PieSlice>();
		jobs = Collections.synchronizedList(new ArrayList<UIJob>());
		producers = Collections.synchronizedList(new ArrayList<ProducerThread>());

		if (pieplot != null) {
			synchronized (pieplot) {
				pieplot.notifyAll();
			}
			pieplot.setStartAngle(PiePlot.DEFAULT_START_ANGLE);
		}

		dataset = new DefaultPieDataset();
		dataset.clear();
	}

	/**
	 * Set pie data
	 * 
	 * @param data
	 *            A tree map where the Key is an ratio or a value and Value is
	 *            the section name
	 */
	public void setData(ArrayList<SliceData> data) {
		if (data.isEmpty()) {
			setVisible(false);
			return;
		} else {
			setVisible(true);
		}

		reset();

		Collections.sort(data, new Comparator<SliceData>() {
			@Override
			public int compare(SliceData arg0, SliceData arg1) {
				return -1 * Double.compare(arg0.getValue().doubleValue(), arg1.getValue().doubleValue());
			}
		});

		/* Set the data set and assign a color for each section */
		for (SliceData slice : data) {
			dataset.setValue(slice, slice.getValue());
			pieplot.setSectionPaint(slice, getColor());
			pieplot.setExplodePercent(slice, 0D);
			slices.put(slice, new PieSlice(slice, 360D * slice.getValue().doubleValue() / 100D));
		}

		pieplot.setDataset(dataset);
	}

	/**
	 * Returns the pie's slices
	 */
	@SuppressWarnings("unchecked") //$NON-NLS-1$
	public Collection<SliceData> getData() {
		return (List<SliceData>) pieplot.getDataset().getKeys();
	}
}
