/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.hierarchicalprofiler.chart;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.entity.PieSectionEntity;

import com.freescale.sa.ui.hierarchicalprofiler.editors.HierarchicalProfilerTab;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerFileInput;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerTab;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerTab.ItemPosition;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SubsectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;
import com.freescale.sa.ui.common.flatdata.UIResourceBundle;

public class CallerCalleePieChartViewer extends Composite {

	private static final Logger LOGGER = Logger.getLogger(CallerCalleePieChartViewer.class);

	private HierarchicalProfilerTab profilerInput;
	private CallPairPieChart calleePieChart;
	private CallPairPieChart callerPieChart;

	/**
	 * Mapping between details rows and slices of the pies
	 */
	private HashMap<TableRecordData, SliceData> slices;

	public CallerCalleePieChartViewer(Composite parent, HierarchicalProfilerTab profilerInput) {
		super(parent, SWT.NONE);

		this.profilerInput = profilerInput;
		try {
			initGUI();
		} catch (Exception e) {
			LOGGER.debug(e.getLocalizedMessage());
		}

		slices = new HashMap<TableRecordData, SliceData>();
	}

	/**
	 * Selects pie slice which correspond to the given record.
	 * 
	 * @param record
	 *            The selected record
	 */
	public void selectRecord(TableRecordData record) {
		SliceData slice = null;

		Vector<TableCellData> recordCells = record.cells;

		// Match record by cells value
		for (TableRecordData rec : slices.keySet()) {
			Vector<TableCellData> recCells = rec.cells;
			if (recCells.size() != recordCells.size()) {
				continue;
			}

			boolean found = true;
			for (int i = 0; i < recordCells.size(); i++) {
				if (!recordCells.get(i).value.equals(recCells.get(i).value)) {
					found = false;
					break;
				}
			}

			if (found) {
				slice = slices.get(rec);
				break;
			}
		}

		if (slice == null) {
			return;
		}

		// Select slice from callee pie chart
		for (SliceData s : calleePieChart.getData()) {
			if (s.equals(slice)) {
				calleePieChart.selectSlice(s);
				return;
			}
		}

		// Select slice from caller pie chart
		for (SliceData s : callerPieChart.getData()) {
			if (s.equals(slice)) {
				callerPieChart.selectSlice(s);
				return;
			}
		}
	}

	protected void initGUI() {
		setLayout(new GridLayout(1, false));

		Label callerLabel = new Label(this, SWT.NONE);
		callerLabel.setText(UIResourceBundle.PieChart_Caller);
		callerLabel.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, true, false, 1, 1));

		callerPieChart = new CallPairPieChart(profilerInput, this, SWT.NONE);
		callerPieChart.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		Label calleeLabel = new Label(this, SWT.NONE);
		calleeLabel.setText(UIResourceBundle.PieChart_Callee);
		calleeLabel.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, true, false, 1, 1));

		calleePieChart = new CallPairPieChart(profilerInput, this, SWT.NONE);
		calleePieChart.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		pack();
	}

	public void loadRecords(FlatProfilerFileInput fileInput, ItemPosition itemPosition, ItemPosition linkPosition) {
		ArrayList<SliceData> calleeData = new ArrayList<SliceData>();
		ArrayList<SliceData> callerData = new ArrayList<SliceData>();
		SliceData sliceData;
		slices.clear();

		SectionData detailsSection = fileInput.getSection(itemPosition.getSection());

		SubsectionData subsectionData = detailsSection.get(itemPosition.getSubsection());
		if (subsectionData == null)
			return;

		TableData selectedTable = subsectionData.get(itemPosition.getTable());
		if (selectedTable == null)
			return;

		TableData detailsData = subsectionData.get(linkPosition.getTable());
		TableData summaryTable = ((SubsectionData) detailsData.getParent().getParent()).get(0L);

		// Extract the function name. Value used in tool tips labels.
		String functionName = selectedTable.getRecordByHash(itemPosition.getRowIndex()).cells
				.get(FlatProfilerTab.functionNameIndex).text;

		for (int i = 0; i < detailsData.getRecords().size(); i++) {
			// Populate slices with links to the corresponding tables
			String function = detailsData.getRecord(i).cells.get(FlatProfilerTab.functionNameIndex).value;

			// Get table id for function from summary table
			Integer recordHash = detailsData.getRecord(i).hashCode();
			
			ItemPosition position;

			/* <no debug info> function */
			if (summaryTable.getRecordByHash(recordHash) == null) {
				position = null;
			} else {
				Long linkTableId = summaryTable.getRecordByHash(recordHash).cells.firstElement().linksToMultipleTables
						.firstElement();
				
				// Create an ItemPosition that indicates the complete path in
				// FlatProfileInput of the record from detailsData
				position = new ItemPosition(linkPosition.getSection(), linkPosition.getSubsection(),
						linkTableId, detailsData.getRecord(i).hashCode());
			}

			// Classify the record - callee or caller
			if (!detailsData.getRecord(i).cells.get(HierarchicalProfilerTab.callerNameIndex).value.equals(functionName)) {
				Number percent = Double.parseDouble(detailsData.getRecord(i).cells
						.get(HierarchicalProfilerTab.percentCalleeIndex).value);
				sliceData = new SliceData(function, percent, position, i);
				callerData.add(sliceData);
			} else {
				Number percent = Double.parseDouble(detailsData.getRecord(i).cells
						.get(HierarchicalProfilerTab.percentCallerIndex).value);
				String callee = detailsData.getRecord(i).cells
						.get(HierarchicalProfilerTab.calleeNameIndex).value;
				sliceData = new SliceData(callee, percent, position, i);
				calleeData.add(sliceData);
			}
			slices.put(detailsData.getRecord(i), sliceData);
		}

		callerPieChart.setData(callerData);
		calleePieChart.setData(calleeData);
	}

	static public class CallPairPieChart extends PieChart {
		private HierarchicalProfilerTab profilerInput;
		private ItemPosition selectedItem;

		public CallPairPieChart(final HierarchicalProfilerTab profilerInput, Composite parent, int style) {
			super(parent, style);

			this.profilerInput = profilerInput;

			addChartMouseListener(new ChartMouseListener() {

				@Override
				public void chartMouseMoved(ChartMouseEvent event) {
					// Nothing to do
				}

				@Override
				public void chartMouseClicked(ChartMouseEvent event) {
					if (!(event.getEntity() instanceof PieSectionEntity)) {
						return;
					}

					PieSectionEntity section = (PieSectionEntity) event.getEntity();
					SliceData slice = (SliceData) section.getSectionKey();
	
					/* <no debug info> section */
					if (slice.getData() == null) {
						return;
					}
					
					ItemPosition item = (ItemPosition) slice.getData();
					profilerInput.selectCallPair(slice.getIndex());
					selectedItem = item;
				}
			});
		}

		@Override
		public void mouseDoubleClick(MouseEvent event) {
			if (selectedItem == null)
				return;
			if (profilerInput == null)
				return;

			reset();
			profilerInput.selectFunction(selectedItem);
		}
	}
}
