/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.calltree.utils;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.freescale.sa.ui.common.IHierarchicalItem;

public class Function implements IHierarchicalItem {
	private long nodeId;
	private String name;
	private List<Function> functions = new ArrayList<Function>();
	private boolean onMaxCallStack = true;
	private boolean onMaxSizeStack = true;
	private int level = 0;
	private int numCalls = 0;
	private float pctTotalCallsOfParent;
	private float pctTotalTimesItWasCalled;
	private double inclusiveTime = 0;
	private String funcFile = null;
	private int funcLineNo = 0;
	private Function parentFunc;
	private CallTreeContext context = null;
	private boolean grayedOut;
	private int functionId;
	
	public Function(String name, long id, Function parent, BigInteger callSite, CallTreeContext context, int level,
			boolean grayed, int functionId) {
		this.name = name;
		this.nodeId = id;
		this.context = context;
		this.level = level;
		this.parentFunc = parent;
		this.grayedOut = grayed;
		this.functionId = functionId;

		if (parent != null) {
			for (EdgeRec edge : context.getEdges()) {
				if ((edge.getCallerId() == nodeId) || (edge.calleeId == nodeId)) {
					if (edge.callSite.equals(callSite)) {
						numCalls = edge.numCalls;
						funcFile = edge.callSiteFile;
						funcLineNo = edge.callSiteLineNo;
						pctTotalCallsOfParent = edge.totalCallsOfParent;
						pctTotalTimesItWasCalled = edge.totalTimeItWasCalled;
						inclusiveTime = edge.inclusiveTime;
					}
				}
			}

			// check to see if we are on the maximum call stack
			onMaxCallStack = false;
			if ((parent.onMaxCallStack) && (level <= context.getMaxCallStack().size())) {
				if ((level >= 1) && (context.getMaxCallStack().elementAt(level - 1).funcId == nodeId)
						&& (context.getMaxCallStack().elementAt(level - 1).callSiteId.equals(callSite))) {
					onMaxCallStack = true;
				}
			}

			// check to see if we are on the maximum size stack
			onMaxSizeStack = false;
			if ((parent.onMaxSizeStack) && (level <= context.getMaxSizeStack().size())) {
				if ((level >= 1) && (context.getMaxSizeStack().elementAt(level - 1).funcId == nodeId)
						&& (context.getMaxSizeStack().elementAt(level - 1).callSiteId.equals(callSite))) {
					onMaxSizeStack = true;
				}
			}
		}
	}

	/**
	 * @return the level
	 */
	public int getLevel() {
		return level;
	}

	/**
	 * @param level
	 *            the level to set
	 */
	public void setLevel(int level) {
		this.level = level;
	}

	public void changeName(String name) {
		this.name = name;
	}

	public String getName() {
		return this.name;
	}

	public long getId() {
		return this.nodeId;
	}

	public void addFunction(Function f) {
		functions.add(f);
	}

	public List<Function> getFunctions() {
		return this.functions;
	}

	public boolean isRecursive() {

		/* Check if the fundId is in the parents tree */
		Function parent = this.parentFunc;

		while (parent != null) {
			if (parent.functionId == functionId && functionId != EdgeRec.INVALID_FUNCTION_ID) {
				return true;
			}
			parent = parent.parentFunc;
		}
		return false;
	}

	public int getNumCalls() {
		return numCalls;
	}

	public double getPctTotalCallsOfParent() {
		return pctTotalCallsOfParent;
	}

	public double getPctTotalTimesItWasCalled() {
		return pctTotalTimesItWasCalled;
	}

	public double getInclusiveTime() {
		return inclusiveTime;
	}

	public String getFuncFile() {
		return funcFile;
	}

	public int getFuncLineNo() {
		return funcLineNo;
	}

	/**
	 * @return the parentFunc
	 */
	public Function getParentFunc() {
		return parentFunc;
	}

	/**
	 * @param parentFunc
	 *            the parentFunc to set
	 */
	public void setParentFunc(Function parentFunc) {
		this.parentFunc = parentFunc;
	}

	/**
	 * @return the onMaxCallStack
	 */
	public boolean isOnMaxCallStack() {
		return onMaxCallStack;
	}

	/**
	 * @param onMaxCallStack
	 *            the onMaxCallStack to set
	 */
	public void setOnMaxCallStack(boolean onMaxCallStack) {
		this.onMaxCallStack = onMaxCallStack;
	}

	/**
	 * @return the onMaxSizeStack
	 */
	public boolean isOnMaxSizeStack() {
		return onMaxSizeStack;
	}

	/**
	 * @return the grayedOut
	 */
	public boolean isExcludedSymbol() {
		return grayedOut;
	}

	/**
	 * @param onMaxSizeStack
	 *            the onMaxSizeStack to set
	 */
	public void setOnMaxSizeStack(boolean onMaxSizeStack) {
		this.onMaxSizeStack = onMaxSizeStack;
	}

	/**
	 * Returns a list with the children for arg0 for the parent that has the ID
	 * equal to l.
	 * 
	 * @param l
	 *            The ID of the parent.
	 * @param callTree
	 *            CallTreeInput useful for non UI runs. It should not be null
	 *            when UI is used.
	 */
	public List<Function> getChildrenFromTree(long l, CallTreeContext context) {
		List<Function> childrens = new ArrayList<Function>();

		for (int i = 0; i < context.getEdges().size(); i++) {
			if (context.getEdges().get(i).getCallerId() == l) {
				Function func = new Function(context.getEdges().get(i).calleeShortName,
						context.getEdges().get(i).calleeId, this, context.getEdges().get(i).callSite, context,
						level + 1, context.getEdges().get(i).isGrayedOut(), context.getEdges().get(i).getCalleeFunctionId());
				childrens.add(func);
			}
		}
		return childrens;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (int) (nodeId ^ (nodeId >>> 32));
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof Function)) {
			return false;
		}
		Function other = (Function) obj;
		if (funcFile == null) {
			if (other.funcFile != null) {
				return false;
			}
		} else if (!funcFile.equals(other.funcFile)) {
			return false;
		}
		if (nodeId != other.nodeId) {
			return false;
		}
		if (funcLineNo != other.funcLineNo) {
			return false;
		}
		if (Double.doubleToLongBits(inclusiveTime) != Double.doubleToLongBits(other.inclusiveTime)) {
			return false;
		}
		if (level != other.level) {
			return false;
		}
		if (name == null) {
			if (other.name != null) {
				return false;
			}
		} else if (!name.equals(other.name)) {
			return false;
		}
		if (numCalls != other.numCalls) {
			return false;
		}
		if (onMaxCallStack != other.onMaxCallStack) {
			return false;
		}
		if (onMaxSizeStack != other.onMaxSizeStack) {
			return false;
		}
		if (parentFunc == null) {
			if (other.parentFunc != null) {
				return false;
			}
		} else if (!parentFunc.equals(other.parentFunc)) {
			return false;
		}
		if (Float.floatToIntBits(pctTotalCallsOfParent) != Float.floatToIntBits(other.pctTotalCallsOfParent)) {
			return false;
		}
		if (Float.floatToIntBits(pctTotalTimesItWasCalled) != Float.floatToIntBits(other.pctTotalTimesItWasCalled)) {
			return false;
		}

		return true;
	}

	@Override
	public IHierarchicalItem getParent() {
		return null;
	}

	@Override
	public <T extends IHierarchicalItem> void setParent(T parent) {
	}

	@Override
	public Collection<? extends IHierarchicalItem> getChildren() {
		List<Function> children = new ArrayList<Function>();

		for (int i = 0; i < context.getEdges().size(); i++) {
			if (context.getEdges().get(i).getCallerId() == getId()) {
				Function func = new Function(context.getEdges().get(i).calleeShortName,
						context.getEdges().get(i).calleeId, this, context.getEdges().get(i).callSite, context,
						level + 1, context.getEdges().get(i).isGrayedOut(), context.getEdges().get(i).getCalleeFunctionId());

				children.add(func);
			}
		}

		/* Check the number children */
		return children;
	}

	@Override
	public boolean hasChildren() {
		for (EdgeRec edge : context.getEdges()) {
			if (edge.getCallerId() == getId()) {
				return true;
			}
		}
		return false;
	}
}
